
/*
 * LibSylph -- E-Mail client library
 * Copyright (C) 1999-2006 Hiroyuki Yamamoto
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 */

#include "mmc_hlr.h"
#include "defs.h"
#include <libgnomevfs/gnome-vfs.h>
#include <libgnomevfs/gnome-vfs-volume-monitor.h>
#include <libgnomevfs/gnome-vfs-volume.h>

static void mmc_check_for_mount(GnomeVFSVolumeMonitor * monitor, MmcHlrCallback * cb_ptr);

static MmcHlrCallback *gl_mmc_cb_ptr = NULL;

/**
  This function will be called to register the mmc call back
  @param cb_ptr    pointer to the MmcHlrCallback
  @return flag to indicate whether regsitering was successful 
  */
gboolean mmc_hlr_init(MmcHlrCallback * cb_ptr)
{
	GnomeVFSVolumeMonitor *monitor = NULL;
	if ((cb_ptr != NULL) &&
	    (cb_ptr->mmc_hlr_volume_mounted_cb != NULL) &&
	    (cb_ptr->mmc_hlr_volume_unmounted_cb != NULL)) {
		gnome_vfs_init();
		monitor = gnome_vfs_get_volume_monitor();
		/* check for mounted volumes */
		if (monitor == NULL) {
			return FALSE;
		}
		mmc_check_for_mount(monitor, cb_ptr);
		g_signal_connect(monitor, "volume_mounted",
				 G_CALLBACK(cb_ptr->mmc_hlr_volume_mounted_cb), NULL);
		g_signal_connect(monitor, "volume_unmounted",
				 G_CALLBACK(cb_ptr->mmc_hlr_volume_unmounted_cb), NULL);
		gl_mmc_cb_ptr = cb_ptr;
		return TRUE;
	} else {
		return FALSE;
	}
}


/**
  This function will be called to check for mounted volumes 
  @param monitor    pointer to the volume being monitored 
  @param cb_ptr    pointer to the MmcHlrCallback
  @return none 
*/
static void mmc_check_for_mount(GnomeVFSVolumeMonitor * monitor, MmcHlrCallback * cb_ptr)
{

	GList *mounted_volume_list = NULL;
	GList *cur = NULL;
	gchar *path = NULL;
	gchar *mmc_path = NULL;
	GnomeVFSVolume *volume = NULL;
	GnomeVFSURI *mmc_uri = NULL;

	mounted_volume_list = gnome_vfs_volume_monitor_get_mounted_volumes(monitor);

	if (mounted_volume_list == NULL) {
		//Should we not have to return proper code here
		return;
	}
	for (cur = mounted_volume_list; cur != NULL; cur = cur->next) {
		path = NULL;
		mmc_path = NULL;
		volume = (GnomeVFSVolume *) cur->data;
		if (volume != NULL) {
			path = gnome_vfs_volume_get_activation_uri(volume);
		}
		if (path == NULL) {
			continue;
		}
		mmc_uri = gnome_vfs_uri_new(path);
		g_free(path);
		if (mmc_uri != NULL) {
			mmc_path = (gchar *) gnome_vfs_uri_get_path(mmc_uri);
		}
		if (mmc_path == NULL) {
			gnome_vfs_uri_unref(mmc_uri);
			continue;
		}
		if (strcmp(mmc_path, MMC_PATH) == 0) {
			cb_ptr->mmc_hlr_volume_mounted_cb(monitor, volume);
			break;
		}
		gnome_vfs_uri_unref(mmc_uri);
	}
	/* free the element in the list */
	for (cur = mounted_volume_list; cur != NULL; cur = cur->next) {
		volume = (GnomeVFSVolume *) cur->data;
		gnome_vfs_volume_unref(volume);
	}
	/* Free the volume list */
	g_list_free(mounted_volume_list);
}

/**
  This function will be called to clean up static memory 
  @param none 
  @return none 
*/
void mmc_cleanup(void)
{
	g_free(gl_mmc_cb_ptr);

	gl_mmc_cb_ptr = NULL;

}
