
/*
 * LibSylph -- E-Mail client library
 * Copyright (C) 1999-2006 Hiroyuki Yamamoto
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 */

#ifdef HAVE_CONFIG_H
#  include "config.h"
#endif
#include <string.h>

#include "htmltotextconv.h"
#include "common.h"
#include "utils.h"
#include  "defs.h"

gboolean
html_to_text_converter(const gchar * htmlfilename, const gchar * charset, gchar ** textfile)
{
	CodeConverter *conv_viewer = NULL;
	HTMLParser *parser_viewer = NULL;
	gchar *str_viewer = NULL;
	FILE *html_fptr = NULL, *conv_fptr = NULL;
	GString *tempbuf = NULL;

	if ((htmlfilename == NULL) || (textfile == NULL)) {
		debug_print(" Invalid arguments \n");
		return FALSE;
	}
	*textfile = NULL;
	tempbuf = g_string_new(NULL);
	if (tempbuf == NULL) {
		debug_print("Memory alloc for GString failed \n");
		return FALSE;
	}

	g_string_printf(tempbuf, "%s.conv", htmlfilename);
	*textfile = tempbuf->str;
	g_string_free(tempbuf, FALSE);

	conv_viewer = conv_code_converter_new(charset);

	if ((html_fptr = fopen(htmlfilename, "rb")) == NULL) {
		debug_print(" Error in opening file \n");
		return FALSE;
	}

	if ((conv_fptr = fopen(*textfile, "wb")) == NULL) {
		send_engine_error_to_ui(errno);
		debug_print(" Error in writing file \n");
		fclose(html_fptr);
		g_free(*textfile);
		*textfile = NULL;
		return FALSE;
	}

	parser_viewer = html_parser_new(html_fptr, conv_viewer);

	while ((str_viewer = html_parse(parser_viewer)) != NULL) {
		fwrite(str_viewer, strlen(str_viewer), 1, conv_fptr);
	}

	html_parser_destroy(parser_viewer);
	conv_code_converter_destroy(conv_viewer);
	fclose(html_fptr);
	fclose(conv_fptr);
	return TRUE;
}


gboolean
html_to_text_convert_buffer(const gchar * htmldata, const gchar * charset, gchar ** convertedtext)
{
	CodeConverter *conv_viewer = NULL;
	HTMLParser *parser_viewer = NULL;
	gchar *str_viewer = NULL;
	FILE *html_fptr = NULL;
	GString *textSting = NULL;
	gchar tmpfile[MAXPATHLEN + 1] = { 0 };

	if ((htmldata == NULL) || (convertedtext == NULL)) {
		debug_print(" Invalid arguments \n");
		return FALSE;
	}

	*convertedtext = NULL;

	textSting = g_string_new(NULL);
	if (textSting == NULL) {
		debug_print("Memory alloc for GString failed \n");
		return FALSE;
	}

	/* Create temp file */
	g_snprintf(tmpfile, MAXPATHLEN, "%s%chtmltext.%08x",
		   get_tmp_dir(), G_DIR_SEPARATOR, (gint) htmldata);

	conv_viewer = conv_code_converter_new(charset);

	if ((html_fptr = fopen(tmpfile, "w+b")) == NULL) {
		send_engine_error_to_ui(errno);
		debug_print(" Error in opening file \n");
		return FALSE;
	}
	fwrite(htmldata, strlen(htmldata), 1, html_fptr);
	rewind(html_fptr);
	parser_viewer = html_parser_new(html_fptr, conv_viewer);

	while ((str_viewer = html_parse(parser_viewer)) != NULL) {
		g_string_append(textSting, str_viewer);
	}

	html_parser_destroy(parser_viewer);
	conv_code_converter_destroy(conv_viewer);
	fclose(html_fptr);
	unlink(tmpfile);
	*convertedtext = textSting->str;
	g_string_free(textSting, FALSE);
	return TRUE;
}
