
/*
 * LibSylph -- E-Mail client library
 * Copyright (C) 1999-2006 Hiroyuki Yamamoto
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 */

#ifndef _FOLDER_H_
#define _FOLDER_H_

#include <glib.h>
#include <time.h>
#include <libgnomevfs/gnome-vfs.h>
#include <libgnomevfs/gnome-vfs-volume-monitor.h>
#include <libgnomevfs/gnome-vfs-volume.h>
#include "procmsg.h"

typedef struct _Folder Folder;
typedef struct _InboxStatus InboxStatus;
typedef struct _FolderAttachInfo FolderAttachInfo;

#define MSG_0   0
#define MSG_1   1
#define MSG_2   2
#define MSG_3   3
#define MSG_4   4


#define FOLDER_SUCCESS 0
#define FOLDER_FAIL   -1
#define FOLDER_XFER 1

#define FOLDER_ERROR_BASE 5000
#define FOLDER_CREATE_ERROR (FOLDER_ERROR_BASE+1)
#define FOLDER_DELETE_ERROR (FOLDER_ERROR_BASE+2)
#define FOLDER_RENAME_ERROR (FOLDER_ERROR_BASE+3)
#define FOLDER_FILE_OPEN_ERROR (FOLDER_ERROR_BASE+4)
#define FOLDER_FILE_WRITE_ERROR (FOLDER_ERROR_BASE+5)
#define FOLDER_READ_ERROR (FOLDER_ERROR_BASE+6)
#define FOLDER_REMOVE_MSG_ERROR (FOLDER_ERROR_BASE+7)
#define FOLDER_REMOVE_MSGS_ERROR (FOLDER_ERROR_BASE+8)
#define FOLDER_MOVE_MSG_ERROR (FOLDER_ERROR_BASE+9)
#define FOLDER_DEST_FILE_NOT_FOUND (FOLDER_ERROR_BASE+10)
#define FOLDER_SRC_FILE_NOT_FOUND (FOLDER_ERROR_BASE+11)
#define FOLDER_MSG_FILE_ERROR (FOLDER_ERROR_BASE+12)
#define FOLDER_MSG_FOLDER_ERROR (FOLDER_ERROR_BASE+13)
#define FOLDER_MSG_PATH_ERROR (FOLDER_ERROR_BASE+14)
#define FOLDER_UPDATE_MSG_STATUS_ERROR (FOLDER_ERROR_BASE+15)
#define FOLDER_POPULATE_ERROR (FOLDER_ERROR_BASE+16)
#define FOLDER_GET_HEADER_ERROR (FOLDER_ERROR_BASE+17)
#define FOLDER_GET_MSG_FOLDER_ERROR (FOLDER_ERROR_BASE+18)
#define FOLDER_SCAN_ERROR (FOLDER_ERROR_BASE+19)
#define FOLDER_REMOVE_ALL_MSGS_ERROR (FOLDER_ERROR_BASE+20)

#define FOLDER_MOVE_MSGS_ERROR (FOLDER_ERROR_BASE +21)
#define FOLDER_STANDARD_FOLDER_DELETION_ERROR (FOLDER_ERROR_BASE +22)
#define FOLDER_STANDARD_FOLDER_RENAME_ERROR (FOLDER_ERROR_BASE +23)
#define FOLDER_STANDARD_FOLDER_CREATION_ERROR (FOLDER_ERROR_BASE +24)
#define FOLDER_EXISTING_ERROR (FOLDER_ERROR_BASE +25)
#define FOLDER_DOESNOT_EXIST_ERROR (FOLDER_ERROR_BASE +26)
#define FOLDER_MSG_ERROR (FOLDER_ERROR_BASE +27)
#define FOLDER_ERROR (FOLDER_ERROR_BASE +28)
#define FOLDER_PARAM_MISSING_ERROR (FOLDER_ERROR_BASE +29)
#define FOLDER_INVALID_PATH_ERROR (FOLDER_ERROR_BASE +30)
#define FOLDER_PATH_NOT_EXIST_ERROR (FOLDER_ERROR_BASE +31)
#define FOLDER_COPY_MSG_ERROR (FOLDER_ERROR_BASE+32)
#define FOLDER_COPY_MSGS_ERROR (FOLDER_ERROR_BASE +33)
#define FOLDER_INITIALIZATION_ERROR (FOLDER_ERROR_BASE +34)
#define FOLDER_ATTACHMENT_NOT_DOWNLOADED (FOLDER_ERROR_BASE +35)
#define FOLDER_ATTACHMENT_REMOVE_ERROR (FOLDER_ERROR_BASE +36)
#define FOLDER_CREATE_MESSAGE_FILE_ERROR (FOLDER_ERROR_BASE +37)
#define FOLDER_READ_MESSAGE_FILE_ERROR (FOLDER_ERROR_BASE +38)
#define FOLDER_MESSAGE_LIST_READ_ERROR (FOLDER_ERROR_BASE +39)
#define FOLDER_MESSAGE_LIST_WRITE_ERROR (FOLDER_ERROR_BASE +40)
#define FOLDER_WRITE_MESSAGE_FILE_ERROR (FOLDER_ERROR_BASE +41)
#define FOLDER_CREATE_MSG_ID_FILE_ERROR (FOLDER_ERROR_BASE +42)
#define FOLDER_ROOT_CA_FILE_NOT_FOUND_ERROR (FOLDER_ERROR_BASE +43)
#define FOLDER_MEM_ALLOC_FAILURE (FOLDER_ERROR_BASE + 44)
#define FOLDER_MSG_DELETED_ERROR (FOLDER_ERROR_BASE + 45)
#define FOLDER_NO_PERMS (FOLDER_ERROR_BASE + 46)
#define ARCHIVE_FOLDER_CREATE_ERROR (FOLDER_ERROR_BASE + 47)
#define FOLDER_ARCHIVE_ERROR (FOLDER_ERROR_BASE + 48)
#define FOLDER_ARCHIVE_DELETE_FILE (FOLDER_ERROR_BASE + 49)
#define FOLDER_ARCHIVE_STATUS_ERROR (FOLDER_ERROR_BASE + 50)
#define FOLDER_ARCHIVE_MSG_DELETE_FILE (FOLDER_ERROR_BASE + 51)
#define FOLDER_ARCHIVE_MSG_MOVE_FILE (FOLDER_ERROR_BASE + 52)
#define FOLDER_ARCHIVE_MSG_COPY_FILE (FOLDER_ERROR_BASE + 53)
#define FOLDER_OP_NOT_SUPPORTED (FOLDER_ERROR_BASE + 54)
#define FOLDER_ARCHIVE_FOLDER_CREATION_ERROR (FOLDER_ERROR_BASE +55)
#define FOLDER_ARCHIVE_FOLDER_DELETE_ERROR (FOLDER_ERROR_BASE +56)
#define FOLDER_PROGRESS (FOLDER_ERROR_BASE +57)
#define FOLDER_COMPLETE (FOLDER_ERROR_BASE +58)
#define FOLDER_REMOVE_SERVER_MSGS_ERROR (FOLDER_ERROR_BASE+59)
#define FOLDER_DEL_PROGRESS (FOLDER_ERROR_BASE +60)
#define FOLDER_DEL_MOVE_MSG_ERROR (FOLDER_ERROR_BASE +61)
#define FOLDER_DEL_COPY_MSG_ERROR (FOLDER_ERROR_BASE +62)

#define FOLDER(obj) ((Folder *)obj)
#define FOLDER_TYPE(obj) (FOLDER(obj)->type)

#define NUM_MSGS_TO_BE_RETURNED 25
#define FILE_PERMS 0666

#define MAX_STR_LEN(str1,str2) \
( \
 strlen(str1)>strlen(str2)?strlen(str1):strlen(str2) \
)
#define MIN_STR_LEN(str1,str2) \
( \
 strlen(str1)<strlen(str2)?strlen(str1):strlen(str2) \
)

struct _Folder {
	gchar *name;
	gchar *path;
	gint unread;
	gint new_msgs;
	gint total;
	guint size;
	glong max;
	gboolean update_req;
	FolderType type;
};

struct _InboxStatus {
	gint exists;
	gint recent;
	gint unseen;
	guint32 uid_validity;
};

struct _FolderAttachInfo {
	gchar *msgid;
	gchar *section_number;
};

typedef enum {
	UNREAD = 1,
	SEEN = 2,
	DELETE = 3,
	REMOTE
} ListType;

typedef void (*FolderProgressCb) (gint cur, gint total);

gint init_folder(void);

gint folder_get_last_msg_num(void);

gint folder_create(const gchar * name, const gchar * path,
		   FolderType type, gboolean create_archive);

Folder *folder_set_props(const gchar * name, const gchar * path, const gchar * type);

Folder *folder_new(const gchar * name, const gchar * path, FolderType type);

void folder_set_name(Folder * folder, const gchar * name);

gint folder_delete(const gchar * name);

gint folder_rename(const gchar * folder_name, const gchar * name);

Folder *folder_find_obj_by_name(const gchar * name);

GList *folder_get_list(void);

gchar *folder_get_path(const Folder * folder);

gchar *folder_get_list_path(void);

void message_save_as(gchar * folder, gchar * filename);

gint folder_move_msg(const gchar * src_folder, const gchar * dest_folder,
		     const gchar * msgid, gchar ** dest_msgid_ptr);

gint folder_copy_msg(const gchar * src_folder, const gchar * dest_folder,
		     const gchar * msgid, gchar ** dest_msgid_ptr);

gint folder_copy_msgs(const gchar * src_folder, const gchar * dest_folder, GSList * msg_list);

gint folder_remove_msg(const gchar * folder, const gchar * msg_id);

gint folder_remove_msgs(const gchar * folder, GSList * msg_list);

gint folder_remove_all_msg(const gchar * folder);

gchar *folder_get_message_file_path(const gchar * msgid);

gint folder_update_msg_status(PrefsAccount * prefs_account, GSList * msg_status);

MsgInfo *folder_parse_msg(const gchar * file);

Folder *folder_get_default_inbox();

gint folder_scan(Folder * folder);

Folder *folder_get_msg_folder(const gchar * msgid);

gboolean folder_read_folder_func(GNode * node, gpointer data);

gboolean folder_build_tree(GNode * node, gpointer data);

gchar *folder_get_new_msg_file(const gchar * account_name, const gchar * destfolder,
			       gint * msgnum_ptr);

gchar *folder_fetch_msg(const Folder * folder, const gchar * msgid);

gint folder_rename_files(const Folder * folder, const gchar * new_name);

gchar *folder_get_mark_file(const Folder * item);

GList *folder_add(Folder * folder);

GSList *folder_populate_folder(const gchar * name);

GSList *folder_get_msgid_files(const gchar * folder, const gchar * msgid);

GSList *folder_get_msgid_all_files(const gchar * folder, const gchar * msgid,
				   gboolean * body_download_flag, gboolean add_mhdr);


gint folder_create_message_id_file(void);

gboolean is_message_unread(const gchar * msgid);

gboolean is_message_seen(const gchar * msgid);

gint folder_message_add_list(const gchar * msgid, ListType type);

gint folder_message_del_list(const gchar * msgid, ListType type);

gint folder_remove_attachment(const gchar * msgid, const gchar * section_number);

void folder_cleanup(void);

gint folder_write_list(ListType type, const gchar * msgid);

gboolean folder_is_msg_deleted(const gchar * msgid);

gboolean folder_is_msg_remote(const gchar * msgid);

gint folder_add_list(const gchar * msgid, gboolean remote_listflag);

gint folder_delete_list_msg(const gchar * msgid, gboolean remote_listflag);

GSList *folder_get_queued_msgid_list(const gchar * account_name, ListType type);

gint folder_move_msgs(const gchar * src_folder, const gchar * dest_folder, GSList * msg_list);

gboolean check_folder_perms(const gchar * folder, gboolean check_write_perms);

gboolean folder_init_lists(void);

gint folder_update_unread(const gchar * msgid, const gchar * destmsgid, gboolean move_operation);

gboolean check_msgid(const gchar * msgid);

gint folder_copy_from_mmc(const gchar * msgid);
gint folder_update_mmc_folder(const gchar * path);
gint folder_remove_mmc_folder(void);
GSList *folder_sort_msglist(GSList * msgid_list, gboolean retr_flag);

gchar *folder_get_msgid_from_path(const gchar * msgidpath);

MsgInfo *folder_get_info(const gchar * name, const gchar * file_path, gboolean archive_flag);

gboolean folder_is_msg_valid(const MsgInfo * msginfo);

/**
 This function returns size of archive folder 
 @param  none 
 @return size of the archive folder or else,FOLDER_FAIL 
 */
gint folder_get_arch_size(gchar * file_path, gboolean archive);


/** 
  This function calls the appropriate S/MIME function to 
  verify/decrypt an S/MIME message that is being opened
  @param file - Contents that need to be verified/decrypted
  @param smime - Whether message is signed, encrypted or both
  @param outfile - output file containing decrypted/verified content
  @param detached - File containing clear signed contents
  @param saved - Whether cert to be saved or not
  @param signer_email_id - Sender email ID
  @param recip_email_id - Recipient email ID
  @param signer_cert_state - Sender Cert state
  @param private_cert_state - Recipient cert state
  @param signature - Whether verification passed or not
  @param trust - Signer certificate trusted or not
  @param cert_path - Path of certificate if saved is TRUE
  @param smime - Whether mail is signed/encrypted or both
  @return 0 for success and error code for failure
*/
gint folder_process_smime_mail(const gchar * file, gchar * outfile,
			       gchar * detached, gboolean * saved,
			       const gchar * signer_email_id, const gchar * recip_email_id,
			       gint * signer_cert_state, gint * private_cert_state,
			       gboolean * signature,
			       gboolean * trust, gchar * cert_path, gint * smime,
			       gint * sender_cert_id, gint * recip_cert_id);

GList *populate_folder_list(void);


/**
 This function will be called create source message list & 
 destination message list for move/copy/delete operations. 
 @param hdr_list    the list of destination messages to be added 
 to destination folder incase of move/copy.Incase of delete,
 this contains messages to be deleted. 
 @param msgid    the message id of the message, on which the 
 folder operation is performed  
 @param dest_folder    the destination folder for folder operation 
 @param prefs_account    the pointer to the account structure of the
 message 
 @param delete_flag    indicates whether the operation is delete 
 @param update_list_ptr    OUT paramter,list of message to be updated
 in the source folder 
 @param dest_msgid     message id of destination message    
 @return updated list of destination messages  
*/
GSList *folder_append_header_list(GSList * hdr_list, const gchar * msgid,
				  const gchar * dest_folder, const PrefsAccount * prefs_account,
				  gboolean delete_flag, GSList ** update_list_ptr,
				  const gchar * dest_msgid);

GSList *folder_rem_parts(const gchar * srcfolder, const gchar * msgid, gboolean * read_flag_ptr);
GSList *folder_get_queued_complete_list(void);


/**
 This function will be called by the email search to verify and decrypt,
 SMIME mail.
 @param msgid message id
 @return path to the file contains decrypted mail
               (to be freed by the calling function)
  
*/
gchar *folder_process_smime_message(const gchar * msgid);


/** This function is used to return the S/MIME type of
    the mail

    @param header - MsgHeader
    @param file - File containing the email content
    @return gint returns S/MIME type
*/
gint folder_get_smime_type(MsgHeader * header, const gchar * file);


/**
  This function gets the required file paths before calling the 
  process_smime_message. This function is called from al.c,
  al_support.c, pop_recv.c modules.

  @param outfile - output file containing decrypted/verified content
  @param detached - File containing clear signed contents
  @param cert_path - Path of certificate if saved is TRUE
  @param file - Contents that need to be verified/decrypted
  @param saved - Whether cert to be saved or not
  @param signer_email_id - Sender email ID
  @param recip_email_id - Recipient email ID
  @param signer_cert_state - Sender Cert state
  @param private_cert_state - Recipient cert state
  @param signature - Whether verification passed or not
  @param trust - Signer certificate trusted or not
  @param smime - Whether its signed/encrypted or both
*/
gint
folder_prepare_process_smime(gchar * outfile, gchar * detached, gchar * cert_path,
			     const gchar * file, gboolean * saved,
			     const gchar * signer_email_id, const gchar * recip_email_id,
			     gint * recip_cert_state, gint * private_cert_state,
			     gboolean * signature, gboolean * trust, gint * smime,
			     gint * sender_cert_id, gint * recip_cert_id);

gint folder_mmc_move_file(const gchar * src_path, const gchar * dest_path, gboolean overwrite_flag);

gchar *folder_get_file_path(const gchar * folder,
			    const gchar * msgid, gboolean destination_flag, gchar ** dest_msgid);

/** 
  This function increments the count in Folder structure for a 
  folder. 

  @param folder->name - Folder name whose count should be incremented
  @return FOLDER_SUCCESS on success or corresponding failure
*/
gint folder_increment_msg(const gchar * folder);


/**
  Functions returns the mounted path of archival folder
 
  @return gchar* - path of the folder
*/
gchar *folder_get_archive_path(void);

/**
  Function changes time stamp on all folder to current time stamp
  to enable cache file re-generation
  
  @param folder_list List of folders to touch
  @return FOLDER_SUCCESS on success and FOLDER_FAIL on failure
*/
gint folder_change_time(GList * folder_list);

/**
 This function checks if the message ID in list is a pop header

 @param file_list - List containing message IDs
 @return TRUE if message ID is for pop3 else FALSE
*/

gboolean folder_check_pop_header(GSList * file_list);

gchar *folder_get_msgid(const gchar * filename);


/** 
  Function to fetch all unread mails required for desktop notification

  @param - outparam to give count of all unread mails
  @return GSList - List of unread mails msgid
*/
GSList *get_unread_mails(gint * count);


/**
    This functions sets info about unread mails in TN
      
    @return TRUE o success and FALSE on failure
*/
gboolean folder_send_tn_unread_mails(void);

/**
    This functions sets info about mail status in TN.
      
    @param TRUE if a new mail is arrived.
    	   FALSE if operation other than mail arrival takes place.
*/
void folder_notify_tn(gboolean mail_arrived);

/**
  Functions returns the mounted file path on archival
 folder
  @param  name of the file 
  @return gchar* - path of the folder
*/
gchar *folder_get_mounted_path(gchar * name);

/**
 This function gets msginfo's for the files in the list

 @param list_file_info - List containing files paths
 @param archive_flag - Whether file is archival
 @param index - Index from which list_file_info should be read

 @return List of message infos
*/
GSList *folder_populate_from_list(GList * list_file_info, gint index, const gchar * folder_name);

/**
  Loads information of all files in a directory

  @param folder_name - Name of the folder whose files need to loaded
  @return GSList of all files in the directory
*/
GList *folder_get_file_list(const gchar * folder_name);

/**
   This function used to compare function used in g_slist_sort function.
   This function compares the message numbers of message id and return 
   the result.This function will work only if the list contaisn 
   message id of the same account. 

   @param a first element
   @param b second element 
   @return -1 in case of any invalid input, otherwise returns the compared 
    result. 
*/
gint folder_msgnum_sort_comparefunc(gconstpointer a, gconstpointer b);
gboolean folder_check_partial(const gchar * file);

/**
  This function uses the presence of cache file to check whether only settings were restored. 

  @param    none 
  @return    TRUE if only settings were restored 
*/
gboolean folder_settings_only_restored(void);

gint check_archive(const gchar * msgid, gboolean * archive_flag_ptr);

gint del_arch_file(const gchar * msgid, gboolean archive_flag);

gint update_mmc_flag(gboolean flag, const gchar * volume);
#endif
