
/*
 * LibSylph -- E-Mail client library
 * Copyright (C) 1999-2006 Hiroyuki Yamamoto
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 */

#include <stdio.h>

#include <sys/stat.h>
#include <string.h>
#include <stdlib.h>
#include <glib.h>
#include <glib/gprintf.h>
#include <libgnomevfs/gnome-vfs-xfer.h>
#include <libgnomevfs/gnome-vfs-file-info.h>
#include <libgnomevfs/gnome-vfs-ops.h>
#include <time.h>

#include "xml.h"
#include "common.h"
#include "utils.h"
#include "prefs_account.h"
#include "account.h"
#include "prefs.h"
#include "procmsg.h"
#include "procheader.h"
#include "defs.h"
#include "codeconv.h"
#include "folder.h"
#include "engine_errno.h"
#include "smime_security.h"
#include "email_gconf.h"
#include "xml_cache.h"




/* Static variables */

static GList *gl_folder_list = NULL;
/* this is list of folders in the application specific directory */

static GHashTable *gl_message_hash = NULL;
/* use hashtable to make is_message_unread faster */

static GSList *gl_message_list = NULL;
/* this is list of unread messages */

static GSList *gl_msg_seen_list = NULL;
/* this is list of msgids to be set \Seen flag to be set in the server */

static gchar *gl_unread_list_path = NULL;
static gchar *gl_seen_list_path = NULL;
static gchar *gl_remote_list_path = NULL;
static gchar *gl_delete_list_path = NULL;

static GSList *gl_msg_del_list = NULL;
/* this is list of unread messages */

static GSList *gl_msg_rem_list = NULL;
/* this is list of remote messages */

static gint gl_last_msg_num = 0;
/* this is last message number used in the Inbox */

/* Global variables */

extern gboolean gl_mmc_status_flag;
/* This flag indicates mmc status.Defined in utils.c */

static gchar *gl_volume_mounted_path = NULL;
/* This is mmc path */

extern gint engine_errno;
/* This flag indicates error number.Defined in utils.c */

/* static functions */

/**
  This function returns the path of the file

  @type - Type of list for which path is required
  @return gchar* - Path of the file
*/
static const gchar *folder_get_list_file_path(ListType type);

static GList *folder_prepare_list(const gchar * path,
				  const gchar * root_folder, GList * foldlist, Folder * root_obj);

static gint folder_create_standard_folder(const gchar * rootdir);

static gchar *folder_create_msg_id(const gchar * msgid,
				   const gchar * dest_folder, gboolean newmsg_flag);

static gint
folder_xfer_progress_callback(const GnomeVFSXferProgressInfo * info, const gpointer data);

static gint folder_update_folder(Folder * folder, const gchar * new_name);

static gchar *folder_get_arch_msg(const gchar * file_path, const gchar * name, gchar ** msgid);

MsgInfo *folder_get_info(const gchar * name, const gchar * file_path, gboolean archive_flag);

static gint folder_get_archive_status(void);

static gint folder_mmc_copy_file(const gchar * src_path,
				 const gchar * dest_path, gboolean overwrite_flag);

static gint folder_update_folder_list(const gchar * full_path, const gchar * name, gint type);


static gchar *folder_alloc_file_path(const gchar * src_folder, const gchar * msgid);

static gint folder_message_read_list(ListType type);

static gint folder_read_list(gboolean remote_listflag);

static gboolean folder_is_standard_folder(const gchar * name);

static gint folder_create_dir_if_not_exist(const gchar * dir, gboolean all_perms);

static gboolean check_folder(const gchar * folder);
static gint rem_file(const gchar * file);

static gint folder_msgid_sort_comparefunc(gconstpointer a, gconstpointer b);


static glong folder_get_size(const gchar * filename);

static glong folder_get_all_size(const gchar * msgid);

static gint folder_update_count(Folder * folder_obj, const gchar * msgid, const gchar * file_path);

/** 
  This function is called by folder_process_smime_mail. It copies
  verified/decrypted contents from inter_file to outfile. 
  Contents in inter_file does not have the mime header:
  MIME-Version:1. Hence this copying is required. outfile
  contains MIME-Version:1 at the beginning and the verified/
  decrypted contents after copied into it.
  @param outfile - Final output file to be passed to engine
  @param inter_file - File with decrypted/vertified contents
  @return SMIME_OK for success and POP3_FAIL for failure
*/
static gint folder_copy_smime_contents(gchar * outfile, gchar * inter_file);

/**
  This function checks if message ID given belongs to a message in
  inbox, then checks if the message exists in inbox 
  
  @param msg_id - Message ID
  @return TRUE if message is in inbox and exists in inbox, FALSE else
*/
static gchar *folder_message_inbox_unread(gchar * msg_id);

/**
  This function reads message information 
  
  @param dest_file_path    file path of the message 
  @param dest_msgid    Message ID
  @param src_msgid    source Message ID
  @param prefs_account    account information structure 
  @return pointer to the message information 
*/
static MsgInfo *folder_read_msginfo(const gchar * dest_file_path,
				    const gchar * dest_msgid,
				    const gchar * src_msgid, const PrefsAccount * prefs_account);

/**
 This function is will create message header structure and set msgid variable  
 @param msgid    message id of the message
 @param update_list_ptr   (OUT)Gslist of messages to update 
 @param none
*/
static void folder_update_src_list(GSList ** update_list_ptr, const gchar * msgid);
/**
 This function is to delete all the temporary files related to the msgid
 @param msgid
 @param none
*/
static void folder_delete_tmp_files(const gchar * msgid);

/**
  Function which sets gconf values to be sent to TN

   @param msginfo - msginfo contains required info to be set in gconf
   @param counter - The count of of mail whose info is in msginfo

   @return gint - 0 on success
*/
static void folder_set_gconf_data(MsgInfo * msginfo, gint counter);

/**
  Initializes all gconf values to NULL

  @return void
*/
static void folder_initialize_gconf_values(void);

/**
  This functions returns path of the folder

  @param folder_name - name of folder whose path is required
  @return gchar* which is the folder path
*/
static gchar *folder_get_path_name(const gchar * folder_name);

/**
 This function is to compare all the messages in the list,using the msgid
 in the RetrMsgData structure of data in a node.
 @param a    data of node to compare 
 @param b    data of node to compare 
 @return 0 if a matches with b
*/
static gint folder_retr_sort_comparefunc(gconstpointer a, gconstpointer b);

/**
 This function creates new email setting directory and restores the 
 old setting files to it. 
 @param root    new settings directory 
 @return FOLDER_SUCCESS on success or else FOLDER_FAIL 
*/
static gint folder_restore_old_settings(const gchar * root);
/**
 This function moves or removes files in restored directory. 
 @param move_flag  indicates whether messsge specific files 
 have to be moved or removed     
 @return FOLDER_SUCCESS on success or else FOLDER_FAIL 
*/
static gint folder_move_restored_files(gboolean move_flag);

/**
 This function checks the whether folder is standard folder 
 @param name    folder name
 @return TRUE on standard folder FALSE on user folder
*/

static gboolean folder_is_standard_folder(const gchar * name)
{
	if (strncmp(name, INBOX_DIR, MAX_STR_LEN(INBOX_DIR, name)) == 0 ||
	    strncmp(name, OUTBOX_DIR, MAX_STR_LEN(OUTBOX_DIR, name)) == 0
	    || strncmp(name, DRAFT_DIR, MAX_STR_LEN(DRAFT_DIR, name)) == 0
	    || strncmp(name, SENT_DIR, MAX_STR_LEN(SENT_DIR, name)) == 0) {
		return TRUE;
	} else {
		return FALSE;
	}
}

/**
 This function checks the whether folder destination is the standard folder,
 which can not be used as destination for folder operations. 
 @param name    folder name
 @return TRUE if folder can not be used, FALSE otherwise 
*/

static gboolean check_dest_folder(const gchar * name)
{
	if (strncmp(name, OUTBOX_DIR, MAX_STR_LEN(OUTBOX_DIR, name)) == 0
	    || strncmp(name, SENT_DIR, MAX_STR_LEN(SENT_DIR, name)) == 0) {
		return TRUE;
	} else {
		return FALSE;
	}
}

/**
 This function frees all the memory pointed by the pointers 
 @param src_file_path     pointer to the source file path,
 which will be freed. 
 @param dest_file_path     pointer to the destination file path,
 which will be freed. 
 @param temp_msg_id     pointer to the message id,
 which will be freed. 
 @param dest_file     pointer to the destination file,which will be freed. 
 @return none
*/
static void
folder_gchar_free(gchar * src_file_path, gchar * dest_file_path,
		  gchar * temp_msg_id, gchar * dest_file)
{
	if (src_file_path != NULL) {
		g_free(src_file_path);
	}
	if (dest_file_path != NULL) {
		g_free(dest_file_path);
	}
	if (temp_msg_id != NULL) {
		g_free(temp_msg_id);
	}
	if (dest_file != NULL) {
		g_free(dest_file);
	}
}

/**
  This function allocates & returns the path for the message 
  @param src_folder    folder name
  @param msgid    message id (filename) 
  @return file path.This has to be freed by the calling function.
 */
static gchar *folder_alloc_file_path(const gchar * src_folder, const gchar * msgid)
{
	gchar *file_path = NULL;

	file_path = g_strdup_printf("%s%c%s%c%s", get_mail_dir(),
				    G_DIR_SEPARATOR, src_folder, G_DIR_SEPARATOR, msgid);
	return file_path;

}

/** 
  This function will be called to create a new Folder
  @param name    Name of the folder to be created
  @param path    Path where the folder has to be created
  @param type    type of the folder (F_LOCAL F_REMOTE F_UNKNOWN)
  @param create_archive    indicates that archive folder can be created 
  @return  FOLDER_SUCCESS(0) on success else any folder error code
*/

gint folder_create(const gchar * name, const gchar * path, FolderType type, gboolean create_archive)
{
	gchar *full_path = NULL;
	const gchar *tmp_path = NULL;
	gint ret = 0;

	CHECK_RETURN_VAL_IF_FAIL(name != NULL, FOLDER_CREATE_ERROR);
	CHECK_RETURN_VAL_IF_FAIL(path != NULL, FOLDER_CREATE_ERROR);
	CHECK_RETURN_VAL_IF_FAIL(check_folder(name) == TRUE, FOLDER_CREATE_ERROR);

	/* check for folder creation path */
	tmp_path = get_mail_dir();
	if (strncmp(path, tmp_path, strlen(tmp_path)) != 0) {
		return FOLDER_CREATE_ERROR;
	}

	if (folder_is_standard_folder(name) == TRUE) {
		return FOLDER_STANDARD_FOLDER_CREATION_ERROR;
	}
	if ((create_archive == FALSE) && (strcmp(name, ARCHIVE) == 0)) {
		return FOLDER_ARCHIVE_FOLDER_CREATION_ERROR;
	}
	if (folder_find_obj_by_name(name) != NULL) {
		return FOLDER_EXISTING_ERROR;
	}

	full_path = g_strdup_printf("%s%s%s", path, G_DIR_SEPARATOR_S, name);

	CHECK_RETURN_VAL_IF_FAIL(full_path != NULL, FOLDER_CREATE_ERROR);

	if (make_dir(full_path) < 0) {
		/* if the archive folder is already present,update the list */
		if ((create_archive == FALSE) || (is_dir_exist(full_path) == FALSE)) {
			g_free(full_path);
			return FOLDER_CREATE_ERROR;
		}
	}
	/* update the folder list */
	ret = folder_update_folder_list(full_path, name, type);
	g_free(full_path);
	return ret;
}


/** 
  This function is used to create a new folder instance.
  @param name    name of the folder object to be initialized
  @param path    path of the folder object to be initialized
  @return folder object.The memory allocated for folder object 
  has to be freed by the calling function.
*/
Folder *folder_new(const gchar * name, const gchar * path, FolderType type)
{
	Folder *folder = NULL;

	CHECK_RETURN_VAL_IF_FAIL(name != NULL, NULL);
	CHECK_RETURN_VAL_IF_FAIL(path != NULL, NULL);

	folder = g_new0(Folder, 1);
	if (folder == NULL) {
		return NULL;
	}
	folder->name = g_strdup(name);
	folder->path = g_strdup(path);
	folder->type = type;
	folder->new_msgs = 0;
	folder->unread = 0;
	folder->total = 0;
	folder->size = 0;
	folder->max = 0;
	return folder;
}

/** 
  This function Removes or Deletes a folder,along with the subdirectories
     in it. 
  @param folder_name    Name of the folder to be deleted
  @return  FOLDER_SUCCESS(0) on success else any folder error code
*/
gint folder_delete(const gchar * folder_name)
{
	gchar *path = NULL;
	Folder *dir = NULL;

	Folder *fold = NULL;
	gchar *recur_dir = NULL;
	GList *list = NULL;

	CHECK_RETURN_VAL_IF_FAIL(folder_name != NULL, FOLDER_DELETE_ERROR);
	CHECK_RETURN_VAL_IF_FAIL(check_folder(folder_name) == TRUE, FOLDER_DELETE_ERROR);

	/* check for standard folder name */

	if (folder_is_standard_folder(folder_name)) {
		return FOLDER_STANDARD_FOLDER_DELETION_ERROR;
	}
	if (strcmp(folder_name, ARCHIVE) == 0) {
		return FOLDER_ARCHIVE_FOLDER_DELETE_ERROR;
	}

	dir = folder_find_obj_by_name(folder_name);

	if (dir == NULL) {
		return FOLDER_DOESNOT_EXIST_ERROR;
	}
	path = folder_get_path(dir);
	CHECK_RETURN_VAL_IF_FAIL(path != NULL, FOLDER_DELETE_ERROR);

	if (remove_dir_recursive(path) < 0) {
		return FOLDER_DELETE_ERROR;
	}

	/* Go recursively and delete all sub-folders from the list */
	recur_dir = g_strdup_printf("%s%c", folder_name, G_DIR_SEPARATOR);
	for (list = gl_folder_list; list != NULL; list = list->next) {
		fold = (Folder *) list->data;
		if (fold == NULL) {
			continue;
		}
		if ((strcmp(folder_name, fold->name) == 0) ||
		    (g_str_has_prefix(fold->name, recur_dir) == TRUE)) {
			g_free(fold->name);
			g_free(fold->path);
			g_free(fold);
			gl_folder_list = g_list_delete_link(gl_folder_list,list);
			list = gl_folder_list;
		}
	}

	g_free(recur_dir);

	return FOLDER_SUCCESS;
}


/** 
  This Function will be called to rename a particular folder. 
  @param folder_name   Name of the folder to be renamed
  @param new_name    New name to be given to the folder
  @return    FOLDER_SUCCESS(0) on success else any error code
 */
gint folder_rename(const gchar * folder_name, const gchar * new_name)
{
	gchar *new_path = NULL;
	Folder *old_folder = NULL;
	gint ret_val = FOLDER_SUCCESS;

	CHECK_RETURN_VAL_IF_FAIL(folder_name != NULL, FOLDER_RENAME_ERROR);
	CHECK_RETURN_VAL_IF_FAIL(new_name != NULL, FOLDER_RENAME_ERROR);
	CHECK_RETURN_VAL_IF_FAIL(check_folder(new_name) == TRUE, FOLDER_RENAME_ERROR);
	CHECK_RETURN_VAL_IF_FAIL(check_folder(folder_name) == TRUE, FOLDER_RENAME_ERROR);

	if ((folder_is_standard_folder(folder_name) == TRUE) ||
	    (folder_is_standard_folder(new_name) == TRUE)) {
		return FOLDER_STANDARD_FOLDER_RENAME_ERROR;
	}
	old_folder = folder_find_obj_by_name(folder_name);
	if (old_folder == NULL) {
		return FOLDER_DOESNOT_EXIST_ERROR;
	}
	/* Rename the files within the folder */
	if (folder_rename_files(old_folder, new_name) != FOLDER_SUCCESS) {
		return FOLDER_FAIL;
	}


	new_path = g_strdup_printf("%s%c%s", get_mail_dir(), G_DIR_SEPARATOR, new_name);


	CHECK_RETURN_VAL_IF_FAIL(new_path != NULL, FOLDER_RENAME_ERROR);

	if (rename(old_folder->path, new_path) < 0) {
		g_free(new_path);
		return FOLDER_RENAME_ERROR;
	}
	g_free(new_path);

	/* update the folder info after this */
	ret_val = folder_update_folder(old_folder, new_name);

	return ret_val;
}


/** 
  This Function returns a Folder object by taking the name of the folder
    as a parameter 
  @param folder	    Name of the folder whose Folder object has to be returned
  @return pointer to the Folder object which was found in the list.
  This should not be freed by the calling function.
*/
Folder *folder_find_obj_by_name(const gchar * folder)
{
	Folder *dir = NULL;
	GList *list = NULL;
	gint flag = 0;
	CHECK_RETURN_VAL_IF_FAIL(folder != NULL, NULL);
	CHECK_RETURN_VAL_IF_FAIL(check_folder(folder) == TRUE, NULL);

	for (list = gl_folder_list; list != NULL; list = list->next) {
		dir = (Folder *) list->data;
		if (dir == NULL) {
			continue;
		}
		if (strcmp(folder, dir->name) == 0) {
			flag++;
			break;
		}
	}
	if (flag > 0) {
		return dir;
	} else {
		return NULL;
	}
}


/** 
  This function is used to get the path of the folder.
  @param folder	   folder structure whose path has to be returned
  @return path	   the path of the folder,this should not be freed 
  by the calling function.
*/
gchar *folder_get_path(const Folder * folder)
{
	gchar *path = NULL;

	CHECK_RETURN_VAL_IF_FAIL(folder != NULL, NULL);

	path = folder->path;
	debug_print("\nFolder get file path ret:%s\n", path);
	return path;
}




/** 
  This function reads the messagelist.txt file,and creates gl_message_list
  which contains the message ids of unread messages.
  @param  ListType(UNREAD list or SEEN list )
  @return FOLDER_SUCCESS(0) on successful read else error code
*/
static gint folder_message_read_list(ListType type)
{
	gchar *message_file_path = NULL;
	static gchar msgid[MSG_ID_LEN + 1];
	FILE *fp = NULL;
	struct stat s;
	gchar *add_msgid = NULL;

	if (type == UNREAD) {
		message_file_path = g_strdup_printf("%s%s%c%s", get_home_dir(),
						    CONFIG_FILE_PATH,
						    G_DIR_SEPARATOR, MESSAGE_LIST);
	} else if (type == SEEN) {
		message_file_path = g_strdup_printf("%s%s%c%s", get_home_dir(),
						    CONFIG_FILE_PATH, G_DIR_SEPARATOR, SEEN_LIST);
	}

	if (is_file_entry_exist(message_file_path) == FALSE) {
		g_free(message_file_path);
		return FOLDER_SUCCESS;
	}
	if (stat(message_file_path, &s) < 0) {
		g_free(message_file_path);
		return FOLDER_CREATE_MESSAGE_FILE_ERROR;
	}
	if ((s.st_mode & S_IRUSR) == 0) {
		g_free(message_file_path);
		return FOLDER_CREATE_MESSAGE_FILE_ERROR;
	}

	fp = fopen(message_file_path, "r");
	g_free(message_file_path);
	if (fp == NULL) {
		return FOLDER_READ_MESSAGE_FILE_ERROR;
	}
	while (1) {

                if(!fgets(msgid,MSG_ID_LEN + 1,fp)){
			break;
                 }

		if (type == UNREAD) {
                        add_msgid = g_strndup(msgid,strlen(msgid) - 1);
			gl_message_list = g_slist_append(gl_message_list, add_msgid);
			if (!gl_message_hash) {
				gl_message_hash = g_hash_table_new_full(g_str_hash,
									g_str_equal, NULL, NULL);
			}
			g_hash_table_insert(gl_message_hash, add_msgid, GINT_TO_POINTER(1));

		} else if (type == SEEN) {
			gl_msg_seen_list = g_slist_append(gl_msg_seen_list, g_strdup(msgid));
		}
	}
	fclose(fp);
	return FOLDER_SUCCESS;
}

/** 
  This function is used to read the message files(MSG_DEL_FILE or 
  MSG_REM_FILE) to the relevent lists (gl_msg_del_list or gl_msg_rem_list)
  @param remote_listflag    indicates which file to read 
  @return FOLDER_SUCCESS(0) on successful read else folder error code
*/
static gint folder_read_list(gboolean remote_listflag)
{
	gchar *message_file_path = NULL;
	FILE *fp = NULL;
	static gchar msgid_string[MSG_ID_LEN + 1];
	GSList *tmp_list = NULL;
	gchar *list_file = NULL;

	if (remote_listflag == TRUE) {
		list_file = MSG_REM_FILE;
	} else {
		list_file = MSG_DEL_FILE;
	}

	message_file_path = g_strdup_printf("%s%s%c%s", get_home_dir(),
					    CONFIG_FILE_PATH, G_DIR_SEPARATOR, list_file);
	if (is_file_exist(message_file_path) == FALSE) {
		g_free(message_file_path);
		return FOLDER_SUCCESS;
	}
	fp = fopen(message_file_path, "r");
	g_free(message_file_path);
	if (fp == NULL) {
		return FOLDER_READ_MESSAGE_FILE_ERROR;
	}
	while (1) {
                if(!fgets(msgid_string, MSG_ID_LEN + 1,fp)){
                   break;
                 }
		if (msgid_string)
			tmp_list = g_slist_append(tmp_list, g_strndup(msgid_string, strlen(msgid_string) - 1));
	}
	fclose(fp);
	if (remote_listflag == TRUE) {
		gl_msg_rem_list = tmp_list;
	} else {
		gl_msg_del_list = tmp_list;
	}
	return FOLDER_SUCCESS;

}

static const gchar *folder_get_list_file_path(ListType type)
{
	switch (type) {
	case UNREAD:
		if (!gl_unread_list_path)
			gl_unread_list_path = g_strdup_printf("%s%s%c%s",
							      get_home_dir(), CONFIG_FILE_PATH,
							      G_DIR_SEPARATOR, MESSAGE_LIST);
		return gl_unread_list_path;
	case SEEN:
		if (!gl_seen_list_path)
			gl_seen_list_path = g_strdup_printf("%s%s%c%s",
							    get_home_dir(), CONFIG_FILE_PATH,
							    G_DIR_SEPARATOR, SEEN_LIST);
		return gl_seen_list_path;
	case REMOTE:
		if (!gl_remote_list_path)
			gl_remote_list_path = g_strdup_printf("%s%s%c%s",
							      get_home_dir(), CONFIG_FILE_PATH,
							      G_DIR_SEPARATOR, MSG_REM_FILE);
		return gl_remote_list_path;
	case DELETE:
		if (!gl_delete_list_path)
			gl_delete_list_path = g_strdup_printf("%s%s%c%s",
							      get_home_dir(), CONFIG_FILE_PATH,
							      G_DIR_SEPARATOR, MSG_DEL_FILE);
		return gl_delete_list_path;

	default:
		debug_print("unsuppored type %d", type);
		return NULL;
	}
}

/** 
  This function is used is used to write the gl_message_list/gl_msg_seen_list
  to the file 

  @param ListType UNREAD/SEEN/DELETE/REMOTE
  @param msgid: the msgid to add to the list or NULL to write the
  whole list
  
  @return FOLDER_SUCCESS(0) on successful read else FOLDER_FAIL(-1)
*/
gint folder_write_list(ListType type, const gchar * msgid)
{
	FILE *fp = NULL;
	GSList *cur = NULL;
	GSList *msg_list = NULL;
	const gchar *list_file_path = NULL;


	list_file_path = folder_get_list_file_path(type);
	if (!list_file_path) {
		debug_print("could not get list file name");
		return FOLDER_FAIL;
	}

	switch (type) {
	case UNREAD:
		msg_list = gl_message_list;
		break;
	case SEEN:
		msg_list = gl_msg_seen_list;
		break;
	case REMOTE:
		msg_list = gl_msg_rem_list;
		break;
	case DELETE:
		msg_list = gl_msg_del_list;
		break;
	}


	if (msg_list == NULL) {
		if (unlink(list_file_path) == -1) {
			debug_print("failed to unlink %s: %s", list_file_path, strerror(errno));
		}
		return FOLDER_SUCCESS;
	}

	/* if a msgid is provided, add *that* message id to the
	 *  file. otherwise, overwrite the file with *all* msgids
	 */
	fp = fopen(list_file_path, msgid ? "a" : "w");
	if (fp == NULL) {
		send_engine_error_to_ui(errno);
		return FOLDER_WRITE_MESSAGE_FILE_ERROR;
	}

         if (msgid) {
                if((fputs(msgid,fp) == EOF) || (fputs("\n",fp) == EOF)){
                   fclose(fp);
                   return FOLDER_FAIL;
                }
        } else {
                for (cur = msg_list; cur != NULL; cur = cur->next){
                    if((fputs((gchar *) cur->data,fp) == EOF) || (fputs("\n",fp) == EOF)){
                          fclose(fp);
                          return FOLDER_FAIL;
                     }
                }
        }
	fclose(fp);
	return FOLDER_SUCCESS;
}

/** 
  This function Removes a single specified message from a specified Folder.
  @param folder	   Folder name from where the message has to be removed.
  @param msgid	  The messageID of the message that needs to be removed.
  @return FOLDER_SUCCESS(0) on success else folder error code
*/

gint folder_remove_msg(const gchar * folder, const gchar * msgid)
{
	gchar *file = NULL;
	gchar *file_escaped = NULL;
	Folder *dir = NULL;
	gint uid = 0;
	GnomeVFSResult vfs_result = GNOME_VFS_OK;

	CHECK_RETURN_VAL_IF_FAIL(folder != NULL, FOLDER_REMOVE_MSG_ERROR);
	CHECK_RETURN_VAL_IF_FAIL(msgid != NULL, FOLDER_REMOVE_MSG_ERROR);

	if ((strcmp(folder, ARCHIVE) != 0)) {
		CHECK_RETURN_VAL_IF_FAIL(check_msgid(msgid) == TRUE, FOLDER_REMOVE_MSG_ERROR);
	}

	CHECK_RETURN_VAL_IF_FAIL(check_folder(folder) == TRUE, FOLDER_REMOVE_MSG_ERROR);

	dir = folder_find_obj_by_name(folder);
	CHECK_RETURN_VAL_IF_FAIL(dir != NULL, FOLDER_REMOVE_MSG_ERROR);
	if (strcmp(folder, ARCHIVE) == 0) {
		if (CHECK_MMC_FLAG == FALSE) {
			return FOLDER_REMOVE_MSG_ERROR;
		}
		uid = get_msgid_num(msgid);
		file = g_strdup_printf("%s%c%s%c%d", gl_volume_mounted_path,
				       G_DIR_SEPARATOR, R_ARCHIVE, G_DIR_SEPARATOR, uid);
	} else {
		file = folder_get_message_file_path(msgid);
	}

	debug_print("message file : %s \n", file);

	file_escaped = gnome_vfs_escape_path_string(file);
	vfs_result = gnome_vfs_unlink(file_escaped);
	g_free(file_escaped);
	g_free(file);
	if (vfs_result == GNOME_VFS_OK) {
		if (dir->total > 0)
			dir->total--;
		folder_delete_tmp_files(msgid);
		return FOLDER_SUCCESS;
	} else {
		return FOLDER_REMOVE_MSG_ERROR;
	}

}

gint folder_increment_msg(const gchar * folder)
{
	Folder *dir = NULL;

	CHECK_RETURN_VAL_IF_FAIL(folder != NULL, FOLDER_REMOVE_MSG_ERROR);

	CHECK_RETURN_VAL_IF_FAIL(check_folder(folder) == TRUE, FOLDER_REMOVE_MSG_ERROR);

	dir = folder_find_obj_by_name(folder);
	CHECK_RETURN_VAL_IF_FAIL(dir != NULL, FOLDER_REMOVE_MSG_ERROR);

	dir->total = dir->total + 1;

	return FOLDER_SUCCESS;
}



/** 
  This function Removes a set of messages specified from a specified folder.
  @param folder	   The folder name in which list of messages exist.
  @param msg_list    The list of messageIDs of the messages to be deleted.
  This list has to be allocated and freed by the calling function.
  @return FOLDER_SUCCESS(0) on success else any error code
*/
gint folder_remove_msgs(const gchar * folder, GSList * msg_list)
{
	GSList *list = NULL;
	gchar *msgid = NULL;

	CHECK_RETURN_VAL_IF_FAIL(folder != NULL, FOLDER_REMOVE_MSGS_ERROR);
	CHECK_RETURN_VAL_IF_FAIL(msg_list != NULL, FOLDER_REMOVE_MSGS_ERROR);
	CHECK_RETURN_VAL_IF_FAIL(check_folder(folder) == TRUE, FOLDER_REMOVE_MSGS_ERROR);

	for (list = msg_list; list != NULL; list = list->next) {
		msgid = g_strdup_printf("%s", (gchar *) list->data);
		if (folder_remove_msg(folder, msgid) == FOLDER_REMOVE_MSG_ERROR) {
			g_free(msgid);
			return FOLDER_REMOVE_MSGS_ERROR;
		}
		g_free(msgid);
	}
	return FOLDER_SUCCESS;
}


/** 
  This function moves a specified message from a folder to another.
  @param src_folder	Source folder for move operation.
  @param dest_folder	Destination Folder for move operation.
  @param msg_id The MessageID of the message that has to be moved.
  @param dest_msgid_ptr    OUT parameter,destination message id generated
  for the message after copying. 
  @return FOLDER_SUCCESS(0) on success else any error code.
*/
gint
folder_move_msg(const gchar * src_folder, const gchar * dest_folder,
		const gchar * msg_id, gchar ** dest_msgid_ptr)
{
	Folder *src = NULL;
	Folder *dest = NULL;
	gchar *src_file_path = NULL;
	gchar *dest_file_path = NULL;
	gchar *dest_msgid = NULL;
	gint ret_val = FOLDER_SUCCESS;

	CHECK_RETURN_VAL_IF_FAIL(src_folder != NULL, FOLDER_MOVE_MSG_ERROR);
	CHECK_RETURN_VAL_IF_FAIL(dest_folder != NULL, FOLDER_MOVE_MSG_ERROR);
	CHECK_RETURN_VAL_IF_FAIL(msg_id != NULL, FOLDER_MOVE_MSG_ERROR);
	CHECK_RETURN_VAL_IF_FAIL(check_msgid(msg_id) == TRUE, FOLDER_MOVE_MSG_ERROR);
	CHECK_RETURN_VAL_IF_FAIL(check_folder(src_folder) == TRUE, FOLDER_MOVE_MSG_ERROR);
	CHECK_RETURN_VAL_IF_FAIL(check_folder(dest_folder) == TRUE, FOLDER_MOVE_MSG_ERROR);

	if (check_dest_folder(dest_folder) == TRUE) {
		return FOLDER_MOVE_MSG_ERROR;
	}

	src = folder_find_obj_by_name(src_folder);
	CHECK_RETURN_VAL_IF_FAIL(src != NULL, FOLDER_MOVE_MSGS_ERROR);

	dest = folder_find_obj_by_name(dest_folder);
	CHECK_RETURN_VAL_IF_FAIL(dest != NULL, FOLDER_MOVE_MSGS_ERROR);


	src_file_path = folder_get_file_path(src_folder, msg_id, FALSE, NULL);

	if (src_file_path != NULL) {
		dest_file_path = folder_get_file_path(dest_folder, msg_id, TRUE, &dest_msgid);
	}

	if (src_file_path == NULL) {
		debug_print("%s: %s memory alloc error\n", __FUNCTION__, __FILE__);
		return FOLDER_MOVE_MSG_ERROR;
	}

	if (is_file_exist(dest_file_path) == FALSE) {
		if (folder_mmc_move_file(src_file_path, dest_file_path, FALSE)
		    != FOLDER_SUCCESS) {
			g_free(dest_msgid);
			ret_val = FOLDER_MOVE_MSG_ERROR;
		} else {
			if (src->total > 0) {
				src->total--;
			}
			dest->total++;
			if (dest_msgid_ptr != NULL) {
				*dest_msgid_ptr = dest_msgid;
			}
		}

	} else {
		g_free(dest_msgid);
		ret_val = FOLDER_MOVE_MSG_ERROR;
	}
	folder_gchar_free(src_file_path, dest_file_path, NULL, NULL);
	return ret_val;
}

/** 
  This function copies a specified message from a folder to another.
  @param src_folder    Source folder for copy operation.
  @param dest_folder    Destination Folder for copy operation. 
  @param msg_id    The MessageID of the message that has to be copied.
  @param dest_msgid_ptr    OUT parameter,destination message id generated
  for the message after copying. 
  @return FOLDER_SUCCESS(0) on success else folder error code.
*/
gint
folder_copy_msg(const gchar * src_folder, const gchar * dest_folder,
		const gchar * msg_id, gchar ** dest_msgid_ptr)
{
	gchar *dest_msgid = NULL;
	gint ret_val = FOLDER_SUCCESS;
	Folder *src = NULL;
	Folder *dest = NULL;
	gchar *src_file_path = NULL;
	gchar *dest_file_path = NULL;
	CHECK_RETURN_VAL_IF_FAIL(src_folder != NULL, FOLDER_COPY_MSG_ERROR);
	CHECK_RETURN_VAL_IF_FAIL(dest_folder != NULL, FOLDER_COPY_MSG_ERROR);
	CHECK_RETURN_VAL_IF_FAIL(msg_id != NULL, FOLDER_COPY_MSG_ERROR);
	if ((strcmp(src_folder, ARCHIVE) != 0)
	    && (strcmp(dest_folder, ARCHIVE) != 0)) {
		CHECK_RETURN_VAL_IF_FAIL(check_msgid(msg_id) == TRUE, FOLDER_COPY_MSG_ERROR);
	}

	CHECK_RETURN_VAL_IF_FAIL(check_folder(src_folder) == TRUE, FOLDER_COPY_MSG_ERROR);
	CHECK_RETURN_VAL_IF_FAIL(check_folder(dest_folder) == TRUE, FOLDER_COPY_MSG_ERROR);

	if (check_dest_folder(dest_folder) == TRUE) {
		return FOLDER_COPY_MSG_ERROR;
	}

	src = folder_find_obj_by_name(src_folder);
	CHECK_RETURN_VAL_IF_FAIL(src != NULL, FOLDER_COPY_MSG_ERROR);
	dest = folder_find_obj_by_name(dest_folder);
	CHECK_RETURN_VAL_IF_FAIL(dest != NULL, FOLDER_COPY_MSG_ERROR);

	src_file_path = folder_get_file_path(src_folder, msg_id, FALSE, NULL);
	if (src_file_path != NULL) {
		dest_file_path = folder_get_file_path(dest_folder, msg_id, TRUE, &dest_msgid);
	}
	if (src_file_path == NULL) {
		return FOLDER_COPY_MSG_ERROR;
	}

	if (is_file_entry_exist(dest_file_path) == FALSE) {
		if (folder_mmc_copy_file(src_file_path, dest_file_path, FALSE)
		    != FOLDER_SUCCESS) {
			g_free(dest_msgid);
			ret_val = FOLDER_COPY_MSG_ERROR;
		}

	} else {
		ret_val = FOLDER_COPY_MSG_ERROR;
	}
	folder_gchar_free(src_file_path, dest_file_path, NULL, NULL);

	if (ret_val == FOLDER_SUCCESS) {
		/* Incase of archive,this can happen */
		if (src->total > 0) {
			src->total--;
		}
		dest->total++;
		if (dest_msgid_ptr != NULL) {
			*dest_msgid_ptr = dest_msgid;
		}
	}

	return ret_val;

}


/** 
  This function is used to Move a list of messages from one folder to 
  another.
  @param src_folder    The source folder from where the List of Messages 
  has to be moved.
  @param dest_folder    The Destination folder to where the list of 
  messages has to be Moved.
  @param msg_list    The List of MessageIDs of the Messages, to be moved.
  This list has to be allocated and freed by the calling function.
  @return FOLDER_SUCCESS(0) on success else any error code
*/
gint folder_move_msgs(const gchar * src_folder, const gchar * dest_folder, GSList * msg_list)
{
	GSList *cur = NULL;
	gint ret_val = FOLDER_SUCCESS;

	CHECK_RETURN_VAL_IF_FAIL(src_folder != NULL, FOLDER_MOVE_MSGS_ERROR);
	CHECK_RETURN_VAL_IF_FAIL(dest_folder != NULL, FOLDER_MOVE_MSGS_ERROR);
	CHECK_RETURN_VAL_IF_FAIL(msg_list != NULL, FOLDER_MOVE_MSGS_ERROR);
	CHECK_RETURN_VAL_IF_FAIL(check_folder(src_folder) == TRUE, FOLDER_MOVE_MSGS_ERROR);
	CHECK_RETURN_VAL_IF_FAIL(check_folder(dest_folder) == TRUE, FOLDER_MOVE_MSGS_ERROR);

	for (cur = msg_list; cur != NULL; cur = cur->next) {
		if (folder_move_msg(src_folder, dest_folder, cur->data, NULL) != FOLDER_SUCCESS) {
			ret_val = FOLDER_MOVE_MSGS_ERROR;
		}
	}
	return ret_val;
}

/** 
  This function is used to copy a list of messages in a folder to another.
  @param src_folder    The source folder from where the List of Messages 
  has to be copied.
  @param dest_folder    The Destination folder to which the List of 
  Messages has to be Moved.
  @param msg_list    The List of MessageIDs of the Messages that 
  needs to be moved.This list has to be allocated and freed by the 
  calling function.
  @return FOLDER_SUCCESS(0) on success else any error code
*/
gint folder_copy_msgs(const gchar * src_folder, const gchar * dest_folder, GSList * msg_list)
{
	GSList *cur = NULL;
	gint ret_val = FOLDER_SUCCESS;

	CHECK_RETURN_VAL_IF_FAIL(src_folder != NULL, FOLDER_COPY_MSGS_ERROR);
	CHECK_RETURN_VAL_IF_FAIL(dest_folder != NULL, FOLDER_COPY_MSGS_ERROR);
	CHECK_RETURN_VAL_IF_FAIL(msg_list != NULL, FOLDER_COPY_MSGS_ERROR);
	CHECK_RETURN_VAL_IF_FAIL(check_folder(src_folder) == TRUE, FOLDER_COPY_MSGS_ERROR);
	CHECK_RETURN_VAL_IF_FAIL(check_folder(dest_folder) == TRUE, FOLDER_COPY_MSGS_ERROR);

	if (check_dest_folder(dest_folder) == TRUE) {
		return FOLDER_COPY_MSGS_ERROR;
	}
	for (cur = msg_list; cur != NULL; cur = cur->next) {
		if (folder_copy_msg(src_folder, dest_folder, cur->data, NULL) != FOLDER_SUCCESS) {
			ret_val = FOLDER_COPY_MSGS_ERROR;
		}
	}
	return ret_val;

}

/** 
  This function is used to parse a given message for the Message Headers
  @param file	Name of the message File
  @return pointer to the Message Information.The memory allocated for the 
  message Information has to be freed by the calling function. 
*/

MsgInfo *folder_parse_msg(const gchar * file)
{
	struct stat s;
	MsgInfo *msginfo = NULL;
	MsgFlags flags = {0, 0};

	CHECK_RETURN_VAL_IF_FAIL(file != NULL, NULL);
	flags.tmp_flags = 0;
	if (stat(file, &s) < 0) {
		return NULL;
	}

	if (!S_ISREG(s.st_mode)) {
		return NULL;
	}
	if (folder_check_partial(file) == FALSE) {
		return NULL;
	}
	msginfo = procheader_parse_file(file, flags, TRUE);
	if (msginfo == NULL) {
		return NULL;
	}
	msginfo->header->mtime = s.st_mtime;
	msginfo->header->size = folder_get_size(file);
	if (is_message_unread(folder_get_msgid_from_path(file))) {
                msginfo->header->flags.perm_flags |= MSG_UNREAD;
        }


	return msginfo;
}



/** 
  This Function is used to delete all Messages present in a 
  folder or To empty a specific Folder.
  @param folder    Name of the folder to be emptied.
  @return FOLDER_SUCCESS or error code 
*/
gint folder_remove_all_msg(const gchar * path)
{
	if (path == NULL) {
		return FOLDER_REMOVE_MSGS_ERROR;
	}

	if (remove_all_files(path) < 0) {
		return FOLDER_REMOVE_MSGS_ERROR;
	} else {
		return FOLDER_SUCCESS;
	}
}

/** 
  This function will be called by the function "folder_get_inbox_status"
  and it returns the inbox status of the local inbox also gets the latest 
  message number in the local inbox for pop recv mail
  @param folder     The folder that needs to scanned
  @return    FOLDER_SUCCESS(0) or FOLDER_SCAN_ERROR in case of error 
*/
gint folder_scan(Folder * folder)
{
	gchar *path = NULL;
	gchar *name = NULL;
	DIR *dp = NULL;
	struct dirent *d = NULL;
	gint max = 0;
	gint num = 0;
	gint n_msg = 0;

	CHECK_RETURN_VAL_IF_FAIL(folder != NULL, FOLDER_SCAN_ERROR);

	path = folder_get_path(folder);
	CHECK_RETURN_VAL_IF_FAIL(path != NULL, FOLDER_SCAN_ERROR);

	if (change_dir(path) < 0) {
		return FOLDER_SCAN_ERROR;
	}
	if ((dp = opendir(".")) == NULL) {
		return FOLDER_SCAN_ERROR;
	}
	d = readdir(dp);

	while (d != NULL) {
		if (check_msgid(d->d_name) == FALSE) {
			d = readdir(dp);
			continue;
		}

		if (dirent_is_regular_file(d)) {
			name = folder_get_msgid(d->d_name);
			num = get_msgid_num(name);
			if (num >= 0) {
				n_msg++;
				if (max < num) {
					max = num;
				}
			}
			g_free(name);
		}
		d = readdir(dp);
	}
	closedir(dp);
	if (n_msg == 0) {
		folder->new_msgs = folder->unread = folder->total = 0;
	} else {
		gint new_msgs = 0, unread = 0, total = 0, min = 0, max = 0;
		procmsg_get_mark_sum(folder, &new_msgs, &unread, &total, &min, &max, 0);
		if (n_msg > total) {
			new_msgs = n_msg - total;
			unread += n_msg - total;
		}
	}
	gl_last_msg_num = max;
	if ((gl_last_msg_num + 1) < 0) {
		gl_last_msg_num = 0;
	}
	return FOLDER_SUCCESS;
}

/** 
  This function is used to get the message file path of a particular message
  @param msgid    messageID of the message
  @return path of the message file,this has to be freed by the calling function 
*/
gchar *folder_get_message_file_path(const gchar * msgid)
{
	gchar *path = NULL, *file = NULL;
	Folder *folder = NULL;

	CHECK_RETURN_VAL_IF_FAIL(msgid != NULL, NULL);
	CHECK_RETURN_VAL_IF_FAIL(check_msgid(msgid) == TRUE, NULL);

	folder = (Folder *) folder_get_msg_folder(msgid);
	CHECK_RETURN_VAL_IF_FAIL(folder != NULL, NULL);

	path = folder_get_path(folder);
	CHECK_RETURN_VAL_IF_FAIL(path != NULL, NULL);

	if (is_dir_perm_exist(path, FALSE) == FALSE) {
		engine_errno = ENGINE_ERROR_NO_FOLDER_PERM;
		return NULL;
	}

	file = g_strdup_printf("%s%c%s", path, G_DIR_SEPARATOR, msgid);

	return file;
}

/** 
  This function is used to get the message folder
  @param msgid    messageID of the message
  @return Pointer to the folder object of the folder of message.
  Returned folder object should not be freed by the calling function 
*/
Folder *folder_get_msg_folder(const gchar * msgid)
{
	Folder *folder = NULL;
	gchar *dir = NULL;
	gchar *tmp = NULL;
	gchar *tmp1 = NULL;
	gchar *subdir = NULL;
	int string_len = 0;

	CHECK_RETURN_VAL_IF_FAIL(msgid != NULL, NULL);
	tmp = strchr(msgid, MSGID_SEPARATOR);
	if (tmp == NULL) {
		return NULL;
	}
	if (++tmp == NULL) {
		return NULL;
	}

	tmp1 = strrchr(tmp, MSGID_SEPARATOR);
	if (tmp1 == NULL) {
		return NULL;
	}

	string_len = tmp1 - tmp;
	dir = g_malloc0(string_len + 1);
	if (dir == NULL) {
		return NULL;
	}
	strncpy(dir, tmp, string_len);
	subdir = dir;
	while (subdir != NULL) {
		subdir = strchr(dir, MSGID_SEPARATOR);
		if (subdir != NULL) {
			*subdir = G_DIR_SEPARATOR;
		}
	}
	folder = (Folder *) folder_find_obj_by_name(dir);
	CHECK_RETURN_VAL_IF_FAIL(folder != NULL, NULL);

	free(dir);
	return folder;
}

/** 
  This Function is used to send the last index of the messages
  @param none
  @return	Last index of the Messages
*/
gint folder_get_last_msg_num()
{
	gint ret_val = gl_last_msg_num;
	gl_last_msg_num++;
	if ((gl_last_msg_num + 1) < 0) {
		gl_last_msg_num = 0;
	}
	return ret_val;
}

/** 
  This function is used to get the new message file name,given 
  account name destination folder and message number
  @param account_name	account_name of the current user
  @param destfolder	Destination folder, based on 
  which the file name is returned
  @param msgnum_ptr	OUT parameter,Message number pointer.i
  Message number is returned in this.This should not be freed by 
  the calling function.
  @return new message file name.Memory allocated by file name 
  should be freed by the calling function.
*/
gchar *folder_get_new_msg_file(const gchar * account_name,
			       const gchar * destfolder, gint * msgnum_ptr)
{
	gchar *dest_file = NULL;
	gchar *tmp_msgid = NULL;
	gint msg_num = 0;
	Folder *dest = NULL;

	CHECK_RETURN_VAL_IF_FAIL(account_name != NULL, NULL);
	CHECK_RETURN_VAL_IF_FAIL(destfolder != NULL, NULL);
	CHECK_RETURN_VAL_IF_FAIL(msgnum_ptr != NULL, NULL);

	dest = folder_find_obj_by_name(destfolder);
	CHECK_RETURN_VAL_IF_FAIL(dest != NULL, NULL);

	msg_num = folder_scan(dest);

	for (;;) {
		GET_MSG_ID(tmp_msgid, account_name, destfolder, msg_num + 1);
		dest_file = g_strdup_printf("%s%c%s", dest_file, G_DIR_SEPARATOR, tmp_msgid);
		if (is_file_entry_exist(dest_file)) {
			msg_num++;
			g_free(dest_file);
		} else {
			break;
		}
	}
	*msgnum_ptr = msg_num;
	return dest_file;
}

/**
 This functions returns the size of the file
 @param filename   name of the file
 @return size of the file
 */
static glong folder_get_size(const gchar * filename)
{
	struct stat s;
	glong ret = FOLDER_FAIL;
	gchar *msgid = NULL;

	if (get_file_extension(filename) != NULL) {
		msgid = folder_get_msgid(filename);
		ret = folder_get_all_size(msgid);
		g_free(msgid);
		return ret;
	}
	if (stat(filename, &s) < 0) {
		return FOLDER_FAIL;
	}
	return s.st_size;
}

/**
 This functions gets the msgid of the file
 @param filename    name of the file whose message id is required
 @return msgid    Message id of the file.This has to be freed 
 by the calling function.
 */
gchar *folder_get_msgid(const gchar * filename)
{
	gchar *t4 = NULL;
	gchar *msgid = NULL;
	gchar *id = NULL;

	if ((id = strrchr(filename, MSGID_SEPARATOR)) == NULL) {
		return NULL;
	}
	t4 = strchr(id, SECTION_SEPARATOR);
	if (t4 != NULL) {
		msgid = g_strndup(filename, t4 - (filename));
	} else {
		msgid = g_strdup(filename);
	}
	return msgid;
}

/** 
  This Function returns the list of message information,
  for the files in a folder 
  @param name    Name of the folder
  @param index   Index deciding which 10 message headers need to be returned
  @return List of Message Headers.
  This is a global list,which should not be freed by the calling function.
*/

GSList *folder_populate_folder(const gchar * folder_name)
{
	GSList *msglist = NULL;
	MsgInfo *msginfo = NULL;
	gchar *path = NULL;
	gchar *tmp_path = NULL;
	gchar *file_path = NULL;
	gboolean archive_flag = FALSE;
	GnomeVFSFileInfo *file_info = NULL;
	GList *list_file_info = NULL;
	GList *cur = NULL;
	Folder *folder = NULL;
	GnomeVFSResult vfs_result = GNOME_VFS_OK;


	if (folder_name == NULL) {
		return NULL;
	}

	if (g_ascii_strncasecmp(folder_name, ARCHIVE, strlen(ARCHIVE)) == 0) {

		if (CHECK_MMC_FLAG == TRUE) {
			debug_print(" %s %s mmc is enabled - folder name%s\n",
				    __FILE__, __FUNCTION__, folder_name);

			path = g_strdup_printf("%s%c%s", gl_volume_mounted_path,
					       G_DIR_SEPARATOR, R_ARCHIVE);
			archive_flag = TRUE;
		} else {
			debug_print(" %s %s mmc is disabled - folder name %s \n",
				    __FILE__, __FUNCTION__, folder_name);
			return NULL;
		}
	} else {

		folder = folder_find_obj_by_name(folder_name);
		if (folder != NULL) {
			tmp_path = folder_get_path(folder);
		}
		if (tmp_path != NULL) {
			path = g_strdup(tmp_path);
		}

	}
	if (path == NULL) {
		return NULL;
	}
	vfs_result = gnome_vfs_directory_list_load(&list_file_info, path,
						   GNOME_VFS_FILE_INFO_DEFAULT);
	if (vfs_result != GNOME_VFS_OK) {
		g_free(path);
		return NULL;
	}
	for (cur = list_file_info; cur != NULL; cur = cur->next) {
		file_info = (GnomeVFSFileInfo *) cur->data;
		if (file_info->type == GNOME_VFS_FILE_TYPE_REGULAR) {

			file_path = g_strdup_printf("%s%c%s", path,
						    G_DIR_SEPARATOR, file_info->name);
			msginfo = folder_get_info(file_info->name, file_path, archive_flag);
			if (msginfo != NULL) {

				msglist = g_slist_append(msglist, msginfo->header);
				g_free(msginfo->attach_list);
				g_free(msginfo->settings);
				g_free(msginfo);
			}
			/* free the file path */
			g_free(file_path);
		}
	}

	g_free(path);
	return msglist;
}

/** 
  This Function returns the list of files for a message 
  @param name	Name of the folder
  @param msgid	  message id of the file for which set of files are required
  @return List of Message files related to the msgid.
  This list has to be freed by the calling function.
*/

GSList *folder_get_msgid_files(const gchar * name, const gchar * msgid)
{
	GSList *msg_file_list = NULL;
	DIR *dp = NULL;
	struct dirent *d = NULL;
	gchar *path = NULL;
	gchar *t1 = NULL;
	gchar *t2 = NULL;
	gchar *file_name = NULL;
	Folder *folder = NULL;

	CHECK_RETURN_VAL_IF_FAIL(name != NULL, NULL);

	folder = folder_find_obj_by_name(name);
	CHECK_RETURN_VAL_IF_FAIL(folder != NULL, NULL);

	path = folder_get_path(folder);
	CHECK_RETURN_VAL_IF_FAIL(path != NULL, NULL);
	if (change_dir(path) < 0) {
		return NULL;
	}
	if ((dp = opendir(".")) == NULL) {
		return NULL;
	}
	d = readdir(dp);
	while (d != NULL) {
		t1 = NULL;
		t2 = NULL;
		if (check_msgid(d->d_name) == FALSE) {
			d = readdir(dp);
			continue;
		}
		if (dirent_is_regular_file(d)) {
			if (msgid == NULL) {
				msg_file_list = g_slist_append(msg_file_list, g_strdup(d->d_name));
			} else {
				t1 = strstr(d->d_name, msgid);
				if ((t1 != NULL)
				    && (d->d_name[strlen(msgid)] == SECTION_SEPARATOR)) {
					file_name = g_strdup(d->d_name);
					msg_file_list = g_slist_append(msg_file_list, file_name);
				} else if (strcmp(d->d_name, msgid) == 0) {
					file_name = g_strdup(d->d_name);
					msg_file_list = g_slist_append(msg_file_list, file_name);
				}
			}
		}
		d = readdir(dp);
	}
	closedir(dp);

	return msg_file_list;
}

/** 
  This Function returns the list of all part files for a message 
  @param name	Name of the folder
  @param msgid	for message id
  @param body_download_flag    This is OUT parameter,which will indicate
  whether mime header( using body strcuture command) is downloaded 
  for a message id 
  @return List of all Message files,to be freed by the calling function
*/

GSList *folder_get_msgid_all_files(const gchar * name, const gchar * msgid,
				   gboolean * body_download_flag, gboolean add_mhdr)
{
	GSList *msg_file_list = NULL;
	DIR *dp = NULL;
	struct dirent *d = NULL;
	gchar *path = NULL, *t1 = NULL, *file_name = NULL;
	Folder *folder = NULL;
	gchar *tmp = NULL;
	CHECK_RETURN_VAL_IF_FAIL(msgid != NULL, NULL);

	folder = folder_find_obj_by_name(name);
	CHECK_RETURN_VAL_IF_FAIL(folder != NULL, NULL);

	path = folder_get_path(folder);
	CHECK_RETURN_VAL_IF_FAIL(path != NULL, NULL);

	if (change_dir(path) < 0) {
		return NULL;
	}

	if ((dp = opendir(".")) == NULL) {
		return NULL;
	}

	d = readdir(dp);

	while (d != NULL) {
		t1 = NULL;
		if (check_msgid(d->d_name) == FALSE) {
			d = readdir(dp);
			continue;
		}
		if (dirent_is_regular_file(d)) {
			t1 = strstr(d->d_name, msgid);
			if ((t1 != NULL) && (d->d_name[strlen(msgid)] == SECTION_SEPARATOR)) {
				file_name = g_strdup(d->d_name);
				tmp = strstr(t1, MIME_SUFFIX);
				if (tmp == NULL) {
					if (body_download_flag != NULL) {
						*body_download_flag = TRUE;
					}
				}

				if ((add_mhdr == TRUE) || (tmp == NULL)) {
					msg_file_list = g_slist_append(msg_file_list, file_name);
				}
			} else if (strcmp(d->d_name, msgid) == 0) {
				file_name = g_strdup(d->d_name);
				msg_file_list = g_slist_append(msg_file_list, file_name);
			}
		}
		d = readdir(dp);
	}
	closedir(dp);

	return msg_file_list;
}

/**
  This function returns the file name corresponding to the message msgid
  @param folder    folder in which message is stored
  @param msgid    msgid of the message
  @return file name,this has to be freed by the calling function 
*/

gchar *folder_fetch_msg(const Folder * folder, const gchar * msgid)
{
	gchar *path = NULL;
	gchar *file = NULL;

	CHECK_RETURN_VAL_IF_FAIL(folder != NULL, NULL);
	CHECK_RETURN_VAL_IF_FAIL(msgid != NULL, NULL);

	path = folder_get_path(folder);
	file = g_strconcat(path, G_DIR_SEPARATOR_S, msgid, NULL);
	if (file == NULL) {
		return NULL;
	}
	if (is_file_exist(file) == FALSE) {
		g_free(file);
		return NULL;
	}
	return file;
}

/**
  This function renames the file containing the message
  @param folder    folder in which message is stored
  @param new_name    new name of the folder
  @return returns FOLDER_SUCCESS on successfully renaming else folder error code

*/

gint folder_rename_files(const Folder * folder, const gchar * new_name)
{
	gchar *path = NULL;
	gchar *new_file_name = NULL;
	DIR *dp = NULL;
	struct dirent *d = NULL;
	Folder *sub_folder = NULL;
	gchar *sub_dir_name = NULL;
	gchar *old_name = NULL;
	gchar *old_sub_name = NULL;
	gchar *src_file = NULL;
	gchar *dest_file = NULL;
	gint ret_val = FOLDER_SUCCESS;

	CHECK_RETURN_VAL_IF_FAIL(folder != NULL, FALSE);

	path = folder_get_path(folder);

	CHECK_RETURN_VAL_IF_FAIL(path != NULL, FALSE);

	if ((dp = opendir(path)) == NULL) {
		return FOLDER_FAIL;
	}
	old_name = folder->name;
	if (folder->name == NULL) {
		return FOLDER_FAIL;
	}
	while ((d = readdir(dp)) != NULL) {

		if (dirent_is_directory(d) == TRUE) {

			if (check_folder(d->d_name) == FALSE) {
				continue;
			}

			old_sub_name = g_strdup_printf("%s%c%s", old_name,
						       G_DIR_SEPARATOR, d->d_name);


			sub_folder = folder_find_obj_by_name(old_sub_name);
			g_free(old_sub_name);

			if (sub_folder == NULL) {
				return FOLDER_FAIL;
			}
			sub_dir_name = g_strdup_printf("%s%c%s", new_name,
						       G_DIR_SEPARATOR, d->d_name);
			/* rename the files in sub directories also */
			if (folder_rename_files(sub_folder, sub_dir_name) == FOLDER_FAIL) {
				g_free(sub_dir_name);
				return FOLDER_FAIL;
			}
			/* Update the folder info afterthat */
			ret_val = folder_update_folder(sub_folder, sub_dir_name);
			g_free(sub_dir_name);
			if (ret_val == FOLDER_FAIL) {
				return FOLDER_FAIL;
			}
		}
		if (check_msgid(d->d_name) == FALSE) {
			continue;
		}
		new_file_name = folder_create_msg_id(d->d_name, new_name, FALSE);
		if (new_file_name == NULL) {
			return FOLDER_FAIL;
		}
		src_file = g_strdup_printf("%s%c%s", path, G_DIR_SEPARATOR, d->d_name);
		dest_file = g_strdup_printf("%s%c%s", path, G_DIR_SEPARATOR, new_file_name);
		if (rename(src_file, dest_file) < 0) {
			debug_print("file %s cannot be renamed \n", d->d_name);
		}
		if (is_message_unread(d->d_name) == TRUE) {
			folder_message_del_list(d->d_name, UNREAD);
			folder_message_add_list(new_file_name, UNREAD);
		}
		g_free(src_file);
		g_free(dest_file);
		g_free(new_file_name);
	}
	closedir(dp);
	return FOLDER_SUCCESS;

}


/** 
  This function adds a new folder in to the global folder list
  @param folder    new folder name to be added.
  This should not be freed by the calling function.
  @return global folder list,this should not freed by the calling function 
*/

GList *folder_add(Folder * folder)
{

	CHECK_RETURN_VAL_IF_FAIL(folder != NULL, NULL);

	gl_folder_list = g_list_append(gl_folder_list, folder);
	if (gl_folder_list == NULL) {
		return NULL;
	}
	return (GList *) gl_folder_list;
}

/**
  This function gets the mark file
  @param folder pointer to folder item
  @return file path.This has to be freed by the calling function.
*/
gchar *folder_get_mark_file(const Folder * item)
{
	gchar *path = NULL;
	gchar *file = NULL;

	CHECK_RETURN_VAL_IF_FAIL(item != NULL, NULL);
	CHECK_RETURN_VAL_IF_FAIL((gchar *) item->path != NULL, NULL);

	path = folder_get_path(item);

	CHECK_RETURN_VAL_IF_FAIL(path != NULL, NULL);
	if (!is_dir_exist(path)) {
		make_dir_hier(path);
	}

	file = g_strdup_printf("%s%c%s", path, G_DIR_SEPARATOR, MARK_FILE);

	return file;
}

/**
  This function intializes the folder lists for storing unread messages,i
  remote messages etc.
  @param none
  @return status(TRUE or FALSE).returns TRUE on successfully reading 
  from message list files and initializing the lists
 */
gboolean folder_init_lists(void)
{

	/* update the unread list first,so that unread mails can be counted */
	if (folder_message_read_list(UNREAD) != FOLDER_SUCCESS) {
		return FALSE;
	}
	if (folder_message_read_list(SEEN) != FOLDER_SUCCESS) {
		return FALSE;
	}
	gl_folder_list = populate_folder_list();
	if (gl_folder_list == NULL) {
		return FALSE;
	}
	if (folder_read_list(FALSE) != FOLDER_SUCCESS) {
		return FALSE;
	}
	if (folder_read_list(TRUE) != FOLDER_SUCCESS) {
		return FALSE;
	}
	return TRUE;
}

/** 
  This function intializes the gl_folder_list(global folder list),
  creates standard directories etc.
  @param none
  @return (FOLDER_SUCCESS)success or error code
*/
gint init_folder(void)
{
	gchar *dir = NULL;
	gchar *root = NULL;

	dir = g_strdup_printf("%s%c%s", STANDARD_PATH, G_DIR_SEPARATOR, APPLN_PATH);
	if (folder_create_dir_if_not_exist(dir, FALSE) == FOLDER_FAIL) {
		g_free(dir);
		return FOLDER_INITIALIZATION_ERROR;
	}
	g_free(dir);
	dir = g_strdup_printf("%s%c%s%c%s", STANDARD_PATH, G_DIR_SEPARATOR,
			      APPLN_PATH, G_DIR_SEPARATOR, EMAIL_PATH);
	if (folder_create_dir_if_not_exist(dir, FALSE) == FOLDER_FAIL) {
		g_free(dir);
		return FOLDER_INITIALIZATION_ERROR;
	}
	if (folder_create_standard_folder(dir) == FOLDER_INITIALIZATION_ERROR) {
		g_free(dir);
		return FOLDER_INITIALIZATION_ERROR;
	}
	g_free(dir);

	dir = g_strdup_printf("%s%s", get_home_dir(), PATH_OSSO);
	if (folder_create_dir_if_not_exist(dir, FALSE) == FOLDER_FAIL) {
		g_free(dir);
		return FOLDER_INITIALIZATION_ERROR;
	}
	g_free(dir);

	root = g_strdup_printf("%s%s", get_home_dir(), CONFIG_FILE_PATH);

	if (folder_restore_old_settings(root) == FOLDER_FAIL) {
		g_free(root);
		return FOLDER_INITIALIZATION_ERROR;

	}
	g_free(root);

	dir = g_strdup_printf("%s%c%s%c%s%c%s", STANDARD_PATH, G_DIR_SEPARATOR,
			      APPLN_PATH, G_DIR_SEPARATOR, EMAIL_PATH, G_DIR_SEPARATOR, TMP_DIR);
	remove_dir_recursive(dir);
	if (folder_create_dir_if_not_exist(dir, TRUE) == FOLDER_FAIL) {
		return FOLDER_INITIALIZATION_ERROR;
	}
	g_free(dir);

	dir = g_strdup_printf("%s%c%s%c%s%c%s", STANDARD_PATH, G_DIR_SEPARATOR,
			      APPLN_PATH, G_DIR_SEPARATOR, EMAIL_PATH, G_DIR_SEPARATOR,
			      MIME_TMP_DIR);
	remove_dir_recursive(dir);
	if (folder_create_dir_if_not_exist(dir, TRUE) == FOLDER_FAIL) {
		return FOLDER_INITIALIZATION_ERROR;
	}
	g_free(dir);

	if (folder_create_dir_if_not_exist(VAR_PATH, TRUE) == FOLDER_FAIL) {
		return FOLDER_INITIALIZATION_ERROR;
	}

	dir = g_strdup_printf("%s%c%s", VAR_PATH, G_DIR_SEPARATOR, EMAIL_PATH);
	if (folder_create_dir_if_not_exist(dir, TRUE) == FOLDER_FAIL) {
		return FOLDER_INITIALIZATION_ERROR;
	}
	g_free(dir);


	return FOLDER_SUCCESS;
}

/**
  This function adds  the message id to  the gl_message_list list 
  @param msgid    message id of the unread message,to add to the list
  @return FOLDER_SUCCESS on successfully adding,else folder error 
*/
gint folder_message_add_list(const gchar * msgid, ListType type)
{
	gchar *folder_name = NULL;
	gchar *add_msgid = NULL;

	if (type == UNREAD) {
		if (is_message_unread(msgid) == FALSE) {
			add_msgid = g_strdup(msgid);
			gl_message_list = g_slist_append(gl_message_list, add_msgid);
			if (!gl_message_hash)
				gl_message_hash = g_hash_table_new_full(g_str_hash,
									g_str_equal, NULL, NULL);
			g_hash_table_insert(gl_message_hash, add_msgid, GINT_TO_POINTER(1));
			if (gl_message_list == NULL) {
				return FOLDER_MESSAGE_LIST_WRITE_ERROR;
			}

			return folder_write_list(type, msgid);
		}

	} else if (type == SEEN) {
		if (is_message_seen(msgid) == FALSE) {
			folder_name = get_fold_name(msgid);
			if (folder_name != NULL) {
				if (g_strncasecmp(folder_name, INBOX_DIR,
						  MAX_STR_LEN(folder_name, INBOX_DIR)) == 0) {
					gl_msg_seen_list =
					    g_slist_append(gl_msg_seen_list, g_strdup(msgid));
				}
			}
			if (gl_msg_seen_list == NULL) {
				return FOLDER_MESSAGE_LIST_WRITE_ERROR;
			}
			return folder_write_list(type, msgid);
		}
	}

	return FOLDER_SUCCESS;
}

/**
  This function checks if the message id is unread
  @param msgid    message id(filename),for which status has to be checked 
  @return TRUE if the message is unread, FALSE status otherwise
*/
gboolean is_message_unread(const gchar * msgid)
{
	if (msgid == NULL) {
		return FALSE;
	}

	if (!gl_message_hash)
		return FALSE;
	return (g_hash_table_lookup(gl_message_hash, msgid) != NULL);
}

/**This function checks if the message id is Seen
   @param msgid    message id(filename),for which status has to be checked
   @return TRUE if the message is unread, FALSE status otherwise
 */
gboolean is_message_seen(const gchar * msgid)
{
	GSList *temp = NULL;
	gboolean flag = FALSE;
	for (temp = gl_msg_seen_list; temp != NULL; temp = temp->next) {
		if (strcmp((gchar *) temp->data, msgid) == 0) {
			flag = TRUE;
			break;
		}
	}
	return flag;
}

/**
  This function deletes the message id from  the gl_message_list 
  @param msgid    message id
  @return TRUE if successfully deleted,FALSE otherwise 
*/
gint folder_message_del_list(const gchar * msgid, ListType type)
{
	GSList *tmp = NULL;
	gchar *dir = NULL;
	Folder *folder = NULL;
	GList *folder_list = NULL;
	if (type == UNREAD) {
		if (is_message_unread(msgid) == FALSE) {
			return FOLDER_SUCCESS;
		}
		if (gl_message_list == NULL) {
			return FOLDER_MESSAGE_LIST_WRITE_ERROR;
		} else {
			/* remove from hash */
			if (gl_message_hash)
				g_hash_table_remove(gl_message_hash, msgid);
			dir = get_fold_name(msgid);
			folder = folder_find_obj_by_name(dir);
			for (tmp = gl_message_list; tmp != NULL; tmp = tmp->next) {
				if ((tmp->data != NULL)
				    && (strcmp((gchar *) tmp->data, msgid) == 0)) {
					g_free(tmp->data);
					gl_message_list = g_slist_delete_link(gl_message_list, tmp);

					if (folder != NULL) {
						folder->unread--;
						/* touch the folder to reflect the current time stamp
						 * so that the cache file can get re-generated
						 * for the folder in which an unread mail has been 
						 * changed to read mail
						 */
						folder_list = g_list_append(folder_list, folder);
						folder_change_time(folder_list);
						folder_list = g_list_remove(folder_list, folder);
						g_list_free(folder_list);
					}
					break;
				}
			}
		}
	} else if (type == SEEN) {
		if (is_message_seen(msgid) == FALSE) {
			return FOLDER_SUCCESS;
		}
		if (gl_msg_seen_list == NULL) {
			return FOLDER_MESSAGE_LIST_WRITE_ERROR;
		} else {
			for (tmp = gl_msg_seen_list; tmp != NULL; tmp = tmp->next) {
				if ((tmp->data != NULL)
				    && (strcmp((gchar *) tmp->data, msgid) == 0)) {
					g_free(tmp->data);
					gl_msg_seen_list =
					    g_slist_delete_link(gl_msg_seen_list, tmp);
					break;
				}
			}
		}
	}
	folder_write_list(type, NULL);
	return FOLDER_SUCCESS;
}

/**
  This function adds the message id to  the remote or delete list 
  @param msgid    message id of the message,to be added to the list
  @param remote_listflag   this indicates which list to use.
  This is TRUE if remote message list(gl_msg_rem_list) has to be used. 
  @return FOLDER_SUCCESS if successfully added 
*/
gint folder_add_list(const gchar * msgid, gboolean remote_listflag)
{
	if (remote_listflag == TRUE) {
		if (folder_is_msg_remote(msgid) == FALSE) {
			gl_msg_rem_list = g_slist_append(gl_msg_rem_list, g_strdup(msgid));
			return folder_write_list(REMOTE, msgid);
		}
	} else {
		if (folder_is_msg_deleted(msgid) == FALSE) {
			gl_msg_del_list = g_slist_append(gl_msg_del_list, g_strdup(msgid));
			return folder_write_list(DELETE, msgid);
		}
	}

	return FOLDER_SUCCESS;
}

/**
  This function checks the messge id exists in the delete list or not
  @param msgid message id of the file
  @return TRUE if message is found in the list 
*/
gboolean folder_is_msg_deleted(const gchar * msgid)
{
	GSList *temp = NULL;
	gboolean flag = FALSE;
	for (temp = gl_msg_del_list; temp != NULL; temp = temp->next) {
		if (strcmp((gchar *) temp->data, msgid) == 0) {
			flag = TRUE;
			break;
		}
	}
	return flag;
}

/** 
  This function prepares the folder list,by calling 
  the function recursively for sub folders
 
   @param path    pointer to the folder path 
   @param root_folder    pointer to the root folder,
   in which the folder exists 
   @param rootlist   the list of root folder,to append to 
   @param root_obj folder object of the root folder
   @return List of folders.This list has to be freed by the caller
*/
static GList *folder_prepare_list(const gchar * path, const gchar * root_folder,
				  GList * rootlist, Folder * root_obj)
{
	Folder *folder = NULL;
	DIR *dp = NULL;
	struct dirent *d = NULL;
	gchar *file_path = NULL;
	struct stat s;
	gchar *folder_name = NULL;
	GList *foldlist = NULL;

	foldlist = rootlist;
	if ((dp = opendir(path)) == NULL) {
		return foldlist;
	}
	d = readdir(dp);

	while (d != NULL) {
		if (check_folder(d->d_name) == FALSE) {
			file_path = g_strdup_printf("%s%c%s", path, G_DIR_SEPARATOR, d->d_name);
			folder_update_count(root_obj, d->d_name, file_path);
			g_free(file_path);
			d = readdir(dp);
			continue;
		}
		/* if the MMC is unmounted,donot add archival folder to the list */
		if ((CHECK_MMC_FLAG == FALSE)
		    && (strcmp(d->d_name, ARCHIVE) == 0)) {
			debug_print("\nfolder populate:MMC is unmounted\n");
			d = readdir(dp);
			continue;
		}
		file_path = g_strdup_printf("%s%c%s", path, G_DIR_SEPARATOR, d->d_name);
		if (stat(file_path, &s) < 0) {
			g_free(file_path);
			debug_print("stat failed \n");
			return foldlist;
		}
                /* skip if not a directory or not readable */
		if (S_ISDIR(s.st_mode) == FALSE || !(s.st_mode & S_IRUSR) ||
                                !(s.st_mode & S_IXUSR)) {
			g_free(file_path);
			d = readdir(dp);
			continue;
		}
		if (root_folder == NULL) {

			folder_name = g_strdup(d->d_name);

		} else {

			folder_name =
			    g_strdup_printf("%s%c%s", root_folder, G_DIR_SEPARATOR, d->d_name);
		}

		folder = folder_new(folder_name, file_path, F_LOCAL);
		if (folder != NULL) {
			foldlist = g_list_append(foldlist, folder);
		}
		foldlist = folder_prepare_list(file_path, folder_name, foldlist, folder);

		g_free(folder_name);
		g_free(file_path);
		d = readdir(dp);
	}
	closedir(dp);
	return foldlist;
}

/**
  This function checks the messge is remote or not
  @param msgid    message id
  @return TRUE if the message is remote also 
*/
gboolean folder_is_msg_remote(const gchar * msgid)
{
	GSList *temp = NULL;

	for (temp = gl_msg_rem_list; temp != NULL; temp = temp->next) {
		if (strcmp((gchar *) temp->data, msgid) == 0) {
			return TRUE;
		}
	}
	return FALSE;
}

/**
  This function returns list of queued message ids  
  @param account_name    the account, for which the list has to be filtered
  @return list of msgids.This list has to be freed by the caller
*/
GSList *folder_get_queued_msgid_list(const gchar * account_name, ListType type)
{
	GSList *temp = NULL;
	GSList *prev = NULL;
	GSList *ret_list = NULL;
	gchar *msgid = NULL;
	gchar *tmp_account_name = NULL;
	if (type == DELETE) {
		temp = gl_msg_del_list;
	} else if (type == SEEN) {
		temp = gl_msg_seen_list;
	}


	while (temp != NULL) {
		msgid = (gchar *) temp->data;
		if (check_msgid(msgid) == FALSE) {
			prev = temp;
			temp = temp->next;
			g_free(msgid);
			if (type == DELETE) {
				gl_msg_del_list = g_slist_delete_link(gl_msg_del_list, prev);
			} else if (type == SEEN) {
				gl_msg_seen_list = g_slist_delete_link(gl_msg_seen_list, prev);
			}

			continue;
		}
		tmp_account_name = get_account_name(msgid);
		if (strcmp(tmp_account_name, account_name) == 0) {
			ret_list = g_slist_append(ret_list, g_strdup(msgid));
		}
		temp = temp->next;
	}
	return ret_list;
}

/**
  This function deletes the message id from the global delete/remote list
  @param msgid    message id of the file,to be deleted from the list
  @param remote_listflag    this flag indicates which list to use. 
  @return FOLDER_SUCCESS(0) if successfully deleted else error code 
*/
gint folder_delete_list_msg(const gchar * msgid, gboolean remote_listflag)
{
	GSList *tmp = NULL;
	GSList *cur_list = NULL;

	if (remote_listflag == TRUE) {
		cur_list = gl_msg_rem_list;
	} else {
		cur_list = gl_msg_del_list;
	}
	if (cur_list == NULL) {
		return FOLDER_MESSAGE_LIST_WRITE_ERROR;
	} else {
		for (tmp = cur_list; tmp != NULL; tmp = tmp->next) {
			if ((tmp->data != NULL)
			    && (strcmp((gchar *) tmp->data, msgid) == 0)) {
				cur_list = g_slist_remove(cur_list, tmp->data);
				break;
			}
		}
	}
	if (remote_listflag == TRUE) {
		gl_msg_rem_list = cur_list;
	} else {
		gl_msg_del_list = cur_list;
	}
	return folder_write_list(remote_listflag ? REMOTE : DELETE, NULL);

}

/**
 This function create the directory if not exists
 @param dir    path of the directory
 @param all_perms   whether permissions are to be given for all users 
 @return success(FOLDER_SUCCESS) or folder error 
*/
static gint folder_create_dir_if_not_exist(const gchar * dir, gboolean all_perms)
{
	struct stat s;
	gint mode = 0;

	if (all_perms == TRUE) {
		mode = S_IRWXU | S_IRWXG | S_IRWXO;
	} else {
		mode = S_IRWXU;
	}
	if (is_dir_exist(dir) == FALSE) {
		if (make_dir(dir) < 0) {
			return FOLDER_FAIL;
		}
		if (chmod(dir, mode) < 0) {
			return FOLDER_FAIL;
		}
		return FOLDER_SUCCESS;
	}
	if (stat(dir, &s) < 0) {
		return FOLDER_FAIL;
	} else {
		if (s.st_mode & S_IRWXU) {
			return FOLDER_SUCCESS;
		} else {
			return FOLDER_FAIL;
		}
	}
}

/**
 This function returns the global folder list
 @param none
 @return global list of folders.This should not be freed 
*/
GList *folder_get_list(void)
{
	return gl_folder_list;
}

/** 
   This function validates the msgid format
   @param msgid    pointer to the msgid
   @return TRUE on valid message id
           FALSE on invalid message id  
*/
gboolean check_msgid(const gchar * msgid)
{
	gchar *tmp = NULL, *tmp1 = NULL;
	if (msgid == NULL) {
		return FALSE;
	}
	tmp = strchr(msgid, MSGID_SEPARATOR);
	if (tmp == NULL) {
		return FALSE;
	}
	tmp1 = strchr(tmp + 1, MSGID_SEPARATOR);
	if (tmp1 == NULL) {
		return FALSE;
	}
	return TRUE;
}

/** 
  This function validates the folder name 
  @param folder    pointer to the folder name
  @return TRUE on valid folder name 
           FALSE on invalid folder name  
*/
static gboolean check_folder(const gchar * folder)
{
	gchar *tmp = NULL;
	gchar *tmp1 = NULL;

	tmp = strchr(folder, MSGID_SEPARATOR);
	if (tmp != NULL) {
		return FALSE;
	} else {
		tmp1 = strchr(folder, SECTION_SEPARATOR);
		if (tmp1 != NULL) {
			return FALSE;
		} else {
			return TRUE;
		}
	}
}




/** 
  This function is to delete the particular message attachment
  @param msgid    message id
  @param section_number    section number of the message attachment
  @return FOLDER_SUCCESS on success & error code on failure
*/
gint folder_remove_attachment(const gchar * msgid, const gchar * section_number)
{
	gchar *encoded_file = NULL, *decoded_file = NULL;
	CHECK_RETURN_VAL_IF_FAIL(msgid != NULL, FOLDER_ATTACHMENT_REMOVE_ERROR);
	CHECK_RETURN_VAL_IF_FAIL(section_number != NULL, FOLDER_ATTACHMENT_REMOVE_ERROR);

	encoded_file = g_strdup_printf("%s%c%s%c%s%c%s", get_mail_dir(),
				       G_DIR_SEPARATOR,
				       INBOX_DIR, G_DIR_SEPARATOR, msgid,
				       SECTION_SEPARATOR, section_number);

	decoded_file = g_strdup_printf("%s%c%s%c%s%c%s", get_mime_tmp_dir(),
				       G_DIR_SEPARATOR, msgid,
				       SECTION_SEPARATOR, section_number,
				       SECTION_SEPARATOR, DEC_SUFFIX);

	debug_print("encoded_file : %s\n", encoded_file);
	debug_print("decoded_file : %s\n", decoded_file);
	if (is_file_exist(encoded_file) == FALSE && is_file_exist(decoded_file) == FALSE) {
		g_free(encoded_file);
		g_free(decoded_file);
		return FOLDER_ATTACHMENT_NOT_DOWNLOADED;
	}
	if (is_file_exist(encoded_file) == TRUE) {
		if (rem_file(encoded_file) != FOLDER_SUCCESS) {
			g_free(encoded_file);
			g_free(decoded_file);
			return FOLDER_ATTACHMENT_REMOVE_ERROR;
		}
		g_free(encoded_file);
	}
	if (is_file_exist(decoded_file) == TRUE) {
		if (rem_file(decoded_file) != FOLDER_SUCCESS) {
			g_free(decoded_file);
			return FOLDER_ATTACHMENT_REMOVE_ERROR;
		}
		g_free(decoded_file);
	}
	return FOLDER_SUCCESS;
}

/**
  This function removes the file
  @param file    file path
  @return FOLDER_SUCCESS if successfully removed 
 */
static gint rem_file(const gchar * file)
{
	struct stat s;

	if (stat(file, &s) < 0) {
		return FOLDER_ATTACHMENT_REMOVE_ERROR;
	}
	if ((s.st_mode & S_IWUSR) == 0) {
		return FOLDER_ATTACHMENT_REMOVE_ERROR;
	}
	if (unlink(file) < 0) {
		return FOLDER_ATTACHMENT_REMOVE_ERROR;
	}
	return FOLDER_SUCCESS;
}


/** 
  This Function populates the folder information while startup
  @param none
  @return global List of folders.
  This list should not be freed by the caller 
*/

GList *populate_folder_list(void)
{
	GList *foldlist = NULL;
	const gchar *path = NULL;

	path = get_mail_dir();
	CHECK_RETURN_VAL_IF_FAIL(path != NULL, NULL);
	foldlist = folder_prepare_list(path, NULL, foldlist, NULL);
	return foldlist;
}

/**
  This function is used to clean up the global data lists used in folder
  @param none
  @return none
*/
void folder_cleanup(void)
{
	GList *flist = gl_folder_list;
	GSList *mlist = gl_message_list;
	GSList *mdlist = gl_msg_del_list;

	while (gl_folder_list != NULL) {
		Folder *folder = (Folder *) gl_folder_list->data;
		g_free(folder->name);
		g_free(folder->path);
		g_free(folder);
		gl_folder_list = gl_folder_list->next;
	}
	g_list_free(flist);
	while (gl_message_list != NULL) {
		g_free(gl_message_list->data);
		gl_message_list = gl_message_list->next;
	}

	if (gl_message_hash)
		g_hash_table_destroy(gl_message_hash);

	g_slist_free(mlist);
	while (gl_msg_del_list != NULL) {
		g_free(gl_msg_del_list->data);
		gl_msg_del_list = gl_msg_del_list->next;
	}
	g_slist_free(mdlist);
	if (gl_msg_rem_list != NULL) {
		slist_free_strings(gl_msg_rem_list);
		g_slist_free(gl_msg_rem_list);
	}

	g_free(gl_unread_list_path);
	g_free(gl_seen_list_path);
	g_free(gl_remote_list_path);
	g_free(gl_delete_list_path);

	gl_unread_list_path = NULL;
	gl_seen_list_path = NULL;
	gl_remote_list_path = NULL;
	gl_delete_list_path = NULL;

	gl_folder_list = NULL;
	gl_message_list = NULL;
	gl_msg_del_list = NULL;
	gl_message_hash = NULL;
}

/** 
  This function checks the folder permissions 
   @param name    pointer to the folder name
   @param check_write_perms    flag to indicate whether write 
   permissions to be checked 
   @return TRUE on folder having folder permissions 
           FALSE on invalid folder permissions  
*/
gboolean check_folder_perms(const gchar * name, gboolean check_write_perms)
{
	gchar *path = NULL;
	gboolean retval = FALSE;
	Folder *folder = NULL;

	if (name == NULL) {
		return FALSE;
	}

	folder = folder_find_obj_by_name(name);

	if (folder == NULL) {
		return FALSE;
	}
	path = folder_get_path(folder);
	if (path == NULL) {
		return FALSE;
	}

	retval = is_dir_perm_exist(path, check_write_perms);
	return retval;

}

/**
  This function prepares the new msgid for copy or move operation
  @param account    account name
  @param dest_folder    folder name
  @param msgnum    message number
  @param newmsg_flag    indicates whether message id has to be created
			with new number 
  @return newmsg_flag message id of the message for destination folder.
  This message id has to be freed by the caller. 
*/
static gchar *folder_create_msg_id(const gchar * msgid, const gchar * dest_folder,
				   gboolean newmsg_flag)
{
	gchar *new_msgid = NULL;
	gchar *account = NULL;
	gchar *tmp = NULL;
	gchar *folder_name = NULL;
	gchar *file_path = NULL;
	Folder *folder = NULL;
	gint num = 0;

	account = get_account_name(msgid);

	folder_name = g_strdup(dest_folder);

	tmp = folder_name;

	while (tmp != NULL) {
		tmp = strchr(tmp, G_DIR_SEPARATOR);
		if (tmp != NULL) {
			*tmp = MSGID_SEPARATOR;
		}
	}
	if (newmsg_flag == FALSE) {
		num = get_msgid_num(msgid);
		new_msgid =
		    g_strdup_printf("%s%c%s%c%u", account, MSGID_SEPARATOR,
				    folder_name, MSGID_SEPARATOR, num);
		g_free(folder_name);
		g_free(account);
		return new_msgid;
	}

	folder = (Folder *) folder_find_obj_by_name(dest_folder);
	if (!folder ) {
		osso_log(LOG_INFO, "oops...folder does not exists\n");
		g_free(folder_name);
		g_free(account);
		return NULL;
	}

	num = folder->max;

	while (num != (++folder->max)) {
		if (strcmp(folder_name, INBOX_DIR) == 0) {
			new_msgid =
		    		g_strdup_printf("%s>%c%s%c%lu", account, MSGID_SEPARATOR,
				    		folder_name, MSGID_SEPARATOR, folder->max);
		}else { 
			new_msgid =
		   		 g_strdup_printf("%s%c%s%c%lu", account, MSGID_SEPARATOR,
				    		folder_name, MSGID_SEPARATOR, folder->max);
		}
		file_path = folder_get_message_file_path(new_msgid);

		if (is_file_entry_exist(file_path) == FALSE) {
			g_free(file_path);
			break;
		}
		g_free(file_path);
		g_free(new_msgid);
		new_msgid = NULL;

	}
	g_free(folder_name);
	g_free(account);
	return new_msgid;
}

/**
  This function updates the unread status after move or copy operation
  @param msgid    message id of the moved/copied message
  @param dest_msgid    destination message id 
  @param move_operation    whether the folder operation is move or copy 
  @return success(FOLDER_SUCCESS) or failure(FOLDER_FAIL) 
*/
gint folder_update_unread(const gchar * msgid, const gchar * destmsgid, gboolean move_operation)
{
	gchar *dir = NULL;
	Folder *folder = NULL;

	if ((msgid == NULL) || (destmsgid == NULL)) {
		return FOLDER_FAIL;
	}
	if (is_message_unread(msgid) == TRUE) {

		if (move_operation == TRUE) {
			folder_message_del_list(msgid, UNREAD);
		}
		folder_message_add_list(destmsgid, UNREAD);
		dir = get_fold_name(destmsgid);
		folder = folder_find_obj_by_name(dir);
		if (folder != NULL) {
			folder->unread++;
		}
	}
	return FOLDER_SUCCESS;
}

/**
  This function creates the standard folder like Inbox,
  Outbox etc if they do not exist 
  @param rootdir    directory application's directory for storing Mails 
  @return success(FOLDER_SUCCESS) or failure(FOLDER_FAIL) 
*/
static gint folder_create_standard_folder(const gchar * rootdir)
{
	gchar *dir = NULL;
	gchar *root = NULL;

	root = g_strdup_printf("%s%c%s", rootdir, G_DIR_SEPARATOR, MAIL);
	if (folder_create_dir_if_not_exist(root, FALSE) == FOLDER_FAIL) {
		g_free(root);
		return FOLDER_INITIALIZATION_ERROR;
	}

	dir = g_strdup_printf("%s%c%s", root, G_DIR_SEPARATOR, INBOX_DIR);
	if (folder_create_dir_if_not_exist(dir, FALSE) == FOLDER_FAIL) {
		folder_gchar_free(dir, root, NULL, NULL);
		return FOLDER_INITIALIZATION_ERROR;
	}
	g_free(dir);

	dir = g_strdup_printf("%s%c%s", root, G_DIR_SEPARATOR, OUTBOX_DIR);
	if (folder_create_dir_if_not_exist(dir, FALSE) == FOLDER_FAIL) {
		folder_gchar_free(dir, root, NULL, NULL);
		return FOLDER_INITIALIZATION_ERROR;
	}
	g_free(dir);

	dir = g_strdup_printf("%s%c%s", root, G_DIR_SEPARATOR, DRAFT_DIR);
	if (folder_create_dir_if_not_exist(dir, FALSE) == FOLDER_FAIL) {
		folder_gchar_free(dir, root, NULL, NULL);
		return FOLDER_INITIALIZATION_ERROR;
	}
	g_free(dir);
	dir = g_strdup_printf("%s%c%s", root, G_DIR_SEPARATOR, SENT_DIR);
	if (folder_create_dir_if_not_exist(dir, FALSE) == FOLDER_FAIL) {
		folder_gchar_free(dir, root, NULL, NULL);
		return FOLDER_INITIALIZATION_ERROR;
	}
	g_free(dir);
	g_free(root);
	return FOLDER_SUCCESS;
}

/**
  This function creates the archive folders on mmc device
  @param path    path of the mmc mounted
  @return FOLDER_SUCCESS(success) or folder error code
*/
gint folder_update_mmc_folder(const gchar * path)
{
	gchar *spu_tmp_archive_folder = NULL;
	gchar *folder_path = NULL;
	GnomeVFSResult vfs_result = GNOME_VFS_OK;
	GnomeVFSFileInfo *file_info = NULL;
	GnomeVFSURI *dir_uri = NULL;
	GnomeVFSURI *dir_uri_str = NULL;
	gint ret = 0;

	if (path == NULL) {
		debug_print("no mem\n");
		return FOLDER_CREATE_ERROR;
	}
	spu_tmp_archive_folder = g_strdup_printf("%s%c%s", get_mail_dir(),
						 G_DIR_SEPARATOR, ARCHIVE);
	gl_volume_mounted_path = g_strdup(path);
	if (spu_tmp_archive_folder == NULL || gl_volume_mounted_path == NULL) {
		return FOLDER_CREATE_ERROR;
	}

	folder_path = g_strdup_printf("%s", get_mail_dir());

	ret = folder_create(ARCHIVE, folder_path, F_LOCAL, TRUE);
	g_free(folder_path);

	if ((ret != FOLDER_SUCCESS) && (ret != FOLDER_EXISTING_ERROR)) {
		g_free(spu_tmp_archive_folder);
		return FOLDER_CREATE_ERROR;
	}
	g_free(spu_tmp_archive_folder);

	dir_uri = gnome_vfs_uri_new(path);
	dir_uri_str = gnome_vfs_uri_append_string(dir_uri, R_ARCHIVE);

	gnome_vfs_uri_unref(dir_uri);

	file_info = gnome_vfs_file_info_new();
	vfs_result = gnome_vfs_get_file_info_uri(dir_uri_str, file_info,
						 GNOME_VFS_FILE_INFO_DEFAULT);
	if (vfs_result == GNOME_VFS_ERROR_NOT_FOUND) {
		vfs_result = gnome_vfs_make_directory_for_uri(dir_uri_str, FILE_PERMS);
		if (vfs_result != GNOME_VFS_OK) {
			gnome_vfs_uri_unref(dir_uri_str);
			return ARCHIVE_FOLDER_CREATE_ERROR;
		}
		debug_print("%s : %s create folder success on mmc\n", __FILE__, __FUNCTION__);

	} else if (file_info->type == GNOME_VFS_FILE_TYPE_REGULAR) {
		gnome_vfs_uri_unref(dir_uri_str);
		return FOLDER_ARCHIVE_DELETE_FILE;
	} else if (file_info->type == GNOME_VFS_FILE_TYPE_DIRECTORY) {
		gnome_vfs_uri_unref(dir_uri_str);
		return FOLDER_SUCCESS;
	}

	return FOLDER_SUCCESS;
}

/**
   This function removes the archive folder in global folder list 
   @param none
   @return success(FOLDER_SUCCESS) or error code
*/
gint folder_remove_mmc_folder(void)
{
	Folder *dir = NULL;
	gchar *xml_path = NULL;
	GnomeVFSResult vfs_result = GNOME_VFS_OK;

	dir = folder_find_obj_by_name(ARCHIVE);

	if (dir == NULL) {
		return FOLDER_DOESNOT_EXIST_ERROR;
	}

	gl_folder_list = g_list_remove(gl_folder_list, dir);

	if (dir->path != NULL) {
		xml_path = g_strdup_printf("%s%c%s", dir->path, G_DIR_SEPARATOR, XML_CACHE);
		vfs_result = gnome_vfs_unlink(xml_path);
		if (vfs_result != GNOME_VFS_OK) {
			osso_log(LOG_INFO, "Unable to delete archive cache file\n");
		}
		osso_log(LOG_DEBUG, "Removed cache file\n");
		g_free(xml_path);
	}

	g_free(dir->name);
	g_free(dir->path);
	g_free(dir);

	return FOLDER_SUCCESS;
}

/** 
  This function checks the last number of file name used in
   archive folder
  @param none
  @return number to be used for the new file name,FOLDER_FAIL on error 
*/
static gint folder_get_archive_status(void)
{
	gchar *path = NULL;
	gint max = 0;
	gint num = 0;
	GnomeVFSResult vfs_result = GNOME_VFS_OK;
	GList *list_file_info = NULL;
	GList *cur = NULL;
	GnomeVFSFileInfo *file_info = NULL;

	path = g_strdup_printf("%s%c%s", gl_volume_mounted_path, G_DIR_SEPARATOR, R_ARCHIVE);

	vfs_result = gnome_vfs_directory_list_load(&list_file_info, path,
						   GNOME_VFS_FILE_INFO_DEFAULT);
	g_free(path);
	if (vfs_result == GNOME_VFS_OK) {
		for (cur = list_file_info; cur != NULL; cur = cur->next) {
			file_info = (GnomeVFSFileInfo *) cur->data;
			if (file_info == NULL) {
				continue;
			}
			if (file_info->name == NULL) {
				continue;
			}
			debug_print("%s : %s file_name  : %s\n",
				    __FILE__, __FUNCTION__, file_info->name);
			if ((num = to_number(file_info->name)) >= 0) {
				if (max < num) {
					max = num;
				}
			}
		}
	} else {
		debug_print("%s : %s folder_archive_status_error : %d\n",
			    __FILE__, __FUNCTION__, vfs_result);
		return FOLDER_FAIL;
	}

	return max + 1;
}

/**
  This function copies a message from or to the MMC
  @param src_path    the path of the source file
  @param dest_path   the path of the destination file
  @param overwrite_flag   the flag indicates whether destination file 
  has can be overwritten or not, incase of duplicate files 
  @return success(FOLDER_SUCCESS) or error code
*/
static gint
folder_mmc_copy_file(const gchar * src_path, const gchar * dest_path, gboolean overwrite_flag)
{
	GnomeVFSURI *src_uri = NULL;
	GnomeVFSURI *dest_uri = NULL;
	gchar *src_path_escaped = NULL;
	gchar *dest_path_escaped = NULL;
	GnomeVFSResult vfs_result;
	GnomeVFSXferOverwriteMode mode = 0;

	if (src_path == NULL || dest_path == NULL) {
		return FOLDER_ARCHIVE_MSG_COPY_FILE;
	}

	src_path_escaped = gnome_vfs_escape_path_string(src_path);
	if(src_path_escaped != NULL) {
		src_uri = gnome_vfs_uri_new(src_path_escaped);
		g_free(src_path_escaped);
	} else {
		src_uri = gnome_vfs_uri_new(src_path);
	}

	if (src_uri == NULL) {
		return FOLDER_ARCHIVE_MSG_COPY_FILE;
	}

	dest_path_escaped = gnome_vfs_escape_path_string(dest_path);
	if(dest_path_escaped != NULL) {
		dest_uri = gnome_vfs_uri_new(dest_path_escaped);
		g_free(dest_path_escaped);
	} else {
		dest_uri = gnome_vfs_uri_new(dest_path);
	}
	
	if (dest_uri == NULL) {
		gnome_vfs_uri_unref(src_uri);
		return FOLDER_ARCHIVE_MSG_COPY_FILE;
	}


	debug_print(" %s %s src path :%s \n", __FILE__, __FUNCTION__, src_path);
	debug_print(" %s %s dest path :%s \n", __FILE__, __FUNCTION__, dest_path);
	if (overwrite_flag == TRUE) {
		mode = GNOME_VFS_XFER_OVERWRITE_MODE_REPLACE;
	} else {
		mode = GNOME_VFS_XFER_OVERWRITE_MODE_QUERY;
	}
	vfs_result =
	    gnome_vfs_xfer_uri(src_uri, dest_uri,
			       GNOME_VFS_XFER_DEFAULT,
			       GNOME_VFS_XFER_ERROR_MODE_QUERY, mode, (GnomeVFSXferProgressCallback)
			       folder_xfer_progress_callback, NULL);
	gnome_vfs_uri_unref(src_uri);
	gnome_vfs_uri_unref(dest_uri);
	if (vfs_result != GNOME_VFS_OK) {
		return FOLDER_ARCHIVE_MSG_COPY_FILE;

	}
	return FOLDER_SUCCESS;
}

/**
  This function moves  the mail from or to mmc folder
  @param src_path    name of the source file
  @param dest_path    name of the destination file
  @param overwrite_flag   the flag indicates whether destination file 
  has can be overwritten or not, incase of duplicate files 
  @return success(FOLDER_SUCCESS) or error code
*/
gint folder_mmc_move_file(const gchar * src_path, const gchar * dest_path, gboolean overwrite_flag)
{
	GnomeVFSResult vfs_result = GNOME_VFS_ERROR_GENERIC;
	gchar *tmp_vfs_path = NULL;

	if (src_path == NULL || dest_path == NULL) {
		return FOLDER_ARCHIVE_MSG_MOVE_FILE;
	}

	if (folder_mmc_copy_file(src_path, dest_path, overwrite_flag)
	    == FOLDER_SUCCESS) {
		tmp_vfs_path = gnome_vfs_escape_path_string(src_path);
		vfs_result = gnome_vfs_unlink(tmp_vfs_path);
	}
	if (vfs_result == GNOME_VFS_OK) {
		debug_print("Error: %s\n", gnome_vfs_result_to_string(vfs_result));
		return FOLDER_SUCCESS;
	} else {
		debug_print("Error: %s\n", gnome_vfs_result_to_string(vfs_result));
		return FOLDER_ARCHIVE_MSG_MOVE_FILE;
	}

}

/**
  This is gnome_vfs_xfer callback function. 
  Just for checking the status of operation
  @param info    pointer to the GnomeVFSXferProgressInfo structure.
  @param data    pointer to the user data.
  @return FOLDER_XFER always 
*/
/*@unused@*/
static gint
folder_xfer_progress_callback(const GnomeVFSXferProgressInfo * info, const gpointer data)
{
	if (info == NULL) {
		return FOLDER_XFER;
	}
	debug_print("[%s][%s] : starts here :\n", __FILE__, __FUNCTION__);
	debug_print("the status is %d\n", info->status);
	debug_print("the phase is %d\n", info->phase);
	if (info->phase == GNOME_VFS_XFER_PHASE_COMPLETED) {
		debug_print("done with trasfering the file \n");
	}

	debug_print("[%s][%s] : ends here :\n", __FILE__, __FUNCTION__);
	return FOLDER_XFER;
}

/**
  This function copies the file of msgid, from mmc to local folder 
  @param msgid    message identifier of the message
  @return success(FOLDER_SUCCESS) or error code
*/
gint folder_copy_from_mmc(const gchar * msgid)
{
	gchar *src_file_path = NULL;
	gchar *dest_file_path = NULL;
	gint ar_num = 0;
	gint rv = FOLDER_FAIL;

	ar_num = get_msgid_num(msgid);
	if (ar_num == FOLDER_FAIL) {
		return FOLDER_FAIL;
	}
	src_file_path = g_strdup_printf("%s%c%s%c%d", gl_volume_mounted_path,
					G_DIR_SEPARATOR, R_ARCHIVE, G_DIR_SEPARATOR, ar_num);
	dest_file_path = folder_get_message_file_path(msgid);
	if (dest_file_path == NULL) {
		return rv;
	}
	rv = folder_mmc_copy_file(src_file_path, dest_file_path, FALSE);
	g_free(src_file_path);
	g_free(dest_file_path);
	return rv;
}

/**
  This function updates the node in folder list with new folder name 
  @param folder    folder object,whose path & name has to be changed.
  The memory allocated for the folder name and path have to be freed 
  only when the folder iobject is deleted. 
  @param new_name    new name assigned to the folder 
  @return success(FOLDER_SUCCESS) or error code
*/
static gint folder_update_folder(Folder * folder, const gchar * new_name)
{
	if ((folder == NULL) || (new_name == NULL)) {
		return FOLDER_FAIL;
	}
	g_free(folder->path);

	folder->path = g_strdup_printf("%s%c%s", get_mail_dir(), G_DIR_SEPARATOR, new_name);

	g_free(folder->name);

	folder->name = g_strdup(new_name);

	return FOLDER_SUCCESS;
}

/**
  This function updates the node in folder list with new folder name 
  @param folder    pointer to folder information
  @param msgid    message id of the file,for which filepath is created 
  @param destination_flag    indicates whether the path required is for 
  destination file or source file. 
  @param dest_msgid    OUT parameter,pointer to the destination msgid  
  @return path of the file.This has to be freed by the calling function. 
*/
gchar *folder_get_file_path(const gchar * folder,
			    const gchar * msgid, gboolean destination_flag, gchar ** dest_msgid)
{
	gint ar_num = 0;
	gchar *file_msgid = NULL;
	gchar *ret_file_path = NULL;
	gboolean archive_flag = FALSE;
	gchar *default_account_name = NULL;

	if (strcmp(folder, ARCHIVE) == 0) {
		archive_flag = TRUE;
		if (CHECK_MMC_FLAG == FALSE) {
			return NULL;
		}
	}

	if (destination_flag == TRUE) {
		if (archive_flag == TRUE) {
			ar_num = folder_get_archive_status();

			if (ar_num < 0) {
				return NULL;
			}
			ret_file_path =
			    g_strdup_printf("%s%c%s%c%d", gl_volume_mounted_path,
					    G_DIR_SEPARATOR, R_ARCHIVE, G_DIR_SEPARATOR, ar_num);
			default_account_name = account_get_default_account_name();

			file_msgid = g_strdup_printf("%s%s%d",
						     default_account_name,
						     ARCHIVE_COPY_PREFIX, ar_num);
			g_free(default_account_name);
			*dest_msgid = file_msgid;
		} else {
			file_msgid = folder_create_msg_id(msgid, folder, TRUE);
			if (file_msgid == NULL) {
				return NULL;
			}
			ret_file_path = folder_alloc_file_path(folder, file_msgid);
			*dest_msgid = file_msgid;
		}
	} else {
		if (archive_flag == TRUE) {
			ret_file_path =
			    g_strdup_printf("%s%c%s%c%d", gl_volume_mounted_path,
					    G_DIR_SEPARATOR, R_ARCHIVE, G_DIR_SEPARATOR,
					    get_msgid_num(msgid));

		} else {
			ret_file_path = folder_alloc_file_path(folder, msgid);
		}

	}
	return ret_file_path;
}

/**
  This function parses the file to get msginfo 
  @param file_name    name of the file to be parsed 
  @param file_path    path of the file to be parsed 
  @param archive_flag    indicates whether the file belongs to archive folder 
  @return pointer to the message information,
  this has to be freed by the caller 
*/
MsgInfo *folder_get_info(const gchar * name, const gchar * file_path, gboolean archive_flag)
{
	MsgInfo *msginfo = NULL;
	gchar *msgid = NULL;
	gchar *dest_path = NULL;
	GnomeVFSResult vfs_result = GNOME_VFS_OK;

	if (archive_flag == TRUE) {
		dest_path = folder_get_arch_msg(file_path, name, &msgid);
		if (dest_path == NULL) {
			return NULL;
		}
	} else if (check_msgid(name) == TRUE) {
		msgid = (gchar *) name;
		dest_path = (gchar *) file_path;
	} else {
		return NULL;
	}
	msginfo = folder_parse_msg(dest_path);

	if (msginfo == NULL) {
		return NULL;
	}
	g_free(msginfo->header->msgid);

	msginfo->header->msgid = folder_get_msgid(msgid);
	if (is_message_unread(msginfo->header->msgid)) {
		msginfo->header->flags.perm_flags |= MSG_UNREAD;
	}

	/* set remote copy flag */
	msginfo->header->msgtag.remote_copy = folder_is_msg_remote(msginfo->header->msgid);

	if (archive_flag == TRUE) {
		vfs_result = gnome_vfs_unlink(dest_path);
		g_free(msgid);
		g_free(dest_path);
	}

	return msginfo;

}

/**
  This function copies the parse file from MMC to temporary folder 
  @param file_path    path of the file to be parsed(MMC) 
  @param name   name of the file to be copied 
  @param msgid    pointer to the filename,which willbe filled by function 
  @return destination file path on temporary folder.
  This has to be freed by the calling function. 
*/
static gchar *folder_get_arch_msg(const gchar * file_path, const gchar * name, gchar ** msgid)
{
	gchar *dest_path = NULL;
	gchar *file_name = NULL;
	gchar *default_account_name = NULL;

	default_account_name = account_get_default_account_name();
	file_name = g_strdup_printf("%s%s%s", default_account_name, ARCHIVE_COPY_PREFIX, name);

	g_free(default_account_name);

	dest_path = g_strdup_printf("%s%c%s%c%s",
				    get_mail_dir(), G_DIR_SEPARATOR,
				    ARCHIVE, G_DIR_SEPARATOR, file_name);


	if (folder_mmc_copy_file(file_path, dest_path, FALSE) == FOLDER_FAIL) {
		g_free(dest_path);
		g_free(msgid);
		*msgid = NULL;
		return NULL;
	}
	*msgid = file_name;

	return dest_path;
}

/**
  This function updates folder list 
  @param file_path    path of the file to be parsed(MMC) 
  @param name    name of the file to be copied 
  @param type    type of folder 
  @return success(FOLDER_SUCCESS) or error code 
*/
static gint folder_update_folder_list(const gchar * full_path, const gchar * name, gint type)
{
	Folder *new_folder = NULL;

	new_folder = (Folder *) folder_new(name, full_path, type);
	if (new_folder == NULL) {
		return FOLDER_CREATE_ERROR;
	}

	gl_folder_list = g_list_append(gl_folder_list, new_folder);

	if (gl_folder_list == NULL) {
		g_free(new_folder->path);
		g_free(new_folder->name);
		g_free(new_folder);
		return FOLDER_CREATE_ERROR;
	}
	return FOLDER_SUCCESS;
}

/**
  This function checks the partial message file to know whether it 
  contains header information and needs to be parsed. 
  @param file    path of the file to be checked 
  @return TRUE if file has to be parsed,FALSE otherwise 
*/
gboolean folder_check_partial(const gchar * file)
{
	gchar *file_name = NULL;
	const gchar *sec_ptr = NULL;

	file_name = strrchr(file, G_DIR_SEPARATOR);
	if (file_name == NULL) {
		return FALSE;
	}
	if ((sec_ptr = get_file_extension(file_name)) != NULL) {
		sec_ptr++;
		if (sec_ptr == NULL) {
			return FALSE;
		}

		if ((strcmp(sec_ptr, HEADER_SECTION) != 0) && (strcmp(sec_ptr, MSG_HDR) != 0)) {
			return FALSE;
		}

	}
	return TRUE;
}


/**
   This function used to sort the message id list.

   @param msgid_list GSlist of message id list 
   @return FALSE in case of GSList NULL, otherwise returns TRUE 
*/

GSList *folder_sort_msglist(GSList * msgid_list, gboolean retr_flag)
{
	if (msgid_list == NULL) {
		return NULL;
	}
	if (retr_flag == TRUE) {
		msgid_list = g_slist_sort(msgid_list, folder_retr_sort_comparefunc);

	} else {
		msgid_list = g_slist_sort(msgid_list, folder_msgid_sort_comparefunc);

	}
	return msgid_list;
}


/**
   This function used to compare function used in g_slist_sort function.
   This function compares the two strings and return the result 

   @param a first element
   @param b second element 
   @return -1 in case of any invalid input, otherwise returns the compared 
    result. 
*/

static gint folder_msgid_sort_comparefunc(gconstpointer a, gconstpointer b)
{
	gint rv = 0;
	gchar *account1 = NULL;
	gchar *account2 = NULL;
	if (check_msgid((gchar *) a) == FALSE) {
		return -1;
	}
	if (check_msgid((gchar *) b) == FALSE) {
		return -1;
	}
	account1 = get_account_name((gchar *) a);
	account2 = get_account_name((gchar *) b);
	rv = g_ascii_strcasecmp(account1, account2);
	g_free(account1);
	g_free(account2);
	return rv;
}


/**
   This function used to get the message id fromthe given path 

    @param mesgidpath message file path
    @return message id on success or returns NULL on fail
*/
gchar *folder_get_msgid_from_path(const gchar * msgidpath)
{
	gchar *msgid = NULL;

	if (msgidpath == NULL) {
		return NULL;
	}
	msgid = strrchr(msgidpath, G_DIR_SEPARATOR);
	if ((msgid == NULL) && (msgid + 1 == NULL)) {
		return NULL;
	}
	return g_strdup(msgid + 1);
}

/**
 This functions returns the size of the all 
 part files of a message
 @param msgid   file path of the file
 @return size of the file
 */
static glong folder_get_all_size(const gchar * filepath)
{
	struct stat s;
	GSList *msg_list = NULL;
	GSList *cur = NULL;
	glong size = FOLDER_FAIL;
	gchar *folder = NULL;
	gchar *msgid = NULL;

	msgid = folder_get_msgid_from_path(filepath);
	folder = get_fold_name(msgid);

	msg_list = folder_get_msgid_all_files(folder, msgid, NULL, TRUE);

	g_free(folder);
	g_free(msgid);
	if (msg_list == NULL) {
		return FOLDER_FAIL;
	}
	for (cur = msg_list; cur != NULL; cur = cur->next) {

		if (stat(cur->data, &s) < 0) {
			size = FOLDER_FAIL;
			break;
		}
		size += s.st_size;
	}
	slist_free_strings(msg_list);
	g_slist_free(msg_list);
	return size;
}

/**
 This function checks the total and unread count of the folder,
 by checking the msgid 
 @param folder_obj folder object of the folder
 @param msgid   message id of the file to be checked
 @return FOLDER_SUCCESS on successfuly updating.FOLDER_FAIL otherwise 
 */
static gint folder_update_count(Folder * folder_obj, const gchar * msgid, const gchar * file_path)
{
	struct stat s;
	glong num = 0;

	if ((folder_obj == NULL) || (check_msgid(msgid) == FALSE)) {
		return FOLDER_FAIL;
	}

	if (stat(file_path, &s) < 0) {
		return FOLDER_FAIL;
	}
	if (!S_ISREG(s.st_mode)) {
		return FOLDER_FAIL;
	}
	if (folder_check_partial(file_path) == FALSE) {
		return FOLDER_FAIL;

	}
	folder_obj->total++;
	if (is_message_unread(msgid) == TRUE) {
		folder_obj->unread++;
	}
	if ((num = get_msgid_num(msgid)) >= 0) {
		if (folder_obj->max < num) {
			folder_obj->max = num;
		}
	}
	return FOLDER_SUCCESS;
}

/**
 This function checks whether the meassge read is valid,by checking 
 for mandatory RFC822 fields 
 @param msginfo  message information of the file to be checked
 @return TRUE if file is correct, FALSE otherwise 
 */
gboolean folder_is_msg_valid(const MsgInfo * msginfo)
{
	if (msginfo == NULL) {
		return FALSE;
	}
	if (msginfo->header == NULL) {
		return FALSE;
	}
	if ((msginfo->header->from == NULL)
	    || (msginfo->header->msgid == NULL)
	    || (msginfo->header->date == 0)) {
		return FALSE;
	}
	return TRUE;
}

gint folder_get_arch_size(gchar * file_path, gboolean archive)
{
	gchar *path = NULL;
	gint size = 0;
	GnomeVFSFileInfo *file_info = NULL;
	GList *list_file_info = NULL;
	GList *cur = NULL;
	GnomeVFSResult vfs_result = GNOME_VFS_OK;

	if (archive == TRUE) {
		path = g_strdup_printf("%s%c%s", gl_volume_mounted_path,
				       G_DIR_SEPARATOR, R_ARCHIVE);
	} else {
		path = g_strdup(file_path);
	}
	vfs_result = gnome_vfs_directory_list_load(&list_file_info, path,
						   GNOME_VFS_FILE_INFO_DEFAULT);
	if (vfs_result != GNOME_VFS_OK) {
		return -1;
	}
	for (cur = list_file_info; cur != NULL; cur = cur->next) {
		file_info = (GnomeVFSFileInfo *) cur->data;
		if (file_info->type == GNOME_VFS_FILE_TYPE_REGULAR) {

			size += file_info->size;
		}
	}

	return size;
}

gint
folder_process_smime_mail(const gchar * file,
			  gchar * outfile,
			  gchar * detached, gboolean * saved,
			  const gchar * signer_email_id,
			  const gchar * recip_email_id,
			  gint * signer_cert_state,
			  gint * private_cert_state, gboolean * signature,
			  gboolean * trust, gchar * cert_path, gint * smime,
			  gint * sender_cert_id, gint * recip_cert_id)
{
	gchar inter_file[BSIZE];
	gint ret_value = 0;
	gint smime_type = SMIME_NONE;
	gboolean decr_verify = FALSE;
	gboolean encrypt_sign = FALSE;

	debug_print("In function folder_process_smime_mail\n");

	CHECK_RETURN_VAL_IF_FAIL((file != NULL), FOLDER_FAIL);
	CHECK_RETURN_VAL_IF_FAIL((outfile != NULL), FOLDER_FAIL);
	CHECK_RETURN_VAL_IF_FAIL((detached != NULL), FOLDER_FAIL);
	CHECK_RETURN_VAL_IF_FAIL((cert_path != NULL), FOLDER_FAIL);

	smime_type = find_smime_type(file);
	if (smime_type == 0) {
		debug_print("S/MIME mail is tampered\n");
		return ENGINE_ERROR_MIME_INVALID;
	}

	if (FALSE == (get_temp_smime_file(inter_file, BSIZE + 1))) {
		debug_print("Unable to get file path\n");
		return ENGINE_ERROR_FILE_ERROR;
	}

	/* Remove any file with this name if it exists */
	remove_smime_file(inter_file);

	switch (smime_type) {
	case SMIME_SIGN:
		*smime = SMIME_SIGN;
		ret_value = verify_message(file, inter_file, detached,
					   saved, signer_email_id,
					   signer_cert_state, signature,
					   trust, cert_path, sender_cert_id);
		smime_type = find_smime_type(inter_file);
		if (smime_type == 0) {
			break;
		} else if (smime_type == 2) {
			encrypt_sign = TRUE;
			file = g_strdup(inter_file);
			memset(inter_file, '\0', BSIZE);
			if (get_temp_smime_file(inter_file, BSIZE + 1) == FALSE ) {
				debug_print("Unable to get file path\n");
				return ENGINE_ERROR_FILE_ERROR;
			}
		}
	case SMIME_ENCRYPT:
		ret_value = decrypt_message(file, inter_file,
					    recip_email_id, private_cert_state,
					    detached, saved, signer_email_id,
					    signer_cert_state, signature, trust,
					    cert_path, decr_verify, smime,
					    sender_cert_id, recip_cert_id);
		if (encrypt_sign == TRUE)
			*smime = SMIME_SIGN_ENCRYPT;
		break;

	case SMIME_SIGN_ENCRYPT:
		*smime = SMIME_SIGN_ENCRYPT;
		ret_value = decrypt_verify(file, recip_email_id,
					   private_cert_state, inter_file,
					   detached, saved, signer_email_id,
					   signer_cert_state, signature,
					   trust, cert_path, sender_cert_id, recip_cert_id);
		break;
	default:
		break;
	}

	if (ret_value != SMIME_OK) {
		debug_print("Unable to process smime message\n");
		engine_errno = ret_value;
		return ret_value;
	}

	if (folder_copy_smime_contents(outfile, inter_file) != SMIME_OK) {
		debug_print("Unable to copy the file contents\n");
		remove_smime_file(inter_file);
		return ENGINE_ERROR_FILE_ERROR;
	}

	remove_smime_file(file);
	remove_smime_file(inter_file);
	return ret_value;
}

static
gint folder_copy_smime_contents(gchar * outfile, gchar * inter_file)
{
	gchar buffer[BSIZE];
	gint ret_value = SMIME_OK;
	FILE *outfp = NULL, *interfp = NULL;

	CHECK_RETURN_VAL_IF_FAIL((outfile != NULL), FOLDER_FAIL);
	CHECK_RETURN_VAL_IF_FAIL((inter_file != NULL), FOLDER_FAIL);

	/* copy the contents of this file into output file
	 * The output file has MIME-Version:1 field 
	 */
	if ((outfp = fopen(outfile, "w")) == NULL) {
		FILE_OP_ERROR(outfile, "fopen");
		send_engine_error_to_ui(errno);
		return FOLDER_FAIL;
	}

	if (change_file_mode_rw(outfp, outfile) < 0) {
		FILE_OP_ERROR(inter_file, "chmod");
		fclose(outfp);
		return FOLDER_FAIL;
	}

	if ((interfp = fopen(inter_file, "r")) == NULL) {
		FILE_OP_ERROR(inter_file, "fopen");
		fclose(outfp);
		return FOLDER_FAIL;
	}

	if (change_file_mode_rw(interfp, inter_file) < 0) {
		FILE_OP_ERROR(inter_file, "chmod");
		fclose(outfp);
		fclose(interfp);
		return FOLDER_FAIL;
	}

	fprintf(outfp, "Mime-Version: 1.0\n");
	memset(buffer, '\0', sizeof(buffer));
	while (fgets(buffer, sizeof(buffer), interfp) != NULL) {
		strcrchomp(buffer);
		if (fputs(buffer, outfp) == EOF) {
			fclose(outfp);
			fclose(interfp);
			remove_smime_file(inter_file);
			return FOLDER_FAIL;
		}
		memset(buffer, '\0', sizeof(buffer));
	}

	fclose(outfp);
	fclose(interfp);
	return ret_value;
}

GSList *folder_append_header_list(GSList * hdr_list, const gchar * msgid,
				  const gchar * dest_folder,
				  const PrefsAccount * prefs_account,
				  gboolean delete_flag, GSList ** update_list_ptr,
				  const gchar * new_msgid)
{
	GnomeVFSResult vfs_result;
	gchar *dest_file_path = NULL;
	const gchar *dest_msgid = NULL;
	MsgInfo *msginfo = NULL;
	gboolean archive_flag = FALSE;

	osso_log(LOG_DEBUG, "\n[%s] start\n", __FUNCTION__);
	if (new_msgid == NULL) {
		dest_msgid = msgid;
	} else {
		dest_msgid = new_msgid;
	}

	/* if the operation is delete,just need msgid */
	if (delete_flag == TRUE) {
		msginfo = g_new0(MsgInfo, 1);
		msginfo->header = g_new0(MsgHeader, 1);
		msginfo->header->msgid = g_strdup(dest_msgid);
		msginfo->header->msgtag.fullydownloaded = FALSE;
		msginfo->header->msgtag.bodydownloaded = FALSE;
	} else {
		if ((g_ascii_strncasecmp(dest_folder, ARCHIVE, strlen(ARCHIVE)) == 0)
		    && (CHECK_MMC_FLAG == TRUE)) {
			archive_flag = TRUE;

			if (CHECK_MMC_FLAG == TRUE) {
				osso_log(LOG_DEBUG, " %s %s mmc is enabled - folder name%s\n",
					 __FILE__, __FUNCTION__, dest_folder);

				if (folder_copy_from_mmc(dest_msgid) != FOLDER_SUCCESS) {
					return hdr_list;
				}
				archive_flag = TRUE;

			} else {
				debug_print(" %s %s mmc is disabled - folder name %s \n",
					    __FILE__, __FUNCTION__, dest_folder);
				return hdr_list;
			}
		}
		dest_file_path = folder_get_message_file_path(dest_msgid);

		if (dest_file_path == NULL) {
			return hdr_list;
		}
		msginfo = folder_read_msginfo(dest_file_path, dest_msgid, msgid, prefs_account);
		folder_update_src_list(update_list_ptr, msgid);
	}
	if (msginfo != NULL) {
		msginfo->header->msgnum = get_msgid_num(dest_msgid);
		hdr_list = g_slist_append(hdr_list, msginfo->header);
		msginfo->header = NULL;
		procmsg_msginfo_free(msginfo);
	}
	if (delete_flag == TRUE) {
		return hdr_list;

	}

	if (archive_flag == TRUE) {
		vfs_result = gnome_vfs_unlink(dest_file_path);

		if (vfs_result != GNOME_VFS_OK) {
			g_free(dest_file_path);
			return hdr_list;
		}
	}
	g_free(dest_file_path);

	return hdr_list;

}

GSList *folder_rem_parts(const gchar * srcfolder, const gchar * msgid, gboolean * read_flag_ptr)
{
	GSList *msgid_list = NULL;
	gboolean body_download_flag = FALSE;
	body_download_flag = FALSE;

	msgid_list =
	    folder_get_msgid_all_files((gchar *) srcfolder,
				       (gchar *) msgid, &body_download_flag, FALSE);
	*read_flag_ptr = FALSE;
	if (msgid_list != NULL) {
		if ((is_message_unread((gchar *) msgid) == FALSE)
		    && (body_download_flag == TRUE)) {
			*read_flag_ptr = TRUE;
		}
	}
	return msgid_list;
}

GSList *folder_get_queued_complete_list(void)
{
	GSList *cur = NULL;
	GSList *folder_delete_list = NULL;

	for (cur = gl_msg_del_list; cur != NULL; cur = cur->next) {
		folder_delete_list = folder_append_header_list(folder_delete_list,
							       cur->data, NULL,
							       NULL, TRUE, NULL, NULL);
	}
	return folder_delete_list;
}
static MsgInfo *folder_read_msginfo(const gchar * dest_file_path,
				    const gchar * dest_msgid,
				    const gchar * src_msgid, const PrefsAccount * prefs_account)
{
	MsgInfo *msginfo = NULL;

	msginfo = folder_parse_msg(dest_file_path);

	if (msginfo == NULL) {
		return NULL;
	}
	g_free(msginfo->header->msgid);

	msginfo->header->msgid = g_strdup(dest_msgid);
	if (is_message_unread(dest_msgid)) {
		msginfo->header->flags.perm_flags |= MSG_UNREAD;
	}
	if (prefs_account == NULL) {
		msginfo->header->msgtag.protocol = PROTO_INVALID;
	} else {

		msginfo->header->msgtag.protocol = prefs_account->receive_settings.protocol;
		msginfo->header->msgtag.recvtype = prefs_account->receive_settings.recv_type;
	}
	if (procmime_check_attachments(dest_file_path, FALSE) == TRUE) {
		msginfo->header->flags.perm_flags |= MSG_ATTACH;
	}
	msginfo->header->msgtag.remote_copy = FALSE;
	msginfo->header->msgtag.fullydownloaded = TRUE;
	msginfo->header->msgtag.bodydownloaded = TRUE;
	msginfo->header->smime_type = folder_get_smime_type(msginfo->header, dest_file_path);
	return msginfo;
}

gint folder_get_smime_type(MsgHeader * header, const gchar * file)
{
	gint smime_header = 0;
	gint smime = 0;

	smime_header = header->smime_type;
	smime = find_smime_type(file);

	if (smime == SMIME_NONE) {
		return smime_header;
	} else {
		return smime;
	}
}
static void folder_update_src_list(GSList ** update_list_ptr, const gchar * msgid)
{
	MsgHeader *header = NULL;


	if (update_list_ptr != NULL) {
		header = g_new0(MsgHeader, 1);
		header->msgid = g_strdup(msgid);
		/* other params are not needed */
		header->msgtag.fullydownloaded = TRUE;
		header->msgtag.bodydownloaded = TRUE;
		*update_list_ptr = g_slist_append(*update_list_ptr, header);
	}
}

gchar *folder_process_smime_message(const gchar * msgid)
{
	gint ret_val = FOLDER_FAIL;
	gchar *file = NULL;
	gchar *signer_email_id = NULL, *recip_email_id = NULL;
	gchar *out_file = NULL;
	gint smime = 0;
	gboolean trust = FALSE;
	gboolean signature = TRUE;
	gint private_cert_state = 0;
	gint recip_cert_state = 0;
	gboolean saved = FALSE;
	PrefsAccount *ac_prefs = NULL;
	gchar *account_name = NULL;
	gint sender_cert_id = 0;
	gint recip_cert_id = 0;

	gchar outfile[BSIZE];
	gchar detached[BSIZE];
	gchar cert_path[BSIZE];

	MsgInfo *msg_info = NULL;

	memset(outfile, '\0', BSIZE);
	memset(detached, '\0', BSIZE);
	memset(cert_path, '\0', BSIZE);

	if (get_temp_smime_file(outfile, BSIZE + 1) == FALSE) {
		debug_print("%s %s : outfile failure \n", __FILE__, __FUNCTION__);
		return NULL;
	}
	if (get_temp_smime_file(detached, BSIZE + 1) == FALSE) {
		debug_print("%s %s : detached failure \n", __FILE__, __FUNCTION__);
		return NULL;
	}
	if (get_temp_smime_file(cert_path, BSIZE + 1) == FALSE) {
		debug_print("%s %s : detached failure \n", __FILE__, __FUNCTION__);
		return NULL;
	}

	file = folder_get_message_file_path(msgid);
	if (file == NULL) {
		debug_print("%s %s : file path not found \n", __FILE__, __FUNCTION__);
		return NULL;
	}
	msg_info = folder_parse_msg(file);
	if (msg_info == NULL) {
		debug_print("%s %s : msg info NULL \n", __FILE__, __FUNCTION__);
		g_free(file);
		return NULL;
	}

	signer_email_id = g_strdup(parse_from_address(msg_info->header->from));
	procmsg_msginfo_free(msg_info);

	account_name = get_account_name(msgid);
	if (account_name == NULL) {
		g_free(signer_email_id);
		g_free(file);
		return NULL;
	}
	ac_prefs = (PrefsAccount *) account_get_account(account_name);
	g_free(account_name);
	/* not to be freed this Prefs_account */
	if (ac_prefs == NULL) {
		recip_email_id = NULL;
	} else {
		recip_email_id = ac_prefs->user_settings.email;
	}

	ret_val = folder_process_smime_mail(file, outfile, detached, &saved,
					    signer_email_id,
					    recip_email_id,
					    &recip_cert_state,
					    &private_cert_state, &signature,
					    &trust, cert_path, &smime,
					    &sender_cert_id, &recip_cert_id);
	g_free(signer_email_id);
	g_free(file);

	if (ret_val != FOLDER_SUCCESS) {
		return NULL;
	} else {
		out_file = g_strdup_printf("%s", outfile);
		return out_file;
	}
}

gint
folder_prepare_process_smime(gchar * outfile, gchar * detached,
			     gchar * cert_path, const gchar * file,
			     gboolean * saved,
			     const gchar * signer_email_id,
			     const gchar * recip_email_id,
			     gint * recip_cert_state,
			     gint * private_cert_state,
			     gboolean * signature, gboolean * trust,
			     gint * smime, gint * sender_cert_id, gint * recip_cert_id)
{

	gint retval = 0;

	CHECK_RETURN_VAL_IF_FAIL((file != NULL), FOLDER_FAIL);

	debug_print("In function folder_prepare_process_smime\n");

	if (FALSE == (get_temp_smime_file(outfile, BSIZE + 1))) {
		debug_print("Unable to create file\n");
		return ENGINE_ERROR_FILE_ERROR;
	}

	if (FALSE == (get_temp_smime_file(detached, BSIZE + 1))) {
		debug_print("Unable to create file\n");
		return ENGINE_ERROR_FILE_ERROR;
	}

	if (FALSE == (get_temp_smime_file(cert_path, BSIZE + 1))) {
		debug_print("Unable to create file\n");
		return ENGINE_ERROR_FILE_ERROR;
	}

	/* Remove any files with the same name and path */
	remove_smime_file(outfile);
	remove_smime_file(detached);
	remove_smime_file(cert_path);

	if ((retval = folder_process_smime_mail(file, outfile,
						detached, saved,
						signer_email_id,
						recip_email_id,
						recip_cert_state,
						private_cert_state, signature,
						trust, cert_path, smime,
						sender_cert_id, recip_cert_id)) != FOLDER_SUCCESS) {
		debug_print("Unable to verify/decrypt S/MIME\n");
	}

	engine_errno = retval;
	return retval;
}

static void folder_delete_tmp_files(const gchar * msgid)
{
	DIR *dp = NULL;
	struct dirent *d = NULL;
	const gchar *path = NULL, *t1 = NULL;

	if (msgid == NULL) {
		return;
	}

	path = get_mime_tmp_dir();

	if (path == NULL) {
		return;
	}

	if (change_dir(path) < 0) {
		return;
	}

	if ((dp = opendir(".")) == NULL) {
		return;
	}

	d = readdir(dp);

	while (d != NULL) {
		t1 = NULL;
		if (dirent_is_regular_file(d)) {
			t1 = strstr(d->d_name, msgid);
			if (t1 != NULL) {
				unlink(d->d_name);
			}
		}
		d = readdir(dp);
	}
	closedir(dp);
	return;
}

gchar *folder_get_archive_path(void)
{
	gchar *archive_path = NULL;

	archive_path = g_strdup_printf("%s%c%s", gl_volume_mounted_path,
				       G_DIR_SEPARATOR, R_ARCHIVE);

	return archive_path;
}

gint folder_change_time(GList * folder_list)
{
	Folder *dir = NULL;
	GList *list = NULL;
	gint ret_val = FOLDER_SUCCESS;
	gchar *path = NULL;

	GnomeVFSResult result = GNOME_VFS_OK;
	GnomeVFSFileInfo *stat_buf = NULL;

	stat_buf = gnome_vfs_file_info_new();
	if (folder_list == NULL) {
		/*Touch all folders */
		list = gl_folder_list;
	} else {
		list = folder_list;
	}

	for (; list != NULL; list = list->next) {
		dir = (Folder *) list->data;

		if ((dir->name != NULL) && (dir->path != NULL)) {

			/* If archival folder mounted path is to be used */
			if (g_ascii_strcasecmp(dir->name, ARCHIVE) != 0) {
				path = g_strdup(dir->path);
			} else {
				path = g_strdup_printf("%s%c%s",
						       gl_volume_mounted_path,
						       G_DIR_SEPARATOR, R_ARCHIVE);
			}

			if ((result = gnome_vfs_get_file_info(path, stat_buf,
							      GNOME_VFS_FILE_INFO_FOLLOW_LINKS |
							      GNOME_VFS_FILE_INFO_DEFAULT)) !=
			    GNOME_VFS_OK) {
				osso_log(LOG_WARNING, "could not stat folder\n");
				continue;
			}

			/* change the time on folder. Does the equivalent
			 * of utime
			 */
			stat_buf->mtime = time(NULL);

			if ((result = gnome_vfs_set_file_info(path, stat_buf,
							      GNOME_VFS_SET_FILE_INFO_TIME)) !=
			    GNOME_VFS_OK) {
				osso_log(LOG_WARNING, "Unable to change time");
				ret_val = FOLDER_FAIL;
			}
			gnome_vfs_file_info_clear(stat_buf);
			g_free(path);

		}
	}
	gnome_vfs_file_info_unref(stat_buf);
	return ret_val;
}

gboolean folder_check_pop_header(GSList * file_list)
{
	gchar *pop_header = NULL;
	gint counter = 0;
	gint length = 0;
	const gchar *sec_ptr = NULL;

	if (file_list == NULL) {
		return FALSE;
	}

	length = g_slist_length(file_list);

	if (length == 0) {
		return FALSE;
	}

	for (counter = 0; counter < length; counter++) {
		pop_header = g_slist_nth_data(file_list, counter);
		if (pop_header == NULL) {
			return FALSE;
		}
		if ((sec_ptr = get_file_extension(pop_header)) != NULL) {
			sec_ptr++;
		}
		if (sec_ptr == NULL) {
			return FALSE;
		}

		if (strcmp(sec_ptr, MSG_HDR) == 0) {
			return TRUE;
		}
	}

	return FALSE;
}

static gint folder_retr_sort_comparefunc(gconstpointer a, gconstpointer b)
{
	gint rv = 0;
	gchar *account1 = NULL;
	gchar *account2 = NULL;
	gchar *msgid1 = NULL;
	gchar *msgid2 = NULL;

	msgid1 = ((RetrMsgData *) a)->msgid;
	msgid2 = ((RetrMsgData *) b)->msgid;

	if (check_msgid(msgid1) == FALSE) {
		return -1;
	}
	if (check_msgid(msgid2) == FALSE) {
		return -1;
	}
	account1 = get_account_name(msgid1);
	account2 = get_account_name(msgid2);
	rv = g_ascii_strcasecmp(account1, account2);
	g_free(account1);
	g_free(account2);
	return rv;
}

GSList *get_unread_mails(gint * unread_count)
{
	GSList *unread_list = NULL;
	gint counter = 0;
	gint len = 0;
	gchar *msg_id = NULL;
	gchar *path = NULL;

	len = g_slist_length(gl_message_list);

	*unread_count = 0;


	for (counter = len - 1; counter >= 0; counter--) {
		msg_id = g_slist_nth_data(gl_message_list, counter);

		if (msg_id == NULL)
			continue;

		if ((path = folder_message_inbox_unread(msg_id)) != NULL) {

			if (path == NULL) {
				continue;
			}

                        unread_list = g_slist_append(unread_list, path);
			*unread_count = *unread_count + 1;
		}

	}

	return unread_list;
}

static
gchar *folder_message_inbox_unread(gchar * msg_id)
{
	gchar *path = NULL;
	gchar *path_hdr = NULL;
	gchar *path_0 = NULL;

	if (msg_id == NULL)
		return NULL;

	if (g_strrstr(msg_id, INBOX_DIR) == NULL)
		return NULL;

	path = folder_get_message_file_path(msg_id);

	if (is_file_entry_exist(path) == TRUE) {
		return path;
	}

	path_hdr = g_strdup_printf("%s%c%s", path, DOT, MSG_HDR);

	if (is_file_entry_exist(path_hdr) == TRUE) {
		g_free(path);
		return path_hdr;
	}

	path_0 = g_strdup_printf("%s%c%s", path, DOT, HEADER_SECTION);

	if (is_file_entry_exist(path_0) == TRUE) {
		g_free(path_hdr);
		g_free(path);
		return path_0;
	}

	g_free(path);
	g_free(path_hdr);
	g_free(path_0);
	return NULL;
}


static
void folder_set_gconf_data(MsgInfo * msginfo, gint counter)
{

	gboolean is_attach = TRUE;
	gfloat time = 0;


	if ((msginfo == NULL) || (msginfo->header == NULL) || (msginfo->header->msgid == NULL))
		return;
	if(msginfo->header->rx_date == 0)
		time = msginfo->header->mtime;
	else
		time = msginfo->header->rx_date;

	switch (counter) {
	case MSG_0:
		osso_email_gconf_set_mailid(EMAIL_GCONF_UNREAD_MAIL1_MSGID, msginfo->header->msgid);

		osso_email_gconf_set_from(EMAIL_GCONF_UNREAD_MAIL1_FROM, msginfo->header->from);

		osso_email_gconf_set_subject(EMAIL_GCONF_UNREAD_MAIL1_SUBJECT,
					     msginfo->header->subject);
		osso_email_gconf_set_time(EMAIL_GCONF_UNREAD_MAIL1_TIME,time);

		is_attach = MSG_IS_ATTACH(msginfo->header->flags);
		osso_email_gconf_set_has_attachment(EMAIL_GCONF_UNREAD_MAIL1_HAS_ATTACH, is_attach);
		break;

	case MSG_1:
		osso_email_gconf_set_mailid(EMAIL_GCONF_UNREAD_MAIL2_MSGID, msginfo->header->msgid);
		osso_email_gconf_set_from(EMAIL_GCONF_UNREAD_MAIL2_FROM, msginfo->header->from);
		osso_email_gconf_set_subject(EMAIL_GCONF_UNREAD_MAIL2_SUBJECT,
					     msginfo->header->subject);
		osso_email_gconf_set_time(EMAIL_GCONF_UNREAD_MAIL2_TIME,time);
		is_attach = MSG_IS_ATTACH(msginfo->header->flags);
		osso_email_gconf_set_has_attachment(EMAIL_GCONF_UNREAD_MAIL2_HAS_ATTACH, is_attach);

		break;

	case MSG_2:
		osso_email_gconf_set_mailid(EMAIL_GCONF_UNREAD_MAIL3_MSGID, msginfo->header->msgid);
		osso_email_gconf_set_from(EMAIL_GCONF_UNREAD_MAIL3_FROM, msginfo->header->from);
		osso_email_gconf_set_subject(EMAIL_GCONF_UNREAD_MAIL3_SUBJECT,
					     msginfo->header->subject);
		osso_email_gconf_set_time(EMAIL_GCONF_UNREAD_MAIL3_TIME,time);
		is_attach = MSG_IS_ATTACH(msginfo->header->flags);
		osso_email_gconf_set_has_attachment(EMAIL_GCONF_UNREAD_MAIL3_HAS_ATTACH, is_attach);
		break;

	case MSG_3:
		osso_email_gconf_set_mailid(EMAIL_GCONF_UNREAD_MAIL4_MSGID, msginfo->header->msgid);
		osso_email_gconf_set_from(EMAIL_GCONF_UNREAD_MAIL4_FROM, msginfo->header->from);
		osso_email_gconf_set_subject(EMAIL_GCONF_UNREAD_MAIL4_SUBJECT,
					     msginfo->header->subject);
		osso_email_gconf_set_time(EMAIL_GCONF_UNREAD_MAIL4_TIME,time);
		is_attach = MSG_IS_ATTACH(msginfo->header->flags);
		osso_email_gconf_set_has_attachment(EMAIL_GCONF_UNREAD_MAIL4_HAS_ATTACH, is_attach);
		break;

	case MSG_4:
		osso_email_gconf_set_mailid(EMAIL_GCONF_UNREAD_MAIL5_MSGID, msginfo->header->msgid);
		osso_email_gconf_set_from(EMAIL_GCONF_UNREAD_MAIL5_FROM, msginfo->header->from);
		osso_email_gconf_set_subject(EMAIL_GCONF_UNREAD_MAIL5_SUBJECT,
					     msginfo->header->subject);
		osso_email_gconf_set_time(EMAIL_GCONF_UNREAD_MAIL5_TIME,time);
		is_attach = MSG_IS_ATTACH(msginfo->header->flags);
		osso_email_gconf_set_has_attachment(EMAIL_GCONF_UNREAD_MAIL5_HAS_ATTACH, is_attach);
		break;

	default:
		break;
	}
}

static
void folder_initialize_gconf_values(void)
{
	osso_email_gconf_unset(EMAIL_GCONF_UNREAD_MAIL1_MSGID);
	osso_email_gconf_unset(EMAIL_GCONF_UNREAD_MAIL2_MSGID);
	osso_email_gconf_unset(EMAIL_GCONF_UNREAD_MAIL3_MSGID);
	osso_email_gconf_unset(EMAIL_GCONF_UNREAD_MAIL4_MSGID);
	osso_email_gconf_unset(EMAIL_GCONF_UNREAD_MAIL5_MSGID);

	osso_email_gconf_unset(EMAIL_GCONF_UNREAD_MAIL1_SUBJECT);
	osso_email_gconf_unset(EMAIL_GCONF_UNREAD_MAIL2_SUBJECT);
	osso_email_gconf_unset(EMAIL_GCONF_UNREAD_MAIL3_SUBJECT);
	osso_email_gconf_unset(EMAIL_GCONF_UNREAD_MAIL4_SUBJECT);
	osso_email_gconf_unset(EMAIL_GCONF_UNREAD_MAIL5_SUBJECT);

	osso_email_gconf_unset(EMAIL_GCONF_UNREAD_MAIL1_FROM);
	osso_email_gconf_unset(EMAIL_GCONF_UNREAD_MAIL2_FROM);
	osso_email_gconf_unset(EMAIL_GCONF_UNREAD_MAIL3_FROM);
	osso_email_gconf_unset(EMAIL_GCONF_UNREAD_MAIL4_FROM);
	osso_email_gconf_unset(EMAIL_GCONF_UNREAD_MAIL5_FROM);

	osso_email_gconf_unset(EMAIL_GCONF_UNREAD_MAIL1_HAS_ATTACH);
	osso_email_gconf_unset(EMAIL_GCONF_UNREAD_MAIL2_HAS_ATTACH);
	osso_email_gconf_unset(EMAIL_GCONF_UNREAD_MAIL3_HAS_ATTACH);
	osso_email_gconf_unset(EMAIL_GCONF_UNREAD_MAIL4_HAS_ATTACH);
	osso_email_gconf_unset(EMAIL_GCONF_UNREAD_MAIL5_HAS_ATTACH);
	
	osso_email_gconf_unset(EMAIL_GCONF_UNREAD_MAIL1_TIME);
	osso_email_gconf_unset(EMAIL_GCONF_UNREAD_MAIL2_TIME);
	osso_email_gconf_unset(EMAIL_GCONF_UNREAD_MAIL3_TIME);
	osso_email_gconf_unset(EMAIL_GCONF_UNREAD_MAIL4_TIME);
	osso_email_gconf_unset(EMAIL_GCONF_UNREAD_MAIL5_TIME);
}

gchar *folder_get_mounted_path(gchar * name)
{
	gchar *path = NULL;

	path = g_strdup_printf("%s%c%s%c%s",
			       gl_volume_mounted_path,
			       G_DIR_SEPARATOR, R_ARCHIVE, G_DIR_SEPARATOR, name);
	return path;

}

static gchar *folder_get_path_name(const gchar * folder_name)
{
	gchar *path = NULL;
	gchar *tmp_path = NULL;
	Folder *folder = NULL;

	if (folder_name == NULL) {
		return NULL;
	}

	if (g_ascii_strncasecmp(folder_name, ARCHIVE, strlen(ARCHIVE)) == 0) {
		if (CHECK_MMC_FLAG == TRUE) {
			debug_print("%s %s mmc is enabled - folder name%s\n",
				    __FILE__, __FUNCTION__, folder_name);
			path = g_strdup_printf("%s%c%s",
					       gl_volume_mounted_path, G_DIR_SEPARATOR, R_ARCHIVE);
		} else {
			debug_print("%s %s mmc is disabled - folder name %s \n",
				    __FILE__, __FUNCTION__, folder_name);
			return NULL;
		}

	} else {
		folder = folder_find_obj_by_name(folder_name);
		if (folder != NULL) {
			tmp_path = folder_get_path(folder);
		}
		if (tmp_path != NULL) {
			path = g_strdup(tmp_path);
		}
	}

	if (path == NULL) {
		return NULL;
	}

	return path;
}

GList *folder_get_file_list(const gchar * folder_name)
{

	GList *list_file_info = NULL;
	GList *tmp = NULL;
	GList *next = NULL;
	gchar *path = NULL;
	gchar *path_escaped = NULL;
	gchar *file_path = NULL;
	GnomeVFSResult vfs_result = GNOME_VFS_OK;
	GnomeVFSFileInfo *file_info = NULL;
	gboolean archive_flag = FALSE;
	gint num = 0;

	if (folder_name == NULL) {
		return NULL;
	}
	if (g_ascii_strncasecmp(folder_name, ARCHIVE, strlen(ARCHIVE)) == 0) {
		archive_flag = TRUE;
	}
	path = folder_get_path_name(folder_name);

	if (path == NULL) {
		return NULL;
	}
	path_escaped = gnome_vfs_escape_path_string(path);
	vfs_result = gnome_vfs_directory_list_load(&list_file_info, path_escaped,
						   GNOME_VFS_FILE_INFO_DEFAULT);
	g_free(path_escaped);

	if (vfs_result != GNOME_VFS_OK) {
		g_free(path);
		return NULL;
	}

	for (tmp = list_file_info; tmp != NULL;) {

		file_info = (GnomeVFSFileInfo *) tmp->data;

		if (tmp->data == NULL) {
			continue;
		}
		if (archive_flag == TRUE) {
			/* check if the message is valid for archival folder */
			if ((num = to_number(file_info->name)) >= 0) {
				tmp = tmp->next;

			} else {
				next = tmp->next;
				list_file_info = g_list_remove_link(list_file_info, tmp);
				gnome_vfs_file_info_unref(file_info);
				tmp = next;

			}
			continue;
		}
		file_path = g_strdup_printf("%s%c%s", path, G_DIR_SEPARATOR, file_info->name);
		/* remove non header files from list */
		if ((folder_check_partial(file_path) == FALSE) ||
		    (check_msgid(file_info->name) == FALSE)) {
			next = tmp->next;
			list_file_info = g_list_remove_link(list_file_info, tmp);
			gnome_vfs_file_info_unref(file_info);
			tmp = next;
		} else {
			tmp = tmp->next;
		}
		g_free(file_path);

	}
	g_free(path);
	return list_file_info;
}

GSList *folder_populate_from_list(GList * list_file_info, gint index, const gchar * folder_name)
{
	MsgInfo *msginfo = NULL;
	GSList *msglist = NULL;
	GnomeVFSFileInfo *file_info = NULL;
	gchar *file_path = NULL;
	gint counter = 0;
	gint length = 0;
	gint max = 0;
	gchar *path = NULL;
	gboolean archive_flag = FALSE;

	length = g_list_length(list_file_info);

	if (index >= length)
		return NULL;

	if (g_ascii_strncasecmp(folder_name, ARCHIVE, strlen(ARCHIVE)) == 0) {
		if (CHECK_MMC_FLAG == TRUE) {
			debug_print("%s %s mmc is enabled - folder name%s\n",
				    __FILE__, __FUNCTION__, folder_name);
			path = g_strdup_printf("%s%c%s", gl_volume_mounted_path,
					       G_DIR_SEPARATOR, R_ARCHIVE);
			archive_flag = TRUE;
		} else {
			debug_print("%s %s mmc is disabled - folder name %s \n",
				    __FILE__, __FUNCTION__, folder_name);
			return NULL;
		}
	}

	if (length > index + NUM_MAILS) {
		max = index + NUM_MAILS;
	} else {
		max = length;
	}

	path = folder_get_path_name(folder_name);
	for (counter = index; counter < max; counter++) {
		file_info = (GnomeVFSFileInfo *) g_list_nth_data(list_file_info, counter);
		if (file_info->type == GNOME_VFS_FILE_TYPE_REGULAR) {
			file_path = g_strdup_printf("%s%c%s", path,
						    G_DIR_SEPARATOR, file_info->name);
			msginfo = folder_get_info(file_info->name, file_path, archive_flag);
			if (msginfo != NULL) {
				msglist = g_slist_append(msglist, msginfo->header);
				g_free(msginfo->attach_list);
				g_free(msginfo->settings);
				g_free(msginfo);
			}
			g_free(file_path);
			file_path = NULL;
		}
	}
	g_free(path);

	return msglist;
}

gint folder_msgnum_sort_comparefunc(gconstpointer a, gconstpointer b)
{
	gint rv = 0;
	gint msgid_num1 = 0;
	gint msgid_num2 = 0;


	/* since both belong to same account,check msgnum */

	msgid_num1 = get_msgid_num((gchar *) a);
	msgid_num2 = get_msgid_num((gchar *) b);

	if (msgid_num1 < msgid_num2) {
		rv = -1;
	} else {
		rv = 1;
	}
	return rv;
}
static gint folder_move_restored_files(gboolean move_flag)
{
	gint ret = FOLDER_SUCCESS;
	gchar *dest = NULL;
	gchar *config_dir = NULL;
	DIR *dp = NULL;
	struct dirent *d = NULL;

	config_dir = g_strdup_printf("%s%s", get_home_dir(), REST_CONFIG_FILE_PATH);

	if (change_dir(config_dir) < 0) {
		g_free(config_dir);
		return FOLDER_FAIL;
	}
	if ((dp = opendir(".")) == NULL) {
		g_free(config_dir);
		return FOLDER_FAIL;
	}
	d = readdir(dp);

	while (d != NULL) {

		if (dirent_is_regular_file(d)) {

			/* Delete the file,since settings are already saved */
			if (strcmp(d->d_name, ACCOUNT_RC) == 0) {
				if (unlink(d->d_name) < 0) {
					ret = FOLDER_FAIL;
					g_warning("Unable to delete file:%s", d->d_name);
					break;
				}

			} else if ((strcmp(d->d_name, UI_SETTINGS_RC) == 0) || (move_flag == TRUE)) {

				dest =
				    g_strdup_printf("%s%s%c%s", get_home_dir(), CONFIG_FILE_PATH,
						    G_DIR_SEPARATOR, d->d_name);
				if (move_file(d->d_name, dest, TRUE) < 0) {
					ret = FOLDER_FAIL;
				        g_free(dest);
					break;
				}
				g_free(dest);

			} else if (unlink(d->d_name) < 0) {
				ret = FOLDER_FAIL;
				g_warning("Unable to delete file:%s", d->d_name);
				break;
			}
		}
		d = readdir(dp);
	}
	g_free(config_dir);
	closedir(dp);
	return ret;
}
static gint folder_restore_old_settings(const gchar * root)
{
	gchar *restored_path = NULL;
	gboolean settings_only_restored = FALSE;

	if (folder_create_dir_if_not_exist(root, FALSE) == FOLDER_FAIL) {
		return FOLDER_INITIALIZATION_ERROR;
	}
	restored_path = g_strdup_printf("%s%s", get_home_dir(), REST_CONFIG_FILE_PATH);
	/* there is no restored setting */
	if (is_dir_exist(restored_path) == FALSE) {
		g_free(restored_path);
		return FOLDER_SUCCESS;
	}
	g_free(restored_path);

	settings_only_restored = folder_settings_only_restored();

	if (account_update_list() == FALSE) {

		return FOLDER_INITIALIZATION_ERROR;
	}
	if (settings_only_restored == TRUE) {

		/* this removes message specific files in restored directory */
		folder_move_restored_files(FALSE);

	} else {
		/* this moves the files in this case */
		folder_move_restored_files(TRUE);

	}

	return FOLDER_SUCCESS;

}

gboolean folder_settings_only_restored(void)
{
	gchar *cache_file = NULL;
	gboolean settings_only_restored = FALSE;

	/* copy message related settings */
	cache_file = g_strdup_printf("%s%c%s%c%s",
				     get_mail_dir(),
				     G_DIR_SEPARATOR, INBOX_DIR, G_DIR_SEPARATOR, XML_CACHE_FILE);
	/* check if only settings were restored */
	if (is_file_entry_exist(cache_file) == FALSE) {
		settings_only_restored = TRUE;
	}
	g_free(cache_file);
	return settings_only_restored;

}

/**
  This function checks if the message is on archive folder.
  If the message is on archive folder,it is copied to temporary folder.
  @param msgid    pointer to the message information
  @param archive_flag_ptr    pointer to the flag which indicates whether
  message is on archive folder
  @return success(FILE_PROCESS_SUCCESS) or failure(AL_PROCESS_FAIL)
 */
gint check_archive(const gchar * msgid, gboolean * archive_flag_ptr)
{
        gchar *folder_name = NULL;

        folder_name = get_fold_name(msgid);
        if (folder_name == NULL) {
                return FILE_PROCESS_FAIL;
        }
        if (strcmp(folder_name, ARCHIVE) == 0) {
                *archive_flag_ptr = TRUE;
                /* check if MMC is mounted */
                if (CHECK_MMC_FLAG == FALSE) {
                        return FILE_PROCESS_FAIL;
                }
                if (folder_copy_from_mmc(msgid) != FOLDER_SUCCESS) {
                        return FILE_PROCESS_FAIL;
                }
	        } else {
	                *archive_flag_ptr = FALSE;

	        }
	        g_free(folder_name);
	        return FILE_PROCESS_SUCCESS;
}


/**
 This function checks if the message is on archive folder.
 If the message is on archive folder,it is copied to temporary folder.
 @param msgid    pointer to the message information
 @param archive_flag_ptr    pointer to the flag which indicates whether
 message is on archive folder
 @return success(FILE_PROCESS_SUCCESS) or failure(FILE_PROCESS_FAIL)
 */
gint del_arch_file(const gchar * msgid, gboolean archive_flag)
{
        gchar *archive_tmp_file_path = NULL;
        GnomeVFSResult vfs_result;

        if (archive_flag == FALSE) {
             	return FILE_PROCESS_SUCCESS;
        }
        archive_tmp_file_path = g_strdup_printf("%s%c%s%c%s",
                                           	get_mail_dir(),
					        G_DIR_SEPARATOR, ARCHIVE, G_DIR_SEPARATOR, msgid);

        vfs_result = gnome_vfs_unlink(archive_tmp_file_path);

        g_free(archive_tmp_file_path);
        if (vfs_result != GNOME_VFS_OK) {
                return FILE_PROCESS_FAIL;
        }

        return FILE_PROCESS_SUCCESS;
}

/**
 This function updates the mmc folder in the engine
 @param flag    TRUE on mmc mounted , FALSE on mmc unmounted
 @param  volume    newly mounted volume information
 @return FOLDER_SUCCESS(0) on on success,error code on failure
 */
gint update_mmc_flag(gboolean flag, const gchar * volume)
{
        gint ret_val = FOLDER_FAIL;

        if (flag == TRUE) {
                SET_MMC_FLAG;
                debug_print("%s : %s before folder_update_mmc_folder \n", __FILE__, __FUNCTION__);
                ret_val = folder_update_mmc_folder(volume);
                return ret_val;
        } else {
                UNSET_MMC_FLAG;
                debug_print("%s : %s before folder_remove_mmc_folder \n", __FILE__, __FUNCTION__);
                ret_val = folder_remove_mmc_folder();
	        return FOLDER_SUCCESS;
        }
}

