
/*
 * LibSylph -- E-Mail client library
 * Copyright (C) 1999-2006 Hiroyuki Yamamoto
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 */

#ifdef HAVE_CONFIG_H
#  include "config.h"
#endif

#include <stdio.h>
#include <string.h>
#include <errno.h>
#include <libosso.h>
#include <log-functions.h>

#include "common.h"
#include "account.h"
#include "defs.h"
#include "prefs_account.h"
#include "procmsg.h"
#include "utils.h"
#include "prefs.h"
#include "callbacks.h"
#include "folder.h"

GSList *account_list = NULL;
AcctMgmtCallback *acct_mgmt_cb = NULL;


static gint account_is_set_settings_valid(PrefsAccount * prefs_account, const SettingType type);


/**
   This function is called to reset the uid related parameters in the account 
   information if the settings were recovered from restored file.
   UID related parameters are reset only if the settings only were restored.
   @param    prefs_account  
   @return   none 
*/
static void account_adjust_settings_on_check(PrefsAccount * prefs_account);

/**
   This function is called to initialise account module. This function 
   get the information from accountrc file and populate the account list

   @param acct_cb  Call back struct contains the account call back methods
   @return AC_OK on success or return the account module error
*/

gint account_init(AcctMgmtCallback * acct_cb)
{
	gint retval = AC_OK;
	gint rv = AC_OK;

	if (acct_cb == NULL) {
		debug_print("Received account cb callback as NULL\n");
		retval = ACCOUNT_CALLBACK_NOT_REGISTER;
	}
	acct_mgmt_cb = acct_cb;
	if ((rv = account_init_list()) != 0) {
		retval = rv;
	}
	return retval;
}


/**
   This function is used to initialize the global variable account_list. 
   This function reads the account configuration file to initialize all 
   the accounts configured in the email application. Account configuration
   file(accountrc) should be existing /etc/osso/email/ directory.

   Example of accountrc file format :
   [Account: 1]
   account_name=myaccount1
   is_default=1
   userid=user
   passwd=mypassword
   remember_passwd=0
   ...
   ...
   [Account: 2]
   account_name=myaccount2
   is_default=1
   userid=user
   passwd=mypassword
   remember_passwd=0
   ...

   @return AC_OK on success and updates the global account list structure
           (or) account module specified error. 
*/

gint account_init_list(void)
{
	PrefsAccount *ac_prefs = NULL;
	gchar *rcpath = NULL;
	GSList *ac_label_list = NULL, *cur = NULL;
	gchar buf[PREFSBUFSIZE];
	FILE *fp = NULL;
	gboolean rv = FALSE;
	gint retval = AC_OK;
	gboolean restored_file_used = TRUE;

	if (account_list != NULL) {
		account_free_list();
	}

	rcpath =
	    g_strdup_printf("%s%s%c%s", get_home_dir(), REST_CONFIG_FILE_PATH, G_DIR_SEPARATOR,
			    ACCOUNT_RC);

	if (is_file_entry_exist(rcpath) == FALSE) {
		g_free(rcpath);
		rcpath =
		    g_strdup_printf("%s%s%c%s", get_home_dir(), CONFIG_FILE_PATH, G_DIR_SEPARATOR,
				    ACCOUNT_RC);
		restored_file_used = FALSE;
	}
	if ((fp = fopen(rcpath, "rb")) == NULL) {

		if (ENOENT != errno) {
			FILE_OP_ERROR(rcpath, "fopen");
		}
		g_free(rcpath);
		return ACCOUNT_RC_FILE_NOT_EXIST;
	}

	g_free(rcpath);

	memset(buf, '\0', PREFSBUFSIZE);

	while (fgets(buf, sizeof(buf), fp) != NULL) {
		if (strncmp(buf, "[Account: ", 10) == 0) {
			strretchomp(buf);
			memmove(buf, buf + 1, strlen(buf));
			buf[strlen(buf) - 1] = '\0';
			ac_label_list = g_slist_append(ac_label_list, g_strdup(buf));
		}
		memset(buf, '\0', PREFSBUFSIZE);
	}
	fclose(fp);

	for (cur = (GSList *) ac_label_list; cur != NULL; cur = cur->next) {
		ac_prefs = (PrefsAccount *) prefs_account_new();
		rv = prefs_account_read_config(ac_prefs, (gchar *) cur->data);
		if (rv == FALSE) {
			retval = ACOUNT_RC_FILE_READ_FAIL;
			break;
		}
		if (ac_prefs != NULL) {
			account_list = g_slist_append(account_list, ac_prefs);
			if (restored_file_used) {
				account_adjust_settings_on_check(ac_prefs);
			}
		}
	}

	while (ac_label_list) {
		g_free(ac_label_list->data);
		ac_label_list = g_slist_remove(ac_label_list, ac_label_list->data);
	}
	return retval;
}

/**
   This function is get all account information.
   It creates a list of account_name field from all the account lists.
        
   @param GSList account_name_list OUT  contains account list. If this 
          parameter already contains some account list it will be freed first 
          and updated with existing account information.

   @return total number of accounts contains in the returned list. 
*/

gint account_get_list(GSList ** account_name_list)
{
	GSList *cur = NULL;
	PrefsAccount *prefs_account = NULL;
	gchar *account_name = NULL;
	gint total_num = 0;

	if (account_name_list == NULL) {
		ACCT_MGMT_ERROR_CB(ACCOUNT_PREFS_ACCOUNT_LIST_NULL);
		debug_print(" Account name list invalid ...\n");
		return total_num;
	}

	while (*account_name_list) {
		g_free((*account_name_list)->data);
		*account_name_list = g_slist_remove(*account_name_list, (*account_name_list)->data);
	}

	for (cur = account_list; cur != NULL; cur = cur->next) {
		prefs_account = (PrefsAccount *) cur->data;
		if ((prefs_account == NULL) || (prefs_account->user_settings.account_name == NULL)) {
			continue;
		}
		account_name = g_strndup(prefs_account->user_settings.account_name,
					 ACCOUNT_NAME_MAX_LEN);
		(*account_name_list) = g_slist_append((*account_name_list), account_name);
		account_name = NULL;
		total_num++;
	}
	return total_num;
}


/**
   This function is get specified type information of a particular 
   account. The retrived information is updated in supplied prefs account 
   structure.
      
   @param account_name is the name of the account
   @param type  type of settings required.
   @param prefs_account  structure of containing the account 
          settings requested   
   @return TRUE on success or FALSE on failure 
*/

gboolean
account_get_settings(const gchar * account_name,
		     const SettingType type, PrefsAccount ** prefs_account)
{
	PrefsAccount *ac_prefs = NULL;
	GSList *cur = NULL;

	if ((account_name == NULL) || (prefs_account == NULL)) {
		ACCT_MGMT_ERROR_CB(ACCOUNT_INVALID_ACCOUNT_NAME);
		return FALSE;
	}

	if (account_check_for_valid_account((gchar *) account_name) == FALSE) {
		ACCT_MGMT_ERROR_CB(ACCOUNT_INVALID_ACCOUNT_NAME);
		return FALSE;
	}

	for (cur = account_list; cur != NULL; cur = cur->next) {
		ac_prefs = (PrefsAccount *) cur->data;
		if ((ac_prefs)
		    && (ac_prefs->user_settings.account_name != NULL)) {
			if (strncmp
			    (ac_prefs->user_settings.account_name, account_name,
			     ACCOUNT_NAME_MAX_LEN) == 0) {
				break;
			}
		}
		ac_prefs = NULL;
	}

	if (!ac_prefs) {
		ACCT_MGMT_ERROR_CB(ACCOUNT_PREFS_ACCOUNT_NOT_EXIST);
		return FALSE;
	}

	if (*prefs_account == NULL) {
		*prefs_account = g_new0(PrefsAccount, 1);
		if (*prefs_account == NULL) {
			ACCT_MGMT_ERROR_CB(ACCOUNT_PREFS_ACCOUNT_MEM_FAIL);
			return FALSE;
		}
	}

	if (type & USER_SETTING) {
		/* user setting is asked */
		g_free((*prefs_account)->user_settings.account_name);
		(*prefs_account)->user_settings.account_name =
		    g_strndup(ac_prefs->user_settings.account_name, ACCOUNT_NAME_MAX_LEN);
		(*prefs_account)->user_settings.is_default = ac_prefs->user_settings.is_default;
		g_free((*prefs_account)->user_settings.userid);
		(*prefs_account)->user_settings.userid = g_strdup(ac_prefs->user_settings.userid);
		g_free((*prefs_account)->user_settings.passwd);
		(*prefs_account)->user_settings.passwd = g_strdup(ac_prefs->user_settings.passwd);
		(*prefs_account)->user_settings.remember_passwd =
		    ac_prefs->user_settings.remember_passwd;
		g_free((*prefs_account)->user_settings.email);
		(*prefs_account)->user_settings.email = g_strdup(ac_prefs->user_settings.email);
		g_free((*prefs_account)->user_settings.alias_name);
		(*prefs_account)->user_settings.alias_name =
		    g_strdup(ac_prefs->user_settings.alias_name);
	}

	if (type & RECEIVE_SETTING) {
		(*prefs_account)->receive_settings.protocol = ac_prefs->receive_settings.protocol;
		g_free((*prefs_account)->receive_settings.recv_server);
		(*prefs_account)->receive_settings.recv_server =
		    g_strdup(ac_prefs->receive_settings.recv_server);
		(*prefs_account)->receive_settings.port = ac_prefs->receive_settings.port;
		(*prefs_account)->receive_settings.passwd_auth_type =
		    ac_prefs->receive_settings.passwd_auth_type;
		(*prefs_account)->receive_settings.recv_type = ac_prefs->receive_settings.recv_type;
		(*prefs_account)->receive_settings.enable_size_limit =
		    ac_prefs->receive_settings.enable_size_limit;
		(*prefs_account)->receive_settings.size_limit =
		    ac_prefs->receive_settings.size_limit;
		(*prefs_account)->receive_settings.recent_mails =
		    ac_prefs->receive_settings.recent_mails;
		(*prefs_account)->receive_settings.last_uid_validity =
		    ac_prefs->receive_settings.last_uid_validity;
		(*prefs_account)->receive_settings.first_msgid =
		    ac_prefs->receive_settings.first_msgid;
		(*prefs_account)->receive_settings.last_msgid =
		    ac_prefs->receive_settings.last_msgid;
		(*prefs_account)->receive_settings.leave_msg = ac_prefs->receive_settings.leave_msg;
		(*prefs_account)->receive_settings.is_not_editable =
		    ac_prefs->receive_settings.is_not_editable;
		(*prefs_account)->receive_settings.num_of_msgs =
		    ac_prefs->receive_settings.num_of_msgs;

	}

	if (type & SEND_SETTING) {
		g_free((*prefs_account)->send_settings.smtp_server);
		(*prefs_account)->send_settings.smtp_server =
		    g_strdup(ac_prefs->send_settings.smtp_server);
		(*prefs_account)->send_settings.smtp_auth_type =
		    ac_prefs->send_settings.smtp_auth_type;
		(*prefs_account)->send_settings.smtp_port = ac_prefs->send_settings.smtp_port;
		g_free((*prefs_account)->send_settings.username);
		(*prefs_account)->send_settings.username =
		    g_strdup(ac_prefs->send_settings.username);
		g_free((*prefs_account)->send_settings.password);
		(*prefs_account)->send_settings.password =
		    g_strdup(ac_prefs->send_settings.password);
	}

	if (type & ADVANCED_SETTING) {
		(*prefs_account)->advanced_settings.email_type =
		    ac_prefs->advanced_settings.email_type;
		(*prefs_account)->advanced_settings.email_sending =
		    ac_prefs->advanced_settings.email_sending;
		(*prefs_account)->advanced_settings.include_signature =
		    ac_prefs->advanced_settings.include_signature;
		g_free((*prefs_account)->advanced_settings.signature_file);
		(*prefs_account)->advanced_settings.signature_file =
		    g_strdup(ac_prefs->advanced_settings.signature_file);
		(*prefs_account)->advanced_settings.request_read_report =
		    ac_prefs->advanced_settings.request_read_report;
		(*prefs_account)->advanced_settings.copy_to_own_address =
		    ac_prefs->advanced_settings.copy_to_own_address;
		(*prefs_account)->advanced_settings.org_in_reply =
		    ac_prefs->advanced_settings.org_in_reply;
		(*prefs_account)->advanced_settings.use_ssl = ac_prefs->advanced_settings.use_ssl;
		(*prefs_account)->advanced_settings.use_tls = ac_prefs->advanced_settings.use_tls;
		(*prefs_account)->advanced_settings.smtp_use_ssl =
		    ac_prefs->advanced_settings.smtp_use_ssl;
		(*prefs_account)->advanced_settings.smtp_use_tls =
		    ac_prefs->advanced_settings.smtp_use_tls;
		g_free((*prefs_account)->advanced_settings.imap_dir);
		(*prefs_account)->advanced_settings.imap_dir =
		    g_strdup(ac_prefs->advanced_settings.imap_dir);
#ifdef CL_USE
		g_free((*prefs_account)->advanced_settings.iap_name);
		(*prefs_account)->advanced_settings.iap_name =
		    g_strdup(ac_prefs->advanced_settings.iap_name);
#endif
		(*prefs_account)->advanced_settings.use_iap_smtp_serv =
		    ac_prefs->advanced_settings.use_iap_smtp_serv;
		(*prefs_account)->advanced_settings.smime_security =
		    ac_prefs->advanced_settings.smime_security;
		(*prefs_account)->advanced_settings.display_sign_encrypt =
		    ac_prefs->advanced_settings.display_sign_encrypt;
		(*prefs_account)->advanced_settings.display_trust =
		    ac_prefs->advanced_settings.display_trust;
		(*prefs_account)->advanced_settings.display_validity =
		    ac_prefs->advanced_settings.display_validity;

	}
	return TRUE;
}


/**
    This function is get the account information based on account name 

     @param account_name name of the account
     @return with account information on success or NULL on failure
*/

PrefsAccount *account_get_account(const gchar * account_name)
{
	PrefsAccount *ac_prefs = NULL;
	GSList *cur = NULL;

        osso_log(LOG_DEBUG, "%s called with account_name=%s",
                        __FUNCTION__, account_name);

	if ((account_name == NULL) || (account_check_for_valid_account(account_name) == FALSE)) {
		return NULL;
	}

	for (cur = account_list; cur != NULL; cur = cur->next) {
		ac_prefs = (PrefsAccount *) cur->data;
		if ((ac_prefs)
		    && (ac_prefs->user_settings.account_name != NULL)) {
			if (strncmp
			    (ac_prefs->user_settings.account_name, account_name,
			     ACCOUNT_NAME_MAX_LEN) == 0) {
                                osso_log(LOG_DEBUG, "found account %s",
                                                account_name);
				return ac_prefs;
                        }
		}
	}
	return NULL;
}


/**
    This function is get the account information based on account name 
    Incase of account not exist error will be returned through callbak. 

    @param account_name IN name of the account
    @return with account information on success or NULL on failure
*/

PrefsAccount *account_get_account_cb(const gchar * account_name)
{
	PrefsAccount *ac_prefs = NULL;
	ac_prefs = account_get_account(account_name);

	if (ac_prefs == NULL) {
		ACCT_MGMT_ERROR_CB(ACCOUNT_PREFS_ACCOUNT_NOT_EXIST);
	}
	return ac_prefs;
}


/**
   This function is to check for valid settings in the account 

   @param prefs_account IN  structure of containing the account settings   
   @param type  type of settings required.
   @return errorcode on failure or AC_OK on success 
*/

static gint account_is_set_settings_valid(PrefsAccount * prefs_account, const SettingType type)
{
	if (prefs_account == NULL) {
		return ACCOUNT_INVALID_ACCOUNT_NAME;
	}

	if (type & USER_SETTING) {
		if ((prefs_account->user_settings.account_name == NULL) ||
		    (account_check_for_valid_account
		     (prefs_account->user_settings.account_name) == FALSE)) {
			return ACCOUNT_INVALID_ACCOUNT_NAME;
		}
		if (prefs_account->user_settings.email == NULL) {
			return ACCOUNT_EMAIL_NULL;
		}
	}
	if (type & RECEIVE_SETTING) {
		if ((prefs_account->receive_settings.protocol != PROTO_POP3) &&
		    (prefs_account->receive_settings.protocol != PROTO_IMAP4)) {
			return ACCOUNT_PROTOCOL_INVALID;
		}
		if (prefs_account->receive_settings.recv_server == NULL) {
			return ACCOUNT_RECV_SERVER_NULL;
		}
		if ((prefs_account->receive_settings.recv_type != HEADERS) &&
		    (prefs_account->receive_settings.recv_type != BODY) &&
		    (prefs_account->receive_settings.recv_type != BODY_ATTCH)) {
			return ACCOUNT_RECV_TYPE_INVALID;
		}
	}
	if (type & SEND_SETTING) {
		if (prefs_account->send_settings.smtp_server == NULL) {
			return ACCOUNT_SMTP_SERVER_NULL;
		}
		if ((prefs_account->send_settings.smtp_auth_type != SMTPAUTH_LOGIN)
		    && (prefs_account->send_settings.smtp_auth_type != SMTPAUTH_CRAM_MD5)
		    && (prefs_account->send_settings.smtp_auth_type != SMTPAUTH_NO_AUTH)) {
			return ACCOUNT_SMTP_AUTH_TYPE_INVALID;
		}
	}
	return AC_OK;
}

/**
   This function is set specified type information of a particular account. 
   The retrived information is updated in supplied prefs account structure.
   This function also updates the account resource file with the new 
   settings. 
 
   @param account_name  name of the account
   @param type  type of settings required.
   @param prefs_account  structure of containing the account settings   
   @return TRUE on success or FALSE on failure 
*/

gboolean
account_set_settings(const gchar * account_name,
		     const SettingType type, PrefsAccount * prefs_account)
{

	PrefsAccount *ac_prefs = NULL;
	GSList *cur = NULL;
	gint  rv = AC_OK;
	gboolean user_info_changed = FALSE;
	gboolean server_info_changed = FALSE;

	if ((account_name == NULL) || (prefs_account == NULL) ||
	    (account_check_for_valid_account(account_name) == FALSE)) {
		ACCT_MGMT_ERROR_CB(ACCOUNT_INVALID_ACCOUNT_NAME);
		return FALSE;
	}

	if ((rv = account_is_set_settings_valid(prefs_account, type)) != AC_OK) {
		ACCT_MGMT_ERROR_CB(rv);
		return FALSE;
	}

	for (cur = account_list; cur != NULL; cur = cur->next) {
		ac_prefs = (PrefsAccount *) cur->data;
		if ((ac_prefs)
		    && (ac_prefs->user_settings.account_name != NULL)) {
			if (strncmp
			    (ac_prefs->user_settings.account_name, account_name,
			     ACCOUNT_NAME_MAX_LEN) == 0) {
				g_free(ac_prefs->user_settings.account_name);
				ac_prefs->user_settings.account_name = g_strdup(account_name);
				break;
			}
		}
		ac_prefs = NULL;
	}


	/* We didn't find the account */
	if (ac_prefs == NULL) {
		/* A new account has to be created */
		/* Add the prefs_account to global list */
		ac_prefs = g_new0(PrefsAccount, 1);
		if (ac_prefs == NULL) {
			ACCT_MGMT_ERROR_CB(ACCOUNT_PREFS_ACCOUNT_MEM_FAIL);
			return FALSE;
		}
		ac_prefs->user_settings.account_name =
		    g_strndup(account_name, ACCOUNT_NAME_MAX_LEN);
		account_list = g_slist_append(account_list, ac_prefs);

	}
	/* User Settings */
	if (type & USER_SETTING) {
		ac_prefs->user_settings.is_default = prefs_account->user_settings.is_default;
		if (ac_prefs->user_settings.userid != NULL) {
			if (strcmp(ac_prefs->user_settings.userid,
				   prefs_account->user_settings.userid) != 0) {
				user_info_changed = TRUE;
			}
		}
		g_free(ac_prefs->user_settings.userid);
		ac_prefs->user_settings.userid = g_strdup(prefs_account->user_settings.userid);

		g_free(ac_prefs->user_settings.passwd);
		ac_prefs->user_settings.passwd = g_strdup(prefs_account->user_settings.passwd);

		ac_prefs->user_settings.remember_passwd =
		    prefs_account->user_settings.remember_passwd;

		g_free(ac_prefs->user_settings.email);
		ac_prefs->user_settings.email = g_strdup(prefs_account->user_settings.email);

		g_free(ac_prefs->user_settings.alias_name);
		ac_prefs->user_settings.alias_name =
		    g_strdup(prefs_account->user_settings.alias_name);
	}

	if (type & RECEIVE_SETTING) {
		ac_prefs->receive_settings.protocol = prefs_account->receive_settings.protocol;
		if (ac_prefs->receive_settings.recv_server != NULL) {
			if (strcmp(ac_prefs->receive_settings.recv_server,
				   prefs_account->receive_settings.recv_server) != 0) {
				server_info_changed = TRUE;
			}
		}
		g_free(ac_prefs->receive_settings.recv_server);
		ac_prefs->receive_settings.recv_server =
		    g_strdup(prefs_account->receive_settings.recv_server);
		ac_prefs->receive_settings.port = prefs_account->receive_settings.port;
		ac_prefs->receive_settings.passwd_auth_type =
		    prefs_account->receive_settings.passwd_auth_type;
		ac_prefs->receive_settings.recv_type = prefs_account->receive_settings.recv_type;
		ac_prefs->receive_settings.enable_size_limit =
		    prefs_account->receive_settings.enable_size_limit;
		ac_prefs->receive_settings.size_limit = prefs_account->receive_settings.size_limit;
		/* reset these only on change in server info */
		if ((user_info_changed == TRUE) || (server_info_changed == TRUE)) {
			ac_prefs->receive_settings.first_msgid = 0;
			ac_prefs->receive_settings.last_msgid = 0;
			/* remove pop3 uidl file if it exists */
			if (ac_prefs->receive_settings.protocol == PROTO_POP3) {
				account_remove_uidl_file(ac_prefs);
			}
		}
		ac_prefs->receive_settings.recent_mails =
		    prefs_account->receive_settings.recent_mails;
		ac_prefs->receive_settings.leave_msg = prefs_account->receive_settings.leave_msg;
		ac_prefs->receive_settings.is_not_editable =
		    prefs_account->receive_settings.is_not_editable;

	}

	if (type & SEND_SETTING) {
		g_free(ac_prefs->send_settings.smtp_server);
		ac_prefs->send_settings.smtp_server =
		    g_strdup(prefs_account->send_settings.smtp_server);
		ac_prefs->send_settings.smtp_auth_type =
		    prefs_account->send_settings.smtp_auth_type;
		ac_prefs->send_settings.smtp_port = prefs_account->send_settings.smtp_port;
		g_free(ac_prefs->send_settings.username);
		ac_prefs->send_settings.username = g_strdup(prefs_account->send_settings.username);
		g_free(ac_prefs->send_settings.password);
		ac_prefs->send_settings.password = g_strdup(prefs_account->send_settings.password);
	}

	if (type & ADVANCED_SETTING) {
		ac_prefs->advanced_settings.email_type =
		    prefs_account->advanced_settings.email_type;
		ac_prefs->advanced_settings.email_sending =
		    prefs_account->advanced_settings.email_sending;
		ac_prefs->advanced_settings.include_signature =
		    prefs_account->advanced_settings.include_signature;
		g_free(ac_prefs->advanced_settings.signature_file);
		ac_prefs->advanced_settings.signature_file =
		    g_strdup(prefs_account->advanced_settings.signature_file);
		ac_prefs->advanced_settings.signature_flag =
		    prefs_account->advanced_settings.signature_flag;
		ac_prefs->advanced_settings.request_read_report =
		    prefs_account->advanced_settings.request_read_report;
		ac_prefs->advanced_settings.copy_to_own_address =
		    prefs_account->advanced_settings.copy_to_own_address;
		ac_prefs->advanced_settings.org_in_reply =
		    prefs_account->advanced_settings.org_in_reply;
		ac_prefs->advanced_settings.use_ssl = prefs_account->advanced_settings.use_ssl;
		ac_prefs->advanced_settings.use_tls = prefs_account->advanced_settings.use_tls;
		ac_prefs->advanced_settings.smtp_use_ssl =
		    prefs_account->advanced_settings.smtp_use_ssl;
		ac_prefs->advanced_settings.smtp_use_tls =
		    prefs_account->advanced_settings.smtp_use_tls;
		g_free(ac_prefs->advanced_settings.imap_dir);
		ac_prefs->advanced_settings.imap_dir =
		    g_strdup(prefs_account->advanced_settings.imap_dir);
#if CL_USE
		g_free(ac_prefs->advanced_settings.iap_name);
		ac_prefs->advanced_settings.iap_name =
		    g_strdup(prefs_account->advanced_settings.iap_name);
#endif
		ac_prefs->advanced_settings.use_iap_smtp_serv =
		    prefs_account->advanced_settings.use_iap_smtp_serv;
		ac_prefs->advanced_settings.smime_security =
		    prefs_account->advanced_settings.smime_security;
		ac_prefs->advanced_settings.display_sign_encrypt =
		    prefs_account->advanced_settings.display_sign_encrypt;
		ac_prefs->advanced_settings.display_trust =
		    prefs_account->advanced_settings.display_trust;
		ac_prefs->advanced_settings.display_validity =
		    prefs_account->advanced_settings.display_validity;
	}
	return prefs_account_save_config_all(account_list);

}

/**
   This function is to delete the account from account list 
   This function also updates the account resource file with the new 
   settings. 
      
   @param account_name IN name of the account
   @return TRUE on success or FALSE on failure 
*/

gboolean account_delete(const gchar * account_name)
{

	PrefsAccount *ac_prefs = NULL;
	GSList *cur = NULL;
	if (account_name == NULL)
		return FALSE;

	for (cur = account_list; cur != NULL; cur = cur->next) {
		ac_prefs = (PrefsAccount *) cur->data;
		if ((ac_prefs)
		    && (ac_prefs->user_settings.account_name != NULL)) {
			if (strncmp
			    (ac_prefs->user_settings.account_name, account_name,
			     ACCOUNT_NAME_MAX_LEN)
			    == 0)
				break;
		}
		ac_prefs = NULL;
	}

	if (ac_prefs != NULL) {
		/* remove pop3 uidl file if it exists */
		if (ac_prefs->receive_settings.protocol == PROTO_POP3) {
			account_remove_uidl_file(ac_prefs);
		}
		account_list = g_slist_remove(account_list, ac_prefs);
		account_free_account(ac_prefs);
	} else {
		return FALSE;
	}

	/* And save the configuration into file */
	return prefs_account_save_config_all(account_list);
}

/**
    This function is rename the existing account with the new name.
    The input parameters old_name, new_name shouldn't contain the 
    special character "_".
      
    @param old_name  name of the existing account
    @param new_name  name of the account to be renamed
    @return TRUE on success or FALSE on failure 
*/

gboolean account_rename(const gchar * old_name, const gchar * new_name)
{
	PrefsAccount *ac_prefs = NULL;
	GSList *cur = NULL;

	if ((old_name == NULL) || (new_name == NULL)) {
		ACCT_MGMT_ERROR_CB(ACCOUNT_INVALID_ACCOUNT_NAME);
		return FALSE;
	}

	if (account_check_for_valid_account((gchar *) old_name) == FALSE) {
		ACCT_MGMT_ERROR_CB(ACCOUNT_INVALID_ACCOUNT_NAME);
		return FALSE;
	}

	if (account_check_for_valid_account((gchar *) new_name) == FALSE) {
		ACCT_MGMT_ERROR_CB(ACCOUNT_INVALID_ACCOUNT_NAME);
		return FALSE;
	}

	for (cur = account_list; cur != NULL; cur = cur->next) {
		ac_prefs = (PrefsAccount *) cur->data;
		if ((ac_prefs)
		    && (ac_prefs->user_settings.account_name != NULL)) {
			if (strcmp(ac_prefs->user_settings.account_name, old_name) == 0) {
				break;
			}
		}
		ac_prefs = NULL;
	}

	if (!ac_prefs) {
		ACCT_MGMT_ERROR_CB(ACCOUNT_PREFS_ACCOUNT_NOT_EXIST);
		return FALSE;
	}

	g_free(ac_prefs->user_settings.account_name);
	ac_prefs->user_settings.account_name = g_strdup(new_name);

	return prefs_account_save_config_all(account_list);
}

/**
   This function is get the default account  from the account list
  
   @return prefsAccount struct or NULL incase of no default account 
*/
PrefsAccount *account_get_default_account(void)
{
	PrefsAccount *ac_prefs = NULL;
	GSList *cur = NULL;
	if (account_list == NULL) {
		return NULL;
	}
	for (cur = account_list; cur != NULL; cur = cur->next) {
		ac_prefs = (PrefsAccount *) cur->data;
		if (ac_prefs->user_settings.is_default == TRUE) {
			break;
		}
		ac_prefs = NULL;
	}
	return ac_prefs;
}

/**
   This function is rename the existing account. 
     
   @param ac_prefs prefsAccount struct to be freed 
   @return TRUE on success or FALSE on account not exist 
*/
gboolean account_free_account(PrefsAccount * ac_prefs)
{

	debug_print("[%s]:[%s]  starts here...\n", __FILE__, __FUNCTION__);
	if (ac_prefs == NULL) {
		debug_print("[%s]:[%d] Account can't be freed ...\n", __FILE__, __LINE__);
		return FALSE;
	}
	g_free(ac_prefs->user_settings.account_name);
	g_free(ac_prefs->user_settings.userid);
	g_free(ac_prefs->user_settings.passwd);
	g_free(ac_prefs->user_settings.email);
	g_free(ac_prefs->user_settings.alias_name);
	if (ac_prefs->user_settings.session) {
		session_destroy(ac_prefs->user_settings.session);
		ac_prefs->user_settings.session = NULL;
	}

	g_free(ac_prefs->receive_settings.recv_server);
	g_free(ac_prefs->send_settings.smtp_server);
	g_free(ac_prefs->send_settings.username);
	g_free(ac_prefs->send_settings.password);

	g_free(ac_prefs->advanced_settings.signature_file);
	g_free(ac_prefs->advanced_settings.imap_dir);
#if CL_USE
	g_free(ac_prefs->advanced_settings.iap_name);
#endif
	g_free(ac_prefs);
	ac_prefs = NULL;

	return TRUE;
}

/**
   This function is to free the account list 
     
   @return TRUE on success or FALSE on account list is NULL 
*/

gboolean account_free_list(void)
{

	while (account_list) {
		account_free_account(account_list->data);
		account_list = g_slist_remove(account_list, account_list->data);
	}
	account_list = NULL;
	return TRUE;
}

/**
   This function is to display the account list 
     
   @return TRUE on success or FALSE on account list is NULL 
*/

gboolean account_display_list(void)
{

	PrefsAccount *ac_prefs = NULL;
	GSList *cur = NULL;

	if (account_list == NULL) {
		return FALSE;
	}
	for (cur = account_list; cur != NULL; cur = cur->next) {
		ac_prefs = (PrefsAccount *) cur->data;
		printf(" Account Name : %s \n", ac_prefs->user_settings.account_name);
	}
	return TRUE;
}

/**
   This function is to validate the account name 
   Incase of account contains special character "_", account 
   becomes invalid.

   @param account_name name of the account to verify 
     
   @return TRUE on success or FALSE on account not exist 
*/

gboolean account_check_for_valid_account(const gchar * account_name)
{

	gchar *tmp = NULL;
	if (account_name == NULL) {
		return FALSE;
	}
	tmp = strchr(account_name, MSGID_SEPARATOR);
	if (tmp != NULL) {
		debug_print(" Invalid Account Name %s \n", account_name);
		return FALSE;
	}
	return TRUE;
}

/**
   This function is to called to free the global memory allocated 
   to account module.
     
   @return TRUE on success or FALSE incase of account list is NULL 
*/

gboolean account_cleanup(void)
{
	g_free(acct_mgmt_cb);
	acct_mgmt_cb = NULL;
	return account_free_list();
}

/**
   This function is called to get the dummy account 
     
   @return Dummy account or NULL in case of any valid account exist 
*/

PrefsAccount *account_get_dummy_account(void)
{
	static gint dunmmy_not_defined = FALSE;
	static PrefsAccount dummy_account;
	/* dummy account not required when accountrc file contains 
	 * atleast one account */
	if (account_list != NULL) {
		return NULL;
	}
	if (dunmmy_not_defined == FALSE) {
		debug_print("Create Dummy account...\n");
		memset(&dummy_account, '\0', sizeof(dummy_account));

		/* User Settings */
		dummy_account.user_settings.account_name = AC_DEFAULT;
		dummy_account.user_settings.is_default = TRUE;
		dummy_account.user_settings.userid = AC_DEFAULT;
		dummy_account.user_settings.passwd = AC_DEFAULT;
		dummy_account.user_settings.email = AC_DEFAULT_EMAIL;

		/* Receive Settings */
		dummy_account.receive_settings.protocol = PROTO_IMAP4;
		dummy_account.receive_settings.recv_server = AC_DEFAULT;
		dummy_account.receive_settings.port = AC_DEFAULT_PORT;
		dummy_account.receive_settings.passwd_auth_type = AUTH_LOGIN;

		/*Send Settings */
		dummy_account.send_settings.smtp_server = AC_DEFAULT;
		dummy_account.send_settings.smtp_auth_type = SMTPAUTH_NO_AUTH;
		dummy_account.send_settings.username = AC_DEFAULT;
		dummy_account.send_settings.password = AC_DEFAULT;

		/*Advanced Settings */
		dummy_account.advanced_settings.email_type = EMAIL_TEXT;
		dummy_account.advanced_settings.email_sending = SEND_IMMEDIATE;
		dunmmy_not_defined = TRUE;
	}
	return &dummy_account;
}

/**
   This function writes the account list to the file
   
   @param account_list  list of accounts to be updated in accountrc file
   @return TRUE on successfully updating the account list in accountrc
           file (or) return FALSE
*/
gboolean account_update_list(void)
{
	return prefs_account_save_config_all(account_list);
}

/**
        This function gets the default account name.
	If default account is not present,then returns "default" 
        @param none
        @return account name to use as default 
*/
gchar *account_get_default_account_name(void)
{
	PrefsAccount *prefs_account = NULL;
	gchar *default_account_name = NULL;

	prefs_account = account_get_default_account();

	if ((prefs_account != NULL)
	    && (prefs_account->user_settings.account_name != NULL)) {
		default_account_name = g_strdup(prefs_account->user_settings.account_name);
		debug_print("Default accnt found by account\n");
	} else {
		debug_print("Account not found. Dummy account set as default\n");
		default_account_name = g_strdup("default");
	}
	return default_account_name;
}


GSList *account_get_prefs_account_list(void)
{
	return account_list;
}

void account_remove_uidl_file(PrefsAccount * ac_prefs)
{
	gchar *path = NULL;

	path = g_strdup_printf("%s%c%s%s", get_rc_dir(), G_DIR_SEPARATOR,
			       ac_prefs->user_settings.account_name, UID_EXT);

	if (is_file_entry_exist(path) == TRUE) {
		unlink(path);
	}
	g_free(path);
}
void account_rename_uidl_file(const gchar * old_name, const gchar * new_name)
{
	gchar *path = NULL;
	gchar *newpath = NULL;

	path = g_strdup_printf("%s%c%s%s", get_rc_dir(), G_DIR_SEPARATOR, old_name, UID_EXT);
	newpath = g_strdup_printf("%s%c%s%s", get_rc_dir(), G_DIR_SEPARATOR, new_name, UID_EXT);

	if (is_file_entry_exist(path) == TRUE) {
		move_file(path, newpath, TRUE);
	}
	g_free(path);
	g_free(newpath);
}

PrefsAccount *account_copy_prefs(PrefsAccount * s_update_prefsAccount)
{
	PrefsAccount *prefsaccount_temp = NULL;
	if (s_update_prefsAccount == NULL) {
		debug_print("Input param check failed\n");
		return NULL;
	}

	prefsaccount_temp = g_new0(PrefsAccount, 1);
	if (s_update_prefsAccount->user_settings.account_name != NULL) {
		prefsaccount_temp->user_settings.account_name =
		    g_strdup(s_update_prefsAccount->user_settings.account_name);
	} else {
		prefsaccount_temp->user_settings.account_name = NULL;
	}
	if (s_update_prefsAccount->user_settings.userid != NULL) {
		prefsaccount_temp->user_settings.userid =
		    g_strdup(s_update_prefsAccount->user_settings.userid);
	} else {
		prefsaccount_temp->user_settings.userid = NULL;
	}
	if (s_update_prefsAccount->user_settings.passwd != NULL) {
		prefsaccount_temp->user_settings.passwd =
		    g_strdup(s_update_prefsAccount->user_settings.passwd);
	} else {
		prefsaccount_temp->user_settings.passwd = NULL;
	}
	if (s_update_prefsAccount->send_settings.username != NULL) {
		prefsaccount_temp->send_settings.username =
		    g_strdup(s_update_prefsAccount->send_settings.username);
	} else {
		prefsaccount_temp->send_settings.username = NULL;
	}
	if (s_update_prefsAccount->send_settings.password != NULL) {
		prefsaccount_temp->send_settings.password =
		    g_strdup(s_update_prefsAccount->send_settings.password);
	} else {
		prefsaccount_temp->send_settings.password = NULL;
	}
	if (s_update_prefsAccount->user_settings.email != NULL) {
		prefsaccount_temp->user_settings.email =
		    g_strdup(s_update_prefsAccount->user_settings.email);
	} else {
		prefsaccount_temp->user_settings.email = NULL;
	}
	if (s_update_prefsAccount->user_settings.alias_name != NULL) {
		prefsaccount_temp->user_settings.alias_name =
		    g_strdup(s_update_prefsAccount->user_settings.alias_name);
	} else {
		prefsaccount_temp->user_settings.alias_name = NULL;
	}
	if (s_update_prefsAccount->receive_settings.recv_server != NULL) {
		prefsaccount_temp->receive_settings.recv_server =
		    g_strdup(s_update_prefsAccount->receive_settings.recv_server);
	} else {
		prefsaccount_temp->receive_settings.recv_server = NULL;
	}
	if (s_update_prefsAccount->send_settings.smtp_server != NULL) {
		prefsaccount_temp->send_settings.smtp_server =
		    g_strdup(s_update_prefsAccount->send_settings.smtp_server);
	} else {
		prefsaccount_temp->send_settings.smtp_server = NULL;
	}
	if (s_update_prefsAccount->advanced_settings.signature_file != NULL) {
		prefsaccount_temp->advanced_settings.signature_file =
		    g_strdup(s_update_prefsAccount->advanced_settings.signature_file);
	} else {
		prefsaccount_temp->advanced_settings.signature_file = NULL;
	}
#if CL_USE
	if (s_update_prefsAccount->advanced_settings.iap_name != NULL) {
		prefsaccount_temp->advanced_settings.iap_name =
		    g_strdup(s_update_prefsAccount->advanced_settings.iap_name);
	} else {
		prefsaccount_temp->advanced_settings.iap_name = NULL;
	}
#endif
	prefsaccount_temp->receive_settings.protocol =
	    s_update_prefsAccount->receive_settings.protocol;
	prefsaccount_temp->advanced_settings.email_type =
	    s_update_prefsAccount->advanced_settings.email_type;
	prefsaccount_temp->advanced_settings.email_sending =
	    s_update_prefsAccount->advanced_settings.email_sending;
	prefsaccount_temp->advanced_settings.request_read_report =
	    s_update_prefsAccount->advanced_settings.request_read_report;
	prefsaccount_temp->receive_settings.port = s_update_prefsAccount->receive_settings.port;
	prefsaccount_temp->user_settings.is_default =
	    s_update_prefsAccount->user_settings.is_default;
	prefsaccount_temp->receive_settings.recv_type =
	    s_update_prefsAccount->receive_settings.recv_type;
	prefsaccount_temp->receive_settings.size_limit =
	    s_update_prefsAccount->receive_settings.size_limit;
	prefsaccount_temp->receive_settings.passwd_auth_type =
	    s_update_prefsAccount->receive_settings.passwd_auth_type;
	prefsaccount_temp->receive_settings.recent_mails =
	    s_update_prefsAccount->receive_settings.recent_mails;
	prefsaccount_temp->advanced_settings.copy_to_own_address =
	    s_update_prefsAccount->advanced_settings.copy_to_own_address;
	prefsaccount_temp->advanced_settings.use_ssl =
	    s_update_prefsAccount->advanced_settings.use_ssl;
	prefsaccount_temp->advanced_settings.use_tls =
	    s_update_prefsAccount->advanced_settings.use_tls;
	prefsaccount_temp->advanced_settings.smtp_use_ssl =
	    s_update_prefsAccount->advanced_settings.smtp_use_ssl;
	prefsaccount_temp->advanced_settings.smtp_use_tls =
	    s_update_prefsAccount->advanced_settings.smtp_use_tls;
	prefsaccount_temp->receive_settings.enable_size_limit =
	    s_update_prefsAccount->receive_settings.enable_size_limit;
	prefsaccount_temp->send_settings.smtp_auth_type =
	    s_update_prefsAccount->send_settings.smtp_auth_type;
	prefsaccount_temp->advanced_settings.include_signature =
	    s_update_prefsAccount->advanced_settings.include_signature;
	prefsaccount_temp->advanced_settings.org_in_reply =
	    s_update_prefsAccount->advanced_settings.org_in_reply;
	prefsaccount_temp->send_settings.smtp_auth_type =
	    s_update_prefsAccount->send_settings.smtp_auth_type;
	prefsaccount_temp->send_settings.smtp_port = s_update_prefsAccount->send_settings.smtp_port;
	prefsaccount_temp->advanced_settings.smime_security =
	    s_update_prefsAccount->advanced_settings.smime_security;
	prefsaccount_temp->advanced_settings.display_sign_encrypt =
	    s_update_prefsAccount->advanced_settings.display_sign_encrypt;
	prefsaccount_temp->advanced_settings.display_trust =
	    s_update_prefsAccount->advanced_settings.display_trust;
	prefsaccount_temp->advanced_settings.display_validity =
	    s_update_prefsAccount->advanced_settings.display_validity;
	prefsaccount_temp->advanced_settings.signature_flag =
	    s_update_prefsAccount->advanced_settings.signature_flag;
	prefsaccount_temp->receive_settings.leave_msg =
	    s_update_prefsAccount->receive_settings.leave_msg;

	prefsaccount_temp->receive_settings.is_not_editable =
	    s_update_prefsAccount->receive_settings.is_not_editable;
	prefsaccount_temp->receive_settings.enable_size_limit = 1;

	return prefsaccount_temp;
}
static void account_adjust_settings_on_check(PrefsAccount * prefs_account)
{
	/* adjust uid related data if only settings were restored */
	if (folder_settings_only_restored() == TRUE) {
		prefs_account->receive_settings.last_uid_validity = 0;
		prefs_account->receive_settings.last_msgid = 0;
		prefs_account->receive_settings.first_msgid = 0;
		prefs_account->receive_settings.num_of_msgs = 0;
	}

}
gboolean is_account_list_valid(void)
{
	if (account_list == NULL) {
		return FALSE;
	}
	return TRUE;
}
