
/**
 * This file is part of osso-email-common
 *
 * Copyright (c) 2006 Nokia Corporation.
 * 
 * Contact: Dirk-Jan Binnema <dirk-jan.binnema@nokia.com>
 * 
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public License
 * version 2.1 as published by the Free Software Foundation.
 * 
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA
 * 02110-1301 USA
 */


#include <sys/types.h>
#include <sys/stat.h>
#include <errno.h>
#include <stdio.h>
#include <string.h>
#include <libosso.h>
#include <log-functions.h>
#include "utils.h"
#include "defs.h"
#include  "iap_confserv.h"
#include "common.h"
#include "dbus_utils.h"
#include "dbus_common_utils.h"
#include "dbus_utils_xml.h"

#if CL_USE

static GSList *gl_iap_confserv_list; 

static void get_servinfo( IapInfo *iap_info, void *data);

/**
   This method is used to get the locally stored iap list 
   @param none 
   @return list of iap configured servers 
*/
GSList *get_iap_confserv_list(void) 
{
	return gl_iap_confserv_list;
}

/**
   This method is used duplicate the iap info 
   @param iapinfo contains the iap specific smtp server data  
   @return duplicate of iapinfo struct   
*/

static IapInfo *dup_iap_serv_data( const IapInfo *src) 
{
	IapInfo *des = NULL;

	if(!src) {
		return NULL;
	}
	des = g_new0(IapInfo,1);
	des->iap_name = g_strdup(src->iap_name);
	if(src->smtp_servname) {
		des->smtp_servname = g_strdup(src->smtp_servname);
	}
	if(src->smtp_username) {
		des->smtp_username = g_strdup(src->smtp_username);
	}
	if(src->smtp_passwd) {
		des->smtp_passwd = g_strdup(src->smtp_passwd);
	}
	des->smtp_port = src->smtp_port; 
	des->smtp_auth_type = src->smtp_auth_type; 
	des->smtp_use_ssl = src->smtp_use_ssl; 
	des->smtp_use_tls = src->smtp_use_tls; 
	return des;
}

/**
   This method is used to get the iap server info 
   @param iapinfo which is populated by from locally maintained data 
   @param data additional info ( here nothing is supplied) 
   @return duplicate of iapinfo struct   
*/
static void get_servinfo( IapInfo *iap_info, void *data)
{
	GSList *tlist = NULL; 
	IapInfo *tiapinfo = NULL;

	if(!iap_info) {
		return;
	}
	if(!(iap_info->iap_name)) {
		return ;
	}
	for( tlist = gl_iap_confserv_list; tlist != NULL ; tlist = tlist->next) {
		tiapinfo = (IapInfo *)tlist->data;
		if(!tiapinfo) {
			continue;
		}
		if(tiapinfo->iap_name) {
			if( compare_utf8_strings( tiapinfo->iap_name, iap_info->iap_name) == 0) {
				if(tiapinfo->smtp_servname) {
					iap_info->smtp_servname = g_strdup(tiapinfo->smtp_servname);
				}
				if(tiapinfo->smtp_username) {
					iap_info->smtp_username = g_strdup(tiapinfo->smtp_username);
				}	
				if(tiapinfo->smtp_passwd) {
					iap_info->smtp_passwd = g_strdup(tiapinfo->smtp_passwd);
				}
				iap_info->smtp_port      = tiapinfo->smtp_port; 
				iap_info->smtp_auth_type = tiapinfo->smtp_auth_type; 
				iap_info->smtp_use_ssl   = tiapinfo->smtp_use_ssl; 
				iap_info->smtp_use_tls   = tiapinfo->smtp_use_tls; 
				break;
			}
		}	
	}	
	return ;
}

/**
   This method is used to sync up the iap list with locally maintained data 
   @param iaplist list of iapinfo  
   @return nothing 
*/
void iap_confsmtp_sync_serv_info_list(GSList *iaplist) 
{
	GSList *tlist = NULL;
	IapInfo *tinfo = NULL, *linfo = NULL;
	/* Free the old data and update with new data */	
	cl_iap_info_list_free(gl_iap_confserv_list);
	gl_iap_confserv_list = NULL;
	
	for(tlist = iaplist; tlist; tlist = tlist->next) {
		tinfo = tlist->data;
		if(tinfo == NULL) {
			continue;
		}
		if((tinfo) && (tinfo->iap_name)) {
			linfo = dup_iap_serv_data(tinfo);
			gl_iap_confserv_list= g_slist_append(gl_iap_confserv_list, linfo);
			linfo = NULL;
		}
			
	}
}

/**
   This method is used to update locally maintained list with latest data 
   @param iaplist list of iapinfo  
   @return nothing 
*/
gboolean iap_confsmtp_get_serv_info_list( GSList **iaplist) 
{
	if(!(iaplist)) {
		return FALSE;
	}
	if(*iaplist != NULL) {
		g_slist_foreach( *iaplist, (GFunc)get_servinfo, NULL);
	}
	iap_confsmtp_sync_serv_info_list(*iaplist);
	iap_smtpconf_savelist(); 
	return TRUE;
}

/**
   This method is used to get the path of iap configured smtp data 
   @param none
   @return path of xml file 
*/
gchar *iap_smtpconfserv_rcpath(void)
{
	return(g_strconcat((const gchar *) get_rc_dir_dbus(),
			     G_DIR_SEPARATOR_S, IAP_CONFSERV_RC, NULL));
}

/**
   This method is used to save the iap configured smtp data in xml file 
   @param none
   @return TRUE on success 
*/
gboolean iap_smtpconf_savelist(void) 
{
	gchar *rcpath= NULL;
	xmlDocPtr doc = NULL;
	xmlNodePtr root = NULL;
	
	rcpath = iap_smtpconfserv_rcpath();	
	DBUS_XML_PACK_INIT;
	osso_email_xml_pack_iapinfo_list(root, doc,
					XML_IAPINFO_LST, 
					gl_iap_confserv_list);
	xmlSaveFormatFile(rcpath, doc, 1);
	DBUS_XML_PACK_UN_INIT;
	g_free(rcpath);
	return TRUE;
}

/**
   This method is used to read the iap configured smtp data from xml file 
   @param none
   @return OSSO_PACK_OK on success and OSSO_PACK_ERROR on fail 
*/
gint iap_smtpconf_readlist(void) 
{
	gchar *rcpath= iap_smtpconfserv_rcpath();	
	xmlDocPtr doc = NULL;
	xmlNodePtr cur = NULL;
       	xmlNodePtr node = NULL;
	void *ret_list = NULL;
		
	if (access(rcpath, R_OK) != 0) {
		osso_log(LOG_INFO, "cannot open %s for reading: %s", rcpath, strerror(errno));
		g_free(rcpath);
		return OSSO_PACK_ERROR;
	}

	doc = xmlParseFile(rcpath);
	cur = xmlDocGetRootElement(doc);

	if ((cur == NULL) || (strcmp(cur->name, XML_DATA) != 0)) {
		osso_log(LOG_INFO, "Invald Root element \n");
		xmlFreeDoc(doc);
		g_free(rcpath);
		return OSSO_PACK_ERROR;
	}

	for (node = cur->xmlChildrenNode; node != NULL; node = node->next) {
		if (node->type == XML_ELEMENT_NODE) {
			if (g_ascii_strcasecmp(node->name, XML_IAPINFO_LST) == 0) {
				osso_email_xml_unpack_iapinfo_list(node, doc, XML_IAPINFO_LST, (void **)&ret_list);
				gl_iap_confserv_list = ret_list;
				break;
			}
		}
	}
	xmlFreeDoc(doc);
	g_free(rcpath);
	return OSSO_PACK_OK;
}

/**
   This method is used to get iap info based on iapname 
   @param iapname iap name 
   @return iapinfo contains smtp server data 
*/
const IapInfo *iap_smtpconf_get_iap_info( const gchar *iapname) 
{
	GSList *tlist = NULL; 
	IapInfo *tiapinfo = NULL;
	
	if(!(iapname)) {
		return NULL;
	}
	for( tlist = gl_iap_confserv_list; tlist != NULL ; tlist = tlist->next) {
		tiapinfo = (IapInfo *)tlist->data;
		if(tiapinfo == NULL) {
			continue;
		}
		if(tiapinfo->iap_name) {
			if( compare_utf8_strings( tiapinfo->iap_name, iapname) == 0) {
				return tiapinfo;
			}	
		}
	}
	return NULL;
}

/**
   This method is used to set iap info 
   @param iapinfo contains smtp server data 
   @return TRUE on successful updatation 
*/
gboolean iap_smtpconf_set_iap_info( const IapInfo *iapinfo) 
{
	GSList *tlist = NULL; 
	IapInfo *tiapinfo = NULL;
	gboolean rv = FALSE;
	
	if(!(iapinfo) || (!iapinfo->iap_name)) {
		return rv;
	}
	for( tlist = gl_iap_confserv_list; tlist != NULL ; tlist = tlist->next) {
		if(tlist->data == NULL) {
			continue;
		}
		tiapinfo = (IapInfo *)tlist->data;		
		if(tiapinfo->iap_name) {
			if( compare_utf8_strings( tiapinfo->iap_name, iapinfo->iap_name) == 0) {
				cl_iap_info_free(tiapinfo);
				tiapinfo = dup_iap_serv_data(iapinfo);
				tlist->data = tiapinfo;
				rv = TRUE;
				break;
			}
		}	
	}
	iap_smtpconf_savelist();
	return rv;
}

#endif
