
/**
 * This file is part of osso-email-common
 *
 * Copyright (c) 2006 Nokia Corporation.
 * 
 * Contact: Dirk-Jan Binnema <dirk-jan.binnema@nokia.com>
 * 
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public License
 * version 2.1 as published by the Free Software Foundation.
 * 
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA
 * 02110-1301 USA
 */


#ifdef HAVE_CONFIG_H
#  include "config.h"
#endif

#if CL_USE


#include "cl.h"
#include "email_gconf.h"

#define 	CB_MAGIC_DATA	0xacdcacdc

static email_cl_wa cl_wa;

static void cl_iap_cb(ConIcConnection * connection, ConIcConnectionEvent * event, gpointer data);
static gint cl_validate_iap(const gchar * iap_name);
static IapInfo *iap_info_new(const gchar * iap_name);
static gboolean cl_new_ic_connection(void);
static void cl_delete_ic_connection(void);

/**
    This function is called to create new ic connection. 
    @param none
    @return TRUE on success or  FALSE 
*/
static gboolean cl_new_ic_connection(void)
{
	GValue value;
	if (cl_wa.connection) {	/* connection already cretaed */
		return TRUE;
	}

	cl_wa.connection = con_ic_connection_new();
	if (!cl_wa.connection) {
		return FALSE;
	}
	memset(&value, 0, sizeof(value));
	g_value_init(&value, G_TYPE_BOOLEAN);
	g_signal_connect(G_OBJECT(cl_wa.connection), "connection-event",
			 G_CALLBACK(cl_iap_cb), GINT_TO_POINTER(CB_MAGIC_DATA));
	g_value_set_boolean(&value, TRUE);
	g_object_set_property(G_OBJECT(cl_wa.connection), "automatic-connection-events", &value);
	return TRUE;
}

/**
    This function is called to delete the ic connection. 
    @param none
    @return TRUE on success or  FALSE 
*/
static void cl_delete_ic_connection(void)
{
	GValue value;
	if (!cl_wa.connection) {
		return;
	}
	memset(&value, 0, sizeof(value));
	g_value_init(&value, G_TYPE_BOOLEAN);
	g_value_set_boolean(&value, FALSE);
	g_object_set_property(G_OBJECT(cl_wa.connection), "automatic-connection-events", &value);
	g_object_unref(cl_wa.connection);
	cl_wa.connection = NULL;
}

/**
    This is initialization function for connectivity layer 
    for email Application
    @param none
    @return CL_OK on success or CL_INIT_FAILURE on fail 
*/
gint cl_init(void *cb_func)
{
	DBusError derror = { 0 };
	email_cl_wa *cl_wap = NULL;

	g_type_init();
	dbus_error_init(&derror);
	memset(&cl_wa, '\0', sizeof(email_cl_wa));
	cl_wap = &cl_wa;
	if (cl_new_ic_connection() == FALSE) {
		osso_log(LOG_INFO, " Error in creating new ic connection \n");
		return CL_INIT_FAILURE;
	}
	cl_wap->gmainctx = g_main_context_default();
	cl_wap->loop = g_main_loop_new(cl_wap->gmainctx, FALSE);
	cl_wap->dbusconn = dbus_bus_get(DBUS_BUS_SYSTEM, &derror);
	cl_wap->conn_type = CON_IC_CONNECT_FLAG_NONE;
	cl_wap->cancel_mails_cb = cb_func;

	if (cl_wap->dbusconn == NULL) {
		osso_log(LOG_ERR, "dbus connection failure(%s)\n", derror.message);
		dbus_error_free(&derror);
		cl_uninit();
		return CL_INIT_FAILURE;
	}
	dbus_connection_setup_with_g_main(cl_wap->dbusconn, cl_wap->gmainctx);
	g_object_set_data(G_OBJECT(cl_wa.connection),"needs_disconnection", GINT_TO_POINTER(FALSE));
        osso_log(LOG_INFO, "cl init success\n");
	return CL_OK;
}

/**
    This function is called to uninitialise connectivity context
    @param none
    @return none 
*/

void cl_uninit(void)
{
	if (cl_wa.dbusconn) {
		/* transition to OSSO1.1: no need to disconnect
		dbus_connection_disconnect(cl_wa.dbusconn);
		*/
		dbus_connection_unref(cl_wa.dbusconn);
	}
	cl_delete_ic_connection();
	g_main_loop_unref(cl_wa.loop);
	g_main_context_unref(cl_wa.gmainctx);
}

/**
   This function is called by the connectivity-ic lib for event notification
   @param event iap_event information
   @param arg additional info passed by the osso ic lib
   @return TRUE or FALSE
*/

static void cl_iap_cb(ConIcConnection * connection, ConIcConnectionEvent * event, gpointer data)
{
	const gchar *iap_name;
	ConIcConnectionStatus status = 0;
	email_cl_wa *cl_wap = &cl_wa;

	osso_log(LOG_INFO, "connection callback starts \n");

	g_assert(GPOINTER_TO_INT(data) == CB_MAGIC_DATA);
	g_assert(CON_IC_IS_CONNECTION_EVENT(event));

	status = con_ic_connection_event_get_status(event);
	iap_name = con_ic_event_get_iap_id(CON_IC_EVENT(event));

	osso_log(LOG_INFO, "connection CB  status= %d  iapname= %s \n", status, iap_name);

	switch (status) {
	case CON_IC_STATUS_CONNECTED:
		if (iap_name == NULL)
			return;
		if (cl_wap->cur_iap == NULL) {
			cl_wap->cur_iap = g_strdup(iap_name);

		} else {
			g_free(cl_wap->cur_iap);
			cl_wap->cur_iap = g_strdup(iap_name);
		}
		break;
	case CON_IC_STATUS_DISCONNECTED:
	case CON_IC_STATUS_DISCONNECTING:
		if (cl_wap->cur_iap != NULL) {
			if (cl_wap->send_recv_cancel_skip == FALSE) {
				if (cl_wap->cancel_mails_cb) {
					cl_wap->cancel_mails_cb();
				}
			}
			g_free(cl_wap->cur_iap);
			cl_wap->cur_iap = NULL;
		}
		break;
	default:
		break;
	}
	cl_wap->status = status;
	cl_wap->send_recv_cancel_skip = FALSE;
}

/**
   This function is called to connect specific iap 
   @param iap_name iap name
   @param con_type specifies to the way to connect.
   @return CL_OK on success or returns the cl layer error code 
*/
gint cl_iap_connect(const gchar * iap_name,  gboolean con_type)
{
	gint rv = CL_OK;
	gboolean con_ic_success = FALSE;
	gint ui_closed = FALSE;

	osso_log(LOG_INFO, "cl_iap_connect starts here \n");
	if (iap_name != NULL) {
		/* requested iap is currently connected iap ???  */
		if ((cl_wa.cur_iap) && (strcmp(iap_name, cl_wa.cur_iap) == 0)) {
			cl_iap_set_type(FALSE);
			return CL_OK;
		}
		if (cl_validate_iap(iap_name) != CL_OK) {
			osso_log(LOG_ERR, "Invalid iap name = %s \n", iap_name);
			cl_iap_set_type(FALSE);
			return CL_INVALID_IAP_NAME;
		}
	 	osso_log(LOG_INFO, "IAP (%s) connect Request d\n", iap_name);
		if(cl_wa.connection) {
			con_ic_success = con_ic_connection_connect_by_id(cl_wa.connection, iap_name, cl_iap_get_type()); 
		}
        } else {
                cl_iap_set_type(con_type);
                osso_log(LOG_INFO, "IAP (ANY) connect Request d\n");
		if(cl_wa.connection) {
			con_ic_success = con_ic_connection_connect(cl_wa.connection, cl_iap_get_type());
		}
        }	
	g_object_set_data(G_OBJECT(cl_wa.connection),"needs_disconnection", GINT_TO_POINTER(TRUE));
	/* connects to the iap */
	if (con_ic_success == FALSE) {
		osso_log(LOG_ERR, "osso_iap_connect failure \n");
		g_free(cl_wa.cur_iap);
		cl_wa.status = -1;
		cl_wa.cur_iap = NULL;
		rv = CL_ERROR_IAP_CONNECT;
		goto endshere;
	}

	cl_wa.status = -1;
	while (cl_wa.status == -1) {
		g_main_context_iteration(NULL, FALSE);
		if(osso_email_gconf_get_int(EMAIL_GCONF_UI_CLOSED,&ui_closed)) {
                        if(ui_closed) {
                                break;
                         }
                  }
	}
	if (cl_wa.status == CON_IC_STATUS_CONNECTED) {
		osso_log(LOG_INFO, "connected\n");
		rv = CL_OK;
	} else {
		osso_log(LOG_INFO, "not connected\n");
		rv = CL_ERROR_IAP_CONNECT;
	}
      endshere:
	cl_iap_set_type(FALSE);
	return rv;
}

/**
   This function is called to disconnect the specific iap
   @param iap_name iap name
   @return success or error code
*/
gint cl_iap_disconnect(const gchar * iap_name)
{
	gint rv = CL_OK;
	gint ui_closed = FALSE;
	gboolean con_ic_success = FALSE;
	osso_log(LOG_INFO, "%s (%s) starts here \n", __FUNCTION__, iap_name);

	if(!(cl_active_iap())){
		osso_log(LOG_INFO, "There is no active iap to disconnect");
		return rv;
	}
	
	cl_wa.send_recv_cancel_skip = TRUE;
	if(!GPOINTER_TO_INT(g_object_get_data(G_OBJECT(cl_wa.connection), "needs_disconnection")))
		return rv;

	if (cl_wa.connection) {
		g_object_set_data(G_OBJECT(cl_wa.connection),"needs_disconnection", GINT_TO_POINTER(FALSE));
		con_ic_success = con_ic_connection_disconnect(cl_wa.connection);
	}
	if (con_ic_success == FALSE) {
		osso_log(LOG_ERR, "osso_iap_disconnect failure \n");
		rv = CL_ERROR_IAP_DISCONNECT;
		goto endshere;
	}
	cl_wa.status = -1;
	while (cl_wa.status == -1) {
		g_main_context_iteration(NULL, FALSE);
		if(osso_email_gconf_get_int(EMAIL_GCONF_UI_CLOSED,&ui_closed)) {
			if(ui_closed) {
			        break;
			}
                }
	}
	if (cl_wa.status == CON_IC_STATUS_DISCONNECTED) {
		rv = CL_OK;
	} else {
		rv = CL_ERROR_IAP_DISCONNECT;
	}
      endshere:
	return rv;
}

/**
   This function creates a new IapInfo instance with the passed values
   @param iap_name iap name
   @return pointer IapInfo or NULL
*/

static IapInfo *iap_info_new(const gchar * iap_name)
{
	IapInfo *iap_info = NULL;
	if (iap_name == NULL) {
		osso_log(LOG_INFO, "invalid parameter \n");
		return NULL;
	}
	iap_info = g_new0(IapInfo, 1);
	iap_info->iap_name = g_strdup(iap_name);
	return iap_info;
}

/**
   This function is used to free the memory
   @param iap_info iap information
   @return nothing 
*/
void cl_iap_info_free(IapInfo * iap_info)
{
	if (iap_info != NULL) {
		g_free(iap_info->iap_name);
		g_free(iap_info->bearer);
		g_free(iap_info);
	}
}

/**
   This function is used to free the memory
   @param iap_info iap information
   @return CL_OK on success 
*/
gint cl_iap_info_list_free(GSList * iap_info_list)
{
	if (iap_info_list != NULL) {
		g_slist_foreach(iap_info_list, (GFunc) cl_iap_info_free, NULL);
		g_slist_free(iap_info_list);
	}
	return CL_OK;
}

/**
   This function return iap_list as an out parameter
   @param iap_list
   @return return status
*/

gint cl_get_iap_info_list(GSList ** iap_list)
{
	GSList *cur = NULL;
	IapInfo *iap_info = NULL;
	GSList *conniap_list = NULL;
	GSList *iap_info_list = NULL;
	ConIcIap *iap = NULL;

	if (iap_list == NULL) {
		return CL_INVALID_PARAM;
	}
	if (cl_new_ic_connection() == FALSE) {
		return CL_ERROR_GET_LIST;
	}

	/* get the configurred iaps  ans prepare info list */
	conniap_list = con_ic_connection_get_all_iaps(cl_wa.connection);
	for (cur = conniap_list; cur != NULL; cur = cur->next) {
		if (cur->data != NULL) {
			iap = CON_IC_IAP(cur->data);
			iap_info = iap_info_new(con_ic_iap_get_name(iap));
			if (iap_info != NULL) {
				iap_info->bearer = g_strdup(con_ic_iap_get_bearer_type(iap));
				iap_info_list = g_slist_append(iap_info_list, iap_info);
			}
			osso_log(LOG_INFO, " %s IAP name %s  bearer type = %s \n", __FUNCTION__,
				 iap_info->iap_name, iap_info->bearer);
			g_object_unref(iap);
		}
	}
	g_slist_free(conniap_list);
	*iap_list = iap_info_list;
	return CL_OK;
}

/**
  This function is called to validate the iap name from the iap list
  returned by osso-ic library.
  
  @param iap_name iap name
  @return on success returns CL_OK or returns error code
*/
static gint cl_validate_iap(const gchar * iap_name)
{
	GSList *iap_lt = NULL;
	GSList *cur = NULL;
	const gchar *name = NULL;
	ConIcIap *iap = NULL;
	gboolean iap_found = FALSE;

	if (iap_name == NULL) {
		return CL_INVALID_PARAM;
	}
	/* get configured iaps in the device */
	if (cl_new_ic_connection() == FALSE) {
		return CL_ERROR_GET_LIST;
	}
	iap_lt = con_ic_connection_get_all_iaps(cl_wa.connection);
	if (iap_lt == NULL) {
		return CL_ERROR_GET_LIST;
	}

	/* check for the iap */
	for (cur = iap_lt; cur != NULL; cur = cur->next) {
		iap = CON_IC_IAP(cur->data);
		if (iap) {
			name = con_ic_iap_get_name(iap);
		}
		if (name == NULL) {
			continue;
		}
		if (strcmp((gchar *) iap_name, name) == 0) {
			iap_found = TRUE;
			break;
		}
		g_object_unref(iap);
	}
	g_slist_free(iap_lt);
	return (iap_found == TRUE ? CL_OK : CL_INVALID_IAP);
}

/**
  This function is called to set the connection type. 
  @param type for timed out connection set to TRUE 
  @return nothing 
*/
void cl_iap_set_type(gboolean type)
{
	if (type == TRUE) {
		cl_wa.conn_type = CON_IC_CONNECT_FLAG_AUTOMATICALLY_TRIGGERED;
	} else {
		cl_wa.conn_type = CON_IC_CONNECT_FLAG_NONE;
	}
}

/**
  This function is called to get the connection type. 
  @param none 
  @return nothing 
*/

gint cl_iap_get_type(void)
{
	return cl_wa.conn_type;
}

/**
  This function is used to get the status of current iap 
  @param option  TRUE means set status and FALSE means get the status
  @param val  to be set  
  @return nothing 
*/

gboolean cl_current_iap_status(gboolean option, gint val)
{
	if (option == FALSE) {	/* get the iap status */
		return (cl_wa.status == CON_IC_STATUS_CONNECTED ? TRUE : FALSE);
	}
	if (cl_wa.status == 0) {	/* iap status is in connecting mode */
		cl_wa.status = val;
	}
	return cl_wa.status;
}

void cl_discon_cur_iap(void)
{
	if (cl_wa.cur_iap != NULL) {
		if(!GPOINTER_TO_INT(g_object_get_data(G_OBJECT(cl_wa.connection), "needs_disconnection"))){
			g_free(cl_wa.cur_iap);
			return;
		}
		if (cl_wa.connection) {
			g_object_set_data(G_OBJECT(cl_wa.connection),"needs_disconnection", GINT_TO_POINTER(FALSE));
			con_ic_connection_disconnect(cl_wa.connection);
		}
		g_free(cl_wa.cur_iap);
	}
}

/**
  This function is used to disconnect at the time of connecting 
  @param void 
  @return nothing 
*/
void cl_cancel_iap_conn(void)
{
	osso_log(LOG_ERR, "%s : Cancel IAP connection\n", __FUNCTION__);
	if (cl_wa.status == 0) {
		cl_wa.status = CL_CONNECTION_CANCEL_REQUEST;
		cl_wa.send_recv_cancel_skip = FALSE;
		if(!GPOINTER_TO_INT(g_object_get_data(G_OBJECT(cl_wa.connection), "needs_disconnection")))
			return;
		if (cl_wa.connection) {
			g_object_set_data(G_OBJECT(cl_wa.connection),"needs_disconnection", GINT_TO_POINTER(FALSE));
			con_ic_connection_disconnect(cl_wa.connection);
		}
	}
	return;
}

/**
  This function is returns the current activ iap 
  @param void 
  @return the current iap name 
*/
const gchar *cl_active_iap(void)
{
	return (cl_wa.cur_iap);
}

/**
  This function checks whether the current active connection is wlan 
  @param void 
  @return gboolean 
*/
gboolean active_con_is_wlan(void)
{
	const gchar *bearer_name = NULL;
	ConIcIap *iap = NULL;
	const gchar *active_iap = NULL, *iapname = NULL;
	GSList *conniap_list = NULL, *cur = NULL;
	gboolean wlan = FALSE;

	active_iap = cl_active_iap();
	if (!active_iap)
		return FALSE;

	/* get configured iaps in the device */
	if (cl_new_ic_connection() == FALSE) {
		osso_log(LOG_INFO, "no iaps conf ");
		return FALSE;
	}

	conniap_list = con_ic_connection_get_all_iaps(cl_wa.connection);
	for (cur = conniap_list; cur && cur->data; cur = cur->next) {
		iap = CON_IC_IAP(cur->data);
		iapname = con_ic_iap_get_name(iap);
		if (strcmp(iapname, active_iap) == 0) {
			bearer_name = con_ic_iap_get_bearer_type(iap);
			if (bearer_name
			    && (g_ascii_strncasecmp(bearer_name, "WLAN", strlen("WLAN")) == 0)) {
				wlan = TRUE;
				osso_log(LOG_INFO, "active iap is wlan!!");
				break;
			}
		}
	}
	g_slist_foreach(conniap_list, (GFunc) g_object_unref, NULL);
	g_slist_free(conniap_list);
	return wlan;
}

#endif				/* end of CL_USE */
