
/**
 * This file is part of osso-email-common
 *
 * Copyright (c) 2006 Nokia Corporation.
 * 
 * Contact: Dirk-Jan Binnema <dirk-jan.binnema@nokia.com>
 * 
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public License
 * version 2.1 as published by the Free Software Foundation.
 * 
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA
 * 02110-1301 USA
 */

#define DBUS_API_SUBJECT_TO_CHANGE

#ifdef HAVE_CONFIG_H
#  include "config.h"
#endif

#include <stdlib.h>
#include <glib.h>
#include <glib/gprintf.h>
#include <libxml/parser.h>
#include "common.h"
#include "utils.h"
#include "prefs_account.h"
#include "prefs.h"
#include "procmsg.h"
#include "procheader.h"
#include "defs.h"
#include "codeconv.h"
#include "folder.h"
#include "procmime.h"
#include "base64.h"
#include "dbus_utils_xml.h"
#include "dbus_common_utils.h"
#include "statesave.h"
#include "libosso.h"

#if CL_USE
#include "cl.h"
#endif

static MimeInfo tmpMimeInfo;

static DBusData MimeInfoDBusData[] = {
	{"encoding", D_STRING, &tmpMimeInfo.encoding,
	 osso_email_xml_pack_str},
	{"encoding_type", D_INT, &tmpMimeInfo.encoding_type,
	 osso_email_xml_pack_int},
	{"mime_type", D_INT, &tmpMimeInfo.mime_type,
	 osso_email_xml_pack_int},
	{"content_type", D_STRING, &tmpMimeInfo.content_type,
	 osso_email_xml_pack_str},
	{"charset", D_STRING, &tmpMimeInfo.charset,
	 osso_email_xml_pack_str},
	{"name", D_STRING, &tmpMimeInfo.name,
	 osso_email_xml_pack_str},
	{"boundary", D_STRING, &tmpMimeInfo.boundary,
	 osso_email_xml_pack_str},
	{"content_disposition", D_STRING, &tmpMimeInfo.content_disposition,
	 osso_email_xml_pack_str},
	{"filename", D_STRING, &tmpMimeInfo.filename,
	 osso_email_xml_pack_str},
	{"fpos", D_LONG, &tmpMimeInfo.fpos,
	 osso_email_xml_pack_long},
	{"size", D_INT, &tmpMimeInfo.size,
	 osso_email_xml_pack_int},
	{"level", D_INT, &tmpMimeInfo.level,
	 osso_email_xml_pack_int},
	{"section_number", D_STRING, &tmpMimeInfo.section_number,
	 osso_email_xml_pack_str},
	{"is_downloaded", D_BOOL, &tmpMimeInfo.is_downloaded,
	 osso_email_xml_pack_bool},
	{"is_purged", D_BOOL, &tmpMimeInfo.is_purged,
	 osso_email_xml_pack_bool},
	/* SMime Support */
	{"cert_state_sender", D_ENUM, &tmpMimeInfo.cert_state_sender,
	 osso_email_xml_pack_int},
	{"cert_state_recipient", D_ENUM, &tmpMimeInfo.cert_state_recip,
	 osso_email_xml_pack_int},
	{"trust", D_BOOL, &tmpMimeInfo.trust,
	 osso_email_xml_pack_bool},
	{"signature_verify", D_BOOL, &tmpMimeInfo.signature_verify,
	 osso_email_xml_pack_bool},
	{"saving", D_BOOL, &tmpMimeInfo.saving,
	 osso_email_xml_pack_bool},
	{"cert_path", D_STRING, &tmpMimeInfo.cert_path,
	 osso_email_xml_pack_str},
	{"smime_type", D_ENUM, &tmpMimeInfo.smime_type,
	 osso_email_xml_pack_int},
	{"Inline_Images", D_MIME_INLINE_INFO_LIST, &tmpMimeInfo,
	 osso_email_xml_pack_inlineimageinfo_list},
	{"file_pos_list", D_INT_LIST, &tmpMimeInfo,
	 osso_email_xml_pack_int_list},
	{"sender_cert_id", D_INT, &tmpMimeInfo.sender_cert_id,
	 osso_email_xml_pack_int},
	{"recip_cert_id", D_INT, &tmpMimeInfo.recip_cert_id,
	 osso_email_xml_pack_int},
	{"parent_boundary_info", D_PARENT, &tmpMimeInfo,
	 osso_email_xml_pack_str},
	{"Child", D_CHILD, &tmpMimeInfo,
	 osso_email_xml_pack_mimeinfo},
	{"Sub", D_SUB, &tmpMimeInfo,
	 osso_email_xml_pack_mimeinfo},
	{NULL, 0, NULL}
};

static MsgHeader tmpMsgHeader;

static DBusData MsgHeaderDBusData[] = {
	{"msgnum", D_INT, &tmpMsgHeader.msgnum, osso_email_xml_pack_int},
	{"size", D_INT, &tmpMsgHeader.size, osso_email_xml_pack_int},
	{"mime", D_LONG, &tmpMsgHeader.mtime, osso_email_xml_pack_long},
	{"date_t", D_LONG, &tmpMsgHeader.date_t, osso_email_xml_pack_long},
	{"rx_date", D_LONG, &tmpMsgHeader.rx_date, osso_email_xml_pack_long},
	{"flags", D_STRUCT, &tmpMsgHeader.flags, osso_email_xml_pack_msgflags},
	{"fromname", D_STRING, &tmpMsgHeader.fromname, osso_email_xml_pack_str},
	{"date", D_STRING, &tmpMsgHeader.date, osso_email_xml_pack_str},
	{"from", D_STRING, &tmpMsgHeader.from, osso_email_xml_pack_str},
	{"to", D_STRING, &tmpMsgHeader.to, osso_email_xml_pack_str},
	{"cc", D_STRING, &tmpMsgHeader.cc, osso_email_xml_pack_str},
	{"bcc", D_STRING, &tmpMsgHeader.bcc, osso_email_xml_pack_str},
	{"subject", D_STRING, &tmpMsgHeader.subject,
	 osso_email_xml_pack_str_base64},
	{"msgid", D_STRING, &tmpMsgHeader.msgid, osso_email_xml_pack_str},
	{"replyto", D_STRING, &tmpMsgHeader.replyto, osso_email_xml_pack_str},
	{"inreplyto", D_STRING, &tmpMsgHeader.inreplyto, osso_email_xml_pack_str},
	{"username", D_STRING, &tmpMsgHeader.username, osso_email_xml_pack_str},
	{"password", D_STRING, &tmpMsgHeader.password, osso_email_xml_pack_str},
	{"account_name", D_STRING, &tmpMsgHeader.account_name,
	 osso_email_xml_pack_str},
	{"msgtag", D_STRUCT, &tmpMsgHeader.msgtag, osso_email_xml_pack_msgtag},
	{"recv_priority", D_INT, &tmpMsgHeader.recv_priority,
	 osso_email_xml_pack_int},
	{"read_receipt_request", D_BOOL, &tmpMsgHeader.read_receipt_request,
	 osso_email_xml_pack_bool},
	{"copy_to_own_address", D_BOOL, &tmpMsgHeader.copy_to_own_address,
	 osso_email_xml_pack_bool},
	{"smime_type", D_ENUM, &tmpMsgHeader.smime_type, osso_email_xml_pack_int},
	{"attach_tag", D_INT, &tmpMsgHeader.attach_tag, osso_email_xml_pack_int},
	{NULL, 0, NULL}
};

static Folder tmpFolder;
static DBusData FolderDBusData[] = {
	{"name", D_STRING, &tmpFolder.name, osso_email_xml_pack_str},
	{"path", D_STRING, &tmpFolder.path, osso_email_xml_pack_str},
	{"unread", D_INT, &tmpFolder.unread, osso_email_xml_pack_int},
	{"new_msgs", D_INT, &tmpFolder.new_msgs, osso_email_xml_pack_int},
	{"total", D_INT, &tmpFolder.total, osso_email_xml_pack_int},
	{"type", D_INT, &tmpFolder.type, osso_email_xml_pack_int},
	{NULL, 0, NULL}
};

static FolderAttachInfo tmpFolderAttachInfo;
static DBusData FolderAttachInfoDBusData[] = {
	{"msgid", D_STRING, &tmpFolderAttachInfo.msgid, osso_email_xml_pack_str},
	{"section_number", D_STRING, &tmpFolderAttachInfo.section_number,
	 osso_email_xml_pack_str},
	{NULL, 0, NULL}
};

static MsgFlags tmpMsgFlags;

static DBusData MsgFlagsDBusData[] = {
	{"perm_flags", D_ENUM, &tmpMsgFlags.perm_flags, osso_email_xml_pack_int},
	{"tmp_flags", D_ENUM, &tmpMsgFlags.tmp_flags, osso_email_xml_pack_int},
	{NULL, 0, NULL}
};

static Tag tmpMsgTag;
static DBusData MsgTagDBusData[] = {
	{"protocol", D_ENUM, &tmpMsgTag.protocol, osso_email_xml_pack_int},
	{"recvtype", D_ENUM, &tmpMsgTag.recvtype, osso_email_xml_pack_int},
	{"foldertype", D_ENUM, &tmpMsgTag.foldertype, osso_email_xml_pack_int},
	{"fullydownloaded", D_BOOL, &tmpMsgTag.fullydownloaded,
	 osso_email_xml_pack_bool},
	{"bodydownloaded", D_BOOL, &tmpMsgTag.bodydownloaded,
	 osso_email_xml_pack_bool},
	{"remote_copy", D_BOOL, &tmpMsgTag.remote_copy, osso_email_xml_pack_bool},
	{"timestamp", D_LONG, &tmpMsgTag.timestamp, osso_email_xml_pack_long},
	{NULL, 0, NULL}
};

static InlineImageInfo tmpInlineImageInfo;

static DBusData InlineImageInfoDBusData[] = {
	{"uri_path", D_STRING, &tmpInlineImageInfo.uri_path,
	 osso_email_xml_pack_str},
	{"content_id", D_STRING, &tmpInlineImageInfo.content_id,
	 osso_email_xml_pack_str},
	{"image_name", D_STRING, &tmpInlineImageInfo.image_name,
	 osso_email_xml_pack_str},
	{"image_path", D_STRING, &tmpInlineImageInfo.image_path,
	 osso_email_xml_pack_str},
	{NULL, 0, NULL}
};

static AttachInfo tmpAttachInfo;

static DBusData AttachInfoDBusData[] = {
	{"type", D_ENUM, &tmpAttachInfo.type, osso_email_xml_pack_int},
	{"filepath", D_STRING, &tmpAttachInfo.filepath, osso_email_xml_pack_str},
	{"msgid", D_STRING, &tmpAttachInfo.msgid, osso_email_xml_pack_str},
	{"attachmentname", D_STRING, &tmpAttachInfo.attachmentname,
	 osso_email_xml_pack_str},
	{"email_path", D_STRING, &tmpAttachInfo.email_path,
	 osso_email_xml_pack_str},
	{NULL, 0, NULL}
};

static EmailSpecificSettings tmpSettings;

static DBusData EMailSettingsDBusData[] = {
	{"priority", D_ENUM, &tmpSettings.priority, osso_email_xml_pack_int},
	{"type", D_ENUM, &tmpSettings.type, osso_email_xml_pack_int},
	{"account", D_STRING, &tmpSettings.account, osso_email_xml_pack_str},
	{"read_receipt_request", D_BOOL, &tmpSettings.read_receipt_request,
	 osso_email_xml_pack_bool},
	{"options", D_ENUM, &tmpSettings.options, osso_email_xml_pack_int},
	{"copy_to_own_address", D_BOOL, &tmpSettings.copy_to_own_address,
	 osso_email_xml_pack_bool},
	{"smime_type", D_ENUM, &tmpSettings.smime_type, osso_email_xml_pack_int},
	{NULL, 0, NULL}
};

static UserSettings tmpUserSettings;
static DBusData UserSettingsDBusData[] = {
	{"account_name", D_STRING, &tmpUserSettings.account_name,
	 osso_email_xml_pack_str},
	{"is_default", D_BOOL, &tmpUserSettings.is_default, osso_email_xml_pack_bool},
	{"userid", D_STRING, &tmpUserSettings.userid, osso_email_xml_pack_str_base64},
	{"passwd", D_STRING, &tmpUserSettings.passwd, osso_email_xml_pack_str_base64},
	{"remember_passwd", D_BOOL, &tmpUserSettings.remember_passwd,
	 osso_email_xml_pack_bool},
	{"email", D_STRING, &tmpUserSettings.email, osso_email_xml_pack_str},
	{"alias_name", D_STRING, &tmpUserSettings.alias_name, osso_email_xml_pack_str},
	{NULL, 0, NULL}

};

static ReceiveSettings tmpRecvSettings;

static DBusData RecvSettingsDBusData[] = {
	{"protocol", D_ENUM, &tmpRecvSettings.protocol, osso_email_xml_pack_int},
	{"recv_server", D_STRING, &tmpRecvSettings.recv_server,
	 osso_email_xml_pack_str},
	{"port", D_USHORT, &tmpRecvSettings.port, osso_email_xml_pack_ushort},
	{"passwd_auth_type", D_ENUM, &tmpRecvSettings.passwd_auth_type,
	 osso_email_xml_pack_int},
	{"recv_type", D_ENUM, &tmpRecvSettings.recv_type, osso_email_xml_pack_int},
	{"enable_size_limit", D_ENUM, &tmpRecvSettings.enable_size_limit,
	 osso_email_xml_pack_int},
	{"size_limit", D_INT, &tmpRecvSettings.size_limit, osso_email_xml_pack_int},
	{"recent_mails", D_INT, &tmpRecvSettings.recent_mails, osso_email_xml_pack_int},
	{"leave_msg", D_INT, &tmpRecvSettings.leave_msg, osso_email_xml_pack_int},
	{"is_not_editable", D_INT, &tmpRecvSettings.is_not_editable,
	 osso_email_xml_pack_int},
	{NULL, 0, NULL}
};

static SendSettings tmpSendSettings;

static DBusData SendSettingsDBusData[] = {
	{"smtp_server", D_STRING, &tmpSendSettings.smtp_server,
	 osso_email_xml_pack_str},
	{"smtp_auth_type", D_ENUM, &tmpSendSettings.smtp_auth_type,
	 osso_email_xml_pack_int},
	{"smtp_port", D_USHORT, &tmpSendSettings.smtp_port,
	 osso_email_xml_pack_ushort},
	{"smtp_username", D_STRING, &tmpSendSettings.username,
	 osso_email_xml_pack_str},
	{"smtp_password", D_STRING, &tmpSendSettings.password,
	 osso_email_xml_pack_str},
	{NULL, 0, NULL}
};

static AdvancedSettings tmpAdvSettings;

static DBusData AdvSettingsDBusData[] = {
	{"email_type", D_ENUM, &tmpAdvSettings.email_type, osso_email_xml_pack_int},
	{"email_sending", D_ENUM, &tmpAdvSettings.email_sending,
	 osso_email_xml_pack_int},
	{"include_signature", D_BOOL, &tmpAdvSettings.include_signature,
	 osso_email_xml_pack_bool},
	{"signature_file", D_STRING, &tmpAdvSettings.signature_file,
	 osso_email_xml_pack_str},
	{"signature_flag", D_BOOL, &tmpAdvSettings.signature_flag,
	 osso_email_xml_pack_bool},
	{"request_read_report", D_BOOL, &tmpAdvSettings.request_read_report,
	 osso_email_xml_pack_bool},
	{"copy_to_own_address", D_BOOL, &tmpAdvSettings.copy_to_own_address,
	 osso_email_xml_pack_bool},
	{"original_in_reply", D_BOOL, &tmpAdvSettings.org_in_reply,
	 osso_email_xml_pack_bool},
	{"use_ssl", D_BOOL, &tmpAdvSettings.use_ssl, osso_email_xml_pack_bool},
	{"imap_dir", D_STRING, &tmpAdvSettings.imap_dir, osso_email_xml_pack_str},
#ifdef CL_USE
	{"iap_name", D_STRING, &tmpAdvSettings.iap_name, osso_email_xml_pack_str},
	{"use_iap_smtp_serv", D_BOOL, &tmpAdvSettings.use_iap_smtp_serv, osso_email_xml_pack_bool},
#endif
	{"smime_security", D_INT, &tmpAdvSettings.smime_security,
	 osso_email_xml_pack_int},
	{"display_sign_encrypt", D_BOOL, &tmpAdvSettings.display_sign_encrypt,
	 osso_email_xml_pack_bool},
	{"display_trust", D_BOOL, &tmpAdvSettings.display_trust,
	 osso_email_xml_pack_bool},
	{"display_validity", D_BOOL, &tmpAdvSettings.display_validity,
	 osso_email_xml_pack_bool},
	{"use_tls", D_BOOL, &tmpAdvSettings.use_tls, osso_email_xml_pack_bool},
	{"smtp_use_ssl",
	 D_BOOL, &tmpAdvSettings.smtp_use_ssl, osso_email_xml_pack_bool},

	{"smtp_use_tls", D_BOOL, &tmpAdvSettings.smtp_use_tls,
	 osso_email_xml_pack_bool},
	{NULL, 0, NULL}
};


/* State save Information */

static osso_pack_return_t
osso_email_xml_pack_folderSortInfo(xmlNodePtr node, xmlDocPtr doc,
				   void *nodeType, void *folderSortInfo);
static osso_pack_return_t
osso_email_xml_unpack_foldersortInfo(xmlNodePtr node, xmlDocPtr doc, gchar * nodeType, void *data);
static osso_pack_return_t
osso_email_xml_pack_openWindowData(xmlNodePtr node, xmlDocPtr doc, void *nodeType, void *openMsg);
static osso_pack_return_t
osso_email_xml_unpack_openWindowData(xmlNodePtr node, xmlDocPtr doc, gchar * nodeType, void *data);
static osso_pack_return_t
osso_email_xml_pack_folderSortInfo_list(xmlNodePtr node,
					xmlDocPtr doc, void *nodeType, void *folderSortInfo_list);
static osso_pack_return_t
osso_email_xml_unpack_folderSortInfo_list(xmlNodePtr node, xmlDocPtr doc,
					  gchar * nodeType, GSList ** data);
static osso_pack_return_t
osso_email_xml_pack_openMsgs_list(xmlNodePtr node,
				  xmlDocPtr doc, void *nodeType, void *openMsgs_list);
static osso_pack_return_t
osso_email_xml_unpack_open_msgs_list(xmlNodePtr node, xmlDocPtr doc,
				     gchar * nodeType, GSList ** data);
static gint
osso_email_xml_unpack_retr_msg(xmlNodePtr node, xmlDocPtr doc,
			       gchar * nodeType, RetrMsgData * retr_msg);
static osso_pack_return_t
osso_email_xml_pack_retr_msg(xmlNodePtr node, xmlDocPtr doc, void *nodeType, void *data);

static EmailStateInfo tmpEmailStateInfo;

static DBusData EmailStateInfoDBusData[] = {
	{"oom_signal_recv", D_BOOL, &tmpEmailStateInfo.oom_signal_received,
	 osso_email_xml_pack_bool},
	{"split_view_mode", D_BOOL, &tmpEmailStateInfo.split_view_mode,
	 osso_email_xml_pack_bool},
	{"full_screen_mode", D_BOOL, &tmpEmailStateInfo.full_screen_mode,
	 osso_email_xml_pack_bool},
	{"tool_bar_hidden", D_BOOL, &tmpEmailStateInfo.tool_bar_hidden,
	 osso_email_xml_pack_bool},
	{"cc_field_on", D_BOOL, &tmpEmailStateInfo.cc_field_on,
	 osso_email_xml_pack_bool},
	{"bcc_field_on", D_BOOL, &tmpEmailStateInfo.bcc_field_on,
	 osso_email_xml_pack_bool},
	{"folderview_scroll_pos", D_INT, &tmpEmailStateInfo.folderview_scroll_position,
	 osso_email_xml_pack_int},
	{"msgview_scroll_pos", D_INT, &tmpEmailStateInfo.msgview_scroll_position,
	 osso_email_xml_pack_int},
	{"current_folder", D_STRING, &tmpEmailStateInfo.current_folder,
	 osso_email_xml_pack_str},
	{"current_msgid", D_STRING, &tmpEmailStateInfo.current_msg_id,
	 osso_email_xml_pack_str},
	{NULL, 0, NULL}
};

static OpenedWindowData tmpOpenedWindowData;

static DBusData OpenedWindowDataDBusData[] = {
	{"Msgid", D_STRING, &tmpOpenedWindowData.msgid,
	 osso_email_xml_pack_str},
	{"msgNum", D_INT, &tmpOpenedWindowData.msgNum,
	 osso_email_xml_pack_int},
	{"topSequence", D_INT, &tmpOpenedWindowData.topSequence,
	 osso_email_xml_pack_int},
	{"OldMsgid", D_STRING, &tmpOpenedWindowData.oldmsgid,
	 osso_email_xml_pack_str},
	{"iseditor", D_BOOL, &tmpOpenedWindowData.iseditor,
	 osso_email_xml_pack_bool},
	{"scroll_position", D_INT, &tmpOpenedWindowData.scroll_position,
	 osso_email_xml_pack_int},
	{"zoom_percentage", D_INT, &tmpOpenedWindowData.zoom_percentage,
	 osso_email_xml_pack_int},
	{"cursor_x_pos", D_INT, &tmpOpenedWindowData.cursor_x_pos,
	 osso_email_xml_pack_int},
	{"cursor_y_pos", D_INT, &tmpOpenedWindowData.cursor_y_pos,
	 osso_email_xml_pack_int},
	{NULL, 0, NULL}
};

static FolderSortInfo tmpFolderSortInfo;

static DBusData FolderSortInfoDBusData[] = {
	{"Folder_name", D_STRING, &tmpFolderSortInfo.name,
	 osso_email_xml_pack_str},
	{"sort_key", D_INT, &tmpFolderSortInfo.sort_key,
	 osso_email_xml_pack_int},
	{"sort_order", D_INT, &tmpFolderSortInfo.sort_order,
	 osso_email_xml_pack_int},
	{NULL, 0, NULL}
};

static EmailPersistentInfo tmpEmailPersistentInfo;

static DBusData EmailPersistentInfoDBusData[] = {
	{"split_view_mode", D_BOOL, &tmpEmailPersistentInfo.split_view_mode,
	 osso_email_xml_pack_bool},
	{"mainview_pane_pos", D_INT,
	 &tmpEmailPersistentInfo.mainview_pane_pos, osso_email_xml_pack_int},
	{"listview_from_col_width", D_INT,
	 &tmpEmailPersistentInfo.listview_from_col_width, osso_email_xml_pack_int},
	{"listview_subject_col_width", D_INT,
	 &tmpEmailPersistentInfo.listview_subject_col_width, osso_email_xml_pack_int},
	{"mainview_tool_bar_hidden", D_BOOL,
	 &tmpEmailPersistentInfo.mainview_tool_bar_hidden,
	 osso_email_xml_pack_bool},
	{"viewer_tool_bar_hidden", D_BOOL,
	 &tmpEmailPersistentInfo.viewer_tool_bar_hidden,
	 osso_email_xml_pack_bool},
	{"editor_tool_bar_hidden", D_BOOL,
	 &tmpEmailPersistentInfo.editor_tool_bar_hidden,
	 osso_email_xml_pack_bool},
	{"mainview_fullscreen_tool_bar_hidden", D_BOOL,
	 &tmpEmailPersistentInfo.mainview_fullscreen_tool_bar_hidden,
	 osso_email_xml_pack_bool},
	{"viewer_fullscreen_tool_bar_hidden", D_BOOL,
	 &tmpEmailPersistentInfo.viewer_fullscreen_tool_bar_hidden,
	 osso_email_xml_pack_bool},
	{"editor_fullscreen_tool_bar_hidden", D_BOOL,
	 &tmpEmailPersistentInfo.editor_fullscreen_tool_bar_hidden,
	 osso_email_xml_pack_bool},
	{"cc_field_on", D_BOOL, &tmpEmailPersistentInfo.cc_field_on,
	 osso_email_xml_pack_bool},
	{"bcc_field_on", D_BOOL, &tmpEmailPersistentInfo.bcc_field_on,
	 osso_email_xml_pack_bool},
	{"read_receipt", D_BOOL, &tmpEmailPersistentInfo.read_receipt,
	 osso_email_xml_pack_bool},
	{"msg_size_limit", D_INT, &tmpEmailPersistentInfo.msg_size_limit,
	 osso_email_xml_pack_int},
	{"play_sound", D_BOOL, &tmpEmailPersistentInfo.play_sound,
	 osso_email_xml_pack_bool},
	{"msg_format_type", D_INT, &tmpEmailPersistentInfo.msg_format_type,
	 osso_email_xml_pack_int},
	{"inc_original", D_BOOL, &tmpEmailPersistentInfo.inc_original,
	 osso_email_xml_pack_bool},
	{"delete_servermail_also", D_BOOL,
	 &tmpEmailPersistentInfo.delete_servermail_also,
	 osso_email_xml_pack_bool},
	{"sent_folder_clean_time_stamp", D_INT,
	 &tmpEmailPersistentInfo.sent_folder_clean_time_stamp,
	 osso_email_xml_pack_int},
	{"sent_folder_clean_option", D_INT,
	 &tmpEmailPersistentInfo.sent_folder_clean_option,
	 osso_email_xml_pack_int},
	{"sent_folder_clean_time_value", D_INT,
	 &tmpEmailPersistentInfo.sent_folder_clean_time_value,
	 osso_email_xml_pack_int},
	{"viewing_html_fontname_index", D_INT,
	 &tmpEmailPersistentInfo.viewing_html_fontname_index,
	 osso_email_xml_pack_int},
	{"viewing_plain_fontname_index", D_INT,
	 &tmpEmailPersistentInfo.viewing_plain_fontname_index,
	 osso_email_xml_pack_int},
	{"editing_html_fontname_index", D_INT,
	 &tmpEmailPersistentInfo.editing_html_fontname_index,
	 osso_email_xml_pack_int},
	{"editing_plain_fontname_index", D_INT,
	 &tmpEmailPersistentInfo.editing_plain_fontname_index,
	 osso_email_xml_pack_int},
	{NULL, 0, NULL}
};

#if CL_USE

static IapInfo tmpIapInfo;

static DBusData IapInfoDBusData[] = {
	{"iap_name", D_STRING, &tmpIapInfo.iap_name, osso_email_xml_pack_str},
	{"servname", D_STRING, &tmpIapInfo.smtp_servname, osso_email_xml_pack_str},
	{"username", D_STRING, &tmpIapInfo.smtp_username, osso_email_xml_pack_str},
	{"passwd", D_STRING, &tmpIapInfo.smtp_passwd, osso_email_xml_pack_str},
	{"port", D_USHORT, &tmpIapInfo.smtp_port, osso_email_xml_pack_ushort},
	{"auth_type", D_ENUM, &tmpIapInfo.smtp_auth_type, osso_email_xml_pack_int},
	{"use_ssl", D_BOOL, &tmpIapInfo.smtp_use_ssl, osso_email_xml_pack_bool},
	{"use_tls", D_BOOL, &tmpIapInfo.smtp_use_tls, osso_email_xml_pack_bool},
	{"iap_type", D_ENUM, &tmpIapInfo.iap_type, osso_email_xml_pack_int},
	{NULL, 0, NULL}
};
#endif

static osso_pack_return_t
osso_email_xml_pack_struct(xmlNodePtr node, xmlDocPtr doc, void *name, void *sdata);
static gboolean osso_email_xml_unpack_match_mimeinfo_attribute(gchar * nodename, gint * idx);
static gboolean osso_email_xml_unpack_match_msgheader_attribute(gchar * nodename, gint * idx);

/**
  This function is used to pack the 'string data' received into a xml 'doc' 
  with identifier 'name' 

   @param  node   Xml Node pointer to pack the data 
   @param  doc    Xml Doc  pointer to pack the data
   @param  name   Tag Identifier of the data  in the xml document 
   @param  data   String to be packed in xml format
   @return none
*/

osso_pack_return_t osso_email_xml_pack_str(xmlNodePtr node, xmlDocPtr doc, void *name, void *data)
{
	osso_email_xml_pack_str_base64(node, doc, name, data);

	return OSSO_PACK_OK;
}

/**
  This function is used to unpack the 'string data' stored in a xml 'doc' 
  with identifier 'name' 

   @param  node   Xml Node pointer of the packed data 
   @param  doc    Xml Doc  pointer of the packed data
   @param  name   Tag Identifier of the string data  in the xml document 
   @param  data   unpaked string  data will be stored in this pointer  
   @return OSSO_PACK_OK on success (or) OSSO_PACK_INVALID for invalid 
           parameter (or) OSSO_PACK_ERROR for other kind of errors
*/

osso_pack_return_t osso_email_xml_unpack_str(xmlNodePtr node, xmlDocPtr doc, void *name, void *data)
{
	osso_email_xml_unpack_str_base64(node, doc, name, data);
	return OSSO_PACK_OK;
}

/**
  This function is used to pack the 'string data' received into a xml 'doc'     
  with identifier 'name'. This method is used to pack the string in base64
  format. 

   @param  node   Xml Node pointer to pack the data
   @param  doc    Xml Doc  pointer to pack the data
   @param  name   Tag Identifier of the data  in the xml document 
   @param  data   String to be packed in xml format
   @return OSSO_PACK_OK on success (or) OSSO_PACK_INVALID for invalid 
           parameter (or) OSSO_PACK_ERROR for other kind of errors
*/


osso_pack_return_t
osso_email_xml_pack_str_base64(xmlNodePtr node, xmlDocPtr doc, void *name, void *data)
{
	guchar *inbuf = NULL;
	gchar outbuf[B64_BUFFSIZE + 1];
	gint len = 0;
	GString *base64buf = NULL;

	if (node == NULL) {
		osso_log(LOG_INFO, "Invalid input params in %s", __FUNCTION__);
		return OSSO_PACK_INVALID;
	}

	inbuf = *((guchar **) data);

	if (inbuf != NULL) {
		len = strlen((gchar *) inbuf);
		base64buf = g_string_new(NULL);
		if (base64buf == NULL) {
			osso_log(LOG_ERR, "Memory alloc fail \n");
			return OSSO_PACK_ALLOC_FAIL;
		}
		while (len > 0) {
			memset(outbuf, '\0', B64_BUFFSIZE);
			if (len > B64_LINE_SIZE) {
				base64_encode_dbus(outbuf, inbuf, B64_LINE_SIZE);
				inbuf += B64_LINE_SIZE;
			} else {
				base64_encode_dbus(outbuf, inbuf, len);
				inbuf += len;
			}
			g_string_append_len(base64buf, outbuf, strlen(outbuf));

			len = strlen((gchar *) inbuf);
		}
		xmlNewChild(node, NULL, BAD_CAST(gchar *) name, BAD_CAST base64buf->str);
		g_string_free(base64buf, TRUE);
	} else {
		xmlNewChild(node, NULL, BAD_CAST(gchar *) name, BAD_CAST * ((gchar **) data));
	}
	return OSSO_PACK_OK;
}

/**
  This function is used to unpack the 'base64 formatted string data' 
  stored in a xml 'doc' with identifier 'name'

   @param  node   Xml Node pointer of the packed data
   @param  doc    Xml Doc  pointer of the packed data
   @param  name   Tag Identifier of the string data  in the xml document
   @param  data   unpaked string  data will be stored in this pointer
   @return OSSO_PACK_OK on success (or) OSSO_PACK_INVALID for invalid 
           parameter (or) OSSO_PACK_ERROR for other kind of errors
*/

osso_pack_return_t
osso_email_xml_unpack_str_base64(xmlNodePtr node, xmlDocPtr doc, void *name, void *data)
{
	gchar *str = NULL;
	guchar *outbuf = NULL;
	gint outbuflen = 0;

	if (node == NULL || doc == NULL || data == NULL) {
		osso_log(LOG_INFO, "Invalid input params in %s", __FUNCTION__);
		return OSSO_PACK_INVALID;
	}
	str = (gchar *) xmlNodeListGetString(doc, node->xmlChildrenNode, 1);
	if (str != NULL) {
		outbuf = g_malloc0(strlen((gchar *) str) + 1);
		if (outbuf == NULL) {
			osso_log(LOG_ERR, "Memory alloc fail \n");
			return OSSO_PACK_ALLOC_FAIL;
		}
		outbuflen = base64_decode_dbus(outbuf, str, -1);
		outbuf[outbuflen] = '\0';
		*((gchar **) data) = (gchar *) outbuf;
		xmlFree(str);
	} else {
		*((gchar **) data) = NULL;
	}
	return OSSO_PACK_OK;
}

/**
  This function is used to pack the 'integer data' received into a xml 'doc'
  with identifier 'name'.

   @param  node   Xml Node pointer to pack the data
   @param  doc    Xml Doc  pointer to pack the data
   @param  name   Tag Identifier of the integer data  in the xml document
   @param  data   integer to be packed in xml format
   @return OSSO_PACK_OK on success (or) OSSO_PACK_INVALID for invalid 
           parameter (or) OSSO_PACK_ERROR for other kind of errors
*/

osso_pack_return_t osso_email_xml_pack_int(xmlNodePtr node, xmlDocPtr doc, void *name, void *data)
{
	gchar temp[10];
	if (node == NULL) {
		osso_log(LOG_INFO, "Invalid input params in %s", __FUNCTION__);
		return OSSO_PACK_INVALID;
	}
	memset(temp, '\0', sizeof(temp));
	snprintf(temp, sizeof(temp), "%d", *(gint *) data);
	xmlNewChild(node, NULL, BAD_CAST name, BAD_CAST temp);
	return OSSO_PACK_OK;
}

/**
  This function is used to unpack the 'integer data' stored in a 
  xml 'doc' with identifier 'name'

   @param  node   Xml Node pointer of the packed data
   @param  doc    Xml Doc  pointer of the packed data
   @param  name   Tag Identifier of the integer data  in the xml document
   @param  data   unpaked integer  data will be stored in this pointer
   @return OSSO_PACK_OK on success (or) OSSO_PACK_INVALID for invalid 
           parameter (or) OSSO_PACK_ERROR for other kind of errors
*/

osso_pack_return_t osso_email_xml_unpack_int(xmlNodePtr node, xmlDocPtr doc, void *name, void *data)
{
	gchar *str = NULL;
	if (node == NULL || doc == NULL || data == NULL) {
		osso_log(LOG_INFO, "Invalid input params in %s", __FUNCTION__);
		return OSSO_PACK_INVALID;
	}
	str = (gchar *) xmlNodeListGetString(doc, node->xmlChildrenNode, 1);
	if (str != NULL) {
		*((gint *) data) = atoi(str);
		xmlFree(str);
	} else {
		*((gint *) data) = 0;
	}
	return OSSO_PACK_OK;
}

/**
  This function is used to pack the 'long integer data' received into 
  a xml 'doc' with identifier 'name'. 

   @param  node   Xml Node pointer to pack the data
   @param  doc    Xml Doc  pointer to pack the data
   @param  name   Tag Identifier of the long int data  in the xml document
   @param  data   Long Integer data to be packed in xml format
   @return OSSO_PACK_OK on success (or) OSSO_PACK_INVALID for invalid 
           parameter (or) OSSO_PACK_ERROR for other kind of errors
*/

osso_pack_return_t osso_email_xml_pack_long(xmlNodePtr node, xmlDocPtr doc, void *name, void *data)
{
	gchar temp[BUF_SIZE];
	if (node == NULL) {
		osso_log(LOG_INFO, "Invalid input params in %s", __FUNCTION__);
		return OSSO_PACK_INVALID;
	}
	memset(temp, '\0', sizeof(temp));
	snprintf(temp, sizeof(temp), "%ld", *(long *) data);
	xmlNewChild(node, NULL, BAD_CAST name, BAD_CAST temp);
	return OSSO_PACK_OK;
}

/**
  This function is used to unpack the 'long integer data' stored in a
  xml 'doc' with identifier 'name'

   @param  node   Xml Node pointer of the packed data
   @param  doc    Xml Doc  pointer of the packed data
   @param  name   Tag Identifier of the long  integer data in the xml document
   @param  data   unpaked lonng integer  data will be stored in this pointer
   @return OSSO_PACK_OK on success (or) OSSO_PACK_INVALID for invalid 
           parameter (or) OSSO_PACK_ERROR for other kind of errors
*/

osso_pack_return_t
osso_email_xml_unpack_long(xmlNodePtr node, xmlDocPtr doc, void *name, void *data)
{
	gchar *str = NULL;
	if (node == NULL || doc == NULL || data == NULL) {
		osso_log(LOG_INFO, "Invalid input params in %s", __FUNCTION__);
		return OSSO_PACK_INVALID;
	}
	str = (gchar *) xmlNodeListGetString(doc, node->xmlChildrenNode, 1);
	if (str != NULL) {
		*((long *) data) = atol(str);
		xmlFree(str);
	} else {
		*((long *) data) = 0;
	}
	return OSSO_PACK_OK;
}

/**
  This function is used to pack the 'unsigned short integer data' received
  into a xml 'doc' with identifier 'name'.

   @param  node   Xml Node pointer to pack the data
   @param  doc    Xml Doc  pointer to pack the data
   @param  name   Tag Identifier of the unsigned short int data  
                  in the xml document
   @param  data   Unsigned short Integer data to be packed in xml format
   @return OSSO_PACK_OK on success (or) OSSO_PACK_INVALID for invalid 
           parameter (or) OSSO_PACK_ERROR for other kind of errors
*/

osso_pack_return_t
osso_email_xml_pack_ushort(xmlNodePtr node, xmlDocPtr doc, void *name, void *data)
{
	gchar temp[BUF_SIZE];
	if (node == NULL) {
		osso_log(LOG_INFO, "Invalid input params in %s", __FUNCTION__);
		return OSSO_PACK_INVALID;
	}
	memset(temp, '\0', sizeof(temp));
	snprintf(temp, sizeof(temp), "%u", *(gint *) data);
	xmlNewChild(node, NULL, BAD_CAST name, BAD_CAST temp);
	return OSSO_PACK_OK;
}

/**
  This function is used to unpack the 'unsigned short integer data' 
  stored in a xml 'doc' with identifier 'name'

   @param  node   Xml Node pointer of the packed data
   @param  doc    Xml Doc  pointer of the packed data
   @param  name   Tag Identifier of the unsigned short data in the xml document
   @param  data   unsigned short integer data will be stored in this pointer
   @return OSSO_PACK_OK on success (or) OSSO_PACK_INVALID for invalid 
           parameter (or) OSSO_PACK_ERROR for other kind of errors
*/


osso_pack_return_t
osso_email_xml_unpack_ushort(xmlNodePtr node, xmlDocPtr doc, void *name, void *data)
{
	gchar *str = NULL;
	if (node == NULL || doc == NULL || data == NULL) {
		osso_log(LOG_INFO, "Invalid input params in %s", __FUNCTION__);
		return OSSO_PACK_INVALID;
	}
	str = (gchar *) xmlNodeListGetString(doc, node->xmlChildrenNode, 1);
	if (str != NULL) {
		*((gushort *) data) = atoi(str);
	} else {
		*((gushort *) data) = 0;
	}
	xmlFree(str);
	return OSSO_PACK_OK;
}

/**
  This function is used to pack the 'boolean data' received     
  into a xml 'doc' with identifier 'name'.

   @param  node   Xml Node pointer to pack the data
   @param  doc    Xml Doc  pointer to pack the data
   @param  name   Tag Identifier of the boolean data in the xml document
   @param  data   Boolean value to be packed in xml format
   @return OSSO_PACK_OK on success (or) OSSO_PACK_INVALID for invalid 
           parameter (or) OSSO_PACK_ERROR for other kind of errors
*/

osso_pack_return_t osso_email_xml_pack_bool(xmlNodePtr node, xmlDocPtr doc, void *name, void *data)
{
	gchar temp[BUF_SIZE];
	if (node == NULL) {
		osso_log(LOG_INFO, "Invalid input params in %s", __FUNCTION__);
		return OSSO_PACK_INVALID;
	}
	memset(temp, '\0', sizeof(temp));
	snprintf(temp, sizeof(temp), "%d", *(int *) data);
	xmlNewChild(node, NULL, BAD_CAST name, BAD_CAST temp);
	return OSSO_PACK_OK;
}

/**
  This function is used to unpack the 'boolean data' stored in a 
  xml 'doc' with identifier 'name'

   @param  node   Xml Node pointer of the packed data
   @param  doc    Xml Doc  pointer of the packed data
   @param  name   Tag Identifier of the boolean data in the xml document
   @param  data   Boolean data will be stored in this pointer
   @return OSSO_PACK_OK on success (or) OSSO_PACK_INVALID for invalid 
           parameter (or) OSSO_PACK_ERROR for other kind of errors
*/


osso_pack_return_t
osso_email_xml_unpack_bool(xmlNodePtr node, xmlDocPtr doc, void *name, void *data)
{
	gchar *str = NULL;
	if (node == NULL || doc == NULL || data == NULL) {
		osso_log(LOG_INFO, "Invalid input params in %s", __FUNCTION__);
		return OSSO_PACK_INVALID;
	}
	str = (gchar *) xmlNodeListGetString(doc, node->xmlChildrenNode, 1);
	if (str != NULL) {
		*((gboolean *) data) = (gboolean) (atoi(str));
	} else {
		*((gboolean *) data) = 0;
	}
	xmlFree(str);
	return OSSO_PACK_OK;
}

/**
  This function is used to pack the 'structure data ' received
  into a xml 'doc' with identifier 'name'.

   @param  node   Xml Node pointer to pack the data
   @param  doc    Xml Doc  pointer to pack the data
   @param  name   Tag Identifier of the structure data in the xml document
   @param  data   Structure data to be packed in xml format
   @return OSSO_PACK_OK on success (or) OSSO_PACK_INVALID for invalid 
           parameter (or) OSSO_PACK_ERROR for other kind of errors
*/

static osso_pack_return_t
osso_email_xml_pack_struct(xmlNodePtr node, xmlDocPtr doc, void *name, void *sdata)
{
	DBusData *cur = NULL;
	MimeInfo *childInfo = NULL;
	MimeInfo *parentInfo = NULL;
	GSList *imageinfo_list = NULL;
	GSList *f_pos_list = NULL;

	if (node == NULL || doc == NULL || sdata == NULL) {
		osso_log(LOG_INFO, "Invalid input params in %s", __FUNCTION__);
		return OSSO_PACK_INVALID;
	}

	for (cur = (DBusData *) sdata; cur->name != NULL; cur++) {
		switch (cur->type) {
		case D_INT:
		case D_USHORT:
		case D_LONG:
		case D_STRING:
		case D_BOOL:
		case D_ENUM:
		case D_STRUCT:
		case D_BOOLARR:
			if ((cur != NULL) && (cur->packFunc != NULL)) {
				cur->packFunc(node, doc, cur->name, cur->data);
			}
			break;
		case D_MIME_INLINE_INFO_LIST:
			imageinfo_list = ((MimeInfo *) cur->data)->inline_images;
			if (imageinfo_list == NULL) {
				break;
			}
			if ((cur != NULL) && (cur->packFunc != NULL)) {
				cur->packFunc(node, doc, cur->name, imageinfo_list);
			}
			break;
		case D_INT_LIST:
			f_pos_list = ((MimeInfo *) cur->data)->file_pos_list;
			if (f_pos_list == NULL) {
				break;
			}
			if ((cur != NULL) && (cur->packFunc != NULL)) {
				cur->packFunc(node, doc, cur->name, f_pos_list);
			}
			break;
		case D_PARENT:
			/* This is bad Just need boundary info of parent */
			if (name == NULL) {
				break;
			}
			if (g_ascii_strcasecmp((char *) name, XML_MIME_INFO) == 0) {
				parentInfo = ((MimeInfo *) cur->data)->parent;
				if ((parentInfo != NULL)
				    && (parentInfo->boundary != NULL)) {
					if ((cur != NULL) && (cur->packFunc != NULL)) {
						cur->packFunc(node, doc, cur->name,
							      &parentInfo->boundary);
					}
				}
			}
			break;
		case D_CHILD:
			childInfo = ((MimeInfo *) cur->data)->children;
			while (childInfo) {
				if ((cur != NULL) && (cur->packFunc != NULL)) {
					cur->packFunc(node, doc, cur->name, childInfo);
				}
				childInfo = childInfo->next;
			}
			break;
		case D_SUB:
			if (((MimeInfo *) cur->data)->sub != NULL) {
				if ((cur != NULL) && (cur->packFunc != NULL)) {
					cur->packFunc(node, doc, cur->name,
						      ((MimeInfo *) cur->data)->sub);
				}
			}
			break;
		default:
			osso_log(LOG_INFO, "Invalid option... \n");
			break;
		}
	}
	return OSSO_PACK_OK;
}

/**
  This function is used to pack the 'MsgInfo structure data ' received
  into a xml 'doc' with identifier 'name'.

   @param  node   Xml Node pointer to pack the data
   @param  doc    Xml Doc  pointer to pack the data
   @param  name   Tag Identifier of the MsgInfo struct in the xml document
   @param  data   MsgInfo Structure data to be packed in xml format
   @return OSSO_PACK_OK on success (or) OSSO_PACK_INVALID for invalid 
           parameter (or) OSSO_PACK_ERROR for other kind of errors
*/


osso_pack_return_t
osso_email_xml_pack_msginfo(xmlNodePtr node, xmlDocPtr doc, void *nodeType, void *msgInfo)
{
	xmlNodePtr root = NULL, child = NULL;
	MsgInfo *minfo = NULL;

	if (doc == NULL || msgInfo == NULL || nodeType == NULL) {
		osso_log(LOG_INFO, "Invalid input params in %s", __FUNCTION__);
		return OSSO_PACK_INVALID;
	}
	minfo = (MsgInfo *) msgInfo;
	if (node == NULL) {
		root = xmlNewNode(NULL, BAD_CAST nodeType);
		xmlAddChild(node, root);
		node = root;
	}
	child = xmlNewNode(NULL, BAD_CAST nodeType);
	xmlAddChild(node, child);
	node = child;

	osso_email_xml_pack_msgheader(node, doc, XML_MSG_HEADER, minfo->header);
	osso_email_xml_pack_settings(node, doc, XML_SETTINGS, minfo->settings);
	osso_email_xml_pack_inlineimageinfo_list(node, doc,
						 XML_INLINEIMAGE_INFO_LST, minfo->image_list);
	osso_email_xml_pack_attachinfo_list(node, doc, XML_ATTACHINFO_LST, minfo->attach_list);
	osso_email_xml_pack_int_list(node, doc, XML_CERTS_UID_LST, minfo->recip_certs_uid);
	osso_email_xml_pack_string_list(node, doc, XML_ALGOS_LST, minfo->recip_algos);
	osso_email_xml_pack_str(node, doc, PASSWORD_KEY, &minfo->password);

	return OSSO_PACK_OK;
}

/**
  This function is used to pack the 'MsgHeader structure data ' received
  into a xml 'doc' with identifier 'name'.

   @param  node   Xml Node pointer to pack the data
   @param  doc    Xml Doc  pointer to pack the data
   @param  name   Tag Identifier of the Msgheader struct in the xml document
   @param  data   MsgHeader Structure data to be packed in xml format
   @return OSSO_PACK_OK on success (or) OSSO_PACK_INVALID for invalid 
           parameter (or) OSSO_PACK_ERROR for other kind of errors
*/

osso_pack_return_t
osso_email_xml_pack_msgheader(xmlNodePtr node, xmlDocPtr doc, void *nodeType, void *msgHeader)
{
	xmlNodePtr root = NULL, child = NULL;

	if ((doc == NULL) || (msgHeader == NULL) || (nodeType == NULL)) {
		osso_log(LOG_INFO, "Invalid input params in %s", __FUNCTION__);
		return OSSO_PACK_INVALID;
	}
	NEW_NODE(node, root, child, nodeType);
	memset(&tmpMsgHeader, '\0', sizeof(tmpMsgHeader));
	tmpMsgHeader = *(MsgHeader *) msgHeader;
	return osso_email_xml_pack_struct(node, doc, NULL, MsgHeaderDBusData);
}

/**
  This function is used to pack the 'Folder structure data ' received
  into a xml 'doc' with identifier 'name'.

   @param  node   Xml Node pointer to pack the data
   @param  doc    Xml Doc  pointer to pack the data
   @param  name   Tag Identifier of the Folder structin the xml document
   @param  data   Folder Structure data to be packed in xml format
   @return OSSO_PACK_OK on success (or) OSSO_PACK_INVALID for invalid 
           parameter (or) OSSO_PACK_ERROR for other kind of errors
*/

osso_pack_return_t
osso_email_xml_pack_folder(xmlNodePtr node, xmlDocPtr doc, void *nodeType, void *folder)
{
	xmlNodePtr root = NULL, child = NULL;
	if ((doc == NULL) || (folder == NULL) || (nodeType == NULL)) {
		osso_log(LOG_INFO, "Invalid input params in %s", __FUNCTION__);
		return OSSO_PACK_INVALID;
	}
	NEW_NODE(node, root, child, nodeType);
	memset(&tmpFolder, '\0', sizeof(tmpFolder));
	tmpFolder = *(Folder *) folder;
	return osso_email_xml_pack_struct(node, doc, NULL, FolderDBusData);
}

/**
  This function is used to pack the 'FolderAttachinfo structure data ' 
  received into a xml 'doc' with identifier 'name'.

   @param  node   Xml Node pointer to pack the data
   @param  doc    Xml Doc  pointer to pack the data
   @param  name   Tag Identifier of the Folder Attachnfo structure 
                  in the xml document
   @param  data   Folder AttchInfo Structure data to be packed in xml format
   @return OSSO_PACK_OK on success (or) OSSO_PACK_INVALID for invalid 
           parameter (or) OSSO_PACK_ERROR for other kind of errors
*/

osso_pack_return_t
osso_email_xml_pack_folder_attachinfo(xmlNodePtr node, xmlDocPtr doc,
				      void *nodeType, void *folder_attach_info)
{
	xmlNodePtr root = NULL, child = NULL;
	if ((doc == NULL) || (folder_attach_info == NULL) || (nodeType == NULL)) {
		osso_log(LOG_INFO, "Invalid input params in %s", __FUNCTION__);
		return OSSO_PACK_INVALID;
	}
	NEW_NODE(node, root, child, nodeType);
	memset(&tmpFolderAttachInfo, '\0', sizeof(tmpFolderAttachInfo));
	tmpFolderAttachInfo = *(FolderAttachInfo *) folder_attach_info;
	return osso_email_xml_pack_struct(node, doc, NULL, FolderAttachInfoDBusData);
}

/**
  This function is used to pack the 'MsgFlags structure data ' received 
  into a xml 'doc' with identifier 'name'.

   @param  node   Xml Node pointer to pack the data
   @param  doc    Xml Doc  pointer to pack the data
   @param  name   Tag Identifier of the MsgFlags struct in the xml document
   @param  data   MsgFlags Structure data to be packed in xml format
   @return OSSO_PACK_OK on success (or) OSSO_PACK_INVALID for invalid 
           parameter (or) OSSO_PACK_ERROR for other kind of errors
*/

osso_pack_return_t
osso_email_xml_pack_msgflags(xmlNodePtr node, xmlDocPtr doc, void *nodeType, void *flags)
{
	xmlNodePtr root = NULL, child = NULL;

	if ((doc == NULL) || (flags == NULL) || (nodeType == NULL)) {
		osso_log(LOG_INFO, "Invalid input params in %s", __FUNCTION__);
		return OSSO_PACK_INVALID;
	}
	NEW_NODE(node, root, child, nodeType);
	memset(&tmpMsgFlags, '\0', sizeof(tmpMsgFlags));
	tmpMsgFlags = *(MsgFlags *) flags;
	osso_email_xml_pack_struct(node, doc, NULL, MsgFlagsDBusData);
	node = root;
	return OSSO_PACK_OK;
}

/**
  This function is used to pack the 'Message tag structure data ' received 
  into a xml 'doc' with identifier 'name'.

   @param  node   Xml Node pointer to pack the data
   @param  doc    Xml Doc  pointer to pack the data
   @param  name   Tag Identifier of the MsgTag struct in the xml document
   @param  data   Tag Structure data to be packed in xml format
   @return OSSO_PACK_OK on success (or) OSSO_PACK_INVALID for invalid 
           parameter (or) OSSO_PACK_ERROR for other kind of errors
*/

osso_pack_return_t
osso_email_xml_pack_msgtag(xmlNodePtr node, xmlDocPtr doc, void *nodeType, void *tag)
{
	xmlNodePtr root = NULL, child = NULL;

	if ((doc == NULL) || (tag == NULL) || (nodeType == NULL)) {
		osso_log(LOG_INFO, "Invalid input params in %s", __FUNCTION__);
		return OSSO_PACK_INVALID;
	}

	NEW_NODE(node, root, child, nodeType);
	memset(&tmpMsgTag, '\0', sizeof(tmpMsgTag));
	tmpMsgTag = *(Tag *) tag;
	return osso_email_xml_pack_struct(node, doc, NULL, MsgTagDBusData);
}

/**
  This function is used to pack the 'EmailSpecific settings data' received
  into a xml 'doc' with identifier 'name'.

   @param  node   Xml Node pointer to pack the data
   @param  doc    Xml Doc  pointer to pack the data
   @param  name   Tag Identifier of the EmailSpecificSettings struct 
                  data in the xml document
   @param  data   EmailSpecificSettings Structure data to be packed 
                  in xml format
   @return OSSO_PACK_OK on success (or) OSSO_PACK_INVALID for invalid 
           parameter (or) OSSO_PACK_ERROR for other kind of errors
*/

osso_pack_return_t
osso_email_xml_pack_settings(xmlNodePtr node, xmlDocPtr doc, void *nodeType, void *settings)
{
	xmlNodePtr root = NULL, child = NULL;

	if ((doc == NULL) || (settings == NULL) || (nodeType == NULL)) {
		osso_log(LOG_INFO, "Invalid input params in %s", __FUNCTION__);
		return OSSO_PACK_INVALID;
	}
	NEW_NODE(node, root, child, nodeType);

	memset(&tmpSettings, '\0', sizeof(tmpSettings));
	tmpSettings = *(EmailSpecificSettings *) settings;
	return osso_email_xml_pack_struct(node, doc, NULL, EMailSettingsDBusData);
}


/**
  This function is used to pack the 'InlineImageInfo struct  data' received
  into a xml 'doc' with identifier 'name'.

   @param  node   Xml Node pointer to pack the data
   @param  doc    Xml Doc  pointer to pack the data
   @param  name   Tag Identifier of the InlineImageInfo struct in the 
                  xml document
   @param  data   InlineImageInfo Structure data to be packed 
                  in xml format
   @return OSSO_PACK_OK on success (or) OSSO_PACK_INVALID for invalid 
           parameter (or) OSSO_PACK_ERROR for other kind of errors
*/

osso_pack_return_t
osso_email_xml_pack_inlineimageinfo(xmlNodePtr node, xmlDocPtr doc, void *nodeType, void *imageinfo)
{
	xmlNodePtr root = NULL, child = NULL;

	if ((doc == NULL) || (imageinfo) == NULL || (nodeType == NULL)) {
		osso_log(LOG_INFO, "Invalid input params in %s", __FUNCTION__);
		return OSSO_PACK_INVALID;
	}
	NEW_NODE(node, root, child, nodeType);

	memset(&tmpInlineImageInfo, '\0', sizeof(tmpInlineImageInfo));
	tmpInlineImageInfo = *(InlineImageInfo *) imageinfo;
	return osso_email_xml_pack_struct(node, doc, NULL, InlineImageInfoDBusData);
}

/**
  This function is used to pack the 'GSList of InlineImageInfo struct 
  data' received into a xml 'doc' with identifier 'name'.

   @param  node   Xml Node pointer to pack the data
   @param  doc    Xml Doc  pointer to pack the data
   @param  name   Tag Identifier of the GSList of InlineImageInfo struct
                  data in the xml document
   @param  data   GSList of InlineImageInfo Structure data to be packed
                  in xml format
   @return OSSO_PACK_OK on success (or) OSSO_PACK_INVALID for invalid 
           parameter (or) OSSO_PACK_ERROR for other kind of errors
*/

osso_pack_return_t
osso_email_xml_pack_inlineimageinfo_list(xmlNodePtr node, xmlDocPtr doc, void *nodeType, void *data)
{
	xmlNodePtr root = NULL, child = NULL;
	GSList *image_list = NULL;
	GSList *cur = NULL;


	if ((doc == NULL) || (data) == NULL || (nodeType == NULL)) {
		osso_log(LOG_INFO, "Invalid input params in %s", __FUNCTION__);
		return OSSO_PACK_INVALID;
	}
	image_list = (GSList *) data;

	NEW_NODE(node, root, child, nodeType);
	for (cur = image_list; cur != NULL; cur = cur->next) {
		if (cur->data == NULL) {
			continue;
		}
		osso_email_xml_pack_inlineimageinfo(node, doc, XML_INLINEIMAGE_INFO, cur->data);
	}
	return OSSO_PACK_OK;
}



osso_pack_return_t
osso_email_xml_pack_int_list(xmlNodePtr node, xmlDocPtr doc, void *nodeType, void *data)
{
	xmlNodePtr root = NULL, child = NULL;
	GSList *f_pos_list = NULL;
	GSList *cur = NULL;


	if ((doc == NULL) || (data) == NULL || (nodeType == NULL)) {
		osso_log(LOG_INFO, "Invalid input params in %s", __FUNCTION__);
		return OSSO_PACK_INVALID;
	}
	f_pos_list = (GSList *) data;

	NEW_NODE(node, root, child, nodeType);
	for (cur = f_pos_list; cur != NULL; cur = cur->next) {
		/* this check is not required as the memeory is not allocated 
		 * in the list and GUINT_TO_POINTER is used instead */
#if 0
		if (cur->data == NULL) {
			continue;
		}
#endif
		osso_email_xml_pack_int(node, doc, XML_INT, &cur->data);
	}
	return OSSO_PACK_OK;
}

/**
  This function is used to pack the 'Attchinfo struct  data' received
  into a xml 'doc' with identifier 'name'.

   @param  node   Xml Node pointer to pack the data
   @param  doc    Xml Doc  pointer to pack the data
   @param  name   Tag Identifier of the Attchinfo struct in the xml document
   @param  data   Attchinfo  Structure data to be packed 
                  in xml format
   @return OSSO_PACK_OK on success (or) OSSO_PACK_INVALID for invalid 
           parameter (or) OSSO_PACK_ERROR for other kind of errors
*/

osso_pack_return_t
osso_email_xml_pack_attachinfo(xmlNodePtr node, xmlDocPtr doc, void *nodeType, void *attachinfo)
{
	xmlNodePtr root = NULL, child = NULL;

	if ((doc == NULL) || (attachinfo) == NULL || (nodeType == NULL)) {
		osso_log(LOG_INFO, "Invalid input params in %s", __FUNCTION__);
		return OSSO_PACK_INVALID;
	}
	NEW_NODE(node, root, child, nodeType);

	memset(&tmpAttachInfo, '\0', sizeof(tmpAttachInfo));
	tmpAttachInfo = *(AttachInfo *) attachinfo;
	return osso_email_xml_pack_struct(node, doc, NULL, AttachInfoDBusData);
}

/**
  This function is used to pack the 'GSList of Attchinfo struct  data' 
  received into a xml 'doc' with identifier 'name'.

   @param  node   Xml Node pointer to pack the data
   @param  doc    Xml Doc  pointer to pack the data
   @param  name   Tag Identifier of the GSList of AttachInfo struct
                  data in the xml document
   @param  data   GSList of Attchinfo Structure data to be packed
                  in xml format
   @return OSSO_PACK_OK on success (or) OSSO_PACK_INVALID for invalid 
           parameter (or) OSSO_PACK_ERROR for other kind of errors
*/

osso_pack_return_t
osso_email_xml_pack_attachinfo_list(xmlNodePtr node, xmlDocPtr doc, void *nodeType, void *data)
{
	xmlNodePtr root = NULL, child = NULL;
	GSList *attach_list = NULL;
	GSList *cur = NULL;


	if ((doc == NULL) || (data) == NULL || (nodeType == NULL)) {
		osso_log(LOG_INFO, "Invalid input params in %s", __FUNCTION__);
		return OSSO_PACK_INVALID;
	}
	attach_list = (GSList *) data;

	NEW_NODE(node, root, child, nodeType);

	for (cur = attach_list; cur != NULL; cur = cur->next) {
		if (cur->data == NULL) {
			continue;
		}
		osso_email_xml_pack_attachinfo(node, doc, XML_ATTACHINFO, cur->data);
	}
	return OSSO_PACK_OK;
}

/**
  This function is used to pack the 'GSList of Folder struct  data' 
  received into a xml 'doc' with identifier 'name'.

   @param  node   Xml Node pointer to pack the data
   @param  doc    Xml Doc  pointer to pack the data
   @param  name   Tag Identifier of the GSList of Folder struct data i
                  in the xml document
   @param  data   GSList of Folder Structure data to be packed
                  in xml format
   @return OSSO_PACK_OK on success (or) OSSO_PACK_INVALID for invalid 
           parameter (or) OSSO_PACK_ERROR for other kind of errors
*/

osso_pack_return_t
osso_email_xml_pack_folder_list(xmlNodePtr node, xmlDocPtr doc, void *nodeType, void *data)
{
	xmlNodePtr root = NULL, child = NULL;
	GSList *folder_list = NULL;
	GSList *cur = NULL;
	if ((doc == NULL) || (data) == NULL || (nodeType == NULL)) {
		osso_log(LOG_INFO, "Invalid input params in %s", __FUNCTION__);
		return OSSO_PACK_INVALID;
	}
	folder_list = (GSList *) data;
	NEW_NODE(node, root, child, nodeType);
	for (cur = folder_list; cur != NULL; cur = cur->next) {
		if (cur->data == NULL) {
			continue;
		}
		osso_email_xml_pack_folder(node, doc, XML_FOLDER, cur->data);
	}
	return OSSO_PACK_OK;
}

/**
  This function is used to pack the 'GSList of strings data'
  received into a xml 'doc' with identifier 'name'.

   @param  node   Xml Node pointer to pack the data
   @param  doc    Xml Doc  pointer to pack the data
   @param  name   Tag Identifier of the structure data in the xml document
   @param  data   GSList of Strings data to be packed in xml format
   @return OSSO_PACK_OK on success (or) OSSO_PACK_INVALID for invalid 
           parameter (or) OSSO_PACK_ERROR for other kind of errors
*/

osso_pack_return_t
osso_email_xml_pack_string_list(xmlNodePtr node, xmlDocPtr doc, void *nodeType, void *data)
{
	xmlNodePtr root = NULL, child = NULL;
	GSList *string_list = NULL;
	GSList *cur = NULL;
	if ((doc == NULL) || (data) == NULL || (nodeType == NULL)) {
		osso_log(LOG_INFO, "Invalid input params in %s", __FUNCTION__);
		return OSSO_PACK_INVALID;
	}
	string_list = (GSList *) data;
	NEW_NODE(node, root, child, nodeType);
	for (cur = string_list; cur != NULL; cur = cur->next) {
		if (cur->data == NULL) {
			continue;
		}
		osso_email_xml_pack_str(node, doc, XML_STRING, &cur->data);
	}
	return OSSO_PACK_OK;
}

/**
  This function is used to unpack the 'GSList of Strings  data' 
  stored in a xml 'doc' with identifier 'name'

   @param  node   Xml Node pointer of the packed data
   @param  doc    Xml Doc  pointer of the packed data
   @param  name   Tag Identifier of the GSList of Strings data 
                  in the xml document
   @param  data  GSList of Strings data will be stored in this pointer
   @return OSSO_PACK_OK on success (or) OSSO_PACK_INVALID for invalid 
           parameter (or) OSSO_PACK_ERROR for other kind of errors
*/


osso_pack_return_t
osso_email_xml_unpack_string_list(xmlNodePtr node, xmlDocPtr doc, gchar * nodeType, GSList ** data)
{
	GSList *string_list = NULL;
	gchar *str = NULL;
	if (doc == NULL || node == NULL || data == NULL) {
		osso_log(LOG_INFO, "Invalid input params in %s", __FUNCTION__);
		return OSSO_PACK_INVALID;
	}

	for (node = node->xmlChildrenNode; node != NULL; node = node->next) {
		if (node->type == XML_ELEMENT_NODE) {
			if (g_ascii_strcasecmp((gchar *) node->name, XML_STRING) == 0) {
				osso_email_xml_unpack_str(node, doc, XML_STRING, &str);
				string_list = g_slist_append(string_list, str);
			}
		}
	}
	*data = string_list;
	return OSSO_PACK_OK;
}

/**
  This function is used to pack the 'GSList of MsgIds data'
  received into a xml 'doc' with identifier 'name'.

   @param  node   Xml Node pointer to pack the data
   @param  doc    Xml Doc  pointer to pack the data
   @param  name   Tag Identifier of the structure data in the xml document
   @param  data   GSList of MsgIds data to be packed in xml format
   @return OSSO_PACK_OK on success (or) OSSO_PACK_INVALID for invalid 
           parameter (or) OSSO_PACK_ERROR for other kind of errors
*/

osso_pack_return_t
osso_email_xml_pack_msgid_list(xmlNodePtr node, xmlDocPtr doc, void *nodeType, void *data)
{
	xmlNodePtr root = NULL, child = NULL;
	GSList *msgid_list = NULL;
	GSList *cur = NULL;
	if ((doc == NULL) || (data) == NULL || (nodeType == NULL)) {
		osso_log(LOG_INFO, "Invalid input params in %s", __FUNCTION__);
		return OSSO_PACK_INVALID;
	}
	msgid_list = (GSList *) data;
	NEW_NODE(node, root, child, nodeType);
	for (cur = msgid_list; cur != NULL; cur = cur->next) {
		if (cur->data == NULL) {
			continue;
		}
		osso_email_xml_pack_str(node, doc, XML_MSG_ID, &cur->data);
	}
	return OSSO_PACK_OK;
}

/**
  This function is used to unpack the 'GSList of MsgIds  data'
  stored in a xml 'doc' with identifier 'name'

   @param  node   Xml Node pointer of the packed data
   @param  doc    Xml Doc  pointer of the packed data
   @param  name   Tag Identifier of the GSList of MsgIds data 
                  in the xml document
   @param  data  GSList of MsgIds data will be stored in this pointer
   @return OSSO_PACK_OK on success (or) OSSO_PACK_INVALID for invalid 
           parameter (or) OSSO_PACK_ERROR for other kind of errors
*/

osso_pack_return_t
osso_email_xml_unpack_msgid_list(xmlNodePtr node, xmlDocPtr doc, gchar * nodeType, GSList ** data)
{
	GSList *msgid_list = NULL;
	gchar *msgid = NULL;
	if (doc == NULL || node == NULL || data == NULL) {
		osso_log(LOG_INFO, "Invalid input params in %s", __FUNCTION__);
		return OSSO_PACK_INVALID;
	}

	for (node = node->xmlChildrenNode; node != NULL; node = node->next) {
		if (node->type == XML_ELEMENT_NODE) {
			if (g_ascii_strcasecmp((gchar *) node->name, XML_MSG_ID) == 0) {
				osso_email_xml_unpack_str(node, doc, XML_MSG_ID, &msgid);
				msgid_list = g_slist_append(msgid_list, msgid);
			}
		}
	}
	*data = msgid_list;
	return OSSO_PACK_OK;
}

/**
  This function is used to pack the 'GSList of Folder AttachInfo struct data'
  received into a xml 'doc' with identifier 'name'.

   @param  node   Xml Node pointer to pack the data
   @param  doc    Xml Doc  pointer to pack the data
   @param  name   Tag Identifier of the structure data in the xml document
   @param  data   GSList of Folder AttachInfo struct data to be packed 
                  in xml format
   @return OSSO_PACK_OK on success (or) OSSO_PACK_INVALID for invalid 
           parameter (or) OSSO_PACK_ERROR for other kind of errors
*/

osso_pack_return_t
osso_email_xml_pack_folder_attachinfo_list(xmlNodePtr node, xmlDocPtr doc,
					   void *nodeType, void *data)
{
	xmlNodePtr root = NULL, child = NULL;
	GSList *folder_list = NULL;
	GSList *cur = NULL;
	if ((doc == NULL) || (data) == NULL || (nodeType == NULL)) {
		osso_log(LOG_INFO, "Invalid input params in %s", __FUNCTION__);
		return OSSO_PACK_INVALID;
	}
	folder_list = (GSList *) data;
	NEW_NODE(node, root, child, nodeType);
	for (cur = folder_list; cur != NULL; cur = cur->next) {
		if (cur->data == NULL) {
			continue;
		}
		osso_email_xml_pack_folder_attachinfo(node, doc, XML_FOLDER_ATTACHINFO, cur->data);
	}
	return OSSO_PACK_OK;
}

/**
  This function is used to pack the 'GSList of MsgInfo struct data'
  received into a xml 'doc' with identifier 'name'.

   @param  node   Xml Node pointer to pack the data
   @param  doc    Xml Doc  pointer to pack the data
   @param  name   Tag Identifier of the structure data in the xml document
   @param  data   GSList of MsgInfo struct data to be packed 
                  in xml format
   @return OSSO_PACK_OK on success (or) OSSO_PACK_INVALID for invalid 
           parameter (or) OSSO_PACK_ERROR for other kind of errors
*/

osso_pack_return_t
osso_email_xml_pack_msginfo_list(xmlNodePtr node, xmlDocPtr doc, void *nodeType, void *data)
{
	xmlNodePtr root = NULL, child = NULL;
	GSList *cur = NULL;
	GSList *msginfo_list = NULL;

	if ((data == NULL) || (nodeType == NULL)) {
		osso_log(LOG_INFO, "Invalid input params in %s", __FUNCTION__);
		return OSSO_PACK_INVALID;
	}
	msginfo_list = (GSList *) data;

	NEW_NODE(node, root, child, nodeType);

	for (cur = msginfo_list; cur != NULL; cur = cur->next) {
		if (cur->data == NULL) {
			continue;
		}
		osso_email_xml_pack_msginfo(node, doc, XML_MSG_INFO, cur->data);
	}
	return OSSO_PACK_OK;
}

/**
  This function is used to pack the 'GSList of MimeInfo struct data'
  received into a xml 'doc' with identifier 'name'.

   @param  node   Xml Node pointer to pack the data
   @param  doc    Xml Doc  pointer to pack the data
   @param  name   Tag Identifier of the structure data in the xml document
   @param  data   GSList of MimeInfo struct data to be packed 
                  in xml format
   @return OSSO_PACK_OK on success (or) OSSO_PACK_INVALID for invalid 
           parameter (or) OSSO_PACK_ERROR for other kind of errors
*/

osso_pack_return_t
osso_email_xml_pack_mimeinfo_list(xmlNodePtr node, xmlDocPtr doc, void *nodeType, void *data)
{
	xmlNodePtr root = NULL, child = NULL;
	GSList *cur = NULL;
	GSList *mimeinfo_list = NULL;

	if ((data == NULL) || (nodeType == NULL)) {
		osso_log(LOG_INFO, "Invalid input params in %s", __FUNCTION__);
		return OSSO_PACK_INVALID;
	}
	mimeinfo_list = (GSList *) data;

	NEW_NODE(node, root, child, nodeType);

	for (cur = mimeinfo_list; cur != NULL; cur = cur->next) {
		if (cur->data == NULL) {
			continue;
		}
		osso_email_xml_pack_mimeinfo(node, doc, XML_MIME_INFO, cur->data);
	}
	return OSSO_PACK_OK;
}

/**
  This function is used to pack the 'GSList of MsgHeader struct data'
  received into a xml 'doc' with identifier 'name'.

   @param  node   Xml Node pointer to pack the data
   @param  doc    Xml Doc  pointer to pack the data
   @param  name   Tag Identifier of the structure data in the xml document
   @param  data   GSList of MsgHeader struct data to be packed
                  in xml format
   @return OSSO_PACK_OK on success (or) OSSO_PACK_INVALID for invalid 
           parameter (or) OSSO_PACK_ERROR for other kind of errors
*/

osso_pack_return_t
osso_email_xml_pack_msgheader_list(xmlNodePtr node, xmlDocPtr doc, void *nodeType, void *data)
{
	xmlNodePtr root = NULL, child = NULL;
	GSList *cur = NULL;
	GSList *msgheader_list = NULL;

	if ((data == NULL) || (nodeType == NULL)) {
		osso_log(LOG_INFO, "Invalid input params in %s", __FUNCTION__);
		return OSSO_PACK_INVALID;
	}
	msgheader_list = (GSList *) data;

	NEW_NODE(node, root, child, nodeType);

	for (cur = msgheader_list; cur != NULL; cur = cur->next) {
		if (cur->data == NULL) {
			continue;
		}
		osso_email_xml_pack_msgheader(node, doc, "MsgHeader", cur->data);
	}
	return OSSO_PACK_OK;
}

/**
  This function is used to pack the 'MimeInfo struct data'
  received into a xml 'doc' with identifier 'name'.

   @param  node   Xml Node pointer to pack the data
   @param  doc    Xml Doc  pointer to pack the data
   @param  name   Tag Identifier of the structure data in the xml document
   @param  data   MimeInfo struct data to be packed in xml format
   @return OSSO_PACK_OK on success (or) OSSO_PACK_INVALID for invalid 
           parameter (or) OSSO_PACK_ERROR for other kind of errors
*/

osso_pack_return_t
osso_email_xml_pack_mimeinfo(xmlNodePtr node, xmlDocPtr doc, void *nodeType, void *mimeInfo)
{
	xmlNodePtr root = NULL, child = NULL;
	if (mimeInfo == NULL) {
		osso_log(LOG_INFO, "Invalid input params in %s", __FUNCTION__);
		return OSSO_PACK_INVALID;
	}

	NEW_NODE(node, root, child, nodeType);
	memset(&tmpMimeInfo, '\0', sizeof(tmpMimeInfo));
	tmpMimeInfo = *(MimeInfo *) mimeInfo;
	return osso_email_xml_pack_struct(node, doc, nodeType, MimeInfoDBusData);
}

/**
  This function is used to get the index matching element in MimeInfo struct  

   @param  nodename  node element name  
   @param  index  index of the matching element 

   @return OSSO_PACK_OK on success or ERROR failure 
*/

static gboolean osso_email_xml_unpack_match_mimeinfo_attribute(gchar * nodename, gint * idx)
{
	gint index = 0;
	if ((nodename == NULL) || (idx == NULL)) {
		return FALSE;
	}
	for (index = 0; MimeInfoDBusData[index].name != NULL; index++) {
		if (g_ascii_strcasecmp(nodename, MimeInfoDBusData[index].name) == 0) {
			*idx = index;
			return TRUE;
		}
	}
	return FALSE;
}

/**
  This function is used to unpack the 'MimeInfo struct  data'
  stored in a xml 'doc' with identifier 'name'

   @param  node   Xml Node pointer of the packed data
   @param  doc    Xml Doc  pointer of the packed data
   @param  name   Tag Identifier of the MimeInfo  data
                  in the xml document
   @param  data   MimeInfo struct data will be stored in this pointer
   @return OSSO_PACK_OK on success (or) OSSO_PACK_INVALID for invalid 
           parameter (or) OSSO_PACK_ERROR for other kind of errors
*/

gint osso_email_xml_unpack_mimeinfo(xmlNodePtr node, xmlDocPtr doc, gchar * nodeType, void *data)
{
	MimeInfo *childInfo = NULL, *subInfo = NULL, *tmimeInfo = NULL;
	MimeInfo *parentInfo = NULL;
	gint idx = 0;
	MimeInfo *mimeInfo = (MimeInfo *) data;

	if (doc == NULL || node == NULL || mimeInfo == NULL) {
		osso_log(LOG_INFO, "Invalid input params in %s", __FUNCTION__);
		return OSSO_PACK_INVALID;
	}

	node = node->xmlChildrenNode;
	while (node != NULL) {
		if (node->type != XML_ELEMENT_NODE) {
			node = node->next;
			continue;
		}
		idx = 0;
		if (osso_email_xml_unpack_match_mimeinfo_attribute((gchar *) node->name, &idx) ==
		    FALSE) {
			node = node->next;
			continue;
		}
		switch (idx) {
		case 0:	/*Encoding */
			osso_email_xml_unpack_str(node, doc, NULL, &mimeInfo->encoding);
			break;
		case 1:	/*encoding_type */
			osso_email_xml_unpack_int(node, doc, NULL, &mimeInfo->encoding_type);
			break;
		case 2:	/*mime_type */
			osso_email_xml_unpack_int(node, doc, NULL, &mimeInfo->mime_type);
			break;
		case 3:	/*content_type */
			osso_email_xml_unpack_str(node, doc, NULL, &mimeInfo->content_type);
			break;
		case 4:	/*charset */
			osso_email_xml_unpack_str(node, doc, NULL, &mimeInfo->charset);
			break;
		case 5:	/*name */
			osso_email_xml_unpack_str(node, doc, NULL, &mimeInfo->name);
			break;
		case 6:	/*boundary */
			osso_email_xml_unpack_str(node, doc, NULL, &mimeInfo->boundary);
			break;
		case 7:	/*content_disposition */
			osso_email_xml_unpack_str(node, doc, NULL, &mimeInfo->content_disposition);
			break;
		case 8:	/*filename */
			osso_email_xml_unpack_str(node, doc, NULL, &mimeInfo->filename);
			break;
		case 9:	/*fpos */
			osso_email_xml_unpack_long(node, doc, NULL, &mimeInfo->fpos);
			break;
		case 10:	/*size */
			osso_email_xml_unpack_int(node, doc, NULL, &mimeInfo->size);
			break;
		case 11:	/*level */
			osso_email_xml_unpack_int(node, doc, NULL, &mimeInfo->level);
			break;
		case 12:	/*section_number */
			osso_email_xml_unpack_str(node, doc, NULL, &mimeInfo->section_number);
			break;
		case 13:	/*is downloaded */
			osso_email_xml_unpack_int(node, doc, NULL, &mimeInfo->is_downloaded);
			break;
		case 14:	/*is purged */
			osso_email_xml_unpack_int(node, doc, NULL, &mimeInfo->is_purged);
			break;
		case 15:	/*cert_state  sender */
			osso_email_xml_unpack_int(node, doc, NULL, &mimeInfo->cert_state_sender);
			break;
		case 16:	/*cert_state recipient */
			osso_email_xml_unpack_int(node, doc, NULL, &mimeInfo->cert_state_recip);
			break;
		case 17:	/*trust */
			osso_email_xml_unpack_bool(node, doc, NULL, &mimeInfo->trust);
			break;
		case 18:	/*Signature verify */
			osso_email_xml_unpack_bool(node, doc, NULL, &mimeInfo->signature_verify);
			break;
		case 19:	/*saving */
			osso_email_xml_unpack_bool(node, doc, NULL, &mimeInfo->saving);
			break;
		case 20:	/*cert path */
			osso_email_xml_unpack_str(node, doc, NULL, &mimeInfo->cert_path);
			break;
		case 21:	/*smime type */
			osso_email_xml_unpack_int(node, doc, NULL, &mimeInfo->smime_type);
			break;
		case 22:	/*GSList of inline images */
			osso_email_xml_unpack_inlineimageinfo_list(node, doc, NULL,
								   &mimeInfo->inline_images);
			break;
		case 23:
			osso_email_xml_unpack_int_list(node, doc, NULL, &mimeInfo->file_pos_list);
			break;
		case 24:
			osso_email_xml_unpack_int(node, doc, NULL, &mimeInfo->sender_cert_id);
			break;

		case 25:
			osso_email_xml_unpack_int(node, doc, NULL, &mimeInfo->recip_cert_id);
			break;

		case 26:	/* This is bad Just need boundary info of parent */
			if ((nodeType != NULL)
			    && (g_ascii_strcasecmp(nodeType, XML_MIME_INFO) == 0)) {
				parentInfo = g_new0(MimeInfo, 1);
				if (parentInfo == NULL) {
					osso_log(LOG_ERR, "Memory alloc fail \n");
					return OSSO_PACK_ALLOC_FAIL;
				}
				osso_email_xml_unpack_str(node, doc, NULL, &parentInfo->boundary);
				mimeInfo->parent = parentInfo;
			}
			break;
		case 27:	/*children */
			childInfo = g_new0(MimeInfo, 1);
			if (childInfo == NULL) {
				return OSSO_PACK_ALLOC_FAIL;
			}
			childInfo->parent = mimeInfo;
			osso_email_xml_unpack_mimeinfo(node, doc, XML_CHILD, childInfo);
			if (childInfo->sub != NULL) {
				childInfo->sub->parent = mimeInfo;
			}
			tmimeInfo = mimeInfo->children;
			if (tmimeInfo == NULL) {
				mimeInfo->children = childInfo;
			} else {
				while (tmimeInfo->next) {
					tmimeInfo = tmimeInfo->next;
				}
				tmimeInfo->next = childInfo;
			}
			break;
		case 28:	/*sub */
			subInfo = g_new0(MimeInfo, 1);
			if (subInfo == NULL) {
				osso_log(LOG_ERR, "Memory alloc fail");
				return OSSO_PACK_ALLOC_FAIL;
			}
			osso_email_xml_unpack_mimeinfo(node, doc, XML_SUB, subInfo);
			mimeInfo->sub = subInfo;
			subInfo->main = mimeInfo;
			subInfo->parent = mimeInfo->parent;
			break;
		default:
			osso_log(LOG_INFO, "Invalid member");
			break;
		}
		node = node->next;
	}
	return OSSO_PACK_OK;
}

/**
  This function is used to unpack the 'MsgInfo struct  data'
  stored in a xml 'doc' with identifier 'name'

   @param  node   Xml Node pointer of the packed data
   @param  doc    Xml Doc  pointer of the packed data
   @param  name   Tag Identifier of the MsgInfo struct data
                  in the xml document
   @param  data   MsgInfo struct data will be stored in this pointer
   @return OSSO_PACK_OK on success (or) OSSO_PACK_INVALID for invalid 
           parameter (or) OSSO_PACK_ERROR for other kind of errors
*/
gint osso_email_xml_unpack_msginfo(xmlNodePtr node, xmlDocPtr doc, gchar * nodeType, void *data)
{
	MsgInfo *msgInfo = NULL;
	GSList *attach_list = NULL;
	GSList *image_list = NULL;
	GSList *recip_certs_uid = NULL;
	GSList *recip_algos = NULL;
	gchar *passwd = NULL;

	if (doc == NULL || node == NULL || data == NULL) {
		osso_log(LOG_INFO, "Invalid input params in %s", __FUNCTION__);
		return OSSO_PACK_INVALID;
	}
	msgInfo = (MsgInfo *) data;

	for (node = node->xmlChildrenNode; node != NULL; node = node->next) {
		if (node->type == XML_ELEMENT_NODE) {
			if (g_ascii_strcasecmp(node->name, XML_MSG_HEADER) == 0) {
				msgInfo->header = g_new0(MsgHeader, 1);
				if (msgInfo->header == NULL) {
					osso_log(LOG_ERR, "Memory alloc fail");
					return OSSO_PACK_ALLOC_FAIL;
				}
				osso_email_xml_unpack_msgheader(node, doc, XML_MSG_HEADER,
								msgInfo->header);
			} else if (g_ascii_strcasecmp(node->name, XML_INLINEIMAGE_INFO_LST) == 0) {
				osso_email_xml_unpack_inlineimageinfo_list(node, doc,
									   XML_INLINEIMAGE_INFO_LST,
									   &image_list);
				msgInfo->image_list = image_list;
			} else if (g_ascii_strcasecmp(node->name, XML_SETTINGS) == 0) {
				msgInfo->settings = g_new0(EmailSpecificSettings, 1);
				if (msgInfo->settings == NULL) {
					osso_log(LOG_ERR, "Memory alloc fail \n");
					return OSSO_PACK_ALLOC_FAIL;
				}
				osso_email_xml_unpack_settings(node, doc, XML_SETTINGS,
							       msgInfo->settings);
			} else if (g_ascii_strcasecmp(node->name, XML_ATTACHINFO_LST) == 0) {
				osso_email_xml_unpack_attachinfo_list(node, doc,
								      XML_ATTACHINFO_LST,
								      &attach_list);
				msgInfo->attach_list = attach_list;
			} else if (g_ascii_strcasecmp(node->name, XML_CERTS_UID_LST) == 0) {
				osso_email_xml_unpack_int_list(node, doc,
							       XML_CERTS_UID_LST, &recip_certs_uid);
				msgInfo->recip_certs_uid = recip_certs_uid;
			} else if (g_ascii_strcasecmp(node->name, XML_ALGOS_LST) == 0) {
				osso_email_xml_unpack_string_list(node, doc,
								  XML_ALGOS_LST, &recip_algos);
				msgInfo->recip_algos = recip_algos;

			} else if (g_ascii_strcasecmp(node->name, PASSWORD_KEY) == 0) {
				osso_email_xml_unpack_str(node, doc, PASSWORD_KEY, &passwd);
				msgInfo->password = passwd;
			}



		}
	}
	return OSSO_PACK_OK;
}

/**
  This function is used to get the index matching element in MsgHeader struct  

   @param  nodename  node element name  
   @param  index  index of the matching element 

   @return OSSO_PACK_OK on success or FALSE failure 
*/

static gboolean osso_email_xml_unpack_match_msgheader_attribute(gchar * nodename, gint * idx)
{
	gint index = 0;
	if ((nodename == NULL) || (idx == NULL)) {
		return FALSE;
	}
	for (index = 0; MsgHeaderDBusData[index].name != NULL; index++) {
		if (g_ascii_strcasecmp(nodename, MsgHeaderDBusData[index].name) == 0) {
			*idx = index;
			return TRUE;
		}
	}
	return FALSE;
}

/**
  This function is used to unpack the 'MsgHeader struct  data'
  stored in a xml 'doc' with identifier 'name'

   @param  node   Xml Node pointer of the packed data
   @param  doc    Xml Doc  pointer of the packed data
   @param  name   Tag Identifier of the MsgHeader struct data
                  in the xml document
   @param  data   MsgHeader struct data will be stored in this pointer
   @return OSSO_PACK_OK on success (or) OSSO_PACK_INVALID for invalid 
           parameter (or) OSSO_PACK_ERROR for other kind of errors
*/

osso_pack_return_t
osso_email_xml_unpack_msgheader(xmlNodePtr node, xmlDocPtr doc, gchar * nodeType, void *data)
{
	MsgHeader *msgHeader = NULL;
	gint idx = 0;

	if (doc == NULL || node == NULL || data == NULL) {
		osso_log(LOG_INFO, "Invalid input params in %s", __FUNCTION__);
		return OSSO_PACK_INVALID;
	}
	msgHeader = (MsgHeader *) data;

	node = node->xmlChildrenNode;

	while (node != NULL) {
		if (node->type != XML_ELEMENT_NODE) {
			node = node->next;
			continue;
		}
		idx = 0;
		if (osso_email_xml_unpack_match_msgheader_attribute((gchar *) node->name, &idx) ==
		    FALSE) {
			node = node->next;
			continue;
		}
		switch (idx) {
		case 0:	/*Msg Num */
			osso_email_xml_unpack_int(node, doc, NULL, &msgHeader->msgnum);
			break;
		case 1:	/*size */
			osso_email_xml_unpack_int(node, doc, NULL, &msgHeader->size);
			break;
		case 2:	/*mtime */
			osso_email_xml_unpack_int(node, doc, NULL, &msgHeader->mtime);
			break;
		case 3:	/* date_t */
			osso_email_xml_unpack_int(node, doc, NULL, &msgHeader->date_t);
			break;
		case 4:	/* rx_date */
			osso_email_xml_unpack_int(node, doc, NULL, &msgHeader->rx_date);
			break;
		case 5:	/*flags */
			osso_email_xml_unpack_flags(node, doc, NULL, &msgHeader->flags);
			break;
		case 6:	/*from name */
			osso_email_xml_unpack_str(node, doc, NULL, &msgHeader->fromname);
			break;
		case 7:	/* date */
			osso_email_xml_unpack_str(node, doc, NULL, &msgHeader->date);
			break;
		case 8:	/* from */
			osso_email_xml_unpack_str(node, doc, NULL, &msgHeader->from);
			break;
		case 9:	/* to */
			osso_email_xml_unpack_str(node, doc, NULL, &msgHeader->to);
			break;
		case 10:	/* cc */
			osso_email_xml_unpack_str(node, doc, NULL, &msgHeader->cc);
			break;
		case 11:	/* bcc */
			osso_email_xml_unpack_str(node, doc, NULL, &msgHeader->bcc);
			break;
		case 12:	/*subject */
			osso_email_xml_unpack_str_base64(node, doc, NULL, &msgHeader->subject);
			break;
		case 13:	/*msgid */
			osso_email_xml_unpack_str(node, doc, NULL, &msgHeader->msgid);
			break;
		case 14:	/*replyto */
			osso_email_xml_unpack_str(node, doc, NULL, &msgHeader->replyto);
			break;
		case 15:	/*inreplyto */
			osso_email_xml_unpack_str(node, doc, NULL, &msgHeader->inreplyto);
			break;
		case 16:	/*username */
			osso_email_xml_unpack_str(node, doc, NULL, &msgHeader->username);
			break;
		case 17:	/*password */
			osso_email_xml_unpack_str(node, doc, NULL, &msgHeader->password);
			break;
		case 18:	/*account_name */
			osso_email_xml_unpack_str(node, doc, NULL, &msgHeader->account_name);
			break;
		case 19:	/*msgtag */
			osso_email_xml_unpack_tag(node, doc, NULL, &msgHeader->msgtag);
			break;
		case 20:	/*recv_priority */
			osso_email_xml_unpack_int(node, doc, NULL, &msgHeader->recv_priority);
			break;
		case 21:	/*read_receipt_request */
			osso_email_xml_unpack_bool(node, doc, NULL,
						   &msgHeader->read_receipt_request);
			break;
		case 22:	/*copy_to_own_address */
			osso_email_xml_unpack_bool(node, doc, NULL,
						   &msgHeader->copy_to_own_address);
			break;
		case 23:	/*smime_type */
			osso_email_xml_unpack_int(node, doc, NULL, &msgHeader->smime_type);
			break;
		case 24:	/* attach_tag */
			osso_email_xml_unpack_int(node, doc, NULL, &msgHeader->attach_tag);
			break;
		default:
			osso_log(LOG_INFO, " Invalid string in unpacking  msgeheader info\n");
			break;
		}
		node = node->next;
	}
	return OSSO_PACK_OK;
}

/**
  This function is used to unpack the 'Folder struct  data'
  stored in a xml 'doc' with identifier 'name'

   @param  node   Xml Node pointer of the packed data
   @param  doc    Xml Doc  pointer of the packed data
   @param  name   Tag Identifier of the Folder struct data
                  in the xml document
   @param  data   Folder struct data will be stored in this pointer
   @return OSSO_PACK_OK on success (or) OSSO_PACK_INVALID for invalid 
           parameter (or) OSSO_PACK_ERROR for other kind of errors
*/

osso_pack_return_t
osso_email_xml_unpack_folder(xmlNodePtr node, xmlDocPtr doc, gchar * nodeType, void *data)
{
	Folder *folder = NULL;
	gint idx = 0;
	if (doc == NULL || node == NULL || data == NULL) {
		osso_log(LOG_INFO, "Invalid input params in %s", __FUNCTION__);
		return OSSO_PACK_INVALID;
	}
	folder = (Folder *) data;
	node = node->xmlChildrenNode;
	while (node != NULL) {
		if (node->type != XML_ELEMENT_NODE) {
			node = node->next;
			continue;
		}
		tmpFolder = *folder;
		idx = 0;
		if (g_ascii_strcasecmp((gchar *) node->name, FolderDBusData[idx++].name) == 0) {
			/*name */
			osso_email_xml_unpack_str(node, doc, NULL, &folder->name);
		} else if (g_ascii_strcasecmp((gchar *) node->name, FolderDBusData[idx++].name) ==
			   0) {
			/*path */
			osso_email_xml_unpack_str(node, doc, NULL, &folder->path);
		} else if (g_ascii_strcasecmp((gchar *) node->name, FolderDBusData[idx++].name) ==
			   0) {
			/*unread */
			osso_email_xml_unpack_int(node, doc, NULL, &folder->unread);
		} else if (g_ascii_strcasecmp((gchar *) node->name, FolderDBusData[idx++].name) ==
			   0) {
			/*new_msgs */
			osso_email_xml_unpack_int(node, doc, NULL, &folder->new_msgs);
		} else if (g_ascii_strcasecmp((gchar *) node->name, FolderDBusData[idx++].name) ==
			   0) {
			/*total */
			osso_email_xml_unpack_int(node, doc, NULL, &folder->total);
		} else if (g_ascii_strcasecmp((gchar *) node->name, FolderDBusData[idx++].name) ==
			   0) {
			/*type */
			osso_email_xml_unpack_int(node, doc, NULL, &folder->type);
		}
		node = node->next;
	}
	return OSSO_PACK_OK;
}

/**
  This function is used to unpack the 'Folder AttachInfo struct  data'
  stored in a xml 'doc' with identifier 'name'

   @param  node   Xml Node pointer of the packed data
   @param  doc    Xml Doc  pointer of the packed data
   @param  name   Tag Identifier of the Folder AttachInfo  struct data
                  in the xml document
   @param  data   Folder AttachInfo  struct data will be stored in this pointer
   @return OSSO_PACK_OK on success (or) OSSO_PACK_INVALID for invalid 
           parameter (or) OSSO_PACK_ERROR for other kind of errors
*/


osso_pack_return_t
osso_email_xml_unpack_folder_attachinfo(xmlNodePtr node, xmlDocPtr doc,
					gchar * nodeType, void *data)
{
	FolderAttachInfo *folder_attach_info = NULL;
	gint idx = 0;
	if (doc == NULL || node == NULL || data == NULL) {
		osso_log(LOG_INFO, "Invalid input params in %s", __FUNCTION__);
		return OSSO_PACK_INVALID;
	}
	folder_attach_info = (FolderAttachInfo *) data;
	node = node->xmlChildrenNode;
	while (node != NULL) {
		if (node->type != XML_ELEMENT_NODE) {
			node = node->next;
			continue;
		}
		tmpFolderAttachInfo = *folder_attach_info;
		idx = 0;
		if (g_ascii_strcasecmp((gchar *) node->name, FolderAttachInfoDBusData[idx++].name)
		    == 0) {
			/*msgid */
			osso_email_xml_unpack_str(node, doc, NULL, &folder_attach_info->msgid);
		} else
		    if (g_ascii_strcasecmp
			((gchar *) node->name, FolderAttachInfoDBusData[idx++].name) == 0) {
			/*section_number */
			osso_email_xml_unpack_str(node, doc, NULL,
						  &folder_attach_info->section_number);
		}
		node = node->next;
	}
	return OSSO_PACK_OK;
}

/**
  This function is used to unpack the 'Msg Flags struct  data' stored 
  in a xml 'doc' with identifier 'name'

   @param  node   Xml Node pointer of the packed data
   @param  doc    Xml Doc  pointer of the packed data
   @param  name   Tag Identifier of the Msg Flags struct data
                  in the xml document
   @param  data   MsgFlags  struct data will be stored in this pointer
   @return OSSO_PACK_OK on success (or) OSSO_PACK_INVALID for invalid 
           parameter (or) OSSO_PACK_ERROR for other kind of errors
*/

osso_pack_return_t
osso_email_xml_unpack_flags(xmlNodePtr node, xmlDocPtr doc, gchar * nodeType, void *data)
{
	MsgFlags *flags = NULL;
	gint idx = 0;

	if (doc == NULL || node == NULL || data == NULL) {
		osso_log(LOG_INFO, "Invalid input params in %s", __FUNCTION__);
		return OSSO_PACK_INVALID;
	}
	flags = (MsgFlags *) data;

	node = node->xmlChildrenNode;

	while (node != NULL) {
		if (node->type != XML_ELEMENT_NODE) {
			node = node->next;
			continue;
		}

		tmpMsgFlags = *flags;
		idx = 0;
		if (g_ascii_strcasecmp((gchar *) node->name, MsgFlagsDBusData[idx++].name) == 0) {
			/*perm_flags */
			osso_email_xml_unpack_int(node, doc, NULL, &flags->perm_flags);
		} else if (g_ascii_strcasecmp((gchar *) node->name, MsgFlagsDBusData[idx++].name) ==
			   0) {
			/*tmp_flags */
			osso_email_xml_unpack_int(node, doc, NULL, &flags->tmp_flags);
		}
		node = node->next;
	}
	return OSSO_PACK_OK;
}

/**
  This function is used to unpack the 'MsgTag struct  data' stored 
  in a xml 'doc' with identifier 'name'

   @param  node   Xml Node pointer of the packed data
   @param  doc    Xml Doc  pointer of the packed data
   @param  name   Tag Identifier of the Msg Tag struct data
                  in the xml document
   @param  data   Tag  struct data will be stored in this pointer
   @return OSSO_PACK_OK on success (or) OSSO_PACK_INVALID for invalid 
           parameter (or) OSSO_PACK_ERROR for other kind of errors
*/

osso_pack_return_t
osso_email_xml_unpack_tag(xmlNodePtr node, xmlDocPtr doc, gchar * nodeType, void *data)
{
	Tag *tag = NULL;
	gint idx = 0;

	if (doc == NULL || node == NULL || data == NULL) {
		osso_log(LOG_INFO, "Invalid input params in %s", __FUNCTION__);
		return OSSO_PACK_INVALID;
	}
	tag = (Tag *) data;

	node = node->xmlChildrenNode;

	while (node != NULL) {
		if (node->type != XML_ELEMENT_NODE) {
			node = node->next;
			continue;
		}

		idx = 0;
		if (g_ascii_strcasecmp((gchar *) node->name, MsgTagDBusData[idx++].name) == 0) {
			/*protocol */
			osso_email_xml_unpack_int(node, doc, NULL, &tag->protocol);
		} else if (g_ascii_strcasecmp((gchar *) node->name, MsgTagDBusData[idx++].name) ==
			   0) {
			/*recvtype */
			osso_email_xml_unpack_int(node, doc, NULL, &tag->recvtype);
		} else if (g_ascii_strcasecmp((gchar *) node->name, MsgTagDBusData[idx++].name) ==
			   0) {
			/*foldertype */
			osso_email_xml_unpack_int(node, doc, NULL, &tag->foldertype);
		} else if (g_ascii_strcasecmp((gchar *) node->name, MsgTagDBusData[idx++].name) ==
			   0) {
			/*fullydownloaded */
			osso_email_xml_unpack_bool(node, doc, NULL, &tag->fullydownloaded);
		} else if (g_ascii_strcasecmp((gchar *) node->name, MsgTagDBusData[idx++].name) ==
			   0) {
			/*bodydownloaded */
			osso_email_xml_unpack_bool(node, doc, NULL, &tag->bodydownloaded);
		} else if (g_ascii_strcasecmp((gchar *) node->name, MsgTagDBusData[idx++].name) ==
			   0) {
			/*remote_copy */
			osso_email_xml_unpack_bool(node, doc, NULL, &tag->remote_copy);
		} else if (g_ascii_strcasecmp((gchar *) node->name, MsgTagDBusData[idx++].name) ==
			   0) {
			/*timestamp */
			osso_email_xml_unpack_bool(node, doc, NULL, &tag->timestamp);
		}
		node = node->next;
	}
	return OSSO_PACK_OK;
}

/**
  This function is used to unpack the 'EmailSpecificSettings struct  data' 
  stored in a xml 'doc' with identifier 'name'

   @param  node   Xml Node pointer of the packed data
   @param  doc    Xml Doc  pointer of the packed data
   @param  name   Tag Identifier of the EmailSpecificSettings struct data
                  in the xml document
   @param  data   EmailSpecificSettings  struct data will be stored 
                  in this pointer 
   @return OSSO_PACK_OK on success (or) OSSO_PACK_INVALID for invalid 
           parameter (or) OSSO_PACK_ERROR for other kind of errors
*/

osso_pack_return_t
osso_email_xml_unpack_settings(xmlNodePtr node, xmlDocPtr doc, gchar * nodeType, void *data)
{
	EmailSpecificSettings *settings = NULL;
	gint idx = 0;

	if (doc == NULL || node == NULL || data == NULL) {
		osso_log(LOG_INFO, "Invalid input params in %s", __FUNCTION__);
		return OSSO_PACK_INVALID;
	}
	settings = (EmailSpecificSettings *) data;

	node = node->xmlChildrenNode;

	while (node != NULL) {
		if (node->type != XML_ELEMENT_NODE) {
			node = node->next;
			continue;
		}

		idx = 0;
		if (g_ascii_strcasecmp((gchar *) node->name, EMailSettingsDBusData[idx++].name) ==
		    0) {
			/*priority */
			osso_email_xml_unpack_int(node, doc, NULL, &settings->priority);
		} else
		    if (g_ascii_strcasecmp((gchar *) node->name, EMailSettingsDBusData[idx++].name)
			== 0) {
			/*type */
			osso_email_xml_unpack_int(node, doc, NULL, &settings->type);
		} else
		    if (g_ascii_strcasecmp((gchar *) node->name, EMailSettingsDBusData[idx++].name)
			== 0) {
			/*account */
			osso_email_xml_unpack_str(node, doc, NULL, &settings->account);
		} else
		    if (g_ascii_strcasecmp((gchar *) node->name, EMailSettingsDBusData[idx++].name)
			== 0) {
			/*read_receipt_request */
			osso_email_xml_unpack_bool(node, doc, NULL,
						   &settings->read_receipt_request);
		} else
		    if (g_ascii_strcasecmp((gchar *) node->name, EMailSettingsDBusData[idx++].name)
			== 0) {
			/*options */
			osso_email_xml_unpack_int(node, doc, NULL, &settings->options);
		} else
		    if (g_ascii_strcasecmp((gchar *) node->name, EMailSettingsDBusData[idx++].name)
			== 0) {
			/*Copy To Own Address */
			osso_email_xml_unpack_bool(node, doc, NULL, &settings->copy_to_own_address);
		} else
		    if (g_ascii_strcasecmp((gchar *) node->name, EMailSettingsDBusData[idx++].name)
			== 0) {
			/* SMime Type */
			osso_email_xml_unpack_int(node, doc, NULL, &settings->smime_type);
		}
		node = node->next;
	}
	return OSSO_PACK_OK;
}

/**
  This function is used to unpack the 'AttachInfo struct  data'
  stored in a xml 'doc' with identifier 'name'

   @param  node   Xml Node pointer of the packed data
   @param  doc    Xml Doc  pointer of the packed data
   @param  name   Tag Identifier of the AttachInfo struct data
                  in the xml document
   @param  data   AttachInfo  struct data will be stored
                  in this pointer
   @return OSSO_PACK_OK on success (or) OSSO_PACK_INVALID for invalid 
           parameter (or) OSSO_PACK_ERROR for other kind of errors
*/

osso_pack_return_t
osso_email_xml_unpack_attachinfo(xmlNodePtr node, xmlDocPtr doc, gchar * nodeType, void *data)
{
	AttachInfo *attinfo = NULL;
	gint idx = 0;

	if (doc == NULL || node == NULL || data == NULL) {
		osso_log(LOG_INFO, "Invalid input params in %s", __FUNCTION__);
		return OSSO_PACK_INVALID;
	}
	attinfo = (AttachInfo *) data;

	node = node->xmlChildrenNode;

	while (node != NULL) {
		if (node->type != XML_ELEMENT_NODE) {
			node = node->next;
			continue;
		}

		idx = 0;
		if (g_ascii_strcasecmp((gchar *) node->name, AttachInfoDBusData[idx++].name) == 0) {
			/*type */
			osso_email_xml_unpack_int(node, doc, NULL, &attinfo->type);
		} else if (g_ascii_strcasecmp((gchar *) node->name, AttachInfoDBusData[idx++].name)
			   == 0) {
			/*Filetpath */
			osso_email_xml_unpack_str(node, doc, NULL, &attinfo->filepath);
		} else if (g_ascii_strcasecmp((gchar *) node->name, AttachInfoDBusData[idx++].name)
			   == 0) {
			/* MsgId */
			osso_email_xml_unpack_str(node, doc, NULL, &attinfo->msgid);
		} else if (g_ascii_strcasecmp((gchar *) node->name, AttachInfoDBusData[idx++].name)
			   == 0) {
			/*attachmentname */
			osso_email_xml_unpack_str(node, doc, NULL, &attinfo->attachmentname);
		} else if (g_ascii_strcasecmp((gchar *) node->name, AttachInfoDBusData[idx++].name)
			   == 0) {
			/* email attachment path */
			osso_email_xml_unpack_str(node, doc, NULL, &attinfo->email_path);
		}
		node = node->next;
	}
	return OSSO_PACK_OK;
}



/**
  This function is used to unpack the 'InlineImageInfo struct  data'
  stored in a xml 'doc' with identifier 'name'

   @param  node   Xml Node pointer of the packed data
   @param  doc    Xml Doc  pointer of the packed data
   @param  name   Tag Identifier of the InlineImageInfo struct data
                  in the xml document
   @param  data   InlineImageInfo  struct data will be stored
                  in this pointer
   @return OSSO_PACK_OK on success (or) OSSO_PACK_INVALID for invalid 
           parameter (or) OSSO_PACK_ERROR for other kind of errors
*/

osso_pack_return_t
osso_email_xml_unpack_inlineimageinfo(xmlNodePtr node, xmlDocPtr doc, gchar * nodeType, void *data)
{
	InlineImageInfo *imageinfo = NULL;
	gint idx = 0;

	if (doc == NULL || node == NULL || data == NULL) {
		osso_log(LOG_INFO, "Invalid input params in %s", __FUNCTION__);
		return OSSO_PACK_INVALID;
	}
	imageinfo = (InlineImageInfo *) data;

	node = node->xmlChildrenNode;

	while (node != NULL) {
		if (node->type != XML_ELEMENT_NODE) {
			node = node->next;
			continue;
		}

		idx = 0;
		if (g_ascii_strcasecmp((gchar *) node->name,
				       InlineImageInfoDBusData[idx++].name) == 0) {
			/*uri_path */
			osso_email_xml_unpack_str(node, doc, NULL, &imageinfo->uri_path);
		} else if (g_ascii_strcasecmp((gchar *) node->name,
					      InlineImageInfoDBusData[idx++].name) == 0) {
			/*content_id */
			osso_email_xml_unpack_str(node, doc, NULL, &imageinfo->content_id);
		} else if (g_ascii_strcasecmp((gchar *) node->name,
					      InlineImageInfoDBusData[idx++].name) == 0) {
			/*image_name */
			osso_email_xml_unpack_str(node, doc, NULL, &imageinfo->image_name);
		} else if (g_ascii_strcasecmp((gchar *) node->name,
					      InlineImageInfoDBusData[idx++].name) == 0) {
			/*image_path */
			osso_email_xml_unpack_str(node, doc, NULL, &imageinfo->image_path);
		}
		node = node->next;
	}
	return OSSO_PACK_OK;
}

/**
  This function is used to unpack the 'GsList of AttachInfo struct  data'
  stored in a xml 'doc' with identifier 'name'

   @param  node   Xml Node pointer of the packed data
   @param  doc    Xml Doc  pointer of the packed data
   @param  name   Tag Identifier of the GSList of AttachInfo struct data
                  in the xml document
   @param  data   GSList of  AttachInfo  struct data will be stored
                  in this pointer
   @return OSSO_PACK_OK on success (or) OSSO_PACK_INVALID for invalid 
           parameter (or) OSSO_PACK_ERROR for other kind of errors
*/

osso_pack_return_t
osso_email_xml_unpack_attachinfo_list(xmlNodePtr node, xmlDocPtr doc,
				      gchar * nodeType, GSList ** data)
{
	GSList *attach_list = NULL;
	AttachInfo *attachinfo = NULL;

	if (doc == NULL || node == NULL || data == NULL) {
		osso_log(LOG_INFO, "Invalid input params in %s", __FUNCTION__);
		return OSSO_PACK_INVALID;
	}

	for (node = node->xmlChildrenNode; node != NULL; node = node->next) {
		if (node->type == XML_ELEMENT_NODE) {
			if (g_ascii_strcasecmp((gchar *) node->name, XML_ATTACHINFO) == 0) {
				attachinfo = g_new0(AttachInfo, 1);
				if (attachinfo == NULL) {
					osso_log(LOG_ERR, "Memory alloc fail \n");
					return OSSO_PACK_ALLOC_FAIL;
				}
				osso_email_xml_unpack_attachinfo(node, doc, XML_ATTACHINFO,
								 attachinfo);
				attach_list = g_slist_append(attach_list, attachinfo);
			}
		}
	}
	*data = attach_list;
	return OSSO_PACK_OK;
}


/**
  This function is used to unpack the 'GsList of Inline Image Info struct 
  data' stored in a xml 'doc' with identifier 'name'

   @param  node   Xml Node pointer of the packed data
   @param  doc    Xml Doc  pointer of the packed data
   @param  name   Tag Identifier of the GSList of Inline Image Info struct 
                  data in the xml document
   @param  data   GSList of  Inline Image Info struct data will be stored
                  in this pointer
   @return OSSO_PACK_OK on success (or) OSSO_PACK_INVALID for invalid 
           parameter (or) OSSO_PACK_ERROR for other kind of errors
*/

osso_pack_return_t
osso_email_xml_unpack_inlineimageinfo_list(xmlNodePtr node, xmlDocPtr doc,
					   gchar * nodeType, GSList ** data)
{
	GSList *image_list = NULL;
	InlineImageInfo *inlineimageinfo = NULL;

	if (doc == NULL || node == NULL || data == NULL) {
		osso_log(LOG_INFO, "Invalid input params in %s", __FUNCTION__);
		return OSSO_PACK_INVALID;
	}

	for (node = node->xmlChildrenNode; node != NULL; node = node->next) {
		if (node->type == XML_ELEMENT_NODE) {
			if (g_ascii_strcasecmp((gchar *) node->name, XML_INLINEIMAGE_INFO) == 0) {
				inlineimageinfo = g_new0(InlineImageInfo, 1);
				if (inlineimageinfo == NULL) {
					osso_log(LOG_ERR, "Memory alloc fail \n");
					return OSSO_PACK_ALLOC_FAIL;
				}
				osso_email_xml_unpack_inlineimageinfo(node, doc,
								      XML_INLINEIMAGE_INFO,
								      inlineimageinfo);
				image_list = g_slist_append(image_list, inlineimageinfo);
			}
		}
	}
	*data = image_list;
	return OSSO_PACK_OK;
}


osso_pack_return_t
osso_email_xml_unpack_int_list(xmlNodePtr node, xmlDocPtr doc, gchar * nodeType, GSList ** data)
{
	GSList *f_pos_list = NULL;
	gint *fpos = NULL;

	if (doc == NULL || node == NULL || data == NULL) {
		osso_log(LOG_INFO, "Invalid input params in %s", __FUNCTION__);
		return OSSO_PACK_INVALID;
	}

	for (node = node->xmlChildrenNode; node != NULL; node = node->next) {
		if (node->type == XML_ELEMENT_NODE) {
			if (g_ascii_strcasecmp((gchar *) node->name, XML_INT) == 0) {
				osso_email_xml_unpack_int(node, doc, XML_INT, &fpos);
				f_pos_list = g_slist_append(f_pos_list, fpos);
			}
		}
	}
	*data = f_pos_list;
	return OSSO_PACK_OK;
}

/**
  This function is used to unpack the 'GsList of Folder struct  data'
  stored in a xml 'doc' with identifier 'name'

   @param  node   Xml Node pointer of the packed data
   @param  doc    Xml Doc  pointer of the packed data
   @param  name   Tag Identifier of the GSList of Folder struct data
                  in the xml document
   @param  data   GSList of  Folder  struct data will be stored
                  in this pointer
   @return OSSO_PACK_OK on success (or) OSSO_PACK_INVALID for invalid 
           parameter (or) OSSO_PACK_ERROR for other kind of errors
*/

osso_pack_return_t
osso_email_xml_unpack_folder_list(xmlNodePtr node, xmlDocPtr doc, gchar * nodeType, GSList ** data)
{
	GSList *folder_list = NULL;
	Folder *folder = NULL;
	if (doc == NULL || node == NULL || data == NULL) {
		osso_log(LOG_INFO, "Invalid input params in %s", __FUNCTION__);
		return OSSO_PACK_INVALID;
	}

	for (node = node->xmlChildrenNode; node != NULL; node = node->next) {
		if (node->type == XML_ELEMENT_NODE) {
			if (g_ascii_strcasecmp((gchar *) node->name, XML_FOLDER) == 0) {
				folder = g_new0(Folder, 1);
				if (folder == NULL) {
					osso_log(LOG_ERR, "Memory alloc fail \n");
					return OSSO_PACK_ALLOC_FAIL;
				}
				osso_email_xml_unpack_folder(node, doc, XML_FOLDER, folder);
				folder_list = g_slist_append(folder_list, folder);
			}
		}
	}
	*data = folder_list;
	return OSSO_PACK_OK;
}

/**
  This function is used to unpack the 'GsList of Folder Attachinfo struct data'
  stored in a xml 'doc' with identifier 'name'

   @param  node   Xml Node pointer of the packed data
   @param  doc    Xml Doc  pointer of the packed data
   @param  name   Tag Identifier of the GSList of Folder Attachinfo struct data
                  in the xml document
   @param  data   GSList of  Folder Attachinfo  struct data will be stored
                  in this pointer
   @return OSSO_PACK_OK on success (or) OSSO_PACK_INVALID for invalid 
           parameter (or) OSSO_PACK_ERROR for other kind of errors
*/

gint
osso_email_xml_unpack_folderattachinfo_list(xmlNodePtr node, xmlDocPtr doc,
					    gchar * nodeType, GSList ** data)
{
	GSList *attachinfo_list = NULL;
	FolderAttachInfo *attachinfo = NULL;

	if (doc == NULL || node == NULL || data == NULL) {
		osso_log(LOG_INFO, "Invalid input params in %s", __FUNCTION__);
		return OSSO_PACK_INVALID;
	}

	for (node = node->xmlChildrenNode; node != NULL; node = node->next) {
		if (node->type == XML_ELEMENT_NODE) {
			if (g_ascii_strcasecmp((gchar *) node->name, XML_FOLDER_ATTACHINFO) == 0) {
				attachinfo = g_new0(FolderAttachInfo, 1);
				if (attachinfo == NULL) {
					osso_log(LOG_ERR, "Memory alloc fail \n");
					return OSSO_PACK_ALLOC_FAIL;
				}
				osso_email_xml_unpack_folder_attachinfo(node, doc,
									XML_FOLDER_ATTACHINFO,
									attachinfo);
				attachinfo_list = g_slist_append(attachinfo_list, attachinfo);
			}
		}
	}
	*data = attachinfo_list;
	return OSSO_PACK_OK;
}

/**
  This function is used to unpack the 'GsList of MsgInfo struct  data'
  stored in a xml 'doc' with identifier 'name'

   @param  node   Xml Node pointer of the packed data
   @param  doc    Xml Doc  pointer of the packed data
   @param  name   Tag Identifier of the GSList of MsgInfo struct data
                  in the xml document
   @param  data   GSList of  MsgInfo  struct data will be stored
                  in this pointer
   @return OSSO_PACK_OK on success (or) OSSO_PACK_INVALID for invalid 
           parameter (or) OSSO_PACK_ERROR for other kind of errors
*/


gint
osso_email_xml_unpack_msginfo_list(xmlNodePtr node, xmlDocPtr doc, gchar * nodeType, GSList ** data)
{
	GSList *msginfo_list = NULL;
	MsgInfo *msgInfo = NULL;

	if (doc == NULL || node == NULL || data == NULL) {
		osso_log(LOG_INFO, "Invalid input params in %s", __FUNCTION__);
		return OSSO_PACK_INVALID;
	}

	for (node = node->xmlChildrenNode; node; node = node->next) {
		if (node->type == XML_ELEMENT_NODE) {
			if (g_ascii_strcasecmp((gchar *) node->name, XML_MSG_INFO) == 0) {
				msgInfo = g_new0(MsgInfo, 1);
				if (msgInfo == NULL) {
					osso_log(LOG_ERR, "Memory alloc fail \n");
					return OSSO_PACK_ALLOC_FAIL;
				}
				osso_email_xml_unpack_msginfo(node, doc, XML_MSG_INFO, msgInfo);
				msginfo_list = g_slist_append(msginfo_list, msgInfo);
			}
		}
	}
	*data = msginfo_list;
	return OSSO_PACK_OK;
}

/**
  This function is used to unpack the 'GsList of MimeInfo struct  data'
  stored in a xml 'doc' with identifier 'name'

   @param  node   Xml Node pointer of the packed data
   @param  doc    Xml Doc  pointer of the packed data
   @param  name   Tag Identifier of the GSList of MimeInfo struct data
                  in the xml document
   @param  data   GSList of  MimeInfo  struct data will be stored
                  in this pointer
   @return OSSO_PACK_OK on success (or) OSSO_PACK_INVALID for invalid 
           parameter (or) OSSO_PACK_ERROR for other kind of errors
*/


gint
osso_email_xml_unpack_mimeinfo_list(xmlNodePtr node, xmlDocPtr doc,
				    gchar * nodeType, GSList ** data)
{
	GSList *mimeinfo_list = NULL;
	MimeInfo *mimeInfo = NULL;

	if (doc == NULL || node == NULL || data == NULL) {
		osso_log(LOG_INFO, "Invalid input params in %s", __FUNCTION__);
		return OSSO_PACK_INVALID;
	}

	for (node = node->xmlChildrenNode; node; node = node->next) {
		if (node->type == XML_ELEMENT_NODE) {
			if (g_ascii_strcasecmp((gchar *) node->name, XML_MIME_INFO) == 0) {
				mimeInfo = g_new0(MimeInfo, 1);
				if (mimeInfo == NULL) {
					osso_log(LOG_ERR, "Memory alloc fail \n");
					return OSSO_PACK_ALLOC_FAIL;
				}
				osso_email_xml_unpack_mimeinfo(node, doc, XML_MIME_INFO, mimeInfo);
				mimeinfo_list = g_slist_append(mimeinfo_list, mimeInfo);
			}
		}
	}
	*data = mimeinfo_list;
	return OSSO_PACK_OK;
}

/**
  This function is used to unpack the 'GSList of MsgHeader struct  data'
  stored in a xml 'doc' with identifier 'name'

   @param  node   Xml Node pointer of the packed data
   @param  doc    Xml Doc  pointer of the packed data
   @param  name   Tag Identifier of the GSList of MsgHeader  struct data
                  in the xml document
   @param  data   GSList of MsgHeader  struct data will be stored
                  in this pointer
   @return OSSO_PACK_OK on success (or) OSSO_PACK_INVALID for invalid 
           parameter (or) OSSO_PACK_ERROR for other kind of errors
*/

osso_pack_return_t
osso_email_xml_unpack_msgheader_list(xmlNodePtr node, xmlDocPtr doc,
				     gchar * nodeType, GSList ** data)
{
	GSList *msgheader_list = NULL;
	MsgHeader *msgheader = NULL;

	if (doc == NULL || node == NULL || data == NULL) {
		osso_log(LOG_INFO, "Invalid input params in %s", __FUNCTION__);
		return OSSO_PACK_INVALID;
	}

	for (node = node->xmlChildrenNode; node != NULL; node = node->next) {
		if (node->type == XML_ELEMENT_NODE) {
			if (g_ascii_strcasecmp((gchar *) node->name, XML_MSG_HEADER) == 0) {
				msgheader = g_new0(MsgHeader, 1);
				if (msgheader == NULL) {
					osso_log(LOG_ERR, "Memory alloc fail \n");
					return OSSO_PACK_ALLOC_FAIL;
				}
				osso_email_xml_unpack_msgheader(node, doc, XML_MSG_HEADER,
								msgheader);
				msgheader_list = g_slist_append(msgheader_list, msgheader);
			}
		}
	}
	*data = msgheader_list;
	return OSSO_PACK_OK;
}

/**
  This function is used to unpack the 'Boolean Array  data'
  stored in a xml 'doc' with identifier 'name'

   @param  node   Xml Node pointer of the packed data
   @param  doc    Xml Doc  pointer of the packed data
   @param  name   Tag Identifier of the Boolean Array  struct data
                  in the xml document  
   @param  data   Boolean Array data will be stored
                  in this pointer
   @return OSSO_PACK_OK on success (or) OSSO_PACK_INVALID for invalid 
           parameter (or) OSSO_PACK_ERROR for other kind of errors
*/

osso_pack_return_t
osso_email_xml_unpack_bool_arr(xmlNodePtr node, xmlDocPtr doc, void *name, void *data)
{
	gboolean *ptr = NULL;

	if (node == NULL || doc == NULL || data == NULL) {
		osso_log(LOG_INFO, "Invalid input params in %s", __FUNCTION__);
		return OSSO_PACK_INVALID;
	}
	ptr = (gboolean *) data;

	for (node = node->xmlChildrenNode; node != NULL; node = node->next) {
		if (node->type == XML_ELEMENT_NODE) {
			if (g_ascii_strcasecmp((gchar *) node->name, XML_SCHED_DAY) == 0) {

				osso_email_xml_unpack_bool(node, doc, XML_SCHED_DAY, ptr);
				ptr++;

			}
		}
	}
	return OSSO_PACK_OK;
}

/**
  This function is used to pack the 'PrefsAccount struct data'
  received into a xml 'doc' with identifier 'name'.

   @param  node   Xml Node pointer to pack the data
   @param  doc    Xml Doc  pointer to pack the data
   @param  name   Tag Identifier of the structure data in the xml document
   @param  data   PrefsAccount struct data to be packed in xml format
   @return OSSO_PACK_OK on success (or) OSSO_PACK_INVALID for invalid 
           parameter (or) OSSO_PACK_ERROR for other kind of errors
*/

osso_pack_return_t
osso_email_xml_pack_prefsaccount(xmlNodePtr node, xmlDocPtr doc, void *nodeType, void *prefsAccount)
{
	xmlNodePtr child = NULL;
	PrefsAccount *account = NULL;

	if (doc == NULL || prefsAccount == NULL || nodeType == NULL) {
		osso_log(LOG_INFO, "Invalid input params in %s", __FUNCTION__);
		return OSSO_PACK_INVALID;
	}
	account = (PrefsAccount *) prefsAccount;
	child = xmlNewNode(NULL, BAD_CAST nodeType);
	xmlAddChild(node, child);
	node = child;

	osso_email_xml_pack_user_settings(node, doc, XML_USER_SETTINGS, &account->user_settings);

	osso_email_xml_pack_recv_settings(node, doc, XML_RECV_SETTINGS, &account->receive_settings);

	osso_email_xml_pack_send_settings(node, doc, XML_SEND_SETTINGS, &account->send_settings);

	osso_email_xml_pack_adv_settings(node, doc, XML_ADV_SETTINGS, &account->advanced_settings);

	return OSSO_PACK_OK;
}

/**
  This function is used to pack the 'GSList of PrefsAccount struct data'
  received into a xml 'doc' with identifier 'name'.

   @param  node   Xml Node pointer to pack the data
   @param  doc    Xml Doc  pointer to pack the data
   @param  name   Tag Identifier of the structure data in the xml document
   @param  data   GSList of  PrefsAccount struct data to be packed in xml
                  format
   @return OSSO_PACK_OK on success (or) OSSO_PACK_INVALID for invalid 
           parameter (or) OSSO_PACK_ERROR for other kind of errors
*/

osso_pack_return_t
osso_email_xml_pack_prefsaccount_list(xmlNodePtr node, xmlDocPtr doc, void *nodeType, void *data)
{
	xmlNodePtr root = NULL, child = NULL;
	GSList *cur = NULL;
	GSList *account_list = NULL;

	if ((data == NULL) || (nodeType == NULL)) {
		osso_log(LOG_INFO, "Invalid input params in %s", __FUNCTION__);
		return OSSO_PACK_INVALID;
	}
	account_list = (GSList *) data;

	NEW_NODE(node, root, child, nodeType);

	for (cur = account_list; cur != NULL; cur = cur->next) {
		if (cur->data == NULL) {
			continue;
		}
		osso_email_xml_pack_prefsaccount(node, doc, XML_PREFS_ACCOUNT, cur->data);
	}
	return OSSO_PACK_OK;
}

/**
   Unpack function for GSList of PrefsAccount struct

   @param  node   Xml Node pointer to pack string
   @param  doc    Xml Doc  pointer to pack string
   @param  name   Tag for  packed string
   @param  data   List of PrefsAccount struct to be unpacked from xml format
   @return OSSO_PACK_OK on success (or) OSSO_PACK_INVALID for invalid 
           parameter (or) OSSO_PACK_ERROR for other kind of errors
*/

osso_pack_return_t
osso_email_xml_unpack_prefsaccount_list(xmlNodePtr node, xmlDocPtr doc,
					gchar * nodeType, GSList ** data)
{
	GSList *account_list = NULL;
	PrefsAccount *prefs_account = NULL;

	if (doc == NULL || node == NULL || data == NULL) {
		osso_log(LOG_INFO, "Invalid input params in %s", __FUNCTION__);
		return OSSO_PACK_INVALID;
	}

	for (node = node->xmlChildrenNode; node != NULL; node = node->next) {
		if (node->type == XML_ELEMENT_NODE) {
			if (g_ascii_strcasecmp((gchar *) node->name, XML_PREFS_ACCOUNT) == 0) {
				prefs_account = g_new0(PrefsAccount, 1);
				if (prefs_account == NULL) {
					osso_log(LOG_ERR, "Memory alloc fail \n");
					return OSSO_PACK_ALLOC_FAIL;
				}
				osso_email_xml_unpack_prefsaccount(node, doc, XML_PREFS_ACCOUNT,
								   prefs_account);
				account_list = g_slist_append(account_list, prefs_account);
			}
		}
	}
	*data = account_list;
	return OSSO_PACK_OK;
}

/**
  This function is used to pack the 'UserSettings struct data' received 
  into a xml 'doc' with identifier 'name'.

   @param  node   Xml Node pointer to pack the data
   @param  doc    Xml Doc  pointer to pack the data
   @param  name   Tag Identifier of the structure data in the xml document
   @param  data   UserSettings struct data to be packed in xml 
                  format
   @return OSSO_PACK_OK on success (or) OSSO_PACK_INVALID for invalid 
           parameter (or) OSSO_PACK_ERROR for other kind of errors
*/

osso_pack_return_t
osso_email_xml_pack_user_settings(xmlNodePtr node, xmlDocPtr doc, void *nodeType,
				  void *user_settings)
{
	xmlNodePtr root = NULL, child = NULL;

	if ((doc == NULL) || (user_settings == NULL)
	    || (nodeType == NULL)) {
		osso_log(LOG_INFO, "Invalid input params in %s", __FUNCTION__);
		return OSSO_PACK_INVALID;
	}
	NEW_NODE(node, root, child, nodeType);
	memset(&tmpUserSettings, '\0', sizeof(tmpUserSettings));
	tmpUserSettings = *(UserSettings *) user_settings;
	osso_email_xml_pack_struct(node, doc, NULL, UserSettingsDBusData);
	return OSSO_PACK_OK;
}

/**
  This function is used to pack the 'ReceiveSettings struct data' received 
  into a xml 'doc' with identifier 'name'.

   @param  node   Xml Node pointer to pack the data
   @param  doc    Xml Doc  pointer to pack the data
   @param  name   Tag Identifier of the structure data in the xml document
   @param  data   ReceiveSettings struct data to be packed in xml
                  format
   @return OSSO_PACK_OK on success (or) OSSO_PACK_INVALID for invalid 
           parameter (or) OSSO_PACK_ERROR for other kind of errors
*/

osso_pack_return_t
osso_email_xml_pack_recv_settings(xmlNodePtr node, xmlDocPtr doc, void *nodeType,
				  void *recv_settings)
{
	xmlNodePtr root = NULL, child = NULL;

	if ((doc == NULL) || (recv_settings == NULL)
	    || (nodeType == NULL)) {
		osso_log(LOG_INFO, "Invalid input params in %s", __FUNCTION__);
		return OSSO_PACK_INVALID;
	}
	NEW_NODE(node, root, child, nodeType);
	memset(&tmpRecvSettings, '\0', sizeof(tmpRecvSettings));
	tmpRecvSettings = *(ReceiveSettings *) recv_settings;
	osso_email_xml_pack_struct(node, doc, NULL, RecvSettingsDBusData);
	return OSSO_PACK_OK;
}

/**
  This function is used to pack the 'SendSettings struct data' received
  into a xml 'doc' with identifier 'name'.

   @param  node   Xml Node pointer to pack the data
   @param  doc    Xml Doc  pointer to pack the data
   @param  name   Tag Identifier of the structure data in the xml document
   @param  data   SendSettings struct data to be packed in xml
                  format
   @return OSSO_PACK_OK on success (or) OSSO_PACK_INVALID for invalid 
           parameter (or) OSSO_PACK_ERROR for other kind of errors
*/

osso_pack_return_t
osso_email_xml_pack_send_settings(xmlNodePtr node, xmlDocPtr doc, void *nodeType,
				  void *send_settings)
{
	xmlNodePtr root = NULL, child = NULL;

	if ((doc == NULL) || (send_settings == NULL)
	    || (nodeType == NULL)) {
		osso_log(LOG_INFO, "Invalid input params in %s", __FUNCTION__);
		return OSSO_PACK_INVALID;
	}
	NEW_NODE(node, root, child, nodeType);
	memset(&tmpSendSettings, '\0', sizeof(tmpSendSettings));
	tmpSendSettings = *(SendSettings *) send_settings;
	osso_email_xml_pack_struct(node, doc, NULL, SendSettingsDBusData);
	return OSSO_PACK_OK;
}

/**
  This function is used to pack the 'AdvancedSettings struct data' received
  into a xml 'doc' with identifier 'name'.

   @param  node   Xml Node pointer to pack the data
   @param  doc    Xml Doc  pointer to pack the data
   @param  name   Tag Identifier of the structure data in the xml document
   @param  data   AdvancedSettings struct data to be packed in xml
                  format
   @return OSSO_PACK_OK on success (or) OSSO_PACK_INVALID for invalid 
           parameter (or) OSSO_PACK_ERROR for other kind of errors
*/

osso_pack_return_t
osso_email_xml_pack_adv_settings(xmlNodePtr node, xmlDocPtr doc, void *nodeType, void *adv_settings)
{
	xmlNodePtr root = NULL, child = NULL;

	if ((doc == NULL) || (adv_settings == NULL) || (nodeType == NULL)) {
		osso_log(LOG_INFO, "Invalid input params in %s", __FUNCTION__);
		return OSSO_PACK_INVALID;
	}
	NEW_NODE(node, root, child, nodeType);
	memset(&tmpAdvSettings, '\0', sizeof(tmpAdvSettings));
	tmpAdvSettings = *(AdvancedSettings *) adv_settings;
	osso_email_xml_pack_struct(node, doc, NULL, AdvSettingsDBusData);
	return OSSO_PACK_OK;
}


/**
  This function is used to pack the 'Boolean Array data' received
  into a xml 'doc' with identifier 'name'.

   @param  node   Xml Node pointer to pack the data
   @param  doc    Xml Doc  pointer to pack the data
   @param  name   Tag Identifier of the structure data in the xml document
   @param  data   Boolean Array struct data to be packed in xml
                  format
   @return OSSO_PACK_OK on success (or) OSSO_PACK_INVALID for invalid 
           parameter (or) OSSO_PACK_ERROR for other kind of errors
*/

osso_pack_return_t
osso_email_xml_pack_bool_arr(xmlNodePtr node, xmlDocPtr doc, void *nodeType, void *data)
{
	xmlNodePtr root = NULL, child = NULL;
	gboolean *ptr = data;
	gint i = 0;
	gint size = BOOL_ARR_SIZE;
	if ((doc == NULL) || (data == NULL) || (nodeType == NULL)) {
		osso_log(LOG_INFO, "Invalid input params in %s", __FUNCTION__);
		return OSSO_PACK_INVALID;
	}

	NEW_NODE(node, root, child, nodeType);
	for (i = 0; i < size; i++) {
		osso_email_xml_pack_bool(node, doc, XML_SCHED_DAY, &ptr[i]);
	}
	return OSSO_PACK_OK;
}

/**
  This function is used to unpack the 'PrefsAccount struct  data'
  stored in a xml 'doc' with identifier 'name'

   @param  node   Xml Node pointer of the packed data
   @param  doc    Xml Doc  pointer of the packed data
   @param  name   Tag Identifier of the  PrefsAccount  struct data
                  in the xml document
   @param  data   PrefsAccount struct data will be stored in this pointer
   @return OSSO_PACK_OK on success (or) OSSO_PACK_INVALID for invalid 
           parameter (or) OSSO_PACK_ERROR for other kind of errors
*/

osso_pack_return_t
osso_email_xml_unpack_prefsaccount(xmlNodePtr node, xmlDocPtr doc, gchar * nodeType, void *data)
{
	PrefsAccount *account = NULL;
	if (doc == NULL || node == NULL || data == NULL) {
		osso_log(LOG_INFO, "Invalid input params in %s", __FUNCTION__);
		return OSSO_PACK_INVALID;
	}
	account = (PrefsAccount *) data;

	for (node = node->xmlChildrenNode; node != NULL; node = node->next) {
		if (node->type == XML_ELEMENT_NODE) {
			if (g_ascii_strcasecmp((gchar *) node->name, XML_USER_SETTINGS) == 0) {
				osso_email_xml_unpack_user_settings(node, doc, XML_USER_SETTINGS,
								    &account->user_settings);
			} else if (g_ascii_strcasecmp((gchar *) node->name, XML_RECV_SETTINGS) == 0) {
				osso_email_xml_unpack_receive_settings(node, doc,
								       XML_RECV_SETTINGS,
								       &account->receive_settings);

			} else if (g_ascii_strcasecmp((gchar *) node->name, XML_SEND_SETTINGS) == 0) {
				osso_email_xml_unpack_send_settings(node, doc, XML_SEND_SETTINGS,
								    &account->send_settings);
			} else if (g_ascii_strcasecmp((gchar *) node->name, XML_ADV_SETTINGS) == 0) {
				osso_email_xml_unpack_adv_settings(node, doc, XML_ADV_SETTINGS,
								   &account->advanced_settings);

			}
		}
	}
	return OSSO_PACK_OK;
}

/**
  This function is used to unpack the 'USerSettings struct  data'
  stored in a xml 'doc' with identifier 'name'

   @param  node   Xml Node pointer of the packed data
   @param  doc    Xml Doc  pointer of the packed data
   @param  name   Tag Identifier of the  USerSettings  struct data
                  in the xml document
   @param  data   USerSettings struct data will be stored in this pointer
   @return OSSO_PACK_OK on success (or) OSSO_PACK_INVALID for invalid 
           parameter (or) OSSO_PACK_ERROR for other kind of errors
*/

osso_pack_return_t
osso_email_xml_unpack_user_settings(xmlNodePtr node, xmlDocPtr doc, gchar * nodeType, void *data)
{
	UserSettings *user_settings = NULL;

	gint idx = 0;

	if (doc == NULL || node == NULL || data == NULL) {
		osso_log(LOG_INFO, "Invalid input params in %s", __FUNCTION__);
		return OSSO_PACK_INVALID;
	}
	user_settings = (UserSettings *) data;

	node = node->xmlChildrenNode;

	while (node != NULL) {
		if (node->type != XML_ELEMENT_NODE) {
			node = node->next;
			continue;
		}
		idx = 0;
		if (g_ascii_strcasecmp((gchar *) node->name, UserSettingsDBusData[idx++].name) == 0) {
			/*account name */
			osso_email_xml_unpack_str(node, doc, NULL, &user_settings->account_name);
		} else
		    if (g_ascii_strcasecmp((gchar *) node->name, UserSettingsDBusData[idx++].name)
			== 0) {
			/*is_default */
			osso_email_xml_unpack_bool(node, doc, NULL, &user_settings->is_default);
		} else
		    if (g_ascii_strcasecmp((gchar *) node->name, UserSettingsDBusData[idx++].name)
			== 0) {
			/*userid */
			//osso_email_xml_unpack_str ( node, doc, NULL, &user_settings->userid );
			osso_email_xml_unpack_str_base64(node, doc, NULL, &user_settings->userid);
		} else
		    if (g_ascii_strcasecmp((gchar *) node->name, UserSettingsDBusData[idx++].name)
			== 0) {
			/*passwd */
			//osso_email_xml_unpack_str ( node, doc, NULL, &user_settings->passwd );
			osso_email_xml_unpack_str_base64(node, doc, NULL, &user_settings->passwd);
		} else
		    if (g_ascii_strcasecmp((gchar *) node->name, UserSettingsDBusData[idx++].name)
			== 0) {
			/*remember passwd */
			osso_email_xml_unpack_bool(node, doc, NULL,
						   &user_settings->remember_passwd);
		} else
		    if (g_ascii_strcasecmp((gchar *) node->name, UserSettingsDBusData[idx++].name)
			== 0) {
			/* email  */
			osso_email_xml_unpack_str(node, doc, NULL, &user_settings->email);
		} else
		    if (g_ascii_strcasecmp((gchar *) node->name, UserSettingsDBusData[idx++].name)
			== 0) {
			/* alias name */
			osso_email_xml_unpack_str(node, doc, NULL, &user_settings->alias_name);
		}

		node = node->next;
	}
	return OSSO_PACK_OK;
}

/**
  This function is used to unpack the 'ReceiveSettings struct  data'
  stored in a xml 'doc' with identifier 'name'

   @param  node   Xml Node pointer of the packed data
   @param  doc    Xml Doc  pointer of the packed data
   @param  name   Tag Identifier of the  ReceiveSettings  struct data
                  in the xml document
   @param  data   ReceiveSettings struct data will be stored in this pointer
   @return OSSO_PACK_OK on success (or) OSSO_PACK_INVALID for invalid 
           parameter (or) OSSO_PACK_ERROR for other kind of errors
*/

osso_pack_return_t
osso_email_xml_unpack_receive_settings(xmlNodePtr node, xmlDocPtr doc, gchar * nodeType, void *data)
{
	ReceiveSettings *recv_settings = NULL;

	gint idx = 0;

	if (doc == NULL || node == NULL || data == NULL) {
		osso_log(LOG_INFO, "Invalid input params in %s", __FUNCTION__);
		return OSSO_PACK_INVALID;
	}
	recv_settings = (ReceiveSettings *) data;

	node = node->xmlChildrenNode;

	while (node != NULL) {
		if (node->type != XML_ELEMENT_NODE) {
			node = node->next;
			continue;
		}
		idx = 0;
		if (g_ascii_strcasecmp((gchar *) node->name, RecvSettingsDBusData[idx++].name) == 0) {
			/*protocol */
			osso_email_xml_unpack_int(node, doc, NULL, &recv_settings->protocol);

		} else
		    if (g_ascii_strcasecmp((gchar *) node->name, RecvSettingsDBusData[idx++].name)
			== 0) {
			/* recv server */
			osso_email_xml_unpack_str(node, doc, NULL, &recv_settings->recv_server);
		} else
		    if (g_ascii_strcasecmp((gchar *) node->name, RecvSettingsDBusData[idx++].name)
			== 0) {
			/*port */
			osso_email_xml_unpack_ushort(node, doc, NULL, &recv_settings->port);

		} else
		    if (g_ascii_strcasecmp((gchar *) node->name, RecvSettingsDBusData[idx++].name)
			== 0) {
			/* passwd_auth_type */
			osso_email_xml_unpack_int(node, doc, NULL,
						  &recv_settings->passwd_auth_type);

		} else
		    if (g_ascii_strcasecmp((gchar *) node->name, RecvSettingsDBusData[idx++].name)
			== 0) {
			/* recv_type */
			osso_email_xml_unpack_int(node, doc, NULL, &recv_settings->recv_type);

		} else
		    if (g_ascii_strcasecmp((gchar *) node->name, RecvSettingsDBusData[idx++].name)
			== 0) {
			/* enable_size_limit */
			osso_email_xml_unpack_int(node, doc, NULL,
						  &recv_settings->enable_size_limit);

		} else
		    if (g_ascii_strcasecmp((gchar *) node->name, RecvSettingsDBusData[idx++].name)
			== 0) {
			/* size_limit */
			osso_email_xml_unpack_int(node, doc, NULL, &recv_settings->size_limit);

		} else
		    if (g_ascii_strcasecmp((gchar *) node->name, RecvSettingsDBusData[idx++].name)
			== 0) {
			/* recent_mails */
			osso_email_xml_unpack_int(node, doc, NULL, &recv_settings->recent_mails);
		} else if (g_ascii_strcasecmp((gchar *) node->name,
					      RecvSettingsDBusData[idx++].name) == 0) {
			/* leave_msg */
			osso_email_xml_unpack_int(node, doc, NULL, &recv_settings->leave_msg);
		} else if (g_ascii_strcasecmp((gchar *) node->name,
					      RecvSettingsDBusData[idx++].name) == 0) {
			osso_email_xml_unpack_int(node, doc, NULL, &recv_settings->is_not_editable);
		}
		node = node->next;
	}
	return OSSO_PACK_OK;
}

/**
  This function is used to unpack the 'SendSettings struct  data'
  stored in a xml 'doc' with identifier 'name'

   @param  node   Xml Node pointer of the packed data
   @param  doc    Xml Doc  pointer of the packed data
   @param  name   Tag Identifier of the  SendSettings  struct data
                  in the xml document
   @param  data   SendSettings struct data will be stored in this pointer
   @return OSSO_PACK_OK on success (or) OSSO_PACK_INVALID for invalid 
           parameter (or) OSSO_PACK_ERROR for other kind of errors
*/

osso_pack_return_t
osso_email_xml_unpack_send_settings(xmlNodePtr node, xmlDocPtr doc, gchar * nodeType, void *data)
{
	SendSettings *send_settings = NULL;

	gint idx = 0;

	if (doc == NULL || node == NULL || data == NULL) {
		osso_log(LOG_INFO, "Invalid input params in %s", __FUNCTION__);
		return OSSO_PACK_INVALID;
	}
	send_settings = (SendSettings *) data;

	node = node->xmlChildrenNode;

	while (node != NULL) {
		if (node->type != XML_ELEMENT_NODE) {
			node = node->next;
			continue;
		}
		idx = 0;
		if (g_ascii_strcasecmp((gchar *) node->name, SendSettingsDBusData[idx++].name) == 0) {
			/* smtp_server */
			osso_email_xml_unpack_str(node, doc, NULL, &send_settings->smtp_server);

		} else if (g_ascii_strcasecmp((gchar *) node->name,
					      SendSettingsDBusData[idx++].name) == 0) {

			/* smtp_auth_type  */
			osso_email_xml_unpack_int(node, doc, NULL, &send_settings->smtp_auth_type);

		} else if (g_ascii_strcasecmp((gchar *) node->name,
					      SendSettingsDBusData[idx++].name) == 0) {
			/* smtp_port  */
			osso_email_xml_unpack_ushort(node, doc, NULL, &send_settings->smtp_port);
		} else if (g_ascii_strcasecmp((gchar *) node->name,
					      SendSettingsDBusData[idx++].name) == 0) {
			/* smtp username  */
			osso_email_xml_unpack_str(node, doc, NULL, &send_settings->username);
		} else if (g_ascii_strcasecmp((gchar *) node->name,
					      SendSettingsDBusData[idx++].name) == 0) {
			/* smtp password  */
			osso_email_xml_unpack_str(node, doc, NULL, &send_settings->password);
		}
		node = node->next;
	}
	return OSSO_PACK_OK;

}

/**
  This function is used to unpack the 'AdvancedSettings struct  data'
  stored in a xml 'doc' with identifier 'name'

   @param  node   Xml Node pointer of the packed data
   @param  doc    Xml Doc  pointer of the packed data
   @param  name   Tag Identifier of the  AdvancedSettings  struct data
                  in the xml document
   @param  data   AdvancedSettings struct data will be stored in this pointer
   @return OSSO_PACK_OK on success (or) OSSO_PACK_INVALID for invalid 
           parameter (or) OSSO_PACK_ERROR for other kind of errors
*/

osso_pack_return_t
osso_email_xml_unpack_adv_settings(xmlNodePtr node, xmlDocPtr doc, gchar * nodeType, void *data)
{
	AdvancedSettings *adv_settings = NULL;

	gint idx = 0;

	if (doc == NULL || node == NULL || data == NULL) {
		osso_log(LOG_INFO, "Invalid input params in %s", __FUNCTION__);
		return OSSO_PACK_INVALID;
	}
	adv_settings = (AdvancedSettings *) data;

	node = node->xmlChildrenNode;

	while (node != NULL) {
		if (node->type != XML_ELEMENT_NODE) {
			node = node->next;
			continue;
		}

		idx = 0;
		if (g_ascii_strcasecmp((gchar *) node->name, AdvSettingsDBusData[idx++].name) == 0) {
			/* email_type */
			osso_email_xml_unpack_int(node, doc, NULL, &adv_settings->email_type);
		} else if (g_ascii_strcasecmp((gchar *) node->name, AdvSettingsDBusData[idx++].name)
			   == 0) {
			/* email_sending */
			osso_email_xml_unpack_int(node, doc, NULL, &adv_settings->email_sending);
		} else if (g_ascii_strcasecmp((gchar *) node->name, AdvSettingsDBusData[idx++].name)
			   == 0) {
			/* include_signature */
			osso_email_xml_unpack_bool(node, doc, NULL,
						   &adv_settings->include_signature);
		} else if (g_ascii_strcasecmp((gchar *) node->name, AdvSettingsDBusData[idx++].name)
			   == 0) {
			/* signature_file */
			osso_email_xml_unpack_str(node, doc, NULL, &adv_settings->signature_file);
		} else if (g_ascii_strcasecmp((gchar *) node->name, AdvSettingsDBusData[idx++].name)
			   == 0) {
			/* signature_flag */
			osso_email_xml_unpack_bool(node, doc, NULL, &adv_settings->signature_flag);
		} else if (g_ascii_strcasecmp((gchar *) node->name, AdvSettingsDBusData[idx++].name)
			   == 0) {
			/* request_read_report */
			osso_email_xml_unpack_bool(node, doc, NULL,
						   &adv_settings->request_read_report);
		} else if (g_ascii_strcasecmp((gchar *) node->name, AdvSettingsDBusData[idx++].name)
			   == 0) {
			/* copy_to_own_address */
			osso_email_xml_unpack_bool(node, doc, NULL,
						   &adv_settings->copy_to_own_address);
		} else if (g_ascii_strcasecmp((gchar *) node->name, AdvSettingsDBusData[idx++].name)
			   == 0) {
			/* original_in_reply */
			osso_email_xml_unpack_bool(node, doc, NULL, &adv_settings->org_in_reply);
		} else if (g_ascii_strcasecmp((gchar *) node->name, AdvSettingsDBusData[idx++].name)
			   == 0) {
			/* use_ssl */
			osso_email_xml_unpack_bool(node, doc, NULL, &adv_settings->use_ssl);
		} else if (g_ascii_strcasecmp((gchar *) node->name,
					      AdvSettingsDBusData[idx++].name) == 0) {
			/* imap_dir */
			osso_email_xml_unpack_str(node, doc, NULL, &adv_settings->imap_dir);
#ifdef CL_USE
		} else if (g_ascii_strcasecmp((gchar *) node->name,
					      AdvSettingsDBusData[idx++].name) == 0) {
			/* iap_name */
			osso_email_xml_unpack_str(node, doc, NULL, &adv_settings->iap_name);
		} else if (g_ascii_strcasecmp((gchar *) node->name,
					      AdvSettingsDBusData[idx++].name) == 0) {
			/* use iap specific smtp data */
			osso_email_xml_unpack_bool(node, doc, NULL,
						   &adv_settings->use_iap_smtp_serv);
#endif
		} else if (g_ascii_strcasecmp(node->name, AdvSettingsDBusData[idx++].name) == 0) {
			/* smime_security */
			osso_email_xml_unpack_int(node, doc, NULL, &adv_settings->smime_security);
		} else if (g_ascii_strcasecmp(node->name, AdvSettingsDBusData[idx++].name) == 0) {
			/* display_sign_encrypt */
			osso_email_xml_unpack_bool(node, doc, NULL,
						   &adv_settings->display_sign_encrypt);
		} else if (g_ascii_strcasecmp(node->name, AdvSettingsDBusData[idx++].name) == 0) {
			/* display_trust */
			osso_email_xml_unpack_bool(node, doc, NULL, &adv_settings->display_trust);
		} else if (g_ascii_strcasecmp(node->name, AdvSettingsDBusData[idx++].name) == 0) {
			/* display_validity */
			osso_email_xml_unpack_bool(node, doc, NULL,
						   &adv_settings->display_validity);
		} else if (g_ascii_strcasecmp((gchar *) node->name, AdvSettingsDBusData[idx++].name)
			   == 0) {
			/* use_tls */
			osso_email_xml_unpack_bool(node, doc, NULL, &adv_settings->use_tls);
		} else if (g_ascii_strcasecmp((gchar *) node->name, AdvSettingsDBusData[idx++].name)
			   == 0) {
			/* smtp_use_ssl */
			osso_email_xml_unpack_bool(node, doc, NULL, &adv_settings->smtp_use_ssl);
		} else if (g_ascii_strcasecmp((gchar *) node->name, AdvSettingsDBusData[idx++].name)
			   == 0) {
			/* smtp_use_tls */
			osso_email_xml_unpack_bool(node, doc, NULL, &adv_settings->smtp_use_tls);
		}
		node = node->next;
	}
	return OSSO_PACK_OK;
}


/**
  This function is used to free the mime info header
   @param  mimeinfo  structure which is to be freed 
   @return  OSSO_PACK_OK on success or FALSE on failure
*/
void osso_email_xml_mimeinfo_free_all(MimeInfo * mimeinfo)
{
	MimeInfo *next = NULL;
	while (mimeinfo != NULL) {
		xmlFree(mimeinfo->encoding);
		xmlFree(mimeinfo->content_type);
		xmlFree(mimeinfo->charset);
		xmlFree(mimeinfo->name);
		xmlFree(mimeinfo->boundary);
		xmlFree(mimeinfo->content_disposition);
		xmlFree(mimeinfo->filename);
		xmlFree(mimeinfo->section_number);
		xmlFree(mimeinfo->cert_path);
		procmsg_ImageInfo_list_free_DBUS(mimeinfo->inline_images);

		osso_email_xml_mimeinfo_free_all(mimeinfo->sub);
		osso_email_xml_mimeinfo_free_all(mimeinfo->children);
		next = mimeinfo->next;
		g_free(mimeinfo);
		mimeinfo = next;
	}
}

/*\@}*/
/**
 * \defgroup Pack unpack of statesave information in xml format 
 */
/*\@{*/

/*\@}*/

/**
  This function is used to pack the 'FolderSortInfo structure data ' received
  into a xml 'doc' with identifier 'name'.

   @param  node   Xml Node pointer to pack the data
   @param  doc    Xml Doc  pointer to pack the data
   @param  name   Tag Identifier of the Msgheader struct in the xml document
   @param  data   FolderSortInfo Structure data to be packed in xml format
   @return OSSO_PACK_OK on success (or) OSSO_PACK_INVALID for invalid 
           parameter (or) OSSO_PACK_ERROR for other kind of errors
*/

static osso_pack_return_t
osso_email_xml_pack_folderSortInfo(xmlNodePtr node, xmlDocPtr doc, void *nodeType,
				   void *folderSortInfo)
{
	xmlNodePtr root = NULL, child = NULL;

	if ((doc == NULL) || (folderSortInfo == NULL) || (nodeType == NULL)) {
		osso_log(LOG_INFO, "Invalid input params in %s", __FUNCTION__);
		return OSSO_PACK_INVALID;
	}
	NEW_NODE(node, root, child, nodeType);
	memset(&tmpFolderSortInfo, '\0', sizeof(tmpFolderSortInfo));
	tmpFolderSortInfo = *(FolderSortInfo *) folderSortInfo;
	osso_email_xml_pack_struct(node, doc, NULL, FolderSortInfoDBusData);
	return OSSO_PACK_OK;
}

/**
  This function is used to pack the 'OpenedWindowData structure data ' received
  into a xml 'doc' with identifier 'name'.

   @param  node   Xml Node pointer to pack the data
   @param  doc    Xml Doc  pointer to pack the data
   @param  name   Tag Identifier of the Msgheader struct in the xml document
   @param  data   OpenedWindowData Structure data to be packed in xml format
   @return OSSO_PACK_OK on success (or) OSSO_PACK_INVALID for invalid 
           parameter (or) OSSO_PACK_ERROR for other kind of errors
*/

static osso_pack_return_t
osso_email_xml_pack_openWindowData(xmlNodePtr node, xmlDocPtr doc, void *nodeType, void *openMsg)
{
	xmlNodePtr root = NULL, child = NULL;

	if ((doc == NULL) || (openMsg == NULL) || (nodeType == NULL)) {
		osso_log(LOG_INFO, "Invalid input params in %s", __FUNCTION__);
		return OSSO_PACK_INVALID;
	}
	NEW_NODE(node, root, child, nodeType);
	memset(&tmpOpenedWindowData, '\0', sizeof(tmpOpenedWindowData));
	tmpOpenedWindowData = *(OpenedWindowData *) openMsg;
	osso_email_xml_pack_struct(node, doc, NULL, OpenedWindowDataDBusData);
	return OSSO_PACK_OK;
}

/**
  This function is used to pack the 'EmailStateInfo structure data ' received
  into a xml 'doc' with identifier 'name'.

   @param  node   Xml Node pointer to pack the data
   @param  doc    Xml Doc  pointer to pack the data
   @param  name   Tag Identifier of the MsgInfo struct in the xml document
   @param  data   EmailStateInfo Structure data to be packed in xml format
   @return OSSO_PACK_OK on success (or) OSSO_PACK_INVALID for invalid 
           parameter (or) OSSO_PACK_ERROR for other kind of errors
*/


osso_pack_return_t
osso_email_xml_pack_emailStateInfo(xmlNodePtr node, xmlDocPtr doc, void *nodeType, void *data)
{
	xmlNodePtr root = NULL, child = NULL;
	EmailStateInfo *emailStateInfo = NULL;

	if (doc == NULL || data == NULL || nodeType == NULL) {
		osso_log(LOG_INFO, "Invalid input params in %s", __FUNCTION__);
		return OSSO_PACK_INVALID;
	}
	emailStateInfo = (EmailStateInfo *) data;

	NEW_NODE(node, root, child, nodeType);
	memset(&tmpEmailStateInfo, '\0', sizeof(tmpEmailStateInfo));

	tmpEmailStateInfo = *(EmailStateInfo *) emailStateInfo;
	osso_email_xml_pack_struct(node, doc, NULL, EmailStateInfoDBusData);

	osso_email_xml_pack_msgid_list(node, doc, XML_MARKED_MSGS_LST, emailStateInfo->marked_msgs);

	osso_email_xml_pack_folderSortInfo_list(node, doc, XML_FOLDER_SORT_INFO_LST,
						emailStateInfo->folder_sort_info_list);

	osso_email_xml_pack_openMsgs_list(node, doc, XML_OPEN_MSGS_LST,
					  emailStateInfo->open_msgs_list);

	osso_email_xml_pack_msgheader_list(node, doc, XML_MSG_HEADER_LST,
					   emailStateInfo->msgheaders_list);
	return OSSO_PACK_OK;
}


/**
  This function is used to pack the 'EmailPersistentInfo structure data ' 
  received into a xml 'doc' with identifier 'name'.

   @param  node   Xml Node pointer to pack the data
   @param  doc    Xml Doc  pointer to pack the data
   @param  name   Tag Identifier of the MsgInfo struct in the xml doc
   @param  data   EmailPersistentInfo Struct to be packed in xml format
   @return OSSO_PACK_OK on success (or) OSSO_PACK_INVALID for invalid 
           parameter (or) OSSO_PACK_ERROR for other kind of errors
*/

osso_pack_return_t
osso_email_xml_pack_emailPersistentInfo(xmlNodePtr node, xmlDocPtr doc, void *nodeType, void *data)
{
	xmlNodePtr root = NULL, child = NULL;
	EmailPersistentInfo *perInfo = NULL;

	if (doc == NULL || data == NULL || nodeType == NULL) {
		osso_log(LOG_INFO, "Invalid input params in %s", __FUNCTION__);
		return OSSO_PACK_INVALID;
	}
	perInfo = (EmailPersistentInfo *) data;

	NEW_NODE(node, root, child, nodeType);
	memset(&tmpEmailPersistentInfo, '\0', sizeof(tmpEmailPersistentInfo));

	tmpEmailPersistentInfo = *(EmailPersistentInfo *) perInfo;
	osso_email_xml_pack_struct(node, doc, NULL, EmailPersistentInfoDBusData);
	osso_email_xml_pack_folderSortInfo_list(node, doc,
						XML_FOLDER_SORT_INFO_LST,
						perInfo->folder_sort_info_list);
	return OSSO_PACK_OK;
}

/**
  This function is used to pack the GSList of FolderSortInfo structures received
  into a xml 'doc' with identifier 'name'.

   @param  node   Xml Node pointer to pack the data
   @param  doc    Xml Doc  pointer to pack the data
   @param  name   Tag Identifier of the MsgInfo struct in the xml document
   @param  data   GSlist of FolderSortInfo Structures to be packed in xml format
   @return OSSO_PACK_OK on success (or) OSSO_PACK_INVALID for invalid 
           parameter (or) OSSO_PACK_ERROR for other kind of errors
*/


static osso_pack_return_t
osso_email_xml_pack_folderSortInfo_list(xmlNodePtr node, xmlDocPtr doc, void *nodeType,
					void *folderSortInfo_list)
{
	xmlNodePtr root = NULL, child = NULL;
	GSList *sort_list = NULL;
	GSList *cur = NULL;

	if ((node == NULL) || (nodeType == NULL)) {
		osso_log(LOG_INFO, "Invalid input params in %s", __FUNCTION__);
		return OSSO_PACK_INVALID;
	}
	sort_list = (GSList *) folderSortInfo_list;
	NEW_NODE(node, root, child, nodeType);
	for (cur = folderSortInfo_list; cur != NULL; cur = cur->next) {
		if (cur->data == NULL) {
			continue;
		}
		osso_email_xml_pack_folderSortInfo(node, doc, XML_FOLDER_SORT_INFO, cur->data);
	}
	return OSSO_PACK_OK;
}

/**
  This function is used to pack the 'GSList OpenedWindowData structure data' received
  into a xml 'doc' with identifier 'name'.

   @param  node   Xml Node pointer to pack the data
   @param  doc    Xml Doc  pointer to pack the data
   @param  name   Tag Identifier of the MsgInfo struct in the xml document
   @param  data   GSList of OpenedWindowData structs data to be packed in xml format
   @return OSSO_PACK_OK on success (or) OSSO_PACK_INVALID for invalid 
           parameter (or) OSSO_PACK_ERROR for other kind of errors
*/

static osso_pack_return_t
osso_email_xml_pack_openMsgs_list(xmlNodePtr node, xmlDocPtr doc, void *nodeType,
				  void *openMsgs_list)
{
	xmlNodePtr root = NULL, child = NULL;
	GSList *cur = NULL;
	GSList *windows_list = NULL;

	if ((node == NULL) || (nodeType == NULL)) {
		osso_log(LOG_INFO, "Invalid input params in %s", __FUNCTION__);
		return OSSO_PACK_INVALID;
	}
	windows_list = (GSList *) openMsgs_list;

	NEW_NODE(node, root, child, nodeType);
	for (cur = windows_list; cur != NULL; cur = cur->next) {
		if (cur->data == NULL) {
			continue;
		}
		osso_email_xml_pack_openWindowData(node, doc, XML_OPEN_WINDOW_DATA, cur->data);
	}
	return OSSO_PACK_OK;
}


/**
  This function is used to unpack the 'EmailStateInfo struct  data'
  stored in a xml 'doc' with identifier 'name'

   @param  node   Xml Node pointer of the packed data
   @param  doc    Xml Doc  pointer of the packed data
   @param  name   Tag Identifier of the EmailStateInfo struct data
                  in the xml document
   @param  data   MsgInfo struct data will be stored in this pointer
   @return OSSO_PACK_OK on success (or) OSSO_PACK_INVALID for invalid 
           parameter (or) OSSO_PACK_ERROR for other kind of errors
*/
osso_pack_return_t
osso_email_xml_unpack_emailStateInfo(xmlNodePtr node, xmlDocPtr doc, gchar * nodeType, void *data)
{
	EmailStateInfo *stateInfo = NULL;
	//GSList *markedMsg_list = NULL;
	gint idx = 0;
	if (doc == NULL || node == NULL || data == NULL) {
		osso_log(LOG_INFO, "Invalid input params in %s", __FUNCTION__);
		return OSSO_PACK_INVALID;
	}
	node = node->xmlChildrenNode;

	stateInfo = (EmailStateInfo *) data;

	while (node != NULL) {
		if (node->type != XML_ELEMENT_NODE) {
			node = node->next;
			continue;
		}
		idx = 0;
		if (g_ascii_strcasecmp((gchar *) node->name, EmailStateInfoDBusData[idx++].name) ==
		    0) {
			/*oom_signal_received */
			osso_email_xml_unpack_bool(node, doc, NULL,
						   &stateInfo->oom_signal_received);
		} else
		    if (g_ascii_strcasecmp((gchar *) node->name, EmailStateInfoDBusData[idx++].name)
			== 0) {
			/*split_view_mode */
			osso_email_xml_unpack_bool(node, doc, NULL, &stateInfo->split_view_mode);
		} else
		    if (g_ascii_strcasecmp((gchar *) node->name, EmailStateInfoDBusData[idx++].name)
			== 0) {
			/*full_screen_mode */
			osso_email_xml_unpack_bool(node, doc, NULL, &stateInfo->full_screen_mode);
		} else
		    if (g_ascii_strcasecmp((gchar *) node->name, EmailStateInfoDBusData[idx++].name)
			== 0) {
			/*tool_bar_hidden */
			osso_email_xml_unpack_bool(node, doc, NULL, &stateInfo->tool_bar_hidden);
		} else
		    if (g_ascii_strcasecmp((gchar *) node->name, EmailStateInfoDBusData[idx++].name)
			== 0) {
			/*cc_field_on */
			osso_email_xml_unpack_bool(node, doc, NULL, &stateInfo->cc_field_on);
		} else
		    if (g_ascii_strcasecmp((gchar *) node->name, EmailStateInfoDBusData[idx++].name)
			== 0) {
			/*bcc_field_on */
			osso_email_xml_unpack_bool(node, doc, NULL, &stateInfo->bcc_field_on);
		} else
		    if (g_ascii_strcasecmp((gchar *) node->name, EmailStateInfoDBusData[idx++].name)
			== 0) {
			/*folderview_scroll_position */
			osso_email_xml_unpack_int(node, doc, NULL,
						  &stateInfo->folderview_scroll_position);
		} else
		    if (g_ascii_strcasecmp((gchar *) node->name, EmailStateInfoDBusData[idx++].name)
			== 0) {
			/*msgview_scroll_position */
			osso_email_xml_unpack_int(node, doc, NULL,
						  &stateInfo->msgview_scroll_position);
		} else
		    if (g_ascii_strcasecmp((gchar *) node->name, EmailStateInfoDBusData[idx++].name)
			== 0) {
			/*current_folder */
			osso_email_xml_unpack_str(node, doc, NULL, &stateInfo->current_folder);
		} else
		    if (g_ascii_strcasecmp((gchar *) node->name, EmailStateInfoDBusData[idx++].name)
			== 0) {
			/*current_msg_id */
			osso_email_xml_unpack_str(node, doc, NULL, &stateInfo->current_msg_id);
		} else if (g_ascii_strcasecmp((gchar *) node->name, XML_MARKED_MSGS_LST) == 0) {
			/*marked_msgs */
			osso_email_xml_unpack_msgid_list(node, doc, NULL, &stateInfo->marked_msgs);
		} else if (g_ascii_strcasecmp((gchar *) node->name, XML_FOLDER_SORT_INFO_LST) == 0) {
			/*folder_sort_info_list */
			osso_email_xml_unpack_folderSortInfo_list(node, doc, NULL,
								  &stateInfo->
								  folder_sort_info_list);
		} else if (g_ascii_strcasecmp((gchar *) node->name, XML_OPEN_MSGS_LST) == 0) {
			/*open_msgs_list */
			osso_email_xml_unpack_open_msgs_list(node, doc, NULL,
							     &stateInfo->open_msgs_list);
		} else if (g_ascii_strcasecmp((gchar *) node->name, XML_MSG_HEADER_LST) == 0) {
			/*msg header list */
			osso_email_xml_unpack_msgheader_list(node, doc, NULL,
							     &stateInfo->msgheaders_list);
		}
		node = node->next;
	}
	return OSSO_PACK_OK;
}


/**
  This function is used to unpack the 'EmailPersistentInfo struct  data'
  stored in a xml 'doc' with identifier 'name'

   @param  node   Xml Node pointer of the packed data
   @param  doc    Xml Doc  pointer of the packed data
   @param  name   Tag Identifier of the EmailPersistentInfo struct data
                  in the xml document
   @param  data   MsgInfo struct data will be stored in this pointer
   @return OSSO_PACK_OK on success (or) OSSO_PACK_INVALID for invalid 
           parameter (or) OSSO_PACK_ERROR for other kind of errors
*/
osso_pack_return_t
osso_email_xml_unpack_emailPersistentInfo(xmlNodePtr node,
					  xmlDocPtr doc, gchar * nodeType, void *data)
{
	EmailPersistentInfo *persInfo = NULL;
	gint idx = 0;

	if (doc == NULL || node == NULL || data == NULL) {
		osso_log(LOG_INFO, "Invalid input params in %s", __FUNCTION__);
		return OSSO_PACK_INVALID;
	}

	node = node->xmlChildrenNode;
	persInfo = (EmailPersistentInfo *) data;

	while (node != NULL) {
		if (node->type != XML_ELEMENT_NODE) {
			node = node->next;
			continue;
		}
		idx = 0;
		if (g_ascii_strcasecmp(node->name, EmailPersistentInfoDBusData[idx++].name) == 0) {
			/*split_view_mode */
			osso_email_xml_unpack_bool(node, doc, NULL, &persInfo->split_view_mode);
		} else if (g_ascii_strcasecmp(node->name,
					      EmailPersistentInfoDBusData[idx++].name) == 0) {
			/*main view pane position */
			osso_email_xml_unpack_int(node, doc, NULL, &persInfo->mainview_pane_pos);
		} else if (g_ascii_strcasecmp(node->name,
					      EmailPersistentInfoDBusData[idx++].name) == 0) {
			/*List view from column width */
			osso_email_xml_unpack_int(node, doc, NULL, &persInfo->listview_from_col_width);
		} else if (g_ascii_strcasecmp(node->name,
					      EmailPersistentInfoDBusData[idx++].name) == 0) {
			/*List view subject column width */
			osso_email_xml_unpack_int(node, doc, NULL, &persInfo->listview_subject_col_width);
		} else if (g_ascii_strcasecmp(node->name,
					      EmailPersistentInfoDBusData[idx++].name) == 0) {
			/*main view tool_bar_hidden */
			osso_email_xml_unpack_bool(node, doc, NULL,
						   &persInfo->mainview_tool_bar_hidden);
		} else if (g_ascii_strcasecmp(node->name,
					      EmailPersistentInfoDBusData[idx++].name) == 0) {
			/* viewer tool_bar_hidden */
			osso_email_xml_unpack_bool(node, doc, NULL,
						   &persInfo->viewer_tool_bar_hidden);
		} else if (g_ascii_strcasecmp(node->name,
					      EmailPersistentInfoDBusData[idx++].name) == 0) {
			/* editor tool_bar_hidden */
			osso_email_xml_unpack_bool(node, doc, NULL,
						   &persInfo->editor_tool_bar_hidden);
		} else if (g_ascii_strcasecmp(node->name,
					      EmailPersistentInfoDBusData[idx++].name) == 0) {
			/*main view fullscreen tool_bar_hidden */
			osso_email_xml_unpack_bool(node, doc, NULL,
						   &persInfo->mainview_fullscreen_tool_bar_hidden);
		} else if (g_ascii_strcasecmp(node->name,
					      EmailPersistentInfoDBusData[idx++].name) == 0) {
			/*viewer fullscreen tool_bar_hidden */
			osso_email_xml_unpack_bool(node, doc, NULL,
						   &persInfo->viewer_fullscreen_tool_bar_hidden);
		} else if (g_ascii_strcasecmp(node->name,
					      EmailPersistentInfoDBusData[idx++].name) == 0) {
			/*editor fullscreen tool_bar_hidden */
			osso_email_xml_unpack_bool(node, doc, NULL,
						   &persInfo->editor_fullscreen_tool_bar_hidden);
		} else if (g_ascii_strcasecmp(node->name,
					      EmailPersistentInfoDBusData[idx++].name) == 0) {
			/*cc_field_on */
			osso_email_xml_unpack_bool(node, doc, NULL, &persInfo->cc_field_on);
		} else if (g_ascii_strcasecmp(node->name, EmailPersistentInfoDBusData[idx++].name) == 0) {
			/*bcc_field_on */
			osso_email_xml_unpack_bool(node, doc, NULL, &persInfo->bcc_field_on);
		} else
		    if (g_ascii_strcasecmp
			((gchar *) node->name, EmailPersistentInfoDBusData[idx++].name)
			== 0) {
			/*read_receipt */
			osso_email_xml_unpack_bool(node, doc, NULL, &persInfo->read_receipt);
		} else
		    if (g_ascii_strcasecmp
			((gchar *) node->name, EmailPersistentInfoDBusData[idx++].name)
			== 0) {
			/*msg_size_limit */
			osso_email_xml_unpack_int(node, doc, NULL, &persInfo->msg_size_limit);
		} else
		    if (g_ascii_strcasecmp
			((gchar *) node->name, EmailPersistentInfoDBusData[idx++].name)
			== 0) {
			/*play_sound */
			osso_email_xml_unpack_bool(node, doc, NULL, &persInfo->play_sound);
		} else
		    if (g_ascii_strcasecmp
			((gchar *) node->name, EmailPersistentInfoDBusData[idx++].name)
			== 0) {
			/*msg_format_type */
			osso_email_xml_unpack_int(node, doc, NULL, &persInfo->msg_format_type);
		} else
		    if (g_ascii_strcasecmp
			((gchar *) node->name, EmailPersistentInfoDBusData[idx++].name)
			== 0) {
			/*inc_original */
			osso_email_xml_unpack_bool(node, doc, NULL, &persInfo->inc_original);
		} else if (g_ascii_strcasecmp(node->name,
					      EmailPersistentInfoDBusData[idx++].name) == 0) {
			/*delete_servermail_also */
			osso_email_xml_unpack_bool(node, doc, NULL,
						   &persInfo->delete_servermail_also);
		} else if (g_ascii_strcasecmp(node->name,
					      EmailPersistentInfoDBusData[idx++].name) == 0) {
			/*sent_folder_clean_time_stamp */
			osso_email_xml_unpack_int(node, doc, NULL,
						  &persInfo->sent_folder_clean_time_stamp);
		} else if (g_ascii_strcasecmp(node->name,
					      EmailPersistentInfoDBusData[idx++].name) == 0) {
			/*sent_folder_clean_option */
			osso_email_xml_unpack_int(node, doc, NULL,
						  &persInfo->sent_folder_clean_option);
		} else if (g_ascii_strcasecmp(node->name,
					      EmailPersistentInfoDBusData[idx++].name) == 0) {
			/*sent_folder_clean_time_value */
			osso_email_xml_unpack_int(node, doc, NULL,
						  &persInfo->sent_folder_clean_time_value);
		} else if (g_ascii_strcasecmp(node->name,
					      EmailPersistentInfoDBusData[idx++].name) == 0) {
			/*viewing_html_fontname_index */
			osso_email_xml_unpack_int(node, doc, NULL,
						  &persInfo->viewing_html_fontname_index);
		} else if (g_ascii_strcasecmp(node->name,
					      EmailPersistentInfoDBusData[idx++].name) == 0) {
			/*viewing_plain_fontname_index */
			osso_email_xml_unpack_int(node, doc, NULL,
						  &persInfo->viewing_plain_fontname_index);
		} else if (g_ascii_strcasecmp(node->name,
					      EmailPersistentInfoDBusData[idx++].name) == 0) {
			/*editing_html_fontname_index */
			osso_email_xml_unpack_int(node, doc, NULL,
						  &persInfo->editing_html_fontname_index);
		} else if (g_ascii_strcasecmp(node->name,
					      EmailPersistentInfoDBusData[idx++].name) == 0) {
			/*editing_plain_fontname_index */
			osso_email_xml_unpack_int(node, doc, NULL,
						  &persInfo->editing_plain_fontname_index);
		} else if (g_ascii_strcasecmp(node->name, XML_FOLDER_SORT_INFO_LST) == 0) {
			/*folder_sort_info_list */
			osso_email_xml_unpack_folderSortInfo_list(node, doc, NULL,
								  &persInfo->folder_sort_info_list);
		}
		node = node->next;
	}
	return OSSO_PACK_OK;
}

/**
  This function is used to unpack the 'GsList of FolderSortInfo struct  data'
  stored in a xml 'doc' with identifier 'name'

   @param  node   Xml Node pointer of the packed data
   @param  doc    Xml Doc  pointer of the packed data
   @param  name   Tag Identifier of the GSList of FolderSortInfo struct data
                  in the xml document
   @param  data   GSList of  MsgInfo  struct data will be stored
                  in this pointer
   @return OSSO_PACK_OK on success (or) OSSO_PACK_INVALID for invalid 
           parameter (or) OSSO_PACK_ERROR for other kind of errors
*/


static osso_pack_return_t
osso_email_xml_unpack_folderSortInfo_list(xmlNodePtr node, xmlDocPtr doc,
					  gchar * nodeType, GSList ** data)
{
	GSList *sort_list = NULL;
	FolderSortInfo *folderSortInfo = NULL;

	if (doc == NULL || node == NULL || data == NULL) {
		osso_log(LOG_INFO, "Invalid input params in %s", __FUNCTION__);
		return OSSO_PACK_INVALID;
	}

	for (node = node->xmlChildrenNode; node; node = node->next) {
		if (node->type == XML_ELEMENT_NODE) {
			if (g_ascii_strcasecmp(node->name, XML_FOLDER_SORT_INFO) == 0) {
				folderSortInfo = g_new0(FolderSortInfo, 1);
				if (folderSortInfo == NULL) {
					osso_log(LOG_ERR, "Memory alloc fail \n");
					return OSSO_PACK_ALLOC_FAIL;
				}
				osso_email_xml_unpack_foldersortInfo(node, doc,
								     XML_FOLDER_SORT_INFO,
								     folderSortInfo);
				sort_list = g_slist_append(sort_list, folderSortInfo);
			}
		}
	}
	*data = sort_list;
	return OSSO_PACK_OK;
}

/**
  This function is used to unpack the 'GsList of OpenedWindowData struct  data'
  stored in a xml 'doc' with identifier 'name'

   @param  node   Xml Node pointer of the packed data
   @param  doc    Xml Doc  pointer of the packed data
   @param  name   Tag Identifier of the GSList of OpenedWindowData struct data
                  in the xml document
   @param  data   GSList of  window_list  struct data will be stored
                  in this pointer
   @return OSSO_PACK_OK on success (or) OSSO_PACK_INVALID for invalid 
           parameter (or) OSSO_PACK_ERROR for other kind of errors
*/


static osso_pack_return_t
osso_email_xml_unpack_open_msgs_list(xmlNodePtr node, xmlDocPtr doc,
				     gchar * nodeType, GSList ** data)
{
	GSList *window_list = NULL;
	OpenedWindowData *openmsg = NULL;

	if (doc == NULL || node == NULL || data == NULL) {
		osso_log(LOG_INFO, "Invalid input params in %s", __FUNCTION__);
		return OSSO_PACK_INVALID;
	}

	for (node = node->xmlChildrenNode; node; node = node->next) {
		if (node->type == XML_ELEMENT_NODE) {
			if (g_ascii_strcasecmp((gchar *) node->name, XML_OPEN_WINDOW_DATA) == 0) {
				openmsg = g_new0(OpenedWindowData, 1);
				if (openmsg == NULL) {
					osso_log(LOG_ERR, "Memory alloc fail \n");
					return OSSO_PACK_ALLOC_FAIL;
				}
				osso_email_xml_unpack_openWindowData(node, doc,
								     XML_OPEN_WINDOW_DATA, openmsg);
				window_list = g_slist_append(window_list, openmsg);
			}
		}
	}
	*data = window_list;
	return OSSO_PACK_OK;
}

/**
  This function is used to unpack the 'FolderSortInfo struct  data'
  stored in a xml 'doc' with identifier 'name'

   @param  node   Xml Node pointer of the packed data
   @param  doc    Xml Doc  pointer of the packed data
   @param  name   Tag Identifier of the  FolderSortInfo struct data
                  in the xml document
   @param  data   Folder AttachInfo  struct data will be stored in this pointer
   @return OSSO_PACK_OK on success (or) OSSO_PACK_INVALID for invalid 
           parameter (or) OSSO_PACK_ERROR for other kind of errors
*/


static osso_pack_return_t
osso_email_xml_unpack_foldersortInfo(xmlNodePtr node, xmlDocPtr doc, gchar * nodeType, void *data)
{
	FolderSortInfo *sortInfo = NULL;
	gint idx = 0;
	if (doc == NULL || node == NULL || data == NULL) {
		osso_log(LOG_INFO, "Invalid input params in %s", __FUNCTION__);
		return OSSO_PACK_INVALID;
	}
	sortInfo = (FolderSortInfo *) data;
	node = node->xmlChildrenNode;
	while (node != NULL) {
		if (node->type != XML_ELEMENT_NODE) {
			node = node->next;
			continue;
		}
		idx = 0;
		if (g_ascii_strcasecmp((gchar *) node->name, FolderSortInfoDBusData[idx++].name) ==
		    0) {
			/*name */
			osso_email_xml_unpack_str(node, doc, NULL, &sortInfo->name);
		} else
		    if (g_ascii_strcasecmp((gchar *) node->name, FolderSortInfoDBusData[idx++].name)
			== 0) {
			/*sort_key */
			osso_email_xml_unpack_int(node, doc, NULL, &sortInfo->sort_key);
		} else
		    if (g_ascii_strcasecmp((gchar *) node->name, FolderSortInfoDBusData[idx++].name)
			== 0) {
			/*sort_order */
			osso_email_xml_unpack_int(node, doc, NULL, &sortInfo->sort_order);
		}
		node = node->next;
	}
	return OSSO_PACK_OK;
}


/**
  This function is used to unpack the 'OpenedWindowData struct  data'
  stored in a xml 'doc' with identifier 'name'

   @param  node   Xml Node pointer of the packed data
   @param  doc    Xml Doc  pointer of the packed data
   @param  name   Tag Identifier of the Folder AttachInfo  struct data
                  in the xml document
   @param  data   OpenedWindowData  struct data will be stored in this pointer
   @return OSSO_PACK_OK on success (or) OSSO_PACK_INVALID for invalid 
           parameter (or) OSSO_PACK_ERROR for other kind of errors
*/


static osso_pack_return_t
osso_email_xml_unpack_openWindowData(xmlNodePtr node, xmlDocPtr doc, gchar * nodeType, void *data)
{
	OpenedWindowData *windowdata = NULL;
	gint idx = 0;
	if (doc == NULL || node == NULL || data == NULL) {
		osso_log(LOG_INFO, "Invalid input params in %s", __FUNCTION__);
		return OSSO_PACK_INVALID;
	}
	windowdata = (OpenedWindowData *) data;
	node = node->xmlChildrenNode;
	while (node != NULL) {
		if (node->type != XML_ELEMENT_NODE) {
			node = node->next;
			continue;
		}
		idx = 0;
		if (g_ascii_strcasecmp((gchar *) node->name, OpenedWindowDataDBusData[idx++].name)
		    == 0) {
			/*msgid */
			osso_email_xml_unpack_str(node, doc, NULL, &windowdata->msgid);
		} else
		    if (g_ascii_strcasecmp
			((gchar *) node->name, OpenedWindowDataDBusData[idx++].name) == 0) {
			/*msgNum */
			osso_email_xml_unpack_int(node, doc, NULL, &windowdata->msgNum);
		} else
		    if (g_ascii_strcasecmp
			((gchar *) node->name, OpenedWindowDataDBusData[idx++].name) == 0) {
			/*topSequence */
			osso_email_xml_unpack_int(node, doc, NULL, &windowdata->topSequence);
		} else
		    if (g_ascii_strcasecmp
			((gchar *) node->name, OpenedWindowDataDBusData[idx++].name) == 0) {
			/*oldmsgid */
			osso_email_xml_unpack_str(node, doc, NULL, &windowdata->oldmsgid);
		} else
		    if (g_ascii_strcasecmp
			((gchar *) node->name, OpenedWindowDataDBusData[idx++].name) == 0) {
			/*iseditor */
			osso_email_xml_unpack_bool(node, doc, NULL, &windowdata->iseditor);
		} else
		    if (g_ascii_strcasecmp
			((gchar *) node->name, OpenedWindowDataDBusData[idx++].name) == 0) {
			/*scroll_position */
			osso_email_xml_unpack_int(node, doc, NULL, &windowdata->scroll_position);
		} else
		    if (g_ascii_strcasecmp
			((gchar *) node->name, OpenedWindowDataDBusData[idx++].name) == 0) {
			/*zoom_percentage */
			osso_email_xml_unpack_int(node, doc, NULL, &windowdata->zoom_percentage);
		} else
		    if (g_ascii_strcasecmp
			((gchar *) node->name, OpenedWindowDataDBusData[idx++].name) == 0) {
			/*cursor_x_pos */
			osso_email_xml_unpack_int(node, doc, NULL, &windowdata->cursor_x_pos);
		} else
		    if (g_ascii_strcasecmp
			((gchar *) node->name, OpenedWindowDataDBusData[idx++].name) == 0) {
			/*cursor_y_pos */
			osso_email_xml_unpack_int(node, doc, NULL, &windowdata->cursor_y_pos);
		}
		node = node->next;
	}
	return OSSO_PACK_OK;
}

osso_pack_return_t
osso_email_xml_pack_retr_list(xmlNodePtr node, xmlDocPtr doc, void *nodeType, GSList * msgid_list)
{
	xmlNodePtr root = NULL, child = NULL;
	GSList *cur = NULL;

	if ((doc == NULL) || (msgid_list == NULL) || (nodeType == NULL)) {
		osso_log(LOG_INFO, "Invalid input params in %s", __FUNCTION__);
		return OSSO_PACK_INVALID;
	}
	NEW_NODE(node, root, child, nodeType);
	for (cur = msgid_list; cur != NULL; cur = cur->next) {
		if (cur->data == NULL) {
			continue;
		}
		osso_email_xml_pack_retr_msg(node, doc, XML_RETR_MSG, cur->data);
	}
	return OSSO_PACK_OK;
}
static osso_pack_return_t
osso_email_xml_pack_retr_msg(xmlNodePtr node, xmlDocPtr doc, void *nodeType, void *data)
{
	xmlNodePtr root = NULL, child = NULL;
	RetrMsgData *retr_data = NULL;

	if ((doc == NULL) || (data == NULL) || (nodeType == NULL)) {
		osso_log(LOG_INFO, "Invalid input params in %s", __FUNCTION__);
		return OSSO_PACK_INVALID;
	}
	retr_data = (RetrMsgData *) data;

	NEW_NODE(node, root, child, nodeType);
	osso_email_xml_pack_str(node, doc, XML_MSG_ID, &retr_data->msgid);
	osso_email_xml_pack_bool(node, doc, XML_BOOL, &retr_data->flag);
	/*
	   memset ( &tmpMsgHeader, '\0', sizeof ( tmpMsgHeader ) );
	   tmpMsgHeader = *( MsgHeader * ) msgHeader;
	   reoturn osso_email_xml_pack_struct ( node, doc, NULL, MsgHeaderDBusData );
	 */
	return OSSO_PACK_OK;

}

osso_pack_return_t
osso_email_xml_unpack_retr_list(xmlNodePtr node, xmlDocPtr doc, gchar * nodeType, GSList ** data)
{
	GSList *msgid_list = NULL;
	RetrMsgData *retr_msg = NULL;

	if (doc == NULL || node == NULL || data == NULL) {
		osso_log(LOG_INFO, "Invalid input params in %s", __FUNCTION__);
		return OSSO_PACK_INVALID;
	}

	for (node = node->xmlChildrenNode; node != NULL; node = node->next) {
		if (node->type == XML_ELEMENT_NODE) {
			if (g_ascii_strcasecmp((gchar *) node->name, XML_RETR_MSG) == 0) {

				retr_msg = g_new0(RetrMsgData, 1);

				osso_email_xml_unpack_retr_msg(node, doc, XML_RETR_MSG, retr_msg);
				msgid_list = g_slist_append(msgid_list, retr_msg);
			}
		}
	}

	*data = msgid_list;
	return OSSO_PACK_OK;
}

static gint
osso_email_xml_unpack_retr_msg(xmlNodePtr node, xmlDocPtr doc,
			       gchar * nodeType, RetrMsgData * retr_msg)
{

	//RetrMsgData *retr_msg = NULL;

	if (doc == NULL || node == NULL || retr_msg == NULL) {
		osso_log(LOG_INFO, "Invalid input params in %s", __FUNCTION__);
		return OSSO_PACK_INVALID;
	}
	//retr_msg = ( RetrMsgData * ) data;

	node = node->xmlChildrenNode;

	while (node != NULL) {
		if (node->type != XML_ELEMENT_NODE) {
			node = node->next;
			continue;
		}


		if (g_ascii_strcasecmp((gchar *) node->name, XML_MSG_ID) == 0) {
			osso_email_xml_unpack_str(node, doc, XML_MSG_ID, &(retr_msg->msgid));
			node = node->next;

		}
		if (g_ascii_strcasecmp((gchar *) node->name, XML_BOOL) == 0) {
			osso_email_xml_unpack_bool(node, doc, XML_BOOL, &(retr_msg->flag));

		}
		node = node->next;

	}
	return OSSO_PACK_OK;
}

#if CL_USE

/**
  This function is used to pack the 'IapInfo structure data' received
  into a xml 'doc' with identifier 'name'.

   @param  node   Xml Node pointer to pack the data
   @param  doc    Xml Doc  pointer to pack the data
   @param  name   Tag Identifier of the IapInfo struct 
   @param  data   IapInfo structs data to be packed in xml format
   @return OSSO_PACK_OK on success (or) OSSO_PACK_INVALID for invalid 
           parameter (or) OSSO_PACK_ERROR for other kind of errors
*/

osso_pack_return_t
osso_email_xml_pack_iapinfo(xmlNodePtr node, xmlDocPtr doc, void *nodeType, void *iapinfo)
{
	xmlNodePtr root = NULL, child = NULL;
	if ((doc == NULL) || (iapinfo == NULL) || (nodeType == NULL)) {
		osso_log(LOG_INFO, "Invalid input params in %s", __FUNCTION__);
		return OSSO_PACK_INVALID;
	}
	NEW_NODE(node, root, child, nodeType);
	memset(&tmpIapInfo, '\0', sizeof(tmpIapInfo));
	tmpIapInfo = *(IapInfo *) iapinfo;
	osso_email_xml_pack_struct(node, doc, NULL, IapInfoDBusData);
	return OSSO_PACK_OK;
}

/**
  This function is used to unpack the 'IapInfo struct  data'
  stored in a xml 'doc' with identifier 'name'

   @param  node   Xml Node pointer of the packed data
   @param  doc    Xml Doc  pointer of the packed data
   @param  name   Tag Identifier of the IapInfo struct data
                  in the xml document
   @param  data   IapInfo struct data will be stored in this pointer
   @return OSSO_PACK_OK on success (or) OSSO_PACK_INVALID for invalid 
           parameter (or) OSSO_PACK_ERROR for other kind of errors
*/
osso_pack_return_t
osso_email_xml_unpack_iapinfo(xmlNodePtr node, xmlDocPtr doc, gchar * nodeType, void *data)
{
	IapInfo *iapinfo = NULL;
	gint idx = 0;
	if (doc == NULL || node == NULL || data == NULL) {
		return OSSO_PACK_INVALID;
	}
	iapinfo = (IapInfo *) data;
	node = node->xmlChildrenNode;
	while (node != NULL) {
		if (node->type != XML_ELEMENT_NODE) {
			node = node->next;
			continue;
		}
		tmpIapInfo = *iapinfo;
		idx = 0;
		if (g_ascii_strcasecmp(node->name, IapInfoDBusData[idx++].name) == 0) {
			/*iap_name */
			osso_email_xml_unpack_str(node, doc, NULL, &iapinfo->iap_name);
		} else if (g_ascii_strcasecmp(node->name, IapInfoDBusData[idx++].name) == 0) {
			/*iap specific smtp serv */
			osso_email_xml_unpack_str(node, doc, NULL, &iapinfo->smtp_servname);
		} else if (g_ascii_strcasecmp(node->name, IapInfoDBusData[idx++].name) == 0) {
			/*iap specific smtp user name */
			osso_email_xml_unpack_str(node, doc, NULL, &iapinfo->smtp_username);
		} else if (g_ascii_strcasecmp(node->name, IapInfoDBusData[idx++].name) == 0) {
			/*iap specific smtp passwd */
			osso_email_xml_unpack_str(node, doc, NULL, &iapinfo->smtp_passwd);
		} else if (g_ascii_strcasecmp(node->name, IapInfoDBusData[idx++].name) == 0) {
			/*iap specific smtp port */
			osso_email_xml_unpack_ushort(node, doc, NULL, &iapinfo->smtp_port);
		} else if (g_ascii_strcasecmp(node->name, IapInfoDBusData[idx++].name) == 0) {
			/* iap specific smtp auth type */
			osso_email_xml_unpack_int(node, doc, NULL, &iapinfo->smtp_auth_type);
		} else if (g_ascii_strcasecmp(node->name, IapInfoDBusData[idx++].name) == 0) {
			/* iap specific smtp to use ssl or not  */
			osso_email_xml_unpack_bool(node, doc, NULL, &iapinfo->smtp_use_ssl);
		} else if (g_ascii_strcasecmp(node->name, IapInfoDBusData[idx++].name) == 0) {
			/* iap specific smtp to use tls  */
			osso_email_xml_unpack_bool(node, doc, NULL, &iapinfo->smtp_use_tls);
		} else if (g_ascii_strcasecmp(node->name, IapInfoDBusData[idx++].name) == 0) {
			/*iap status */
			osso_email_xml_unpack_int(node, doc, NULL, &iapinfo->iap_status);
		}
		node = node->next;
	}
	return OSSO_PACK_OK;
}

/**
  This function is used to pack the 'GSList IapInfo structure data' received
  into a xml 'doc' with identifier 'name'.

   @param  node   Xml Node pointer to pack the data
   @param  doc    Xml Doc  pointer to pack the data
   @param  name   Tag Identifier of the GSList IapInfo struct 
   @param  data   GSList of IapInfo structs data to be packed in xml format
   @return OSSO_PACK_OK on success (or) OSSO_PACK_INVALID for invalid 
           parameter (or) OSSO_PACK_ERROR for other kind of errors
*/

osso_pack_return_t
osso_email_xml_pack_iapinfo_list(xmlNodePtr node, xmlDocPtr doc, void *nodeType, void *data)
{
	xmlNodePtr root = NULL, child = NULL;
	GSList *iapinfo_list = NULL;
	GSList *cur = NULL;
	if ((doc == NULL) || (data) == NULL || (nodeType == NULL)) {
		return OSSO_PACK_INVALID;
	}
	iapinfo_list = (GSList *) data;
	NEW_NODE(node, root, child, nodeType);
	for (cur = iapinfo_list; cur != NULL; cur = cur->next) {
		if (cur->data == NULL) {
			continue;
		}
		osso_email_xml_pack_iapinfo(node, doc, XML_IAPINFO, cur->data);
	}
	return OSSO_PACK_OK;
}

/**
  This function is used to unpack the 'GSList ofIapInfo struct  data'
  stored in a xml 'doc' with identifier 'name'

   @param  node   Xml Node pointer of the packed data
   @param  doc    Xml Doc  pointer of the packed data
   @param  name   Tag Identifier of the GSList of IapInfo struct data
                  in the xml document
   @param  data   GSList of IapInfo struct data will be stored in this pointer
   @return OSSO_PACK_OK on success (or) OSSO_PACK_INVALID for invalid 
           parameter (or) OSSO_PACK_ERROR for other kind of errors
*/
osso_pack_return_t
osso_email_xml_unpack_iapinfo_list(xmlNodePtr node, xmlDocPtr doc, gchar * nodeType, void **data)
{
	GSList *iapInfo_list = NULL;
	IapInfo *iapinfo = NULL;
	if (doc == NULL || node == NULL || data == NULL) {
		return OSSO_PACK_INVALID;
	}
	for (node = node->xmlChildrenNode; node != NULL; node = node->next) {
		if (node->type == XML_ELEMENT_NODE) {
			if (g_ascii_strcasecmp(node->name, XML_IAPINFO) == 0) {
				iapinfo = g_new0(IapInfo, 1);
				if (iapinfo == NULL) {
					osso_log(LOG_ERR, "Memory alloc fail \n");
					return OSSO_PACK_ALLOC_FAIL;
				}
				osso_email_xml_unpack_iapinfo(node, doc, XML_IAPINFO, iapinfo);
				iapInfo_list = g_slist_append(iapInfo_list, iapinfo);
			}
		}
	}
	*data = iapInfo_list;
	return OSSO_PACK_OK;
}

#endif
