
/**
 * This file is part of osso-email-common
 *
 * Copyright (c) 2006 Nokia Corporation.
 * 
 * Contact: Dirk-Jan Binnema <dirk-jan.binnema@nokia.com>
 * 
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public License
 * version 2.1 as published by the Free Software Foundation.
 * 
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA
 * 02110-1301 USA
 */

#define DBUS_API_SUBJECT_TO_CHANGE

#ifndef DBUS_UTILS_H_
#define DBUS_UTILS_H_

#include "common.h"
#include "utils.h"
#include "callbacks.h"
#include "procmsg.h"
#include "procmime.h"
#include "folder.h"
#include "dbus_utils_def.h"
#include "dbus_utils_xml.h"
#include "base64.h"
#include "statesave.h"

#if CL_USE
#include "cl.h"
#endif

#include <glib.h>

/* Dbus includes */
#include <dbus/dbus.h>

typedef struct _EmailDBusFuncs EmailDBusFuncs;

struct _EmailDBusFuncs {
	gchar *hname;
	 gint(*msghandler) (const gchar * recv_str, gchar ** reply);
};


#define CHECK_STATUS(condition, message,  iffail)   \
{                                                   \
     if(condition)  {                               \
        osso_log(LOG_INFO, " %s\n",message);        \
        iffail;                                     \
     }                                              \
}

#define GET_DATA_FROM_RECV_STRING( recvstr, datastr, iffail) \
{                                                            \
    if( recvstr == NULL ) {                                  \
       osso_log(LOG_INFO, "data received  is NULL \n");      \
       iffail;                                               \
    }                                                        \
    datastr = strstr(recvstr, seperator);                    \
    if( (datastr == NULL) || (++datastr == NULL) ) {         \
      iffail;                                                \
    }                                                        \
}

/**
   Pack  function for Message Info struct

   @param  msginfo  MsgInfo struct to be packed in Xml format
   @param  message  Packed data to be stored in message
   @return OSSO_PACK_OK on success (or) OSSO_PACK_INVALID for invalid 
           parameter (or) OSSO_PACK_ERROR for other kind of errors
*/

osso_pack_return_t osso_email_utils_pack_msgInfo(MsgInfo * msginfo, gchar ** message);

/**
   Unpack  function for Message Info struct

   @param  msginfo  MsgInfo struct to be unpacked from Xml format
   @param  message  Data which is to be unpacked
   @return OSSO_PACK_OK on success (or) OSSO_PACK_INVALID for invalid 
           parameter (or) OSSO_PACK_ERROR for other kind of errors
*/

osso_pack_return_t osso_email_utils_unpack_msgInfo(gchar * message, MsgInfo * msginfo);

/**
   Pack  function for Message Header struct

   @param  msgheader  MsgHeader struct to be packed in Xml format
   @param  message  Packed data to be stored in message
   @return OSSO_PACK_OK on success (or) OSSO_PACK_INVALID for invalid 
           parameter (or) OSSO_PACK_ERROR for other kind of errors
*/

osso_pack_return_t osso_email_utils_pack_msgHeader(MsgHeader * msgheader, gchar ** message);

/**
   Unpack  function for Message Header struct

   @param  msgheader  MsgHeader struct to be unpacked from Xml format
   @param  message    Data which is to be unpacked
   @return OSSO_PACK_OK on success (or) OSSO_PACK_INVALID for invalid 
           parameter (or) OSSO_PACK_ERROR for other kind of errors
*/

osso_pack_return_t osso_email_utils_unpack_msgHeader(const gchar * message, MsgHeader * msgheader);

/**
   Pack  function for PrefsAccount struct

   @param  prefsaccount  PrefsAccount struct to be packed in Xml format
   @param  message  Packed data to be stored in message
   @return OSSO_PACK_OK on success (or) OSSO_PACK_INVALID for invalid 
           parameter (or) OSSO_PACK_ERROR for other kind of errors
*/

osso_pack_return_t
osso_email_utils_pack_prefsAccount(PrefsAccount * prefsaccount, gchar ** message);


/**
   Unpack  function for PrefsAccount struct

   @param  prefsaccount PrefsAccount  struct to be unpacked from Xml format
   @param  message    Data which is to be unpacked
   @return OSSO_PACK_OK on success (or) OSSO_PACK_INVALID for invalid 
           parameter (or) OSSO_PACK_ERROR for other kind of errors
*/

osso_pack_return_t
osso_email_utils_unpack_prefsAccount(gchar * message, PrefsAccount * prefsaccount);

/**
   Pack  function for MimeInfo struct

   @param  mimeinfo  MimeInfo struct to be packed in Xml format
   @return OSSO_PACK_OK on success (or) OSSO_PACK_INVALID for invalid 
           parameter (or) OSSO_PACK_ERROR for other kind of errors
*/

gchar *osso_email_utils_pack_mimeInfo(MimeInfo * mimeinfo);

/**
   Unpack  function for MimeInfo struct

   @param  message    Data which is to be unpacked
   @return OSSO_PACK_OK on success (or) OSSO_PACK_INVALID for invalid 
           parameter (or) OSSO_PACK_ERROR for other kind of errors
*/

MimeInfo *osso_email_utils_unpack_mimeInfo(gchar * message);

/**
   Pack  function for Folder struct

   @param  folder  Folder struct to be packed in Xml format
   @param  message  Packed data to be stored in message
   @return OSSO_PACK_OK on success (or) OSSO_PACK_INVALID for invalid 
           parameter (or) OSSO_PACK_ERROR for other kind of errors
*/

osso_pack_return_t osso_email_utils_pack_Folder(Folder * folder, gchar ** message);


/**
   Unpack  function for Folder struct

   @param  folder   Folder  struct to be unpacked from Xml format
   @param  message  Data which is to be unpacked
   @return OSSO_PACK_OK on success (or) OSSO_PACK_INVALID for invalid 
           parameter (or) OSSO_PACK_ERROR for other kind of errors
*/

osso_pack_return_t osso_email_utils_unpack_Folder(gchar * message, Folder * folder);

/**
   Pack  function for FolderAttachInfo struct

   @param  folder_attach  FolderAttachInfo struct to be packed in Xml format
   @param  message  Packed data to be stored in message
   @return OSSO_PACK_OK on success (or) OSSO_PACK_INVALID for invalid 
           parameter (or) OSSO_PACK_ERROR for other kind of errors
*/
osso_pack_return_t
osso_email_utils_pack_folder_attach(FolderAttachInfo * folder_attach, gchar ** list_string);
/**
   Unpack  function for FolderAttachInfo struct

   @param  folder_attach FolderAttachInfo struct to be unpacked from Xml format
   @param  message  Data which is to be unpacked
   @return OSSO_PACK_OK on success (or) OSSO_PACK_INVALID for invalid 
           parameter (or) OSSO_PACK_ERROR for other kind of errors
*/
osso_pack_return_t
osso_email_utils_unpack_folder_attach(gchar * message, FolderAttachInfo * folder_attach);
/**
   Pack  function for GSList of  String

   @param  attach_list  GSList of Strings to be packed in Xml format
   @param  list_str  Packed data to be stored in message
   @return OSSO_PACK_OK on success (or) OSSO_PACK_INVALID for invalid 
           parameter (or) OSSO_PACK_ERROR for other kind of errors
*/
osso_pack_return_t
osso_email_utils_pack_from_gslist_to_string(GSList * attach_list, gchar ** list_str);

/**
   Pack  function for GSList of  MsgInfo struct

   @param  msginfo_list  GSList of MsgInfo struct to be packed in Xml format
   @param  list  Packed data to be stored in message
   @return OSSO_PACK_OK on success (or) OSSO_PACK_INVALID for invalid 
           parameter (or) OSSO_PACK_ERROR for other kind of errors
*/

osso_pack_return_t
osso_email_utils_pack_from_gslist_to_msginfo(GSList * msginfo_list, gchar ** list);

/**
   Pack  function for GSList of  MsgHeader struct

   @param  msgheader_list GSList of MsgHeader struct to be packed in Xml format
   @param  list  Packed data to be stored in message
   @return OSSO_PACK_OK on success (or) OSSO_PACK_INVALID for invalid 
           parameter (or) OSSO_PACK_ERROR for other kind of errors
*/

osso_pack_return_t
osso_email_utils_pack_from_gslist_to_msgheader(GSList * msgheader_list, gchar ** list);

/**
   Pack  function for GSList of Message Ids

   @param  string_list GSList of Message Ids  to be packed in Xml format
   @param  list  Packed data to be stored in message
   @return OSSO_PACK_OK on success (or) OSSO_PACK_INVALID for invalid 
           parameter (or) OSSO_PACK_ERROR for other kind of errors
*/

osso_pack_return_t osso_email_utils_pack_msgid_to_gslist(GSList * string_list, gchar ** list);

/**
   Pack  function for GSList of Strings

   @param  string_list GSList of Strings to be packed in Xml format
   @param  list  Packed data to be stored in message
   @return OSSO_PACK_OK on success (or) OSSO_PACK_INVALID for invalid 
           parameter (or) OSSO_PACK_ERROR for other kind of errors
*/

osso_pack_return_t osso_email_utils_pack_string_to_gslist(GSList * string_list, gchar ** list);

/**
   Pack  function for Message headers in Folder populate folder

   @param  msgheader_list GSList of MsgHeader Struct to be packed in Xml format
   @param  list  Packed data to be stored in message
   @return OSSO_PACK_OK on success (or) OSSO_PACK_INVALID for invalid 
           parameter (or) OSSO_PACK_ERROR for other kind of errors
*/

osso_pack_return_t
osso_email_utils_pack_populate_folder_to_gslist(GSList * msgheader_list, gchar ** list);

/**
   Pack  function for GSList of Message Header List

   @param  msgheader_list GSList of MsgHeader Struct to be packed in Xml format
   @param  list  Packed data to be stored in message
   @return OSSO_PACK_OK on success (or) OSSO_PACK_INVALID for invalid 
           parameter (or) OSSO_PACK_ERROR for other kind of errors
*/

osso_pack_return_t
osso_email_utils_pack_msgheader_to_gslist(GSList * msgheader_list, gchar ** list);

/**
   Pack  function for GSList of Prefs Account

   @param  account_list GSList of PrefsAccount Struct to be packed in Xml format
   @param  list  Packed data to be stored in message
   @return OSSO_PACK_OK on success (or) OSSO_PACK_INVALID for invalid 
           parameter (or) OSSO_PACK_ERROR for other kind of errors
*/

osso_pack_return_t
osso_email_utils_pack_prefs_account_to_gslist(GSList * account_list, gchar ** list);

/**
   Pack  function for GSList of Folder Struct

   @param  folder_list GSList of Folder Struct to be packed in Xml format
   @param  list  Packed data to be stored in message
   @return OSSO_PACK_OK on success (or) OSSO_PACK_INVALID for invalid 
           parameter (or) OSSO_PACK_ERROR for other kind of errors
*/

osso_pack_return_t osso_email_utils_pack_folder_to_gslist(GSList * folder_list, gchar ** list);

/**
   Pack  function for GSList of Folder Attach Struct

   @param  attach_list GSList of Folder Attach Struct to be packed in Xml format
   @param  list  Packed data to be stored in message
   @return OSSO_PACK_OK on success (or) OSSO_PACK_INVALID for invalid 
           parameter (or) OSSO_PACK_ERROR for other kind of errors
*/

osso_pack_return_t
osso_email_utils_pack_folder_attach_to_gslist(GSList * attach_list, gchar ** list);

/**
   Pack  function for GSList of AttachInfo struct
   @param  attachinfo_list GSList of AttachInfo Struct to be packed
           in Xml format
   @param  list  Packed data to be stored in message
   @return OSSO_PACK_OK on success (or) OSSO_PACK_INVALID for invalid 
           parameter (or) OSSO_PACK_ERROR for other kind of errors
*/
osso_pack_return_t
osso_email_utils_pack_from_gslist_to_attachinfo(GSList * attachinfo_list, gchar ** list);

/**
   Unpack  function GSList of AttachInfo struct

   @param  list GSList of AttachInfo  struct to be unpacked from Xml format
   @param  data  Data which is to be unpacked
   @return OSSO_PACK_OK on success (or) OSSO_PACK_INVALID for invalid 
           parameter (or) OSSO_PACK_ERROR for other kind of errors
*/

osso_pack_return_t osso_email_utils_unpack_attachinfo_in_gslist(gchar * data, GSList ** list);

/**
   Unpack  function GSList of Message Info struct

   @param  list GSList of Message Info  struct to be unpacked from Xml format
   @param  data  Data which is to be unpacked
   @return OSSO_PACK_OK on success (or) OSSO_PACK_INVALID for invalid 
           parameter (or) OSSO_PACK_ERROR for other kind of errors
*/
osso_pack_return_t osso_email_utils_unpack_msginfo_in_gslist(gchar * data, GSList ** list);

/**
   Unpack  function GSList of Message Header struct

   @param  list GSList of Message Header  struct to be unpacked from Xml format
   @param  data  Data which is to be unpacked
   @return OSSO_PACK_OK on success (or) OSSO_PACK_INVALID for invalid 
           parameter (or) OSSO_PACK_ERROR for other kind of errors
*/

osso_pack_return_t osso_email_utils_unpack_msgheader_in_gslist(gchar * data, GSList ** list);

/**
   Unpack  function GSList of Prefs Account struct

   @param  list GSList of Prefs Account struct to be unpacked from Xml format
   @param  data  Data which is to be unpacked
   @return OSSO_PACK_OK on success (or) OSSO_PACK_INVALID for invalid 
           parameter (or) OSSO_PACK_ERROR for other kind of errors
*/

osso_pack_return_t osso_email_utils_unpack_prefsaccount_in_gslist(gchar * data, GSList ** list);

/**
   Unpack  function GSList of Folder struct

   @param  list GSList of Folder struct to be unpacked from Xml format
   @param  data  Data which is to be unpacked
   @return OSSO_PACK_OK on success (or) OSSO_PACK_INVALID for invalid 
           parameter (or) OSSO_PACK_ERROR for other kind of errors
*/

osso_pack_return_t osso_email_utils_unpack_folder_in_gslist(gchar * data, GSList ** list);

/**
   Unpack  function GSList of Folder Attach struct

   @param  list GSList of Folder Attach struct to be unpacked from Xml format
   @param  data  Data which is to be unpacked
   @return OSSO_PACK_OK on success (or) OSSO_PACK_INVALID for invalid 
           parameter (or) OSSO_PACK_ERROR for other kind of errors
*/

osso_pack_return_t osso_email_utils_unpack_folderattach_in_gslist(gchar * data, GSList ** list);


/**
   Unpack  function GSList of AttachInfo struct

   @param  list GSList of AttachInfo struct to be unpacked from Xml format
   @param  data  Data which is to be unpacked
   @return OSSO_PACK_OK on success (or) OSSO_PACK_INVALID for invalid 
           parameter (or) OSSO_PACK_ERROR for other kind of errors
*/

osso_pack_return_t osso_email_utils_unpack_attachinfo_to_gslist(gchar * data, GSList ** list);

/**
   Unpack  function GSList of Message Headers contains in the packed data

   @param  list GSList of MsgHeader struct to be unpacked from Xml format
   @param  data  Data which is to be unpacked
   @return OSSO_PACK_OK on success (or) OSSO_PACK_INVALID for invalid 
           parameter (or) OSSO_PACK_ERROR for other kind of errors
*/

osso_pack_return_t osso_email_utils_unpack_populate_folder_to_gslist(gchar * data, GSList ** list);

/**
   Unpack  function GSList of Message ID list in the packed data

   @param  list GSList of Message Ids to be unpacked from Xml format
   @param  data  Data which is to be unpacked
   @return OSSO_PACK_OK on success (or) OSSO_PACK_INVALID for invalid 
           parameter (or) OSSO_PACK_ERROR for other kind of errors
*/

osso_pack_return_t osso_email_utils_unpack_msgid_list_in_gslist(gchar * data, GSList ** list);


/**
   Unpack  function GSList of Strings in the packed data

   @param  list GSList of Strings to be unpacked from Xml format
   @param  data  Data which is to be unpacked
   @return OSSO_PACK_OK on success (or) OSSO_PACK_INVALID for invalid 
           parameter (or) OSSO_PACK_ERROR for other kind of errors
*/

osso_pack_return_t osso_email_utils_unpack_string_in_gslist(gchar * data, GSList ** list);

/**
   Function to free Folder Attach Struct

   @param  folder_attach_list GSList of Folder Attach struct
   @return OSSO_PACK_OK on success (or) OSSO_PACK_INVALID for invalid 
           parameter (or) OSSO_PACK_ERROR for other kind of errors
*/

osso_pack_return_t osso_email_utils_folder_attach_free(GSList * folder_attach_list);

/**
   Function to concat two strings seperated by separator(:) 

   @param  str1   First string to concatanate
   @param  str2   Second string to concatanate
   @return NULL on failure or concated string on success 
*/
gchar *osso_email_utils_concat_strings(const gchar * str1, const gchar * str2);

/**
   Function to concat String and integer seperated by separator(:) 

   @param  str   String  contains Function Identifier 
   @param  val   integer value to concat to the string 
   @return NULL on failure or concated string on success 
*/

gchar *osso_email_utils_concat_str_int(const gchar * str1, const gint val);

/**
   Function to free the mimeinfo allocated during xml packing. 

   @param  mimeinfo structure to be freed
   @return OSSO_PACK_OK on successful free of mimeinfo (or) 
    OSSO_PACK_INVALID for invalid input. 
*/

osso_pack_return_t osso_email_utils_mimeinfo_free_all(MimeInfo * mimeinfo);

/**
   Function to convert the given string to base64 format 

   @param  inbuf  string which is to converted in base64 format 
   @param  convdata converted data for given input string 
   @return OSSO_PACK_OK on success (or) OSSO_PACK_INVALID for invalid 
           parameter (or) OSSO_PACK_ERROR for other kind of errors
*/

osso_pack_return_t osso_email_utils_pack_str_to_base64(const guchar * inbuf, gchar ** convdata);

/**
   Function to convert the given base64 format string to normal 

   @param  inbuf  base64 string which is to converted to normal string 
   @param  convdata converted string for given input base64 string 
   @return OSSO_PACK_OK on success (or) OSSO_PACK_INVALID for invalid 
           parameter (or) OSSO_PACK_ERROR for other kind of errors
*/
osso_pack_return_t
osso_email_utils_unpack_base64_to_str(const gchar * base64data, gchar ** convdata);

#if CL_USE
/**
   Pack  function for GSList of iapinfo struct

   @param  list GSList of iapinfo struct packed from Xml format
   @param  data populated with xml packed data 
   @return OSSO_PACK_OK on success (or) OSSO_PACK_INVALID for invalid 
           parameter (or) OSSO_PACK_ERROR for other kind of errors
*/
osso_pack_return_t osso_email_utils_pack_iapinfo_list(GSList * iapinfo_list, gchar ** list);

/**
   Unpack  function GSList of iapinfo struct

   @param  list GSList of Message Ids to be unpacked from Xml format
   @param  data  contains the xml string which is to be unpacked
   @return OSSO_PACK_OK on success (or) OSSO_PACK_INVALID for invalid 
           parameter (or) OSSO_PACK_ERROR for other kind of errors
*/

osso_pack_return_t osso_email_utils_unpack_iapinfo_list(gchar * data, GSList ** list);
#endif

/**
   This Method is called to write persistent data in to a file 
   @param  perInfo contains the Persistent information to be saved 
   @return on successful operation returns OSSO_PACK_OK. Otherwise 
    returns errorcode
*/

osso_pack_return_t dbus_utils_persistentInfo_write_to_file(EmailPersistentInfo * perInfo);

/**
   This Method is called to write persistent data in to a file 
   @param  perInfo persistent data updated in this struct 
   @return on successful operation returns OSSO_PACK_OK. Otherwise 
    returns errorcode
*/

osso_pack_return_t dbus_utils_persistentInfo_read_from_file(EmailPersistentInfo * perInfo);

#endif
