
/**
 * This file is part of osso-email-common
 *
 * Copyright (c) 2006 Nokia Corporation.
 * 
 * Contact: Dirk-Jan Binnema <dirk-jan.binnema@nokia.com>
 * 
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public License
 * version 2.1 as published by the Free Software Foundation.
 * 
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA
 * 02110-1301 USA
 */

#include <stdio.h>
#include <glib.h>

#include "dbus_utils.h"
#include "dbus_common_utils.h"
#include "libosso.h"
#include "defs.h"

/**
   Pack  function for Message Info struct

   @param  msginfo  MsgInfo struct to be packed in Xml format
   @param  message  Packed data to be stored in message
   @return OSSO_PACK_OK on success (or) OSSO_PACK_INVALID for invalid 
           parameter (or) OSSO_PACK_ERROR for other kind of errors
*/

osso_pack_return_t osso_email_utils_pack_msgInfo(MsgInfo * msginfo, gchar ** message)
{
	xmlDocPtr doc = NULL;
	xmlNodePtr root = NULL;
	xmlChar *xmlbuff = NULL;
	int buffersize = 0;

	CHECK_STATUS((msginfo == NULL), "MsgInfo arg null", return OSSO_PACK_INVALID);
	CHECK_STATUS((message == NULL), "pack str arg null", return OSSO_PACK_INVALID);

	DBUS_XML_PACK_INIT;
	osso_email_xml_pack_msginfo(root, doc, XML_MSG_INFO, msginfo);
	xmlDocDumpFormatMemory(doc, &xmlbuff, &buffersize, 1);
	DBUS_XML_PACK_UN_INIT;

	*message = (gchar *) xmlbuff;

	return OSSO_PACK_OK;
}


/**
   Unpack  function for Message Info struct

   @param  msginfo  MsgInfo struct to be unpacked from Xml format
   @param  message  Data which is to be unpacked
   @return OSSO_PACK_OK on success (or) OSSO_PACK_INVALID for invalid 
           parameter (or) OSSO_PACK_ERROR for other kind of errors
*/

osso_pack_return_t osso_email_utils_unpack_msgInfo(gchar * message, MsgInfo * msginfo)
{
	xmlNodePtr cur = NULL, node = NULL;
	xmlDocPtr doc = NULL;

	CHECK_STATUS((msginfo == NULL), "MsgInfo arg null", return OSSO_PACK_INVALID);
	CHECK_STATUS((message == NULL), "pack str arg null", return OSSO_PACK_INVALID);

	DBUS_XML_UNPACK_INIT(message, return OSSO_PACK_ERROR);

	for (node = cur->xmlChildrenNode; node != NULL; node = node->next) {
		if (node->type == XML_ELEMENT_NODE) {
			if (g_ascii_strcasecmp(node->name, XML_MSG_INFO) == 0) {
				osso_email_xml_unpack_msginfo(node, doc, XML_MSG_INFO, msginfo);
				break;
			}
		}
	}

	DBUS_XML_UNPACK_UN_INIT;
	return OSSO_PACK_OK;
}


/**
   Pack  function for Message Header struct

   @param  msgheader  MsgHeader struct to be packed in Xml format
   @param  message  Packed data to be stored in message
   @return OSSO_PACK_OK on success (or) OSSO_PACK_INVALID for invalid 
           parameter (or) OSSO_PACK_ERROR for other kind of errors
*/

osso_pack_return_t osso_email_utils_pack_msgHeader(MsgHeader * msgheader, gchar ** message)
{
	xmlDocPtr doc = NULL;
	xmlNodePtr root = NULL;
	xmlChar *xmlbuff = NULL;
	int buffersize = 0;

	CHECK_STATUS((msgheader == NULL), "Msghdr arg null", return OSSO_PACK_INVALID);
	CHECK_STATUS((message == NULL), "pack str arg null", return OSSO_PACK_INVALID);

	DBUS_XML_PACK_INIT;
	osso_email_xml_pack_msgheader(root, doc, XML_MSG_HEADER, msgheader);
	xmlDocDumpFormatMemory(doc, &xmlbuff, &buffersize, 1);
	DBUS_XML_PACK_UN_INIT;

	*message = (gchar *) xmlbuff;

	return OSSO_PACK_OK;
}


/**
   Unpack  function for Message Header struct

   @param  msgheader  MsgHeader struct to be unpacked from Xml format
   @param  message    Data which is to be unpacked
   @return OSSO_PACK_OK on success (or) OSSO_PACK_INVALID for invalid 
           parameter (or) OSSO_PACK_ERROR for other kind of errors
*/

osso_pack_return_t osso_email_utils_unpack_msgHeader(const gchar * message, MsgHeader * msgheader)
{
	xmlNodePtr cur = NULL, node = NULL;
	xmlDocPtr doc = NULL;

	CHECK_STATUS((msgheader == NULL), "Msghdr arg null", return OSSO_PACK_INVALID);
	CHECK_STATUS((message == NULL), "pack str arg null", return OSSO_PACK_INVALID);

	DBUS_XML_UNPACK_INIT(message, return OSSO_PACK_ERROR);

	for (node = cur->xmlChildrenNode; node != NULL; node = node->next) {
		if (node->type == XML_ELEMENT_NODE) {
			if (g_ascii_strcasecmp(node->name, XML_MSG_HEADER) == 0) {
				osso_email_xml_unpack_msgheader(node, doc,
								XML_MSG_HEADER, msgheader);
				break;
			}
		}
	}

	DBUS_XML_UNPACK_UN_INIT;
	return OSSO_PACK_OK;
}

/**
   Pack  function for PrefsAccount struct

   @param  prefsaccount  PrefsAccount struct to be packed in Xml format
   @param  message  Packed data to be stored in message
   @return OSSO_PACK_OK on success (or) OSSO_PACK_INVALID for invalid 
           parameter (or) OSSO_PACK_ERROR for other kind of errors
*/

osso_pack_return_t osso_email_utils_pack_prefsAccount(PrefsAccount * prefsaccount, gchar ** message)
{
	xmlDocPtr doc = NULL;
	xmlNodePtr root = NULL;
	xmlChar *xmlbuff = NULL;
	int buffersize = 0;

	CHECK_STATUS((prefsaccount == NULL), "PrefsAccount arg null", return OSSO_PACK_INVALID);
	CHECK_STATUS((message == NULL), "pack str arg null", return OSSO_PACK_INVALID);

	DBUS_XML_PACK_INIT;
	osso_email_xml_pack_prefsaccount(root, doc, XML_PREFS_ACCOUNT, prefsaccount);
	xmlDocDumpFormatMemory(doc, &xmlbuff, &buffersize, 1);
	DBUS_XML_PACK_UN_INIT;

	*message = (gchar *) xmlbuff;
	return OSSO_PACK_OK;

}


/**
   Unpack  function for PrefsAccount struct

   @param  prefsaccount PrefsAccount  struct to be unpacked from Xml format
   @param  message    Data which is to be unpacked
   @return OSSO_PACK_OK on success (or) OSSO_PACK_INVALID for invalid 
           parameter (or) OSSO_PACK_ERROR for other kind of errors
*/

osso_pack_return_t
osso_email_utils_unpack_prefsAccount(gchar * message, PrefsAccount * prefsaccount)
{
	xmlNodePtr cur = NULL, node = NULL;
	xmlDocPtr doc = NULL;

	CHECK_STATUS((prefsaccount == NULL), "PrefsAccount arg null", return OSSO_PACK_INVALID);
	CHECK_STATUS((message == NULL), "pack str arg null", return OSSO_PACK_INVALID);

	DBUS_XML_UNPACK_INIT(message, return OSSO_PACK_ERROR);

	for (node = cur->xmlChildrenNode; node != NULL; node = node->next) {
		if (node->type == XML_ELEMENT_NODE) {
			if (g_ascii_strcasecmp(node->name, XML_PREFS_ACCOUNT) == 0) {
				osso_email_xml_unpack_prefsaccount(node, doc,
								   XML_PREFS_ACCOUNT, prefsaccount);
				break;
			}
		}
	}
	DBUS_XML_UNPACK_UN_INIT;
	return OSSO_PACK_OK;
}

/**
   Pack  function for MimeInfo struct

   @param  mimeinfo  MimeInfo struct to be packed in Xml format
   @return OSSO_PACK_OK on success (or) OSSO_PACK_INVALID for invalid 
           parameter (or) OSSO_PACK_ERROR for other kind of errors
*/

gchar *osso_email_utils_pack_mimeInfo(MimeInfo * mimeinfo)
{
	xmlDocPtr doc = NULL;
	xmlNodePtr root = NULL;
	xmlChar *xmlbuff = NULL;
	int buffersize = 0;

	CHECK_STATUS((mimeinfo == NULL), "MimeInfo arg null", return NULL);
	DBUS_XML_PACK_INIT;
	osso_email_xml_pack_mimeinfo(root, doc, XML_MIME_INFO, mimeinfo);
	xmlDocDumpFormatMemory(doc, &xmlbuff, &buffersize, 1);
	DBUS_XML_PACK_UN_INIT;

	return (gchar *) xmlbuff;
}


/**
   Unpack  function for MimeInfo struct

   @param  message    Data which is to be unpacked
   @return OSSO_PACK_OK on success (or) OSSO_PACK_INVALID for invalid 
           parameter (or) OSSO_PACK_ERROR for other kind of errors
*/

MimeInfo *osso_email_utils_unpack_mimeInfo(gchar * message)
{
	MimeInfo *mimeinfo = NULL;
	xmlNodePtr cur = NULL, node = NULL;
	xmlDocPtr doc = NULL;

	CHECK_STATUS((message == NULL), "message arg null", return NULL);

	mimeinfo = g_new0(MimeInfo, 1);
	if (mimeinfo == NULL) {
		osso_log(LOG_ERR, "MimeInfo alloc fail");
		return NULL;
	}

	DBUS_XML_UNPACK_INIT(message, return NULL);

	for (node = cur->xmlChildrenNode; node != NULL; node = node->next) {
		if (node->type == XML_ELEMENT_NODE) {
			if (g_ascii_strcasecmp((gchar *) node->name, XML_MIME_INFO) == 0) {
				osso_email_xml_unpack_mimeinfo(node, doc, XML_MIME_INFO, mimeinfo);
				break;
			}
		}
	}
	DBUS_XML_UNPACK_UN_INIT;
	return mimeinfo;

}

/**
   Pack  function for Folder struct

   @param  folder  Folder struct to be packed in Xml format
   @param  message  Packed data to be stored in message
   @return OSSO_PACK_OK on success (or) OSSO_PACK_INVALID for invalid 
           parameter (or) OSSO_PACK_ERROR for other kind of errors
*/

osso_pack_return_t osso_email_utils_pack_Folder(Folder * folder, gchar ** message)
{
	xmlDocPtr doc = NULL;
	xmlNodePtr root = NULL;
	xmlChar *xmlbuff = NULL;
	int buffersize = 0;

	CHECK_STATUS((folder == NULL), "folder arg null", return OSSO_PACK_INVALID);
	CHECK_STATUS((message == NULL), "pack str arg null", return OSSO_PACK_INVALID);

	DBUS_XML_PACK_INIT;
	osso_email_xml_pack_folder(root, doc, XML_FOLDER, folder);
	xmlDocDumpFormatMemory(doc, &xmlbuff, &buffersize, 1);
	DBUS_XML_PACK_UN_INIT;

	*message = (gchar *) xmlbuff;

	return OSSO_PACK_OK;
}

/**
   Unpack  function for Folder struct

   @param  folder   Folder  struct to be unpacked from Xml format
   @param  message  Data which is to be unpacked
   @return OSSO_PACK_OK on success (or) OSSO_PACK_INVALID for invalid 
           parameter (or) OSSO_PACK_ERROR for other kind of errors
*/

osso_pack_return_t osso_email_utils_unpack_Folder(gchar * message, Folder * folder)
{
	xmlNodePtr cur = NULL, node = NULL;
	xmlDocPtr doc = NULL;

	CHECK_STATUS((folder == NULL), "folder arg null", return OSSO_PACK_INVALID);
	CHECK_STATUS((message == NULL), "pack str arg null", return OSSO_PACK_INVALID);

	DBUS_XML_UNPACK_INIT(message, return OSSO_PACK_ERROR);

	for (node = cur->xmlChildrenNode; node != NULL; node = node->next) {
		if (node->type == XML_ELEMENT_NODE) {
			if (g_ascii_strcasecmp((gchar *) node->name, XML_FOLDER) == 0) {
				osso_email_xml_unpack_folder(node, doc, XML_FOLDER, folder);
				break;
			}
		}
	}
	DBUS_XML_UNPACK_UN_INIT;

	return OSSO_PACK_OK;
}


/**
   Pack  function for FolderAttachInfo struct

   @param  folder_attach  FolderAttachInfo struct to be packed in Xml format
   @param  message  Packed data to be stored in message
   @return OSSO_PACK_OK on success (or) OSSO_PACK_INVALID for invalid 
           parameter (or) OSSO_PACK_ERROR for other kind of errors
*/
osso_pack_return_t
osso_email_utils_pack_folder_attach(FolderAttachInfo * folder_attach, gchar ** list_string)
{
	xmlDocPtr doc = NULL;
	xmlNodePtr root = NULL;
	xmlChar *xmlbuff = NULL;
	int buffersize = 0;

	CHECK_STATUS((folder_attach == NULL), "folder_attach arg null", return OSSO_PACK_INVALID);
	CHECK_STATUS((list_string == NULL), "pack str arg null", return OSSO_PACK_INVALID);

	DBUS_XML_PACK_INIT;
	osso_email_xml_pack_folder_attachinfo(root, doc, XML_FOLDER_ATTACHINFO, folder_attach);
	xmlDocDumpFormatMemory(doc, &xmlbuff, &buffersize, 1);
	DBUS_XML_PACK_UN_INIT;

	*list_string = (gchar *) xmlbuff;
	return OSSO_PACK_OK;
}


/**
   Unpack  function for FolderAttachInfo struct

   @param  folder_attach FolderAttachInfo struct to be unpacked from Xml format
   @param  message  Data which is to be unpacked
   @return OSSO_PACK_OK on success (or) OSSO_PACK_INVALID for invalid 
           parameter (or) OSSO_PACK_ERROR for other kind of errors
*/
osso_pack_return_t
osso_email_utils_unpack_folder_attach(gchar * message, FolderAttachInfo * folder_attach)
{
	xmlNodePtr cur = NULL, node = NULL;
	xmlDocPtr doc = NULL;

	CHECK_STATUS((folder_attach == NULL), "folder_attach arg null", return OSSO_PACK_INVALID);
	CHECK_STATUS((message == NULL), "pack string arg null", return OSSO_PACK_INVALID);

	DBUS_XML_UNPACK_INIT(message, return OSSO_PACK_ERROR);

	for (node = cur->xmlChildrenNode; node != NULL; node = node->next) {
		if (node->type == XML_ELEMENT_NODE) {
			if (g_ascii_strcasecmp((gchar *) node->name, XML_FOLDER_ATTACHINFO) == 0) {
				osso_email_xml_unpack_folder_attachinfo(node, doc,
									XML_FOLDER_ATTACHINFO,
									folder_attach);
				break;
			}
		}
	}
	DBUS_XML_UNPACK_UN_INIT;

	return OSSO_PACK_OK;
}

/**
   Pack  function for GSList of  String

   @param  attach_list  GSList of Strings to be packed in Xml format
   @param  list_str  Packed data to be stored in message
   @return OSSO_PACK_OK on success (or) OSSO_PACK_INVALID for invalid 
           parameter (or) OSSO_PACK_ERROR for other kind of errors
*/
osso_pack_return_t
osso_email_utils_pack_from_gslist_to_string(GSList * attach_list, gchar ** list_str)
{
	xmlDocPtr doc = NULL;
	xmlNodePtr root = NULL;
	xmlChar *xmlbuff = NULL;
	int buffersize = 0;

	CHECK_STATUS((attach_list == NULL), "attach_list arg null", return OSSO_PACK_INVALID);
	CHECK_STATUS((list_str == NULL), "pack str arg null", return OSSO_PACK_INVALID);

	DBUS_XML_PACK_INIT;
	osso_email_xml_pack_string_list(root, doc, XML_STRING_LST, attach_list);
	xmlDocDumpFormatMemory(doc, &xmlbuff, &buffersize, 1);
	DBUS_XML_PACK_UN_INIT;

	*list_str = (gchar *) xmlbuff;
	return OSSO_PACK_OK;

}

/**
   Pack  function for GSList of  MsgInfo struct

   @param  msginfo_list  GSList of MsgInfo struct to be packed in Xml format
   @param  list  Packed data to be stored in message
   @return OSSO_PACK_OK on success (or) OSSO_PACK_INVALID for invalid 
           parameter (or) OSSO_PACK_ERROR for other kind of errors
*/
osso_pack_return_t
osso_email_utils_pack_from_gslist_to_msginfo(GSList * msginfo_list, gchar ** list)
{
	xmlDocPtr doc = NULL;
	xmlNodePtr root = NULL;
	xmlChar *xmlbuff = NULL;
	int buffersize = 0;

	CHECK_STATUS((msginfo_list == NULL), "msginfo_list arg null", return OSSO_PACK_INVALID);
	CHECK_STATUS((list == NULL), "pack str arg null", return OSSO_PACK_INVALID);

	DBUS_XML_PACK_INIT;
	osso_email_xml_pack_msginfo_list(root, doc, XML_MSG_INFO_LST, msginfo_list);
	xmlDocDumpFormatMemory(doc, &xmlbuff, &buffersize, 1);
	DBUS_XML_PACK_UN_INIT;

	*list = (gchar *) xmlbuff;
	return OSSO_PACK_OK;
}


/**
   Pack  function for GSList of  MsgHeader struct

   @param  msgheader_list GSList of MsgHeader struct to be packed in Xml format
   @param  list  Packed data to be stored in message
   @return OSSO_PACK_OK on success (or) OSSO_PACK_INVALID for invalid 
           parameter (or) OSSO_PACK_ERROR for other kind of errors
*/
osso_pack_return_t
osso_email_utils_pack_from_gslist_to_msgheader(GSList * msgheader_list, gchar ** list)
{
	xmlDocPtr doc = NULL;
	xmlNodePtr root = NULL;
	xmlChar *xmlbuff = NULL;
	int buffersize = 0;

	CHECK_STATUS((msgheader_list == NULL), "msgheader_list arg null", return OSSO_PACK_INVALID);
	CHECK_STATUS((list == NULL), "pack str arg null", return OSSO_PACK_INVALID);

	DBUS_XML_PACK_INIT;
	osso_email_xml_pack_msgheader_list(root, doc, XML_MSG_HEADER_LST, msgheader_list);
	xmlDocDumpFormatMemory(doc, &xmlbuff, &buffersize, 1);
	DBUS_XML_PACK_UN_INIT;

	*list = (gchar *) xmlbuff;
	return OSSO_PACK_OK;
}

/**
   Pack  function for GSList of Message Ids

   @param  string_list GSList of Message Ids  to be packed in Xml format
   @param  list  Packed data to be stored in message
   @return OSSO_PACK_OK on success (or) OSSO_PACK_INVALID for invalid 
           parameter (or) OSSO_PACK_ERROR for other kind of errors
*/

osso_pack_return_t osso_email_utils_pack_msgid_to_gslist(GSList * string_list, gchar ** list)
{
	xmlDocPtr doc = NULL;
	xmlNodePtr root = NULL;
	xmlChar *xmlbuff = NULL;
	int buffersize = 0;

	CHECK_STATUS((string_list == NULL), "GSList of Msg Id null", return OSSO_PACK_INVALID);
	CHECK_STATUS((list == NULL), "pack str arg null", return OSSO_PACK_INVALID);

	DBUS_XML_PACK_INIT;

	osso_email_xml_pack_msgid_list(root, doc, XML_MSG_ID_LST, string_list);

	xmlDocDumpFormatMemory(doc, &xmlbuff, &buffersize, 1);

	DBUS_XML_PACK_UN_INIT;

	*list = (gchar *) xmlbuff;
	return OSSO_PACK_OK;
}


/**
   Pack  function for GSList of Strings

   @param  string_list GSList of Strings to be packed in Xml format
   @param  list  Packed data to be stored in message
   @return OSSO_PACK_OK on success (or) OSSO_PACK_INVALID for invalid 
           parameter (or) OSSO_PACK_ERROR for other kind of errors
*/
osso_pack_return_t osso_email_utils_pack_string_to_gslist(GSList * string_list, gchar ** list)
{
	xmlDocPtr doc = NULL;
	xmlNodePtr root = NULL;
	xmlChar *xmlbuff = NULL;
	int buffersize = 0;

	CHECK_STATUS((string_list == NULL), "GSList of String null", return OSSO_PACK_INVALID);
	CHECK_STATUS((list == NULL), "pack str arg null", return OSSO_PACK_INVALID);

	DBUS_XML_PACK_INIT;

	osso_email_xml_pack_string_list(root, doc, XML_STRING_LST, string_list);

	xmlDocDumpFormatMemory(doc, &xmlbuff, &buffersize, 1);

	DBUS_XML_PACK_UN_INIT;

	*list = (gchar *) xmlbuff;
	return OSSO_PACK_OK;
}

/**
   Pack  function for Message headers in Folder populate folder

   @param  msgheader_list GSList of MsgHeader Struct to be packed in Xml format
   @param  list  Packed data to be stored in message
   @return OSSO_PACK_OK on success (or) OSSO_PACK_INVALID for invalid 
           parameter (or) OSSO_PACK_ERROR for other kind of errors
*/
osso_pack_return_t
osso_email_utils_pack_populate_folder_to_gslist(GSList * msgheader_list, gchar ** list)
{
	xmlDocPtr doc = NULL;
	xmlNodePtr root = NULL;
	xmlChar *xmlbuff = NULL;
	int buffersize = 0;

	CHECK_STATUS((msgheader_list == NULL), "GSList of Message Headers null",
		     return OSSO_PACK_INVALID);
	CHECK_STATUS((list == NULL), "pack str arg null", return OSSO_PACK_INVALID);

	DBUS_XML_PACK_INIT;
	osso_email_xml_pack_msgheader_list(root, doc, XML_MSG_HEADER_LST, msgheader_list);
	xmlDocDumpFormatMemory(doc, &xmlbuff, &buffersize, 1);
	DBUS_XML_PACK_UN_INIT;

	*list = (gchar *) xmlbuff;
	return OSSO_PACK_OK;
}


/**
   Pack  function for GSList of Message Header List

   @param  msgheader_list GSList of MsgHeader Struct to be packed in Xml format
   @param  list  Packed data to be stored in message
   @return OSSO_PACK_OK on success (or) OSSO_PACK_INVALID for invalid 
           parameter (or) OSSO_PACK_ERROR for other kind of errors
*/
osso_pack_return_t osso_email_utils_pack_msgheader_to_gslist(GSList * msgheader_list, gchar ** list)
{
	xmlDocPtr doc = NULL;
	xmlNodePtr root = NULL;
	xmlChar *xmlbuff = NULL;
	int buffersize = 0;

	CHECK_STATUS((msgheader_list == NULL), "GSList of Message Headers null",
		     return OSSO_PACK_INVALID);
	CHECK_STATUS((list == NULL), "pack str arg null", return OSSO_PACK_INVALID);

	DBUS_XML_PACK_INIT;

	osso_email_xml_pack_msgheader_list(root, doc, XML_MSG_HEADER_LST, msgheader_list);

	xmlDocDumpFormatMemory(doc, &xmlbuff, &buffersize, 1);

	DBUS_XML_PACK_UN_INIT;

	*list = (gchar *) xmlbuff;
	return OSSO_PACK_OK;
}

/**
   Pack  function for GSList of Prefs Account

   @param  account_list GSList of PrefsAccount Struct to be packed in Xml format
   @param  list  Packed data to be stored in message
   @return OSSO_PACK_OK on success (or) OSSO_PACK_INVALID for invalid 
           parameter (or) OSSO_PACK_ERROR for other kind of errors
*/
osso_pack_return_t
osso_email_utils_pack_prefs_account_to_gslist(GSList * account_list, gchar ** list)
{
	xmlDocPtr doc = NULL;
	xmlNodePtr root = NULL;
	xmlChar *xmlbuff = NULL;
	int buffersize = 0;

	CHECK_STATUS((account_list == NULL), "GSList of PrefsAccount null",
		     return OSSO_PACK_INVALID);
	CHECK_STATUS((list == NULL), "pack str arg null", return OSSO_PACK_INVALID);

	DBUS_XML_PACK_INIT;

	osso_email_xml_pack_prefsaccount_list(root, doc, XML_PREFS_ACCOUNT_LST, account_list);

	xmlDocDumpFormatMemory(doc, &xmlbuff, &buffersize, 1);

	DBUS_XML_PACK_UN_INIT;

	*list = (gchar *) xmlbuff;
	return OSSO_PACK_OK;
}


/**
   Pack  function for GSList of Folder Struct

   @param  folder_list GSList of Folder Struct to be packed in Xml format
   @param  list  Packed data to be stored in message
   @return OSSO_PACK_OK on success (or) OSSO_PACK_INVALID for invalid 
           parameter (or) OSSO_PACK_ERROR for other kind of errors
*/
osso_pack_return_t osso_email_utils_pack_folder_to_gslist(GSList * folder_list, gchar ** list)
{
	xmlDocPtr doc = NULL;
	xmlNodePtr root = NULL;
	xmlChar *xmlbuff = NULL;
	int buffersize = 0;

	CHECK_STATUS((folder_list == NULL), "Folder List is  null", return OSSO_PACK_INVALID);
	CHECK_STATUS((list == NULL), "pack str arg null", return OSSO_PACK_INVALID);

	DBUS_XML_PACK_INIT;

	osso_email_xml_pack_folder_list(root, doc, XML_FOLDER_LST, folder_list);

	xmlDocDumpFormatMemory(doc, &xmlbuff, &buffersize, 1);

	DBUS_XML_PACK_UN_INIT;

	*list = (gchar *) xmlbuff;
	return OSSO_PACK_OK;

}

/**
   Pack  function for GSList of Folder Attach Struct

   @param  attach_list GSList of Folder Attach Struct to be packed in Xml format
   @param  list  Packed data to be stored in message
   @return OSSO_PACK_OK on success (or) OSSO_PACK_INVALID for invalid 
           parameter (or) OSSO_PACK_ERROR for other kind of errors
*/
osso_pack_return_t
osso_email_utils_pack_folder_attach_to_gslist(GSList * attach_list, gchar ** list)
{
	xmlDocPtr doc = NULL;
	xmlNodePtr root = NULL;
	xmlChar *xmlbuff = NULL;
	int buffersize = 0;

	CHECK_STATUS((attach_list == NULL), "Folder Atatch List is  null",
		     return OSSO_PACK_INVALID);
	CHECK_STATUS((list == NULL), "pack str arg null", return OSSO_PACK_INVALID);

	DBUS_XML_PACK_INIT;

	osso_email_xml_pack_folder_attachinfo_list(root, doc, XML_FOLDER_ATTACHINFO_LST,
						   attach_list);

	xmlDocDumpFormatMemory(doc, &xmlbuff, &buffersize, 1);

	DBUS_XML_PACK_UN_INIT;

	*list = (gchar *) xmlbuff;
	return OSSO_PACK_OK;

}

/**
   Pack  function for GSList of AttachInfo struct
   @param  attachinfo_list GSList of AttachInfo Struct to be packed
           in Xml format
   @param  list  Packed data to be stored in message
   @return OSSO_PACK_OK on success (or) OSSO_PACK_INVALID for invalid 
           parameter (or) OSSO_PACK_ERROR for other kind of errors
*/

osso_pack_return_t
osso_email_utils_pack_from_gslist_to_attachinfo(GSList * attachinfo_list, gchar ** list)
{
	xmlDocPtr doc = NULL;
	xmlNodePtr root = NULL;
	xmlChar *xmlbuff = NULL;
	int buffersize = 0;

	CHECK_STATUS((attachinfo_list == NULL), "AtatchInfo List is  null",
		     return OSSO_PACK_INVALID);
	CHECK_STATUS((list == NULL), "pack str arg null", return OSSO_PACK_INVALID);

	DBUS_XML_PACK_INIT;

	osso_email_xml_pack_folder_attachinfo_list(root, doc, XML_ATTACHINFO_LST, attachinfo_list);

	xmlDocDumpFormatMemory(doc, &xmlbuff, &buffersize, 1);

	DBUS_XML_PACK_UN_INIT;

	*list = (gchar *) xmlbuff;
	return OSSO_PACK_OK;
}


/**
   Unpack  function GSList of AttachInfo struct

   @param  list GSList of AttachInfo  struct to be unpacked from Xml format
   @param  data  Data which is to be unpacked
   @return OSSO_PACK_OK on success (or) OSSO_PACK_INVALID for invalid 
           parameter (or) OSSO_PACK_ERROR for other kind of errors
*/

osso_pack_return_t osso_email_utils_unpack_attachinfo_in_gslist(gchar * data, GSList ** list)
{
	xmlNodePtr cur = NULL, node = NULL;
	xmlDocPtr doc = NULL;
	GSList *ret_list = NULL;

	CHECK_STATUS((list == NULL), "AttachInfo GSList is  null", return OSSO_PACK_INVALID);
	CHECK_STATUS((data == NULL), "pack str arg null", return OSSO_PACK_INVALID);

	DBUS_XML_UNPACK_INIT(data, return OSSO_PACK_ERROR);

	for (node = cur->xmlChildrenNode; node != NULL; node = node->next) {
		if (node->type == XML_ELEMENT_NODE) {
			if (g_ascii_strcasecmp((gchar *) node->name, XML_ATTACHINFO_LST) == 0) {
				osso_email_xml_unpack_attachinfo_list(node, doc, XML_ATTACHINFO_LST,
								      &ret_list);
			}
		}
	}
	DBUS_XML_UNPACK_UN_INIT;

	*list = ret_list;
	return OSSO_PACK_OK;
}


/**
   Unpack  function GSList of Message Info struct

   @param  list GSList of Message Info  struct to be unpacked from Xml format
   @param  data  Data which is to be unpacked
   @return OSSO_PACK_OK on success (or) OSSO_PACK_INVALID for invalid 
           parameter (or) OSSO_PACK_ERROR for other kind of errors
*/
osso_pack_return_t osso_email_utils_unpack_msginfo_in_gslist(gchar * data, GSList ** list)
{
	xmlNodePtr cur = NULL, node = NULL;
	xmlDocPtr doc = NULL;
	GSList *ret_list = NULL;

	CHECK_STATUS((list == NULL), "MessageInfo GSList is  null", return OSSO_PACK_INVALID);
	CHECK_STATUS((data == NULL), "pack str arg null", return OSSO_PACK_INVALID);

	DBUS_XML_UNPACK_INIT(data, return OSSO_PACK_ERROR);

	for (node = cur->xmlChildrenNode; node != NULL; node = node->next) {
		if (node->type == XML_ELEMENT_NODE) {
			if (g_ascii_strcasecmp((gchar *) node->name, XML_MSG_INFO_LST) == 0) {
				osso_email_xml_unpack_msginfo_list(node, doc, XML_MSG_INFO_LST,
								   &ret_list);
			}
		}
	}
	DBUS_XML_UNPACK_UN_INIT;

	*list = ret_list;
	return OSSO_PACK_OK;
}


/**
   Unpack  function GSList of Message Header struct

   @param  list GSList of Message Header  struct to be unpacked from Xml format
   @param  data  Data which is to be unpacked
   @return OSSO_PACK_OK on success (or) OSSO_PACK_INVALID for invalid 
           parameter (or) OSSO_PACK_ERROR for other kind of errors
*/
osso_pack_return_t osso_email_utils_unpack_msgheader_in_gslist(gchar * data, GSList ** list)
{

	xmlNodePtr cur = NULL, node = NULL;
	xmlDocPtr doc = NULL;
	GSList *ret_list = NULL;

	CHECK_STATUS((list == NULL), "MessageHeader GSList is  null", return OSSO_PACK_INVALID);
	CHECK_STATUS((data == NULL), "pack str arg null", return OSSO_PACK_INVALID);

	DBUS_XML_UNPACK_INIT(data, return OSSO_PACK_ERROR);

	for (node = cur->xmlChildrenNode; node != NULL; node = node->next) {
		if (node->type == XML_ELEMENT_NODE) {
			if (g_ascii_strcasecmp((gchar *) node->name, XML_MSG_HEADER_LST) == 0) {
				osso_email_xml_unpack_msgheader_list(node, doc, XML_MSG_HEADER_LST,
								     &ret_list);
			}
		}
	}
	DBUS_XML_UNPACK_UN_INIT;

	*list = ret_list;
	return OSSO_PACK_OK;

}

/**
   Unpack  function GSList of Prefs Account struct

   @param  list GSList of Prefs Account struct to be unpacked from Xml format
   @param  data  Data which is to be unpacked
   @return OSSO_PACK_OK on success (or) OSSO_PACK_INVALID for invalid 
           parameter (or) OSSO_PACK_ERROR for other kind of errors
*/
osso_pack_return_t osso_email_utils_unpack_prefsaccount_in_gslist(gchar * data, GSList ** list)
{

	xmlNodePtr cur = NULL, node = NULL;
	xmlDocPtr doc = NULL;
	GSList *ret_list = NULL;

	CHECK_STATUS((list == NULL), "PrefsAccount GSList is  null", return OSSO_PACK_INVALID);
	CHECK_STATUS((data == NULL), "pack string arg null", return OSSO_PACK_INVALID);

	DBUS_XML_UNPACK_INIT(data, return OSSO_PACK_ERROR);

	for (node = cur->xmlChildrenNode; node != NULL; node = node->next) {
		if (node->type == XML_ELEMENT_NODE) {
			if (g_ascii_strcasecmp((gchar *) node->name, XML_PREFS_ACCOUNT_LST) == 0) {
				osso_email_xml_unpack_prefsaccount_list(node, doc,
									XML_PREFS_ACCOUNT_LST,
									&ret_list);
			}
		}
	}
	DBUS_XML_UNPACK_UN_INIT;

	*list = ret_list;
	return OSSO_PACK_OK;

}

/**
   Unpack  function GSList of Folder struct

   @param  list GSList of Folder struct to be unpacked from Xml format
   @param  data  Data which is to be unpacked
   @return OSSO_PACK_OK on success (or) OSSO_PACK_INVALID for invalid 
           parameter (or) OSSO_PACK_ERROR for other kind of errors
*/
osso_pack_return_t osso_email_utils_unpack_folder_in_gslist(gchar * data, GSList ** list)
{
	xmlNodePtr cur = NULL, node = NULL;
	xmlDocPtr doc = NULL;
	GSList *ret_list = NULL;

	CHECK_STATUS((list == NULL), "Folder GSList is  null", return OSSO_PACK_INVALID);
	CHECK_STATUS((data == NULL), "pack string arg null", return OSSO_PACK_INVALID);

	DBUS_XML_UNPACK_INIT(data, return OSSO_PACK_ERROR);

	for (node = cur->xmlChildrenNode; node != NULL; node = node->next) {
		if (node->type == XML_ELEMENT_NODE) {
			if (g_ascii_strcasecmp((gchar *) node->name, XML_FOLDER_LST) == 0) {
				osso_email_xml_unpack_folder_list(node, doc, XML_FOLDER_LST,
								  &ret_list);
				break;
			}
		}
	}
	DBUS_XML_UNPACK_UN_INIT;

	*list = ret_list;
	return OSSO_PACK_OK;
}


/**
   Unpack  function GSList of Folder Attach struct

   @param  list GSList of Folder Attach struct to be unpacked from Xml format
   @param  data  Data which is to be unpacked
   @return OSSO_PACK_OK on success (or) OSSO_PACK_INVALID for invalid 
           parameter (or) OSSO_PACK_ERROR for other kind of errors
*/
osso_pack_return_t osso_email_utils_unpack_folderattach_in_gslist(gchar * data, GSList ** list)
{
	xmlNodePtr cur = NULL, node = NULL;
	xmlDocPtr doc = NULL;
	GSList *ret_list = NULL;

	CHECK_STATUS((list == NULL), "FolderAttach GSList is null", return OSSO_PACK_INVALID);
	CHECK_STATUS((data == NULL), "pack str arg null", return OSSO_PACK_INVALID);

	DBUS_XML_UNPACK_INIT(data, return OSSO_PACK_ERROR);

	for (node = cur->xmlChildrenNode; node != NULL; node = node->next) {
		if (node->type == XML_ELEMENT_NODE) {
			if (g_ascii_strcasecmp((gchar *) node->name, XML_FOLDER_ATTACHINFO_LST) ==
			    0) {
				osso_email_xml_unpack_folderattachinfo_list(node, doc,
									    XML_FOLDER_ATTACHINFO_LST,
									    &ret_list);
				break;
			}
		}
	}
	DBUS_XML_UNPACK_UN_INIT;

	*list = ret_list;
	return OSSO_PACK_OK;
}


/**
   Unpack  function GSList of AttachInfo struct

   @param  list GSList of AttachInfo struct to be unpacked from Xml format
   @param  data  Data which is to be unpacked
   @return OSSO_PACK_OK on success (or) OSSO_PACK_INVALID for invalid 
           parameter (or) OSSO_PACK_ERROR for other kind of errors
*/
osso_pack_return_t osso_email_utils_unpack_attachinfo_to_gslist(gchar * data, GSList ** list)
{
	xmlNodePtr cur = NULL, node = NULL;
	xmlDocPtr doc = NULL;
	GSList *ret_list = NULL;

	CHECK_STATUS((list == NULL), "AttachInfo GSList is  null", return OSSO_PACK_INVALID);
	CHECK_STATUS((data == NULL), "pack str arg null", return OSSO_PACK_INVALID);

	DBUS_XML_UNPACK_INIT(data, return OSSO_PACK_ERROR);

	node = cur->xmlChildrenNode;
	for (node = cur->xmlChildrenNode; node != NULL; node = node->next) {
		if (node->type == XML_ELEMENT_NODE) {
			if (g_ascii_strcasecmp((gchar *) node->name, XML_ATTACHINFO_LST) == 0) {
				osso_email_xml_unpack_attachinfo_list(node, doc, XML_ATTACHINFO_LST,
								      &ret_list);
				break;
			}
		}
	}
	DBUS_XML_UNPACK_UN_INIT;

	*list = ret_list;
	return OSSO_PACK_OK;
}


/**
   Unpack  function GSList of Message Headers contains in the packed data

   @param  list GSList of MsgHeader struct to be unpacked from Xml format
   @param  data  Data which is to be unpacked
   @return OSSO_PACK_OK on success (or) OSSO_PACK_INVALID for invalid 
           parameter (or) OSSO_PACK_ERROR for other kind of errors
*/
osso_pack_return_t osso_email_utils_unpack_populate_folder_to_gslist(gchar * data, GSList ** list)
{
	xmlNodePtr cur = NULL, node = NULL;
	xmlDocPtr doc = NULL;
	GSList *ret_list = NULL;

	CHECK_STATUS((list == NULL), "MsgHeader GSList is  null", return OSSO_PACK_INVALID);
	CHECK_STATUS((data == NULL), "pack str arg null", return OSSO_PACK_INVALID);

	DBUS_XML_UNPACK_INIT(data, return OSSO_PACK_ERROR)

	    for (node = cur->xmlChildrenNode; node != NULL; node = node->next) {
		if (node->type == XML_ELEMENT_NODE) {
			if (g_ascii_strcasecmp((gchar *) node->name, XML_MSG_HEADER_LST) == 0) {
				osso_email_xml_unpack_msgheader_list(node, doc, XML_MSG_HEADER_LST,
								     &ret_list);
				break;
			}
		}
	}
	DBUS_XML_UNPACK_UN_INIT;

	*list = ret_list;
	return OSSO_PACK_OK;
}


/**
   Unpack  function GSList of Message ID list in the packed data

   @param  list GSList of Message Ids to be unpacked from Xml format
   @param  data  Data which is to be unpacked
   @return OSSO_PACK_OK on success (or) OSSO_PACK_INVALID for invalid 
           parameter (or) OSSO_PACK_ERROR for other kind of errors
*/
osso_pack_return_t osso_email_utils_unpack_msgid_list_in_gslist(gchar * data, GSList ** list)
{
	xmlNodePtr cur = NULL, node = NULL;
	xmlDocPtr doc = NULL;
	GSList *ret_list = NULL;

	CHECK_STATUS((list == NULL), "Msg Id GSList is  null", return OSSO_PACK_INVALID);
	CHECK_STATUS((data == NULL), "pack str arg null", return OSSO_PACK_INVALID);

	DBUS_XML_UNPACK_INIT(data, return OSSO_PACK_ERROR)

	    for (node = cur->xmlChildrenNode; node != NULL; node = node->next) {
		if (node->type == XML_ELEMENT_NODE) {
			if (g_ascii_strcasecmp((gchar *) node->name, XML_MSG_ID_LST) == 0) {
				osso_email_xml_unpack_msgid_list(node, doc, XML_MSG_ID_LST,
								 &ret_list);
				break;
			}
		}
	}
	DBUS_XML_UNPACK_UN_INIT;

	*list = ret_list;
	return OSSO_PACK_OK;
}

/**
   Unpack  function GSList of Strings in the packed data

   @param  list GSList of Strings to be unpacked from Xml format
   @param  data  Data which is to be unpacked
   @return OSSO_PACK_OK on success (or) OSSO_PACK_INVALID for invalid 
           parameter (or) OSSO_PACK_ERROR for other kind of errors
*/
osso_pack_return_t osso_email_utils_unpack_string_in_gslist(gchar * data, GSList ** list)
{
	xmlNodePtr cur = NULL, node = NULL;
	xmlDocPtr doc = NULL;
	GSList *ret_list = NULL;

	CHECK_STATUS((list == NULL), "String GSList is  null", return OSSO_PACK_INVALID);
	CHECK_STATUS((data == NULL), "pack str arg null", return OSSO_PACK_INVALID);

	DBUS_XML_UNPACK_INIT(data, return OSSO_PACK_ERROR)

	    for (node = cur->xmlChildrenNode; node != NULL; node = node->next) {
		if (node->type == XML_ELEMENT_NODE) {
			if (g_ascii_strcasecmp((gchar *) node->name, XML_STRING_LST) == 0) {
				osso_email_xml_unpack_string_list(node, doc, XML_STRING_LST,
								  &ret_list);
				break;
			}
		}
	}
	DBUS_XML_UNPACK_UN_INIT;

	*list = ret_list;
	return OSSO_PACK_OK;
}


/**
   Function to free Folder Attach Struct

   @param  folder_attach_list GSList of Folder Attach struct
   @return OSSO_PACK_OK on success (or) OSSO_PACK_INVALID for invalid 
           parameter (or) OSSO_PACK_ERROR for other kind of errors
*/
osso_pack_return_t osso_email_utils_folder_attach_free(GSList * folder_attach_list)
{
	FolderAttachInfo *folder_attach = NULL;

	CHECK_STATUS((folder_attach_list == NULL), "FolderAttach List  is  null",
		     return OSSO_PACK_INVALID);

	while (folder_attach_list) {
		folder_attach = (FolderAttachInfo *) folder_attach_list->data;
		g_free(folder_attach->msgid);
		folder_attach->msgid = NULL;
		g_free(folder_attach->section_number);
		folder_attach->section_number = NULL;

		folder_attach_list = g_slist_remove(folder_attach_list, folder_attach_list->data);
	}
	return OSSO_PACK_OK;
}

/**
   Function to concat two strings seperated by separator(:) 

   @param  str1   First string to concatanate
   @param  str2   Second string to concatanate
   @return NULL on failure or concated string on success 
*/
gchar *osso_email_utils_concat_strings(const gchar * str1, const gchar * str2)
{

	GString *concat_str = NULL;
	gchar *ret_str = NULL;

	if (str1 == NULL) {
		osso_log(LOG_INFO, "Input param is NULL");
		return NULL;
	}
	concat_str = g_string_new(NULL);
	if (concat_str == NULL) {
		osso_log(LOG_ERR, "Memory Alloc Failed");
		return NULL;
	}
	if (str2 != NULL) {
		g_string_printf(concat_str, "%s%s%s", str1, seperator, str2);
	} else {
		g_string_printf(concat_str, "%s%s", str1, seperator);
	}
	ret_str = g_strdup(concat_str->str);
	g_string_free(concat_str, TRUE);
	return ret_str;
}


/**
   Function to concat String and integer seperated by separator(:) 

   @param  str   String  contains Function Identifier 
   @param  val   integer value to concat to the string 
   @return NULL on failure or concated string on success 
*/
gchar *osso_email_utils_concat_str_int(const gchar * str1, const gint val)
{

	GString *concat_str = NULL;
	gchar *ret_str = NULL;

	if (str1 == NULL) {
		osso_log(LOG_INFO, "Input param is NULL");
		return NULL;
	}

	concat_str = g_string_new(NULL);
	if (concat_str == NULL) {
		osso_log(LOG_ERR, "Memory Alloc Failed");
		return NULL;
	}
	g_string_printf(concat_str, "%s%s%d", str1, seperator, val);

	ret_str = g_strdup(concat_str->str);
	g_string_free(concat_str, TRUE);
	return ret_str;
}

/**
   Function to free the mimeinfo allocated during xml packing. 

   @param  mimeinfo structure to be freed
   @return OSSO_PACK_OK on successful free of mimeinfo (or) 
    OSSO_PACK_INVALID for invalid input. 
*/

osso_pack_return_t osso_email_utils_mimeinfo_free_all(MimeInfo * mimeinfo)
{
	MimeInfo *parent = NULL;

	if (mimeinfo == NULL) {
		return OSSO_PACK_INVALID;
	}
	if (mimeinfo->parent != NULL) {
		/* Normally parent will be NULL for root mimeinfo
		 * but when UI application requests for mimeinfo of attachment
		 * (may not be root mimeinfo), it packs the parent mime info also
		 * in this case it needs to be freed
		 */
		parent = mimeinfo->parent;
		if (parent != NULL) {
			g_free(parent->boundary);
			g_free(parent);
		}
	}

	osso_email_xml_mimeinfo_free_all(mimeinfo);
	return OSSO_PACK_OK;
}

/**
   Function to convert the given string to base64 format 

   @param  inbuf  base64 string which is to converted to normal string 
   @param  convdata converted string for given input base64 string 
   @return OSSO_PACK_OK on success (or) OSSO_PACK_INVALID for invalid 
           parameter (or) OSSO_PACK_ERROR for other kind of errors
*/
osso_pack_return_t osso_email_utils_pack_str_to_base64(const guchar * indata, gchar ** convdata)
{
	gchar outbuf[B64_BUFFSIZE + 1];
	gint len = 0;
	GString *base64buf = NULL;
	guchar *inbuf = NULL;

	if ((indata == NULL) || (convdata == NULL)) {
		osso_log(LOG_INFO, "Input param is NULL");
		return OSSO_PACK_INVALID;
	}
	inbuf = (guchar *) indata;

	len = strlen(inbuf);
	base64buf = g_string_new(NULL);
	if (base64buf == NULL) {
		osso_log(LOG_ERR, "Memory Alloc Failed");
		return OSSO_PACK_ALLOC_FAIL;
	}
	while (len > 0) {
		memset(outbuf, '\0', B64_BUFFSIZE);
		base64_encode_dbus(outbuf, inbuf, B64_LINE_SIZE);
		g_string_append_len(base64buf, outbuf, strlen(outbuf));

		if (len > B64_LINE_SIZE) {
			inbuf += B64_LINE_SIZE;
		} else {
			inbuf += len;
		}
		len = strlen((const char *) inbuf);
	}
	*convdata = g_strdup(base64buf->str);
	g_string_free(base64buf, TRUE);
	return OSSO_PACK_OK;
}


/**
   Function to convert the given base64 format string to normal 

   @param  inbuf  base64 string which is to converted to normal string 
   @param  convdata converted string for given input base64 string 
   @return OSSO_PACK_OK on success (or) OSSO_PACK_INVALID for invalid 
           parameter (or) OSSO_PACK_ERROR for other kind of errors
*/
osso_pack_return_t
osso_email_utils_unpack_base64_to_str(const gchar * base64data, gchar ** convdata)
{
	guchar *outbuf = NULL;
	gint outbuflen = 0;

	if ((base64data == NULL) || (convdata == NULL)) {
		return OSSO_PACK_INVALID;
	}
	outbuf = (guchar *) g_malloc0(strlen((const char *) base64data) + 1);
	if (outbuf == NULL) {
		osso_log(LOG_ERR, "Memory Alloc Failed");
		return OSSO_PACK_ALLOC_FAIL;
	}
	outbuflen = base64_decode_dbus(outbuf, base64data, -1);

	outbuf[outbuflen] = '\0';
	*convdata = (gchar *) outbuf;
	return OSSO_PACK_OK;
}

#if CL_USE

/**
   Pack  function for GSList of iapinfo struct

   @param  list GSList of iapinfo struct packed from Xml format
   @param  data populated with xml packed data 
   @return OSSO_PACK_OK on success (or) OSSO_PACK_INVALID for invalid 
           parameter (or) OSSO_PACK_ERROR for other kind of errors
*/

osso_pack_return_t osso_email_utils_pack_iapinfo_list(GSList * iapinfo_list, gchar ** list)
{
	xmlDocPtr doc = NULL;
	xmlNodePtr root = NULL;
	xmlChar *xmlbuff = NULL;
	int buffersize = 0;

	CHECK_STATUS((iapinfo_list == NULL), "iapinfo list null", return OSSO_PACK_INVALID);
	CHECK_STATUS((list == NULL), "pack str arg null", return OSSO_PACK_INVALID);

	DBUS_XML_PACK_INIT;
	osso_email_xml_pack_iapinfo_list(root, doc, XML_IAPINFO_LST, iapinfo_list);
	xmlDocDumpFormatMemory(doc, &xmlbuff, &buffersize, 1);
	DBUS_XML_PACK_UN_INIT;

	*list = xmlbuff;
	return OSSO_PACK_OK;
}

/**
   Unpack  function GSList of iapinfo struct

   @param  list GSList of Message Ids to be unpacked from Xml format
   @param  data  contains the xml string which is to be unpacked
   @return OSSO_PACK_OK on success (or) OSSO_PACK_INVALID for invalid 
           parameter (or) OSSO_PACK_ERROR for other kind of errors
*/
osso_pack_return_t osso_email_utils_unpack_iapinfo_list(gchar * data, GSList ** list)
{
	xmlNodePtr cur = NULL, node = NULL;
	xmlDocPtr doc = NULL;
	void *ret_list = NULL;

	CHECK_STATUS((list == NULL), "IapInfio list null", return OSSO_PACK_INVALID);
	CHECK_STATUS((data == NULL), "pack str arg null", return OSSO_PACK_INVALID);

	DBUS_XML_UNPACK_INIT(data, return OSSO_PACK_ERROR);

	node = cur->xmlChildrenNode;
	for (node = cur->xmlChildrenNode; node != NULL; node = node->next) {
		if (node->type == XML_ELEMENT_NODE) {
			if (g_ascii_strcasecmp(node->name, XML_IAPINFO_LST) == 0) {
				osso_email_xml_unpack_iapinfo_list(node, doc,
								   XML_IAPINFO_LST,
								   (void **) &ret_list);
				break;
			}
		}
	}
	DBUS_XML_UNPACK_UN_INIT;

	*list = (GSList *) ret_list;
	return OSSO_PACK_OK;
}

#endif


osso_pack_return_t dbus_utils_persistentInfo_write_to_file(EmailPersistentInfo * perInfo)
{
	xmlDocPtr doc = NULL;
	xmlNodePtr root = NULL;
	gchar *rcpath = NULL;

	if (perInfo == NULL) {
		osso_log(LOG_INFO, "Received NULL value for stateinfo\n");
		return OSSO_PACK_INVALID;
	}
	rcpath = g_strconcat((const gchar *) get_rc_dir_dbus(),
			     G_DIR_SEPARATOR_S, UI_SETTINGS_RC, NULL);
	DBUS_XML_PACK_INIT;
	osso_email_xml_pack_emailPersistentInfo(root, doc,
						XML_EMAIL_PERSISTENT_INFO, (void *) perInfo);
	xmlSaveFormatFile(rcpath, doc, 1);
	DBUS_XML_PACK_UN_INIT;
	g_free(rcpath);

	return OSSO_PACK_OK;
}


osso_pack_return_t dbus_utils_persistentInfo_read_from_file(EmailPersistentInfo * perInfo)
{
	xmlNodePtr cur = NULL;
	xmlNodePtr node = NULL;
	xmlDocPtr doc = NULL;
	gchar *rcpath = NULL;

	if (perInfo == NULL) {
		osso_log(LOG_INFO, " perInfo is NULL \n");
		return OSSO_PACK_ERROR;
	}
	rcpath = g_strconcat((const gchar *) get_rc_dir_dbus(),
			     G_DIR_SEPARATOR_S, UI_SETTINGS_RC, NULL);
	if (rcpath == NULL) {
		osso_log(LOG_INFO, "rc path is NULL \n");
		return OSSO_PACK_ERROR;
	}

	if (access(rcpath, R_OK) != 0) {
		osso_log(LOG_INFO, "cannot open %s for reading: %s", rcpath, strerror(errno));
		g_free(rcpath);
		return OSSO_PACK_ERROR;
	}

	doc = xmlParseFile(rcpath);

	cur = xmlDocGetRootElement(doc);
	if ((cur == NULL) || (strcmp(cur->name, XML_DATA) != 0)) {
		osso_log(LOG_INFO, "Invald Root element \n");
		xmlFreeDoc(doc);
		g_free(rcpath);
		return OSSO_PACK_ERROR;
	}

	for (node = cur->xmlChildrenNode; node != NULL; node = node->next) {
		if (node->type == XML_ELEMENT_NODE) {
			if (g_ascii_strcasecmp(node->name, XML_EMAIL_PERSISTENT_INFO) == 0) {
				osso_email_xml_unpack_emailPersistentInfo(node, doc,
									  XML_EMAIL_PERSISTENT_INFO,
									  perInfo);
				break;
			}
		}
	}
	xmlFreeDoc(doc);
	g_free(rcpath);
	return OSSO_PACK_OK;
}
