
/**
 * This file is part of osso-email-common
 *
 * Copyright (c) 2006 Nokia Corporation.
 * 
 * Contact: Dirk-Jan Binnema <dirk-jan.binnema@nokia.com>
 * 
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public License
 * version 2.1 as published by the Free Software Foundation.
 * 
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA
 * 02110-1301 USA
 */

#include "dbus_common_utils.h"
#include <glib.h>
#include <ctype.h>
#include <string.h>


static const gchar base64chardbus[64] =
    "ABCDEFGHIJKLMNOPQRSTUVWXYZabcdefghijklmnopqrstuvwxyz0123456789+/";

static const gchar base64valdbus[128] = {
	-1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1,
	-1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1,
	-1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, 62, -1, -1, -1, 63,
	52, 53, 54, 55, 56, 57, 58, 59, 60, 61, -1, -1, -1, -1, -1, -1,
	-1, 0, 1, 2, 3, 4, 5, 6, 7, 8, 9, 10, 11, 12, 13, 14,
	15, 16, 17, 18, 19, 20, 21, 22, 23, 24, 25, -1, -1, -1, -1, -1,
	-1, 26, 27, 28, 29, 30, 31, 32, 33, 34, 35, 36, 37, 38, 39, 40,
	41, 42, 43, 44, 45, 46, 47, 48, 49, 50, 51, -1, -1, -1, -1, -1
};

#define BASE64VALDBUS(c)    (isascii(c) ? base64valdbus[(gint)(c)] : -1)

/**         
 *    This method is to get resource directory path
 *    @input none
 *    @return resource directory path 
 *       
 */

gchar *get_rc_dir_dbus(void)
{
	static gchar *rc_dir = NULL;

	if (!rc_dir) {
		rc_dir = g_strdup_printf("%s%s", g_get_home_dir(), CONFIG_FILE_PATH_DBUS);
	}
	return rc_dir;
}

/**         
 *    This method is to convert to base64 encoding
 *    @param out base64 encoded string, in string to converted to base64 
 *    @return  none
 *       
 */

void base64_encode_dbus(gchar * out, const guchar * in, gint inlen)
{
	const guchar *inp = in;
	gchar *outp = out;

	while (inlen >= 3) {
		*outp++ = base64chardbus[(inp[0] >> 2) & 0x3f];
		*outp++ = base64chardbus[((inp[0] & 0x03) << 4) | ((inp[1] >> 4) & 0x0f)];
		*outp++ = base64chardbus[((inp[1] & 0x0f) << 2) | ((inp[2] >> 6) & 0x03)];
		*outp++ = base64chardbus[inp[2] & 0x3f];

		inp += 3;
		inlen -= 3;
	}

	if (inlen > 0) {
		*outp++ = base64chardbus[(inp[0] >> 2) & 0x3f];
		if (inlen == 1) {
			*outp++ = base64chardbus[(inp[0] & 0x03) << 4];
			*outp++ = '=';
		} else {
			*outp++ = base64chardbus[((inp[0] & 0x03) << 4) | ((inp[1] >> 4) & 0x0f)];
			*outp++ = base64chardbus[((inp[1] & 0x0f) << 2)];
		}
		*outp++ = '=';
	}

	*outp = '\0';
}

/**         
 *    This method is for decoding base64 string
 *    @param out base64 decoded string, in base64 encoded string
 *    @return  none
 *       
 */

gint base64_decode_dbus(guchar * out, const gchar * in, gint inlen)
{
	const gchar *inp = in;
	guchar *outp = out;
	gchar buf[4];

	if (inlen < 0)
		inlen = G_MAXINT;

	while (inlen >= 4 && *inp != '\0') {
		buf[0] = *inp++;
		inlen--;
		if (BASE64VALDBUS(buf[0]) == -1)
			break;

		buf[1] = *inp++;
		inlen--;
		if (BASE64VALDBUS(buf[1]) == -1)
			break;

		buf[2] = *inp++;
		inlen--;
		if (buf[2] != '=' && BASE64VALDBUS(buf[2]) == -1)
			break;

		buf[3] = *inp++;
		inlen--;
		if (buf[3] != '=' && BASE64VALDBUS(buf[3]) == -1)
			break;

		*outp++ = ((BASE64VALDBUS(buf[0]) << 2) & 0xfc) |
		    ((BASE64VALDBUS(buf[1]) >> 4) & 0x03);
		if (buf[2] != '=') {
			*outp++ = ((BASE64VALDBUS(buf[1]) & 0x0f) << 4) |
			    ((BASE64VALDBUS(buf[2]) >> 2) & 0x0f);
			if (buf[3] != '=') {
				*outp++ = ((BASE64VALDBUS(buf[2]) & 0x03) << 6) |
				    (BASE64VALDBUS(buf[3]) & 0x3f);
			}
		}
	}

	return outp - out;
}

/**         
 *    This method is for freeing DBUS  list
 *    @param  list to be freed
 *    @return  none
 *       
 */

void procmsg_ImageInfo_list_free_DBUS(GSList * image_list)
{
	InlineImageInfoDBUS *imageinfo = NULL;
	if (image_list == NULL) {
		return;
	}
	while (image_list != NULL) {
		imageinfo = image_list->data;
		image_list = g_slist_remove(image_list, imageinfo);
		procmsg_ImageInfo_free_DBUS(imageinfo);
	}
}

/**         
 *    This method is for freeing InlineImageInfo list
 *    @param  InlineImageInfo to be freed
 *    @return  none
 *       
 */

void procmsg_ImageInfo_free_DBUS(InlineImageInfoDBUS * info)
{
	if (info != NULL) {
		g_free(info->uri_path);
		g_free(info->content_id);
		g_free(info->image_name);
		g_free(info->image_path);
		g_free(info);
		info = NULL;
	}
}
