/*
 * OProfile User Interface
 *
 * Copyright (C) 2007 Nokia Corporation. All rights reserved.
 *
 * Author: Robert Bradford <rob@openedhand.com>
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * version 2 as published by the Free Software Foundation.
 *
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA
 *
 */


#include <stdlib.h>
#include <glib.h>
#include <errno.h>
#include <sys/types.h>
#include <sys/socket.h>
#include <netinet/in.h>
#include <stdio.h>
#include <unistd.h>
#include <string.h>
#include <wait.h>
#include <unistd.h>
#include <sys/stat.h>
#include <fcntl.h>
#include <netdb.h>
#include <assert.h>
#include <time.h>
#include <stdlib.h>

#include <gtk/gtk.h>
#include <glade/glade.h>
#include <glib.h>
#include <libgnomevfs/gnome-vfs.h>

#include "response.h"
#include "command.h"
#include "client.h"
#include "archive_window.h"
#include "archive_save_window.h"
#include "main.h"
#include "archive.h"

static GSList *archive_files = NULL;
static gboolean get_finished = FALSE;
static gchar *archive_path = NULL;
static guint archive_count = 0;
static gboolean already_saved = FALSE;
static gboolean user_specified = FALSE; /* Whether the current path is user specified */

gchar *
archive_get_archive_path ()
{
  static const gchar *tmpdir;
  if (tmpdir == NULL)
    {
      tmpdir = g_getenv("TMPDIR");
      if (tmpdir == NULL)
        {
	  tmpdir = "/tmp";
	}
    }
  /* TODO: Real path generation here ... */
  if (archive_path == NULL)
    {
      /* Code mostly stolen from strftime(3) man page */
      time_t t;
      struct tm *tmp;
      gchar time_str[15];

      /* Get the time_t for the current time */
      t = time (NULL);

      /* Get the split version */
      tmp = localtime (&t);

      /* Format the time into YYYYMMMDDDHHMMSS */
      strftime (time_str, sizeof (time_str),
          "%Y%m%d%H%M%S", tmp);

      archive_path = g_strdup_printf ("%s/oprofile-archive-%s/", tmpdir, time_str);
      user_specified = FALSE;
    }

  return archive_path;
}

void
archive_set_archive_path (gchar *path)
{
  g_free (archive_path);

  if (path == NULL)
    archive_path = NULL;
  else
    archive_path = g_strdup (path);

  user_specified = TRUE;
}

void
archive_fully_downloaded ()
{
  archive_window_finished ();
  g_slist_free (archive_files);
  archive_files=NULL;

  main_archive_finished_cb (TRUE, NULL);
}

void
archive_get_next_file ()
{
  static GSList *cur_entry = NULL;

  if (get_finished)
    {
      /* Important reset the cur_entry pointer */
      cur_entry = NULL;
      archive_fully_downloaded ();
      return;
    }

  if (cur_entry == NULL)
    {
      cur_entry = archive_files;
    }

  /* Fire off a request to download the next file */
  client_send_command_get (cur_entry->data);
  archive_window_file_get_started (cur_entry->data);

  if (g_slist_next (cur_entry) == NULL)
    {
      get_finished = TRUE;
    } else {
      cur_entry = g_slist_next (cur_entry);
    }
}

void
archive_handle_response (struct response *reply)
{
  gchar **tmp;
  int i;

  tmp = g_strsplit (reply->payload, "\n", 0);

  for (i=0; i < g_strv_length (tmp); i++)
    {
      gchar *str = g_strdup(tmp[i]);

      g_strstrip (str);

      if (!g_str_equal (str, ""))
        archive_files = g_slist_append (archive_files, str);
    }

  archive_count = g_slist_length (archive_files);
  archive_window_file_list_finished (archive_count); 

  /* Bunch of initialisation stuff */

  /* Delete the old archive */
  archive_cleanup ();

  get_finished = FALSE;
  already_saved = FALSE;

  archive_set_archive_path (NULL);
  archive_get_next_file ();
  g_strfreev (tmp);
}

void
archive_file_got ()
{
  archive_window_file_get_finished();
  archive_get_next_file ();
}



gint
archive_save_progress_cb (GnomeVFSXferProgressInfo *info, gpointer user_data)
{

  archive_save_window_progress (info->file_index, info->files_total);

  /* Yield to gtk. */
  while (gtk_events_pending ())
    gtk_main_iteration ();
  
  return 1;
}

void
archive_save (gchar *path)
{
  GnomeVFSResult res;
  GnomeVFSURI *src_uri;
  GnomeVFSURI *dst_uri;

  gchar *archive_path = archive_get_archive_path ();

  gnome_vfs_init ();

  archive_save_window_show ();

  dst_uri = gnome_vfs_uri_new (gnome_vfs_get_uri_from_local_path(path));
  src_uri = gnome_vfs_uri_new (gnome_vfs_get_uri_from_local_path(archive_path));


  if (already_saved)
    {
      /* If we have saved before then we copy it */
      res = gnome_vfs_xfer_uri (src_uri, dst_uri, 
          GNOME_VFS_XFER_RECURSIVE, 
          GNOME_VFS_XFER_ERROR_MODE_ABORT,
          GNOME_VFS_XFER_OVERWRITE_MODE_REPLACE,
          archive_save_progress_cb, NULL);
    } else {
      /* Otherwise we can move it as an optimisation */
      res = gnome_vfs_xfer_uri (src_uri, dst_uri, 
          GNOME_VFS_XFER_RECURSIVE | GNOME_VFS_XFER_REMOVESOURCE, 
          GNOME_VFS_XFER_ERROR_MODE_ABORT,
          GNOME_VFS_XFER_OVERWRITE_MODE_REPLACE,
          archive_save_progress_cb, NULL);
    }

  if (res != GNOME_VFS_OK)
    {
      const gchar *err_string = gnome_vfs_result_to_string (res);

      printf ("GNOME-VFS: error %s", err_string);
    }

  archive_save_window_finished ();

  /* Set the new archive path */
  archive_set_archive_path (path);

  /* Mark that we have done one save already */
  already_saved = TRUE;
}

void
archive_load (gchar *path)
{
  /* Cleanup the old archive */
  archive_cleanup ();

  archive_set_archive_path (path);
}

void
archive_cleanup ()
{
  GnomeVFSResult res;
  GnomeVFSURI *src_uri;
  GList uri_list;

  if (archive_path == NULL)
    return;

  if (user_specified)
    return;

  gnome_vfs_init ();

  src_uri = gnome_vfs_uri_new (gnome_vfs_get_uri_from_local_path(archive_path));

  uri_list.data = src_uri;
  uri_list.next = NULL;
  uri_list.prev = NULL;

  res = gnome_vfs_xfer_delete_list (&uri_list,
      GNOME_VFS_XFER_ERROR_MODE_ABORT,
      GNOME_VFS_XFER_EMPTY_DIRECTORIES,
      NULL, NULL);

  if (res != GNOME_VFS_OK)
    {
      const gchar *err_string = gnome_vfs_result_to_string (res);

      printf ("GNOME-VFS: error %s\n", err_string);
    }

  archive_set_archive_path (NULL);
}
