/* -*- Mode: C++; tab-width: 4; indent-tabs-mode: nil; c-basic-offset: 2 -*- */
/* vim:set ts=2 sw=4 sts=2 et cindent: */
/* ***** BEGIN LICENSE BLOCK *****
 * Version: MPL 1.1
 *
 * The contents of this file are subject to the Mozilla Public License Version
 * 1.1 (the "License"); you may not use this file except in compliance with
 * the License. You may obtain a copy of the License at
 * http://www.mozilla.org/MPL/
 *
 * Software distributed under the License is distributed on an "AS IS" basis,
 * WITHOUT WARRANTY OF ANY KIND, either express or implied. See the License
 * for the specific language governing rights and limitations under the
 * License.
 *
 * The Original Code is the MICROB EAL package.
 *
 * The Initial Developer of the Original Code is Nokia Corporation.
 * Portions created by the Initial Developer are Copyright (C) 2005
 * the Initial Developer. All Rights Reserved.
 *
 * Contact: Oleg Romashin <oleg.romashin@nokia.com>
 *
 * ***** END LICENSE BLOCK ***** */

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include "common.h"
#include "gmozillatransferitem.h"
#include "gmozillamarshal.h"
#include "gmozillaengine.h"
#include "gmozillaweb.h"

// class and instance initialization
enum
{
    TRANSFER_ITEM_STARTED_SIGNAL,
    TRANSFER_ITEM_COMPLETED_SIGNAL,
    TRANSFER_ITEM_ABORTED_SIGNAL,
    TRANSFER_ITEM_ERROR_SIGNAL,
    TRANSFER_ITEM_PROGRESS_SIGNAL,
    TRANSFER_ITEM_LAST_SIGNAL
};

static guint transfer_item_signals [TRANSFER_ITEM_LAST_SIGNAL] = { 0 };

// progress timeout stuff - network down errors.
#define PROGRESS_TOLERANCY_INTERVAL 60000
static gboolean progress_timeout (GMozillaTransferItem *);
void reset_progress_timeout (GMozillaTransferItem *);

// global download var
extern gpointer global_last_gtk_moz_embed_download_object;

// global web
static GObject *webglobal = NULL;

// connecting signals from engine.
static gboolean g_mozilla_transfer_item_connect_signals (GMozillaTransferItem *self);

// callbacks
static void start_download_cb (GtkMozEmbedDownload *donwload, gchar **file_name_with_path, GMozillaTransferItem *self);
static void error_download_cb (GtkMozEmbedDownload *donwload, GMozillaTransferItem *self);
static void aborted_download_cb (GtkMozEmbedDownload *donwload, GMozillaTransferItem *self);
static void completed_download_cb (GtkMozEmbedDownload *donwload, GMozillaTransferItem *self);
static void progress_download_cb (GtkMozEmbedDownload *donwload, glong current, glong total, glong mode, GMozillaTransferItem *self);

static void
g_mozilla_transfer_item_destroy (GMozillaTransferItem *self)
{
    TRACE_LOG();
    if (self->item)
    {
        // gtk_mozilla_web_drop_transfer(GTK_MOZILLAWEBTRANSFERITEM(self->item));
        EAL_IF_GFREE(self->file_name);
        gtk_object_destroy(GTK_OBJECT(self->item));
        self->item = NULL;
    }
}

static void
g_mozilla_transfer_item_start (GMozillaTransferItem *self)
{
    TRACE_LOG();
    // gtk_mozilla_web_transferitem_start(GTK_MOZILLAWEBTRANSFERITEM(self->item));
}

static void
g_mozilla_transfer_item_stop (GMozillaTransferItem *self)
{
    TRACE_LOG();
    if (self->gtk_moz_embed_download)
        gtk_moz_embed_download_do_command (self->gtk_moz_embed_download, GTK_MOZ_EMBED_DOWNLOAD_PAUSE);
    // gtk_mozilla_web_transferitem_stop(GTK_MOZILLAWEBTRANSFERITEM(self->item));

    // reset progress timeout.
    reset_progress_timeout (self);
}

static GWebStatus
g_mozilla_transfer_item_continue (GMozillaTransferItem *self)
{
    TRACE_LOG();

    // fix for bug 55796
    if (!g_mozilla_connectivity_is_connected())
        if (g_mozilla_connectivity_connect(self, FALSE))
            return G_WEB_ERR;

    if (self->gtk_moz_embed_download)
        gtk_moz_embed_download_do_command (self->gtk_moz_embed_download, GTK_MOZ_EMBED_DOWNLOAD_RESUME);

    // init progress timeout.
    init_progress_timeout (self);

    /* if(gtk_mozilla_web_transferitem_continue(GTK_MOZILLAWEBTRANSFERITEM(self->item)) == GTK_MOZILLA_STATUS_OK) { 
       return G_WEB_STATUS_OK;
       }*/
    return G_WEB_ERR;
}

static void
g_mozilla_transfer_item_cancel (GMozillaTransferItem *self)
{
    TRACE_LOG();
    //if(self->item)
    if (self && self->gtk_moz_embed_download) {
        // work around for bugs 48621 and 54465
        {
            gtk_moz_embed_download_do_command (self->gtk_moz_embed_download, GTK_MOZ_EMBED_DOWNLOAD_RESUME);
        }
        gtk_moz_embed_download_do_command (self->gtk_moz_embed_download, GTK_MOZ_EMBED_DOWNLOAD_CANCEL);
    }

    if (webglobal && G_MOZILLA_WEB (webglobal)->ongoing_download_list)
        G_MOZILLA_WEB (webglobal)->ongoing_download_list = 
            g_list_remove (G_MOZILLA_WEB (webglobal)->ongoing_download_list, self);

}

static void
g_mozilla_transfer_item_reload (GMozillaTransferItem *self)
{
    TRACE_LOG();
    // gtk_mozilla_web_transferitem_reload(GTK_MOZILLAWEBTRANSFERITEM(self->item));
}

static GObject*
g_mozilla_transfer_item_get_web_item (GMozillaTransferItem *self)
{
    TRACE_LOG();
    //return self->item;
    return (GObject *) self;
}

static const gchar*
g_mozilla_transfer_item_get_url (GMozillaTransferItem *self)
{
    TRACE_LOG();
    if (self->gtk_moz_embed_download)
        return gtk_moz_embed_download_get_url (self->gtk_moz_embed_download);

    return NULL;
    // return gtk_mozilla_web_transferitem_get_url(GTK_MOZILLAWEBTRANSFERITEM(self->item));
    //return self->gtk_moz_embed_download->data->server;
}

static const gchar*
g_mozilla_transfer_item_get_filename (GMozillaTransferItem *self)
{
    TRACE_LOG();
    if (self->gtk_moz_embed_download)
        return gtk_moz_embed_download_get_file_name (self->gtk_moz_embed_download);

    return NULL;
    // return gtk_mozilla_web_transferitem_get_filename(GTK_MOZILLAWEBTRANSFERITEM(self->item));
}

static glong
g_mozilla_transfer_item_get_total_size (GMozillaTransferItem *self)
{
    TRACE_LOG();
    if (self->gtk_moz_embed_download)
        return gtk_moz_embed_download_get_file_size (self->gtk_moz_embed_download);

    return 0;
    // return gtk_mozilla_web_transferitem_get_total_size(GTK_MOZILLAWEBTRANSFERITEM(self->item));
}

static glong
g_mozilla_transfer_item_get_progress (GMozillaTransferItem *self)
{
    TRACE_LOG();
    if (self->gtk_moz_embed_download)
        return gtk_moz_embed_download_get_progress (self->gtk_moz_embed_download);

    return 0;
    // return gtk_mozilla_web_transferitem_get_progress(GTK_MOZILLAWEBTRANSFERITEM(self->item));
}

static void
g_mozilla_transfer_item_interface_init (GWebTransferItemIface *iface)
{
    TRACE_LOG();
    iface->destroy        = (void (*) (GWebTransferItem*))
        g_mozilla_transfer_item_destroy;
    iface->start          = (void (*) (GWebTransferItem*))
        g_mozilla_transfer_item_start;
    iface->stop           = (void (*) (GWebTransferItem*))
        g_mozilla_transfer_item_stop;
    iface->do_continue    = (GWebStatus (*) (GWebTransferItem*))
        g_mozilla_transfer_item_continue;
    iface->cancel         = (void (*) (GWebTransferItem*))
        g_mozilla_transfer_item_cancel;
    iface->reload         = (void (*) (GWebTransferItem*))
        g_mozilla_transfer_item_reload;
    iface->get_web_item   = (GObject* (*) (GWebTransferItem*))
        g_mozilla_transfer_item_get_web_item;
    iface->get_url        = (const gchar* (*) (GWebTransferItem*))
        g_mozilla_transfer_item_get_url;
    iface->get_filename   = (const gchar* (*) (GWebTransferItem*))
        g_mozilla_transfer_item_get_filename;
    iface->get_total_size = (glong (*) (GWebTransferItem*))
        g_mozilla_transfer_item_get_total_size;
    iface->get_progress   = (glong (*) (GWebTransferItem*))
        g_mozilla_transfer_item_get_progress;
}

static void
g_mozilla_transfer_item_instance_init (GTypeInstance *instance, gpointer  g_class)
{
    TRACE_LOG();
    G_MOZILLA_TRANSFER_ITEM(instance)->item = NULL;
}

static void
g_mozilla_transfer_item_class_init (GMozillaTransferItemClass *klass)
{
    TRACE_LOG();

    GObjectClass *object_class = G_OBJECT_CLASS (klass);

    // set up maemo-eal/ui specific signals
    transfer_item_signals[TRANSFER_ITEM_STARTED_SIGNAL] =
        g_signal_new(G_WEB_TRANSFER_ITEM_SIGNAL_STARTED,
                     G_TYPE_FROM_CLASS(klass),
                     G_SIGNAL_RUN_FIRST,
                     G_STRUCT_OFFSET(GMozillaTransferItemClass, started),
                     NULL,
                     NULL,
                     g_cclosure_marshal_VOID__VOID,
                     G_TYPE_NONE,
                     0);
    transfer_item_signals[TRANSFER_ITEM_COMPLETED_SIGNAL] =
        g_signal_new (G_WEB_TRANSFER_ITEM_SIGNAL_COMPLETED,
                      G_OBJECT_CLASS_TYPE (object_class),
                      G_SIGNAL_RUN_FIRST,
                      G_STRUCT_OFFSET (GMozillaTransferItemClass, completed),
                      NULL, NULL,
                      g_cclosure_marshal_VOID__VOID,
                      G_TYPE_NONE,
                      0);
    transfer_item_signals[TRANSFER_ITEM_ERROR_SIGNAL] =
        g_signal_new (G_WEB_TRANSFER_ITEM_SIGNAL_ERROR,
                      G_OBJECT_CLASS_TYPE (object_class),
                      G_SIGNAL_RUN_FIRST,
                      G_STRUCT_OFFSET (GMozillaTransferItemClass, error),
                      NULL, NULL,
                      g_cclosure_marshal_VOID__VOID,
                      G_TYPE_NONE,
                      0);
    transfer_item_signals[TRANSFER_ITEM_ERROR_SIGNAL] =
        g_signal_new ("error",
                      G_OBJECT_CLASS_TYPE (object_class),
                      G_SIGNAL_RUN_FIRST,
                      G_STRUCT_OFFSET (GMozillaTransferItemClass, error),
                      NULL, NULL,
                      g_cclosure_marshal_VOID__VOID,
                      G_TYPE_NONE,
                      0);
    transfer_item_signals[TRANSFER_ITEM_ABORTED_SIGNAL] =
        g_signal_new ("aborted",
                      G_OBJECT_CLASS_TYPE (object_class),
                      G_SIGNAL_RUN_FIRST,
                      G_STRUCT_OFFSET (GMozillaTransferItemClass, aborted),
                      NULL, NULL,
                      g_cclosure_marshal_VOID__VOID,
                      G_TYPE_NONE,
                      0);
    transfer_item_signals[TRANSFER_ITEM_PROGRESS_SIGNAL] =
        g_signal_new (G_WEB_TRANSFER_ITEM_SIGNAL_PROGRESS,
                      G_OBJECT_CLASS_TYPE (object_class),
                      G_SIGNAL_RUN_FIRST,
                      G_STRUCT_OFFSET (GMozillaTransferItemClass, progress),
                      NULL, NULL,
                      g_mozeal_marshal_VOID__ULONG_ULONG_ULONG,
                      G_TYPE_NONE,
                      3,
                      G_TYPE_ULONG, G_TYPE_ULONG, G_TYPE_ULONG);

}

GType
g_mozilla_transfer_item_get_type (void)
{
    static GType type = 0;
    if (type == 0)
    {
        static const GTypeInfo info =
        {
            sizeof (GMozillaTransferItemClass),
            NULL,   /* base_init */
            NULL,   /* base_finalize */
            (GClassInitFunc) g_mozilla_transfer_item_class_init,   /* class_init */
            NULL,   /* class_finalize */
            NULL,   /* class_data */
            sizeof (GMozillaTransferItem),
            0,      /* n_preallocs */
            g_mozilla_transfer_item_instance_init    /* instance_init */
        };
        static const GInterfaceInfo iface_info =
        {
            (GInterfaceInitFunc) g_mozilla_transfer_item_interface_init,
            NULL,   /* interface_finalize */
            NULL    /* interface_data */
        };
        type = g_type_register_static (G_TYPE_OBJECT,
                                       "GMozillaTransferItem",
                                       &info, 0);
        g_type_add_interface_static (type,
                                     G_TYPE_WEB_TRANSFER_ITEM,
                                     &iface_info);
    }
    return type;
}

/* a macro for creating a new object of our type */
#define GET_NEW (g_object_new(g_mozilla_transfer_item_get_type(), NULL))

GObject*
g_mozilla_transfer_item_new (void)
{
    TRACE_LOG();
    GObject* instance = G_OBJECT(GET_NEW);

    return instance;
}

GObject*
g_mozilla_transfer_item_new_with_item (GObject *item)
{
    TRACE_LOG();

    GObject* instance = g_mozilla_transfer_item_new ();
    if (!instance) return NULL;

    // signal stuff
    g_mozilla_transfer_item_connect_signals (G_MOZILLA_TRANSFER_ITEM(instance));

    G_MOZILLA_TRANSFER_ITEM(instance)->item = item;
    return instance;
}

GObject*
g_mozilla_transfer_item_new_with_url (const gchar *url,
                                      const gchar *filename)
{
    TRACE_LOG();

    GObject* instance = g_mozilla_transfer_item_new ();
    if (!instance) return NULL;

    //FIXME XXX MEMLEAK
    if (global_last_gtk_moz_embed_download_object) {

        G_MOZILLA_TRANSFER_ITEM(instance)->gtk_moz_embed_download = 
            (GtkMozEmbedDownload *) malloc (sizeof (GtkMozEmbedDownload));
        G_MOZILLA_TRANSFER_ITEM(instance)->gtk_moz_embed_download = global_last_gtk_moz_embed_download_object;
        global_last_gtk_moz_embed_download_object = NULL;

    } else  {
        G_MOZILLA_TRANSFER_ITEM(instance)->gtk_moz_embed_download =
            (GtkMozEmbedDownload *) gtk_moz_embed_download_new_with_url_filename (url, filename);
    }

    if (filename)
        G_MOZILLA_TRANSFER_ITEM (instance)->file_name = g_strdup (filename); 
    else G_MOZILLA_TRANSFER_ITEM (instance)->file_name = NULL; 

    // signal stuff
    g_mozilla_transfer_item_connect_signals (G_MOZILLA_TRANSFER_ITEM(instance));

    return instance;
}

static gboolean
g_mozilla_transfer_item_connect_signals (GMozillaTransferItem *self)
{
    TRACE_LOG();

    if (!self || !self->gtk_moz_embed_download) 
        return FALSE;

    // connecting singals coming from engine, gtk_moz_embed_download object.
    g_signal_connect(G_OBJECT(self->gtk_moz_embed_download), "started", G_CALLBACK(start_download_cb), self);
    g_signal_connect(G_OBJECT(self->gtk_moz_embed_download), "progress", G_CALLBACK(progress_download_cb), self);
    g_signal_connect(G_OBJECT(self->gtk_moz_embed_download), "completed", G_CALLBACK(completed_download_cb), self); 
    // FIXME: Take it off, it UI does not support it.
    g_signal_connect(G_OBJECT(self->gtk_moz_embed_download), "aborted", G_CALLBACK(aborted_download_cb), self);
    g_signal_connect(G_OBJECT(self->gtk_moz_embed_download), "error", G_CALLBACK(error_download_cb), self);

    return TRUE;
}

static void 
start_download_cb (GtkMozEmbedDownload *donwload, gchar **file_name_with_path, GMozillaTransferItem *self)
{
    TRACE_LOG();

    g_signal_emit_by_name(G_OBJECT (self), G_WEB_TRANSFER_ITEM_SIGNAL_STARTED);
    if (self->file_name)
        *file_name_with_path = g_strdup (self->file_name);
    else *file_name_with_path = NULL;

    return ;
}

static void 
error_download_cb (GtkMozEmbedDownload *donwload, GMozillaTransferItem *self)
{
    TRACE_LOG();
    g_signal_emit_by_name(G_OBJECT (self), "error");

    return ;
}

static void 
aborted_download_cb (GtkMozEmbedDownload *donwload, GMozillaTransferItem *self)
{
    TRACE_LOG();
    g_signal_emit_by_name(G_OBJECT (self), "aborted");

    return ;
}

static void 
completed_download_cb (GtkMozEmbedDownload *donwload, GMozillaTransferItem *self)
{
    TRACE_LOG();
    g_signal_emit_by_name(G_OBJECT (self), G_WEB_TRANSFER_ITEM_SIGNAL_COMPLETED);

    // reset progress timeout.
    reset_progress_timeout (self);

    if (self->gtk_moz_embed_download) {
        if (self->gtk_moz_embed_download == global_last_gtk_moz_embed_download_object)
            global_last_gtk_moz_embed_download_object = NULL;
        gtk_object_destroy(GTK_OBJECT(self->gtk_moz_embed_download));
    }

    if (webglobal && G_MOZILLA_WEB (webglobal)->ongoing_download_list)
        G_MOZILLA_WEB (webglobal)->ongoing_download_list = g_list_remove (G_MOZILLA_WEB (webglobal)->ongoing_download_list, self);


    return ;
}

static void 
progress_download_cb (GtkMozEmbedDownload *donwload, glong current, glong total, glong mode, GMozillaTransferItem *self)
{
    TRACE_LOG();

    init_progress_timeout (self);

    // emit signal to get download progress and displays on download list dialog
    g_signal_emit_by_name(G_OBJECT (self), G_WEB_TRANSFER_ITEM_SIGNAL_PROGRESS, current, total, 1);
    return ;
}

gboolean progress_timeout (GMozillaTransferItem *self)
{
    TRACE_LOG();

    // reset progress timeout.
    reset_progress_timeout (self);

    // forcing download to get paused
    g_signal_emit_by_name (G_OBJECT (self), "error");
    return FALSE;
}

void init_progress_timeout (GMozillaTransferItem *self)
{
    TRACE_LOG();

    // reset progress timeout.
    reset_progress_timeout (self);

    self->progress_timer_id = g_timeout_add (PROGRESS_TOLERANCY_INTERVAL, (GSourceFunc)
                                             progress_timeout, (gpointer) self);
}

void reset_progress_timeout (GMozillaTransferItem *self)
{
    TRACE_LOG();

    if (self->progress_timer_id) 
    {
        g_source_remove(self->progress_timer_id);
        self->progress_timer_id = 0;
    }
}

void g_mozilla_transfer_item_set_webglobal (GObject *o)
{
    TRACE_LOG();

    if (!webglobal)
        webglobal = o;
}
