/**
 * This file is part of maemo-examples package
 * 
 * Copyright (c) 2007 Nokia Corporation. All rights reserved.
 * 
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 * 
 * The above copyright notice and this permission notice shall be included in
 * all copies or substantial portions of the Software.
 * 
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
 * THE SOFTWARE. */


#include <gtk/gtk.h>
#include <alarm_event.h>
#include <dbus/dbus-glib.h>
#include <dbus/dbus-glib-bindings.h>
#include <time.h>
#include <string.h>

#include "example_common.h"

/* Define structure app-wide data*/
typedef struct
{
	HildonProgram *program;
	HildonWindow  *window;

	DBusGConnection *bus;
} AppData;

/* Define new GObject structure
 * HelloAlarm */
typedef struct
{
	GObject parent;
	/* Define a pointer to store
	 * appdata */
	AppData *appdata;
} HelloAlarm;

/* Define HelloAlarm's class */
typedef struct
{
	GObjectClass parent;

	/* HelloAlarm's say_hello-method.
	 * This will be registered as a DBUS-method */
	void (*say_hello)(HelloAlarm *self);
} HelloAlarmClass;

/* Callback to be called when application recieves
 * alarm message from the alarmd */
void hello_alarm_say_hello(HelloAlarm *self)
{
	GtkWidget *dialog;

	dialog = gtk_message_dialog_new(
			GTK_WINDOW(self->appdata->window),
			GTK_DIALOG_DESTROY_WITH_PARENT,
			GTK_MESSAGE_INFO,
			GTK_BUTTONS_OK,
			"Hello Alarm!");

	gtk_dialog_run(GTK_DIALOG(dialog));
	gtk_widget_destroy(dialog);
}
/* This file is generated from example_alarm_dbus.xml-file
 * using dbus-binding-tool and is needed to get the object's
 * DBUS export info (dbus_glib_hello_alarm_object_info).
 * See the Makefile for how the tool is used */
#include "example_alarm_dbus.h"

/* Initialize class data for HelloAlarm */
static void hello_alarm_class_init(HelloAlarmClass *klass)
{
	/* Define implementation location of the
	 * say_hello-method */
	klass->say_hello = hello_alarm_say_hello;
}

/* Get GType of the HelloAlarm object */
GType hello_alarm_get_type()
{
	static GType type = 0;
	/* If the type is already registered,
	 * return it, otherwise register the
	 * GTypeInfoStructure */
	if(!type)
	{
		static const GTypeInfo info = 
		{
			sizeof(HelloAlarmClass),
			NULL, /* base_init */
			NULL, /* base_finalize */
			(GClassInitFunc)hello_alarm_class_init, /* class_init */
			NULL, /* class_finalize */
			NULL, /* class_data */
			sizeof(HelloAlarm),
			0, /* n_preallocs */
			NULL, /* instance_init */
		};

		type = g_type_register_static(G_TYPE_OBJECT, "HelloAlarmType",
			&info, 0);
	}
	return type;
}

/* Tell alarmd to send a alarm */
static void set_alarm(GtkWidget *button, AppData *appdata)
{
	alarm_event_t event;
	time_t current_time;
	struct tm *alarm_time;

	/* Reset the event-structure to zero */
	memset(&event, 0, sizeof(event));
	
	/* Get current time and add two seconds to it,
	 * meaning that alarm should be sent at
	 * this time */
	time(&current_time);
	alarm_time = localtime(&current_time);
	alarm_time->tm_sec += 2;
	/* Set the alarm time */
	event.alarm_time = mktime(alarm_time);
	
	/* Set interface, service name, path and method
	 * name of the HelloAlarm D-BUS interface */
	event.dbus_interface = "org.maemo.alarm_example";
	event.dbus_service = "org.maemo.alarm_example";
	event.dbus_path = "/org/maemo/alarm_example";
	event.dbus_name = "hello_alarm";
	
	/* The application presents its own dialog,
	 * so the system provided isn't needed */
	event.flags = ALARM_EVENT_NO_DIALOG;

	/* Add the alarm to alarmd and
	 * verify that it was correctly created */
	if(!alarm_event_add(&event))
	{
		hildon_banner_show_information(button, "gtk-dialog-error",
			       "Couldn't set alarm");
		g_warning("Couldn't set alarm. Error code %d\n", alarmd_get_error());
		return;
	}

	hildon_banner_show_information(button, NULL,
			"Alarm set");
}

/* Register GObject's D-BUS interface and
 * request a name for the service */
static gboolean
register_service(DBusGConnection *bus, GObject *obj,
		const DBusGObjectInfo *info,
		gchar *name, gchar *path, GError **error)
{
	DBusGProxy *system;
	gboolean result;
	
	/* Install GObject's info so glib knows
	 * what kind of interface to present on
	 * D-BUS */
	dbus_g_object_type_install_info(G_TYPE_FROM_INSTANCE(obj),
			info);
	/* Register the object to D-BUS using
	 * given object path */
	dbus_g_connection_register_g_object(bus,
			path, obj);
	
	/* Get proxy D-BUS-service */
	system = dbus_g_proxy_new_for_name(bus,
			DBUS_SERVICE_DBUS,
			DBUS_PATH_DBUS,
			DBUS_INTERFACE_DBUS);

	/* Request a name for the service from D-BUS */
	result = org_freedesktop_DBus_request_name(system,
			name, 0, NULL, error);

	/* Check that the name request went OK */
	if(!result)
	{
		g_object_unref(system);
		return FALSE;
	}

	g_object_unref(system);
	return TRUE;
}

int main(int argc, char **argv)
{
	AppData appdata;
	GError *error = NULL;
	GtkWidget *button;
	HelloAlarm *server;

	/* Initialize the program */
	example_gui_initialize(&appdata.program, &appdata.window,
			&argc, &argv, "alarm-example");

	/* Connect to D-BUS session bus and verify that
	 * it succeeded */
	appdata.bus = dbus_g_bus_get(DBUS_BUS_SESSION, &error);
	if(!appdata.bus)
	{
		g_error("Couldn't get DBUS connection: %s\n", error->message);
		g_error_free(error);
		return 1;
	}

	/* Create and initialize a HelloAlarm object and
	 * register it to D-BUS */
	server = g_object_new(hello_alarm_get_type(), NULL);
	server->appdata = &appdata;
	register_service(appdata.bus, G_OBJECT(server),
			&dbus_glib_hello_alarm_object_info,
			"org.maemo.alarm_example",
			"/org/maemo/alarm_example", &error);

	/* Check that the registration was successfull */
	if(error)
	{
		g_error("Couldn't register service: %s", error->message);
		g_error_free(error);
		return 1;
	}

	/* Create a button and connect its clicked-event
	 * to set_alarm-function */
	button = gtk_button_new_with_label("Set alarm");
	gtk_container_add(GTK_CONTAINER(appdata.window), button);
	g_signal_connect(G_OBJECT(button), "clicked",
			G_CALLBACK(set_alarm), &appdata);

	/* Run the mainloop */
	example_gui_run(appdata.program, appdata.window);
	
	/* Disconnect from the dbus */
	dbus_g_connection_unref(appdata.bus);

	return 0;
}
