/**
 * This file is part of maemo-examples package
 * 
 * Copyright (c) 2007 Nokia Corporation. All rights reserved.
 * 
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 * 
 * The above copyright notice and this permission notice shall be included in
 * all copies or substantial portions of the Software.
 * 
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
 * THE SOFTWARE. */


#include <libosso.h>
#include <libebook/e-book.h>
#include <libosso-abook/osso-abook.h>
#include <gtk/gtk.h>

#include "example_common.h"

/* Define structure for data that's passed around
 * the application */
typedef struct
{
	HildonProgram *program;
	HildonWindow *window;

	osso_context_t *osso_context;

	EBookView *book_view;
	OssoABookContactModel *contact_model;
	GtkWidget *contact_view;
} AppData;

/* Callback to be called when user activates (double clicks) a contact
 * in the contact view */
static void contact_activated(OssoABookContactView *view, EContact *contact,
		AppData *appdata)
{
	GtkWidget *starter;

	/* Create new contact starter dialog */
	starter = osso_abook_contact_starter_new();
	/* Contact starter needs a book view to exchange
	 * signals with */
	osso_abook_contact_starter_set_book_view(OSSO_ABOOK_CONTACT_STARTER(starter),
			appdata->book_view);
	/* Specify the contact that is "started" */
	osso_abook_contact_starter_set_contact(OSSO_ABOOK_CONTACT_STARTER(starter), contact);
	
	/* Run dialog and destroy it afterwards. The dialog works "independently", ie
	 * no action or response value checking is needed by the client side code */
	gtk_dialog_run(GTK_DIALOG(starter));
	
	gtk_widget_destroy(GTK_WIDGET(starter));
}

/* Initialize the contact list connections and widgets */
static void create_contact_list(AppData *appdata)
{
	EBook *book;
	GError *error = NULL;
	EBookQuery *query;

	/* Request a handle to the system-wide addressbook
	 * The book isn't opened yet. */
	book = e_book_new_system_addressbook(&error);
	if(!book)
	{
		g_critical("Couldn't open addressbook: %s", error->message);
		g_error_free(error);
		return;
	}
	
	/* Open connection to the address book */
	if(!e_book_open(book, FALSE, &error))
	{
		g_critical("Couldn't open addressbook: %s", error->message);
		g_error_free(error);
		return;
	}

	/* Create a query. This query matches any
	 * records in the book. */
	query = e_book_query_any_field_contains("");

	/* Create new book view with the book and the query */
	if(!e_book_get_book_view(book, query, NULL, -1, &appdata->book_view, &error))
	{
		g_critical("Couldn't create book view: %s", error->message);
		g_error_free(error);
		return;
	}
	e_book_query_unref(query);
	
	/* Start the EBookView, this runs the query and
	 * starts to exchange signals with the EBook */
	e_book_view_start(appdata->book_view);

	/* Create a new data model for the abook contact widget */
	appdata->contact_model = osso_abook_contact_model_new();
	/* Set the model to use the book view as a data source */
	osso_abook_tree_model_set_book_view(OSSO_ABOOK_TREE_MODEL(appdata->contact_model),
			appdata->book_view);

	/* Create new basic contact view using the model */
	appdata->contact_view = osso_abook_contact_view_new_basic(appdata->contact_model);
	g_object_unref(appdata->contact_model);

	/* Connect contact-activated signal to a callback */
	g_signal_connect(G_OBJECT(appdata->contact_view), "contact-activated",
			G_CALLBACK(contact_activated), appdata);

}

/* Callback to be called when user clicks the "Add Contact" button */
static void add_contact(GtkWidget *button, AppData *appdata)
{
	/* This is all that's needed to use the contact adding dialog,
	 * the new users created will be shown automatically in the
	 * contact_view widget */
	osso_abook_add_contact_dialog_run(GTK_WINDOW(appdata->window),
			appdata->book_view);
}

/* Callback to be called when user clicks the "Delete Contact" button */
static void delete_contacts(GtkWidget *button, AppData *appdata)
{
	GList *selection;

	/* Get selected items from the contact view */
	selection = osso_abook_contact_view_get_selection(
			OSSO_ABOOK_CONTACT_VIEW(appdata->contact_view));
	/* Pass the selection to delete contacts dialog */
	osso_abook_delete_contacts_dialog_run(GTK_WINDOW(appdata->window),
			appdata->book_view, selection);

	g_list_free(selection);
}

#define APP_NAME "maemo-abook-example"

int main(int argc, char **argv)
{
	AppData appdata;
	GtkWidget *add_button;
	GtkWidget *delete_button;
	GtkWidget *contact_box;
	GtkWidget *button_box;
	
	/* Create and initialize the HildonProgram and HildonWindow */
	example_gui_initialize(&appdata.program, &appdata.window,
			&argc, &argv, APP_NAME);

	/* Initialize the osso context */
	appdata.osso_context = osso_initialize(
			APP_NAME, "1.0", TRUE, NULL);
	if(!appdata.osso_context)
	{
		g_error("Couldn't initialize osso context");
		return 1;
	}
	
	/* Initialize abook, which also initializes all the
	 * libraries it needs (GTK+, Galago, Gnome-VFS) */
	osso_abook_init(&argc, &argv, appdata.osso_context);

	
	/* Create the contact list widget and rest of the GUI */
	create_contact_list(&appdata);
	add_button = gtk_button_new_with_label("Add contact");
	delete_button = gtk_button_new_with_label("Delete contact");
	contact_box = gtk_vbox_new(FALSE, 8);
	button_box = gtk_hbox_new(FALSE, 8);
	gtk_box_pack_start_defaults(GTK_BOX(contact_box),
			GTK_WIDGET(appdata.contact_view));
	gtk_box_pack_start_defaults(GTK_BOX(button_box),
			GTK_WIDGET(add_button));
	gtk_box_pack_start_defaults(GTK_BOX(button_box),
			GTK_WIDGET(delete_button));

	gtk_box_pack_start_defaults(GTK_BOX(contact_box),
			GTK_WIDGET(button_box));
	
	gtk_container_add(GTK_CONTAINER(appdata.window),
			GTK_WIDGET(contact_box));

	g_signal_connect(G_OBJECT(add_button), "clicked",
			G_CALLBACK(add_contact), &appdata);
	g_signal_connect(G_OBJECT(delete_button), "clicked",
			G_CALLBACK(delete_contacts), &appdata);

	
	/* Run the application */
	example_gui_run(appdata.program, appdata.window);

	/* Deinitialize and free allocated resources */
	e_book_view_stop(appdata.book_view);
	g_object_unref(appdata.book_view);
	
	osso_deinitialize(appdata.osso_context);
	return 0;
}
