/**
    UPnP/AV GnomeVFS browser (Command Line)

    This browser is used for testing UPnP/AV plugin from command line in order
    to confirm the quality of the plugin.
    
    Copyright 2006 Nokia Corporation. All rights reserved.
	
    Contact: Aapo Makela <aapo.makela@nokia.com>
    
    This library is free software; you can redistribute it and/or modify it
    under the terms of the GNU Lesser General Public License version 2.1 as 
    published by the Free Software Foundation.
  
    This library is distributed in the hope that it will be useful, but
    WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU Lesser
    General Public License for more details.

    You should have received a copy of the GNU Lesser General Public License
    along with this library; if not, write to the Free Software Foundation,
    Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
*/

#include <stdlib.h>
#include <time.h>
#include <strings.h>
#include <string.h>

#include <glib.h>
#include <libgnomevfs/gnome-vfs.h>

#include <dbus/dbus.h>
#include <dbus/dbus-glib-lowlevel.h>

#ifdef GUPNP_USE_MAEMO
#include "player.h"
#endif

#define BUF_SIZE 16384

#define ERROR_UNKNOWN_COMMAND "Unknown command. See 'help' for help."
#define UPNP_AV_PROTOCOL "upnpav://"
#define FILE_PROTOCOL "file://"
#define COMMAND_DELIM ";"
#define ERROR_STR     "UPNPAV ERROR"

typedef struct
{
  GMainLoop *mainloop;
  GIOChannel *stdin;
  guint source_id;
  
  gchar *cur_dir;
  gboolean exit;
  
  GnomeVFSMonitorHandle *monitor;
  DBusConnection *sys_conn;
  DBusConnection *ses_conn;
  
  GQueue *command_queue;
  
} BrowserInfo;

typedef gboolean (*CommandFunc) (BrowserInfo *browser, gint argc, const gchar **args);

typedef struct
{
  gint argc;
  gchar **args;
  CommandFunc command;
  
} CommandInfo;

static gboolean command_ls(BrowserInfo *browser, gint argc, const gchar **args);
static gboolean command_cd(BrowserInfo *browser, gint argc, const gchar **args);
static gboolean command_get(BrowserInfo *browser, gint argc, const gchar **args);
static gboolean command_exit(BrowserInfo *browser, gint argc, const gchar **args);
static gboolean command_play(BrowserInfo *browser, gint argc, const gchar **args);
static gboolean command_stop(BrowserInfo *browser, gint argc, const gchar **args);
static gboolean command_sleep(BrowserInfo *browser, gint argc, const gchar **args);
static gboolean command_help(BrowserInfo *browser, gint argc, const gchar **args);
static gboolean command_exists(BrowserInfo *browser, gint argc, const gchar **args);


typedef struct
{
  const gchar *str;
  CommandFunc command;
  
} CommandMapping;

static CommandMapping commands[] = {
    { .str = "ls", 
      .command = command_ls },
    { .str = "cd",
      .command = command_cd },
    { .str = "get",
      .command = command_get },
    { .str = "exit",
      .command = command_exit },
    { .str = "sleep",
      .command = command_sleep },
    { .str = "help",
      .command = command_help },
    { .str = "play",
      .command = command_play },
    { .str = "stop",
      .command = command_stop },
    { .str = "exists",
      .command = command_exists },
    { .str = NULL,
      .command = NULL } };


static void print_prompt(BrowserInfo *browser)
{
  gchar *tmp_str = gnome_vfs_unescape_string(browser->cur_dir, NULL);
  g_print("%s%s > ", UPNP_AV_PROTOCOL, tmp_str);
  g_free(tmp_str);
}


static void directory_monitor(GnomeVFSMonitorHandle *handle,
                              const gchar *monitor_uri,
                              const gchar *info_uri,
                              GnomeVFSMonitorEventType event_type,
                              gpointer user_data)
{
  BrowserInfo *browser = (BrowserInfo*)user_data;
  GnomeVFSURI *uri1, *uri2;
  gchar *uri_str = NULL;
  static const gchar *prev_dir[] =
    { ".." };
  static const gchar *root_dir[] =
    { "/" };
    
  g_return_if_fail(browser != NULL);
  
  uri_str = g_strconcat(UPNP_AV_PROTOCOL, browser->cur_dir, NULL);
  uri1 = gnome_vfs_uri_new(info_uri);
  uri2 = gnome_vfs_uri_new(uri_str);
  g_free(uri_str); uri_str = NULL;
  
  g_print("Monitor event: %s - ", info_uri);
  
  switch(event_type)
  {
    case GNOME_VFS_MONITOR_EVENT_CHANGED:
      g_print("CHANGED\n");
      break;
    case GNOME_VFS_MONITOR_EVENT_DELETED:
      g_print("DELETED\n");
      break;
    case GNOME_VFS_MONITOR_EVENT_CREATED:
      g_print("CREATED\n");
      break;
    case GNOME_VFS_MONITOR_EVENT_METADATA_CHANGED:
      g_print("CHANGED\n");
      break;
    default:
      break;
  }
    
  if (event_type == GNOME_VFS_MONITOR_EVENT_DELETED)
  {
    if (gnome_vfs_uri_equal(uri1, uri2))
    {
      /* Our directory was deleted, go to previous directory */
      if (command_cd(browser, 1, prev_dir) == FALSE)
      {
        /* As a final action: go to root */
        command_cd(browser, 1, root_dir);
      }
    }
  }
}


static gchar *build_path(BrowserInfo *browser, const gchar *path)
{
  gchar *new_path;
  
  if (path[0] == '/') /* Absolute path */
  {
    new_path = g_strdup(path);
    
  } else if (strcmp(path, "..") == 0)  /* Special path */
  {
    new_path =  g_path_get_dirname( browser->cur_dir );
    
  } else { /* Relative path */
    
    new_path = g_build_path("/", browser->cur_dir, path, NULL);
  }
  
  return new_path;
}


static gboolean command_ls(BrowserInfo *browser, gint argc, const gchar **args)
{
  GnomeVFSResult result;
  GnomeVFSDirectoryHandle *dir_handle;
  GnomeVFSFileInfoOptions options = 0;
  GnomeVFSFileInfo *file_info;
  gint idx = 0;
  gchar *uri_str = NULL, *tmp_str = NULL;
  GSList *directories = NULL, *tmp = NULL;
  
  if (argc > 0)
  {
    for (idx = 0; idx < argc; idx++)
    {
      if (strcasecmp(args[idx], "--mime") == 0)
      {
        options = options | GNOME_VFS_FILE_INFO_GET_MIME_TYPE;
        
      } else if (strcasecmp(args[idx], "--links") == 0)
      {
        options = options | GNOME_VFS_FILE_INFO_FOLLOW_LINKS;
        
      } else if (g_str_has_prefix(args[idx], "-")) {
        g_print("%s: Unknown option %s. Possible options '--links' and '--mime'\n",
                ERROR_STR, args[idx]);
        return FALSE;
      } else  {
        directories = g_slist_append(directories, build_path(browser, args[0]));
      }
    }
  }
  
  if (directories == NULL)
  {
    directories = g_slist_append(directories, g_strdup(browser->cur_dir));
  }
  
  g_print("\n");
  
  tmp = directories;
  while (tmp != NULL)
  {
    uri_str = g_strconcat(UPNP_AV_PROTOCOL, tmp->data, NULL);
    result = gnome_vfs_directory_open(&dir_handle,
                                      uri_str,
                                      options);
  
    if (result == GNOME_VFS_OK)
    {
      file_info = gnome_vfs_file_info_new();
    
      while (gnome_vfs_directory_read_next(dir_handle,
                                           file_info) == GNOME_VFS_OK)
      {
        tmp_str = gnome_vfs_escape_string(file_info->name);
        g_print("%s\t\t\t%lld\t[%s]\t\n", tmp_str, 
                file_info->size,
                file_info->mime_type);
        if (GNOME_VFS_FILE_INFO_SYMLINK (file_info))
        {
          g_print("\t==>%s\n", file_info->symlink_name);
        }
        g_free(tmp_str); tmp_str = NULL;
      }
    
      gnome_vfs_file_info_unref(file_info);
      gnome_vfs_directory_close(dir_handle);  
    
    } else {
      g_print("%s: %s: %s\n", ERROR_STR, (gchar*)tmp->data, gnome_vfs_result_to_string(result));
    }
    
    g_free(uri_str);
    g_free(tmp->data);
    tmp = tmp->next;
  }
  
  g_slist_free(directories);
  
  return TRUE;
}


static gboolean command_cd(BrowserInfo *browser, gint argc, const gchar **args)
{
  gchar *new_dir = NULL;
  gchar *uri_str = NULL;
  GnomeVFSFileInfo *file_info;
  GnomeVFSResult result;
  
  if (argc == 0 || strlen(args[0]) < 1) 
  {
    g_print("%s\n", browser->cur_dir);
    return FALSE;
  }
  
  new_dir = build_path(browser, args[0]);
  
  uri_str = g_strconcat(UPNP_AV_PROTOCOL, new_dir, NULL);
  file_info = gnome_vfs_file_info_new ();
  result = gnome_vfs_get_file_info(uri_str, file_info, 
                                   GNOME_VFS_FILE_INFO_DEFAULT);
  
  if (result != GNOME_VFS_OK || 
      file_info->type != GNOME_VFS_FILE_TYPE_DIRECTORY)
  {
    if (file_info->type != GNOME_VFS_FILE_TYPE_DIRECTORY)
    {
      g_print("%s: %s: Not a directory.\n", ERROR_STR, args[0]);
    } else {
      g_print("%s: %s: %s\n", ERROR_STR, args[0], gnome_vfs_result_to_string(result));
    }
    
    g_free(uri_str); uri_str = NULL;
    gnome_vfs_file_info_unref (file_info);    
    g_free(new_dir);
    return FALSE;
  }
  
  gnome_vfs_file_info_unref (file_info);    
    
  /* We need to change our monitors */
  if (browser->monitor != NULL)
  {
    gnome_vfs_monitor_cancel(browser->monitor); browser->monitor = NULL;
  }
  
  result = gnome_vfs_monitor_add(&browser->monitor,
                                 uri_str,
                                 GNOME_VFS_MONITOR_DIRECTORY,
                                 directory_monitor,
                                 browser);
  if (result != GNOME_VFS_OK)
  {
    g_print("%s: Error installing directory monitor: %s", 
            ERROR_STR, gnome_vfs_result_to_string(result));
  }
  g_free(uri_str); uri_str = NULL;
  
  /* Finally change our current directory */
  g_free(browser->cur_dir);
  browser->cur_dir = new_dir;
  
  return TRUE;
}

 
static gboolean command_get(BrowserInfo *browser, gint argc, const gchar **args)
{
  gchar *path = NULL, *uri_str =NULL, *tmp = NULL;
  GnomeVFSResult result;
  GnomeVFSHandle *remote_handle, *local_handle;
  gchar buf[BUF_SIZE];
  GnomeVFSFileSize read, written, written_total;
  
  if (argc == 0 || strlen(args[0]) < 1) 
  {
    g_print("Usage: GET <filename>");
    return FALSE;
  }
  
  /* Create string */
  path = build_path(browser, args[0]);
  uri_str = g_strconcat(UPNP_AV_PROTOCOL, path, NULL);
  tmp = g_path_get_basename(path);
  g_free(path); path = NULL;
    
  /* Open remote file */
  result = gnome_vfs_open(&remote_handle, uri_str, GNOME_VFS_OPEN_READ);
  g_free(uri_str); uri_str = NULL;
  
  if (result != GNOME_VFS_OK)
  {
    g_print("%s: Remote file: %s\n", ERROR_STR, gnome_vfs_result_to_string(result));
    g_free(tmp);
    return FALSE;
  }    
  
  /* Open local file */
  path = g_build_path("/", g_getenv("PWD"), tmp, NULL);
  g_free(tmp); tmp = NULL;
  uri_str = g_strconcat(FILE_PROTOCOL, path, NULL);
  g_free(path); path = NULL;
  
  result =gnome_vfs_create(&local_handle, uri_str, GNOME_VFS_OPEN_WRITE,
                           FALSE, 644);
  g_free(uri_str); uri_str = NULL;
  
  if (result != GNOME_VFS_OK)
  {
    g_print("%s: Local file: %s\n", ERROR_STR, gnome_vfs_result_to_string(result));
    gnome_vfs_close(remote_handle);
    return FALSE;
  }    
  
  /* Copying */
  do {
    result = gnome_vfs_read(remote_handle, buf, BUF_SIZE, &read);
    if (read > 0) 
    {
      g_print(".");
      written_total = 0;
      while (read > 0 && written_total < read && result == GNOME_VFS_OK)
      {
        result = gnome_vfs_write(local_handle, (buf+written_total), 
                                 (read-written_total), &written);
        written_total += written;
      }
    }
  } while(read > 0 && written > 0 && result == GNOME_VFS_OK);
  
  g_print("\nCompleted.");
  
  /* Close files */
  gnome_vfs_close(remote_handle);
  gnome_vfs_close(local_handle);
  
  return TRUE;
}


static gboolean command_play(BrowserInfo *browser, gint argc, const gchar **args)
{
  GnomeVFSFileInfo *file_info;
  GnomeVFSResult result;
  gchar *path = NULL, *uri_str = NULL;
  
  if (argc == 0 || strlen(args[0]) < 1) 
  {
    g_print("Usage: PLAY <filename>");
    return FALSE;
  }
  
  /* Create string */
  path = build_path(browser, args[0]);
  uri_str = g_strconcat(UPNP_AV_PROTOCOL, path, NULL);
  g_free(path); path = NULL;
  
  file_info = gnome_vfs_file_info_new();
  result = gnome_vfs_get_file_info(uri_str, file_info, 
                                   GNOME_VFS_FILE_INFO_DEFAULT);
  g_free(uri_str);
    
  if (result != GNOME_VFS_OK || 
      (file_info->type != GNOME_VFS_FILE_TYPE_SYMBOLIC_LINK &&
       file_info->type != GNOME_VFS_FILE_TYPE_REGULAR))
  {
    if (result == GNOME_VFS_OK)
    {
      g_print("%s: %s: %s\n", ERROR_STR, args[0], "Wrong file type");
    } else {
      g_print("%s: %s: %s\n", ERROR_STR, args[0], gnome_vfs_result_to_string(result));
    }
    
    gnome_vfs_file_info_unref(file_info);
    return FALSE;
  }
  
#ifdef GUPNP_USE_MAEMO
  if (player_play_uri(browser->ses_conn, file_info->symlink_name, NULL) == FALSE)
  {
    g_print("%s: %s: %s\n", ERROR_STR, args[0], "Unable to play");
    gnome_vfs_file_info_unref(file_info);
    return FALSE;
  }
#else
  g_print("This version does not have MAEMO support.\n");
#endif
  return TRUE;
}


static gboolean command_exists(BrowserInfo *browser, gint argc, const gchar **args)
{
	gchar *path = NULL, *uri_str = NULL;
	GnomeVFSURI* uri = NULL;
	
	if (argc == 0 || strlen(args[0]) < 1) 
	{
		g_print("Usage: EXISTS <filename>\n");
		return FALSE;
	}
  
	/* Create string */
	path = build_path(browser, args[0]);
	uri_str = g_strconcat(UPNP_AV_PROTOCOL, path, NULL);
	g_free(path); path = NULL;
	
	uri = gnome_vfs_uri_new(uri_str);
	g_free(uri_str); uri_str = NULL;
	
	g_print(gnome_vfs_uri_exists(uri) ? "TRUE\n" : "FALSE\n");
	gnome_vfs_uri_unref(uri);
	return TRUE;
}


static gboolean command_stop(BrowserInfo *browser, gint argc, const gchar **args)
{
#ifdef GUPNP_USE_MAEMO
  if (player_stop(browser->ses_conn, NULL) == FALSE)
  {
    g_print("%s: Unable to stop\n", ERROR_STR);
    return FALSE;
  }
#endif
  return TRUE;
}


static gboolean command_sleep(BrowserInfo *browser, gint argc, const gchar **args)
{
  gint time;
  
  if (argc == 0 || strlen(args[0]) < 1) 
  {
    g_print("Usage: SLEEP <seconds>");
    return FALSE;
  }
  
  time = atoi(args[0]);
  if (time > 0) sleep(time);
    
  return TRUE;
}


static gboolean command_help(BrowserInfo *browser, gint argc, const gchar **args)
{
  gint idx;
  
  g_print("Possible commands:\n");
  for (idx = 0; commands[idx].str != NULL; idx++ )
  {
    g_print("%s\n", commands[idx].str);
  }
  
  return TRUE;
}


static gboolean command_exit(BrowserInfo *browser, gint argc, const gchar **args)
{
  g_print("Exiting..\n");
  g_main_loop_quit(browser->mainloop);
  browser->exit = TRUE;
  
  return TRUE;
}


static gboolean command_runner(gpointer data)
{
  BrowserInfo *browser = (BrowserInfo*)data;
  CommandInfo *cominfo = NULL;
  
  g_return_val_if_fail(browser != NULL, FALSE);
  
  if (g_queue_is_empty(browser->command_queue)) return TRUE;
    
  cominfo = (CommandInfo*)g_queue_pop_tail(browser->command_queue);
  
  g_return_val_if_fail(cominfo != NULL, FALSE);
  
  /* Do the command */
  cominfo->command(browser, cominfo->argc - 1, (const gchar**)(cominfo->args + 1));
  
  /* Free arguments and cominfo */
  g_strfreev(cominfo->args);
  g_free(cominfo);
  
  if (browser->exit == TRUE)
    return FALSE;
  
  if (g_queue_is_empty(browser->command_queue)) print_prompt(browser);
  
  return TRUE;
}


static gboolean read_data_cb(GIOChannel *channel,
                             GIOCondition condition,
                             gpointer data)
{
  gchar *buf = NULL;
  gsize len = 0;
  BrowserInfo *browser = (BrowserInfo*)data;
  CommandInfo *cominfo = NULL;
  
  gchar **line_commands = NULL;
  gchar **args = NULL;
  gint argc = 0;
  gint idx = 0, command_idx = 0;
  
  if (browser == NULL) return TRUE;
  
  g_io_channel_read_line(channel, &buf, &len, NULL, NULL);
  if (buf == NULL) return TRUE;
  
  /* This is stupid way to split string, as it splits wrongly, if there is
     COMMAND_DELIM in quotes. But I don't want to use lexical scanner.. */
  line_commands = g_strsplit(buf, COMMAND_DELIM, 0);
  g_free(buf); buf = NULL;
  
  for (command_idx = 0; line_commands[command_idx] != NULL;
       command_idx++)
  { 
    if (!g_shell_parse_argv(line_commands[command_idx], &argc, &args, NULL) ||
        argc == 0)
    {
      continue;
    }
    
    /* Try to find command */
    for (idx = 0; commands[idx].str != NULL; idx++ )
    {
      if (strcasecmp(commands[idx].str, args[0]) == 0)
      {
        cominfo = g_new0(CommandInfo, 1);
        cominfo->argc = argc;
        cominfo->args = args;
        cominfo->command = commands[idx].command;
        break;
      }
    }
  
    /* If command was found, then add it to the queue */
    if (cominfo == NULL)
    {
      g_print("%s: %s\n", args[0], ERROR_UNKNOWN_COMMAND);
    } else {
      g_queue_push_head(browser->command_queue, cominfo);
      cominfo = NULL;
    }
  }
  
  /* Free commands array */
  g_strfreev(line_commands);
  
  if (g_queue_is_empty(browser->command_queue)) print_prompt(browser);
  
  return TRUE;
}


int main (int argc, char **argv)
{
  BrowserInfo *browser = NULL;
  DBusError error;
  static const gchar *root_dir[] =
    { "/" };
    
  gnome_vfs_init ();
  
  browser = g_new0(BrowserInfo, 1);
  if (browser == NULL)
  {
    g_error("Unable to allocate memory");
    return 1;
  }
  
  browser->mainloop = g_main_loop_new(NULL, FALSE);
  if (browser->mainloop == NULL)  
  {
    g_error("Unable to initialize main loop.");
    g_free(browser);
    return 1;
  }
  
  browser->command_queue = g_queue_new();
  if (browser->command_queue == NULL)
  {
    g_error("Unable to initialize command queue.");
    g_main_loop_unref(browser->mainloop);
    g_free(browser);
    return 1;
  }

  dbus_error_init(&error);
  browser->sys_conn = dbus_bus_get(DBUS_BUS_SYSTEM, &error);
  browser->ses_conn = dbus_bus_get(DBUS_BUS_SESSION, &error);
  if (browser->sys_conn == NULL || browser->ses_conn == NULL)
  {
    g_error("Unable to initialize connection to DBUS");
    g_queue_free(browser->command_queue);
    g_main_loop_unref(browser->mainloop);
    g_free(browser);
    return 1;
  }
  
  /* setup D-Bus to use the Glib main loop */
  dbus_connection_setup_with_g_main(browser->sys_conn, NULL);
  dbus_connection_setup_with_g_main(browser->ses_conn, NULL);
  g_idle_add(command_runner, browser);
  
  /* Set volume */
#ifdef GUPNP_USE_MAEMO
  player_set_volume(browser->ses_conn, 100, NULL);
#endif
  
  /* Create IO channel to stdin and add watch */
  browser->stdin = g_io_channel_unix_new(0);
  browser->source_id = g_io_add_watch(browser->stdin, 
                                      G_IO_IN, 
                                      read_data_cb,
                                      browser);
  
  command_cd(browser, 1, root_dir);
  print_prompt(browser);
  
  /* Go to mainloop */
  g_main_loop_run(browser->mainloop);
  
  /* Uninitialize everything */
#ifdef GUPNP_USE_MAEMO
  player_stop(browser->ses_conn, NULL);
#endif
  g_source_remove(browser->source_id);
  g_io_channel_shutdown(browser->stdin, TRUE, NULL);
  g_queue_free(browser->command_queue);
  g_main_loop_unref(browser->mainloop);
  
  g_free(browser->cur_dir);
  g_free(browser);
  
  gnome_vfs_shutdown();
  
  return 0;
}
