/**
    Gnome-VFS module for browsing UPnP AV MediaServers

    Copyright 2006 Nokia Corporation. All rights reserved.
	
    Contact: Aapo Makela <aapo.makela@nokia.com>

    This library is free software; you can redistribute it and/or modify it
    under the terms of the GNU Lesser General Public License version 2.1 as 
    published by the Free Software Foundation.
  
    This library is distributed in the hope that it will be useful, but
    WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU Lesser
    General Public License for more details.

    You should have received a copy of the GNU Lesser General Public License
    along with this library; if not, write to the Free Software Foundation,
    Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
*/

#include <libgnomevfs/gnome-vfs-module-shared.h>
#include <libgnomevfs/gnome-vfs-mime.h>

#include "gnomevfs-upnpav-utils.h"

/* GnomeVFS virtual functions */

/**
	Initialize this Gnome-VFS module.
*/
GnomeVFSMethod *vfs_module_init(const char *method, const char *args);
/**
	Uninitialize this Gnome-VFS module.
*/
void vfs_module_shutdown(GnomeVFSMethod *method);

static GnomeVFSResult upnpav_open_directory(
			GnomeVFSMethod *method,
			GnomeVFSMethodHandle **method_handle,
			GnomeVFSURI *uri,
			GnomeVFSFileInfoOptions options,
			GnomeVFSContext *context);

static GnomeVFSResult upnpav_read_directory(
				GnomeVFSMethod *method,
				GnomeVFSMethodHandle *method_handle,
				GnomeVFSFileInfo *file_info,
				GnomeVFSContext *context);
				
static GnomeVFSResult upnpav_close_directory(
			GnomeVFSMethod *method,
			GnomeVFSMethodHandle *method_handle,
			GnomeVFSContext *context);
			
static GnomeVFSResult upnpav_get_file_info(
				GnomeVFSMethod *method,
				GnomeVFSURI *uri,
				GnomeVFSFileInfo *file_info,
				GnomeVFSFileInfoOptions options,
				GnomeVFSContext *context);
				
static GnomeVFSResult upnpav_get_file_info_from_handle (
				GnomeVFSMethod *method,
				GnomeVFSMethodHandle *method_handle,
				GnomeVFSFileInfo *file_info,
				GnomeVFSFileInfoOptions options,
				GnomeVFSContext *context);
			
static GnomeVFSResult upnpav_open(
			GnomeVFSMethod *method,
			GnomeVFSMethodHandle **method_handle,
			GnomeVFSURI *uri,
			GnomeVFSOpenMode mode,
			GnomeVFSContext *context);

static GnomeVFSResult upnpav_close(
			GnomeVFSMethod *method,
			GnomeVFSMethodHandle *method_handle,
			GnomeVFSContext *context);

static GnomeVFSResult upnpav_read(
			GnomeVFSMethod *method,
			GnomeVFSMethodHandle *method_handle,
			gpointer buffer,
			GnomeVFSFileSize num_bytes,
			GnomeVFSFileSize *bytes_read_return,
			GnomeVFSContext *context);

static GnomeVFSResult upnpav_tell(
			GnomeVFSMethod *method,
			GnomeVFSMethodHandle *method_handle,
			GnomeVFSFileSize *offset_return);

static GnomeVFSResult upnpav_seek(
			GnomeVFSMethod *method,
			GnomeVFSMethodHandle *method_handle,
			GnomeVFSSeekPosition  whence,
			GnomeVFSFileOffset    offset,
			GnomeVFSContext *context);

static gboolean upnpav_is_local(
			GnomeVFSMethod *method,
			const GnomeVFSURI *uri);

static GnomeVFSResult upnpav_move(
				GnomeVFSMethod *method,
				GnomeVFSURI *old_uri,
				GnomeVFSURI *new_uri,
				gboolean force_replace,
				GnomeVFSContext *context);

static GnomeVFSResult upnpav_check_same_fs(
			GnomeVFSMethod *method,
			GnomeVFSURI *a,
			GnomeVFSURI *b,
			gboolean *same_fs_return,
			GnomeVFSContext *context);

static GnomeVFSResult upnpav_monitor_add(
			GnomeVFSMethod *method,
      			GnomeVFSMethodHandle **method_handle_return,
      			GnomeVFSURI *uri,
      			GnomeVFSMonitorType monitor_type);

static GnomeVFSResult upnpav_monitor_cancel(GnomeVFSMethod *method,
					    GnomeVFSMethodHandle *method_handle);


/*--- Implementation ---*/

static GnomeVFSMethod upnpav_method = {
	sizeof (GnomeVFSMethod),
	
	upnpav_open,		/* open */
	NULL,			/* create */
	upnpav_close,		/* close */
	upnpav_read,		/* read */
	NULL,			/* write */
	upnpav_seek,		/* seek */
	upnpav_tell, 		/* tell */
	NULL, 			/* truncate handle */
	upnpav_open_directory, 	/* open directory */
	upnpav_close_directory, /* close directory */
	upnpav_read_directory,	/* read directory */
	upnpav_get_file_info, 	/* get file info */
	upnpav_get_file_info_from_handle, /* get file info from handle */
	(GnomeVFSMethodIsLocalFunc)upnpav_is_local, 	/* is local */
	NULL,			/* make directory */
	NULL, 			/* remove directory */
	upnpav_move,  		/* move - NOT SUPPORTED */
	NULL,			/* unlink */
	upnpav_check_same_fs, 	/* check same fs */
	NULL, 			/* set file info */
	NULL, 			/* truncate */
	NULL, 			/* find directory */
	NULL, 			/* create symbolic link */
	upnpav_monitor_add,	/* monitor add */
	upnpav_monitor_cancel, 	/* monitor cancel */
	NULL			/* file control */
};

extern GStaticMutex tree_mutex;
extern GHashTable *monitor_hash;
extern GStaticMutex monitor_mutex;

static GNode *root_node = NULL;
static GnomeVFSMethod *http_method = NULL; 

GnomeVFSMethod *vfs_module_init(const char *method, const char *args)
{
	debug("vfs_module_init called with method '%s' and args '%s'", 
	      method, args);
	
	if (strcmp(method, "upnpav") == 0)
	{
		/* Create root node for media items */
		root_node = upnpav_init_tree();
		if (root_node == NULL) 
		{
			error("Couldn't create root node");
			return NULL;
		}
		
		/* We need http method in order to implement upnpav */
		http_method = gnome_vfs_method_get("http");
		if (http_method == NULL) 
		{
			error("Couldn't get HTTP method");
			return NULL;
		}
		
		return &upnpav_method;
	}
	
	return NULL;
}


void vfs_module_shutdown(GnomeVFSMethod *method)
{
	debug("vfs_module_shutdown for upnpav");
	upnpav_free_tree();
	root_node = NULL;
}


static GnomeVFSResult upnpav_open_directory(
			GnomeVFSMethod *method,
			GnomeVFSMethodHandle **method_handle,
			GnomeVFSURI *uri,
			GnomeVFSFileInfoOptions options,
			GnomeVFSContext *context)
{
	GnomeVFSCancellation *cancellation = NULL;
	gchar *path = NULL;
	GNode *dir_node = NULL;
	UPnPAVDirectoryHandle *handle = NULL;
	
	/* assign NULL to method handle */
	*method_handle = NULL;
	
	if (gnome_vfs_uri_get_path(uri) == NULL || 
	    gnome_vfs_uri_get_path(uri)[0] != '/')
		return GNOME_VFS_ERROR_INVALID_URI;
	
	path = gnome_vfs_unescape_string(gnome_vfs_uri_get_path(uri), NULL);
	debug("upnpav_open_directory %s", path);
	
	g_static_mutex_lock(&tree_mutex);
	
	if (context != NULL) 
		cancellation = gnome_vfs_context_get_cancellation(context);
	
	dir_node = upnpav_search_for_path(root_node, path, cancellation);
	g_free(path); path = NULL;
	
	if (dir_node == NULL || dir_node->data == NULL)
	{
		g_static_mutex_unlock(&tree_mutex);
		
		if (gnome_vfs_context_check_cancellation(context) == TRUE)
			return GNOME_VFS_ERROR_CANCELLED;
		
		return GNOME_VFS_ERROR_NOT_FOUND;
	}
	
	if (((UPnPAVNodeData*)dir_node->data)->container == FALSE)
	{
		g_static_mutex_unlock(&tree_mutex);
		return GNOME_VFS_ERROR_INVALID_URI;
	}
	
	if (((UPnPAVNodeData*)dir_node->data)->browsed == 0)
	{
		/* Browse, if the directory has never been browsed */
		if (upnpav_get_content_directory(dir_node, 
						 cancellation, 
						 NULL) == FALSE)
		{
			g_static_mutex_unlock(&tree_mutex);
			
#ifdef USE_CANCELLATION
			if (gnome_vfs_context_check_cancellation(context)
				== TRUE)
					return GNOME_VFS_ERROR_CANCELLED;
#endif		
			return GNOME_VFS_ERROR_INVALID_URI;
		}
	}
	
	handle = g_new0(UPnPAVDirectoryHandle, 1);
	handle->directory_node = dir_node;
	handle->current_node = g_node_first_child(dir_node);
	handle->options = options;
	((UPnPAVNodeData*)dir_node->data)->directory_handles = 
		g_slist_append(
			((UPnPAVNodeData*)dir_node->data)->directory_handles,
			handle);
	
	*method_handle = (GnomeVFSMethodHandle*)handle;
	
	g_static_mutex_unlock(&tree_mutex);
	
	return GNOME_VFS_OK;
}


static GnomeVFSResult upnpav_read_directory(
				GnomeVFSMethod *method,
				GnomeVFSMethodHandle *method_handle,
				GnomeVFSFileInfo *file_info,
				GnomeVFSContext *context)
{
	GnomeVFSCancellation *cancellation = NULL;
	UPnPAVDirectoryHandle *handle = (UPnPAVDirectoryHandle*) method_handle;
	GNode *file;
	UPnPAVNodeData *data;
	UPnPAVNodeData *dir_data;
	GnomeVFSResult result = GNOME_VFS_OK;
	
	debug("upnpav_read_directory");
    
	g_static_mutex_lock(&tree_mutex);
	
	if (handle->directory_node == NULL && handle->current_node != NULL)
	{
		/* Handle is invalid, so the dir has probably been removed. 
		   Return INVALID URI */
		error("Invalid directory handle detected!");
		g_static_mutex_unlock(&tree_mutex);
		return GNOME_VFS_ERROR_NOT_FOUND;
	}
	
	if (context != NULL) 
		cancellation = gnome_vfs_context_get_cancellation(context);
	
	if (handle->current_node == NULL) 
	{
		g_static_mutex_unlock(&tree_mutex);
		debug("End of directory reached.");
		return GNOME_VFS_ERROR_EOF;
	}
	
	file = handle->current_node;
	data = (UPnPAVNodeData*)file->data;
	
	if (data == NULL) 
	{
		g_static_mutex_unlock(&tree_mutex);
		debug("End of directory reached.");
		return GNOME_VFS_ERROR_EOF;
	}
	
	result = upnpav_determine_fileinfo(data, file_info, &handle->options);
	if (result != GNOME_VFS_OK)
	{
		g_static_mutex_unlock(&tree_mutex);
		error("Unable to determine file info for %s, when browsing", data->name);
		return result;
	}
	
	debug("Iterating file %s", file_info->name);
	
	if ( (g_node_next_sibling(handle->current_node) == NULL ||
	      g_node_next_sibling(g_node_next_sibling(handle->current_node)) 
		== NULL) &&
	    ((UPnPAVNodeData*)handle->directory_node->data)->browsed 
		< MAX_BROWSED )
	{
		/* Were soon going over known directory structure,
		   browse more.. */
		if (upnpav_get_content_directory(handle->directory_node, 
						 cancellation, NULL) == FALSE)
		{
			/* TODO: This should probably be fixed to loop while 
			   browsed < MAX_BROWSED && not cancelled && 
			   g_node_next_sibling(handle->current_node) == NULL 
		
			   Currently listing all contents of the directory
			   fails, if directory contains over 
			   2 * NUM_PER_BROWSE invalid
			   items in a sequence (very rare case)
			*/
			
			g_static_mutex_unlock(&tree_mutex);
			
#ifdef USE_CANCELLATION
			if (gnome_vfs_context_check_cancellation(context)
				== TRUE)
				return GNOME_VFS_ERROR_CANCELLED;
#endif		
			return GNOME_VFS_ERROR_INVALID_URI;
		}
	}
	
	handle->current_node = g_node_next_sibling(handle->current_node);
	
	if (handle->current_node == NULL && 
	    handle->directory_node != NULL &&
	    handle->directory_node->data != NULL) 
	{
		/* We have iterated past the last file and can remove this
		   directory handle */
		dir_data = ((UPnPAVNodeData*)handle->directory_node->data);
		dir_data->directory_handles = g_slist_remove(
						dir_data->directory_handles,
						handle);
		handle->directory_node = NULL;
	}
	
	g_static_mutex_unlock(&tree_mutex);
		
	return GNOME_VFS_OK;
}


static GnomeVFSResult upnpav_close_directory(
			GnomeVFSMethod *method,
			GnomeVFSMethodHandle *method_handle,
			GnomeVFSContext *context)
{
	UPnPAVDirectoryHandle *handle = (UPnPAVDirectoryHandle*) method_handle;
	UPnPAVNodeData *dir_data;
	debug("upnpav_close_directory");
	
	g_static_mutex_lock(&tree_mutex);
	
	if (handle != NULL && handle->directory_node != NULL &&
	    handle->directory_node->data != NULL)
	{
		/* Remove handle from the node, if it still exists */
		dir_data = ((UPnPAVNodeData*)handle->directory_node->data);
		dir_data->directory_handles = g_slist_remove(
						dir_data->directory_handles,
						handle);
		handle->directory_node = NULL;
	}
	
	if (handle != NULL) g_free(handle);
	
	g_static_mutex_unlock(&tree_mutex);
	
	return GNOME_VFS_OK;
}


static GnomeVFSResult upnpav_get_file_info(
			GnomeVFSMethod *method,
			GnomeVFSURI *uri,
			GnomeVFSFileInfo *file_info,
			GnomeVFSFileInfoOptions options,
			GnomeVFSContext *context)
{
	GnomeVFSCancellation *cancellation = NULL;

	gchar *path = NULL;
	GNode *file = NULL;
	UPnPAVNodeData *data;
	GnomeVFSResult result;
	
	if (gnome_vfs_uri_get_path(uri) == NULL || 
	    gnome_vfs_uri_get_path(uri)[0] != '/')
		return GNOME_VFS_ERROR_INVALID_URI;
	
	path = gnome_vfs_unescape_string(gnome_vfs_uri_get_path(uri), NULL);
	debug("upnpav_get_file_info: %s", path);
	
	g_static_mutex_lock(&tree_mutex);
	
	if (context != NULL) 
		cancellation = gnome_vfs_context_get_cancellation(context);
	
	file = upnpav_search_for_path(root_node, path, cancellation);
	g_free(path); path = NULL;
	
	if (file == NULL)
	{
		g_static_mutex_unlock(&tree_mutex);
		debug("No file found. Exiting upnpav_get_file_info..");
		
#ifdef USE_CANCELLATION
		if (gnome_vfs_context_check_cancellation(context) == TRUE)
			return GNOME_VFS_ERROR_CANCELLED;
#endif		
		return GNOME_VFS_ERROR_NOT_FOUND;
	}
	
	data = (UPnPAVNodeData*)file->data;
	
	result = upnpav_determine_fileinfo(data, file_info, &options);
	
	g_static_mutex_unlock(&tree_mutex);
	
	debug("Exiting upnpav_get_file_info..");
	
	return result;
}


static GnomeVFSResult upnpav_get_file_info_from_handle (
				GnomeVFSMethod *method,
				GnomeVFSMethodHandle *method_handle,
				GnomeVFSFileInfo *file_info,
				GnomeVFSFileInfoOptions options,
				GnomeVFSContext *context)
{
	UPnPAVFileHandle *handle = (UPnPAVFileHandle*)method_handle;
	
	debug("upnpav_get_file_info_from_handle");
	
	return upnpav_get_file_info(method,  handle->file_uri, file_info, 
				    options, context);
}


static GnomeVFSResult upnpav_open(
			GnomeVFSMethod *method,
			GnomeVFSMethodHandle **method_handle,
			GnomeVFSURI *uri,
			GnomeVFSOpenMode mode,
			GnomeVFSContext *context)
{
	GnomeVFSCancellation *cancellation = NULL;
	gchar *path = NULL;
	GNode *file = NULL;
	gchar *http_uri_str = NULL;
	UPnPAVFileHandle *handle = NULL;
	GnomeVFSResult result;
	
	if (gnome_vfs_uri_get_path(uri) == NULL || 
	    gnome_vfs_uri_get_path(uri)[0] != '/')
		return GNOME_VFS_ERROR_INVALID_URI;
	
	path = gnome_vfs_unescape_string(gnome_vfs_uri_get_path(uri), NULL);
	debug("upnpav_open %s", path);
	
	g_static_mutex_lock(&tree_mutex);
	
	*method_handle = NULL;
	
	if (context != NULL) 
		cancellation = gnome_vfs_context_get_cancellation(context);
	
	file = upnpav_search_for_path(root_node, path, cancellation);
	g_free(path); path = NULL;
	
	if (file == NULL || file->data == NULL)
	{
		g_static_mutex_unlock(&tree_mutex);
		
#ifdef USE_CANCELLATION
		if (gnome_vfs_context_check_cancellation(context) == TRUE)
			return GNOME_VFS_ERROR_CANCELLED;
#endif		
		return GNOME_VFS_ERROR_NOT_FOUND;
	}	
		
	if (((UPnPAVNodeData*)file->data)->container == TRUE)
	{
		g_static_mutex_unlock(&tree_mutex);
		return GNOME_VFS_ERROR_IS_DIRECTORY;
	}
	
	http_uri_str = ((UPnPAVNodeData*)file->data)->uri;
	
	if (http_uri_str == NULL)
	{
		g_static_mutex_unlock(&tree_mutex);
		return GNOME_VFS_ERROR_NOT_FOUND;
	}
	
	/* Create handle for the file  and try to get URI */
	handle = g_new0(UPnPAVFileHandle, 1);
	
#ifdef MODEL_AS_PLAYLISTS
	handle->uri = g_strconcat(http_uri_str, "\n", NULL);
	handle->curpos = 0;
#else
	/* Try opening a HTTP connection for the URI */
	handle->mode = mode;
	handle->http_uri = gnome_vfs_uri_new(http_uri_str);
	result = http_method->open(
			http_method,
			(GnomeVFSMethodHandle**)&handle->http_handle,
			handle->http_uri, mode, context);
	
	if (result != GNOME_VFS_OK)
	{
		g_free(handle);
		
		g_static_mutex_unlock(&tree_mutex);
		return result;
	}
#endif
	
	handle->file_uri = gnome_vfs_uri_dup(uri);
	*method_handle = (GnomeVFSMethodHandle*)handle;
	
	g_static_mutex_unlock(&tree_mutex);
	
	return GNOME_VFS_OK;
}


static GnomeVFSResult upnpav_close(
			GnomeVFSMethod *method,
			GnomeVFSMethodHandle *method_handle,
			GnomeVFSContext *context)
{
	UPnPAVFileHandle *handle = (UPnPAVFileHandle*)method_handle;
	GnomeVFSResult result;
	debug("upnpav_close");
	
	if (handle == NULL) return GNOME_VFS_ERROR_IO;
	
	gnome_vfs_uri_unref(handle->file_uri);
	gnome_vfs_uri_unref(handle->http_uri);
#ifdef MODEL_AS_PLAYLISTS
	g_free(handle->uri);
	result = GNOME_VFS_OK;
#else
	result = http_method->close(
				http_method, 
				(GnomeVFSMethodHandle*)handle->http_handle, 
				context);
#endif
	g_free(handle);
	
	return result;
}


static GnomeVFSResult upnpav_read(
			GnomeVFSMethod *method,
			GnomeVFSMethodHandle *method_handle,
			gpointer buffer,
			GnomeVFSFileSize num_bytes,
			GnomeVFSFileSize *bytes_read_return,
			GnomeVFSContext *context)
{
	UPnPAVFileHandle *handle = (UPnPAVFileHandle*)method_handle;
	GnomeVFSResult result;
	
	if (handle == NULL) return GNOME_VFS_ERROR_IO;
	
	g_static_mutex_lock(&tree_mutex);
	if (g_node_first_child(root_node) == NULL)
	{
		/* No childs exists, so we don't have a connection */
		g_static_mutex_unlock(&tree_mutex);
		return GNOME_VFS_ERROR_IO;
	}
	g_static_mutex_unlock(&tree_mutex);
	
#ifdef MODEL_AS_PLAYLISTS
	if (handle->curpos >= cg_strlen(handle->uri))
	{
		*bytes_read_return = 0;
		result = GNOME_VFS_ERROR_EOF;
	} else {
		*bytes_read_return = MIN(num_bytes, cg_strlen(handle->uri) - handle->curpos);
		
		memcpy(buffer, handle->uri + handle->curpos, *bytes_read_return);
		handle->curpos += *bytes_read_return;
		
		result = GNOME_VFS_OK;
	}
#else
	result = http_method->read(
				http_method,
				(GnomeVFSMethodHandle*)handle->http_handle,
				buffer, num_bytes,
				bytes_read_return, context);
	if (handle->mode != GNOME_VFS_OPEN_NONE && 
	    result == GNOME_VFS_ERROR_IO)
	{
		/* Circumvent a bug in some important UPnP implementations and
		   try again with a new handle if IO error occurs at start */
		error("I/O Error at startup bug happened, try again...");
		http_method->close(
				http_method, 
				(GnomeVFSMethodHandle*)handle->http_handle, 
				context);
		if (http_method->open(
			http_method,
			(GnomeVFSMethodHandle**)&handle->http_handle,
			handle->http_uri, handle->mode, context) 
				== GNOME_VFS_OK)
		{
			result = http_method->read(
				http_method,
				(GnomeVFSMethodHandle*)handle->http_handle,
				buffer, num_bytes,
				bytes_read_return, context);
		}
	}
	handle->mode = GNOME_VFS_OPEN_NONE;
#endif	
	return result;
}


static GnomeVFSResult upnpav_tell(
			GnomeVFSMethod *method,
			GnomeVFSMethodHandle *method_handle,
			GnomeVFSFileSize *offset_return)
{
	UPnPAVFileHandle *handle = (UPnPAVFileHandle*)method_handle;
	GnomeVFSResult result;
	
	if (handle == NULL) return GNOME_VFS_ERROR_IO;
	
#ifdef MODEL_AS_PLAYLISTS
	if (offset_return != NULL)
	    *offset_return = handle->curpos;
	result = GNOME_VFS_OK;
#else
	result = gnome_vfs_tell(handle->http_handle,
				offset_return);
#endif
	return result;
}


static GnomeVFSResult upnpav_seek(
			GnomeVFSMethod *method,
			GnomeVFSMethodHandle *method_handle,
			GnomeVFSSeekPosition  whence,
			GnomeVFSFileOffset    offset,
			GnomeVFSContext *context)
{
	UPnPAVFileHandle *handle = (UPnPAVFileHandle*)method_handle;
	GnomeVFSResult result;
	
	if (handle == NULL) return GNOME_VFS_ERROR_IO;
	
#ifdef MODEL_AS_PLAYLISTS
	result = GNOME_VFS_ERROR_NOT_SUPPORTED;
#else
	if (VFS_METHOD_HAS_FUNC(http_method,seek))
	{
		result = http_method->seek(
				http_method, 
				(GnomeVFSMethodHandle*)handle->http_handle,
				whence, offset,
				context);
	} else {
		error("Current HTTP module does not support seek");
		result = GNOME_VFS_ERROR_NOT_SUPPORTED;
	}
#endif
	return result;
}


static gboolean upnpav_is_local(
			GnomeVFSMethod *method,
			const GnomeVFSURI *uri)
{
	debug("upnpav_is_local: %s",
		    gnome_vfs_uri_get_path(uri));
	
#ifdef MODEL_AS_PLAYLISTS
	return TRUE;
#else
	return FALSE;
#endif
}


static GnomeVFSResult upnpav_move(
				GnomeVFSMethod *method,
				GnomeVFSURI *old_uri,
				GnomeVFSURI *new_uri,
				gboolean force_replace,
				GnomeVFSContext *context)
{
    debug("upnpav_move");
    
    return GNOME_VFS_ERROR_NOT_SUPPORTED;
}


static GnomeVFSResult upnpav_check_same_fs (
			GnomeVFSMethod *method,
			GnomeVFSURI *a,
			GnomeVFSURI *b,
			gboolean *same_fs_return,
			GnomeVFSContext *context)
{
    debug("upnpav_check_same_fs");
    if (gnome_vfs_uri_get_path(a) == NULL ||
        gnome_vfs_uri_get_path(a)[0] != '/') return GNOME_VFS_ERROR_INVALID_URI;
    if (gnome_vfs_uri_get_path(b) == NULL ||
        gnome_vfs_uri_get_path(b)[0] != '/') return GNOME_VFS_ERROR_INVALID_URI;
    
    /*
    *same_fs_return = 
	(cg_strcmp (gnome_vfs_uri_get_scheme (a), 
		    gnome_vfs_uri_get_scheme (b)) == 0);
    */
    *same_fs_return = FALSE;
    
    return GNOME_VFS_OK;
}


static GnomeVFSResult upnpav_monitor_add(
			GnomeVFSMethod *method,
			GnomeVFSMethodHandle **method_handle_return,
			GnomeVFSURI *uri,
			GnomeVFSMonitorType monitor_type)
{
#ifdef DEBUG
	gchar *tmp = NULL;
#endif
	UPnPAVMonitorHandle *handle = NULL;
	UPnPAVMonitorList *monitors = NULL;
#ifdef DEBUG
	tmp = gnome_vfs_uri_to_string(uri, GNOME_VFS_URI_HIDE_NONE);
	debug("upnpav_monitor_add (%d) %s", monitor_type, tmp);
	g_free(tmp); tmp = NULL;
#endif
	handle = g_new0(UPnPAVMonitorHandle, 1);
	handle->uri = gnome_vfs_uri_dup(uri);
	handle->monitor_type = monitor_type;
	
	g_static_mutex_lock(&monitor_mutex);
	
	monitors = g_hash_table_lookup (monitor_hash, handle->uri);
	if (!monitors) 
	{
		monitors = g_new0(UPnPAVMonitorList, 1);
		g_hash_table_insert (monitor_hash,
				     gnome_vfs_uri_dup(handle->uri),
				     monitors);
	}

	monitors->handles = g_list_prepend (monitors->handles, handle); 
	
	g_static_mutex_unlock(&monitor_mutex);
	
	*method_handle_return = (GnomeVFSMethodHandle *) handle;
	
	return GNOME_VFS_OK;
}


static GnomeVFSResult upnpav_monitor_cancel(GnomeVFSMethod *method,
					    GnomeVFSMethodHandle *method_handle)
{
	UPnPAVMonitorHandle *handle = NULL;
	UPnPAVMonitorList *monitors = NULL;
	debug("upnpav_monitor_cancel");
	
	handle = (UPnPAVMonitorHandle *)method_handle;
	
	g_static_mutex_lock(&monitor_mutex);
	
	monitors = g_hash_table_lookup (monitor_hash, handle->uri);
	if (monitors)
	{
		monitors->handles = g_list_remove(monitors->handles, handle);
		if (!monitors->handles)
		{
			g_hash_table_remove(monitor_hash, handle->uri);
			debug("Removing last monitor for URI %s", 
				    gnome_vfs_uri_to_string(handle->uri, 	
						GNOME_VFS_URI_HIDE_NONE));
			g_free(monitors); monitors = NULL;
		}
	}
	
	gnome_vfs_uri_unref(handle->uri); handle->uri = NULL;
	g_free(handle); handle = NULL;
	
	g_static_mutex_unlock(&monitor_mutex);
	
	return GNOME_VFS_OK;
}
