/**
    Gnome-VFS module for browsing UPnP AV MediaServers
	
    Copyright 2006 Nokia Corporation. All rights reserved.
	
    Contact: Aapo Makela <aapo.makela@nokia.com>
    
    This library is free software; you can redistribute it and/or modify it
    under the terms of the GNU Lesser General Public License version 2.1 as 
    published by the Free Software Foundation.
  
    This library is distributed in the hope that it will be useful, but
    WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU Lesser
    General Public License for more details.

    You should have received a copy of the GNU Lesser General Public License
    along with this library; if not, write to the Free Software Foundation,
    Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
*/

#ifndef _GNOMEVFS_UPNPAV_UTILS_H_
#define _GNOMEVFS_UPNPAV_UTILS_H_

#include <string.h>
#include <glib.h>

#ifdef DEBUG
# define debug(...) g_debug(__VA_ARGS__)
#else
# define debug(...) ((void)(0))
#endif

#define error(...) g_warning(__VA_ARGS__)

#include <libgnomevfs/gnome-vfs.h>

#include <cybergarage/upnp/cupnp.h>

/* #define MODEL_AS_SYMLINKS */
/* #define MODEL_AS_PLAYLISTS */

/* #define USE_EXTENDED_DIRECTORY_TYPES */
#define USE_CANCELLATION
#define USE_HTTP_FILE_INFO

#define MAX_BROWSED G_MAXULONG

#ifdef __cplusplus
extern "C" {
#endif /* __cplusplus */

/**
	Data associated with each node in filetree
*/
typedef struct _UPnPAVNodeData
{
	/** Object name */
	gchar *name;
	/** Object ID */
	gchar *object_id;
	
	/** Resource size */
	guint64 size;
	/** Resource type */
	gchar *mime_type;
	/** Resource URI */
	gchar *uri;
	/** Modification time - if available */
	time_t mtime;
	
	/** Unique device number, not copied */
	gchar *dev_udn;	/* Reference to device's tree node */
	
	/** Container flag */
	gboolean container;
	/** How many items in container are browsed, 0 <= x <= MAX_BROWSED */
	gulong browsed;
	
	/** UpdateID for a container */
	glong update_id;
	
	/** Directory handles pointing to this directory */
	GSList *directory_handles;
	
} UPnPAVNodeData;

/**
	Struct representing file handle
*/
typedef struct _UPnPAVFileHandle {
#ifdef MODEL_AS_PLAYLISTS
	gchar *uri;
	guint curpos;
#else
	GnomeVFSURI *file_uri;
	GnomeVFSURI *http_uri;
	GnomeVFSOpenMode mode;
	GnomeVFSHandle *http_handle;
#endif
} UPnPAVFileHandle;

/**
	Struct representing directory handle
*/
typedef struct _UPnPAVDirectoryHandle
{
	GnomeVFSURI dir_uri;
	GnomeVFSFileInfoOptions options;
	GNode *directory_node;
	GNode *current_node;
} UPnPAVDirectoryHandle;

typedef struct {
	GnomeVFSURI         *uri;
	GnomeVFSMonitorType  monitor_type;
} UPnPAVMonitorHandle;

/* A list of monitors are kept for each URI. */
typedef struct {
	GList *handles; /* list of MonitorHandle* */
} UPnPAVMonitorList;

typedef struct {
	GnomeVFSURI 	    *info_uri;
	GnomeVFSMonitorEventType event_type;
} UPnPAVMonitorNotify;

/*--- Interface ---*/

/**
	Initializes UPnP AV tree.

	@return Root node of the tree.
*/
GNode *upnpav_init_tree(void);

/**
	Uninitializes UPnP AV tree.
*/
void upnpav_free_tree(void);

/**
	Gets remotely content (children) for the given AV container node.
	Tree must be locked!

	@param node
	@param cancellation
	@return
*/
gboolean upnpav_get_content_directory(GNode *node, 
				      GnomeVFSCancellation *cancellation,
				      gint *upnp_status);

/**
	(Recursive) search for the given path from the given AV root node.
	Tree must be locked!
	
	@param parent
	@param path
	@param cancellation
	@return
*/
GNode *upnpav_search_for_path(GNode *parent, const gchar *path, 
			      GnomeVFSCancellation *cancellation);

/**
	Determines file info for the given AV node data.
	Tree must be locked!
	
	@param data
	@param file_info
	@param options
	@return
*/
GnomeVFSResult upnpav_determine_fileinfo(UPnPAVNodeData *data, 
					GnomeVFSFileInfo* file_info, 
					GnomeVFSFileInfoOptions *options);

#ifdef __cplusplus
}
#endif /* __cplusplus */

#endif /* _GNOMEVFS_UPNPAV_UTILS_H_ */
