/*
 * This file is part of hildon-input-method-plugins-example
 *
 * Copyright (C) 2006-2007 Nokia Corporation. All rights reserved.
 *
 * Contact: Mohammad Anwari <Mohammad.Anwari@nokia.com>
 *
 * Redistribution and use in source and binary forms, with or without modification, are permitted provided that the following conditions are met:
 *
    * Redistributions of source code must retain the above copyright notice, this list of conditions and the following disclaimer.
    * Redistributions in binary form must reproduce the above copyright notice, this list of conditions and the following disclaimer in the documentation and/or other materials provided with the distribution.
    * Neither the name of Nokia Corporation nor the names of its contributors may be used to endorse or promote products derived from this software without specific prior written permission.
*
* THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF 
* MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT OWNER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, 
* EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED
* AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF
* THE POSSIBILITY OF SUCH DAMAGE.
* 
*/

#include "hildon-im-plugin.h"
#include "hildon-im-ui.h"

#include <string.h>
#include <glib.h>
#include <gdk/gdk.h>
#include <gdk/gdkx.h>
#include <gtk/gtkwidget.h>

#define HIMEXAMPLE_VKB_TYPE himExample_vkb_get_type()
#define HIMEXAMPLE_VKB(obj) GTK_CHECK_CAST(obj, himExample_vkb_get_type(), HimExampleVKB)
#define HIMEXAMPLE_VKB_CLASS(klass) \
        GTK_CHECK_CLASS_CAST(klass, himExample_vkb_get_type, \
                             HimExampleVKBClass)
#define IS_HIMEXAMPLE_VKB(obj) \
        GTK_CHECK_TYPE(obj, himExample_vkb_get_type())
#define HIMEXAMPLE_VKB_GET_PRIVATE(obj) \
        (G_TYPE_INSTANCE_GET_PRIVATE ((obj), HIMEXAMPLE_VKB_TYPE,\
                                      HimExampleVKBPrivate))

#define FULLSCREEN_WIDTH   600
#define START_X             20
#define START_Y              0
#define HEIGHT              35
#define MAX_KEYPAD_KEYS      3
#define KEYBOARD_ROWS        4
#define KEYPAD_ROWS          4
#define KEYPAD_MARGIN_RIGHT 18

#define KEY_LABEL_MARGIN     3

#define TOTAL_KEYPAD_KEYS   MAX_KEYPAD_KEYS * KEYPAD_ROWS

enum {
  SCREEN_NORMAL,
  SCREEN_FULL,
  SCREEN_MAX
};

enum {
  CASE_UPPER,
  CASE_LOWER,
  CASE_MAX
};

enum {
  KEYSIZE_NORMAL,
  KEYSIZE_BIG,
  KEYSIZE_SPACE,

  KEYSIZE_MAX
};

static const gint button_sizes [SCREEN_MAX][KEYSIZE_MAX] = {
  { 34, 51, -1 },
  { 40, 60, -1 }
};

static const gint keyboard_keys [KEYBOARD_ROWS] = {
  11, 12, 11, 1
};

static const gchar *keyboard_layout [CASE_MAX][KEYBOARD_ROWS] = {
  { "QWERTYUIOP^", "ASDFGHJKL:#|", "ZXCVBNM<>\\~", " " },
  { "qwertyuiop@", "asdfghjkl;'!", "zxcvbnm,./?", " " }
};

static const gchar *keypad_layout [CASE_MAX][KEYPAD_ROWS] = {
  { "'\"£", "$€%", "&()", "_*+" },
  { "123", "456", "789", "-0=" }
};

typedef struct _vkbkey VKBKey;

static GType himExample_vkb_type = 0;
static GtkWidgetClass *parent_class = NULL;

typedef struct {
  GtkWidgetClass parent;
} HimExampleVKBClass;

typedef struct {
  GtkWidget parent;
} HimExampleVKB;

typedef struct {
  HildonIMUI        *ui;
  GtkStyle          *style;
  VKBKey            *keys, 
                    *last_key;
  PangoLayout       *layout;

  gint               screen_mode;
  gint               case_mode;
  gint               total_keys;
  
  gint               keypad_start_x;
} HimExampleVKBPrivate;

struct _vkbkey {
  gchar             *upper;
  gint               upper_length;
  gchar             *lower;
  gint               lower_length;
  gint               x, y, width, height;
};

GType himExample_vkb_get_type (void);
GtkWidget *himExample_vkb_new (HildonIMUI *kbd);

/* Prototype declarations */
static void himExample_vkb_iface_init (HildonIMPluginIface *iface);
static void himExample_vkb_class_init (HimExampleVKBClass *klass);
static void finalize(GObject *obj);
static void get_property (GObject *object, guint prop_id, GValue *value, GParamSpec *pspec);
static void set_property(GObject *object, guint prop_id, const GValue *value, GParamSpec *pspec);
static void himExample_vkb_init (HimExampleVKB *vkb);
static void enable (HildonIMPlugin *plugin, gboolean init);
static void disable(HildonIMPlugin *plugin);
static void input_mode_changed (HildonIMPlugin *plugin);
static void clear(HildonIMPlugin *plugin);
static void client_widget_changed (HildonIMPlugin *plugin);
static void save_data(HildonIMPlugin *plugin);
static void mode_a(HildonIMPlugin *plugin); 
static void mode_b(HildonIMPlugin *plugin);
static void language (HildonIMPlugin *plugin);
static void language_settings_changed (HildonIMPlugin *plugin, gint index);
static void backspace (HildonIMPlugin *plugin);
static void enter (HildonIMPlugin *plugin);
static void tab (HildonIMPlugin *plugin);
static void settings_changed (HildonIMPlugin *plugin, const gchar *key, const GConfValue *value);

static void draw_keyboard (HimExampleVKB *);
static void update_layout (HimExampleVKB *);
static void send_key (HimExampleVKB *, VKBKey *key);

static gboolean expose_cb (GtkWidget *, GdkEventExpose *);
static gboolean destroy_cb (GtkWidget *, GdkEventAny *);
static void realize_cb (GtkWidget *);
static void size_allocate_cb (GtkWidget *, GtkAllocation *);
static gboolean button_press_cb (GtkWidget *, GdkEventButton *);
static gboolean button_release_cb (GtkWidget *, GdkEventButton *);

/**
 * Module functions
 *
 **/

HildonIMPlugin* 
module_create (HildonIMUI *keyboard)
{
  return HILDON_IM_PLUGIN (himExample_vkb_new (keyboard));
}

void
module_exit(void)
{
  /* empty */
}

void
module_init(GTypeModule *module)
{
  static const GTypeInfo type_info = {
    sizeof(HimExampleVKBClass),
    NULL, /* base_init */
    NULL, /* base_finalize */
    (GClassInitFunc) himExample_vkb_class_init,
    NULL, /* class_finalize */
    NULL, /* class_data */
    sizeof(HimExampleVKB),
    0, /* n_preallocs */
    (GInstanceInitFunc) himExample_vkb_init,
  };
  
  static const GInterfaceInfo plugin_info = {
    (GInterfaceInitFunc) himExample_vkb_iface_init,
    NULL, /* interface_finalize */
    NULL, /* interface_data */
  };

  himExample_vkb_type =
          g_type_module_register_type(module,
                                      GTK_TYPE_WIDGET, "HimExampleVKB",
                                      &type_info,
                                      0);
  
  g_type_module_add_interface(module,
                              HIMEXAMPLE_VKB_TYPE,
                              HILDON_IM_TYPE_PLUGIN,
                              &plugin_info);
}

GType
himExample_vkb_get_type (void)
{
  return himExample_vkb_type;
}

/* Standard GTK stuff */
static void
himExample_vkb_iface_init (HildonIMPluginIface *iface)
{
  iface->enable = enable;
  iface->disable = disable;
  iface->enter = enter;
  iface->tab = tab;
  iface->backspace = backspace;
  iface->clear = clear;
  iface->input_mode_changed = input_mode_changed;
  iface->client_widget_changed = client_widget_changed;
  iface->save_data = save_data;
  iface->language = language;
  iface->mode_a = mode_a;
  iface->mode_b = mode_b;
  iface->language_settings_changed = language_settings_changed;
  iface->settings_changed = settings_changed;

  return;
}

static void
himExample_vkb_class_init (HimExampleVKBClass *klass)
{
  GObjectClass *object_class;
  GtkObjectClass *gtk_object_class;
  GtkWidgetClass *widget_class;

  parent_class = g_type_class_peek_parent(klass);
  g_type_class_add_private(klass, sizeof(HimExampleVKBPrivate));

  object_class = G_OBJECT_CLASS(klass);
  gtk_object_class = GTK_OBJECT_CLASS(klass);
  widget_class = GTK_WIDGET_CLASS(klass);

  object_class->set_property  = set_property;
  object_class->get_property  = get_property;
  object_class->finalize      = finalize;

  widget_class->realize       = realize_cb;
  widget_class->destroy_event = destroy_cb;
  widget_class->expose_event  = expose_cb;
  widget_class->size_allocate = size_allocate_cb;

  widget_class->button_press_event    = button_press_cb;
  widget_class->button_release_event  = button_release_cb;

  g_object_class_install_property(object_class, HILDON_IM_PROP_UI,
                                  g_param_spec_object(HILDON_IM_PROP_UI_DESCRIPTION, 
                                  HILDON_IM_PROP_UI_DESCRIPTION,
                                  "Keyboard that uses plugin",
                                  HILDON_IM_TYPE_UI,
                                  G_PARAM_READWRITE
                                  | G_PARAM_CONSTRUCT_ONLY));  
}

/* Input Method plugin information.
 * This structure tells the main UI about this plugin */
const HildonIMPluginInfo *
hildon_im_plugin_get_info(void)
{
  static const HildonIMPluginInfo info =
  {
    "HIM VKB Example",                  /* description */
    "himExample_vkb",                   /* name */
    "Keyboard (EXAMPLE)",               /* menu title */
    NULL,                               /* gettext domain */
    TRUE,                               /* visible in menu */
    FALSE,                              /* cached */
    HILDON_IM_TYPE_DEFAULT,             /* UI type */
    HILDON_IM_GROUP_LATIN,              /* group */
    HILDON_IM_DEFAULT_PLUGIN_PRIORITY,  /* priority */
    NULL,                               /* special character plugin */
    "",                                 /* help page */
    FALSE,                              /* disable common UI buttons */
    HILDON_IM_DEFAULT_HEIGHT,           /* plugin height */
    HILDON_IM_TRIGGER_STYLUS            /* trigger */
  };

  return &info;
}

/*
 * This function returns the list of available languages supported
 * by the plugin.
 * */
gchar ** 
hildon_im_plugin_get_available_languages (gboolean *free)
{
  static gchar *list[] = { "en_GB", NULL };
  *free = FALSE;

  return list;
}

static void 
finalize(GObject *obj)
{
  if (G_OBJECT_CLASS(parent_class)->finalize)
  {
    G_OBJECT_CLASS(parent_class)->finalize(obj);
  }
}

static void
get_property (GObject *object,
              guint prop_id,
              GValue *value,
              GParamSpec *pspec)
{
  HimExampleVKBPrivate *priv;

  g_return_if_fail (IS_HIMEXAMPLE_VKB(object));
  priv = HIMEXAMPLE_VKB_GET_PRIVATE(object);

  switch (prop_id)
  {
    case HILDON_IM_PROP_UI:
      g_value_set_object(value, priv->ui);
      break;

    default:
      G_OBJECT_WARN_INVALID_PROPERTY_ID(object, prop_id, pspec);
      break;
  }
}

static void
set_property(GObject *object,
             guint prop_id,
             const GValue *value,
             GParamSpec *pspec)
{
  HimExampleVKBPrivate *priv;

  g_return_if_fail (IS_HIMEXAMPLE_VKB (object));
  priv = HIMEXAMPLE_VKB_GET_PRIVATE(object);

  switch (prop_id)
  {
    case HILDON_IM_PROP_UI:
      priv->ui = g_value_get_object(value);
      break;
    default:
      G_OBJECT_WARN_INVALID_PROPERTY_ID(object, prop_id, pspec);
      break;
  }
}

/* Implementation of plugin interface starts here */
static void
himExample_vkb_init (HimExampleVKB *vkb)
{
  HimExampleVKBPrivate *priv;

  g_return_if_fail (IS_HIMEXAMPLE_VKB (vkb));

  priv = HIMEXAMPLE_VKB_GET_PRIVATE (vkb);

  priv->keys          = NULL;
  priv->screen_mode   = SCREEN_NORMAL;
  priv->case_mode     = CASE_LOWER;
}

GtkWidget *
himExample_vkb_new (HildonIMUI *kbd)
{
  return g_object_new (HIMEXAMPLE_VKB_TYPE, HILDON_IM_PROP_UI_DESCRIPTION, kbd, NULL);
}

/* Called when the plugin is available to the user */
static void
enable (HildonIMPlugin *plugin, gboolean init)
{
  HimExampleVKBPrivate *priv;
  HimExampleVKB *vkb;

  vkb = HIMEXAMPLE_VKB (plugin);
  priv = HIMEXAMPLE_VKB_GET_PRIVATE (vkb);
  if (init == TRUE) 
  {
    hildon_im_ui_button_set_toggle (priv->ui, 
      HILDON_IM_BUTTON_MODE_A, TRUE);
    hildon_im_ui_button_set_toggle (priv->ui, 
      HILDON_IM_BUTTON_MODE_B, TRUE);
    hildon_im_ui_button_set_label (priv->ui, 
      HILDON_IM_BUTTON_MODE_A, "ABC");
    hildon_im_ui_button_set_label (priv->ui, 
      HILDON_IM_BUTTON_MODE_B, "Shift");
  }

  hildon_im_ui_send_communication_message(priv->ui,
                                          HILDON_IM_CONTEXT_DIRECT_MODE);
}

/* Called when the plugin is disabled */
static void
disable(HildonIMPlugin *plugin)
{
  /* not implemented */
}

/* Called when input mode changed */
static void
input_mode_changed (HildonIMPlugin *plugin)
{
  /* not implemented */
}

/* Called when the plugin is requested to 'clear'/refresh its UI */
static void
clear(HildonIMPlugin *plugin)
{
  /* not implemented */
}

/* Called when the client widget changed */
static void
client_widget_changed (HildonIMPlugin *plugin)
{
  /* not implemented */
}

/* Called when the plugin is requested to save its data */
static void
save_data(HildonIMPlugin *plugin)
{
  /* not implemented */
}

/* Called when the MODE_A button is pressed */
static void
mode_a(HildonIMPlugin *plugin)
{
  HimExampleVKBPrivate *priv;
  HimExampleVKB *vkb;

  vkb = HIMEXAMPLE_VKB (plugin);

  priv = HIMEXAMPLE_VKB_GET_PRIVATE (vkb);
  if (hildon_im_ui_button_get_active (priv->ui,
        HILDON_IM_BUTTON_MODE_B)) {
    hildon_im_ui_button_set_active (priv->ui,
        HILDON_IM_BUTTON_MODE_B, FALSE);
    if (hildon_im_ui_button_get_active (priv->ui,
        HILDON_IM_BUTTON_MODE_A)) {
      priv->case_mode = CASE_UPPER;
    } else {
      priv->case_mode = CASE_LOWER;
    }
  } else {
    if (hildon_im_ui_button_get_active (priv->ui,
        HILDON_IM_BUTTON_MODE_A)) {
      priv->case_mode = CASE_UPPER;
    } else {
      priv->case_mode = CASE_LOWER;
    }
  }
  
  update_layout (vkb);
}

/* Called when the MODE_B button is pressed */
static void
mode_b(HildonIMPlugin *plugin)
{
  HimExampleVKBPrivate *priv;
  HimExampleVKB *vkb;

  vkb = HIMEXAMPLE_VKB (plugin);

  priv = HIMEXAMPLE_VKB_GET_PRIVATE (vkb);

  if (hildon_im_ui_button_get_active (priv->ui,
        HILDON_IM_BUTTON_MODE_B)) {
    if (hildon_im_ui_button_get_active (priv->ui,
        HILDON_IM_BUTTON_MODE_A)) {
      priv->case_mode = CASE_LOWER;
    } else {
      priv->case_mode = CASE_UPPER;
    }
  } else {
    if (hildon_im_ui_button_get_active (priv->ui,
        HILDON_IM_BUTTON_MODE_A)) {
      priv->case_mode = CASE_UPPER;
    } else {
      priv->case_mode = CASE_LOWER;
    }
  }

  update_layout (vkb);
}

/* Called when the language has been changed */
static void
language (HildonIMPlugin *plugin)
{
  /* not implemented */
}

/* CAlled when the language settings has been changed */
static void
language_settings_changed (HildonIMPlugin *plugin,
    gint index)
{
  /* not implemented */
}

/* Called when the backspace button is pressed */
static void
backspace (HildonIMPlugin *plugin)
{
  HimExampleVKBPrivate *priv;

  priv = HIMEXAMPLE_VKB_GET_PRIVATE (HIMEXAMPLE_VKB (plugin));
  hildon_im_ui_send_communication_message (priv->ui, 
      HILDON_IM_CONTEXT_HANDLE_BACKSPACE);
}

/* Called when the enter button is pressed */
static void
enter (HildonIMPlugin *plugin)
{
  HimExampleVKBPrivate *priv;

  priv = HIMEXAMPLE_VKB_GET_PRIVATE (HIMEXAMPLE_VKB (plugin));
  hildon_im_ui_send_communication_message (priv->ui, 
      HILDON_IM_CONTEXT_HANDLE_ENTER);
}

/* Called when the tab button is pressed */
static void
tab (HildonIMPlugin *plugin)
{
  HimExampleVKBPrivate *priv;

  priv = HIMEXAMPLE_VKB_GET_PRIVATE (HIMEXAMPLE_VKB (plugin));
  hildon_im_ui_send_communication_message (priv->ui, 
      HILDON_IM_CONTEXT_HANDLE_TAB);
}

/* Called when the standard input method settings 
 * has been changed */
static void
settings_changed (HildonIMPlugin *plugin,
    const gchar *key, const GConfValue *value)
{
  /* not implemented */
}

/* Supporting functions starts here */
static void 
draw_key (HimExampleVKB *vkb, VKBKey *key, GtkStateType state, gint case_mode)
{
  HimExampleVKBPrivate *priv;
  PangoRectangle extents;

  g_return_if_fail (IS_HIMEXAMPLE_VKB (vkb));
  g_return_if_fail (key != NULL);
  priv = HIMEXAMPLE_VKB_GET_PRIVATE (vkb);

  if (case_mode == CASE_UPPER) {
    pango_layout_set_text (priv->layout, key->upper, key->upper_length);
  } else { 
    pango_layout_set_text (priv->layout, key->lower, key->lower_length);
  }

  priv->style->depth = gdk_drawable_get_depth (((GtkWidget*)vkb)->window);

  pango_layout_get_pixel_extents(priv->layout, &extents, NULL);
  
  gtk_paint_box (priv->style, 
      GTK_WIDGET(vkb)->window, 
      state, 
      GTK_SHADOW_IN, NULL, GTK_WIDGET(vkb), NULL, 
      key->x, key->y, 
      key->width, 
      key->height);

  gtk_paint_layout (GTK_WIDGET(vkb)->style, 
      GTK_WIDGET(vkb)->window, 
      state, 
      TRUE, NULL, GTK_WIDGET(vkb), NULL, 
      key->x + ((key->width - extents.width) / 2), key->y, 
      priv->layout);
}

static void 
draw_keyboard (HimExampleVKB *vkb)
{
  HimExampleVKBPrivate *priv;
  VKBKey          *key;
  gchar           *tmp_upper,
                  *tmp_lower;

  gint i, j, x, y, width;
  gint start_x;

  g_return_if_fail (IS_HIMEXAMPLE_VKB (vkb));
  priv = HIMEXAMPLE_VKB_GET_PRIVATE (vkb);

  priv->total_keys = TOTAL_KEYPAD_KEYS;
  for (i = 0; i < KEYBOARD_ROWS; i ++)
  {
    priv->total_keys += keyboard_keys [i];
  }

  if (priv->keys == NULL)
  {
    priv->keys = g_malloc0 (sizeof(VKBKey) * priv->total_keys);
  }

  /* To handle insufficient memory error is left as an exercise
   * to the readers(TM) */

  key = priv->keys;
  
  start_x = START_X;
  y = START_Y;
  for (i = 0; i < KEYBOARD_ROWS; i ++)
  {
    x = start_x;
    tmp_upper = (gchar*) keyboard_layout [CASE_UPPER][i];
    tmp_lower = (gchar*) keyboard_layout [CASE_LOWER][i];
    for (j = 0; j < keyboard_keys [i]; j ++) 
    {
      if (i == KEYBOARD_ROWS - 1) {
        width = button_sizes [priv->screen_mode][KEYSIZE_NORMAL] * 
          keyboard_keys [1];
      }
      else if (((i % 2) == 0) &&
          ((j == 0) || 
           (j == keyboard_keys [i] - 1))
          ) {
        width = button_sizes [priv->screen_mode][KEYSIZE_BIG];
      } 
      else {
        width = button_sizes [priv->screen_mode][KEYSIZE_NORMAL];
      }
    
      key->upper  = tmp_upper;
      key->lower  = tmp_lower;
      tmp_upper   = g_utf8_find_next_char (tmp_upper, NULL);
      tmp_lower   = g_utf8_find_next_char (tmp_lower, NULL);

      if (tmp_upper != NULL) {
        key->upper_length = tmp_upper - key->upper;
      } else {
        key->upper_length = 0;
      }
      if (tmp_lower != NULL) {
        key->lower_length = tmp_lower - key->lower;
      } else {
        key->lower_length = 0;
      }
      key->x      = x;
      key->y      = y;
      key->width  = width;
      key->height = HEIGHT;
      draw_key (vkb, key, GTK_STATE_NORMAL, priv->case_mode);

      x += width;
      key ++;
    }
    y += HEIGHT;
  }

  start_x = priv->keypad_start_x =
    GTK_WIDGET (vkb)->allocation.width - 
    (button_sizes [priv->screen_mode][KEYSIZE_NORMAL] * 
    MAX_KEYPAD_KEYS + KEYPAD_MARGIN_RIGHT);
  
  y = START_Y;
  for (i = 0; i < KEYPAD_ROWS; i ++)
  {
    x = start_x;
    tmp_upper = (gchar*) keypad_layout [CASE_UPPER][i];
    tmp_lower = (gchar*) keypad_layout [CASE_LOWER][i];
    for (j = 0; j < MAX_KEYPAD_KEYS; j ++) 
    {
      width = button_sizes [priv->screen_mode][KEYSIZE_NORMAL];
      
      key->upper  = tmp_upper;
      key->lower  = tmp_lower;
      tmp_upper   = g_utf8_find_next_char (tmp_upper, NULL);
      tmp_lower   = g_utf8_find_next_char (tmp_lower, NULL);

      if (tmp_upper != NULL) {
        key->upper_length = tmp_upper - key->upper;
      } else {
        key->upper_length = 0;
      }
      if (tmp_lower != NULL) {
        key->lower_length = tmp_lower - key->lower;
      } else {
        key->lower_length = 0;
      }
      key->x      = x;
      key->y      = y;
      key->width  = width;
      key->height = HEIGHT;

      draw_key (vkb, key, GTK_STATE_NORMAL, priv->case_mode);

      x += width;
      key ++;
    }
    y += HEIGHT;
  }
}

static void 
update_layout (HimExampleVKB *vkb)
{
  HimExampleVKBPrivate *priv;

  priv = HIMEXAMPLE_VKB_GET_PRIVATE (vkb);

  draw_keyboard (vkb);
}

static void 
send_key (HimExampleVKB *vkb, VKBKey *key)
{
  HimExampleVKBPrivate *priv;
#define UTF8_SIZE 6
  gchar   text [UTF8_SIZE];

  if (key != NULL) 
  {
    priv = HIMEXAMPLE_VKB_GET_PRIVATE (vkb);    
    memset (text, 0, UTF8_SIZE);
    if (priv->case_mode == CASE_UPPER) 
    {
      memcpy (text, key->upper, key->upper_length);
    } else {
      memcpy (text, key->lower, key->lower_length);
    }

    if (hildon_im_ui_button_get_active (priv->ui,
        HILDON_IM_BUTTON_MODE_B)) {
      hildon_im_ui_button_set_active (priv->ui,
        HILDON_IM_BUTTON_MODE_B, FALSE);
      if (hildon_im_ui_button_get_active (priv->ui,
        HILDON_IM_BUTTON_MODE_A)) {
        priv->case_mode = CASE_UPPER;
      } else {
        priv->case_mode = CASE_LOWER;
      }
      update_layout (vkb);
    }
    hildon_im_ui_send_utf8(priv->ui, text);
  }
}

/* Callbacks starts here */
static gboolean 
expose_cb (GtkWidget *widget, GdkEventExpose *event)
{
  draw_keyboard (HIMEXAMPLE_VKB (widget));
  return TRUE;
}

static gboolean
destroy_cb (GtkWidget *widget, GdkEventAny *event)
{
  HimExampleVKB            *vkb;
  HimExampleVKBPrivate     *priv;

  vkb = HIMEXAMPLE_VKB (widget);
  priv = HIMEXAMPLE_VKB_GET_PRIVATE (vkb);
  if (priv->keys != NULL)
  {
    g_free (priv->keys);
    priv->keys = NULL;
  }

  return FALSE;
}

static void
realize_cb (GtkWidget *widget)
{
  HimExampleVKB        *vkb;
  HimExampleVKBPrivate *priv;
  GdkScreen            *screen;
  GdkWindowAttr         attributes;
  gint                  attributes_mask;
  gint                  depth;

  g_return_if_fail (IS_HIMEXAMPLE_VKB (widget));
  vkb = HIMEXAMPLE_VKB (widget);
  priv = HIMEXAMPLE_VKB_GET_PRIVATE (vkb);

  GTK_WIDGET_SET_FLAGS (widget, GTK_REALIZED);
  attributes.x = widget->allocation.x;
  attributes.y = widget->allocation.y;
  attributes.width = widget->allocation.width;
  attributes.height = widget->allocation.height;
 
  attributes.wclass = GDK_INPUT_OUTPUT;
  attributes.window_type = GDK_WINDOW_CHILD;
  attributes.event_mask = 
    gtk_widget_get_events (widget)  |
    GDK_EXPOSURE_MASK               | 
    GDK_BUTTON_PRESS_MASK           |
    GDK_BUTTON_RELEASE_MASK;
  attributes.visual = gtk_widget_get_visual (widget);
  attributes.colormap = gtk_widget_get_colormap (widget);
  attributes_mask = GDK_WA_X | GDK_WA_Y | GDK_WA_VISUAL | GDK_WA_COLORMAP;

  widget->window = gdk_window_new (
      gtk_widget_get_parent_window (widget), 
      &attributes, attributes_mask);

  widget->style = gtk_style_attach(widget->style, widget->window);

  gdk_window_set_user_data (widget->window, widget);
  gtk_style_set_background (widget->style, widget->window, GTK_STATE_NORMAL);

  priv->style = gtk_rc_get_style_by_paths (
      gtk_widget_get_settings(GTK_WIDGET(widget)),
      ".osso-im-key", NULL, GTK_TYPE_BUTTON);

  screen = gtk_widget_get_screen(widget);
  depth = DefaultDepth (
      GDK_SCREEN_XDISPLAY (screen), 
      GDK_SCREEN_XNUMBER (screen));

  priv->layout = pango_layout_new (
      gtk_widget_get_pango_context (widget));
}

static void 
size_allocate_cb (GtkWidget *widget, 
    GtkAllocation *allocation)
{
  HimExampleVKBPrivate     *priv;

  g_return_if_fail (IS_HIMEXAMPLE_VKB (widget));
  priv = HIMEXAMPLE_VKB_GET_PRIVATE (HIMEXAMPLE_VKB (widget));

  widget->allocation = *allocation;

  if (GTK_WIDGET_REALIZED (widget))
  {
    gdk_window_move_resize (widget->window, 
        allocation->x, allocation->y,
        allocation->width, allocation->height);

    if (allocation->width > FULLSCREEN_WIDTH) {
      priv->screen_mode = SCREEN_FULL;
    } else {
      priv->screen_mode = SCREEN_NORMAL;
    }
    draw_keyboard (HIMEXAMPLE_VKB (widget));
  }
}

static VKBKey *
get_key (HimExampleVKB *vkb, gint x, gint y)
{
  HimExampleVKBPrivate   *priv;

  gint row;
  gint keyboard_width;
  gint i, offset, size;

  priv = HIMEXAMPLE_VKB_GET_PRIVATE (vkb);
  
  keyboard_width = button_sizes [priv->screen_mode][KEYSIZE_NORMAL] * 
    keyboard_keys [1] + START_X;

  /* Ignore if it fall on the 'dead zones' */
  if ((x < START_X) ||
      (y < START_Y) ||
      ((x > keyboard_width) && 
       (x < priv->keypad_start_x)) ||
      (x > GTK_WIDGET (vkb)->allocation.width - KEYPAD_MARGIN_RIGHT)
      ) {    
    return NULL;
  }

  row = y / HEIGHT;
  if ((x < keyboard_width) &&
          (row == (KEYBOARD_ROWS - 1)))
  {
    return priv->keys + (priv->total_keys - TOTAL_KEYPAD_KEYS - 1);
  }

  if (x > priv->keypad_start_x) 
  {
    i = 0;
    while (i < MAX_KEYPAD_KEYS)
    {
      if (x > (priv->keypad_start_x + 
            button_sizes [priv->screen_mode][KEYSIZE_NORMAL] * 
            (i + 1)))
      {
        i ++;
      } else {
        break;
      }
    }
    return priv->keys + (priv->total_keys - 
        TOTAL_KEYPAD_KEYS + 
        (MAX_KEYPAD_KEYS * row) +
        i
        );
  } else {
    if (row == 1) {
      i = 0;
      while (i < keyboard_keys [1])
      {
        if (x > (START_X + 
              button_sizes [priv->screen_mode][KEYSIZE_NORMAL] * 
              (i + 1)))
        {
          i ++;
        } else {
          break;
        }
      }
      return priv->keys + (keyboard_keys [0] + i);
    } else {
      offset = 0;
      for (i = 0; i < row; i ++) 
      {
        offset += keyboard_keys [i];
      }

      size = button_sizes [priv->screen_mode][KEYSIZE_BIG];
      if ((x < size + START_X)) {
        i = 0;
      } else if (x > keyboard_width - size)         
      {
        i = keyboard_keys [row] - 1;
      } else {
        i = 0;
        while (i < keyboard_keys [row])
        {
          if (x > (START_X + size +
                button_sizes [priv->screen_mode][KEYSIZE_NORMAL] * 
                (i + 1)))
          {
            i ++;
          } else {
            break;
          }
        }
        i ++;
      }
      return priv->keys + (offset + i);
    }
  }
  return NULL;
}

static gboolean
button_press_cb (GtkWidget *widget, 
    GdkEventButton *event)
{
  VKBKey *key;
  HimExampleVKB *vkb;
  HimExampleVKBPrivate *priv;

  vkb = HIMEXAMPLE_VKB (widget);
  key = get_key (vkb, event->x, event->y);

  if (key != NULL) 
  {
    priv = HIMEXAMPLE_VKB_GET_PRIVATE (vkb);
    draw_key (vkb, key, GTK_STATE_ACTIVE, priv->case_mode);
    priv->last_key = key;
    
  }
  return FALSE;
}

static gboolean
button_release_cb (GtkWidget *widget, 
    GdkEventButton *event)
{
  HimExampleVKB *vkb;
  HimExampleVKBPrivate *priv;

  vkb = HIMEXAMPLE_VKB (widget);
  priv = HIMEXAMPLE_VKB_GET_PRIVATE (vkb);

  if (priv->last_key != NULL) 
  {
    send_key (vkb, priv->last_key);
    draw_key (HIMEXAMPLE_VKB (widget), 
        priv->last_key, 
        GTK_STATE_NORMAL,
        priv->case_mode);
    priv->last_key = NULL;
  }
  return FALSE;
}

/* The end of the road */
