/***************************************************************************
 * CVSID: $Id$
 *
 * probe-omap.c : Probe OMAP system specific features
 *
 * Copyright (C) 2005 David Zeuthen, <david@fubar.dk>
 * Copyright (C) 2005 Richard Hughes, <richard@hughsie.com>
 * Copyright (C) 2007 Nokia Corporation
 *
 * Licensed under the Academic Free License version 2.1
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 *
 **************************************************************************/

#ifdef HAVE_CONFIG_H
#  include <config.h>
#endif

#include <ctype.h>
#include <fcntl.h>
#include <stdint.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <unistd.h>

#include "libhal/libhal.h"
#include "../../logger.h"

#define strbegin(buf, str) (strncmp (buf, str, strlen (str)) == 0)

#define UDI_NOKIA "/com/nokia"

/* global */
static LibHalChangeSet *cs = NULL;

static inline int
is_field_separator (char c)
{
	return (isspace (c) || ':' == c);
}

static inline void
setkey (char *prop, char *value)
{
	libhal_changeset_set_property_string (cs, prop, value);
	HAL_DEBUG (("Setting %s='%s'", prop, value));
}

/** Finds the start of a null terminated string and sets HAL
 *  property if valid.
 *
 *  @param	buf		The non tabbed prefixed, null terminated string
 *  @param	str		The strings to compare with e.g. "Vendor:"
 *  @param	prop		The HAL property to set
 *  @return			TRUE is found, FALSE otherwise.
 */
static int
setstr (char *buf, char *str, char *prop)
{
	if (strbegin (buf, str)) {
		char *value;

		value = buf + strlen (str);

		/* skip the field separator */
		while (is_field_separator (*value))
			value++;

		setkey (prop, value);

		return TRUE;
	}

	return FALSE;
}

static int
setstrsplit (char *buf, char *str, char *prop_a, char *prop_b)
{
	if (strbegin (buf, str)) {
		char *value_a, *value_b;

		value_a = buf + strlen (str);

		/* skip the field separator */
		while (is_field_separator (*value_a))
			value_a++;

		value_b = value_a;

		/* skip the field contents */
		while (!is_field_separator (*value_b))
			value_b++;

		/* terminate the first part */
		*value_b = '\0';
		value_b++;

		setkey (prop_a, value_a);
		setkey (prop_b, value_b);

		return TRUE;
	}

	return FALSE;
}


static void
parse_hwinfo (char *buf)
{
	/* XXX: Think about this 'hardware' key -> 'system.hardware'? */
	setstr (buf, "product", "system.hardware.product");
	setstr (buf, "hw-build", "system.hardware.version");
	if (setstr (buf, "nolo", "system.firmware.version"))
		setkey ("system.firmware.name", "NOLO");
}

static void
parse_cpuinfo (char *buf)
{
	setstrsplit (buf, "Hardware",
		     "system.hardware.vendor", "system.firmware.product");
	setstr (buf, "Revision", "system.hardware.uuid");
	setstr (buf, "Serial", "system.hardware.serial");
}

static int
parse_file (char *filename, void (*parse_fn)(char *))
{
	char buf[512];
	FILE *f;

	f = fopen (filename, "r");
	if (!f) {
		HAL_ERROR (("not running on an OMAP kernel\n"));
		return -1;
	}

	/* read the output of the child */
	while (fgets (buf, sizeof(buf), f) != NULL) {
		unsigned int i;
		unsigned int len;
		char *nbuf;

		/* trim whitespace */
		len = strlen (buf);

		/* check that will fit in buffer */
		if (len >= sizeof (buf))
			continue;

		/* not big enough for data, and protects us from underflow */
		if (len < 3)
			continue;

		nbuf = buf;

		/* removes the trailing spaces */
		for (i = len - 1; isspace (nbuf[i]) && i >= 0; --i)
			nbuf[i] = '\0';

		parse_fn (nbuf);
	}

	/* as read to EOF, close */
	fclose (f);

	return 0;
}


/** Main entry point
 *
 *  @param	argc		Number of arguments given to program
 *  @param	argv		Arguments given to program
 *  @return			Return code
 */
int
main (int argc, char *argv[])
{
	int ret;
	char *udi;
	DBusError error;
	LibHalContext *ctx = NULL;

	/* assume failure */
	ret = 1;

	setup_logger ();

	udi = getenv ("UDI");
	if (udi == NULL) {
		HAL_ERROR (("UDI not set"));
		goto out;
	}

	dbus_error_init (&error);
	if ((ctx = libhal_ctx_init_direct (&error)) == NULL) {
		HAL_ERROR (("ctx init failed"));
		goto out;
	}

	cs = libhal_device_new_changeset (udi);
	if (cs == NULL) {
		HAL_DEBUG (("Cannot initialize changeset"));
		goto out;
	}

	if (parse_file ("/proc/component_version", parse_hwinfo))
		goto out;
	if (parse_file ("/proc/cpuinfo", parse_cpuinfo))
		goto out;

	/* return success */
	ret = 0;

out:
	if (cs != NULL) {
		libhal_device_commit_changeset (ctx, cs, &error);
		libhal_device_free_changeset (cs);
	}

	/* free ctx */
	if (ctx != NULL) {
		dbus_error_init (&error);
		libhal_ctx_shutdown (ctx, &error);
		libhal_ctx_free (ctx);
	}

	return ret;
}

