/**
 * @file meta-object.c MetaObject API
 *
 * Copyright (C) 2004-2006 Christian Hammond.
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 2 of
 * the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public
 * License along with this program; if not, write to the Free
 * Software Foundation, Inc., 59 Temple Place, Suite 330, Boston,
 * MA  02111-1307  USA
 */
#include <glib.h>
#include "meta-object.h"

static void galagod_meta_object_finalize(GObject *gobject);
static void galagod_meta_object_dbus_message_append(
	DBusMessageIter *iter, const GalagoObject *object);

static GalagoObjectClass *parent_class = NULL;

G_DEFINE_TYPE(GalagodMetaObject, galagod_meta_object, GALAGO_TYPE_OBJECT);

static void
galagod_meta_object_class_init(GalagodMetaObjectClass *klass)
{
	GObjectClass *gobject_class;
	GalagoObjectClass *object_class;

	gobject_class = G_OBJECT_CLASS(klass);
	object_class  = GALAGO_OBJECT_CLASS(klass);

	parent_class = g_type_class_peek_parent(klass);

	gobject_class->finalize = galagod_meta_object_finalize;

	object_class->dbus_message_append = galagod_meta_object_dbus_message_append;

	klass->add_filter    = NULL;
	klass->remove_filter = NULL;
}

static void
galagod_meta_object_init(GalagodMetaObject *meta_object)
{
	meta_object->main_object = NULL;
	meta_object->children    = NULL;
};

static void
galagod_meta_object_finalize(GObject *gobject)
{
	GalagodMetaObjectClass *klass;
	GalagodMetaObject *meta_object = GALAGOD_META_OBJECT(gobject);

	klass = GALAGOD_META_OBJECT_CLASS(meta_object);

	if (meta_object->main_object != NULL)
	{
		if (klass->remove_filter != NULL)
			klass->remove_filter(meta_object->main_object);

		g_object_unref(meta_object->main_object);
	}

	if (meta_object->children != NULL)
	{
		const GList *l;

		for (l = meta_object->children; l != NULL; l = l->next)
			g_object_set_data(l->data, "meta-object", NULL);

		g_list_free(meta_object->children);
	}

	G_OBJECT_CLASS(parent_class)->finalize(gobject);
};

static void
galagod_meta_object_dbus_message_append(DBusMessageIter *iter,
										const GalagoObject *object)
{
	GalagodMetaObject *meta_object = (GalagodMetaObject *)object;

	galago_dbus_message_iter_append_object(iter, meta_object->main_object);
}

void *
galagod_meta_account_get_main_object(const GalagodMetaObject *meta_object)
{
	g_return_val_if_fail(meta_object != NULL,           NULL);
	g_return_val_if_fail(GALAGO_IS_OBJECT(meta_object), NULL);

	return meta_object->main_object;
}

const GList *
galagod_meta_objects_get_children(const GalagodMetaObject *meta_object)
{
	g_return_val_if_fail(meta_object != NULL,           NULL);
	g_return_val_if_fail(GALAGO_IS_OBJECT(meta_object), NULL);

	return meta_object->children;
}

void
galagod_meta_object_add_child(GalagodMetaObject *meta_object, void *object)
{
	g_return_if_fail(meta_object != NULL);
	g_return_if_fail(GALAGO_IS_OBJECT(meta_object));

	/* TODO: Class check. */

	meta_object->children = g_list_append(meta_object->children, object);
}

void
galagod_meta_object_remove_child(GalagodMetaObject *meta_object, void *object)
{
	g_return_if_fail(meta_object != NULL);
	g_return_if_fail(GALAGO_IS_OBJECT(meta_object));

	/* TODO: Class check. */

	meta_object->children = g_list_remove(meta_object->children, object);
}
