/*
 * This file is part of cx3110x
 *
 * Copyright (C) 2005, 2006 Nokia Corporation
 *
 * Contact: Kalle Valo <Kalle.Valo@nokia.com>
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License 
 * version 2 as published by the Free Software Foundation. 
 *
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA
 *
 */

#include <linux/kernel.h>
#include <linux/init.h>
#include <linux/device.h>
#include <linux/platform_device.h>

#include "sm_drv.h"
#include "sm_drv_pda.h"
#include "sm_drv_sysfs.h"

#ifdef CONFIG_SYSFS

extern char wlan_fw_version[32];
static ssize_t sm_drv_show_fw_version(struct device *dev,
				      struct device_attribute *attr,
				      char *buf)
{
	return sprintf(buf, "%s\n", wlan_fw_version);
}

/* 
 * CAL interface.
 * Those sysfs files are for user space
 * to pass the calibration data through.
 */
struct wlan_pda_mac_addr_s wlan_pda_mac_address;
struct wlan_pda_rssi_cal_s * wlan_pda_rssi;
struct wlan_pda_iq_autocal_s * wlan_pda_iq;
struct wlan_pda_output_limit_s * wlan_pda_output_limits;
struct wlan_pda_pa_curve_data_s * wlan_pda_pa_curve_data;
/*
 * Default country. This values depends on
 * the 770 product code. We have a product code 
 * for every SW variant.
 */
struct wlan_pda_default_country_s * wlan_pda_default_country;


static ssize_t sm_drv_store_cal_mac_address(struct device *dev,
					    struct device_attribute *attr, 
					    const char *buf, size_t count)
{
	struct net_device * net_dev = dev_get_drvdata(dev);

	if (!memcpy(wlan_pda_mac_address.mac_addr, buf, ETH_ALEN))
		return 0;
		
	if (!memcpy(net_dev->dev_addr, wlan_pda_mac_address.mac_addr,
		    ETH_ALEN))
		return 0;
	
	DEBUG(DBG_SYSFS,
	      "MAC address received:%.2x:%.2x:%.2x:%.2x:%.2x:%.2x\n",
	      wlan_pda_mac_address.mac_addr[0],
	      wlan_pda_mac_address.mac_addr[1],
	      wlan_pda_mac_address.mac_addr[2],
	      wlan_pda_mac_address.mac_addr[3],
	      wlan_pda_mac_address.mac_addr[4],
	      wlan_pda_mac_address.mac_addr[5]);

	return count;
}

static ssize_t sm_drv_show_cal_mac_address(struct device *dev,
					   struct device_attribute *attr,
					   char *buf)
{
	return sprintf(buf, "%.2x:%.2x:%.2x:%.2x:%.2x:%.2x\n",
		       wlan_pda_mac_address.mac_addr[0],
		       wlan_pda_mac_address.mac_addr[1],
		       wlan_pda_mac_address.mac_addr[2],
		       wlan_pda_mac_address.mac_addr[3],
		       wlan_pda_mac_address.mac_addr[4],
		       wlan_pda_mac_address.mac_addr[5]);
}

static ssize_t sm_drv_store_cal_rssi(struct device *dev,
				     struct device_attribute *attr, 
				     const char *buf, size_t count)
{
	if (wlan_pda_rssi != NULL)
		kfree(wlan_pda_rssi);

	wlan_pda_rssi = kmalloc(count, GFP_KERNEL);
	
	if (!wlan_pda_rssi)
		return 0;

	if (!memcpy(wlan_pda_rssi, buf, count))
		return 0;
	
	return count;
}

static ssize_t sm_drv_show_cal_rssi(struct device *dev,
				    struct device_attribute *attr, 
				    char *buf)
{
	int offset = 0;

	if (wlan_pda_rssi) {
		int i;
		offset += sprintf(buf + offset,
				  "Dumping RSSI calibration values:\n");
		
		for (i = 0; i < NUM_CHANNELS; i++) {
			offset += sprintf(buf + offset,
					  "Freq %d (channel %d):\n",
					  wlan_pda_rssi->rssi_data[i].frequency,
					  i + 1);
			offset += sprintf(buf + offset,
					  "\tRSSI A: %d\n",
					  wlan_pda_rssi->rssi_data[i].rssi_A);
			offset += sprintf(buf + offset,
					  "\tRSSI B: %d\n",
					  wlan_pda_rssi->rssi_data[i].rssi_B);
			offset += sprintf(buf + offset,
					  "\tRSSI LNA: %d\n",
					  wlan_pda_rssi->rssi_data[i].rssi_LNA);
		}
	}

	return offset;
}

static ssize_t sm_drv_store_cal_iq(struct device *dev,
				   struct device_attribute *attr, 
				   const char *buf, size_t count)
{
	if (wlan_pda_iq != NULL)
		kfree(wlan_pda_iq);
	
	wlan_pda_iq = kmalloc(count, GFP_KERNEL);
	if (!wlan_pda_iq)
		return 0;

	if (!memcpy(wlan_pda_iq, buf, count))
		return 0;

	return count;
}

static ssize_t sm_drv_show_cal_iq(struct device *dev,
				  struct device_attribute *attr, 
				  char *buf)
{
	int offset = 0;
	
	if (wlan_pda_iq) {
		int i;
		offset += sprintf(buf + offset, "Dumping IQ tuning values:\n");
		
		for (i = 0; i < NUM_CHANNELS; i++) {
			offset += sprintf(buf + offset,
					  "Freq %d (channel %d):\n",
					  wlan_pda_iq[i].frequency, i + 1);
			offset += sprintf(buf + offset,
					  "\tparam 1: %d\n",
					  wlan_pda_iq[i].iq_param_1);
			offset += sprintf(buf + offset,
					  "\tparam 2: %d\n",
					  wlan_pda_iq[i].iq_param_2);
			offset += sprintf(buf + offset,
					  "\tparam 3: %d\n",
					  wlan_pda_iq[i].iq_param_3);
			offset += sprintf(buf + offset,
					  "\tparam 4: %d\n",
					  wlan_pda_iq[i].iq_param_4);
		}
	}

	return offset;
}

static ssize_t sm_drv_store_cal_output_limits(struct device *dev,
					      struct device_attribute *attr, 
					      const char *buf, size_t count)
{
	if (wlan_pda_output_limits != NULL)
		kfree(wlan_pda_output_limits);
	
	wlan_pda_output_limits = kmalloc(count, GFP_KERNEL);
	
	if (!wlan_pda_output_limits)
		return 0;

	if (!memcpy(wlan_pda_output_limits, buf, count))
		return 0;

	if (wlan_pda_output_limits->pdr_revision != PHASER_PDR1903_REV)
		return 0;
	
	return count;
}

static ssize_t sm_drv_show_cal_output_limits(struct device *dev,
					     struct device_attribute *attr,
					     char *buf)
{
	int offset = 0;
	
	if (wlan_pda_output_limits) {
		int i;
		
		offset += sprintf(buf + offset, "Dumping output limits:\n");
		offset += sprintf(buf + offset, "Output revision: 0x%x\n",
				  wlan_pda_output_limits->pdr_revision);
		offset += sprintf(buf + offset, "Xcvr_bmp: %d\n",
				  wlan_pda_output_limits->xcvr_bmp);
		offset += sprintf(buf + offset, "N PA levels: %d\n",
				  wlan_pda_output_limits->n_pa_levels);
		
		for (i = 0; i < NUM_CHANNELS; i++) {
			offset += sprintf(buf + offset, "\tFrequency: %d\n",
					  wlan_pda_output_limits->power[i].frequency);
			offset += sprintf(buf + offset,
					  "\t\tRate set mask: 0x%x\n",
					  wlan_pda_output_limits->power[i].rate_set_mask);
			offset += sprintf(buf + offset,
					  "\t\tRate set size: 0x%x\n",
					  wlan_pda_output_limits->power[i].rate_set_size);
			offset += sprintf(buf + offset, "\t\tBArker: %d\n",
					  wlan_pda_output_limits->power[i].limit_barker);
			offset += sprintf(buf + offset, "\t\tBPSK:   %d\n",
					  wlan_pda_output_limits->power[i].limit_bpsk);
			offset += sprintf(buf + offset, "\t\tQPSK:   %d\n",
					  wlan_pda_output_limits->power[i].limit_qpsk);
			offset += sprintf(buf + offset, "\t\t16QAM:  %d\n",
					  wlan_pda_output_limits->power[i].limit_16qam);
			offset += sprintf(buf + offset, "\t\t64QAM:  %d\n\n",
					  wlan_pda_output_limits->power[i].limit_64qam);
			
		}
	}
		
	return offset;
}


static ssize_t sm_drv_store_cal_pa_curve_data(struct device *dev,
					      struct device_attribute *attr, 
					      const char *buf, size_t count)
{
	if (wlan_pda_pa_curve_data != NULL)
		kfree(wlan_pda_pa_curve_data);

	wlan_pda_pa_curve_data = kmalloc(count, GFP_KERNEL);
	
	if (!wlan_pda_pa_curve_data)
		return 0;

	if (!memcpy(wlan_pda_pa_curve_data, buf, count))
		return 0;

	if (wlan_pda_pa_curve_data->cal_method_revision
	    != PHASER_PA_CAL_METHOD)
		return 0;
		
	return count;
}

static ssize_t sm_drv_show_cal_pa_curve_data(struct device *dev,
					     struct device_attribute *attr, 
					     char *buf)
{
	int offset = 0;

	if (wlan_pda_pa_curve_data) {
		int i,j;
		
		offset += sprintf(buf + offset, "Dumping PA curve data:\n");
		offset += sprintf(buf + offset,
				  "Calibration method: 0x%x\n",
				  wlan_pda_pa_curve_data->cal_method_revision);
		offset += sprintf(buf + offset,
				  "Points per channel: %d\n",
				  wlan_pda_pa_curve_data->points_per_channel);
		for (i = 0; i < NUM_CHANNELS; i++) {
			offset += sprintf(buf + offset,
					  "\tFreq:%d (channel %d)\n",
					  wlan_pda_pa_curve_data->curve_data[i].frequency,
					  i + 1);
			offset += sprintf(buf + offset,
					  "\tRate mask:0x%x\n",
					  wlan_pda_pa_curve_data->curve_data[i].rate_set_mask);
			offset += sprintf(buf + offset,
					  "\tRate size:%d\n",
					  wlan_pda_pa_curve_data->curve_data[i].rate_set_size);

			for (j = 0; j < POINTS_PER_CHANNEL; j++) {
				offset += sprintf(buf + offset,
						  "\t\tSample %d:\n", j + 1);

				offset += sprintf(buf + offset,
						  "\t\t\t(%d,%d,%d)\n", 
						  wlan_pda_pa_curve_data->curve_data[i].curve_samples[j].rf_power,
						  wlan_pda_pa_curve_data->curve_data[i].curve_samples[j].pa_detector,
						  wlan_pda_pa_curve_data->curve_data[i].curve_samples[j].pcv);
			}

		}
		
	}
	
	return offset;
}

static ssize_t sm_drv_store_default_country(struct device *dev,
					    struct device_attribute *attr,
					    const char *buf, size_t count)
{
	if (wlan_pda_default_country != NULL)
		kfree(wlan_pda_default_country);

	wlan_pda_default_country = kmalloc(count, GFP_KERNEL);
	
	if (!wlan_pda_default_country)
		return 0;

	if (!memcpy(wlan_pda_default_country, buf, count))
		return 0;
	
	return count;
}

static ssize_t sm_drv_show_default_country(struct device *dev,
					   struct device_attribute *attr,
					   char *buf)
{
	if (wlan_pda_default_country)
		return sprintf(buf, "Regulatory domain: 0x%x\n",
			       wlan_pda_default_country->regulatory_domain);
	else
		return sprintf(buf, "\n");
}

static ssize_t sm_drv_store_RSSI_A(struct device *dev,
				   struct device_attribute *attr,
				   const char *buf, size_t count)
{
	struct net_device * net_dev;
	struct net_local *lp;
	int i;
	uint16_t rssi_a = simple_strtol(buf, NULL, 0);
	
	net_dev = (struct net_device *)dev_get_drvdata(dev);
	lp = net_dev->priv;
	
	for (i = 0; i < NUM_CHANNELS; i++)
		wlan_pda_rssi->rssi_data[i].rssi_A = rssi_a;
	
	return count;
}

static ssize_t sm_drv_store_RSSI_B(struct device *dev,
				   struct device_attribute *attr,
				   const char *buf, size_t count)
{
	struct net_device * net_dev;
	struct net_local *lp;
	int i;
	uint16_t rssi_b = simple_strtol(buf, NULL, 0);
	
	net_dev = (struct net_device *)dev_get_drvdata(dev);
	lp = net_dev->priv;
	
	for (i = 0; i < NUM_CHANNELS; i++)
		wlan_pda_rssi->rssi_data[i].rssi_B = rssi_b;
	
	return count;
}


static ssize_t sm_drv_store_LNA(struct device *dev,
				struct device_attribute *attr,
				const char *buf, size_t count)
{
	struct net_device * net_dev;
	struct net_local *lp;
	int i;
	uint16_t lna = simple_strtol(buf, NULL, 0);
	
	net_dev = (struct net_device *)dev_get_drvdata(dev);
	lp = net_dev->priv;
	
	for (i = 0; i < NUM_CHANNELS; i++)
		wlan_pda_rssi->rssi_data[i].rssi_LNA = lna;
	
	return count;
}

static DEVICE_ATTR(fw_version, S_IRUGO | S_IWUSR,
		   sm_drv_show_fw_version, NULL);
static DEVICE_ATTR(cal_mac_address, S_IRUGO | S_IWUSR,
		   sm_drv_show_cal_mac_address, sm_drv_store_cal_mac_address);
static DEVICE_ATTR(cal_rssi, S_IRUGO | S_IWUSR,
		   sm_drv_show_cal_rssi, sm_drv_store_cal_rssi);
static DEVICE_ATTR(cal_iq, S_IRUGO | S_IWUSR,
		   sm_drv_show_cal_iq, sm_drv_store_cal_iq);
static DEVICE_ATTR(cal_output_limits, S_IRUGO | S_IWUSR,
		   sm_drv_show_cal_output_limits,
		   sm_drv_store_cal_output_limits);
static DEVICE_ATTR(cal_pa_curve_data, S_IRUGO | S_IWUSR,
		   sm_drv_show_cal_pa_curve_data,
		   sm_drv_store_cal_pa_curve_data);
static DEVICE_ATTR(default_country, S_IRUGO | S_IWUSR,
		   sm_drv_show_default_country,
		   sm_drv_store_default_country);
static DEVICE_ATTR(RSSI_A, S_IRUGO | S_IWUSR, NULL, sm_drv_store_RSSI_A);
static DEVICE_ATTR(RSSI_B, S_IRUGO | S_IWUSR, NULL, sm_drv_store_RSSI_B);
static DEVICE_ATTR(LNA, S_IRUGO | S_IWUSR, NULL, sm_drv_store_LNA);

extern struct platform_device wlan_omap_device;
int sm_drv_sysfs_create_files(void)
{
	struct device *dev = &(wlan_omap_device.dev);
	int ret;
	
	DEBUG(DBG_SYSFS, "Creating WLAN sysfs\n");

	ret = device_create_file(dev, &dev_attr_fw_version);
	if (ret < 0) {
		cx3110x_error("failed to create dev_attr_fw_version.");
		goto error;
	}
		
	ret = device_create_file(dev, &dev_attr_cal_mac_address);
	if (ret < 0) {
		cx3110x_error("failed to create dev_attr_cal_mac_address.");
		goto error;
	}

	ret = device_create_file(dev, &dev_attr_cal_rssi);
	if (ret < 0) {
		cx3110x_error("failed to create dev_attr_cal_rssi.");
		goto error;
	}

	ret = device_create_file(dev, &dev_attr_cal_iq);
	if (ret < 0) {
		cx3110x_error("failed to create dev_attr_cal_iq.");
		goto error;
	}

	ret = device_create_file(dev, &dev_attr_cal_output_limits);
	if (ret < 0) {
		cx3110x_error("failed to create dev_attr_cal_output_limits.");
		goto error;
	}

	ret = device_create_file(dev , &dev_attr_cal_pa_curve_data);
	if (ret < 0) {
		cx3110x_error("failed to create dev_attr_cal_pa_curve_data.");
		goto error;
	}

	ret = device_create_file(dev, &dev_attr_default_country);
	if (ret < 0) {
		cx3110x_error("failed to create dev_attr_default_country.");
		goto error;
	}

	ret = device_create_file(dev, &dev_attr_RSSI_A);
	if (ret < 0) {
		cx3110x_error("failed to create dev_attr_RSSI_A.");
		goto error;
	}

	ret = device_create_file(dev, &dev_attr_RSSI_B);
	if (ret < 0) {
		cx3110x_error("failed to create dev_attr_RSSI_B.");
		goto error;
	}

	ret = device_create_file(dev, &dev_attr_LNA);
	if (ret < 0) {
		cx3110x_error("failed to create dev_attr_LNA.");
		goto error;
	}
		
	ret = sm_drv_sysfs_umac_create_files();
	if (ret < 0) {
		goto error;
	}

	return ret;
	
error:
	device_remove_file(dev, &dev_attr_fw_version);
	device_remove_file(dev, &dev_attr_cal_mac_address);
	device_remove_file(dev, &dev_attr_cal_rssi);
	device_remove_file(dev, &dev_attr_cal_iq);
	device_remove_file(dev, &dev_attr_cal_output_limits);
	device_remove_file(dev, &dev_attr_cal_pa_curve_data);
	device_remove_file(dev, &dev_attr_default_country);
	device_remove_file(dev, &dev_attr_RSSI_A);
	device_remove_file(dev, &dev_attr_RSSI_B);
	device_remove_file(dev, &dev_attr_LNA);

	return ret;
}

#endif // CONFIG_SYSFS
