/*
 * This file is part of cx3110x
 *
 * Copyright (C) 2003 Conexant Americas Inc. All Rights Reserved.
 * Copyright (C) 2004, 2005, 2006 Nokia Corporation
 *
 * Contact: Kalle Valo <Kalle.Valo@nokia.com>
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License 
 * version 2 as published by the Free Software Foundation. 
 *
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA
 *
 */

#define __KERNEL_SYSCALLS__

#include <linux/version.h>
#include <linux/module.h>

#include <linux/init.h>
#include <linux/netdevice.h>
#include <linux/if_arp.h>
#include <linux/random.h>
#include <linux/clk.h>
#include <linux/spi/spi.h>
#include <linux/platform_device.h>
#include <linux/string.h>
#include <linux/firmware.h>
#include <linux/config.h>
#if !defined(CONFIG_FW_LOADER) && !defined(CONFIG_FW_LOADER_MODULE)
#error No Firmware Loading configured in the kernel !
#endif

#include <asm/irq.h>
#include <asm/hardware.h>

#include "sm_drv.h"
#include "sm_drv_spi.h"
#include "sm_drv_sysfs.h"
#include "sm_drv_ioctl.h"
#include "sm_drv_pda.h"
#include "pda.h"

static void sm_drv_spi_initialize(struct net_device *dev);
static void sm_drv_spi_reset_device(struct net_device *dev, unsigned boot);
static irqreturn_t sm_drv_spi_interrupt(int irq, void *config);
static struct net_device *sm_drv_spi_probe(void);

const char driver_name[64] = DRIVER_NAME;
char wlan_fw_version[32] = "N/A";
char * firmware_file;
struct omap_wlan_cx3110x_config * wlan_config;

#define DEV_NAME    "wlan%d"
extern struct net_device *sm_net_device;

extern unsigned int driver_type;

static u8 wlan_wait_bit(struct net_device * dev, u16 reg, u32 mask,
			u32 expected )
{
	int i;
	char buffer[4];
	
	for (i = 0; i < 2000; i++) {		
		
		sm_spi_read(dev, reg, buffer, 4);
		if (((*(u32*)buffer) & mask) == expected){
			return 1;
		}
		msleep(1);
        }
	return 0;
}

struct s_txdata {
	/* ic message associated with packet */
	struct s_ic_msg *ic_msg;                   
	uint32_t ic_msg_flags;
};

extern struct completion softmac_init_comp;

static void sm_drv_send_zero_mac(struct net_device *dev)
{
	union iwreq_data uwrq;

	memset(&(uwrq.addr.sa_data), 0, ETH_ALEN);
	uwrq.addr.sa_family = ARPHRD_ETHER;

	wireless_send_event(dev, SIOCGIWAP, &uwrq, NULL);
}

static int sm_drv_spi_wakeup(struct net_device *dev) 
{
	struct net_local *lp = dev->priv;
	uint32_t host_ints, host_ints_ack, target_ints;
	unsigned long timeout;
	int err, tries = 0;

again:	
	DEBUG(DBG_BH, "w\n");

	spin_lock_bh(&lp->lock);
	if (lp->device_state != DEVSTATE_ACTIVE) {
		spin_unlock_bh(&lp->lock);
		err = -1;
		goto exit;
	}
	spin_unlock_bh(&lp->lock);
	
	/* Here we wake the target up */
	target_ints = SPI_TARGET_INT_WAKEUP; 
	sm_spi_write(dev, SPI_ADRS_ARM_INTERRUPTS,
		     (unsigned char *)&target_ints, sizeof target_ints );
		
	/* And wait for the READY interrupt */
	timeout = jiffies + HZ;
	
	sm_spi_read(dev, SPI_ADRS_HOST_INTERRUPTS,
		    (unsigned char *)&host_ints, sizeof(host_ints));
	while (!(host_ints & SPI_HOST_INT_READY)) {
		DEBUG(DBG_BH, ".\n");
		if (time_after(jiffies, timeout)) {
			if (tries < 3) {
				tries++;
				cx3110x_warning("Haven't got a READY interrupt"
						" from WAKEUP, trying again. "
						"(host_ints=0x%x, tries=%i)",
						host_ints, tries);
				goto again;
			} else {
				cx3110x_error("Chip did not respond to "
					      "WAKEUP, it's dead. "
					      " Firmware crashed? "
					      "(host_ints=0x%x, tries=%i)",
					      host_ints, tries);
				spin_lock_bh(&lp->lock);
				lp->device_state = DEVSTATE_DEAD;
				spin_unlock_bh(&lp->lock);
				err = -1;
				goto exit;
			}
		}
		
		msleep(1);
		
		sm_spi_read(dev, SPI_ADRS_HOST_INTERRUPTS,
			    (unsigned char *)&host_ints, sizeof(host_ints));
	}

	host_ints_ack = SPI_HOST_INT_READY;
	sm_spi_write(dev, SPI_ADRS_HOST_INT_ACK,
		     (unsigned char *)&host_ints_ack, sizeof(host_ints_ack));

	err = 0;

	if (tries > 0)
		cx3110x_info("chip woke up (host_ints=0x%x, tries=%i)",
			     host_ints, tries);
	
exit:
	DEBUG(DBG_BH, "W\n");
	return err;
}

static int sm_drv_spi_sleep(struct net_device *dev) 
{
	uint32_t target_ints;

	DEBUG(DBG_BH, "S\n");

	target_ints = SPI_TARGET_INT_SLEEP; 
	sm_spi_write(dev, SPI_ADRS_ARM_INTERRUPTS,
		     (unsigned char *)&target_ints, sizeof target_ints );

	return 0;
}

static void sm_drv_spi_reset(struct net_device *dev) 
{
	struct net_local *lp = dev->priv;
	union iwreq_data wrqu;
	uint32_t target_ints = SPI_TARGET_INT_RDDONE;
	uint32_t bgr_scan_disable = 0;

	cx3110x_warning("scan timer expired, resetting.");

	sm_spi_write(dev, SPI_ADRS_ARM_INTERRUPTS,
		     (unsigned char *)&target_ints, sizeof(target_ints));

	wrqu.data.length = 0;
	wrqu.data.flags = 0;

	spin_lock_bh(&lp->lock);

	if (lp->bss_type != DOT11_BSSTYPE_IBSS) {
		spin_unlock_bh(&lp->lock);
		sm_drv_oid_set(dev, DOT11_OID_AUTOSCANDISABLE,
			       (void*)&bgr_scan_disable,
			       sizeof(uint32_t));
		spin_lock_bh(&lp->lock);
	}

	wireless_send_event(dev, SIOCGIWSCAN, &wrqu, NULL);
	sm_drv_send_zero_mac(dev);
	lp->device_state = DEVSTATE_DEAD;

	spin_unlock_bh(&lp->lock);
}

static int sm_drv_spi_rx(struct net_device *dev) 
{
	struct net_local *lp = dev->priv;
	struct spi_local *spi_lp = lp->spi_lp;
	struct s_sm_frame *frame;
	struct s_ic_msg *ic_msg;
	unsigned short length;
	int err;
	int32_t callb_mask = 0;

	err = sm_drv_spi_wakeup(dev);
	if (err < 0) {
		goto exit;
	}

	spin_lock_bh(&lp->lock);

	frame = frame_skb_alloc(dev, lp->sm_descr.mtu + lp->sm_descr.rxoffset,
				0);

	spin_unlock_bh(&lp->lock);
	
	if (frame != NULL) {
		ic_msg = FRAME_ICMSG(frame);
		ic_msg->frame = frame;
            
		/* dummy read to flush SPI DMA controller bug */
		sm_spi_read(dev, SPI_ADRS_GEN_PURP_1,
			    (unsigned char *)&length, sizeof(length));

		sm_spi_read(dev, SPI_ADRS_DMA_DATA,
			    (unsigned char *)&length, sizeof(length));

		if (length == 0) {
			cx3110x_warning("chip requested DMA rx transfer"
					" of a zero length frame");
			frame_skb_free(dev, frame);
			err = -1;
			goto exit;
		}
		
		if ( length > SPI_MAX_PACKET_SIZE ) 
			length = SPI_MAX_PACKET_SIZE;

		sm_spi_dma_read(dev, SPI_ADRS_DMA_DATA,
				(unsigned char *)frame->data, length);

		DEBUG(DBG_BH, "received data 0x%p, %d bytes\n",
		      frame, length);
			
		ic_msg->channel = 0; 
		ic_msg->flags   = 0;
		ic_msg->length  = length;
		ic_msg->address = 0;
		ic_msg->data    = frame->data;

		spin_lock_bh(&lp->lock);
			
		spi_lp->spi_packets++;

		callb_mask = prism_interconnect_message_handle(lp->sm_context,
							       ic_msg);

		spin_unlock_bh(&lp->lock);

		DEBUG(DBG_IC, "After prism_interconnect_message_handle"
		      "(%d, %x, %d, %x, %p)\n",
		      ic_msg->channel, ic_msg->flags, ic_msg->length,
		      ic_msg->address, ic_msg->data);
			
		DEBUG(DBG_IC,"Callback mask: %d\n", callb_mask);
			
		if(callb_mask < 0)
			cx3110x_warning("prism_interconnect_message_handle"
					" returned error %d",
					callb_mask);
	} else
		cx3110x_warning("couldn't allocate RX frame.");

	handle_sm_callback(dev, callb_mask);

exit:
	return err;
}

static int sm_drv_spi_tx(struct net_device *dev) 
{
	struct net_local *lp = dev->priv;
	struct spi_local *spi_lp = lp->spi_lp;
	struct s_ic_msg *ic_msg;
	struct s_dma_regs dma_regs;
	int32_t callb_mask = 0;
	uint32_t host_ints, host_ints_ack, ic_mask;
	unsigned long timeout;
	int err;

	/* See if the UMAC has ic messages pending for us to send */
	while(1)
	{

		ic_mask = IC_MASK_CTRL | IC_MASK_FRAME;
		spin_lock_bh(&lp->lock);
		callb_mask = prism_interconnect_message_query(lp->sm_context,
							      ic_mask,
							      &ic_msg);
		spin_unlock_bh(&lp->lock);
		if(!ic_msg) {
			err = 0;
			goto exit;
		}

		DEBUG(DBG_IC,
		      "prism_interconnect_message_query(%d, %x, %d, %x, %p)\n",
		      ic_msg->channel, ic_msg->flags, ic_msg->length,
		      ic_msg->address, ic_msg->data);

		/* Firmware bug...First TX packets needs to be horribly
		 * delayed */
		if (spi_lp->initial_packets < SPI_DELAY_THRESHOLD)
			msleep(5);

		err = sm_drv_spi_wakeup(dev);
		if (err < 0) {
			err = -1;
			goto exit;
		}
		
		/* program DMA and transfer TX buffer */
		dma_regs.cmd  = SPI_DMA_WRITE_CTRL_ENABLE;        
		dma_regs.len  = cpu_to_le16(ic_msg->length);      
		dma_regs.addr = cpu_to_le32(ic_msg->address); 
	
		sm_spi_write(dev, SPI_ADRS_DMA_WRITE_CTRL,
			     (unsigned char *)&dma_regs,
			     sizeof(struct s_dma_regs));
		
		sm_spi_dma_write(dev, SPI_ADRS_DMA_DATA,
				 ic_msg->data, ic_msg->length);

		DEBUG(DBG_BH, "sent data 0x%p, %d bytes\n",
		      ic_msg->data, ic_msg->length);
		
		/* We wait for the DMA Ready interrupt */
		DEBUG(DBG_BH, "wr\n");
		timeout = jiffies + 2 * HZ;
		sm_spi_read(dev, SPI_ADRS_HOST_INTERRUPTS,
			    (unsigned char *) &host_ints, sizeof(host_ints));
		while(!(host_ints & SPI_HOST_INT_WR_READY)) {
			DEBUG(DBG_BH, "-\n");
			if (time_after(jiffies, timeout)) {
				cx3110x_warning("Haven't got a "
						"WR_READY for DMA write. "
						"(firmware crashed?)");
				err = -1;
				goto exit;
			}
			sm_spi_read(dev, SPI_ADRS_HOST_INTERRUPTS,
				    (unsigned char *) &host_ints,
				    sizeof(host_ints));
		}

		host_ints_ack = SPI_HOST_INT_WR_READY;
		sm_spi_write(dev, SPI_ADRS_HOST_INT_ACK,
			     (unsigned char *) &host_ints_ack,
			     sizeof(host_ints_ack));

		DEBUG(DBG_BH, "WR\n");

		if (ic_msg->flags & IC_FLAG_FREE) {
			spin_lock_bh(&lp->lock);

			if (ic_msg->frame)
				prism_driver_frame_free(lp->sm_context,
							ic_msg->frame);
			else
				prism_driver_free(lp->sm_context, ic_msg);

			spin_unlock_bh(&lp->lock);
		}
	}

	handle_sm_callback(dev, callb_mask);

exit:
	return err;
}

static void sm_drv_spi_wq(struct work_struct *work)
{
	struct net_local *lp = container_of(work, struct net_local, work);
	struct net_device *dev = lp->netdev;
	struct spi_local *spi_lp = lp->spi_lp;
	uint32_t host_ints, host_ints_en, host_ints_ack;
	int err;
	
	sm_spi_read(dev, SPI_ADRS_HOST_INTERRUPTS,
		    (unsigned char *) &host_ints, sizeof(host_ints));

	DEBUG(DBG_BH, "interrupts 0x%x\n", host_ints);

	if(host_ints & SPI_HOST_INT_READY) {
		host_ints_ack = SPI_HOST_INT_READY;
		sm_spi_write(dev, SPI_ADRS_HOST_INT_ACK,
			     (unsigned char *)&host_ints_ack,
			     sizeof(host_ints_ack));

		spin_lock_bh(&lp->lock);
		if (spi_lp->upload_state == UPLOAD_STATE_BOOTING) {
			spin_unlock_bh(&lp->lock);

			sm_drv_spi_initialize(dev);

			spin_lock_bh(&lp->lock);
			spi_lp->upload_state = UPLOAD_STATE_RUNNING;
			spin_unlock_bh(&lp->lock);

			host_ints_en = SPI_HOST_INT_UPDATE |
				SPI_HOST_INT_SW_UPDATE;
			sm_spi_write(dev, SPI_ADRS_HOST_INT_EN,
				     (unsigned char *)&host_ints_en,
				     sizeof(host_ints_en));

			DEBUG(DBG_BH, "initialized\n");

			/* Now we can wait ifconfig...*/
			complete(&softmac_init_comp);

			spin_lock_bh(&lp->lock);
		}
		
		spin_unlock_bh(&lp->lock);

	}

	spin_lock_bh(&lp->lock);

	if (unlikely(lp->device_state == DEVSTATE_RESETTING)) {
		spin_unlock_bh(&lp->lock);
		sm_drv_spi_reset(dev);
		goto exit;
	}

	if (lp->sm_descr.mtu == 0) {
		spin_unlock_bh(&lp->lock);
		DEBUG(DBG_BH, "%s SoftMAC not initialized\n", DRIVER_NAME);
		goto exit;
	}

	spin_unlock_bh(&lp->lock);
    
	/* Check if LMAC has rx frame */
	if((host_ints & SPI_HOST_INT_UPDATE) ||
	   (host_ints & SPI_HOST_INT_SW_UPDATE)) {
		host_ints_ack = SPI_HOST_INT_UPDATE | SPI_HOST_INT_SW_UPDATE;
		sm_spi_write(dev, SPI_ADRS_HOST_INT_ACK, 
			     (unsigned char *)&host_ints_ack,
			     sizeof(host_ints_ack));

		err = sm_drv_spi_rx(dev);
		if (err < 0) {
			goto exit;
		}
	}

        err = sm_drv_spi_tx(dev);
	if (err < 0) {
		goto exit;
	}

exit:
	sm_drv_spi_sleep(dev);	

	return;
}

static void wlan_omap_device_release(struct device * dev)
{
	
}

struct platform_device wlan_omap_device = { 
	.name		= "wlan-omap",
	.id		= -1,
	.dev            = {
		.release = wlan_omap_device_release,
	},
};

static int wlan_omap_suspend(struct device *dev, pm_message_t state)
{
	int ret = 0;
	uint32_t hibernate = 1;
	struct net_device * net_dev;
	struct net_local *lp;

	cx3110x_info("suspending.");

	net_dev = (struct net_device *)dev_get_drvdata(dev);
	lp = net_dev->priv;

	spin_lock_bh(&lp->lock);

	if (lp->sm_initialization) {
		spin_unlock_bh(&lp->lock);
		ret = sm_drv_oid_set((struct net_device *)dev_get_drvdata(dev),
				     GEN_OID_HIBERNATE,
				     &hibernate,
				     sizeof(uint32_t));
		spin_lock_bh(&lp->lock);
	}

	spin_unlock_bh(&lp->lock);

	return ret;
}

static int wlan_omap_resume(struct device *dev)
{
	int ret = 0;
	uint32_t hibernate = 0;
	struct net_device * net_dev;
	struct net_local *lp;

	cx3110x_info("resuming.");

	net_dev = (struct net_device *)dev_get_drvdata(dev);
	lp = net_dev->priv;

	spin_lock_bh(&lp->lock);

	if (lp->sm_initialization) {
		spin_unlock_bh(&lp->lock);
		ret = sm_drv_oid_set((struct net_device *)dev_get_drvdata(dev),
				     GEN_OID_HIBERNATE,
				     &hibernate,
				     sizeof(uint32_t));
		spin_lock_bh(&lp->lock);
	}

	spin_unlock_bh(&lp->lock);

	return ret;
}

static int __init wlan_drv_probe(struct device *dev)
{
	return 0;
}

static int wlan_drv_remove(struct device *dev)
{
	return 0;
}

static struct device_driver wlan_omap_driver = {
	.name   	= "wlan-omap",
	.bus		= &platform_bus_type,
	.probe          = wlan_drv_probe,
	.remove         = wlan_drv_remove,
	.suspend	= wlan_omap_suspend,
	.resume		= wlan_omap_resume
};

int sm_drv_register_platform(struct net_device * sm_drv)
{
	int ret;
	
	DEBUG(DBG_SYSFS, "Registering WLAN platform device\n");

	ret = platform_device_register(&wlan_omap_device);
	if (ret) {
		cx3110x_error("Couldn't register wlan_omap device.");
		return ret;
	}

	/* Our net_device will be our private data */
	dev_set_drvdata(&(wlan_omap_device.dev), (void *)sm_drv);
	
	return 0;
}


void sm_drv_unregister_platform(void)
{
	platform_device_unregister(&wlan_omap_device);
}


int sm_drv_spi_up(struct net_device *dev)
{
	DEBUG(DBG_CALLS, "%s\n", __FUNCTION__);
	
	cx3110x_info("loading %s firmware.", firmware_file);
		
	if (sm_drv_spi_upload_firmware(dev) < 0) {
		cx3110x_error("failed to load %s.", firmware_file);
		return -1;
	} 

	return 0;
}

int sm_drv_spi_down(struct net_device *dev)
{
	DEBUG(DBG_CALLS, "%s\n", __FUNCTION__);

	sm_drv_spi_reset_device(dev, 0);

	return 0;
}

void sm_drv_spi_cli(struct net_device *dev)
{
	unsigned int host_reg = 0;
	
	sm_spi_write(dev, SPI_ADRS_HOST_INT_EN,
		     (unsigned char *) &host_reg, sizeof host_reg );
	
	cx3110x_disable_irq(dev);
}

static void sm_drv_spi_initialize(struct net_device *dev)
{
	struct net_local *lp = dev->priv;
	struct s_sm_setup setup;
	unsigned long rand;
	int i, callb_mask, ret;

	DEBUG(DBG_CALLS, "%s\n", __FUNCTION__);

	spin_lock_bh(&lp->lock);
	
	ret = prism_softmac_describe(&lp->sm_descr, lp->sm_initdata);
	if (ret != SM_ENONE) {
		cx3110x_error("LMAC - UMAC incomptability.");
		goto out;
	}

	setup.mode = lp->sm_mode;
	setup.rx_align = 0;
	setup.max_sta = DEV_MAX_STA;
	setup.max_bss = DEV_MAX_BSS;
	setup.mtu = lp->sm_descr.mtu;
	get_random_bytes(&rand, sizeof(unsigned long));
	setup.seed = rand;

	for(i = 0; i < 8; i++)
		setup.depth[i] = 0;

        /* allocate a memory region for the UMAC context */ 
	lp->sm_context = (uint32_t *)kmalloc(lp->sm_descr.context_size,
					     GFP_ATOMIC);
	if(!lp->sm_context) {
		cx3110x_error("UMAC Context Memory could not be allocated.");
		goto out;
	}

	DEBUG(DBG_CALLS, "sm_drv_spi_initialize: Mode %d\n", setup.mode);
	
	callb_mask = prism_softmac_create(lp->sm_context, &setup, lp->sm_pda,
					  lp->sm_initdata);
	if(callb_mask < 0) {
		cx3110x_error("Could not create SoftMAC (%d).", callb_mask);
		lp->sm_descr.mtu = 0;
	}
	
	/* We can now free the PDA */
	kfree(lp->sm_pda);

	lp->sm_initialization = 1;

out:
	spin_unlock_bh(&lp->lock);

	/*
	 * workaround for the set ladder problem
	 * FIXME: fix this properly
	 */
	if (driver_type == SM_DRIVER_TYPE_MTUM) {
		cx3110x_info("msleep(100)");
		msleep(100);
	}
}

#define TARGET_BOOT_SLEEP 50

static void sm_drv_spi_reset_device(struct net_device * dev, unsigned boot)
{
	unsigned short dev_reg;

	/* We do the ROM boot */
	if (boot)
		dev_reg = SPI_CTRL_STAT_HOST_OVERRIDE 
			| SPI_CTRL_STAT_HOST_RESET 
			| SPI_CTRL_STAT_RAM_BOOT;
	else
		dev_reg = SPI_CTRL_STAT_HOST_OVERRIDE
			| SPI_CTRL_STAT_HOST_RESET
			| SPI_CTRL_STAT_START_HALTED;
	sm_spi_write(dev, SPI_ADRS_DEV_CTRL_STAT, 
		     (unsigned char *)&dev_reg, sizeof(dev_reg));

	msleep(TARGET_BOOT_SLEEP);

	if (boot)
		dev_reg = SPI_CTRL_STAT_HOST_OVERRIDE 
			| SPI_CTRL_STAT_RAM_BOOT;
	else
		dev_reg = SPI_CTRL_STAT_HOST_OVERRIDE
			| SPI_CTRL_STAT_START_HALTED;

	sm_spi_write(dev, SPI_ADRS_DEV_CTRL_STAT, 
		     (unsigned char *)&dev_reg, sizeof(dev_reg));
	msleep(TARGET_BOOT_SLEEP);
}

int sm_drv_fetch_firmware(struct net_device *ndev, char *fw_id,
			  struct device * dev)
{
	struct net_local *lp = ndev->priv;
	const struct firmware *fw_entry = 0;
	int ret = 0;
	
	DEBUG(DBG_CALLS, "%s\n", __FUNCTION__);
	
	ret = request_firmware(&fw_entry, fw_id, dev);
	if (ret) {
		cx3110x_error("sm_drv_fetch_firmware failed for '%s': %d",
			     fw_id, ret);
		return ret;
	}

	spin_lock_bh(&lp->lock);
	
	lp->fw_len = fw_entry->size;
	lp->fw_ptr = (uint8_t *)kmalloc(lp->fw_len, GFP_ATOMIC);
	if (!lp->fw_ptr) {
		cx3110x_error("Couldn't allocate memory for firmware.");
		ret = -1;
		goto out;
	}
	
	memcpy(lp->fw_ptr, fw_entry->data, lp->fw_len);

	DEBUG(DBG_CALLS, "sm_drv_fetch_firmware: file %s (%d bytes)\n",
	      fw_id, fw_entry->size);

	if (lp->fw_len % 4) {
		cx3110x_error("firmware '%s' size is not multiple of 32bit.",
			      fw_id);
		ret = -EILSEQ; /* Illegal byte sequence  */;
		goto err_out;
	}
	
	/* add offset 20 to point to the bra directly */ 
	ret = prism_softmac_parse_bra(lp->fw_ptr + 20, lp->sm_initdata);
	if(ret != SM_ENONE) {
		cx3110x_error("firmware does not contain LMAC description.");
		ret = -1;
		goto err_out;
	}

 out:
	spin_unlock_bh(&lp->lock);

	/* Firmware version is at offset 40 */
	memcpy(wlan_fw_version, fw_entry->data + 40, 16);
	wlan_fw_version[16] = '\0';
	
	cx3110x_info("firmware version %s.", wlan_fw_version);

	release_firmware(fw_entry);
	return ret;
	
 err_out:
	kfree(lp->fw_ptr);
	lp->fw_len = 0;

	spin_unlock_bh(&lp->lock);

	release_firmware(fw_entry);
	return ret;	
}


void sm_drv_release_firmware(struct net_device *ndev)
{
	struct net_local *lp = ndev->priv;

	spin_lock_bh(&lp->lock);

	if (lp->fw_ptr)
		kfree(lp->fw_ptr);
	lp->fw_len = 0;

	spin_unlock_bh(&lp->lock);
}

/* FIXME: fw_len and fw_buffer needs to be locked */
int sm_drv_spi_upload_firmware(struct net_device *dev)
{
	struct net_local *lp = dev->priv;
	struct s_dma_regs dma_regs;
	unsigned long fw_len, address = FIRMWARE_ADDRESS;
	uint32_t host_reg;
	uint8_t * fw_buffer;

	DEBUG(DBG_CALLS, "%s\n", __FUNCTION__);

	fw_len = lp->fw_len;
	fw_buffer = lp->fw_ptr;
	
	if (lp->fw_len == 0) {
		return -1;
	}
	
	/* Stop device */
	sm_drv_spi_reset_device(dev, 0);

	while (fw_len > 0) {

		long _fw_len =
			(fw_len > SPI_MAX_PACKET_SIZE) ?
			SPI_MAX_PACKET_SIZE : fw_len;
		dma_regs.cmd   = SPI_DMA_WRITE_CTRL_ENABLE;
		dma_regs.len   = cpu_to_le16(_fw_len);        
		dma_regs.addr  = cpu_to_le32(address); 
		
		fw_len -= _fw_len;
		address += _fw_len;
		
		sm_spi_write(dev, SPI_ADRS_DMA_WRITE_CTRL,
			     (unsigned char *)(&dma_regs.cmd), sizeof(short));
		if (wlan_wait_bit(dev, SPI_ADRS_DMA_WRITE_CTRL,
				  HOST_ALLOWED, HOST_ALLOWED) == 0) {
			cx3110x_warning("fw_upload not allowed to DMA write.");
			cx3110x_dump_register(dev);
			return -EAGAIN; 
		}
		
		sm_spi_write(dev, SPI_ADRS_DMA_WRITE_LEN,
			     (unsigned char *)(&dma_regs.len), sizeof(short));
		sm_spi_write(dev, SPI_ADRS_DMA_WRITE_BASE,
			     (unsigned char *)(&dma_regs.addr), sizeof(long));
		
		sm_spi_dma_write(dev, SPI_ADRS_DMA_DATA, fw_buffer, _fw_len);
	}

	BUG_ON(fw_len != 0);

	/* Enable host interrupts */
	host_reg = SPI_HOST_INTS_DEFAULT;
	sm_spi_write(dev, SPI_ADRS_HOST_INT_EN,
		     (unsigned char *)&host_reg, sizeof host_reg );

	/* Boot device */
	sm_drv_spi_reset_device(dev, 1);

	DEBUG(DBG_CALLS, "Upload firmware done\n");

	return 0;
}


extern pda_record_t sm_drv_pda[];
static struct wlan_pda_add_regulatory_limits_channel_s
wlan_pda_add_regulatory_limits[4] =
{
	{
		.frequency = 2412,
		.regulatory_domain = 0x10,
		.country_code = {0x0, 0x0, 0x0},
		.modulation = 0x1f,
		.power_limit = 68,
	},
	{
		.frequency = 2412,
		.regulatory_domain = 0x30,
		.country_code = {0x0, 0x0, 0x0},
		.modulation = 0x1f,
		.power_limit = 68,
	},
	{
		.frequency = 2462,
		.regulatory_domain = 0x10,
		.country_code = {0x0, 0x0, 0x0},
		.modulation = 0x1f,
		.power_limit = 68,
	},
	{
		.frequency = 2472,
		.regulatory_domain = 0x01,
		.country_code = {0x0, 0x0, 0x0},
		.modulation = 0x1f,
		.power_limit = 68,
	},
};

struct s_pda * sm_drv_spi_get_pda(struct device *dev)
{
	struct s_pda * pda;
	const unsigned char * default_pda;

	DEBUG(DBG_CALLS, "%s\n", __FUNCTION__);

	if (driver_type == SM_DRIVER_TYPE_UMAC)
		default_pda = umac_pda;
	else
		default_pda = mtum_pda;
	
	/* We first build the pda_record array */
	sm_drv_pda_init(sm_drv_pda,
			(uint8_t *)default_pda, sizeof(default_pda));

	/* Now we can replace any PDR we want */
	if (wlan_pda_mac_address.mac_addr[0] != 0xff)
		sm_drv_pda_replace_pdr(sm_drv_pda, PDR_MAC_ADDRESS, 
				       wlan_pda_mac_address.mac_addr, 
				       sizeof(wlan_pda_mac_address.mac_addr));
	if (wlan_pda_iq)
		sm_drv_pda_replace_pdr(sm_drv_pda,
				       PDR_PRISM_ZIF_TX_IQ_CALIBRATION, 
				       (uint8_t *)wlan_pda_iq,
				       NUM_CHANNELS *
				       sizeof(struct wlan_pda_iq_autocal_s));
	
	if (wlan_pda_pa_curve_data)
		sm_drv_pda_replace_pdr(sm_drv_pda, PDR_PRISM_PA_CAL_CURVE_DATA,
				       (uint8_t *)wlan_pda_pa_curve_data,
				       sizeof(struct wlan_pda_pa_curve_data_s));

	if (wlan_pda_output_limits)
		sm_drv_pda_replace_pdr(sm_drv_pda,
				       PDR_PRISM_PA_CAL_OUTPUT_POWER_LIMITS,
				       (uint8_t *)wlan_pda_output_limits,
				       sizeof(struct wlan_pda_output_limit_s));	

	if (wlan_pda_rssi) {
		sm_drv_pda_replace_pdr(sm_drv_pda,
				       PDR_RSSI_LINEAR_APPROXIMATION_EXTENDED,
				       (uint8_t *)wlan_pda_rssi,
				       sizeof(struct wlan_pda_rssi_cal_s));	
	}

	if (wlan_pda_default_country)
		sm_drv_pda_replace_pdr(sm_drv_pda, PDR_DEFAULT_COUNTRY,
				       (uint8_t *)wlan_pda_default_country,
				       sizeof(struct wlan_pda_default_country_s));	
	
	sm_drv_pda_replace_pdr(sm_drv_pda, PDR_REGULATORY_POWER_LIMITS,
			       (uint8_t *)wlan_pda_add_regulatory_limits,
			       sizeof(wlan_pda_add_regulatory_limits));

	/* Then we get the s_pda from the pda_record array */
	pda = sm_drv_pda_get_pda(sm_drv_pda);

	return pda;
}


static irqreturn_t sm_drv_spi_interrupt(int irq, void *config)
{
	struct net_device *dev = config;
	struct net_local *lp = dev->priv;

	DEBUG(DBG_IRQ, "IRQ\n");

	/* FIXME: move state check to workqueue */
	if(lp->device_state == DEVSTATE_ACTIVE
	   || lp->device_state == DEVSTATE_BOOTING)
		queue_work(lp->workqueue, &lp->work);

	return IRQ_HANDLED;
}

int sm_drv_spi_request_irq(struct net_device *dev)
{
	int rval;
	
	DEBUG(DBG_CALLS, "%s\n", __FUNCTION__);
	
	/* Install interrupt handler */
	rval = cx3110x_request_irq(sm_drv_spi_interrupt, DRIVER_NAME,
				   (void*) dev);

	if (rval < 0)
		return rval;
       	
	return 0;
}

void sm_drv_spi_free_irq( struct net_device *dev )
{
	DEBUG(DBG_CALLS, "%s\n", __FUNCTION__);
	cx3110x_free_irq((void *)dev);
}

struct sm_drv_firmware_file {
	u8  chip_type;
	/* Chip's manufacturer name */
	u8* name; 
	/* LMAC name */
	u8* lm;
	/* MTLM name */
	u8* mtlm; 
} firmware_array [] = {
	/* STLC4370, aka gen2.1 */
	{0x21, "STLC4370", "3825.arm", "mtlm3825.arm"}, 
	/* STLC4550, aka gen2.5 */
	{0x25, "STLC4550", "3826.arm", "mtlm3826.arm"}, 
	/* Default to STLC4370 */
	{0x0,  "Unknown" , "3825.arm", "mtlm3825.arm"}, 
};

static u8 * sm_drv_firmware_name(u8 chip_type, unsigned int driver_type)
{
	int i = 0;
	
	while (firmware_array[i].chip_type) {
		if (firmware_array[i].chip_type == chip_type) {
			cx3110x_info("chip variant %s.",
				     firmware_array[i].name);
			if (driver_type == SM_DRIVER_TYPE_UMAC)
				return firmware_array[i].lm;
			else
				return firmware_array[i].mtlm;
		}
		i++;
	}

	return NULL;
	
}

#define N770_WLAN_CS_GPIO 21
#define N770_WLAN_POWER_GPIO 59
#define N770_WLAN_IRQ_GPIO   36
static struct net_device *sm_drv_spi_probe(void)
{
	struct net_local *lp;
	struct net_device * dev = NULL;
	struct spi_local *spi_lp = NULL;
	const struct omap_wlan_cx3110x_config * cx3110x_wlan_config;
	int ret;
	
	DEBUG(DBG_CALLS, "%s\n", __FUNCTION__);

	/*
	 * Create the Network device instance that will interface with the
	 * kernel Network stack
	 */
	dev = sm_drv_netdev_create(0,0);
	
	if (!dev) {
		cx3110x_error("Cannot allocate network device.");
		return NULL;
	}
	
	/* Register platform device and create sysfs files */
	if(sm_drv_register_platform(dev)) {
		cx3110x_spi_stop(dev);
		return NULL;
	}
	
	if (driver_register(&wlan_omap_driver))
		goto err_out_1;

	ret = sm_drv_sysfs_create_files();
	if (ret < 0)
		goto err_out_2;
	
	lp = dev->priv;
	spi_lp = lp->spi_lp;
	

	spi_lp = kzalloc(sizeof(struct spi_local), GFP_KERNEL);
	if (!spi_lp) {
		cx3110x_error("Cannot allocate SPI local data.");
		goto err_out_2;
	}
	
	spi_lp->netdev = dev;
	
	wlan_config = kzalloc(sizeof(struct omap_wlan_cx3110x_config), 
			      GFP_KERNEL);
	
	if (wlan_config == NULL)
		return NULL;
	
	cx3110x_wlan_config = omap_get_config(OMAP_TAG_WLAN_CX3110X, 
					      struct omap_wlan_cx3110x_config);
	
	/* With some old N770 bootloaders, the chip type s not set */
	if (cx3110x_wlan_config == NULL ||
	    ((cx3110x_wlan_config->chip_type != 0x21) &&
	     (cx3110x_wlan_config->chip_type != 0x25))) {
		wlan_config->chip_type = 0x21;
		wlan_config->power_gpio = N770_WLAN_POWER_GPIO;
		wlan_config->irq_gpio = N770_WLAN_IRQ_GPIO;
		wlan_config->spi_cs_gpio = N770_WLAN_CS_GPIO;
	} else {
		memcpy(wlan_config, cx3110x_wlan_config,
		       sizeof(struct omap_wlan_cx3110x_config));
	}


	lp->spi_lp = spi_lp;

	spin_lock_init(&spi_lp->lock);
	
	/* Initialise initial packet count */
	spi_lp->spi_packets = 0;

	/* Init our Bottom halve handler */
	lp->workqueue = create_singlethread_workqueue("cx3110x");
	INIT_WORK(&lp->work, sm_drv_spi_wq);

	/* Init the wireless stats queue */
	lp->stats_timestamp = 0;

	/* Init the WPA IE list */
	INIT_LIST_HEAD(&lp->bss_wpa_list);
	sema_init(&lp->wpa_sem, 1);

	/* Initial mode is CLIENT */
	lp->sm_mode = SM_MODE_CLIENT;
	
	/* 802.11 initial state */
	lp->link_state = DOT11_STATE_NONE;

	/* No SSID */
	memset(&lp->ssid, 0, sizeof(struct obj_ssid));

	/* Init the tuning values */
	wlan_pda_mac_address.mac_addr[0] = 0xff;
	wlan_pda_iq = NULL;
	wlan_pda_output_limits = NULL;
	wlan_pda_pa_curve_data = NULL;
	wlan_pda_rssi = NULL;

	/* Default country is EU if this is NULL */
	wlan_pda_default_country = NULL;

	lp->sm_initdata = kmalloc(sizeof(struct s_sm_initdata), GFP_KERNEL);
	if(!lp->sm_initdata) {
		cx3110x_error("Cannot allocate SPI initdata.");
		goto err_out_4;
	}

	SET_NETDEV_DEV(dev, &wlan_omap_device.dev);

	register_netdev(dev);
	
	/* Let's fetch the firmware from userspace */
	firmware_file = sm_drv_firmware_name(wlan_config->chip_type,
					     driver_type);
	if (firmware_file == NULL)
		goto err_out_4;
	
	if (sm_drv_fetch_firmware(dev, firmware_file,
				  &(wlan_omap_device.dev)) < 0)
		goto err_out_4;
		
	return dev;
	
 err_out_4:
	kfree(spi_lp);
	kfree(wlan_config);
 err_out_2:
	driver_unregister(&wlan_omap_driver);
 err_out_1:
	platform_device_unregister(&wlan_omap_device);
	return NULL;
}

static void sm_drv_spi_unprobe(struct net_device *dev)
{
	DEBUG(DBG_CALLS, "%s\n", __FUNCTION__);
	driver_unregister(&wlan_omap_driver);
	sm_drv_unregister_platform();
	unregister_netdev(dev);
	sm_drv_wpa_ie_clean(dev);
	sm_drv_release_firmware(dev);
	kfree(wlan_config);
}

static struct spi_driver cx3110x_driver = {
	.driver = {
		.name		= "cx3110x",
		.bus		= &spi_bus_type,
		.owner		= THIS_MODULE,
	},
	
	.probe		= cx3110x_probe,
	.remove		= __devexit_p(cx3110x_remove),
};

static int __init sm_drv_spi_init_module(void)
{
	int status;

	sm_net_device = sm_drv_spi_probe();

	if (!sm_net_device) {
		cx3110x_error("Could not register SoftMAC SPI Driver.");
		return -ENODEV;
	}

	status = spi_register_driver(&cx3110x_driver);
	if (status < 0)
		return status;
	
	cx3110x_info("driver version %s loaded.", DRIVER_VERSION);

	return 0;
}

static void __exit sm_drv_spi_cleanup_module( void )
{
	spi_unregister_driver(&cx3110x_driver);
	sm_drv_spi_unprobe(sm_net_device);

	cx3110x_info("driver version %s unloaded.", DRIVER_VERSION);
}
MODULE_DESCRIPTION("WLAN driver for Nokia N800");
MODULE_AUTHOR("Kalle Valo <Kalle.Valo@nokia.com>");
MODULE_LICENSE("GPL v2");

module_init(sm_drv_spi_init_module);
module_exit(sm_drv_spi_cleanup_module);
