/*
 * This file is part of cx3110x
 *
 * Copyright (C) 2005, 2006 Nokia Corporation
 *
 * Contact: Kalle Valo <Kalle.Valo@nokia.com>
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License 
 * version 2 as published by the Free Software Foundation. 
 *
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA
 *
 */

#include "sm_drv.h"
#include "sm_drv_ioctl.h"

/*
 * Wireless events
 */


int sm_drv_oid_set(struct net_device *dev, unsigned int oid_num,
		   void * oid_object, unsigned int oid_size)
{
	struct net_local *lp = dev->priv;
	int callb_mask;
	struct s_sm_conf smconf;

	if(!lp->sm_initialization) {
		cx3110x_warning("SoftMAC not initialized, chip not booted "
				"(set oid 0x%x).", oid_num);
		return -ENODEV;
	}

	smconf.flags = SM_CONF_OPSET;
	smconf.length = oid_size;
	smconf.oid = oid_num;
	smconf.data = oid_object;

	DEBUG(DBG_IOCTL, "sm_drv_oid_set: Set OID 0x%x: data %p, len %d\n",
	      smconf.oid, smconf.data, smconf.length);
	
	spin_lock_bh(&lp->lock);
	callb_mask = prism_softmac_conf(lp->sm_context, &smconf);
	spin_unlock_bh(&lp->lock);

	if(callb_mask < 0) {
		DEBUG(DBG_IOCTL, "sm_drv_ioctl (SET): sm_conf returned "
		      "error %d\n", callb_mask);	
		return -EINVAL;
	}
	
	handle_sm_callback(dev, callb_mask);

	return 0;
}


int sm_drv_commit(struct net_device *dev, struct iw_request_info *info,
			 char *cwrq, char *extra)
{
	uint32_t commit;
	
	DEBUG(DBG_IOCTL, "COMMIT\n");
	
	/*
	 * Some objects need to be commited.
	 * This is called when one of the ioctls
	 * return -EINPROGRESS.
	 * We just have to write the COMMIT object.
	 */

	commit = 0;

	return sm_drv_oid_set(dev, GEN_OID_COMMIT, (void *)&commit,
			      sizeof(uint32_t));
}


int sm_drv_oid_get(struct net_device *dev, unsigned int oid_num,
		   void * oid_object, unsigned int oid_size)
{
	struct net_local *lp = dev->priv;
	int32_t callb_mask;
	struct s_sm_conf smconf;

	if(!lp->sm_initialization) {
		cx3110x_warning("SoftMAC not initialized, chip not booted "
				"(get oid 0x%x)\n", oid_num);
		return -ENODEV;
	}

	smconf.flags = SM_CONF_OPGET;
	smconf.length = oid_size;
	smconf.oid = oid_num;
	smconf.data = oid_object;

	DEBUG(DBG_IOCTL, "sm_drv_oid_get: Get OID 0x%x, len %d\n",
	      smconf.oid, smconf.length);
	
	spin_lock_bh(&lp->lock);
	callb_mask = prism_softmac_conf(lp->sm_context, &smconf);
	spin_unlock_bh(&lp->lock);


	/* We can reset the data since we didn't call handle_trap yet */
	lp->getresp.data = NULL;
	if(callb_mask == SM_EPENDING) {
		int ret = -EINVAL;
		
		DEBUG(DBG_IOCTL, "sm_drv_oid_get: EPENDING\n");
		/* Call the softmac so it can handle our request */
		handle_sm_callback(dev, SM_IC);
				
		/* Wait for the response.
		 * FIXME: There is a possible race condition here that we get a
		 * response from the LMAC before we go to sleep
		 */
		interruptible_sleep_on_timeout(&lp->getresp_waitq, HZ/4);

		/* We got woken with a response to our get request */
		if(!lp->getresp.data) {
			cx3110x_error("%s: get response data is NULL.",
				      __FUNCTION__);

			return ret;
		}
		if(lp->getresp.oid != smconf.oid) {
			cx3110x_error("%s: get response invalid OID "
				      "(0x%x, 0x%x).",
				      __FUNCTION__, smconf.oid,
				      lp->getresp.oid);
			goto geterr_out;
		}
		if(lp->getresp.length > smconf.length) {
			cx3110x_error("%s: get overflow (oid 0x%x, %d > %d).",
				      __FUNCTION__, smconf.oid,
				      lp->getresp.length, smconf.length);
			goto geterr_out;
		}
		
		memcpy(smconf.data, lp->getresp.data, lp->getresp.length);
		ret = 0;
		
	geterr_out:
		kfree(lp->getresp.data);
		lp->getresp.data = NULL;
		
		return ret;
	} else if(callb_mask < 0) {
		DEBUG(DBG_IOCTL, 
		      "sm_drv_ioctl (Get OID 0x%x): sm_conf error %d\n", 
		      smconf.oid, callb_mask);
		
		return -EINVAL;
	}
	
	handle_sm_callback(dev, callb_mask);

	return 0;

}


/* to convert between channel and freq */
const int frequency_list_bg[] = { 2412, 2417, 2422, 2427, 2432, 2437, 2442,
	2447, 2452, 2457, 2462, 2467, 2472
};

int channel_of_freq(int f)
{
	int c = 0;

	if ((f >= 2412) && (f <= 2484)) {
		while ((c < 14) && (f != frequency_list_bg[c]))
			c++;
		if (c >= 14)
			return 0;
	} else
		return 0;

	return ++c;
}

int freq_of_channel(int c)
{
	if (c < 1 || c > 13)
		return 0;
	return frequency_list_bg[c - 1] * 1000;
}

/* UMAC has a dynamic debug level */
#define SM_DRV_SET_DEBUG                       SIOCIWFIRSTPRIV

int sm_drv_set_debug(struct net_device *dev, struct iw_request_info *info,
			    __u32 * uwrq, char *extra)
{
	struct obj_debug debug;
	DEBUG(DBG_IOCTL, "SET DEBUG\n");

	if (uwrq == NULL)
		return -EINVAL;

	debug.package = *uwrq;
	debug.level = 0xff;

	return sm_drv_oid_set(dev, GEN_OID_DEBUG, &debug,
			      sizeof(struct obj_debug));
}
