/*
 * This file is part of cx3110x
 *
 * Copyright (C) 2003 Conexant Americas Inc. All Rights Reserved.
 * Copyright (C) 2004, 2005, 2006 Nokia Corporation
 *
 * Contact: Kalle Valo <Kalle.Valo@nokia.com>
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License 
 * version 2 as published by the Free Software Foundation. 
 *
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA
 *
 */

#ifndef _SM_DRV_H
#define _SM_DRV_H

/*
 * INCLUDES
 */
#include <linux/skbuff.h>
#include <linux/netdevice.h>
#include <linux/wireless.h>
#include <linux/delay.h>
#include <asm/io.h>

#include "sm_drv_spi.h"
#include "smoid2.h"

#ifdef CONFIG_CX3110X_PROD
#define DRIVER_NAME "cx3110x_mt"
#define LIBUMAC_NAME "libmtum"
#define LMAC_NAME "mtml"
#else
#define DRIVER_NAME "cx3110x"
#define LIBUMAC_NAME "libumac"
#define LMAC_NAME "lmac"
#endif

#define DRIVER_VERSION "2.0.15"
#define DRIVER_PREFIX DRIVER_NAME ": "

/*
 * FUNCTION PROTOTYPES
 */
struct s_sm_frame *skb_to_frame(struct net_device *dev, struct sk_buff *skb);
struct sk_buff *frame_to_skb(struct net_device *dev,
			     struct s_sm_frame *framep);

int sm_drv_open(struct net_device *dev);
int sm_drv_close(struct net_device *dev);
int sm_drv_transmit(struct sk_buff *skb, struct net_device *dev);
void sm_drv_tx_timeout(struct net_device *dev);
struct net_device_stats *sm_drv_statistics(struct net_device *dev);
int sm_drv_ioctl(struct net_device *dev, struct ifreq *ifr, int cmd);
int sm_drv_reset(struct net_device *dev, uint8_t sm_mode);
int sm_drv_up(struct net_device *dev, uint8_t mode);
int sm_drv_down(struct net_device *dev);
struct net_device *sm_drv_netdev_create(int ioaddr, int irq);
struct net_device *sm_drv_get_device( uint32_t *context );
struct s_sm_frame *frame_skb_alloc(struct net_device *dev, uint32_t length,
				   uint16_t data_offset);
void frame_skb_free(struct net_device *dev, struct s_sm_frame *framep);
void sm_drv_psm_fsm(struct net_device *dev, int event);
void handle_sm_callback(struct net_device *dev, int32_t callb_mask);
void driver_timer_expired(unsigned long sm_handle);

extern struct net_device *root_sm_drv_device;
extern struct net_device *sm_drv_get_device(uint32_t *context);

/*
 * MACROS
 */
#define SM_ALIGN(value)    ((value + (sizeof(long) - 1)) & ~(sizeof(long) - 1))
#define FRAME_ICMSG(framep) ((struct s_ic_msg *)((char *)(framep) \
		+ SM_ALIGN(sizeof(struct s_sm_frame))))

/*
 * DEFINES
 */

#define MAX_CAM_TIMEOUT 100000

/* Configuration Items */
#define DEV_MAX_STA                             128
#define DEV_MAX_BSS                             50

/* Device State (lower halfword) and device actions (upper half word) */
#define DEVSTATE_IDLE                   0
#define DEVSTATE_BOOTING                1
#define DEVSTATE_ACTIVE                 2
#define DEVSTATE_DEAD                   3 /* Crash detecting */
#define DEVSTATE_RESETTING              4 /* Device needs resetting */

/* TX timeout */
#define SM_DRV_TX_TIMEOUT               (5120*HZ/100)

#define CX3110X_DEFAULT_DMA_THRESHOLD 100

/* timeout in ms to wait for firmware boot */
#define CX3110X_BOOT_TIMEOUT 2000 

/* Debug flag to get debug information for specific sections */
#define DBG_NONE    0x00000000
#define DBG_CALLS   0x00000001
#define DBG_TRACE   0x00000002
#define DBG_IRQ     0x00000004
#define DBG_BH      0x00000008
#define DBG_SMAC    0x00000010
#define DBG_FRAME   0x00000020
#define DBG_PWRSAVE 0x00000040
#define DBG_IOCTL   0x00000080
#define DBG_UARTPCI 0x00000100
#define DBG_NETLINK 0x00000200
#define DBG_IC      0x00000400
#define DBG_LMAC    0x00000800
#define DBG_CTRL    0x00001000
#define DBG_WDS     0x00002000
#define DBG_OFFLOAD 0x00004000
#define DBG_TIMER   0x00008000
#define DBG_SPI_IO  0x00010000
#define DBG_SYSFS   0x00020000
#define DBG_WPA     0x00040000
#define DBG_PDA     0x00080000

/*
 * Use DBG_OFF to initially disable debugging but make it possible to
 * enable it runtime. use DBG_NONE to compile out all debugging for release
 * builds.
 */
#define DBG_OFF     0xF0000000
#define DBG_ALL     0xFFFFFFFF


/*
 * DEBUG STUFF. The Makefile should specify if debug should be on or off.
 * This is done using -DDEBUG_ON in this case the DEBUG_LEVEL will be set
 * to DBG_OFF which means that by default no debug will be shown, but this
 * can be configured on the fly using sm_conf or sm_dbg tool. When a final
 * built is made and no debug should be used (every insturction saved
 * helps) so when DEBUG_ON in undefined DEBUG_LEVEL will be undefined
 * either.
 */
#ifdef CONFIG_CX3110X_DEBUG
#define DEBUG_LEVEL (DBG_CALLS | DBG_TRACE | DBG_IRQ | DBG_BH | DBG_SMAC \
	| DBG_IOCTL | DBG_SYSFS | DBG_IC | DBG_LMAC | DBG_WPA | DBG_PDA)
#else
//#define DEBUG_LEVEL DBG_ALL
#define DEBUG_LEVEL (DBG_LMAC | DBG_SMAC) 
#undef DEBUG_TIMESTAMP
#endif

#define K_DEBUG( f, m, args... )    do{ if(( f & m )) printk(args); }while(0)

#if DEBUG_LEVEL > DBG_NONE
#define DEBUG( f, args... )     K_DEBUG( f, DEBUG_LEVEL, args )
#else
#define DEBUG( f, args... )
#endif

#define cx3110x_error(fmt, arg...) \
	printk(KERN_ERR DRIVER_PREFIX "ERROR " fmt "\n", ##arg)

#define cx3110x_warning(fmt, arg...) \
	printk(KERN_WARNING DRIVER_PREFIX "WARNING " fmt "\n", ##arg)

#define cx3110x_info(fmt, arg...) \
	printk(KERN_INFO DRIVER_PREFIX fmt "\n", ##arg)

#define cx3110x_debug(level, fmt, arg...) \
	do { \
		if (level & DEBUG_LEVEL) \
			printk(KERN_INFO DRIVER_PREFIX fmt "\n", ##arg); \
	} while (0)

#define MAC2STR(a) (a)[0], (a)[1], (a)[2], (a)[3], (a)[4], (a)[5]
#define MACSTR "%02x:%02x:%02x:%02x:%02x:%02x"

/* We have 2 different drivers; MTUM and UMAC */
#define SM_DRIVER_TYPE_MTUM 0
#define SM_DRIVER_TYPE_UMAC 1

#define MAX_BSS_WPA_IE_COUNT 64
#define MAX_WPA_IE_LEN 255
#define MAX_PROBE_LENGTH 512

struct sm_drv_bss_wpa_ie {
	struct list_head list;
	unsigned long last_update;
	u8 bssid[ETH_ALEN];
	u8 wpa_ie[MAX_WPA_IE_LEN];
	size_t wpa_ie_len;
};


struct net_local {
	/* unique ID for shared IRQ */
	void*                     device_id;
	struct spi_local*	  spi_lp;
	struct net_device*        netdev;
	struct spi_device *       spi_device;

	/* lock for net_local structure and softmac */
	spinlock_t                lock;

	unsigned int              sm_initialization;

	/* Powersave/Active */
	uint32_t                  device_state;
	/* Assciated or not */
	uint32_t                  link_state;

	/* Our bottom halve (BH) handler */
	struct workqueue_struct *workqueue;
	struct work_struct work;
	
        /* Last stats update */
	unsigned long             stats_timestamp;
	
	/* WPA crap */

	/* WPA mode enabled */
	int wpa;

	struct list_head bss_wpa_list;
	int num_bss_wpa;
	struct semaphore wpa_sem;

	/* Our current SSID */
	struct obj_ssid ssid;

	/* MAC address we're associated with */
	uint8_t ap_mac_address[ETH_ALEN];
	
	/* AP frequency extension */
	uint16_t ext;

	/* 
	 * Firmware cache.
	 * We cache the firmware here so that we
	 * don't need to call hotplug and parse_bra
	 * everytime we boot.
	 */
	uint8_t * fw_ptr;
	long fw_len;
	
	/*
	 * Above the most frequently used variables are place we try to
	 * keep this in one cache line.
	 */

	/* SoftMAC mode: Client or AP */
	uint8_t                   sm_mode;
	/* BSS type: INFRA or IBSS */
	uint8_t                   bss_type;

        /* We can queue 1 TX frame if the SoftMAC queue is full */
	struct s_sm_frame*        queued_tx_frame;

	/* Timer for SoftMAC */
	struct timer_list         softmac_timer;

	/* In case we don't get the scan trap... */
	struct timer_list         scan_timer;

	/* temporary init structure storage */
	struct s_sm_initdata*     sm_initdata;

	/* SoftMAC description */
	struct s_sm_descr         sm_descr;

	/* handle to SoftMAC context */
	uint32_t*                 sm_context;

	/* Pointer to PDA */
	struct s_pda*             sm_pda;

	/* statistics */
	struct net_device_stats   stats;
	
	/* Wireless stats */
	struct iw_statistics      iwstatistics;


	/* To wait for configuration to be allowed */
	wait_queue_head_t         conf_waitq;

	/* wait queue and buffer for response ...*/
	wait_queue_head_t         getresp_waitq;

	/* ...to pending get request */
	struct s_sm_conf          getresp;

	int dma_threshold;	
};

#endif /* _SM_DRV_H */
