/* Certificate Management library
 * 
 * Copyright (C) 2005 Nokia. All rights reserved.
 * Author: Ed Bartosh <Eduard.Bartosh@nokia.com>
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License version 2.1 as published by the Free Software Foundation.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the
 * Free Software Foundation, Inc., 59 Temple Place - Suite 330,
 * Boston, MA 02111-1307, USA.
 */

#include "test.h"
#include <sys/types.h>
#include <unistd.h>
#include <signal.h>
#include <sys/wait.h>
#include <sys/stat.h>


#define KILL_FILENAME "kill.cst"
#define IMPORT_FILENAME "import.cst"
#define DEFAULT_TIMEOUT 5

static void test_multi_process_kill(PARAM * param);
static void test_multi_process_kill_child(const char *filename);

static void test_multi_process_import(PARAM * param);
static void test_multi_process_import_child(const char *filename);

static void test_multi_process_search(PARAM * param);
static void test_multi_process_search_child(const char *filename);

void test_multi_process(PARAM * param)
{
    test_multi_process_kill(param);
    test_multi_process_import(param);
    test_multi_process_search(param);
}

void test_multi_process_kill(PARAM * param)
{
    TEST_TITLE("Test process kill");

    int pid = fork();

    if (pid < 0)
    {
        perror("Can't fork()");
        TEST_DO_SIMPLE(pid >= 0);
        return;
    }
    
    if (pid > 0)
    {
        //printf("Main process. Child pid: %i\n", pid);
        const int timeout = DEFAULT_TIMEOUT;
        printf("Wait %i sec...\n", timeout);
        sleep(timeout);
        
        TEST_DO_SIMPLE(0 == kill(pid, SIGKILL));
        CST * st = CST_open_file(KILL_FILENAME, FALSE, NULL);
        TEST_DO_SIMPLE(NULL != st);
        if (st)
        {
            CST_free(st);
        }
    }
    else
    {
        test_multi_process_kill_child(KILL_FILENAME);
    }
}

static void test_multi_process_kill_child(const char *filename)
{
    // printf("Child process.\n");
    if (0 == unlink(filename))
    {
        CST_open_file(filename, FALSE, NULL);
        while (1) {}
    }
    else
    {
        perror("Can't unlink");
    }
}

static void test_multi_process_import(PARAM * param)
{
    TEST_TITLE("Multi process import and search");
    
    unlink(IMPORT_FILENAME);
    
    CST * st = CST_open_file(IMPORT_FILENAME, FALSE, NULL);

    if (NULL != st)
    {
        int pid = fork();

        if (pid < 0)
        {
            perror("Can't fork()");
            return;
        }

        if (pid > 0)
        {
            int timeout = DEFAULT_TIMEOUT;
            printf("Wait %i sec...\n", timeout);
            sleep(timeout);
            GSList * list = CST_search_by_folder(st, CST_FOLDER_PERSONAL);
            TEST_DO_SIMPLE(NULL != list);
            CST_free(st);
            wait(NULL);
        }
        else
        {
            CST_free(st);
            test_multi_process_import_child(IMPORT_FILENAME);
            exit(0);
        }
    }

}

static void test_multi_process_import_child(const char *filename)
{
    CST * st = CST_open_file(filename, FALSE, NULL);
    
    cst_t_seqnum certID = 0;
    cst_t_seqnum keyID = 0;

    if (st)
    {
        FILE * fp = fopen("files/pair/c1.pem", "r");
        if (fp)
        {
            certID = CST_import_cert_adv(st, fp, CST_FOLDER_PERSONAL, NULL);
            fclose(fp);
        }
        else
        {
            perror("Can't open file with cert");
        }
       
        if (certID > 0)
        {
            X509 * x = CST_get_cert(st, certID);
            if (x)
            {
                fp = fopen("files/pair/k1.pem", "r");
                if (fp)
                {
                    keyID = CST_import_priv_key_adv(st, X509_get_subject_name(x), fp, "123", DEF_PASS, NULL);
                    printf("keyID: %u\n", keyID);
                    fclose(fp);
                }
                else
                {
                    perror("Can't open file with private key");
                }
                X509_free(x);
            }
            else
            {
                printf("Can't load cert\n");
            }
        }    

        if ((certID <= 0) || (keyID <= 0))
        {
            printf("Can't import key: %u cert: %u\n", keyID, certID);
        }
        else
        {
            CST_assign(st, certID, keyID, DEF_PASS);
        }
    }
}

static void test_multi_process_search(PARAM * param)
{
    TEST_TITLE("Test search email");

    int pid = fork();

    if (pid < 0)
    {
        perror("Can't fork()");
        TEST_DO_SIMPLE(pid >= 0);
        return;
    }
    
    if (pid > 0)
    {
        //printf("Main process. Child pid: %i\n", pid);
        const int timeout = DEFAULT_TIMEOUT;
        printf("Wait %i sec...\n", timeout);
        sleep(timeout);
        
        CST * st = CST_open_file(IMPORT_FILENAME, FALSE, NULL);
        TEST_DO_SIMPLE(NULL != st);
        if (st)
        {
            GSList * list = CST_search_by_email(st, "sotona999@gmail.com");
            TEST_DO_SIMPLE(NULL != list);
            g_slist_free(list);
            CST_free(st);
        }
    }
    else
    {
        test_multi_process_search_child(IMPORT_FILENAME);
        exit(0);
    }
}

static void test_multi_process_search_child(const char *filename)
{
    CST * st = CST_open_file(filename, FALSE, NULL);
    
    cst_t_seqnum certID = 0;

    if (st)
    {
        FILE * fp = fopen("files/email/mailcert.dat", "r");
        if (fp)
        {
            certID = CST_import_cert_adv(st, fp, CST_FOLDER_PERSONAL, NULL);
            printf("Cert: %u\n", certID);
            fclose(fp);
        }
        else
        {
            perror("Can't open file with cert");
        }
    }
}
