/* Certificate Management library
 * 
 * Copyright (C) 2005 Nokia. All rights reserved.
 * Author: Ed Bartosh <Eduard.Bartosh@nokia.com>
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License version 2.1 as published by the Free Software Foundation.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the
 * Free Software Foundation, Inc., 59 Temple Place - Suite 330,
 * Boston, MA 02111-1307, USA.
 */

/**
    Test correct import multiple files to storage
*/

#include <sys/types.h>
#include <sys/stat.h>
#include <dirent.h>
#include <openssl/x509.h>

#include "cst.h"
#include "test.h"

#define IMP_FMT_DER 1
#define IMP_FMT_PEM 2

void test_import_certs(PARAM * param);
void test_import_keys(PARAM * param);
void test_import_crls(PARAM * param);

void test_import_and_assign(PARAM * param, const char *filename, const int fmt);
void test_import1(PARAM * param, const char *filename, const int fmt);
void test_import2(PARAM * param, const char *filename, const int fmt);

int test_import_dir(CST * st, const char *dir, const cst_t_cert_folder f);
int test_import_file(CST * st, const char *filename, const cst_t_cert_folder f);

void test_import_and_assign_complex(PARAM * param, const char *filename, const int fmt);

void test_import_PKCS12(PARAM * param);
void test_import_corrupted(PARAM * param);

void test_import(PARAM * param)
{
    TEST_TITLE("Test all imports");
    test_import_certs(param);
    test_import_keys(param);
    test_import_crls(param);
    
    TEST_TITLE("PEM format");
    test_import_and_assign_complex(param, "files/cert-clt.pem", IMP_FMT_PEM);
    
    //TEST_TITLE("DER format");
    //test_import_and_assign_complex(param, "files/cert-clt.der", IMP_FMT_DER);
}

static guint get_cert_id_by_x(CST * st, X509 * x)
{
    g_assert(st && x);
    return CST_search_by_UID(st, X509_get_issuer_name(x), X509_get_serialNumber(x));
}

void test_import_certs(PARAM * param)
{
    TEST_TITLE("Import multiple CA not root certificates");
    TEST_DO_SIMPLE(test_import_dir(param->st, "files/ca", CST_FOLDER_CA));

    TEST_TITLE("Import multiple CA root certificates");
    TEST_DO_SIMPLE(test_import_dir(param->st, "files/ca_root", CST_FOLDER_CA));
    
    TEST_TITLE("Import cert1.pem and cert2.pem");
    TEST_DO_SIMPLE(test_import_file(param->st, "files/cert1.pem", CST_FOLDER_SITE));
    TEST_DO_SIMPLE(test_import_file(param->st, "files/cert2.pem", CST_FOLDER_CA));

    TEST_TITLE("Import pair");
    TEST_DO_SIMPLE(test_import_file(param->st, "files/pair/c1.pem", CST_FOLDER_SITE));

    TEST_TITLE("Import email cert");
    TEST_DO_SIMPLE(test_import_file(param->st, "files/email/mailcert.dat", CST_FOLDER_PERSONAL));

    test_set_purpose_ca(param);

    TEST_TITLE("Append multiple - STACK_OF(X509)");
    STACK_OF(X509) * sk = sk_X509_new_null();
    sk_X509_push(sk, param->cert);
    sk_X509_push(sk, param->ca);
    GSList * res = CST_append_sk_X509(param->st, sk);
    TEST_DO_SIMPLE(GPOINTER_TO_INT(res->data) == CST_ERROR_CERT_EXIST);
    TEST_DO_SIMPLE((res->next != NULL) 
            && (GPOINTER_TO_INT(res->next->data) == CST_ERROR_CERT_EXIST)
            && (res->next->next == NULL));
    g_slist_free(res);
    sk_X509_free(sk);

    TEST_TITLE("Get certID by issuer + serial");
    param->cert_uid = get_cert_id_by_x(param->st, param->cert);
    TEST_DO_SIMPLE(param->cert_uid > 0);
    param->ca_uid = get_cert_id_by_x(param->st, param->ca);
    TEST_DO_SIMPLE(param->ca_uid > 0);
    param->user_cert_uid = get_cert_id_by_x(param->st, param->user_cert);
    TEST_DO_SIMPLE(param->user_cert_uid > 0);

    printf("\tcertID: %u, %u, %u\n", param->cert_uid, param->ca_uid, param->user_cert_uid);

    test_import_PKCS12(param);
    test_import_corrupted(param);
}



int test_import_dir(CST * st, const char *dirname, const cst_t_cert_folder f)
{
    DIR *dir;
    struct dirent *dent;
    struct stat stt;
    char filename[FILENAME_MAX];

    dir = opendir(dirname);
    if (!dir)
    {
        TEST_ERROR("Directory not found");
    }

    int counter_ok = 0;
    int counter_bad = 0;
    while ((dent = readdir(dir)))
    {
        printf(".");
        sprintf(filename, "%s/%s", dirname, dent->d_name);

        if (stat(filename, &stt) < 0)
        {
            continue;
        } else if (S_ISREG(stt.st_mode))
        {
            if (test_import_file(st, filename, f))
            {
                counter_ok++;
            } else
            {
                printf("%s\n", filename);
                counter_bad++;
            }
        }
    }

    closedir(dir);

    printf("\n     Total imported: %i. Not imported: %i\n", counter_ok,
           counter_bad);

    if (counter_bad) 
    {
        return FALSE;
    }

    return TRUE;
}

int test_import_file(CST * st, const char *filename, const cst_t_cert_folder f)
{
    int result = TRUE;

    FILE *fp;

    fp = fopen(filename, "r");

    if (!fp)
    {
        return FALSE;
    }

    if (CST_ERROR_OK != CST_import_cert_f(st, fp, NULL, f))
    {
        result = FALSE;
    }

    fclose(fp);
    
    return result;
}

static void test_import_key(PARAM * param, char *filename, 
        X509_NAME * name, int pub, char *password, int error_code)
{
    FILE *fp;

    if ((fp = fopen(filename, "r")))
    {
        if (pub)
        {
            TEST_DO_SIMPLE(CST_import_pub_key(param->st, name, fp) == error_code);
        }
        else
        {
            TEST_DO_SIMPLE(CST_import_priv_key(param->st, name, fp, password, DEF_PASS) == error_code);
        }
        fclose(fp);
    }
    else
    {
        TEST_DO_SIMPLE(FALSE);
    }
}

void test_import_keys(PARAM * param)
{
    TEST_TITLE("Import keys from file");

    test_import_key(param, "files/pub_key.pem", param->name, TRUE, NULL, CST_ERROR_OK);
#if 0
    /* Test check for duplicate */
    test_import_key(param, "files/pub_key.pem", param->name, TRUE, NULL, CST_ERROR_KEY_EXIST);
#endif    
    test_import_key(param, "files/priv_key.pem", param->name, FALSE, NULL, CST_ERROR_OK);
    test_import_key(param, "files/pair/k1.pem", param->user_name, FALSE, "123", CST_ERROR_OK);
    
    /* Test bad password */
    test_import_key(param, "files/pair/k1.pem", param->user_name, FALSE, "12", CST_ERROR_NOT_FOUND);
}

static void test_import_crl(PARAM * param, char *filename, int error_code)
{
    FILE *fp;

    if ((fp = fopen(filename, "r")))
    {
        TEST_DO_SIMPLE(CST_import_CRL(param->st, fp) == error_code);
        fclose(fp);
    }
    else
    {
        TEST_DO_SIMPLE(FALSE);
    }
}

void test_import_crls(PARAM * param)
{
    TEST_TITLE("Import CRL from file");

    test_import_crl(param, "files/crls/crl.pem", CST_ERROR_OK);
    
    /* Now not check duplicate */
    /* test_import_crl(param, "files/crls/crl.pem", CST_ERROR_CRL_EXIST); */

    /* Because some bug in libdb1 */
    /* test_import_crl(param, "files/crls/ThawteServerCA.pem", CST_ERROR_OK); */
}

static cst_t_seqnum imp_cert(CST * st, FILE * fp, int folder, int fmt)
{
    cst_t_seqnum result = 0;
    
    switch (fmt)
    {
    case IMP_FMT_PEM:
        result = CST_import_cert_adv(st, fp, folder, NULL);
        break;
    case IMP_FMT_DER:
        result = CST_import_cert_adv_DER(st, fp, folder, NULL);
        break;
    }

    return result;
}

static cst_t_seqnum imp_key(CST * st, FILE * fp, char *password, int fmt, X509_NAME * name)
{
    cst_t_seqnum result = 0;
    
    switch (fmt)
    {
    case IMP_FMT_PEM:
        result = CST_import_priv_key_adv(st, name, fp, password, DEF_PASS, NULL);
        break;
    case IMP_FMT_DER:
        result = CST_import_priv_key_adv_DER(st, name, fp, DEF_PASS, NULL);
        break;
    }

    return result;
}

void test_import_and_assign(PARAM * param, const char *filename, const int fmt)
{
    TEST_TITLE("Import cert and key from one file and assign it");
    FILE * fp;
    if ((fp = fopen(filename, "r")))
    {
        cst_t_seqnum certID = 0, keyID = 0;

        certID = imp_cert(param->st, fp, CST_FOLDER_PERSONAL, fmt);
        TEST_DO_SIMPLE(certID > 0);
        keyID = imp_key(param->st, fp, "whatever", fmt, param->name);
        TEST_DO_SIMPLE(keyID > 0);
        fclose(fp);
        TEST_DO_SIMPLE(CST_ERROR_OK == CST_assign(param->st, certID, keyID, DEF_PASS));

        // Remove certificate and key from store
        TEST_DO_SIMPLE(CST_ERROR_OK == CST_delete_priv_key(param->st, keyID));
        TEST_DO_SIMPLE(NULL == CST_get_key(param->st, keyID, DEF_PASS));
        TEST_DO_SIMPLE(CST_ERROR_OK == CST_delete_cert(param->st, certID));
        TEST_DO_SIMPLE(NULL == CST_get_cert(param->st, certID));
    }
    else
    {
        TEST_DO_SIMPLE(FALSE);
    }
}

void test_import1(PARAM * param, const char *filename, const int fmt)
{
    TEST_TITLE("Import cert and key in direct seq");
    FILE * fp;
    
    cst_t_seqnum certID = 0, keyID = 0;
    
    if ((fp = fopen(filename, "r")))
    {     
        certID = imp_cert(param->st, fp, CST_FOLDER_PERSONAL, fmt);
        TEST_DO_SIMPLE(certID > 0);
        fclose(fp);
    } 

    if ((fp = fopen(filename, "r")))
    { 
        keyID = imp_key(param->st, fp, "whatever", fmt, param->name);
        TEST_DO_SIMPLE(keyID > 0);
        fclose(fp);
    }    
    
    TEST_DO_SIMPLE(CST_ERROR_OK == CST_assign(param->st, certID, keyID, DEF_PASS));
    
    TEST_DO_SIMPLE(CST_ERROR_OK == CST_delete_priv_key(param->st, keyID));
    TEST_DO_SIMPLE(NULL == CST_get_key(param->st, keyID, DEF_PASS));
    TEST_DO_SIMPLE(CST_ERROR_OK == CST_delete_cert(param->st, certID));
    TEST_DO_SIMPLE(NULL == CST_get_cert(param->st, certID));
}

void test_import2(PARAM * param, const char *filename, const int fmt)
{
    TEST_TITLE("Import cert and key in reverse");
    FILE * fp;
    
    cst_t_seqnum certID = 0, keyID = 0;
   
    if ((fp = fopen(filename, "r")))
    { 
        keyID = imp_key(param->st, fp, "whatever", fmt, param->name);
        TEST_DO_SIMPLE(keyID > 0);
        fclose(fp);
    }
    
    if ((fp = fopen(filename, "r")))
    {     
        certID = imp_cert(param->st, fp, CST_FOLDER_PERSONAL, fmt);
        TEST_DO_SIMPLE(certID > 0);
        fclose(fp);
    } 

    TEST_DO_SIMPLE(CST_ERROR_OK == CST_assign(param->st, certID, keyID, DEF_PASS));
    
    TEST_DO_SIMPLE(CST_ERROR_OK == CST_delete_priv_key(param->st, keyID));
    TEST_DO_SIMPLE(NULL == CST_get_key(param->st, keyID, DEF_PASS));
    TEST_DO_SIMPLE(CST_ERROR_OK == CST_delete_cert(param->st, certID));
    TEST_DO_SIMPLE(NULL == CST_get_cert(param->st, certID));
}

void test_import_and_assign_complex(PARAM * param, const char *filename, const int fmt)
{
    test_import_and_assign(param, filename, fmt);
    test_import1(param, filename, fmt);
    test_import2(param, filename, fmt);
}

int confirm_cb(X509 * xcert,
               cst_t_cert_folder * folder,
               cst_t_cert_purpose * purpose,
               unsigned char ** out_password,
               int is_pair, 
               int *cancel,
               void *data)
{
    printf("\tPKCS12: in confirm_cb (%i)\n", is_pair);
    return TRUE;
}
    
int error_cb(X509 * xcert, int error, void *data)
{
    PARAM * param = (PARAM *) data;
    printf("\tPKCS12: in error_cb %i\n", error);
    TEST_DO_SIMPLE(CST_ERROR_CERT_EXIST == error);
    return FALSE;
}

void test_import_PKCS12(PARAM * param)
{
    TEST_TITLE("Import PKCS12 (1)");
    FILE *fp = fopen("files/test.p12", "r");
    if (fp)
    {
        GError * error = NULL;
        CST_import_PKCS12(param->st, fp, &confirm_cb, &error_cb, "123456", param, &error);
        if (error)
        {
            printf("\tPKCS12: Error message: %i %s\n", error->code, error->message);
            TEST_DO_SIMPLE(FALSE);
        }
        else
        {
            TEST_DO_SIMPLE(TRUE);
        }
        g_clear_error(&error);
        fclose(fp);
    }
}

void test_import_corrupted(PARAM * param)
{    
    TEST_TITLE("Test import corrupted certificate (PEM)");
    FILE * fp;
    fp = fopen("files/corrupted.pem", "r");
    if (fp != NULL)
    {
        //int result = CST_import_cert(param->st, fp, NULL);
        //printf("\tresult=%i\n", result);
        //TEST_DO_SIMPLE(0 != result);

        GError * error = NULL;
        CST_import_PKCS12(param->st, fp, &confirm_cb, &error_cb, "123456", param, &error);
        if (error)
        {
            printf("\tPKCS12: Error message: %i %s\n", error->code, error->message);
            TEST_DO_SIMPLE(CST_ERROR_STRUCTURE_CORRUPT == error->code);
        }
        else
        {
            TEST_DO_SIMPLE(FALSE);
        }
        g_clear_error(&error);
    }
    else
    {
        TEST_DO_SIMPLE(FALSE);
    }
}
