/* Certificate Management library
 * 
 * Copyright (C) 2005 Nokia. All rights reserved.
 * Author: Ed Bartosh <Eduard.Bartosh@nokia.com>
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License version 2.1 as published by the Free Software Foundation.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the
 * Free Software Foundation, Inc., 59 Temple Place - Suite 330,
 * Boston, MA 02111-1307, USA.
 */

/**
   @file cst_key.h

   Key functions and structures
*/

#ifndef CST_KEY_H_
#define CST_KEY_H_

#include "cst.h"
#include "cst_const.h"
#include "cst_crypt.h"
#include <glib.h>
#include <openssl/evp.h>

#ifdef __cplusplus
extern "C" {
#endif

/* Cert description structure */
    typedef struct KEY_st {
        X509_NAME *name;
        unsigned char *der_buffer;
        int der_buffer_len;
        t_bool pub;
        t_bool enc;
        t_key_type type;
        t_seqnum cert_link;
        int uid;
    } KEY;

/**
    Append key to list and update indexes

    @return TRUE if append and FALSE if exist
*/
    gboolean key_put(CST * st, KEY * key);

    int CST_append_EVP_PKEY(CST * st, EVP_PKEY * key,
                            X509_NAME * name, const t_bool pub, 
                            unsigned char *password);

/**
    Remove key from list and update indexes
*/
    void key_remove_i(CST * st, KEY * k);

    void key_remove(CST * st, KEY * k);

/**
    Unlink key from cert
*/
    void key_unlink(CST * st, KEY * k);
/**
    Create CST_KEY structure
*/
    KEY *key_new(CST * st, EVP_PKEY * x, X509_NAME * name, t_bool pub, unsigned char *password);
    KEY *key_new0();

/**
    Init keyificate idx
*/
    int cst_key_list_init(CST * st);

/**
    Destroy keyificate idx
*/
    int cst_key_list_destroy(CST * st);

/**
    Free CST_KEY structure
*/
    void key_free(CST * st, KEY * k);

/**
    Read from buffer(file)
*/
    int key_read(CST * st, unsigned char *buffer, t_rsize len);

/**
    Write to file
*/
    int key_write(KEY * key, CRYPT_FILE * cf);
    int cst_key_all_write(CST * st, CRYPT_FILE * cf);

/**
    Append key to storage
*/
    t_seqnum key_append(CST * st, EVP_PKEY * key, X509_NAME * name, t_bool pub, unsigned char *pasword, GError **error);

/**
    Get EVP_PKEY structure
*/
    EVP_PKEY *key_get_EVP_PKEY(CST * st, const t_bool pub,
            const t_seqnum keyID, unsigned char *password, 
            GError ** error);


/**
    Import key from file
*/
    t_seqnum cst_import_key(CST * st, FILE * file,
                       unsigned char *inpass, unsigned char *outpass,
                       X509_NAME * name, const t_bool pub, GError **error);
    
    t_seqnum cst_import_key_DER(CST * st, FILE * file,
                       unsigned char *outpass,
                       X509_NAME * name, const t_bool pub, GError **error);

/**
    Export key(s) to file
*/
    int cst_export_keys_fmt(CST * st, FILE * file, X509_NAME * account,
                        unsigned char *password, const t_bool pub, 
                        const int fmt);

    int cst_export_key_fmt(CST * st, FILE * file, EVP_PKEY * key,
                       unsigned char *password, const t_bool pub, 
                       const int fmt);

/**
    Delete key (s)
*/
    int cst_delete_key(CST * st, 
                       const t_seqnum id, const t_bool pub);

    int cst_delete_keys(CST * st, X509_NAME * account, const t_bool pub);

/**
    Search key
*/
    GSList * cst_key_search_by_name(CST * st, X509_NAME * name,
                                      const t_bool pub);
    KEY *key_search_x(CST * st, X509_NAME * name,
                      EVP_PKEY * key, const t_bool pub);
    gboolean key_is_exist(CST * st, const KEY * k);

    KEY *key_search_id(CST * st, const t_bool pub, const t_seqnum id);

/**
    Compare key
*/
    gboolean key_equal(const KEY * key_a, const KEY * key_b);

    
    void initial_scan_db_key(CST * st, DBT * key, DBT * data);
    
#ifdef __cplusplus
}
#endif
#endif                          /* CST_KEY_H_ */
