/* Certificate Management library
 * 
 * Copyright (C) 2005 Nokia. All rights reserved.
 * Author: Ed Bartosh <Eduard.Bartosh@nokia.com>
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License version 2.1 as published by the Free Software Foundation.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the
 * Free Software Foundation, Inc., 59 Temple Place - Suite 330,
 * Boston, MA 02111-1307, USA.
 */

/**
    @file cst_key.c

    Certificate Management Library

    Key functions (common)
*/

#include "cst_t.h"
#include <memory.h>
#include <openssl/evp.h>

static int key_save(CST * st, const t_seqnum keyID, KEY * key_info);
static int key_load(CST * st, const t_seqnum keyID, KEY ** key_info);
static int key_parser_info(CST * st, DBT * info, KEY ** ptr_k);
static void key_free_der(KEY * k);

X509_NAME * CST_get_key_account(CST * st, cst_t_seqnum keyID)
{
    X509_NAME * result = NULL;
    CST_LOCK_BEGIN_S(LOCK_SH);
    if (st && (keyID > 0))
    {
        KEY * k = NULL;

        if (CST_ERROR_OK == key_load(st, keyID, &k))
        {
            g_assert(k);
            result = X509_NAME_dup(k->name);
            key_free(st, k);
        }
    }
    CST_LOCK_END;
    return result;
}

int CST_append_EVP_PKEY(CST * st, EVP_PKEY * key, X509_NAME * name,
                        const t_bool pub, unsigned char *password)
{
    if (st && key && name)
    {
        GError *p_error = NULL;
        key_append(st, key, name, pub, password, &p_error);
        if (p_error)
        {
            return p_error->code;
        }
        else
        {
            return CST_ERROR_OK;
        }
    }
    else
    {
        return CST_ERROR_PARAM_INCORRECT;
    }
}

static int key_save(CST * st, const t_seqnum keyID, 
                    KEY * key)
{
    TRACE("*");
    g_assert(st && (keyID > 0) && key);
    
    /* i2d for key->name */
    unsigned char *name_buffer = NULL;
    int name_buffer_size = i2d_X509_NAME(key->name, &name_buffer);
    g_assert(name_buffer_size > 0);
    
    g_assert(key->der_buffer_len > 0);
   
    /* Make body */
    DBT body;
    body.size =    sz_rheader + sz_bool              /* 1. pub */
                 + sz_rheader + sz_key_type          /* 2. type */
                 + sz_rheader + name_buffer_size     /* 3. name */
                 + sz_rheader + sz_seqnum            /* 4. cert_link */
                 + sz_rheader + key->der_buffer_len; /* 5. body */

    body.data = g_malloc0(body.size);
    void *i = body.data;

    /* 1. public */
    write_header_mem(&i, CST_RT_PUB, sz_bool);
    write_mem(&i, &key->pub, sz_bool);

    /* 2. type */
    write_header_mem(&i, CST_RT_KEY_TYPE, sz_key_type);
    t_key_type temp_type = KEY_TYPE_TO_LE(key->type);
    write_mem(&i, &temp_type, sz_key_type);
    
    /* 3. name */
    write_header_mem(&i, CST_RT_NAME, name_buffer_size);
    write_mem(&i, name_buffer, name_buffer_size);
    g_free(name_buffer);
    
    /* 4. cert_link */
    write_header_mem(&i, CST_RT_CERTLINK, sz_seqnum);
    t_seqnum cert_link = SEQNUM_TO_LE(key->cert_link);
    write_mem(&i, &cert_link, sz_seqnum);

    /* 5. body */
    write_header_mem(&i, CST_RT_BODY, key->der_buffer_len);
    write_mem(&i, key->der_buffer, key->der_buffer_len);

    g_assert((body.data + body.size) == i);

    int r = cst_data_put(st, CST_SUFFIX_KEYINFO, keyID, &body);

    g_free(body.data);

    return r;
}

static int key_load(CST * st, const t_seqnum keyID, 
                    KEY ** key_info)
{
    g_assert(st);
    g_assert(keyID > 0);
    g_assert(key_info);

    DBT info;
   
    int result = cst_data_get(st, CST_SUFFIX_KEYINFO, keyID, &info);
    if (CST_ERROR_NOT_FOUND == result) result = CST_ERROR_KEY_NOTFOUND;
    
    if (CST_ERROR_OK == result) 
    {
        if (key_parser_info(st, &info, key_info) == CST_ERROR_OK)
        {
            (*key_info)->uid = keyID;
            result = CST_ERROR_OK;
        }    
    }
    return result;
}

static int key_parser_info(CST * st, DBT * info, KEY ** ptr_k)
{    
    g_assert(st && info);
    
    GSList * list = parse_buffer(info->data, info->size);

    if (!list)
    {
        return CST_error(CST_ERROR_STRUCTURE_CORRUPT);
    }

    PARSE_RES *res = NULL;
    GSList *i = NULL;
    unsigned char *name_buff = NULL;

    KEY * k = key_new0(st);
    *ptr_k = k;

    for (i = list; i != NULL; i = i->next)
    {
        res = (PARSE_RES *) i->data;

        switch (res->type)
        {
        case CST_RT_PUB:
            memcpy(&k->pub, res->data, sz_bool);
            break;
        case CST_RT_KEY_TYPE:
            memcpy(&k->type, res->data, sz_key_type);
            k->type = KEY_TYPE_FROM_LE(k->type);
            break;
        case CST_RT_NAME:
            name_buff = res->data;
            k->name = d2i_X509_NAME(NULL, &name_buff, res->size);
            break;
        case CST_RT_CERTLINK:
            memcpy(&(k->cert_link), res->data, sizeof(k->cert_link));
            k->cert_link = SEQNUM_FROM_LE(k->cert_link);
            break;
        case CST_RT_BODY:
            k->der_buffer_len = res->size;
            k->der_buffer = g_malloc(res->size);
            memcpy(k->der_buffer, res->data, res->size);
            break;
        }
    }
    
    parse_res_free(list);

    return CST_ERROR_OK;
}

/**
    Write key info
*/
int key_write(KEY * k, CRYPT_FILE * cf)
{
    // TODO: check count of written bytes for control
    g_assert(cf && k && k->name);

    unsigned char *name_buffer = NULL;
    int name_buffer_size = i2d_X509_NAME(k->name, &name_buffer);

    g_assert(name_buffer_size > 0);

    /* header */
    write_header(cf, CST_RT_KEY,
                 sz_rheader + sz_bool
                 + sz_rheader + sz_seqnum
                 + sz_rheader + sz_bool
                 + sz_rheader + sz_key_type
                 + sz_rheader + name_buffer_size
                 + sz_rheader + sz_seqnum
                 + sz_rheader + k->der_buffer_len);

    /* public */
    write_header(cf, CST_RT_PUB, sz_bool);
    crypt_writer_put(cf, &k->pub, sz_bool);

    /* id */
    write_header(cf, CST_RT_NEXTKEYID, sz_seqnum);
    t_seqnum id = SEQNUM_TO_LE(k->uid);
    crypt_writer_put(cf, &id, sz_seqnum);

    /* encoded */
    write_header(cf, CST_RT_ENC, sz_bool);
    crypt_writer_put(cf, &k->enc, sz_bool);

    /* type */
    write_header(cf, CST_RT_KEY_TYPE, sz_key_type);
    t_key_type temp_type = KEY_TYPE_TO_LE(k->type);
    crypt_writer_put(cf, &temp_type, sz_key_type);

    /* name */
    write_header(cf, CST_RT_NAME, name_buffer_size);
    crypt_writer_put(cf, name_buffer, name_buffer_size);
    g_free(name_buffer);

    /* cert_link */
    write_header(cf, CST_RT_CERTLINK, sz_seqnum);
    t_seqnum cert_link = SEQNUM_TO_LE(k->cert_link);
    crypt_writer_put(cf, &cert_link, sz_seqnum);

    /* body */
    write_header(cf, CST_RT_BODY, k->der_buffer_len);
    crypt_writer_put(cf, k->der_buffer, k->der_buffer_len);

    return CST_error(CST_ERROR_OK);
}

gboolean cst_key_all_write_op(gpointer key, gpointer value, gpointer data)
{
    g_assert(key && value && data);
    GSList *list = (GSList *) value;
    GSList *i;
    for (i = list; i != NULL; i = i->next)
    {
        key_write((KEY *) (i->data), (CRYPT_FILE *) data);
    }
    return FALSE;
}

int cst_key_all_write(CST * st, CRYPT_FILE * cf)
{
    TRACE("*");
    g_assert(st && cf);
    g_tree_foreach(st->keys, cst_key_all_write_op, cf);
    return CST_error(CST_ERROR_OK);
}

EVP_PKEY *key_get_EVP_PKEY(CST * st, const t_bool pub, 
            const t_seqnum keyID, unsigned char *password, GError ** error)
{
    TRACE("*");
    g_assert(st);

    KEY * k = NULL;
    EVP_PKEY * result = NULL;
    int err = CST_ERROR_OK;

    if (keyID > 0)
    {        
        err = key_load(st, keyID, &k);
        if(CST_ERROR_OK == err)
        {
            g_assert(k);
            
            g_assert(k->der_buffer_len > 0);
            g_assert(k->der_buffer);

            /* TODO: read man d2i_X509 */
            unsigned char *p;
            p = k->der_buffer;

            if (k->pub)
            {
                result = d2i_PublicKey(k->type, NULL, &p, k->der_buffer_len);
            } 
            else
            {
                int decrypted_length = k->der_buffer_len;
                unsigned char *decrypted = cst_decrypt_buffer(k->der_buffer, &decrypted_length, password);
                    
                if (decrypted)
                {
                    p = decrypted;
                    result = d2i_PrivateKey(k->type, NULL, &p, decrypted_length);
                    g_free(decrypted);
                }
            }

            if (!result)
            {
                err = CST_ERROR_PASSWORD_WRONG;
            }

            key_free(st, k);
        }
    }
    else
    {
        err = CST_ERROR_PARAM_INCORRECT;
    }

    if (CST_ERROR_OK != err) g_set_error(error, CST_ERROR_DOMAIN, err, "Can't read key from storage");
    
    return result;
}

/**
    Read info from buffer
*/
int key_read(CST * st, unsigned char *buffer, t_rsize len)
{
    g_assert(st && buffer);
    g_assert(len > 0);

    GSList *list = parse_buffer(buffer, len);

    if (!list)
    {
        return CST_error(CST_ERROR_STRUCTURE_CORRUPT);
    }

    KEY *k = NULL;
    PARSE_RES *res = NULL;
    GSList *i = NULL;
    unsigned char *name_buff = NULL;

    k = key_new0(st);

    for (i = list; i != NULL; i = i->next)
    {
        res = (PARSE_RES *) i->data;

        switch (res->type)
        {
        case CST_RT_PUB:
            memcpy(&k->pub, res->data, sz_bool);
            break;
        case CST_RT_NEXTKEYID:
            memcpy(&(k->uid), res->data, sizeof(k->uid));
            k->uid = SEQNUM_FROM_LE(k->uid);
            break;
        case CST_RT_ENC:
            memcpy(&k->enc, res->data, sz_bool);
            break;
        case CST_RT_KEY_TYPE:
            memcpy(&k->type, res->data, sz_key_type);
            k->type = KEY_TYPE_FROM_LE(k->type);
            break;
        case CST_RT_NAME:
            name_buff = res->data;
            k->name = d2i_X509_NAME(NULL, &name_buff, res->size);
            break;
        case CST_RT_CERTLINK:
            memcpy(&(k->cert_link), res->data, sizeof(k->cert_link));
            k->cert_link = SEQNUM_FROM_LE(k->cert_link);
            break;
        case CST_RT_BODY:
            k->der_buffer_len = res->size;
            k->der_buffer = g_malloc(res->size);
            memcpy(k->der_buffer, res->data, res->size);
            break;
        }
    }

    if (k->uid <= 0)
    {
        k->uid = cst_next_key_uid(st);
    }
    
    if (!k->name || !key_put(st, k))
    {
        TRACE("not put");
        key_free(st, k);
    }

    parse_res_free(list);

    return CST_error(CST_ERROR_OK);
}

/**
    Initialize key list (constructor)
*/
int cst_key_list_init(CST * st)
{
    g_assert(st);
    TRACE("*");

    st->keys = g_tree_new(helper_name_cmp);
    st->idx_key_uid = g_hash_table_new(g_direct_hash, g_direct_equal);

    return CST_error(CST_ERROR_OK);
}

gboolean destroy_key(gpointer key, gpointer value, gpointer data)
{
    g_assert(key && value && data);
    GSList ** plist = (GSList **) data;
    
    *plist = g_slist_concat(*plist, g_slist_copy((GSList *) value));

    return FALSE;
}

/**
    Free key list (destructor)
    Memory used by key info free also
*/
int cst_key_list_destroy(CST * st)
{
    g_assert(st);
    
    TRACE("*");
    
    GSList * keys = NULL;
    g_tree_foreach(st->keys, destroy_key, &keys);
    GSList *i;
    for (i = keys; i != NULL; i = i->next)
    {
        key_remove_i(st, (KEY *) i->data);
    }
    g_slist_free(keys);
    g_tree_destroy(st->keys);

    g_hash_table_destroy(st->idx_key_uid);
    st->idx_key_uid = NULL;
    
    return CST_error(CST_ERROR_OK);
}

/**
    Append key to storage  
    and update indexes
*/
t_seqnum key_append(CST * st, EVP_PKEY * key, X509_NAME * name, t_bool pub, 
                    unsigned char *password, GError **error)
{
    TRACE("*");
    g_assert(st && key && name);

    t_seqnum keyID = 0;
    int result = CST_ERROR_OK;

    CST_LOCK_BEGIN(LOCK_EX);
        KEY *k = key_new(st, key, name, pub, password);
        keyID = k->uid;
        g_assert(k);
        key_put(st, k);

        result = key_save(st, k->uid, k);
        if (CST_ERROR_OK == result)
        {
            key_free_der(k);
        }
        else
        {
            key_remove(st, k);
        }
    CST_LOCK_END;    
       
    g_set_error(error, CST_ERROR_DOMAIN, result, "No comment");
    return keyID;
}

static void key_free_der(KEY * k)
{
    g_assert(k);
    g_free(k->der_buffer);
    k->der_buffer = NULL;
    k->der_buffer_len = 0;
}


/**
    Create empty KEY structure
*/
KEY *key_new0(CST * st)
{
    g_assert(st);
    KEY *k = (KEY *) g_malloc0(sizeof(KEY));
    return k;
}

/**
    Create KEY structure
*/
KEY *key_new(CST * st, EVP_PKEY * key, X509_NAME * name, t_bool pub, unsigned char *password)
{
    TRACE("*");
    g_assert(st && key && name);

    KEY *k = key_new0(st);
    k->uid = cst_next_key_uid(st);
    k->name = X509_NAME_dup(name);
    k->type = key->type;
    k->pub = pub;

    g_assert(k->uid > 0);

    if (pub)
    {
        k->der_buffer_len = i2d_PublicKey(key, &k->der_buffer);
    } else
    {
        unsigned char *decrypted = NULL;
        int decrypted_length;
        decrypted_length = i2d_PrivateKey(key, &decrypted);
        if (decrypted_length > 0)
        {
            k->der_buffer = cst_encrypt_buffer(decrypted, &decrypted_length, password);
            k->der_buffer_len = decrypted_length;
            g_free(decrypted);
        }
    }

    g_assert(k->der_buffer);

    if (k->der_buffer_len <= 0)
    {
        CST_error(CST_ERROR_NOT_INIT);
        key_free(st, k);
        return NULL;
    }

    return k;
}


/**
    Free KEY structure
*/
void key_free(CST * st, KEY * k)
{
    TRACE("*");
    g_assert(st && k);
    if (k->name)
    {
        X509_NAME_free(k->name);
    }    
    g_free(k->der_buffer);
    //key_unlink(st, k);
    g_free(k);
}


/**
    Import key from file
*/
static t_seqnum cst_import_key_fmt(CST * st, FILE * file, unsigned char *inpass,
        unsigned char *outpass, X509_NAME * name, const t_bool pub,
        const int fmt, GError **error)
{
    if (st && file && name)
    {
        EVP_PKEY *x = NULL;
        if (pub)
        {
            switch (fmt)
            {
            case IMPORT_FORMAT_PEM:
                x = PEM_read_PUBKEY(file, NULL, NULL, (void *) inpass);
                break;
            case IMPORT_FORMAT_DER:
                x = d2i_PUBKEY_fp(file, NULL);
                break;
            }
        } else
        {
            switch (fmt)
            {
            case IMPORT_FORMAT_PEM:
                x = PEM_read_PrivateKey(file, NULL, NULL, (void *) inpass);
                break;
            case IMPORT_FORMAT_DER:
                x = d2i_PrivateKey_fp(file, NULL);
                break;
            }
        }

        if (!x)
        {
            g_set_error(error, CST_ERROR_DOMAIN, CST_ERROR_NOT_FOUND, "No comment");
            return 0;
        }
        else
        {
            t_seqnum result = key_append(st, x, name, pub, outpass, error);
            EVP_PKEY_free(x);
            return result;
        }
    }
    else
    {
        g_set_error(error, CST_ERROR_DOMAIN, CST_ERROR_PARAM_INCORRECT, "Incorrect params");
        return 0;
    }
}

t_seqnum cst_import_key(CST * st, FILE * file, unsigned char *inpass,
        unsigned char *outpass, X509_NAME * name, const t_bool pub, 
        GError **error)
{
    return cst_import_key_fmt(st, file, inpass, outpass, name, pub, IMPORT_FORMAT_PEM, error);
}

t_seqnum cst_import_key_DER(CST * st, FILE * file, 
        unsigned char *outpass, X509_NAME * name, const t_bool pub, 
        GError **error)
{
    return cst_import_key_fmt(st, file, NULL, outpass, name, pub, IMPORT_FORMAT_DER, error);
}

/**
    Export keys with given account
*/
int cst_export_keys_fmt(CST * st, FILE * file, X509_NAME * account,
                        unsigned char *password, const t_bool pub, 
                        const int fmt)
{
    int result = CST_ERROR_PARAM_INCORRECT;
    GError * error = NULL;
    
    if (st && file && account)
    {
        CST_LOCK_BEGIN(LOCK_SH);
            if (pub)
            {
                GSList *list = (GSList *) g_tree_lookup(st->keys, account);

                GSList *i;
                KEY *k;
                EVP_PKEY *x;
                result = CST_ERROR_OK;
                for (i = list; i != NULL; i = i->next)
                {
                    k = (KEY *) i->data;
                    if (pub == k->pub)
                    {
                        x = key_get_EVP_PKEY(st, pub, k->uid, NULL, &error);

                        if (x)
                        {
                            result = cst_export_key_fmt(st, file, x, password, pub, fmt);
                            EVP_PKEY_free(x);
                        }
                        else
                        {
                            result = CST_ERROR_KEY_NOTFOUND;
                            if (error)
                            {
                                result = error->code;
                                g_clear_error(&error);
                            }
                        }

                        if (CST_ERROR_OK != result)
                        {
                            break;
                        }
                    }
                }

                CST_error(result);
            }
            else
            {
                result = CST_error(CST_ERROR_NOT_IMPLEMENTED);
            }
        CST_LOCK_END;
        CST_LOCK_SET_ERROR(result);
    }
    
    CST_error(result);
    return result;
}

/**
    Export key from file
*/
int cst_export_key_fmt(CST * st, FILE * file, EVP_PKEY * key,
        unsigned char *password, const t_bool pub, const int fmt)
{
    TRACE("*");
    
    int result = CST_ERROR_PARAM_INCORRECT;

    if (file && key)
    {
        if (pub)
        {
            switch (fmt)
            {
            case IMPORT_FORMAT_PEM:
                if (PEM_write_PUBKEY(file, key))
                {
                    result = CST_ERROR_OK;
                }
                break;
            case IMPORT_FORMAT_DER:
                if (i2d_PUBKEY_fp(file, key))
                {
                    result = CST_ERROR_OK;
                }
                break;
            }
        } else
        {
            switch (fmt)
            {
            case IMPORT_FORMAT_PEM:
                if (PEM_write_PrivateKey(file, key, DEFAULT_ALGORITHM,
                                          NULL, 0, NULL, (void *) password))
                {
                    result = CST_ERROR_OK;
                }
                break;
            case IMPORT_FORMAT_DER:
                if (i2d_PrivateKey_fp(file, key))
                {
                    result = CST_ERROR_OK;
                }
                break;
            }
        }
    }
    return result;
}


static gboolean remove_key_link(gpointer key, gpointer value, gpointer data)
{
        gpointer *adata = (gpointer*)data;
        t_seqnum keyID = * (t_seqnum*)adata[0];
        CST *st = (CST*)adata[1];        
        CERT *cert = (CERT *) value;
        
        if(cert->key_link == keyID)
        {
                cst_set_keylink(st, cert->uid, 0);
                return TRUE;
        }
        return FALSE;        
}

int cst_delete_key(CST * st, const t_seqnum keyID, const t_bool pub)
{
    int result = CST_ERROR_PARAM_INCORRECT;

    if (st && (keyID > 0))
    {
        CST_LOCK_BEGIN(LOCK_EX);
            result = cst_db_delete(st, CST_SUFFIX_KEYINFO, keyID);
            KEY * key = key_search_id(st, pub, keyID);

            if (key)
            {
                key_remove(st, key);
            }

            {
                    gpointer adata[2] = { &keyID, st };
                    g_tree_foreach(st->certs, remove_key_link, &adata[0]);
            }
            
        CST_LOCK_END;
    }

    return CST_error(result);
}

int cst_delete_keys(CST * st, X509_NAME * account, const t_bool pub)
{
    if (st && account)
    {
            int result = CST_ERROR_OK;
        CST_LOCK_BEGIN(LOCK_EX);
            GSList *list =
                g_slist_copy((GSList *) g_tree_lookup(st->keys, account));
            GSList *i;
            KEY *k;
            for (i = list; i != NULL; i = i->next)
            {
                k = (KEY *) i->data;
                int rc = cst_db_delete(st, CST_SUFFIX_KEYINFO, k->uid);
                if(rc != CST_ERROR_OK)
                        result = rc;

                if (pub == k->pub)
                {
                    key_remove(st, k);
                }
            }
            g_slist_free(list);
        CST_LOCK_END;
        return CST_error(result);
    }
    
    return CST_error(CST_ERROR_PARAM_INCORRECT);
}

EVP_PKEY *CST_get_priv_key_by_UID(CST * st,
                                  X509_NAME * issuer,
                                  ASN1_INTEGER * serial,
                                  unsigned char *password)
{
    EVP_PKEY * result = NULL;
    GError * error = NULL;
    int err = CST_ERROR_OK;
   
    if (st && issuer && serial)
    {
        CST_LOCK_BEGIN(LOCK_SH);
            t_seqnum certID = cert_search_by_UID(st, issuer, serial);
            if (certID > 0)
            {
                CERT * c = cert_get_by_id(st, certID);
                if (c)
                {
                    t_seqnum kl = cst_get_keylink(st, c->uid);
                    if (kl > 0)
                    {
                        result = key_get_EVP_PKEY(st, FALSE, kl, password, &error);
                        if (error)
                        {
                            err = error->code;
                            g_clear_error(&error);
                        }
                    }
                    else
                    {
                        err = CST_ERROR_KEY_NOTFOUND;
                    }
                }
                else
                {
                    err = CST_ERROR_CERT_NOTFOUND;
                }
            }
            else
            {
                err = CST_ERROR_CERT_NOTFOUND;
            }
        CST_LOCK_END;
        CST_LOCK_SET_ERROR(err);
    }
    else
    {
        err = CST_ERROR_PARAM_INCORRECT;
    }

    CST_error(err);
    return result;
}

EVP_PKEY *CST_get_priv_key(CST * st, X509 * x, unsigned char *p)
{
    if (st && x)
    {
        return CST_get_priv_key_by_UID(st, X509_get_issuer_name(x),
                                   X509_get_serialNumber(x), p);
    }
    else
    {
        CST_error(CST_ERROR_PARAM_INCORRECT);
        return NULL;
    }
}

EVP_PKEY *CST_get_priv_key_default(CST * st, char *email, unsigned char *password)
{
    EVP_PKEY * result = NULL;
    GError * error = NULL;
    int err = CST_ERROR_OK;
    
    if (st && NAME_IS_NOT_EMPTY(email))
    {
        CST_LOCK_BEGIN(LOCK_SH);
            t_seqnum certID = cert_get_default_id(st, email);

            if (certID > 0)
            {
                CERT * cert = cert_get_by_id(st, certID);
                if (cert)
                {
                    t_seqnum kl = cert->key_link;
                    if (kl > 0)
                    {
                        result = key_get_EVP_PKEY(st, FALSE, kl, password, &error);
                        if (error) 
                        { 
                            err = error->code; 
                            g_clear_error(&error); 
                        }
                    }
                }
            }
        CST_LOCK_END;
        CST_LOCK_SET_ERROR(err);
    }
    else
    {
        err = CST_ERROR_PARAM_INCORRECT;
    }

    CST_error(err);
    return result;
}

EVP_PKEY * CST_get_pub_key(CST * st, const cst_t_seqnum keyID)
{
    EVP_PKEY * result = NULL;
    GError * error = NULL;
    int err = CST_ERROR_OK;
    
    if (st && (keyID > 0))
    {
        CST_LOCK_BEGIN(LOCK_SH);
            result = key_get_EVP_PKEY(st, TRUE, keyID, NULL, &error);
            if (error) 
            { 
                err = error->code; 
                g_clear_error(&error); 
            }
        CST_LOCK_END;
        CST_LOCK_SET_ERROR(err);
    }
    else
    {
        err = CST_ERROR_PARAM_INCORRECT;
    }

    CST_error(err);
    return result;
}

EVP_PKEY * CST_get_key(CST * st, const cst_t_seqnum keyID, 
            unsigned char *password)
{
    EVP_PKEY * result = NULL;
    GError * error = NULL;
    int err = CST_ERROR_OK;
    if (st)
    {
        CST_LOCK_BEGIN(LOCK_SH);
            result = key_get_EVP_PKEY(st, FALSE, keyID, password, &error);
            if (error)
            {
                err = error->code;
                g_clear_error(&error);
            }
        CST_LOCK_END;
        CST_LOCK_SET_ERROR(err);
    }
    else
    {
        err = CST_ERROR_PARAM_INCORRECT;
    }   

    CST_error(err);
    return result;
}

void initial_scan_db_key(CST * st, DBT * key, DBT * data)
{
    KEY * k;
            
    if (0 == key_parser_info(st, data, &k))
    {
        t_seqnum id = extract_id(key); 
        k->uid = id;
        key_free_der(k);
        key_put(st, k);
    }
}
