#include <config.h>
#ifdef DBUS_ENABLE
#include <gpsd_dbus.h>
#include <stdio.h>
#include <string.h>
#include <time.h>

static DBusConnection* connection = NULL;
static DBusError error;
static int listener_status = 0;

static int init_dbus_listener(void);

/* ----------------------------------------------------------------------- */
/*
 * Does what is required to initialize the dbus connection
 * This is pretty basic at this point, as we don't receive commands via dbus.
 * Returns 0 when everything is OK.
 */
int initialize_dbus_connection(void)
{
	dbus_error_init(&error);
	connection = dbus_bus_get(DBUS_BUS_SYSTEM, &error);
	if (connection == NULL) {
		/* report error */
		return 1;
	}
	init_dbus_listener();
	return 0;
}


/* ----------------------------------------------------------------------- */
int cleanup_dbus(void)
{
	if (dbus_error_is_set(&error)) {
		dbus_error_free(&error);
	}
	dbus_connection_unref(connection);
	connection=NULL;
	return 0;
}


/* ----------------------------------------------------------------------- */
void send_dbus_fix(struct gps_device_t* channel) {
/* sends the current fix data for this channel via dbus */
    struct gps_data_t*	gpsdata;
    struct gps_fix_t*	gpsfix;
    DBusMessage*	message;
    /*DBusMessageIter	iter;*/
    dbus_uint32_t	serial; /* collected, but not used */

    /* if the connection is non existent, return without doing anything */
    if (connection == NULL) return;

    gpsdata = &(channel->gpsdata);
    gpsfix = &(gpsdata->fix);

    message = dbus_message_new_signal(
		    "/org/gpsd",
		    "org.gpsd",
		    "fix");

    /* add the interesting information to the message */
    dbus_message_append_args (message,
		    	      DBUS_TYPE_DOUBLE, &(gpsfix->time),
			      DBUS_TYPE_INT32,	&(gpsfix->mode),
			      DBUS_TYPE_DOUBLE,	&(gpsfix->ept),
			      DBUS_TYPE_DOUBLE, &(gpsfix->latitude),
			      DBUS_TYPE_DOUBLE, &(gpsfix->longitude),
			      DBUS_TYPE_DOUBLE, &(gpsfix->eph),
			      DBUS_TYPE_DOUBLE, &(gpsfix->altitude),
			      DBUS_TYPE_DOUBLE, &(gpsfix->epv),
			      DBUS_TYPE_DOUBLE, &(gpsfix->track),
			      DBUS_TYPE_DOUBLE, &(gpsfix->epd),
			      DBUS_TYPE_DOUBLE, &(gpsfix->speed),
			      DBUS_TYPE_DOUBLE, &(gpsfix->eps),
			      DBUS_TYPE_DOUBLE, &(gpsfix->climb),
			      DBUS_TYPE_DOUBLE, &(gpsfix->epc),
			      DBUS_TYPE_INVALID);
    
    dbus_message_set_no_reply(message, TRUE);

    /* message is complete time to send it */
    dbus_connection_send(connection, message, &serial);
    dbus_message_unref(message);
}


/* ----------------------------------------------------------------------- */
int send_dbus_status_changed(struct gps_device_t* channel, int force)
{
/* sends the current changed status via dbus */
    struct gps_data_t*	gpsdata;
    DBusMessage*	message;
    DBusMessageIter	iter;
    dbus_uint32_t	serial; /* collected, but not used */
    dbus_bool_t         st;
    const char *        str;
    int                 ret = -1;
    int                 appended = 0;
    static double online_status = -1.0;
    static int fix_status = -1;
    static int mode_status = -1;
    static time_t last_sent = 0;

    gps_mask_t changed = 0;
    gps_mask_t check_these = ONLINE_SET | STATUS_SET | MODE_SET;

    if (connection == NULL)
	    return ret;

    gpsdata = &(channel->gpsdata);

    if (!(gpsdata->set & check_these)) {
	    return 0;
    }
    changed = gpsdata->set & check_these;
    if (changed) {
	    gpsd_report(4, "Changed mask 0x%x (old values: online=%ld, fix_status=%d, mode=%d)\n",
			changed, (long)online_status, fix_status, mode_status);
    }

    /* Send status once / minute (so that if dbus signal is lost,
     * then we get the status at least sometimes).
     */
    if (last_sent) {
	    time_t curr = time(0);
	    if ((last_sent+60) < curr) {
		    force = 1;
		    gpsd_report(4, "Forced status_changed because of time (%d)\n", curr-(last_sent+60));
	    }
    }


    /* We only send stuff if it has changed from previous value. */
    if (!force && (changed & ONLINE_SET)) {
	    /* Has the status changed from previous value */
	    if (online_status<0) {
		    online_status = gpsdata->online;
	    } else if (online_status == 0.0) {
		    if (gpsdata->online>0.0) {
			    /* ok, send online status */
			    online_status = gpsdata->online;
		    } else {
			    /* status not changed, ignore */
			    changed &= ~ONLINE_SET;
			    gpsd_report(4, "Online status (%f) not sent, status not changed (0x%x) (down).\n", gpsdata->online, changed);
		    }
	    } else if (online_status > 0.0) {
		    if (gpsdata->online==0.0) {
			    /* ok, send online status */
			    online_status = gpsdata->online;
		    } else {
			    /* status not changed, ignore */
			    changed &= ~ONLINE_SET;
			    gpsd_report(4, "Online status (%f) not sent, status not changed (0x%x) (up).\n", gpsdata->online, changed);
		    }
	    }
    }

    /* We only send stuff if it has changed from previous value. */
    if (!force && (changed & STATUS_SET)) {
	    /* Has the status changed from previous value */
	    if (fix_status<0) {
		    fix_status = gpsdata->status;
	    } else if (fix_status == 0) {
		    if (gpsdata->status>0) {
			    /* ok, send fix status */
			    fix_status = gpsdata->status;
		    } else {
			    /* status not changed, ignore */
			    changed &= ~STATUS_SET;
			    gpsd_report(4, "Fix status (%d) not sent, status not changed (0x%x) (down).\n", gpsdata->status, changed);
		    }
	    } else if (fix_status > 0) {
		    if (gpsdata->status==0) {
			    /* ok, send online status */
			    fix_status = gpsdata->status;
		    } else {
			    /* status not changed, ignore */
			    changed &= ~STATUS_SET;
			    gpsd_report(4, "Fix status (%d) not sent, status not changed (0x%x) (up).\n", gpsdata->status, changed);
		    }
	    }
    }


    /* We only send stuff if it has changed from previous value. */
    if (!force && (changed & MODE_SET)) {
	    /* as the status changed from previous value */
	    if (mode_status < 0) {
		    mode_status = gpsdata->fix.mode;
	    } else if (mode_status == 0) {
		    	if (gpsdata->fix.mode > 0) {
				/* Ok, send mode status */
				mode_status = gpsdata->fix.mode;
			} else {
				/* Status not changed, ignore */
				changed &= ~MODE_SET;
				gpsd_report(4, "Mode status (%d) not set, status not changed (0x%x)\n", gpsdata->fix.mode, changed);
			}
	    } else if (mode_status > 0) {
		    if (gpsdata->fix.mode != mode_status) {
			    /* ok, send fix mode */
			    mode_status = gpsdata->fix.mode;
		    } else {
			    /* status not changed, ignore */
			    changed &= ~MODE_SET;
			    gpsd_report(4, "Fix mode (%d) not sent, status not changed (0x%x)\n", gpsdata->fix.mode, changed);
		    }
	    }
    }


    /* If ONLINE_SET is only changed but the value has not really changed,
     * then ignore the signal because it is obsolete and not needed.
     */
    if (!force && !changed) {
	    gpsd_report(4, "Status dbus signal not sent, status not changed.\n");
	    return 0;
    }

    message = dbus_message_new_signal(
		    "/org/gpsd",
		    "org.gpsd",
		    "status_changed");

    dbus_message_iter_init_append(message, &iter);

#define APPEND(changed, dbus_type, value)				\
    if (gpsdata->set & (changed)) {					\
	    str = #changed;						\
	    st = dbus_message_iter_append_basic(&iter,			\
						DBUS_TYPE_STRING,	\
						&str);			\
	    if (st == FALSE) {						\
		    gpsd_report(1, "Out of memory\n");			\
		    return ret;						\
	    }								\
	    st = dbus_message_iter_append_basic(&iter,			\
						dbus_type,		\
						&(value));		\
	    if (st == FALSE) {						\
		    gpsd_report(1, "Out of memory\n");			\
		    return ret;						\
	    }								\
	    appended++;							\
    }

    /* Add stuff here when necessary (also add same SET checks at the beginning
     * of the function)
     */
    APPEND(ONLINE_SET, DBUS_TYPE_DOUBLE, gpsdata->online);
    APPEND(STATUS_SET, DBUS_TYPE_INT32, gpsdata->status);
    APPEND(MODE_SET, DBUS_TYPE_INT32, gpsdata->fix.mode);


    if (!appended)
	    goto OUT;

    dbus_message_set_no_reply(message, TRUE);

    dbus_connection_send(connection, message, &serial);

    gpsd_report(4, "status_changed: changed=0x%x, online=%lu, status=%d, mode=%d\n", changed, (unsigned long)gpsdata->online, gpsdata->status, gpsdata->fix.mode);
    last_sent = time(0);

OUT:
    dbus_message_unref(message);

    return 0;
}


/* ----------------------------------------------------------------------- */
static int init_dbus_listener()
{
	dbus_bool_t st;

	listener_status = 0;

	if (!connection) {
		return -1;
	}


	st = dbus_bus_request_name(connection,
				   GPSD_SERVICE,
				   0,
				   &error);
	if (st<0) {
		gpsd_report(0, "Dbus error: %s\n", error.message);
		return -1;
	} else {
		gpsd_report(4, "dbus result code = %d\n", st);
	}

	/* add a rule for which messages we want to see */
	dbus_bus_add_match(connection, 
			   "type='method_call',"
			   "interface='" GPSD_INTERFACE "',"
			   "path='" GPSD_PATH "'",
			   &error);
	dbus_connection_flush(connection);
	if (dbus_error_is_set(&error)) { 
		gpsd_report(0, "Cannot create dbus listener (%s)\n", error.message);
		return -1;
	}

	listener_status = 1;

	return 0;
}


#define allocated_device(dev)	((dev).gpsdata.gps_device[0] != '\0')

/* ----------------------------------------------------------------------- */
int check_dbus_messages(struct gps_device_t *devices, int max_devices)
{
	DBusMessage* msg;
	DBusMessageIter iter;
        DBusMessage *reply = NULL;
	int st = 0;
	int ok = 1;

	if (!connection) {
		return -1;
	}

	if (!listener_status)
		return -1;

	/* non blocking read of the next available message */
	dbus_connection_read_write(connection, 0);
	msg = dbus_connection_pop_message(connection);

	if (msg == NULL) { 
		return 0;
	}

	if (!dbus_message_is_method_call(msg,
					 GPSD_INTERFACE,
					 GPSD_GET_STATUS_REQ)) {
#if 0
		char *member, *iface, *path, *dest;
		path = dbus_message_get_path(msg);
		iface = dbus_message_get_interface(msg);
		member = dbus_message_get_member(msg);
		dest = dbus_message_get_destination(msg);

		gpsd_report(1, "dbus type=%s, dest=%s, path=%s, interface=%s, member=%s\n",
			    dbus_message_type_to_string(dbus_message_get_type(msg)),
			    dest, path, iface, member);
#endif

		dbus_message_unref(msg);
		return 0;
	} else {
		gpsd_report(4, "Got a dbus message for me.\n");
	}

	if (!dbus_message_iter_init(msg, &iter)) {
		gpsd_report(4, "Dbus message has no arguments!\n");
	} else {
		char *status_str;
		gps_mask_t set = 0;
		int cnt = 0;

		/* Check the arguments and mark what status the caller
		 * wants to know.
		 */
		while(1) {
			status_str = NULL;
			if (dbus_message_iter_get_arg_type(&iter) != DBUS_TYPE_STRING)
				break;
			dbus_message_iter_get_basic(&iter, &status_str);
			dbus_message_iter_next(&iter);

			gpsd_report(3, "[%d] checking status \"%s\"\n", ++cnt, status_str);

			if (!strcmp(status_str, "ONLINE_SET")) {
				set |= ONLINE_SET;
				continue;
			}

			if (!strcmp(status_str, "STATUS_SET")) {
				set |= STATUS_SET;
				continue;
			}
		}


		if (dbus_message_get_no_reply(msg) == FALSE) {
			/* There is really no need for reply but send it
			 * anyways. The status is sent in separate signal.
			 */
			reply = dbus_message_new_method_return(msg);
			if (reply == NULL) {
				gpsd_report(0,
					    "No mem dbus_message_new_method_return()");
				goto OUT;
			}

			if (dbus_message_append_args(reply,
						     DBUS_TYPE_INT32, &ok,
						     DBUS_TYPE_INVALID)==FALSE) {
				gpsd_report(0, "Dbus append args failed.");
				goto OUT;
			}

			if (!dbus_connection_send(connection, reply, NULL)) {
				gpsd_report(0, "Dbus reply send failed.");
			}
			dbus_connection_flush(connection);
		}


		if (set) {
			/* Ok, the caller wants to know some data, find out
			 * which device has this data.
			 */
			int i;
			struct gps_data_t *gpsdata;
			gps_mask_t changed = 0;
			double online = 0;
			int status = 0;

			for (i=0; i<max_devices; i++) {
				if (!(allocated_device(devices[i]))) {
					continue;
				}

				gpsdata = &devices[i].gpsdata;

				if (set & ONLINE_SET) {
					if (gpsdata->online)
						online = 1;
					changed |= ONLINE_SET;
					gpsd_report(3, "Returning online status\n");
				}

				if (set & STATUS_SET) {
					if (gpsdata->status)
						status = gpsdata->status;
					changed |= STATUS_SET;
					gpsd_report(3, "Returning fix status\n");
				}
			}

			if (changed) {
				/* Ok, we have something for the caller. */
				struct gps_device_t device;
				if (changed & ONLINE_SET) {
					if (online)
						device.gpsdata.online = (double)time(0);
					else
						device.gpsdata.online = 0.0;
				}
				if (changed & STATUS_SET) {
					if (status)
						device.gpsdata.status = status;
					else
						device.gpsdata.status = 0;
				}
				device.gpsdata.set = changed;

				gpsd_report(3, "Signaling status changes.\n");

				/* Send the status reply as a signal */
				send_dbus_status_changed(&device, 1);
				st = 1;
			}
		}
	}

OUT:
	dbus_message_unref(msg);

	return st;
}


/* ----------------------------------------------------------------------- */
int get_dbus_socket(int *fd)
{
	if (connection && fd)
		return (int)dbus_connection_get_socket(connection, fd);

	return 0;
}



/* ----------------------------------------------------------------------- */
#ifdef TEST
/*
gcc -DTEST -I. -I/usr/include/dbus-1.0 -I/usr/lib/dbus-1.0/include -I/usr/include/glib-2.0 -I/usr/lib/glib-2.0/include    -DDBUS_API_SUBJECT_TO_CHANGE=1    -g -O2 -Wall -Wcast-align -Wmissing-declarations -Wmissing-prototypes -Wstrict-prototypes -Wpointer-arith -Wreturn-type -D_GNU_SOURCE -ldbus-1 gpsd_dbus.c
*/

#include <string.h>
#include "gpsd.h"

void gpsd_report(int errlevel, const char *fmt, ... )
{
	char buf[BUFSIZ];
	va_list ap;

	strcpy(buf, "a.out");
	strcat(buf, ": ");
	va_start(ap, fmt) ;
	(void)vsnprintf(buf + strlen(buf), sizeof(buf)-strlen(buf), fmt, ap);
	va_end(ap);
	(void)fputs(buf, stdout);
}


main()
{
	struct gps_device_t device = {0};
	
	initialize_dbus_connection();

	device.gpsdata.set = ONLINE_SET | STATUS_SET | USED_SET | DEVICE_SET;
	device.gpsdata.online = (double)time(0);
	device.gpsdata.status = 1;

	if (send_dbus_status_changed(&device)) {
		printf("ERROR\n");
	} else
		printf("OK\n");

	device.gpsdata.set = ONLINE_SET | USED_SET | DEVICE_SET;
	device.gpsdata.online = (double)time(0);
	device.gpsdata.status = 1;

	if (send_dbus_status_changed(&device)) {
		printf("ERROR\n");
	} else
		printf("OK\n");

	device.gpsdata.set = STATUS_SET | USED_SET | DEVICE_SET;
	device.gpsdata.online = (double)time(0);
	device.gpsdata.status = 1;

	if (send_dbus_status_changed(&device)) {
		printf("ERROR\n");
	} else
		printf("OK\n");

	device.gpsdata.set = USED_SET | DEVICE_SET;
	device.gpsdata.online = (double)time(0);
	device.gpsdata.status = 1;

	if (send_dbus_status_changed(&device)) {
		printf("ERROR\n");
	} else
		printf("OK\n");
}
#endif

#endif /* DBUS_ENABLE */

