/**
 * @file vfolder.c VFolder functionality
 *
 * Copyright (C) 2003, 2004 Lars Lindner <lars.lindner@gmx.net>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

#include "support.h"
#include "callbacks.h"
#include "common.h"
#include "conf.h"
#include "debug.h"
#include "vfolder.h"

extern AppData *app_data;
gint rss_current_feed = 0;
gint rss_max_feed = 0;

/**
 * The list of all existing vfolders. Used for updating vfolder
 * information upon item changes
 */
static GSList *vfolders = NULL;

gboolean vfolder_apply_rules_for_item(feedPtr vp, itemPtr ip);

/************************************************************************/
/*                        PRIVATE FUNCTIONS                             */
/************************************************************************/

/**
 * Adds an item to this VFolder, this method is called
 * when any rule of the vfolder matched
 *
 * @param vp the vfolder
 * @param ip the item to add
 */
static void vfolder_add_item(feedPtr vp, itemPtr ip)
{
    GSList *iter = NULL;
    itemPtr tmp = NULL;

    g_assert(NULL != vp);
    g_assert(NULL != ip);

    /* check if the item was already added */
    iter = vp->items;
    while (NULL != iter) {
	tmp = iter->data;
	if ((ip->nr == tmp->nr) && (ip->fp == tmp->sourceFeed))
	    return;
	iter = g_slist_next(iter);
    }

    /* add an item copy to the vfolder */
    tmp = item_new();
    item_copy(ip, tmp);
    // not needed as sourceFeed is same as orig_fp
    tmp->orig_fp = ip->fp; 
    feed_add_item(vp, tmp);
    //now increase the number of search items found here
    app_data->app_ui_data->rss_search_found ++;
    //and if we're searching: then update ui. Because this can be called even when
    //we're not searching: particularly when we add a new feed. (see feed_add_items)
    //by moving it to here, we are actually ensuring that 
    if (app_data->app_ui_data->search_mode == SFM_SEARCH)
        ui_itemlist_search_append(tmp);
}

/** Used to remove a vfolder item copy from a vfolder 
  *
  * @param vp the vfolder
  * @param ip the item to remove
  */
static void vfolder_remove_item_copy(feedPtr vp, itemPtr ip)
{
    GSList *items = NULL;
    gboolean found = FALSE;

    g_assert(vp != NULL);
    ULOG_DEBUG("vfolder_remove_item_copy:....");
    items = vp->items;
    while (NULL != items) {
	if (items->data == ip)
	    found = TRUE;
	items = g_slist_next(items);
    }

    if (found) {
	item_free(ip);
	vp->items = g_slist_remove(vp->items, ip);
    //reduce the number of matching items too
    app_data->app_ui_data->rss_search_found--;
    } else {
	g_warning("vfolder_remove_item(): item not found...");
    }
}

/** Searches a given vfolder for a copy of the passed item and
  * removes them. Used for item remove propagation and for 
  * processing of removing vfolder rules
  *
  * @param vp the vfolder
  * @param ip the item to look for
  */
static void vfolder_remove_matching_item_copy(feedPtr vp, itemPtr ip)
{
    GSList *items = NULL;
    itemPtr tmp = NULL;

    ULOG_DEBUG("vfolder_remove_matching_item_copy");
    items = feed_get_item_list(vp);
    while (NULL != items) {
	tmp = items->data;
	g_assert(NULL != ip->fp);
	g_assert(NULL != tmp->fp);
	if ((ip->nr == tmp->nr) && (ip->fp == tmp->sourceFeed)) {
        ULOG_DEBUG("vfolder_remove_matching_item_copy: ....now removing it");
	    vfolder_remove_item_copy(vp, tmp);
	    break;
	}
	items = g_slist_next(items);
    }
}

/** Checks all items of the given feed list node against a vfolder. 
  * Used by vfolder_refresh().
  *
  * @param np a feed list node
  * @param userdata a vfolder to check against
  */
static void vfolder_apply_rules(nodePtr np, gpointer userdata)
{
    feedPtr vp = (feedPtr) userdata;
    feedPtr fp = (feedPtr) np;
    GSList *items = NULL;

    if (SFM_SEARCH != app_data->app_ui_data->search_mode)
	return;
    /* do not search in vfolders */
    
    if(FST_VFOLDER == feed_get_type(fp))
        return;

    ////debug_enter("vfolder_apply_rules");

    g_assert(app_data != NULL);
    g_assert(app_data->app_ui_data != NULL);
    ULOG_DEBUG("\n\nvfolder_apply_rules: \
        search stuff progressbar set %d : %d\n\n",
                 rss_current_feed, rss_max_feed);

    gtk_progress_bar_set_fraction(GTK_PROGRESS_BAR
				  (app_data->app_ui_data->progress_bar),
				  (float) rss_current_feed / rss_max_feed);

    if (FST_FEED == feed_get_type(fp))
	rss_current_feed = rss_current_feed + 1;

    //if (SFM_SEARCH != app_data->app_ui_data->search_mode)
	//return;

    //debug1(DEBUG_UPDATE, "applying rules for (%s)", feed_get_source(fp));
    feed_load(fp);

    /* check all feed items */
    items = feed_get_item_list(fp);
    /*tvh : 21May2006: Make sure it doesn't always have to go through the list
     by adding second condition
     Again, like displaying a feed, g_idle can be used in stead of ui_update()
      */
    while ((NULL != items) && (app_data->app_ui_data->search_mode == SFM_SEARCH)){
        vfolder_apply_rules_for_item(vp, items->data);
        /* This function is not called from threads */
        ui_update();
        items = g_slist_next(items);
    }

    items = NULL;
    feed_unload(fp);
    ////debug_exit("vfolder_apply_rules");
}

/************************************************************************/
/*                        PUBLIC FUNCTIONS                              */
/************************************************************************/

/* sets up a vfolder feed structure */
feedPtr vfolder_new(void)
{
    feedPtr fp = NULL;
    gchar *tmp = NULL;

    //debug_enter("vfolder_new");

    fp = feed_new();;
    feed_set_type(fp, FST_VFOLDER);
    feed_set_title(fp, "vfolder");
    feed_set_source(fp, "vfolder");
    tmp = conf_new_id();
    feed_set_id(fp, tmp);
    g_free(tmp);
    feed_set_available(fp, TRUE);
    fp->fhp = feed_type_str_to_fhp("vfolder");
    vfolders = g_slist_append(vfolders, fp);

    //debug_exit("vfolder_new");

    return fp;
}

/* Method thats adds a rule to a vfolder. To be used
   on loading time or when creating searching. Does 
   not process items. Just sets up the vfolder */
void vfolder_add_rule(feedPtr vp, const gchar * ruleId,
		      const gchar * value, gboolean additive)
{
    rulePtr rp = NULL;

    //debug_enter("vfolder_add_rule");
    ULOG_DEBUG("vfolder_add_rule: checking new rule");
    if (NULL != (rp = rule_new(vp, ruleId, value, additive))) {
    	vp->rules = g_slist_append(vp->rules, rp);
    } else {
	g_warning("unknown rule id: \"%s\"", ruleId);
    }

    //debug_exit("vfolder_add_rule");
}

/* Method that remove a rule from a vfolder. To be used
   when deleting or changing vfolders. Does not process
   items. */
void vfolder_remove_rule(feedPtr vp, rulePtr rp)
{

    //debug_enter("vfolder_remove_rule");
    vp->rules = g_slist_remove(vp->rules, rp);
    //debug_exit("vfolder_remove_rule");
}

/** 
 * Searches all vfolders for copies of the given item and
 * removes them. Used for item remove propagation.
 */
void vfolder_remove_item(itemPtr ip)
{
    GSList *iter = NULL;
    feedPtr vp = NULL;

    ////debug_enter("vfolder_remove_item");
    ULOG_DEBUG("vfolder_remove_item");

    g_assert(ip != NULL);
    
    /* never process vfolder items! */
    g_assert(FST_VFOLDER != feed_get_type(ip->fp));

    iter = vfolders;
    while (NULL != iter) {
	vp = (feedPtr) iter->data;
	vfolder_remove_matching_item_copy(vp, ip);
	iter = g_slist_next(iter);
    }

    ////debug_exit("vfolder_remove_item");
}

/**
 * Check item against rules of a vfolder. When the item matches an 
 * additive rule it is added to the vfolder and when it matches an excluding 
 * rule the item is removed again from the given vfolder.
 */
gboolean vfolder_apply_rules_for_item(feedPtr vp, itemPtr ip)
{
    rulePtr rp = NULL;
    GSList *iter = NULL;
    int add = -1;
    int remove = -1;
    gboolean added = FALSE;

    g_assert(NULL != vp);
    g_assert(NULL != ip);

    /* Check against all rules using AND boolean logic */
    iter = vp->rules;

    while (NULL != iter) {
	rp = iter->data;
	if (rp->additive) {
	    add == -1 ? add += 2 : add++;
	    if (rule_check_item(rp, ip))
		add--;
	    else
		break;
	} else {
	    remove == -1 ? remove += 2 : remove++;
	    if (rule_check_item(rp, ip))
		remove--;
	}
	iter = g_slist_next(iter);
    }

    if (add == 0) {
	debug2(DEBUG_UPDATE, "adding matching item (%d): %s\n", ip->nr,
	       item_get_title(ip));
	vfolder_add_item(vp, ip);
	added = TRUE;
    }

    if (remove == 0 && added) {
	debug2(DEBUG_UPDATE, "deleting matching item (%d): %s\n", ip->nr,
	       item_get_title(ip));
	vfolder_remove_matching_item_copy(vp, ip);
	added = FALSE;
    }
    return added;
}

/* Method that applies the rules of the given vfolder to 
   all existing items. To be used for creating search
   results or new vfolders. Not to be used when loading
   vfolders from cache. */
void vfolder_refresh(feedPtr vp)
{
    GSList *iter = NULL;

    //debug_enter("vfolder_refresh");

    g_assert(vp != NULL);
    
    /* free vfolder items */
    iter = vp->items;
    while (NULL != iter) {
        item_free(iter->data);
        iter = g_slist_next(iter);
    }
    g_slist_free(vp->items);
    vp->items = NULL;

    rss_current_feed = 0;
    /*tvh: IMPORTANT search is done here */
    ULOG_DEBUG("vfolder_refresh: ui_feedlist__do_for_all_data with vfolder_apply rules to search feedlist");
    ui_feedlist_do_for_all_data(NULL,
				ACTION_FILTER_FEED |
				ACTION_FILTER_DIRECTORY,
				vfolder_apply_rules, vp);

    //debug_exit("vfolder_refresh");
}

/**
 * Method to be called when a feed item was updated. This maybe
 * after user interaction or updated item contents. For simplicity
 * we assume that an updated item still matches. 
 */
void vfolder_update_item(itemPtr ip)
{
    //same as case of vfolder_check_item
    return;
    GSList *iter = NULL, *items = NULL;
    itemPtr tmp = NULL;
    feedPtr vp = NULL;

    //debug_enter("vfolder_update_item");

    g_assert(ip != NULL);
    
    /* never process vfolder items! */
    g_assert(FST_VFOLDER != feed_get_type(ip->fp));

    iter = vfolders;
    while (NULL != iter) {
	vp = (feedPtr) iter->data;

	/* first step: update item copy if found */
	items = feed_get_item_list(vp);
	while (NULL != items) {
	    tmp = items->data;
	    g_assert(NULL != ip->fp);
	    g_assert(NULL != tmp->fp);
	    /* find the item copies */
	    if ((ip->nr == tmp->nr) && (ip->fp == tmp->sourceFeed)) {
		/* check if the item still matches, the item won't get added
		   another time so this call effectivly just checks if the
		   item is still to remain added. */
		if (TRUE == vfolder_apply_rules_for_item(vp, ip)) {
		    debug2(DEBUG_UPDATE,
			   "item (%s) used in vfolder (%s), updating vfolder copy...",
			   ip->title, vp->title);
		    item_copy(ip, tmp);
		} else {
		    debug2(DEBUG_UPDATE,
			   "item (%s) used in vfolder (%s) does not match anymore -> removing...",
			   ip->title, vp->title);
		    /* we need this: vfolder_remove_item(ip); but not here :-) */
		    item_copy(ip, tmp);	/* as long as we don't remove them at least update them */
		}
		break;
	    }
	    items = g_slist_next(items);
	}

	/* second step: update vfolder unread count */
	vp->unreadCount = 0;
	items = feed_get_item_list(vp);
	while (NULL != items) {
	    tmp = items->data;
	    if (FALSE == item_get_read_status(tmp))
		feed_increase_unread_counter(vp);

	    items = g_slist_next(items);
	}
	iter = g_slist_next(iter);
    }

    //debug_exit("vfolder_update_item");
}

/**
 * Method to be called when a new item needs to be checked
 * against all vfolder rules. To be used upon feed list loading
 * and when new items are downloaded.
 * tvh 2006May27: removed this func, as it's only relevent in the case we want to update
 * the search feed when adding new feeds, or getting new items.(see feed_load() )
 * Now this is a arguable feature whether or not to do that.
 * If user pressed STOP searching. Then he/she doesn't
 * want any more of the items. Adding this is then stupid.
 * If user wants to refresh search feed, he/she can redo the search.
 * This is probably the best way
 */
void vfolder_check_item(itemPtr ip)
{
    return;
    GSList *iter = NULL;

    //debug_enter("vfolder_check_item");

    g_assert(ip != NULL);
    
    /* never process vfolder items! */
    g_assert(FST_VFOLDER != feed_get_type(ip->fp));

    iter = vfolders;
    while (NULL != iter) {
	vfolder_apply_rules_for_item(iter->data, ip);
	iter = g_slist_next(iter);
    }

    //debug_exit("vfolder_check_item");
}

/* called when a vfolder is processed by feed_free
   to get rid of the vfolder items */
void vfolder_free(feedPtr vp)
{
    GSList *iter = NULL;

    //debug_enter("vfolder_free");

    vfolders = g_slist_remove(vfolders, vp);

    g_assert(vp != NULL);
    
    /* free vfolder items */
    iter = vp->items;
    while (NULL != iter) {
	item_free(iter->data);
	iter = g_slist_next(iter);
    }
    g_slist_free(vp->items);
    vp->items = NULL;

    /* free vfolder rules */
    iter = vp->rules;
    while (NULL != iter) {
	rule_free(iter->data);
	iter = g_slist_next(iter);
    }
    g_slist_free(vp->rules);
    vp->rules = NULL;

    //debug_exit("vfolder_free");
}

feedHandlerPtr vfolder_init_feed_handler(void)
{
    feedHandlerPtr fhp = NULL;

    fhp = g_new0(struct feedHandler, 1);

    /* prepare feed handler structure, we need this for
       vfolders too to set item and OPML type identifier */
    fhp->typeStr = "vfolder";
    fhp->icon = ICON_FOLDER_CLOSED;
    fhp->directory = FALSE;
    fhp->feedParser = NULL;
    fhp->checkFormat = NULL;
    fhp->merge = FALSE;

    rule_init();

    return fhp;
}
