/**
 * @file update.c feed update request processing
 *
 * Copyright (C) 2003, 2004 Lars Lindner <lars.lindner@gmx.net>
 * Copyright (C) 2004 Nathan J. Conrad <t98502@users.sourceforge.net>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

#include <unistd.h>
#include <stdio.h>
#include <sys/wait.h>
#include <osso-log.h>
#include <strings.h>
#include <gtkhtml/gtkhtml.h>
#include <gtkhtml/gtkhtml-stream.h>
#include <fcntl.h>
#include <libgnomevfs/gnome-vfs.h>
#include <libgnomevfs/gnome-vfs-utils.h>
#include "cache.h"
#include "support.h"
#include "debug.h"
#include "update.h"
#include "conf.h"
#include "ui_queue.h"
#include "ui_mainwindow.h"
#include "net/downloadlib.h"
#include "callbacks.h"
#include "dbus.h"

#define DOWNLOAD_TIMEOUT 120

/* define below if the queue should block if there is no connection */
//#define USE_BLOCKING_OFFLINE
#undef USE_BLOCKING_OFFLINE

/* must never be less than 2, because first thread works exclusivly on high prio requests */
#define DEFAULT_UPDATE_THREAD_CONCURRENCY	3

/* define if the download result processing is happening 
in a separate thread, otherwise it will happen in g_idle */
//#undef DOWNLOAD_PROCESS_THREAD
#define DOWNLOAD_PROCESS_THREAD

#undef UPDATE_DEBUG_MUTEX
//#define UPDATE_DEBUG_MUTEX
#ifdef UPDATE_DEBUG_MUTEX
#define G_MUTEX_LOCK(mx) { ULOG_DEBUG("##Trying to lock mutex: %s, %d, %p", __FUNCTION__, __LINE__, g_thread_self()); g_mutex_lock(mx); ULOG_DEBUG("##Locked mutex: %s, %d, %p", __FUNCTION__, __LINE__, g_thread_self()); }
#define G_MUTEX_UNLOCK(mx) { ULOG_DEBUG("##Trying to unlock mutex: %s, %d, %p", __FUNCTION__, __LINE__, g_thread_self()); g_mutex_unlock(mx); ULOG_DEBUG("##Unlocked mutex: %s, %d, %p", __FUNCTION__, __LINE__, g_thread_self()); }
#define GDK_THREADS_ENTERD { ULOG_DEBUG("@@Trying gdk_threads_enter: %s, %d, %p", __FUNCTION__, __LINE__, g_thread_self()); if (g_thread_self() != mainThread) gdk_threads_enter(); ULOG_DEBUG("@@Success gdk_threads_enter: %s, %d, %p", __FUNCTION__, __LINE__, g_thread_self()); }
#define GDK_THREADS_LEAVED { ULOG_DEBUG("@@Trying gdk_threads_leave: %s, %d, %p", __FUNCTION__, __LINE__, g_thread_self()); if (g_thread_self() != mainThread) gdk_threads_leave(); ULOG_DEBUG("@@Success gdk_threads_leave: %s, %d, %p", __FUNCTION__, __LINE__, g_thread_self()); }
#else
#define G_MUTEX_LOCK(mx) g_mutex_lock(mx);
#define G_MUTEX_UNLOCK(mx) g_mutex_unlock(mx);
#define GDK_THREADS_ENTERD { if (g_thread_self() != mainThread) gdk_threads_enter(); }
#define GDK_THREADS_LEAVED { if (g_thread_self() != mainThread) gdk_threads_leave(); }
#endif

/* communication queues for requesting updates and sending the results */
GAsyncQueue *requests_high_prio = NULL;
GQueue *requests_normal_prio = NULL;
GQueue *requests_image = NULL;
GAsyncQueue *results = NULL;
GHashTable *url_hash = NULL;
GHashTable *download_hash = NULL;
int rss_threads = 0;

extern AppData *app_data;
extern GThread *mainThread;

/* Variable for signaling the threads they can run */
static gboolean canRun = TRUE;
static gpointer dummyPointer = (gpointer) 1;

GThread *update_threads[DEFAULT_UPDATE_THREAD_CONCURRENCY+1];

/* condition mutex for offline mode */
static GMutex *cond_mutex = NULL;
static GMutex *queue_mutex = NULL;
static GCond *queue_cond = NULL;
static GCond *offline_cond = NULL;

G_LOCK_DEFINE_STATIC (image_mode_mutex);
static volatile gboolean image_mode = FALSE;
/*static gint max_download = 0;
static gint current_download = 0;
static gint max_img_download = 0;
static gint current_img_download = 0;*/

G_LOCK_DEFINE_STATIC (timeout_mutex);
static gint timeout_id = 0;

/* Counters */
static gint rss_current_download;
static gint rss_max_download;
static gint images_current_download;
static gint images_max_download;
static gint timeout_count;

/* prototypes */
static void *download_thread_main(void *data);
gboolean set_image_mode_simple(gboolean is_img);

/************************************************************************/
/*                        UTILITY FUNCTIONS                             */
/************************************************************************/

/* read the counter . It needs to be atomic */
gint get_counter(int type)
{
    switch (type) {
    case C_IMG_CURRENT:
        return g_atomic_int_get(&images_current_download);
    case C_IMG_MAX:
        return g_atomic_int_get(&images_max_download);
    case C_RSS_CURRENT:
        return g_atomic_int_get(&rss_current_download);
    case C_RSS_MAX:
        return g_atomic_int_get(&rss_max_download);
       
    default:
        ULOG_DEBUG("oooooooooooooooooooooERRORooooooooooooooooooooooooooooo");
        return 0;
    }
    return 0;    
}

static void inc_counter(int type)
{
    ULOG_DEBUG("=== Inc counter: %d = %d ===", type, get_counter(type));

    switch (type) {
    case C_IMG_CURRENT:
        g_atomic_int_inc(&images_current_download);
        break;
    case C_IMG_MAX:
        g_atomic_int_inc(&images_max_download);
        break;
    case C_RSS_CURRENT:
        g_atomic_int_inc(&rss_current_download);
        break;
    case C_RSS_MAX:
        g_atomic_int_inc(&rss_max_download);
        break;
       
    default:
        ULOG_DEBUG("oooooooooooooooooooooERRORooooooooooooooooooooooooooooo");
    }
}

static void reset_counter(int type)
{
    ULOG_DEBUG("¤¤ Reset counter: %d = %d ¤¤", type, get_counter(type));
    
    switch (type) {
    case C_IMG_CURRENT:
        while (!g_atomic_int_compare_and_exchange(&images_current_download, images_current_download, 0));
        break;
    case C_IMG_MAX:
        while (!g_atomic_int_compare_and_exchange(&images_max_download, images_max_download, 0)) ;
        break;
    case C_RSS_CURRENT:
        while (!g_atomic_int_compare_and_exchange(&rss_current_download, rss_current_download, 0)) ;
        break;
    case C_RSS_MAX:
        while (!g_atomic_int_compare_and_exchange(&rss_max_download, rss_max_download, 0)) ;
        break;
       
    default:
        ULOG_DEBUG("oooooooooooooooooooooERRORooooooooooooooooooooooooooooo");
    }
}

/************************************************************************/
/*                        PRIVATE FUNCTIONS                             */
/************************************************************************/

static gboolean dl_timeout_cb(gpointer data)
{
    download_cancel_all(FALSE);
    return FALSE;
}

static void setup_dl_timeout()
{
    G_LOCK(timeout_mutex);
    if (timeout_id)
        g_source_remove(timeout_id);
    timeout_count++;
    timeout_id = g_timeout_add(DOWNLOAD_TIMEOUT*1000, dl_timeout_cb, NULL);
    G_UNLOCK(timeout_mutex);
}

static void cancel_dl_timeout()
{
    G_LOCK(timeout_mutex);
    if (timeout_id)
        g_source_remove(timeout_id);
    timeout_id = 0;
    G_UNLOCK(timeout_mutex);
}

static void remove_dl_timeout()
{
    G_LOCK(timeout_mutex);
    if (--timeout_count <= 0) {
        timeout_count = 0;
        if (timeout_id)
            g_source_remove(timeout_id);
        timeout_id = 0;
    }
    G_UNLOCK(timeout_mutex);
}

/** This is needed during long downloads to handle
  * user input and to draw stuff to the screen
  */
/* not used anymore atm
static void download_wait_for_pending_events()
{
    GDK_THREADS_ENTERD;
    while (gtk_events_pending ())
	gtk_main_iteration ();
    GDK_THREADS_LEAVED;
}
*/

/** Filter idea (and some of the code) was taken from Snownews 
  *
  * @param cmd
  * @param data
  * @param size
  * @return 
  */
static char *filter(gchar * cmd, gchar * data, size_t * size)
{
    int fd = 0;
    gchar *command = NULL;
    const gchar *tmpdir = g_get_tmp_dir();
    char *tmpfilename = NULL;
    char *out = NULL;
    FILE *file = NULL, *p = NULL;

    tmpfilename =
	g_strdup_printf("%s" G_DIR_SEPARATOR_S "liferea-XXXXXX", tmpdir);

    fd = g_mkstemp(tmpfilename);

    if (fd == -1) {
	odebug1("Error opening temp file %s to use for filtering!",
		tmpfilename);
	g_free(tmpfilename);
	return NULL;
    }


    file = fdopen(fd, "w");
    fwrite(data, strlen(data), 1, file);
    fclose(file);

    *size = 0;
    command = g_strdup_printf("%s < %s", cmd, tmpfilename);
    p = popen(command, "r");
    g_free(command);
    if (NULL != p) {
	while (!feof(p) && !ferror(p)) {
	    size_t len;
	    out = g_realloc(out, *size + 1025);
	    len = fread(&out[*size], 1, 1024, p);
	    if (len > 0)
		*size += len;
	}
	pclose(p);
	out[*size] = '\0';
    }
    /* Clean up. */
    unlink(tmpfilename);
    g_free(tmpfilename);
    return out;
}

gboolean pop_request(struct request **request)
{
    if (is_image_mode())
        *request = g_queue_pop_head(requests_image);
    else
        *request = g_queue_pop_head(requests_normal_prio);
    
    return *request != NULL;
}

#ifndef DOWNLOAD_PROCESS_THREAD
gboolean download_process_results(gpointer data);
#endif

/** Download requests are processed here. High priority requests are
  * processed first
  * 
  * @param data if only high priority requests should be processed
  */
static void *download_thread_main(void *data)
{
    struct request *request = NULL;
    gboolean high_priority = (gboolean) data;
    gboolean mutex_needs_unlock = FALSE;

    for (;canRun;) {
        /* block updating if we are offline */
#ifdef USE_BLOCKING_OFFLINE
        if (!online) {
            ULOG_DEBUG("download_thread_main: now going offline!");
            g_mutex_lock(cond_mutex);
            g_cond_wait(offline_cond, cond_mutex);
            g_mutex_unlock(cond_mutex);
            ULOG_DEBUG("download_thread_main: going online again!");
        }
#endif

        /* do update processing */
        ULOG_DEBUG("download_thread_main: waiting for request... %p", g_thread_self());
        if (high_priority) {
            request = g_async_queue_pop(requests_high_prio);
        } else {
            request = g_async_queue_try_pop(requests_high_prio);
            if ((mutex_needs_unlock = (NULL == request))) {
                G_MUTEX_LOCK(queue_mutex);
                
                if (!pop_request(&request)) {
                    //wait for a request to arrive
                    g_cond_wait(queue_cond, queue_mutex);
                    pop_request(&request);
                }
            }
        }

        if (request && request != dummyPointer) {
            gboolean use_hash = !high_priority && is_image_mode()
                && request->type_image;
            if (use_hash)
                g_hash_table_remove(url_hash, request->source);
            
/*            ULOG_DEBUG("download_thread_main: processing received request (%s)",
                   request->source);*/

            if (request->callback == NULL) {
/*                ULOG_DEBUG("download_thread_main: freeing cancelled request (%s)",
                    request->source);*/
                if (mutex_needs_unlock)
                    G_MUTEX_UNLOCK(queue_mutex);
                download_request_free(request);
            } else {
                if (use_hash)
                    g_hash_table_insert(download_hash, request->source, request);
                if (mutex_needs_unlock)
                    G_MUTEX_UNLOCK(queue_mutex);

                setup_dl_timeout();
                download_process(request);
                remove_dl_timeout();

                if (!canRun) {
                    ULOG_DEBUG("download_thread_main: aborted!");
                    download_request_free(request);
                    return NULL;
                }
                
/*                ULOG_DEBUG("download_thread_main: After download process, request->data: %d",(request->data==NULL?0:1));
                ULOG_DEBUG("download_thread_main: for %s\n",request->source);*/

                /* return the request so the GUI thread can merge the feeds 
                   and display the results... */
//                download_wait_for_pending_events();

                if(app_data && 
                   app_data->app_ui_data &&
                   app_data->app_ui_data->search_mode == SFM_REFRESH)
                {
                    g_async_queue_push(results, (gpointer) request);
#ifndef DOWNLOAD_PROCESS_THREAD
                    if (!app_data->app_ui_data->download_dequeue_source)
                        app_data->app_ui_data->download_dequeue_source = 
                            g_idle_add(download_process_results, NULL);
#endif
                } else if (request != NULL)
                    download_request_free(request);
                    
                /*download_process_results(request);*/
            }
        } else {
            if (mutex_needs_unlock) 
                G_MUTEX_UNLOCK(queue_mutex);
        }
    }
    
    return NULL;
}

static gint g_idle_id = 0;
static gint g_idle_pb_id = 0;

gboolean call_ui_mainwindow_finish(gpointer data)
{
    g_idle_id = 0;
    
    ULOG_DEBUG("Calling ui_mainwindow_finish from idle ....");
    
    if (app_data && app_data->app_ui_data &&
                !app_data->app_ui_data->feed_displayed &&
    !app_data->app_ui_data->nospace)
        ui_mainwindow_finish();
    
    return FALSE;
}

gboolean call_update_progressbar(gpointer data)
{
    g_idle_pb_id = 0;

    ULOG_DEBUG("Update progressbar");    
    GDK_THREADS_ENTERD;
    update_progress_bar();
    GDK_THREADS_LEAVED;
    ULOG_DEBUG("Update progressbar done");
    
    return FALSE;
}

static gboolean
call_refresh_finished(gpointer data)
{
    cancel_dl_timeout();
    on_refresh_finished(app_data->app_ui_data->start_on_background, FALSE);
    set_image_mode(FALSE, FALSE);
    request_iap_disconnect();
    return FALSE;
}

#ifdef DOWNLOAD_PROCESS_THREAD
static void *download_process_results(void *data)
#else
gboolean download_process_results(gpointer data)
#endif
{
    struct request *request = NULL;
//    gchar *text = NULL;
//    float f = 0;
#ifndef DOWNLOAD_PROCESS_THREAD
    gboolean result = canRun;
#endif

    gboolean img = FALSE;
    
#ifdef DOWNLOAD_PROCESS_THREAD
    while ((request = g_async_queue_pop(results)) && canRun && 
           (request != dummyPointer)) 
#else
    if ((request = g_async_queue_try_pop(results)) && canRun)
#endif
    {
        ULOG_DEBUG("download_process_results: %s", request->source);
        GTimer *tmr = g_timer_new();
        ULOG_DEBUG("%%%%%%%%%% Begin Process Request %%%%%%%%%%\n");
        if (request->callback)
        {
        ULOG_DEBUG("1\n");
            if ((img = request->type_image))
            {
        ULOG_DEBUG("2\n");
                G_MUTEX_LOCK(queue_mutex);
        ULOG_DEBUG("3\n");
                g_hash_table_remove(download_hash, request->source);
        ULOG_DEBUG("4\n");
                if (request->data) {
        ULOG_DEBUG("5\n");
                    file_cache_add(app_data->app_ui_data->img_cache, 
                        request->source, NULL, request->data, request->size);
                    ULOG_DEBUG("Caching image: %s", request->source);
                }
        ULOG_DEBUG("6\n");
                G_MUTEX_UNLOCK(queue_mutex);
        ULOG_DEBUG("7\n");
                (request->callback) (request);
            } else
                (request->callback) (request);
        ULOG_DEBUG("8\n");
            
        }
        ULOG_DEBUG("{{{{{{{{{{{{{ Process Request Finished: %f }}}}}}}}}}}}\n",
            g_timer_elapsed(tmr, NULL));
        g_timer_destroy(tmr);

//        ULOG_DEBUG(">>>download_process_results: callback");

        if (request != NULL)
            download_request_free(request);
        
//        G_MUTEX_LOCK(queue_mutex);
        if (img) {
//            ULOG_DEBUG(">>>image");
            inc_counter(C_IMG_CURRENT);
        } else {
//            ULOG_DEBUG(">>>feed");
            inc_counter(C_RSS_CURRENT);
        }

/*        text = NULL;

        if (is_image_mode())
//            text = g_strdup_printf(_("rss_pb_toolbar_receiving"), current_img_download,
//                max_img_download);
            text = g_strdup_printf("Receiving %d/%d", current_img_download,
                max_img_download);
        else
            f = (float) current_download / max_download;*/
//        G_MUTEX_UNLOCK(queue_mutex);
        
/*        if (text) {
            ULOG_DEBUG(">>>image mode");

            gtk_progress_bar_set_text(GTK_PROGRESS_BAR
                    (app_data->app_ui_data->progress_bar), text);
            g_free(text);
        } else {
            ULOG_DEBUG(">>>feed mode");

            gtk_progress_bar_set_fraction(GTK_PROGRESS_BAR
                    (app_data->app_ui_data->progress_bar), f);
        }*/

        /* no need for mutex or worry about thread concurency, because
           only updates the status bar */
        if (!img)
            call_update_progressbar(NULL);
        else if (!g_idle_pb_id)
            g_idle_pb_id = g_idle_add(call_update_progressbar, NULL);
//        download_wait_for_pending_events();
        
        //need to check this function
        if (!img && !g_idle_id)
            g_idle_id = g_idle_add(call_ui_mainwindow_finish, NULL);

//        ULOG_DEBUG(">>>check for interrupt");
        if (app_data->app_ui_data->search_mode == SFM_INTERRUPTED) { 
            g_assert(app_data != NULL);
            g_assert(app_data->app_ui_data != NULL);        	

            download_cancel_all(TRUE);
        } else {
            gboolean refresh_finished = FALSE;
            
//            ULOG_DEBUG(">>>check for more elements");
//            G_MUTEX_LOCK(queue_mutex);

            gint remaining = get_counter(C_RSS_MAX) - get_counter(C_RSS_CURRENT);

            if (!is_image_mode()) {
                if (remaining <= 0) {
//                    ULOG_DEBUG(">>>no more feed to download");
                    reset_counter(C_RSS_MAX);
                    reset_counter(C_RSS_CURRENT);
                    //autoswitch to download images
                    if (get_counter(C_IMG_MAX)/*max_img_download*/)
                        set_image_mode(TRUE, FALSE);
                    else
                        refresh_finished = TRUE;
                }
/*                else
                    ULOG_DEBUG(">>>more data");*/
            } else 
                if (get_counter(C_IMG_MAX) - get_counter(C_IMG_CURRENT) <= 0)
                {
//                    ULOG_DEBUG(">>>no more image to download");
                    reset_counter(C_IMG_CURRENT);
                    reset_counter(C_IMG_MAX);
                    if (remaining > 0)
                        set_image_mode(FALSE, FALSE);
                    else
                        refresh_finished = TRUE;
                }
/*                else
                    ULOG_DEBUG(">>>more image");*/
//            G_MUTEX_UNLOCK(queue_mutex);

#ifndef DOWNLOAD_PROCESS_THREAD
            result = refresh_finished;
#endif
                
            if (refresh_finished) {
		ULOG_DEBUG("Planning to call refresh finished");
		g_idle_add(call_refresh_finished, NULL);
            }
        }
    }
#ifdef DOWNLOAD_PROCESS_THREAD
    return NULL;
#else
    if (!result)
        app_data->app_ui_data->download_dequeue_source = 0;
    return result;
#endif
}

/************************************************************************/
/*                        PUBLIC FUNCTIONS                              */
/************************************************************************/

void clear_image_user_data1(gpointer key, gpointer value, gpointer user_data)
{
    struct request *req = (struct request *)(((GList *)(value))->data);
    if (req->user_data) {
        //TODO maybe close the stream??
        req->user_data = NULL;
    }
}

void clear_image_user_data2(gpointer key, gpointer value, gpointer user_data)
{
    struct request *req = (struct request *)value;
    if (req->user_data) {
        //TODO maybe close the stream??
        req->user_data = NULL;
    }
}

void download_clear_image_user_data()
{
    G_MUTEX_LOCK(queue_mutex);
    g_hash_table_foreach(url_hash, clear_image_user_data1, NULL);
    g_hash_table_foreach(download_hash, clear_image_user_data2, NULL);
    G_MUTEX_UNLOCK(queue_mutex);
}

void download_process(struct request *request)
{
    FILE *f;
    int status = 0;
    GTimer *tmr = NULL;

    g_assert(request != NULL);
    
    request->data = NULL;

    if (request->source[0] == '|') {
        /* if the first char is a | we have a pipe else a file */
        request->size = 0;
        f = popen(&request->source[1], "r");
        if (NULL != f) {
            while (!feof(f)) {
                size_t len;
                request->data =
                    g_realloc(request->data, request->size + 1025);
                len = fread(&request->data[request->size], 1, 1024, f);
                if (len > 0)
                    request->size += len;
            }
            status = pclose(f);
            if (WIFEXITED(status) && WEXITSTATUS(status) == 0)
                request->httpstatus = 200;
            else
                request->httpstatus = 404;
            request->data[request->size] = '\0';
        } else {
            ULOG_DEBUG
                ("Error: Could not open pipe \"%s\"", &request->source[1]);
        }
    } else if  (( (strncasecmp(request->source, "http://", 7) == 0 ) || 
                ( strncasecmp(request->source, "https://", 8) == 0)  ) &&
	            (NULL == strstr(request->source, "obex://"))) {

        /*tvh: IMPORTANT here we need to process local files as well 
         Added more compareing to process http:// or https:// only 
            should not invoke netio stuff when processing file:///....
        */
//        ULOG_DEBUG("download_process: URL found, starting to process %s",request->source);
    ULOG_DEBUG("Source: %s", request->source);
    ULOG_DEBUG("Download process begin: %p", g_thread_self());
    tmr = g_timer_new();
        downloadlib_process_url(request);
    ULOG_DEBUG("Download process end: %p, elapsed: %f", g_thread_self(),
        g_timer_elapsed(tmr, NULL));
    g_timer_destroy(tmr);
//        ULOG_DEBUG("download_process: Received http status %d",request->httpstatus);
//        ULOG_DEBUG("download_process: After process_url, request->data: \n%s",request->data);
//        ULOG_DEBUG("download_process: for %s",request->source);
        
        if (request->httpstatus >= 400) {
//            ULOG_DEBUG("download_process: Received http status >= 400");
            
/*            if(request->data != NULL)
                ULOG_DEBUG("download_process: %s",request->data);*/
            
            g_free(request->data);
            request->data = NULL;
            request->size = 0;
        }
    } else if (NULL != strstr(request->source, "obex://")) {
	   /* g_file is not using prefix 'file://'*/ 
        gchar *file_source = NULL;

    	ULOG_DEBUG("OBEX!\n");
       GnomeVFSHandle *read_handle, *write_handle;
       GnomeVFSFileSize bytes_read, bytes_written;
       GnomeVFSResult vfs_result;
       gchar buffer[128*1024];
       file_source = g_strdup (GATEWAY_TMP_FILE+7);
       vfs_result = gnome_vfs_open (&read_handle, request->source, GNOME_VFS_OPEN_READ);
       if (vfs_result != GNOME_VFS_OK)
       {
	   ULOG_DEBUG("open not ok\n");
	   gnome_vfs_close(read_handle);
       gtk_infoprint(GTK_WINDOW(app_data->app_ui_data->app),
                        _("rss_ni_adding_failed"));
		return;
       }
	   ULOG_DEBUG("OBEX - File is opened! - %s\n", request->source);
       vfs_result = gnome_vfs_create (&write_handle, GATEWAY_TMP_FILE, GNOME_VFS_OPEN_WRITE, FALSE, 0777);
       if (vfs_result != GNOME_VFS_OK) {
	   gnome_vfs_close(read_handle);
	   gnome_vfs_close(write_handle);
	   ULOG_DEBUG("OBEX - create not ok\n");
	   return;
       }
	   ULOG_DEBUG("OBEX - File creation is OK! - %s\n", GATEWAY_TMP_FILE);
       while (gnome_vfs_read(read_handle, buffer, sizeof(buffer), 
		&bytes_read) == GNOME_VFS_OK)
       {
	       ULOG_DEBUG("%d\n", bytes_read);
           vfs_result = gnome_vfs_write (write_handle, buffer, bytes_read, &bytes_written);
           if (vfs_result != GNOME_VFS_OK) {
	       ULOG_DEBUG("write not ok\n");
               gnome_vfs_close(read_handle);
               gnome_vfs_close(write_handle);
			   
               gint gatewaypdf_handle=open(GATEWAY_TMP_FILE, O_RDONLY);
               if (gatewaypdf_handle != -1)
               {
                   close(gatewaypdf_handle);
                   remove(GATEWAY_TMP_FILE);
                 ULOG_DEBUG("OBEX - Download problem!!!!!!!\n");
               } else {
				   
                 ULOG_DEBUG("OBEX - Download is successful!\n");
				   
			   }
	       return;
	   }
	       ULOG_DEBUG("write not ok\n");
                 ULOG_DEBUG("OBEX - Download is successful!\n");
       }
       gnome_vfs_close(read_handle);
       gnome_vfs_close(write_handle);

	   
        request->source = g_strdup (GATEWAY_TMP_FILE);
        ULOG_DEBUG("download_process: processing FILES now for %s\n",file_source);

        if (g_file_test(file_source, G_FILE_TEST_EXISTS)) {
           /* we have a file... */
           if ((!g_file_get_contents
                 (file_source, &(request->data), &(request->size),
                  NULL)) || (request->data[0] == '\0')) {
               request->httpstatus = 403;
               ULOG_DEBUG
                ("Error: Could not open file \"%s\"\n", file_source);
           } else {
               g_assert(NULL != request->data);
               request->httpstatus = 200;
           }
           /* free this potential mem leak. */
           g_free(file_source);
        } else {
           ULOG_DEBUG("Error: There is no such file \"%s\"n",file_source);
           request->httpstatus = 404;
        }
    } else if (NULL != strstr(request->source, "file://")) {
       ULOG_DEBUG("file://...!\n");
        /* g_file is not using prefix 'file://'*/ 
        gchar *file_source = NULL;

        file_source = g_strdup (request->source + 7);
//        ULOG_DEBUG("download_process: processing FILES now for \nfile = %s\n",file_source);

        if (g_file_test(file_source, G_FILE_TEST_EXISTS)) {
           /* we have a file... */
           if ((!g_file_get_contents
                 (file_source, &(request->data), &(request->size),
                  NULL)) || (request->data[0] == '\0')) {
               request->httpstatus = 403;
               ULOG_DEBUG
                ("Error: Could not open file \"%s\"\n", request->source);
           } else {
               g_assert(NULL != request->data);
               request->httpstatus = 200;
           }
           /* free this potential mem leak. */
           g_free(file_source);
        } else {
           ULOG_DEBUG("Error: There is no such file \"%s\"n",request->source);
           request->httpstatus = 404;
        }
    } else {
        ULOG_DEBUG("Wrong source");
    }
    
    /* And execute the postfilter */
    if (request->data != NULL && request->filtercmd != NULL) {
        size_t size;
        gchar *tmp = filter(request->filtercmd, request->data, &size);
        if (tmp != NULL) {
            g_free(request->data);
            request->data = tmp;
            request->size = size;
        }
    }
}

gpointer download_request_new()
{
    struct request *request;

    debug_enter("update_request_new");

    request = g_new0(struct request, 1);

    debug_exit("update_request_new");

    return (gpointer) request;
}

void download_request_free(struct request *request)
{
//    ULOG_DEBUG("\n\n**********************\ndownload_request_free\n**********************\n\n");
    if (request == NULL)
        return;

    if (NULL != request) {
        if (NULL != (gchar * )request->source)
        {
//            ULOG_DEBUG("Freeing request: %s", (gchar *) request->source);
            g_free(request->source);
            request->source = NULL;
        }
        if (request->filtercmd)
        {
            g_free(request->filtercmd);
            request->filtercmd = NULL;
        }
        if (request->etag)
        {
            g_free(request->etag);
            request->etag = NULL;
        }
        if (request->data)
        {
            g_free(request->data);
            request->data = NULL;
        }
        if (request->feed)
           request->feed->request = NULL;
        /*FIXME: What about the other fields?*/

        /* IMPORTANT:Might be dangersous here to free some of this
        because we will destroy the feeds, the folder or even the ui
        if freeing them...Needs coming back
        */
    
        if (request->user_data)
           request->user_data = NULL;

        //TODO: need this?
        g_free(request);
        request = NULL;
    }
//    ULOG_DEBUG("--------------------------------------------");
}


void download_init(void)
{
    int i = 0;
    int count = 0;

    /* counters */
    timeout_count = rss_current_download = rss_max_download = 
        images_current_download = images_max_download = 0;

    downloadlib_init();

    requests_high_prio = g_async_queue_new();
    results = g_async_queue_new();

    requests_normal_prio = g_queue_new();
    requests_image = g_queue_new();

    download_hash = g_hash_table_new(g_str_hash, g_str_equal);
    url_hash = g_hash_table_new(g_str_hash, g_str_equal);

    offline_cond = g_cond_new();
    cond_mutex = g_mutex_new();

    queue_mutex = g_mutex_new();
    queue_cond = g_cond_new();

    rss_threads = DEFAULT_UPDATE_THREAD_CONCURRENCY;

    count = rss_threads;

    gboolean priority_high = TRUE;
    gboolean normal_priority = FALSE;

    for (i = 0; i < count; i++)
    {
        if (i == 0)
            update_threads[i] =
                g_thread_create(download_thread_main, (void *) priority_high, TRUE,
                    NULL);
        else 
            update_threads[i] =
                g_thread_create(download_thread_main, (void *) normal_priority, TRUE,
                    NULL);
    }

#ifdef DOWNLOAD_PROCESS_THREAD
    update_threads[i] = 
        g_thread_create(download_process_results, NULL, TRUE, NULL);
#endif
}

void download_done(void)
{
    gint i;
    
    ULOG_DEBUG("Download_done");
    canRun = FALSE;

    //to be sure the first thread will get the signal
    for (i = 0; i <= DEFAULT_UPDATE_THREAD_CONCURRENCY; i++)
        g_async_queue_push(requests_high_prio, dummyPointer);
#ifdef DOWNLOAD_PROCESS_THREAD
    g_async_queue_push(results, dummyPointer);
#endif
    
    g_cond_broadcast(offline_cond);
    g_cond_broadcast(queue_cond);
    
    ULOG_DEBUG("Download_done... joining threads");
#ifdef DOWNLOAD_PROCESS_THREAD
    for (i = 0; i <= DEFAULT_UPDATE_THREAD_CONCURRENCY; i++) {
#else
    for (i = 0; i < DEFAULT_UPDATE_THREAD_CONCURRENCY; i++) {
#endif
        ULOG_DEBUG(">>Join thread %p", update_threads[i]);
        g_thread_join(update_threads[i]);
        ULOG_DEBUG("<<Thread %p joined", update_threads[i]);
    }
    ULOG_DEBUG("Download_done... joining threads done");
    
    g_async_queue_unref(requests_high_prio); requests_high_prio = NULL;
    g_async_queue_unref(results); requests_high_prio = NULL;

    g_queue_free(requests_normal_prio); requests_normal_prio = NULL;
    g_queue_free(requests_image); requests_image = NULL;

    g_hash_table_destroy(download_hash); download_hash = NULL;
    g_hash_table_destroy(url_hash); url_hash = NULL;

    g_cond_free(offline_cond);
    g_mutex_free(cond_mutex);

    g_mutex_free(queue_mutex);
    g_cond_free(queue_cond);
}

/* Push a download request to the wait queue depending from is priority.
   Should not be used for images! */
void download_queue(struct request *new_request)
{
    g_assert(NULL != new_request);
    g_assert(app_data != NULL);
    g_assert(app_data->app_ui_data != NULL);

    if (!is_image_mode())
        switch_progressbar(PROGRESSBAR_REFRESH_MODE);
//        show_stop_refresh_button(FALSE);
    
//    G_MUTEX_LOCK(queue_mutex);
    inc_counter(C_RSS_MAX);
//    G_MUTEX_UNLOCK(queue_mutex);
    app_data->app_ui_data->pending_requests_nr++;

    ULOG_DEBUG("===== New request: %s\n", new_request->source);
    if (new_request->priority == 1) {
        ULOG_DEBUG("\n\n\n\n\n\nHigh priority request!!!!!!!!!!!!!\n\n\n\n\n\n");
        g_async_queue_push(requests_high_prio, new_request);
    } else {
        G_MUTEX_LOCK(queue_mutex);
        g_queue_push_tail(requests_normal_prio, new_request);
        if (!is_image_mode())
            g_cond_signal(queue_cond);
        G_MUTEX_UNLOCK(queue_mutex);
    }
}

void image_downloaded_result(struct request *requestdata)
{
    GtkHTMLStream *stream=requestdata->user_data;
    ULOG_DEBUG("gtkhtml: Downloaded image: %s",
                requestdata->source);
    if (requestdata->data)
    {
        if (stream) {
            GDK_THREADS_ENTERD;
            if (requestdata->load_image) {
                gchar *tmp = app_data->app_ui_data->load_url;
                app_data->app_ui_data->load_url = NULL;
                g_free(tmp);
            }
/*            gtk_html_stream_write (stream, requestdata->data, requestdata->size);
            gtk_html_stream_close(stream, 
                requestdata->size == -1
                ? GTK_HTML_STREAM_ERROR
                : GTK_HTML_STREAM_OK);*/
            write_image((gpointer)stream, requestdata->data, requestdata->size);
            GDK_THREADS_LEAVED;
        }
    } else 
        if (stream)
        {
            GDK_THREADS_ENTERD;
            if (requestdata->load_image /*&& 
                    requestdata->returncode == NET_ERR_HTTP_404*/) {
                gtk_infoprint(GTK_WINDOW(app_data->app_ui_data->app), _("rss_ni_unable_load_image"));
                gchar *tmp = app_data->app_ui_data->load_url;
                app_data->app_ui_data->load_url = NULL;
                g_free(tmp);
            }
            gtk_html_stream_write (stream, "123", -1);
            gtk_html_stream_close(stream, GTK_HTML_STREAM_OK);
            GDK_THREADS_LEAVED;
        }

    return;
}

gint queue_cmp(gconstpointer a, gconstpointer b, gpointer user_data)
{
    if (((struct request *)b)->load_image) 
        return 1;
        
    return (((struct request *)a)->user_data ? -1 : 0);
}

void print_queue(GQueue *q)
{
    ULOG_DEBUG("Printing queue: ");
    GList *l = q->head;
    while (l) {
        ULOG_DEBUG("  %s, %s", ((struct request *)l->data)->source,
            ((struct request *)l->data)->user_data ? "TRUE" : "-");
        l = l->next;
    }
}

GList *g_queue_push_sorted(GQueue *q, gpointer data)
{
    g_queue_insert_sorted(q, data, queue_cmp, NULL);
    GList *result = q->head;
    while (result->data != data)
        result = result->next;
    return result;
}

/* rewritten glib functions, to insert a link sorted into a queue */
GList *g_list_insert_before_link (GList *list, GList *sibling, GList *node)
{
    if (!list)
        return node;
    else {
        node->prev = sibling->prev;
        node->next = sibling;
        sibling->prev = node;
        if (node->prev)
        {
            node->prev->next = node;
            return list;
        }
        else
            return node;
    }
}

void g_queue_push_link_sorted(GQueue *queue, GList *link)
{
    GList *list;
    g_return_if_fail (queue != NULL);

    list = queue->head;
    while (list && queue_cmp(list->data, link->data, NULL) < 0)
        list = list->next;

    if (list) {
        queue->head = g_list_insert_before_link(queue->head, list, link);
        queue->length++;
    } else
        g_queue_push_tail_link(queue, link);
}

/* Push a download image request to the wait queue. */
void download_queue_image(const gchar *url, gpointer data, gboolean load_image)
{
    g_assert(url != NULL);
    g_assert(app_data != NULL);
    g_assert(app_data->app_ui_data != NULL);
    
    G_MUTEX_LOCK(queue_mutex);
    
    //check if it is not already in the cache ... in some rare cases it will because
    //there ...
    size_t size;
    gpointer cdata = file_cache_find(app_data->app_ui_data->img_cache,
        url, &size,TRUE);
    if (cdata) {
        //ULOG_DEBUG("#####Found in cache %s: ", url);
        
        if (data) {
            GDK_THREADS_ENTERD;
/*            gtk_html_stream_write ((GtkHTMLStream *)data, cdata, size);
            gtk_html_stream_close((GtkHTMLStream *)data, GTK_HTML_STREAM_OK);*/
            write_image(data, cdata, size);
            GDK_THREADS_LEAVED;
        }

        g_free(cdata);
    } else {
        GList *l = NULL;
        struct request *r = NULL;
        if ((l = (GList *)(g_hash_table_lookup(url_hash, url)))) {
            //if the request is already in the download queue move it to the front
            //only if data is not null
            r = (struct request *)l->data;
            r->user_data = data;
            if (data) {
                g_queue_unlink(requests_image, l);
                g_queue_push_link_sorted(requests_image, l);
            }
        } else if ((r = (struct request *)g_hash_table_lookup(download_hash, url))) {
            r->user_data = data;
        } else {
            //not found in the queue, so add it ...

            inc_counter(C_IMG_MAX);

            r = download_request_new();
            r->source = g_strdup(url);
            r->callback = image_downloaded_result;
            r->type_image = TRUE;
            r->user_data = data;
            r->load_image = load_image;
    
            if (data) {
                ULOG_DEBUG("Inserting %s: ", r->source);
                g_hash_table_insert(url_hash, r->source,
                    g_queue_push_sorted(requests_image, r));
 //               print_queue(requests_image);
            } else {
                g_queue_push_tail(requests_image, r);
                g_hash_table_insert(url_hash, r->source,
                    g_queue_peek_tail_link(requests_image));
            }
            
            if (is_image_mode())
                g_cond_signal(queue_cond);
/*            else if (max_download - current_download == 0 && 
                     app_data->app_ui_data->network_iap)
                set_image_mode(TRUE, TRUE);*/
        }
    }
    
    G_MUTEX_UNLOCK(queue_mutex);
}

void download_set_online(gboolean mode)
{
#ifdef USE_BLOCKING_OFFLINE
   if ((online = mode)) {

        ULOG_DEBUG("download_set_online requested");
#endif

        if (app_data && app_data->app_ui_data &&
            app_data->app_ui_data->search_mode != SFM_REFRESH) {

           G_MUTEX_LOCK(queue_mutex);
           if (get_counter(C_RSS_MAX) > 0)
               set_image_mode(FALSE, TRUE);
           else if (get_counter(C_IMG_MAX) > 0)
               set_image_mode(TRUE, TRUE);
           G_MUTEX_UNLOCK(queue_mutex);
       }
       
#ifdef USE_BLOCKING_OFFLINE
       g_mutex_lock(cond_mutex);
       g_cond_broadcast(offline_cond);
       g_mutex_unlock(cond_mutex);
    }
#endif
}

gboolean download_is_online(void)
{
#ifdef USE_BLOCKING_OFFLINE
    return online;
#else
    return TRUE;
#endif
}

void download_empty_aqueue(GAsyncQueue * queue)
{
    gpointer data = NULL;
    g_assert(queue != NULL);

    while ((data = g_async_queue_try_pop(queue)))
        download_request_free((struct request *) data);
}

void download_empty_queue(GQueue * queue)
{
    gpointer data = NULL;
    struct request *req = NULL;
    g_assert(queue != NULL);

    while ((data = g_queue_pop_head(queue))) {
        req = (struct request *) data;
        /* image request will be freed by gtkhtml model*/

        if (req->type_image == TRUE && req->user_data)
            req->callback (req);
        download_request_free(req);
    }
}

gboolean remove_all(gpointer key, gpointer value, gpointer user_data)
{
    return TRUE;
}

void download_cancel_all(gboolean show_banner)
{
    static gboolean canceling = FALSE;

    cancel_dl_timeout();
    
    if (!canceling)
    {
        canceling = TRUE;

        app_data->app_ui_data->search_mode = SFM_INTERRUPTED;

        ULOG_DEBUG("________download_cancel_all_________");
        G_MUTEX_LOCK(queue_mutex);

        gtkhtml_load_images(FALSE);

        downloadlib_cancel_all();
        
        g_hash_table_foreach_remove(url_hash, remove_all, NULL);
        g_hash_table_foreach_remove(download_hash, remove_all, (gpointer)1);
        
        download_empty_aqueue(requests_high_prio);
        download_empty_aqueue(results);
        
        download_empty_queue(requests_normal_prio);
        download_empty_queue(requests_image);

        reset_counter(C_RSS_CURRENT);
        reset_counter(C_RSS_MAX);
        reset_counter(C_IMG_CURRENT);
        reset_counter(C_IMG_MAX);
    
        G_MUTEX_UNLOCK(queue_mutex);
    
        set_image_mode_simple(FALSE);
    
        if (app_data->app_ui_data->feed_directory_loading) {
            app_data->app_ui_data->feed_directory_loading = FALSE;
    
            on_refresh_finished(show_banner, TRUE);
        } else {
            on_refresh_finished(app_data->app_ui_data->start_on_background && show_banner, 
                FALSE);
    /*        if(app_data->app_ui_data->network_disconnected)
                on_refresh_finished(TRUE, FALSE);
            else
                on_refresh_finished(FALSE, FALSE);
            app_data->app_ui_data->network_disconnected = FALSE;*/
        }
    
        ULOG_DEBUG(">>>>>>>>>>download_cancel_all done<<<<<<<<<<<<<");
        canceling = FALSE;
    }
}

gboolean is_image_mode()
{
    gboolean result;
    G_LOCK(image_mode_mutex);
    result = image_mode;
    G_UNLOCK(image_mode_mutex);
    return result;
}

gboolean set_image_mode_simple(gboolean is_img)
{
    G_LOCK(image_mode_mutex);
    gboolean not_eq = (image_mode != is_img);
    image_mode = is_img;
    G_UNLOCK(image_mode_mutex);
    return not_eq;
}

void set_image_mode(gboolean is_img, gboolean mutex_locked)
{
    ULOG_DEBUG("set_image_mode: feed(%d, %d), img(%d, %d)",
        get_counter(C_RSS_CURRENT), 
        get_counter(C_RSS_MAX),
        get_counter(C_IMG_CURRENT),
        get_counter(C_IMG_MAX));
    ULOG_DEBUG(">>>>>>requesting changing mode");
    if (set_image_mode_simple(is_img) || mutex_locked)
    {
/*        if (!mutex_locked)
            G_MUTEX_LOCK(queue_mutex);*/
        //We are switching to download images gathered in the queue
        GDK_THREADS_ENTERD;
        if (is_img) {
            if (get_counter(C_IMG_MAX))
                switch_progressbar(PROGRESSBAR_IMAGE_MODE);
        } else {
            if (get_counter(C_RSS_MAX))
                switch_progressbar(PROGRESSBAR_REFRESH_MODE);

            update_progress_bar();
        }
        
        GDK_THREADS_LEAVED;

/*        if (!mutex_locked)
            G_MUTEX_UNLOCK(queue_mutex);*/
        g_cond_broadcast(queue_cond);
    }
    
    if (app_data->app_ui_data->nospace == TRUE)
    {
      
      gdk_threads_enter();
      
      ui_show_save_nodevicemem_dialog();
      
      gdk_threads_leave();
      
      app_data->app_ui_data->nospace = FALSE;
      
    }
    
}

gboolean is_image_queue_empty()
{
//    gboolean res = FALSE;
//    G_MUTEX_LOCK(queue_mutex);
    ULOG_DEBUG("is_image_queue_empty: feed(%d, %d), img(%d, %d)",
        get_counter(C_RSS_CURRENT), 
        get_counter(C_RSS_MAX),
        get_counter(C_IMG_CURRENT),
        get_counter(C_IMG_MAX));
//    res = get_counter(C_IMG_MAX);
//    G_MUTEX_UNLOCK(queue_mutex);
    return get_counter(C_IMG_MAX) <= 0;
}

gboolean results_empty()
{
    return g_async_queue_try_pop(results) == NULL;
}
