/**
 * @file ui_feedlist.h GUI feed list handling
 * 
 * Copyright (C) 2004 Lars Lindner <lars.lindner@gmx.net>
 * Copyright (C) 2004 Nathan J. Conrad <t98502@users.sourceforge.net>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA 
 */

#ifndef _UI_FEEDLIST_H
#define _UI_FEEDLIST_H

#include <gtk/gtk.h>
#include "feed.h"
#include "folder.h"
#include "appdata.h"

/* constants for attributes in feedstore */
enum {
    FS_LABEL,			/* Displayed name */
    FS_ICON,			/* Icon to use */
    FS_PTR,			/* pointer to the folder or feed */
    FS_UNREAD,			/* Number of unread items */
    FS_LEN
};

typedef struct ui_data {
    GtkTreeIter row;
} ui_data;

extern GtkTreeStore *feedstore;
extern GtkTreeModel *feedmodel;

/* Add/remove/update nodes */

/**
 * Updates all childrens of the given GtkTreeIter
 *
 * @param iter	a folder of the feed list
 */
void ui_feedlist_update_iter(GtkTreeIter * iter);

/**
 * Update the labels of all of the nodes of the feedlist, and update
 * their GtkTreeIter pointers.
 */
#define ui_feedlist_update() (ui_feedlist_update_iter(NULL))

/* Selections */

/** Gets the selected node from the feed list
  *
  * @return the selected node
  */
nodePtr ui_feedlist_get_selected();

/** Gets the selected node from the given tree
  *
  * @param treeview the tree to get the selected node from
  * @return the selected node
  */
nodePtr ui_feedlist_get_selected_from_tree(GtkTreeView *);

/** Selects a feed from the feed list
  *
  * @param the feed to select
  */
void ui_feedlist_select(nodePtr fp);
gboolean ui_feedlist_check_if_folder_empty(folderPtr folder);
typedef void (*nodeActionFunc) (nodePtr fp);
typedef void (*nodeActionDataFunc) (nodePtr fp, gpointer user_data);
typedef gboolean(*nodeActionFuncWithResult) (nodePtr fp);
typedef gboolean(*nodeActionDataFuncWithResult) (nodePtr fp,
						 gpointer user_data);

/**
 * Returns the feed store, creating it if needed.
 *
 * @return feed store pointer.
 */
GtkTreeStore *getFeedStore(void);

/**
 * Initializes the feed list. For example, it creates the various
 * columns and renderers needed to show the list.
 */
void ui_feedlist_init(GtkWidget * mainview);

/** 
 * Determines the currently selected feed list iterator.
 *
 * @param iter	pointer to iter structure to return selected iter
 */
gboolean ui_feedlist_get_iter(GtkTreeIter * iter);


/**
 * Returns the path for the folder which should be selected after deleting a folder
 * in the manage folders dialog.
 * 
 * @returns the folder to be selected.
 */
GtkTreePath *ui_feedlist_get_previous_after_delete_folder(GtkTreeView *treeview);


/**
 * Returns the parent folder of a node, or NULL if the root folder is
 * the parent.
 * 
 * @returns the parent folder.
 */

folderPtr ui_feedlist_get_parent(nodePtr ptr);

/** Selects the proper destination for a new feed based on which feed
 *  is currently selected.
 *
 *  @returns folder into which the feed should be inserted
 */
folderPtr ui_feedlist_get_target_folder();

/** Selects the proper destination for a new feed based on which feed
 *  is currently selected, starting from ptr.
 *
 *  @param ptr start point
 *  @returns folder into which the feed should be inserted
 */
folderPtr ui_feedlist_get_target_folder_ptr(nodePtr ptr);

/** Finds a folder with matching feed URI, usually from applet.
 *
 *  @param from tree iterator to start recursive search from
 *  @param feed points to gchar string of the feed URI to find
 *
 *  @returns feed which contains requested URI
 */
nodePtr ui_feedlist_find_feed(GtkTreeIter * from, gchar * feed);

/** Goes through all the feeds and collects the statistics information
  * that is displayed when the news reader is started
  *
  * @param from an iterator for the current tree node
  * @param nr_feeds the number of feeds
  * @param nr_unread_feeds the number of unread feeds
  * @param nr_marked the number of feeds marked saved for later
  * @param last_poll_time the last refresh time of the feeds
  */
void ui_feedlist_find_mainwindow_data(GtkTreeIter * from, gint * nr_feeds,
				      gint * nr_unread,
				      gint * nr_unread_feeds,
				      gint * nr_marked,
				      glong * last_poll_time);

/** Displays a feed, scrolling the feed view to show requested item.
 *  @param feed gchar string of the feed URI
 *  @param nr feed news item number
 *
 *  @returns whether such feed and item pair was found
 */
gboolean ui_feedlist_display_news(gchar * feed, guint32 nr);

/**
 * Create a new subscription in the currently selected folder.
 *
 * @param source	feed source URL or local file name or piped command
 * @param filter    filename of the filter to use for the new feed
 * @param flags feed request flags to pass to the update requesting subsystem
 */
void ui_feedlist_new_subscription(gchar * source,
				  const gchar * filter, gint flags,
				  folderPtr selected_parent);

/**
 * Create a new subscription in the currently selected folder for feed directory.
 * The implementation differs in that the feeds are only updated when an
 * internet connection exists.
 *
 * @param source	feed source URL or local file name or piped command
 * @param filter    filename of the filter to use for the new feed
 * @param flags feed request flags to pass to the update requesting subsystem
 * @param refresh if the feed should be refreshed
 * @param update_feedlist if the feedlist should be updated
 */				  
void ui_feedlist_new_subscription_for_fd(const gchar * source,
				         const gchar * filter, gint flags,
				         folderPtr selected_parent, gboolean refresh, gboolean update_feedlist);				  

#define	ACTION_FILTER_FEED	1	/** Only matches nodes where IS_FEED(node->type) */
#define	ACTION_FILTER_DIRECTORY	2	/** Only matches nodes where IS_DIRECTORY(node->type) */
#define	ACTION_FILTER_FOLDER	4	/** Only matches nodes where IS_FOLDER(node->type) */
#define	ACTION_FILTER_ROOT	8	/** Only matches the root node */
#define	ACTION_FILTER_ANY	0x0F	/** Matches any node */
#define	ACTION_FILTER_CHILDREN	0x10	/** Matches immediate children of the given node */

/**
 * Helper function to recursivly call feed_save() for all
 * elements of the given type in the feed list.
 *
 * @param ptr	node pointer whose children should be processed (NULL defaults to root)
 * @param filter specifies the types of nodes for which func should be called
 * @param func	the function to process all found elements
 * @param params Set to 1 if there will be user_data. Set to 0 for no user data
 * @param user_data specifies the second argument that func should be passed
 */
void ui_feedlist_do_for_all_full(nodePtr ptr, gint filter,
				 gpointer func,
				 gint params, gpointer user_data);

/**
 * Helper function to recursivly call feed_save() for all
 * elements of the given type in the feed list.
 *
 * @param ptr	node pointer whose children should be processed (NULL defaults to root)
 * @param filter specifies the types of nodes for which func should be called
 * @param func	the function to process all found elements
 */
#define ui_feedlist_do_for_all(ptr, filter, func) \
        ui_feedlist_do_for_all_full(ptr,filter,func,0,NULL)

/**
 * Helper function to recursivly call feed_save() for all
 * elements of the given type in the feed list.
 *
 * @param ptr	node pointer whose children should be processed (NULL defaults to root)
 * @param filter specifies the types of nodes for which func should be called
 * @param func	the function to process all found elements
 * @param user_data specifies the second argument that func should be passed
 */
#define ui_feedlist_do_for_all_data(ptr, filter, func, user_data) \
        ui_feedlist_do_for_all_full(ptr,filter,func,1,user_data)

/**
 * Helper function to recursively call a function for all
 * elements of the given type in the feed list.
 * Traversing the tree is interrupted if the function returns FALSE.
 *
 * @param ptr	node pointer whose children should be processed (NULL defaults to root)
 * @param filter specifies the types of nodes for which func should be called
 * @param func	the function to process all found elements
 * @param params Set to 1 if there will be user_data. Set to 0 for no user data
 * @param user_data specifies the second argument that func should be passed
 * @param TRUE if all nodes were traversed,
 *        FALSE if traversing was interrupted
 */
gboolean ui_feedlist_do_for_all_full_with_int(nodePtr ptr, gint filter,
					      gpointer func, gint params,
					      gpointer user_data);

/**
 * Helper function to recursively call a function for all
 * elements of the given type in the feed list.
 * Traversing the tree is interrupted if the function returns FALSE.
 *
 * @param ptr	node pointer whose children should be processed (NULL defaults to root)
 * @param filter specifies the types of nodes for which func should be called
 * @param func	the function to process all found elements
 */
#define ui_feedlist_do_for_all_with_int(ptr, filter, func) \
        ui_feedlist_do_for_all_full_with_int(ptr,filter,func,0,NULL)

/**
 * Helper function to recursivly call feed_save() for all
 * elements of the given type in the feed list.
 *
 * @param ptr	node pointer whose children should be processed (NULL defaults to root)
 * @param filter specifies the types of nodes for which func should be called
 * @param func	the function to process all found elements
 * @param user_data specifies the second argument that func should be passed
 */
#define ui_feedlist_do_for_all_data_with_int(ptr, filter, func, user_data) \
        ui_feedlist_do_for_all_full_with_int(ptr,filter,func,1,user_data)

/** 
 * Tries to find the first unread feed in the given folder.
 * 
 * @return feed pointer or NULL
 */
feedPtr ui_feedlist_find_unread_feed(nodePtr folder);

/**
 * marks all items of the feed of the given tree iter as read 
 *
 * @param iter	an iterator of the feed list to be processed
 */
void ui_feedlist_mark_items_as_unread(GtkTreeIter * iter);

/**
 * timeout callback to trigger the auto update handling
 * for the feed list
 */
gboolean ui_feedlist_auto_update(void *data);

/**
 * timeout callback to trigger the new subscription fifo
 * checking 
 */
gboolean ui_feedlist_check_subscription_fifo(void *data);

/**
 * Add a node to the feedlist
 *
 * @param parent	the parent of the new folder, or NULL to 
 *			insert in the root folder
 * @param node		the node to add
 * @param position	the position in which the folder should be 
 *			added, or -1 to append the folder to the parent.
 */
void ui_feedlist_add(folderPtr parent, nodePtr node, gint position);

/** Adds a new folder to the root of the feed list
  *
  * @param node the new root folder
  */
void ui_feedlist_add_new_folder_to_root(nodePtr node);

/** Removes a feed from the feed list
  *
  * @param node the feed to remove
  */
void ui_feedlist_remove(nodePtr node);

/**
 * Prompt the user for confirmation of a folder or feed, and
 * recursively remove the feed or folder if the user accepts. This
 * function blocks does not block, so the folder/feeds will not have
 * been deleted when this function returns.
 *
 * @param ptr the node to delete
 * @return TRUE if OK is clicked, FALSE otherwise
 */
gboolean ui_feedlist_delete(nodePtr ptr);

/** Deletes a folder from the feed list
  *
  * @param folder the folder to delete
  */
void ui_feedlist_delete_folder(folderPtr folder);

/**
 * Load and display in itemlist the feed currently selected by treeview.
 */
void ui_feedlist_load_selected(nodePtr ptr);

/** 
 * @name menu and dialog callbacks 
 * @{
 */

/** Opens the feed properties dialog
  *
  * @param callback_data the feed to open the properties dialog for
  * @param callback_action not used
  * @param widget not used
  */
void on_popup_prop_selected(gpointer callback_data,
			    guint callback_action, GtkWidget * widget);


/** Opens the new feed dialog
  *
  * @param button not used
  * @param user_data not used
  */			    
void on_newbtn_clicked_cb(GtkButton * button, gpointer user_data);

/** Opens the new feed dialog
  *
  * @param button not used
  * @param user_data not used
  * @param source: the source to be put in the address entry when displaying
  */

void on_newbtn_clicked(AppData * appdata, const gchar * fd_source);


/* UI folder stuff */

/** Gets the root folder
  *
  * @return the root folder
  */
folderPtr ui_feedlist_get_root_folder(void);

/** Sets the root folder to NULL
  */
void ui_feedlist_reset_root_folder(void);

/**
 * Search GtkTreeModel by a pointer to a node.
 *
 * @param model GtkTreeModel (not NULL)
 * @param node node (not NULL)
 * @param path result path (or NULL)
 * @param iter result iter (or NULL)
 * @return TRUE if node found, FALSE if not
 */
gboolean ui_feedlist_search_model_by_ptr(GtkTreeModel * model,
					 nodePtr node,
					 GtkTreePath ** path,
					 GtkTreeIter * iter);


/**
 * Convert feed store iterator to an iterator for the model
 * of the feed view.
 *
 * @param iter_target Result iterator.
 *   Pointer to an (possibly uninitialized) GtkTreeIter. Not NULL.
 * @param iter_source Source iterator.
 *   Pointer to the GtkTreeIter to be converted. Not NULL.
 */
void ui_feedlist_convert_store_iter_to_visible_iter(GtkTreeIter *
						    iter_target,
						    GtkTreeIter *
						    iter_source);


/**
 * Convert an iterator for the model of the feed view
 * to an iterator for the feed store.
 *
 * @param iter_target Result iterator.
 *   Pointer to an (possibly uninitialized) GtkTreeIter. Not NULL.
 * @param iter_source Source iterator.
 *   Pointer to the GtkTreeIter to be converted. Not NULL.
 */
void ui_feedlist_convert_visible_iter_to_store_iter(GtkTreeIter *
						    iter_target,
						    GtkTreeIter *
						    iter_source);

/**
 * Count all the feeds in a folder.
 * The subfolders are included recursively in the search.
 *
 * @param folder folder (not NULL)
 * @return number of feeds in the folder
 */
gint ui_feedlist_count_feeds(folderPtr folder);


/**
 * Check that the folder can be deleted,
 * confirm deletion if necessary and delete the folder.
 *
 * @param folder folder to be deleted (not NULL)
 * @return TRUE if the folder was deleted, FALSE if not
 */
gboolean ui_feedlist_confirm_and_delete_folder(folderPtr folder);

/** Set the timer for automatic updates
  *
  * @param constant_interval if the last refresh time should be subtracted from the chosen interval
  */
/* TODO: unused ATM.
void set_update_timer(gboolean constant_interval);
*/

/** Updates the number of unread items and the last update time, does
  * a general save and signals the applet 
  */
void ui_feedlist_update_applet();

/** Checks if the given folder is an ancestor of the node supplied as the first
  * parameter
  *
  * @param ptr the node to check for the parent
  * @param fop the folder
  * @return TRUE if the folder is an ancestor
  */
gboolean ui_feedlist_is_ancestor(nodePtr ptr, GtkTreeIter *folder);

/** Deletes read items from all feeds if the time for automatic
  * deletion has passed
  *
  * @param from an iterator for traversing the tree
  */
void ui_feedlist_delete_read_items(GtkTreeIter *from);

/** Deletes a feed or folder and its children in the feed list
  *
  * @param ptr the node to delete
  */
void ui_feedlist_delete_(nodePtr ptr);

/** Checks if a node can be added to the current folder, 
  * if the parent is not factory folder.
  *
  * @param show_info to print the info message
  */
gboolean ui_feedlist_check_factory_add(gboolean show_info);

/** Checks if a node can be deleted (it is not factory node)
  *
  * @param show_info to print the info message
  */
gboolean ui_feedlist_check_factory_delete(gboolean show_info);

void ui_feedlist_enable_def_buttons();

#endif
