/**
 * @file ui_feed.c	UI actions concerning a single feed
 *
 * Copyright (C) 2004 Lars Lindner <lars.lindner@gmx.net>
 * Copyright (C) 2004 Nathan J. Conrad <t98502@users.sourceforge.net>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */


#include <gtk/gtk.h>
#include <libxml/uri.h>
#include <hildon-widgets/gtk-infoprint.h>
#include <strings.h>
#include "debug.h"
#include "support.h"
#include "feed.h"
#include "conf.h"
#include "callbacks.h"
#include "update.h"
#include "htmlview.h"
#include "interface.h"
#include "ui_feed.h"
#include "ui_queue.h"
#include "favicon.h"
#include "ui_notification.h"
#include "ui_feed_directory.h"

#include <osso-log.h>

#define MAX_DELETE_REQ 5

extern guint rss_max_feed;
extern AppData *app_data;


static void on_newdialog_response(GtkDialog * dialog, gint response_id,
				  gpointer user_data);
static void on_propdialog_response(GtkDialog * dialog, gint response_id,
				   gpointer user_data);
static gboolean ui_feed_newdialog_can_close(struct fp_prop_ui_data *ui_data, 
    gboolean msg, gchar **result);
gboolean feed_prop_can_close(struct fp_prop_ui_data *ui_data, gboolean show,
    gchar **ptitle, gchar **psource);

/************************************************************************/
/*                        PRIVATE FUNCTIONS                             */
/************************************************************************/				  
				  
/**
 * Creates a new error description according to the passed
 * HTTP status and the feeds parser errors. If the HTTP
 * status is a success status and no parser errors occured
 * no error messages is created. The created error message 
 * can be queried with feed_get_error_description().
 *
 * @param fp		feed
 * @param httpstatus	HTTP status
 * @param resultcode the update code's return code (see update.h)
 */
static void ui_feed_set_error_description(feedPtr fp, gint httpstatus,
					  gint resultcode)
{
    gchar *tmp1 = NULL, *tmp2 = NULL, *buffer = NULL;
    gboolean errorFound = FALSE;

    g_assert(NULL != fp);
    g_free(fp->errorDescription);
    fp->errorDescription = NULL;

    if (((httpstatus >= 200) && (httpstatus < 400)) &&	/* HTTP codes starting with 2 and 3 mean no error */
	(NULL == fp->parseErrors))
	return;

    if ((200 != httpstatus) || (resultcode != NET_ERR_OK)) {
	/* first specific codes */
	switch (httpstatus) {
	case 401:
	    tmp2 =
		g_strdup
		("You are unauthorized to download this feed. Please update your username and "
		 "password in the feed properties dialog box.");
	    break;
	case 402:
	    tmp2 = g_strdup("Payment Required");
	    break;
	case 403:
	    tmp2 = g_strdup("Access Forbidden");
	    break;
	case 404:
	    tmp2 = g_strdup("Resource Not Found");
	    break;
	case 405:
	    tmp2 = g_strdup("Method Not Allowed");
	    break;
	case 406:
	    tmp2 = g_strdup("Not Acceptable");
	    break;
	case 407:
	    tmp2 = g_strdup("Proxy Authentication Required");
	    break;
	case 408:
	    tmp2 = g_strdup("Request Time-Out");
	    break;
	case 410:
	    tmp2 =
		g_strdup
		("Gone. Resource doesn't exist. Please unsubscribe!");
	    break;
	}
	/* Then, netio errors */
	if (tmp2 == NULL) {
	    switch (resultcode) {
	    case NET_ERR_URL_INVALID:
		tmp2 = g_strdup("URL is invalid");
		break;
	    case NET_ERR_UNKNOWN:
	    case NET_ERR_CONN_FAILED:
	    case NET_ERR_SOCK_ERR:
		tmp2 = g_strdup("Error connecting to remote host");
		break;
	    case NET_ERR_HOST_NOT_FOUND:
		tmp2 = g_strdup("Hostname could not be found");
		break;
	    case NET_ERR_CONN_REFUSED:
		tmp2 =
		    g_strdup
		    ("Network connection was refused by the remote host");
		break;
	    case NET_ERR_TIMEOUT:
		tmp2 = g_strdup("Remote host did not finish sending data");
		break;
		/* Transfer errors */
	    case NET_ERR_REDIRECT_COUNT_ERR:
		tmp2 =
		    g_strdup("Too many HTTP redirects were encountered");
		break;
	    case NET_ERR_REDIRECT_ERR:
	    case NET_ERR_HTTP_PROTO_ERR:
	    case NET_ERR_GZIP_ERR:
		tmp2 = g_strdup("Remote host sent an invalid response");
		break;
		/* These are handled above      
		   case NET_ERR_HTTP_410:
		   case NET_ERR_HTTP_404:
		   case NET_ERR_HTTP_NON_200:
		 */
	    case NET_ERR_AUTH_FAILED:
	    case NET_ERR_AUTH_NO_AUTHINFO:
		tmp2 = g_strdup("Authentication failed");
		break;
	    case NET_ERR_AUTH_GEN_AUTH_ERR:
	    case NET_ERR_AUTH_UNSUPPORTED:
		tmp2 =
		    g_strdup
		    ("Webserver's authentication method incompatible with Liferea");
		break;
	    }
	}
	/* And generic messages in the unlikely event that the above didn't work */
	if (NULL == tmp2) {
	    switch (httpstatus / 100) {
	    case 3:
		tmp2 =
		    g_strdup
		    ("Feed not available: Server requested unsupported redirection!");
		break;
	    case 4:
		tmp2 = g_strdup("Client Error");
		break;
	    case 5:
		tmp2 = g_strdup("Server Error");
		break;
	    default:
		tmp2 = g_strdup("(unknown networking error happened)");
		break;
	    }
	}
	errorFound = TRUE;
	tmp1 = g_strdup_printf(HTTP_ERROR_TEXT, httpstatus, tmp2);
	addToHTMLBuffer(&buffer, tmp1);
	g_free(tmp1);
	g_free(tmp2);
    }

    /* add parsing error messages */
    if (NULL != fp->parseErrors) {
	errorFound = TRUE;
    }

    /* if none of the above error descriptions matched... */
    if (!errorFound) {
    }

    fp->errorDescription = buffer;
}

static gboolean do_unsubscribe(struct fp_prop_ui_data *ui_data)
{
    ULOG_DEBUG("*!*! Query unsubscribe !*!*");
    if((nodePtr)ui_data->fp != ui_feedlist_get_selected())
        ui_feedlist_select((nodePtr)ui_data->fp);

    if (SFM_SEARCH == app_data->app_ui_data->search_mode){
        gtk_infoprint(GTK_WINDOW(app_data->app_ui_data->app), _("rss_ib_unable_unsubscribe_searching"));
        return FALSE;
    }

    /* TODO: downloading images also uses SFM_REFRESH mode */
    if (SFM_REFRESH == app_data->app_ui_data->search_mode){
        gtk_infoprint(GTK_WINDOW(app_data->app_ui_data->app), _("rss_ib_unable_unsubscribe"));
        return FALSE;
    }

    nodePtr ptr = (nodePtr) ui_feedlist_get_selected();

    if (!ptr || ptr->type != FST_FEED) {
        gtk_infoprint(GTK_WINDOW(app_data->app_ui_data->app),
            _("rss_ib_no_feed_selected"));
        return FALSE;
    }

    if (ui_feedlist_check_factory_delete(TRUE))
        return FALSE;

    return TRUE;
}

void on_insensitive_unsubscribe(GtkButton * button, gpointer user_data)
{
    do_unsubscribe((struct fp_prop_ui_data *) user_data);
}

/** Decodes source
  *
  * @param ui_data pointer to an ui data structure
  * @return the decoded source
  */
static gchar *ui_feed_dialog_decode_source(struct fp_prop_ui_data *ui_data)
{
    gchar *source = NULL;

    gchar *str = NULL, *tmp2 = NULL;
    /* First, strip leading and trailing whitespace */
  if (ui_data->sourceURI)
  {
    str =
	g_strstrip(g_strdup((ui_data->sourceURI)));
  }
  else
  {
    str =
	g_strstrip(g_strdup
		   (gtk_entry_get_text(GTK_ENTRY(ui_data->sourceEntry))));
  }

    /* Add http:// if needed */
    if (strstr(str, "://") == NULL) {
	tmp2 = g_strdup_printf("http://%s", str);
	g_free(str);
	str = tmp2;
    }

    /* Add trailing / if needed */
    if (strstr(strstr(str, "://") + 3, "/") == NULL) {
	tmp2 = g_strdup_printf("%s/", str);
	g_free(str);
	str = tmp2;
    }
    
    /* Changing feed:// ->http:// */
if (g_str_has_prefix(str,"feed:"))
{
  str[0]='h';
  str[1]='t';
  str[2]='t';
  str[3]='p';
}
    /* Use the values in the textboxes if also specified in the URL! */
    if ((NULL != ui_data->authcheckbox) &&
	gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON
				     (ui_data->authcheckbox))) {
	xmlURIPtr uri = xmlParseURI(BAD_CAST str);
	if (uri != NULL) {
	    xmlChar *sourceUrl;
	    xmlFree(uri->user);
	    uri->user = g_strdup_printf("%s:%s",
					gtk_entry_get_text(GTK_ENTRY
							   (ui_data->
							    username)),
					gtk_entry_get_text(GTK_ENTRY
							   (ui_data->
							    password)));
	    sourceUrl = xmlSaveUri(uri);
	    source = g_strdup(BAD_CAST sourceUrl);
	    g_free(uri->user);
	    uri->user = NULL;
	    xmlFree(sourceUrl);
	    xmlFreeURI(uri);
	} else
	    source = g_strdup(str);
    } else {
	source = g_strdup(str);
    }
    g_free(str);
    str=NULL;

    return source;
}

/** After closing the add new feed dialog this function is called
  *
  * @param dialog the new feed dialog
  * @param response_id the id sent by hitting one of the dialog buttons
  * @param user_data pointer to an ui data structure
  */
static void on_newdialog_response(GtkDialog * dialog, gint response_id,
				  gpointer user_data)
{
    struct fp_prop_ui_data *ui_data = (struct fp_prop_ui_data *) user_data;
    AppUIData *app_ui_data;
    g_assert(app_data != NULL);
    
    app_ui_data = app_data->app_ui_data;

    gchar *source = NULL;

    g_assert(app_ui_data != NULL);
    
    if(app_ui_data->source != NULL)
    {
	g_free(app_ui_data->source);
	app_ui_data->source = NULL;
    }
	    
    if(app_ui_data->filter != NULL)
    {
        g_free((gchar *)app_ui_data->filter);
        app_ui_data->filter = NULL;
    }
    
    switch (response_id) {
    case GTK_RESPONSE_OK:
        if (!ui_feed_newdialog_can_close(ui_data, FALSE, &source)) {
            gtk_widget_grab_focus(ui_data->sourceEntry);
        }
/*	source = ui_feed_dialog_decode_source(ui_data);

	ULOG_DEBUG("Trying to add source:\n%s", source);
	
	if(strcmp(source, "") == 0) {
	    ULOG_DEBUG("Error case 1");
	    gtk_infoprint(GTK_WINDOW(app_data->app_ui_data->app), _("rss_ni_no_web_address_for_feed"));
	    g_free(source);
	    source = NULL;
	} else if (strcasecmp(source, "http:///") == 0) {
	    ULOG_DEBUG("Error case 2");
	    gtk_infoprint(GTK_WINDOW(app_data->app_ui_data->app), _("rss_ni_no_web_address_for_feed"));
	    g_free(source);
	    source = NULL;
	} else if ((strncasecmp(source, "http://", 7) != 0) &&
		   (strncasecmp(source, "https://", 8) != 0) &&
           (strncasecmp(source, "file://", 7) != 0) ) {
            ULOG_DEBUG("Error case 3. Only takes http,https,file protocol");
	    gtk_infoprint(GTK_WINDOW(app_data->app_ui_data->app), _("rss_ni_adding_failed"));
	    g_free(source);
	    source = NULL;
	} else if (strchr(source,'.') == NULL) {
	    ULOG_DEBUG("Error case 4");
    	    fprintf(stderr,"invalid new feed address\n");
	    gtk_infoprint(GTK_WINDOW(app_data->app_ui_data->app), _("rss_ni_adding_failed"));
	    g_free(source);
	    source = NULL;
	} else {
	    //something like this http://qewr./ is not valid. Should be http://abc.xyz/
	    char *tmp = strchr(source,'.');
	    fprintf(stderr, "tmp = %s\n", tmp);
	    if (strncmp(tmp, "./", 2) == 0) {
	        ULOG_DEBUG("Error case 5");
		fprintf(stderr,"invalid new feed address\n");
	        gtk_infoprint(GTK_WINDOW(app_data->app_ui_data->app), _("rss_ni_adding_failed"));
	        g_free(source);
	        source = NULL;
	    }*/
	    else {
	        ULOG_DEBUG("New subscription: %s", source);	 	    	              
	        //tvh :TODO: Why not modal here?????  	    		   
            /* TODO: SZs vvvvv */
	        gtk_window_set_modal(GTK_WINDOW(app_ui_data->add_feed_dialog), TRUE);
            /* tvh: Moved from dbus iap connected handling here */
   	        gtk_widget_set_sensitive(app_data->app_ui_data->newfeedbtn, FALSE);
	        gtk_widget_set_sensitive(app_data->app_ui_data->changefolderbtn, FALSE);
	        gtk_widget_set_sensitive(app_data->app_ui_data->browsebtn, FALSE);
	        gtk_widget_set_sensitive(ui_data->sourceEntry, FALSE);

/*        ptr = ui_data->parent_folder;

        ui_feed_directory_button_clicked(NULL,NULL,fd_source, ptr);*/
	        ui_feedlist_new_subscription(source, NULL,
					 FEED_REQ_RESET_TITLE |
					 FEED_REQ_RESET_UPDATE_INT |
					 FEED_REQ_AUTO_DISCOVER |
					 FEED_REQ_DOWNLOAD_FAVICON,
					 ui_data->parent_folder);
	    }
//        }
	break;

    case GTK_RESPONSE_CANCEL:
    case GTK_RESPONSE_DELETE_EVENT:
//	app_ui_data->dialog = NULL;
//	app_ui_data->add_feed_dialog = NULL;
	    
        if (SFM_REFRESH == app_ui_data->search_mode) {
	    app_ui_data->search_mode = SFM_INTERRUPTED;
	    download_cancel_all(TRUE);
        }
//	gtk_widget_destroy(GTK_WIDGET(dialog));
//	g_free(ui_data);
        newdialog_destroy();
	break;
    }
}


/** By clicking on one of the buttons to close the feed details dialog
  * this function is called
  *
  * @param dialog the feed details dialog
  * @param response_id  the id sent by hitting one of the dialog buttons
  * @param user_data pointer to an ui data structure
  */
static void on_propdialog_response(GtkDialog * dialog, gint response_id,
				   gpointer user_data)
{
    struct fp_prop_ui_data *ui_data = (struct fp_prop_ui_data *) user_data;
    AppUIData *app_ui_data;
    gboolean close_dialog = TRUE;
    gchar *title = NULL;
	gchar *source = NULL;
    gchar *tmp = NULL;
    
    ULOG_DEBUG(__FUNCTION__);

    g_assert(app_data != NULL);
    
    app_ui_data = app_data->app_ui_data;
    
    g_assert(app_ui_data != NULL);
    
    switch (response_id) {
    case GTK_RESPONSE_OK:
	/* General */
        g_assert(ui_data != NULL);
        g_assert(ui_data->feedNameEntry != NULL);
        
        if (feed_prop_can_close(ui_data, TRUE, &title, &source))
        {
            if (ui_data->parent_folder != NULL) {
               ULOG_INFO("%s: changing folder", __FUNCTION__);
               ULOG_DEBUG("fp: %p", ui_data->fp);
               ui_feedlist_remove((nodePtr) ui_data->fp);
               g_assert(ui_data->fp->ui_data == NULL);
               ui_feedlist_add(ui_data->parent_folder,
                    (nodePtr) ui_data->fp, -1);
               ULOG_INFO("%s: changed folder", __FUNCTION__);
            } else {
               ULOG_INFO("%s: folder was not changed", __FUNCTION__);
            }

           ULOG_DEBUG("%s: setting title", __FUNCTION__);
	   if (!feed_get_nonremovable(ui_data->fp)) {
               feed_set_title(ui_data->fp, title);
               feed_set_source(ui_data->fp, source);
               if (displayed_node == ui_data->fp) {
                   displayed_node = NULL;
                   ui_feedlist_load_selected(ui_data->fp);
               }
           }

           gboolean dn = gtk_toggle_button_get_active(
                               GTK_TOGGLE_BUTTON(ui_data->downloadImagesEntry));
           if (dn != feed_get_download_images(ui_data->fp))
               feed_set_download_images(ui_data->fp, dn);

           ULOG_DEBUG("%s: call feedlist update ...", __FUNCTION__);
           ui_feedlist_update();
           ULOG_DEBUG("%s: call feedlist save ...", __FUNCTION__);
           conf_feedlist_schedule_save();
        } else
            close_dialog = FALSE;
        
        break;
    case RSS_RESPONSE_FEED_PROP_UNSUBSCRIBE:

	   ULOG_INFO
	    ("Unsubscribe was selected ... unsubscribe the feed now.");
	
    	g_assert(ui_data != NULL);
    	g_assert(ui_data->fp != NULL);
    	
        close_dialog = do_unsubscribe(ui_data) && ui_feedlist_delete((nodePtr)ui_data->fp);
/*    	if((nodePtr)ui_data->fp != ui_feedlist_get_selected())
    	    ui_feedlist_select((nodePtr)ui_data->fp);
	
    	close_dialog = ui_feedlist_delete((nodePtr)ui_data->fp);*/

    	break;
    }

    ULOG_DEBUG("on_propdialog_response: done");

    if(close_dialog)
    {
        app_ui_data->dlg_btn_unsubscribe = NULL;
        g_free(ui_data);
        app_ui_data->dialog = NULL;
        gtk_widget_destroy(GTK_WIDGET(dialog));
    }
    /*g_free*/
    if (NULL != tmp)
        g_free(tmp);
}

/** Determines the feeds favicon or default icon 
  *
  * @param fp the feed to get the icon for
  * @return the icon
  */
static GdkPixbuf* ui_feed_get_icon(feedPtr fp) 
{
	gpointer	favicon;
	
	g_assert(FST_FOLDER != fp->type);

	if(NULL != (favicon = feed_get_favicon(fp)))
		return favicon;
	if (feed_get_available(fp))
    	return icons[ICON_RSS_NEWS_FEED];
    else 
    	return icons[ICON_RSS_NEWS_INVALID_FEED];
}


/* Remove the hidden and new attribute from that items, what it has these*/
void ui_feed_remove_hidden_attribute(feedPtr fp)
{
	GSList *item = NULL;
	item = fp->items;
	int i=0;
	ULOG_DEBUG("Entering ui_feed_remove_hidden_attribute\n");
	/*We have to analyse, if we have hidden items. If we don't, we don't do anything. */
	while ((NULL != item)&& !i) {
		if (item_get_hidden((itemPtr) item->data))
		{
			i++;
			ULOG_DEBUG("Hidden item found %d\n",i);
		}
		item = g_slist_next(item);
	}
	item = fp->items;
	/*If we don't have items, what have hidden status, than there are only new 
	items, so we don't need to do anything*/
	if (i)
	{
		while (NULL != item) 
		{
			if (item_get_new_status((itemPtr) item->data))
			{
				if (item_get_hidden((itemPtr) item->data))
				{
//					ULOG_DEBUG("Removing Hidden and New attribute\n");
					item_set_hidden((itemPtr) item->data,FALSE);
					item_set_new_status((itemPtr) item->data,FALSE);
				}
			}
			item = g_slist_next(item);
		}		
	}
}

/* Delete the old items. It will delet only, if we have hidden items
(setted by the ui_feed_set_new_items_hidden function, before the beginning of
the refresh)*/
void ui_feed_delete_old_items(feedPtr fp)
{
	GSList *item = NULL;
	item = fp->items;
	int i=0;
	ULOG_DEBUG("Entering ui_feed_delete_old_items\n");
	/*We have to analyse, if we have hidden items. If we don't, we don't do anything. */
	while (NULL != item && !i) {
		if (item_get_hidden((itemPtr) item->data))
		{
			i++;
//			ULOG_DEBUG("Hidden item found %d\n",i);
		}
		item = g_slist_next(item);
	}
	item = fp->items;
	/*If we don't have items, what have hidden status, than there are only new 
	items, so we don't need to do anything*/
	if (i)
	{
        ULOG_DEBUG("Removing old feeds");
		while (NULL != item) 
		{
			if (FALSE==item_get_new_status((itemPtr) item->data))
			{
				if(FALSE==item_get_mark(item->data))
				{
//					ULOG_DEBUG("Deleting unneded item");
/*          if (item_get_delete_count(item->data)>MAX_DELETE_REQ)
          {*/
					feed_remove_item(fp,item->data);
					item=NULL;
/*          }
          else
          {
            item_increase_delete_count(item->data);
          }*/
				}
			}
            if (item)
                item = g_slist_next(item);
            else
                item = fp->items;
		}		
	}
}


/************************************************************************/
/*                        PUBLIC FUNCTIONS                              */
/************************************************************************/

/** handles completed feed update requests */
void ui_feed_process_update_result(struct request *request)
{
    feedPtr fp = (feedPtr) request->user_data;
    feedHandlerPtr fhp = NULL;
    gchar *old_title = NULL, *old_source = NULL;
    gint old_update_interval = 0;
    gboolean must_load_selected = ((feedPtr) displayed_node == fp);

    g_assert(app_data != NULL);
    g_assert(app_data->app_ui_data != NULL);
    
//    debug_enter("ui_feed_process_update_result");
	/* TODO: Remove this */
    ui_lock();
    
    if ((must_load_selected)&&(NULL != request->data))
        gtkhtml_stop();

	/*Temporarly mark the "old" new items, as hidden*/
    if (!((app_data->app_ui_data->auto_refresh_is_ongoing)&&(must_load_selected)))
    {
	    ui_feed_set_new_items_hidden(fp);
    }
	
    g_assert(NULL != request);

    feed_load(fp);
    

/*    ULOG_DEBUG("\nret from download: request->source: %s",request->source);
    ULOG_DEBUG("request->data: %d",(request->data==NULL?0:1));*/
    
    /* no matter what the result of the update is we need to save update
       status and the last update time to cache */
    fp->needsCacheSave = TRUE;

    feed_set_available(fp, TRUE);

    if (401 == request->httpstatus) {	/* unauthorized */
        ULOG_DEBUG("Returned 401: Unauthorized request");
	feed_set_available(fp, FALSE);
	app_data->app_ui_data->errors_in_refreshing = TRUE;
    } else if (410 == request->httpstatus) {	/* gone */
        ULOG_DEBUG("Returned 410");
	feed_set_available(fp, FALSE);
	feed_set_discontinued(fp, TRUE);
	ULOG_DEBUG
	    ("\"%s\" is discontinued. Liferea won't updated it anymore!",
	     feed_get_title(fp));
	app_data->app_ui_data->errors_in_refreshing = TRUE;
    } else if (304 == request->httpstatus) {
        ULOG_DEBUG("Returned 304");
        /*feed_remove_unmarked(fp, TRUE);*/
    } else if (NULL != request->data) {
  feed_reset_update_counter(fp);
	feed_set_lastmodified(fp, request->lastmodified);
	feed_set_etag(fp, request->etag);

	/* note this is to update the feed URL on permanent redirects */
	if (0 != strcmp(request->source, feed_get_source(fp))) {
	    feed_set_source(fp, request->source);
	    ULOG_DEBUG
		("The URL of \"%s\" has changed permanently and was updated",
		 feed_get_title(fp));
	}

    ULOG_DEBUG("ui_feed_process_update_result: 1");
    if (must_load_selected)
        ui_itemlist_clear();
    
    ULOG_DEBUG("ui_feed_process_update_result: 2");
	/* we save all properties that should not be overwritten in all cases */
	old_update_interval = feed_get_update_interval(fp);
	old_title = g_strdup(feed_get_title(fp));
	old_source = g_strdup(feed_get_source(fp));

    ULOG_DEBUG("ui_feed_process_update_result: 3");
	ULOG_DEBUG("Old title: %s",old_title);

        feed_set_parse_errors(fp, FALSE);

        /* parse the new downloaded feed into new_fp */
	fhp =
	    feed_parse(fp, request->data, request->size,
		       request->flags & FEED_REQ_AUTO_DISCOVER);
    
    /* If opml gotten from add feed dialog. Just returns, because feed directory
    dialog will be popped up by now */
    ULOG_DEBUG("ui_feed_process_update_result: 4");
    if ((fhp == NULL) && (fp->available == FALSE) &&
        (app_data->app_ui_data->add_feed_dialog != NULL))
    {
        //free stuff
      g_free(old_title);
      g_free(old_source);
      fp = NULL;
      
      gtk_widget_destroy(app_data->app_ui_data->add_feed_dialog);
          
      ui_unlock();
      
      return;
    }    
    if (fhp && strcmp(fhp->typeStr, "opml") == 0)                         
    {
        //free stuff
        g_free(old_title);
        g_free(old_source);
        fp = NULL;
        ui_unlock();
        return;
    }    
    ULOG_DEBUG("ui_feed_process_update_result: 5");
	if (fhp == NULL || feed_get_parse_errors(fp)) {
	    ULOG_DEBUG("Could not detect the type of this feed");
	    feed_set_available(fp, FALSE);
	    app_data->app_ui_data->errors_in_refreshing = TRUE;
	} else if(feed_get_new_subscription(fp) == TRUE &&
	        feed_get_feed_directory(fp) == TRUE &&
	        app_data->app_ui_data->add_feed_dialog != NULL) 
    {
        //ULOG_DEBUG("OPML file detected, but not containing any directories,"
        //    "so add it to the list");
	    ULOG_DEBUG("Unnecessary feed directory found. Not adding to the feed tree");
        request->folder = app_data->app_ui_data->ui_data->parent_folder;
	    newdialog_destroy();
        
/*	    if(!app_data->app_ui_data->feed_directory_cancelled)
	    newdialog_destroy();
	    else
		newdialog_return_to_normal();*/
		    
//        app_data->app_ui_data->feed_directory_cancelled = FALSE;		    
//	    if (app_data->app_ui_data->feed_directory_count > 0)
        feed_set_new_subscription(fp, FALSE);
        feed_set_available(fp, FALSE);
        
        ULOG_DEBUG("Processing as a directory...");
        app_data->app_ui_data->feed_directory_loading = TRUE;
        ui_feed_directory_process(request);
	} else {
	    fp->fhp = fhp;
            
        ULOG_DEBUG("Feed type found %s", fhp->typeStr);
	    
	    /* Feed has been added from the add feed dialog and has been found so
	       it is added to the feed tree */
	    if(feed_get_new_subscription(fp) &&
	    app_data->app_ui_data->add_feed_dialog != NULL)
	    {
	        folderPtr selected_parent = app_data->app_ui_data->ui_data->parent_folder;
	        folderPtr parent = NULL;
	    
	        parent = (selected_parent != NULL)
	    	    ? selected_parent : ui_feedlist_get_root_folder();
	     
	        ui_feedlist_add(parent, (nodePtr) fp, -1);
	        ui_feedlist_update();
	        ui_feedlist_select((nodePtr) fp);
	        
	        newdialog_destroy();
	        feed_set_new_subscription(fp, FALSE);
            }
	    
	    /* restore user defined properties if necessary */
	    if (!(request->flags & FEED_REQ_RESET_TITLE))
	    {
//	        ULOG_DEBUG("Resetting the title");
		feed_set_title(fp, old_title);
	    }

    ULOG_DEBUG("ui_feed_process_update_result: 6");
	    if (!(request->flags & FEED_REQ_AUTO_DISCOVER))
		feed_set_source(fp, old_source);
		if (!((app_data->app_ui_data->auto_refresh_is_ongoing)&&(must_load_selected)))
    {
		  ui_feed_delete_old_items(fp);
		  ui_feed_remove_hidden_attribute(fp);
    }

	    if (!(request->flags & FEED_REQ_RESET_UPDATE_INT))
		feed_set_update_interval(fp, old_update_interval);
		
    ULOG_DEBUG("ui_feed_process_update_result: 7");

	    ULOG_DEBUG("\"%s\" updated...",
					 feed_get_title(fp));

        if (feed_get_download_images(fp)) {
/*            ULOG_DEBUG("\"%s\" checking for images...",
                         feed_get_title(fp));*/
            feed_download_images(fp);
        }
        
    ULOG_DEBUG("ui_feed_process_update_result: 8");
//	    if ((feedPtr) displayed_node == fp)
        if (must_load_selected)
        {
            feed_set_feed_read(fp, -1);
		ui_itemlist_load((nodePtr) fp);
            feed_mark_all_items_read(fp);
//            ULOG_DEBUG("displaying node");
        } else
//            ULOG_DEBUG("hiding node");

    ULOG_DEBUG("ui_feed_process_update_result: 9");
	    if (request->flags & FEED_REQ_SHOW_PROPDIALOG)
	    {
	        g_assert(app_data != NULL);
		g_assert(app_data->app_ui_data != NULL);
		
		ui_feed_propdialog_new(GTK_WINDOW
				       (app_data->app_ui_data->app), fp);
	    }
	}
	g_free(old_title);
	g_free(old_source);
    } else {
        ULOG_DEBUG("Feed not available");
	ULOG_DEBUG("\"%s\" is not available",
				     feed_get_title(fp));
	feed_set_available(fp, FALSE);
	app_data->app_ui_data->errors_in_refreshing = TRUE;
    }

    ULOG_DEBUG("ui_feed_process_update_result: 10");
//    ULOG_DEBUG("Download processing ended");
    
    ui_feed_set_error_description(fp, request->httpstatus,
				  request->returncode);

    ULOG_DEBUG("ui_feed_process_update_result: 11");
    fp->request = NULL;
//    ui_notification_update(fp);
    ULOG_DEBUG("ui_feed_process_update_result: 12");
    ui_feedlist_update();
    
    ULOG_DEBUG("ui_feed_process_update_result: 13");
    if(feed_get_available(fp))
        if(request->flags & FEED_REQ_DOWNLOAD_FAVICON)
            favicon_download(fp, app_data->app_ui_data);
		
    ULOG_DEBUG("ui_feed_process_update_result: 14");
    if (feed_unload(fp) == FALSE)
    {
      ui_unlock();
      
      app_data->app_ui_data->search_mode = SFM_INTERRUPTED;
      download_cancel_all(TRUE);
      g_warning("cancel all downloads!");
      
      return;
    }

    /* If the add feed dialog is displayed and the feed is invalid
       or it is a feed directory the feed is freed */
    if((app_data->app_ui_data->add_feed_dialog != NULL &&
       feed_get_available(fp) == FALSE) ||
       (feed_get_feed_directory(fp) == TRUE && 
       feed_get_available(fp) == FALSE))
    {
       ULOG_DEBUG("Deleting an invalid feed");
       ui_feedlist_delete_((nodePtr)fp);
    }
    ULOG_DEBUG("ui_feed_process_update_result: 15");
    ui_unlock();
}

/** updating of a single feed list entry */
void ui_feed_update(feedPtr fp)
{
    GtkTreeModel *model = NULL;
    GtkTreeIter *iter = NULL;
    gchar *label = NULL, *tmp = NULL;
    int unread = 0, marked = 0;

    if (fp->ui_data == NULL)
	return;

    iter = &((ui_data *) fp->ui_data)->row;
    model = GTK_TREE_MODEL(feedstore);

    g_assert(FST_FOLDER != fp->type);

    if (FST_FEED == fp->type)
	rss_max_feed = rss_max_feed + 1;
    unread = feed_get_unread_counter(fp);
    marked = feed_get_mark_counter(fp);
    /*
    label = unhtmlize(g_strdup(feed_get_title(fp)));

    // FIXME: Unescape text here! 
    tmp = g_markup_escape_text(label, -1);
    tmp = g_strstrip(tmp);
    
    g_free(label);
    */

    //let's be simple. why need to unhtmlize?    

    //need to escape though, as textrenderer is using markup


    label = g_strdup(feed_get_title(fp));
    label= utf8_fix(label);
    tmp = g_markup_escape_text(label, -1);    
    tmp = g_strstrip(tmp);
//    ULOG_DEBUG("ui_feed_update: tmp = %s", tmp);
    g_free(label);
    
    if (unread > 0 && marked > 0)
	label =
	    g_strdup_printf
	    ("<span weight=\"bold\" foreground=\"%s\">%s (%d)</span>",
	     "red"/*get_logical_color(EMPTEXTCOLOR)*/, tmp, unread);
    else if (marked > 0)
	label = g_strdup_printf("<span foreground=\"%s\">%s</span>",
				"red"/*get_logical_color(EMPTEXTCOLOR)*/, tmp);
    else if (unread > 0)
	label =
	    g_strdup_printf
	    ("<span foreground=\"%s\" weight=\"bold\">%s (%d)</span>",
	     "black" /*get_logical_color(DEFAULTTEXTCOLOR)*/, tmp, unread);
    else
	label = g_strdup_printf("<span foreground=\"%s\">%s</span>",
				"black" /*get_logical_color(DEFAULTTEXTCOLOR)*/, tmp);
    g_free(tmp);
    
    /* CHECK: valgrind says internal row-changed event from this leaks memory */
    gtk_tree_store_set(feedstore, iter, FS_LABEL, label,
		       FS_UNREAD, unread,
		       FS_ICON, ui_feed_get_icon(fp), -1);
    g_free(label);
}

static void insensitive_btn_press(GtkWidget *widget, struct fp_prop_ui_data *ui_data)
{
    if (ui_feed_newdialog_can_close(ui_data, TRUE, NULL))
        gtk_infoprint(GTK_WINDOW(app_data->app_ui_data->app), _("rss_ib_subcribe_already_started"));
}

/*Temporarly mark the New items as hidden. So we will know for example after 
refreshing, what items are realy new, and what was new just before refreshing*/
void ui_feed_set_new_items_hidden(feedPtr fp)
{
	GSList *item = NULL;
	itemPtr curitem;
    ULOG_DEBUG("Entering ui_feed_set_new_items_hidden\n");
    g_assert(fp != NULL);
    item = fp->items;

	while (NULL != item) {
		
		curitem=item->data;
		if (fp->newCount)
		{
		if (TRUE==curitem->newStatus)
			{
				ULOG_DEBUG("Hiding an item\n");
				curitem->hidden=TRUE;
			}
	}
	item = g_slist_next(item);
    }
	ULOG_DEBUG("Leaving ui_feed_set_new_items_hidden\n");
}

/********************************************************************
 * newdialog                                                       *
 *******************************************************************/

GtkWidget *ui_feed_newdialog_new(GtkWindow * parent, const gchar *source)
{
    GtkWidget *newdialog;
    struct fp_prop_ui_data *ui_data = NULL;
    AppUIData *app_ui_data;
    folderPtr ptr = NULL;
    
    g_assert(app_data != NULL);
    
    app_ui_data = app_data->app_ui_data;

    app_ui_data->ui_data = ui_data = g_new0(struct fp_prop_ui_data, 1);

    g_assert(app_ui_data != NULL);
    
    if(ui_data == NULL) {
        ULOG_DEBUG("Not enough memory for add feed dialog");
        return NULL;
    }
    
    /* Create the dialog */
    app_ui_data->add_feed_dialog = app_ui_data->dialog = ui_data->dialog = newdialog = create_newdialog();
    
    ui_data->label_widget = lookup_widget(newdialog, "folderName");
    
    ptr = ui_feedlist_get_target_folder();

    if(ptr != NULL && ptr != ui_feedlist_get_root_folder()) {
        gtk_label_set_text(GTK_LABEL(ui_data->label_widget),
			   ptr->title);
    
        ui_data->parent_folder = ptr;
    } else {
        ui_data->parent_folder = ui_feedlist_get_root_folder();
    }

    GtkWidget *changefolder = lookup_widget(newdialog, "changefolder");
    g_signal_connect(changefolder, "clicked",
		     G_CALLBACK(display_change_folder_dialog), ui_data);

    GtkWidget *browsebtn = lookup_widget(newdialog, "browsebtn");
    g_signal_connect(browsebtn, "clicked",
		     G_CALLBACK(display_file_chooser_dialog), ui_data);
            

    gtk_window_set_transient_for(GTK_WINDOW(newdialog),
				 GTK_WINDOW(parent));
    gtk_window_set_modal(GTK_WINDOW(newdialog), TRUE);

	/***********************************************************************
	 * Source                                                              *
	 **********************************************************************/

    /* Setup source entry */
    ui_data->sourceEntry = lookup_widget(newdialog, "sourceEntry");
    g_signal_connect(G_OBJECT(ui_data->sourceEntry), "changed", 
            G_CALLBACK(enable_sensitive_newfeedbtn), ui_data);
    if (source != NULL)
    {
        ULOG_DEBUG("ui_feed_newdialog_new: setting default sourcEntry to: %s", source);
        //set it here
         gtk_entry_set_text (GTK_ENTRY(ui_data->sourceEntry), (const gchar *) source);
    }    

    gtk_widget_grab_focus(GTK_WIDGET(ui_data->sourceEntry));
    gtk_entry_set_activates_default(GTK_ENTRY(ui_data->sourceEntry), TRUE);

    ui_data->newfeedbtn = lookup_widget(newdialog, "newfeedbtn");
    gtk_widget_set_sensitive(ui_data->newfeedbtn, source != NULL);
    gtk_widget_grab_default(ui_data->newfeedbtn);
    g_signal_connect(ui_data->newfeedbtn, "insensitive_press",
             G_CALLBACK(insensitive_btn_press), ui_data);
    g_signal_connect(G_OBJECT(changefolder), "insensitive_press",
             G_CALLBACK(on_insensitive_infoprint),
                  _("rss_ib_unable_change_tg_subscribe"));
    g_signal_connect(G_OBJECT(newdialog), "response",
		     G_CALLBACK(on_newdialog_response), ui_data);
    g_signal_connect(G_OBJECT(newdialog), "key-press-event",
		     G_CALLBACK(key_press_for_cancelling_dialog), NULL);
    gtk_widget_show_all(newdialog);

    return newdialog;
}

gboolean ui_feed_newdialog_can_close(struct fp_prop_ui_data *ui_data, gboolean msg, 
    gchar **result)
{
    gchar *source = ui_feed_dialog_decode_source(ui_data);

    ULOG_DEBUG("Trying to add source:\n%s", source);

    if (result)
        *result = NULL;

    if(strcmp(source, "") == 0) {
        ULOG_DEBUG("Error case 1");
        if (msg)
            gtk_infoprint(GTK_WINDOW(app_data->app_ui_data->app), 
                _("rss_ni_no_web_address_for_feed"));
        g_free(source);
        return FALSE;
    } else if (strcasecmp(source, "http:///") == 0) 
    {
        ULOG_DEBUG("Error case 2");
        if (msg)
            gtk_infoprint(GTK_WINDOW(app_data->app_ui_data->app), 
                _("rss_ni_no_web_address_for_feed"));
        g_free(source);
        return FALSE;
    } else if ((strncasecmp(source, "http://", 7) != 0) &&
           (strncasecmp(source, "https://", 8) != 0) &&
           (strncasecmp(source, "file://", 7) != 0)&&
           (strncasecmp(source, "obex://", 7) != 0)&&
           (strncasecmp(source, "feed://", 7) != 0)    ) 
    {
        ULOG_DEBUG("Error case 3. Only takes http,https,file protocol");
        if (msg)
            gtk_infoprint(GTK_WINDOW(app_data->app_ui_data->app), 
                _("rss_ni_adding_failed"));
        g_free(source);
        return FALSE;
    } else if (strchr(source,'.') == NULL) {
        ULOG_DEBUG("Error case 4");
        ULOG_DEBUG("invalid new feed address");
        if (msg)
            gtk_infoprint(GTK_WINDOW(app_data->app_ui_data->app), 
                _("rss_ni_adding_failed"));
        g_free(source);
        return FALSE;
    } else {
        //something like this http://qewr./ is not valid. Should be http://abc.xyz/
        char *tmp = strchr(source,'.');
        fprintf(stderr, "tmp = %s\n", tmp);
        fprintf(stderr, "source = %s\n", source);
        if (strncmp(tmp, "./", 2) == 0) 
        {
            ULOG_DEBUG("Error case 5");
            ULOG_DEBUG("invalid new feed address");
            if (msg)
                gtk_infoprint(GTK_WINDOW(app_data->app_ui_data->app), 
                    _("rss_ni_adding_failed"));
            g_free(source);
            return FALSE;
        }
    }

    if (result)
        *result = source;
    else
        g_free(source);

    return TRUE;
}

gboolean feed_prop_can_close(struct fp_prop_ui_data *ui_data, gboolean show,
    gchar **ptitle, gchar **psource)
{
    if (feed_get_nonremovable(ui_data->fp))
        return TRUE;

    const gchar *title = gtk_entry_get_text(GTK_ENTRY(ui_data->feedNameEntry));
    if (title)
    {
        gchar *tmp = g_strstrip(g_strdup(title));
        if (!*tmp)
            title = NULL;
        g_free(tmp);
    }
    
    if (!title)
    {
        if (show)
            gtk_infoprint(GTK_WINDOW(app_data->app_ui_data->app), 
                _("rss_ni_no_title_for_feed"));
        gtk_widget_grab_focus(ui_data->feedNameEntry);
        return FALSE;
    }

    const gchar *source = gtk_entry_get_text(GTK_ENTRY(ui_data->feedLocationEntry));

    if (source)
    {
        gchar *tmp = g_strstrip(g_strdup(title));
        if (!*tmp)
            source = NULL;
        g_free(tmp);
    }
    
    if ((!source) || (strcmp(source, "") == 0))
    {
        if (show)
            gtk_infoprint(GTK_WINDOW(app_data->app_ui_data->app), 
                _("rss_ni_no_web_address_for_feed"));
        gtk_widget_grab_focus(ui_data->feedLocationEntry);
        return FALSE;
    }

    if (ptitle)
        *ptitle = (gchar *)title;
    if (psource)
        *psource = (gchar *)source;

    return TRUE;
}

void on_feed_prop_changed(GtkWidget *widget, struct fp_prop_ui_data *ui_data)
{
    g_assert(ui_data);
    
    gtk_widget_set_sensitive(ui_data->propokbtn,
        feed_prop_can_close(ui_data, FALSE, NULL, NULL));
}

void on_insensitive_prop(GtkWidget *widget, struct fp_prop_ui_data *ui_data)
{
    g_assert(ui_data);
    
    feed_prop_can_close(ui_data, TRUE, NULL, NULL);
}

GtkWidget *ui_feed_propdialog_new(GtkWindow * parent, feedPtr fp)
{
    GtkWidget *propdialog;
    struct fp_prop_ui_data *ui_data = NULL;
    AppUIData *app_ui_data;
     
    g_assert(app_data != NULL);
    
    app_ui_data = app_data->app_ui_data;

    ui_data = g_new0(struct fp_prop_ui_data, 1);
    ui_data->fp = fp;
    
    g_assert(app_ui_data != NULL);
    
    /* Create the dialog */
    app_ui_data->dialog = ui_data->dialog = propdialog =
	create_propdialog(feed_get_nonremovable(fp));
    app_ui_data->propdialog = propdialog;

    GtkWidget *changefolder = lookup_widget(propdialog, "changefolder");
    g_signal_connect(changefolder, "clicked",
		     G_CALLBACK(display_change_folder_dialog), ui_data);
    g_signal_connect(changefolder, "insensitive_press",
		     G_CALLBACK(on_insensitive_infoprint), _("rss_ib_unable_to_move_feed"));
    gtk_widget_set_sensitive(changefolder, !feed_get_nonremovable(fp));

    GtkWidget *unsubscribe = lookup_widget(propdialog, "unsubscribeBtn");
/*    g_signal_connect(unsubscribe, "insensitive_press",
		     G_CALLBACK(on_insensitive_infoprint), _("rss_ib_unsubscribing_feed_not_allowed"));*/
    g_signal_connect(unsubscribe, "insensitive_press",
		     G_CALLBACK(on_insensitive_unsubscribe), ui_data);
    gtk_widget_set_sensitive(unsubscribe, !feed_get_nonremovable(fp) && app_ui_data->search_mode <= SFM_INTERRUPTED);
    app_ui_data->dlg_btn_unsubscribe = unsubscribe;

    gtk_window_set_transient_for(GTK_WINDOW(propdialog),
				 GTK_WINDOW(parent));
    
    gtk_window_set_modal(GTK_WINDOW(propdialog),TRUE);				 
	/***********************************************************************
	 * General                                                             *
	 **********************************************************************/

    /* If we're not modal, ui_data may be overwritten */
    ui_data->feedNameEntry = lookup_widget(propdialog, "feedNameEntry");
	ui_data->feedLocationEntry = lookup_widget(propdialog, "feedLocationEntry");
    if (feed_get_nonremovable(fp)) {
        gtk_label_set_label(GTK_LABEL(ui_data->feedNameEntry),
		       feed_get_title(fp));
        gtk_label_set_label(GTK_LABEL(ui_data->feedLocationEntry),
		       feed_get_source(fp));
        g_object_set(G_OBJECT(ui_data->feedNameEntry), "autocap", TRUE, NULL);
    } else {
        gtk_entry_set_text(GTK_ENTRY(ui_data->feedNameEntry),
		       feed_get_title(fp));
        g_signal_connect(G_OBJECT(ui_data->feedNameEntry),
            "changed", G_CALLBACK(on_feed_prop_changed), ui_data);
        gtk_entry_set_text(GTK_ENTRY(ui_data->feedLocationEntry),
		       feed_get_source(fp));
        g_signal_connect(G_OBJECT(ui_data->feedLocationEntry),
            "changed", G_CALLBACK(on_feed_prop_changed), ui_data);
        g_object_set(G_OBJECT(ui_data->feedNameEntry), "autocap", TRUE, NULL);
    }
    
    ui_data->downloadImagesEntry = 
    lookup_widget(propdialog, "downloadImagesEntry");
    gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(ui_data->downloadImagesEntry),
               feed_get_download_images(fp));

    GtkWidget *subscribedData =
	lookup_widget(propdialog, "subscribedData");
    gchar *time_str = ui_itemlist_format_date(feed_get_added(fp));
    gtk_label_set_text(GTK_LABEL(subscribedData), time_str);
    g_free(time_str);

    GtkWidget *folderName = lookup_widget(propdialog, "folderName");
    folderPtr parentPtr = (folderPtr) ui_feedlist_get_parent((nodePtr) fp);
    if (parentPtr != NULL) {
 	gtk_label_set_text(GTK_LABEL(folderName), parentPtr->title);
    }

	/***********************************************************************
	 * Source                                                              *
	 **********************************************************************/

    ui_data->propokbtn = lookup_widget(propdialog, "okBtn");
    g_signal_connect(ui_data->propokbtn, "insensitive_press",
		     G_CALLBACK(on_insensitive_prop), ui_data);

    g_signal_connect(G_OBJECT(propdialog), "response",
		     G_CALLBACK(on_propdialog_response), ui_data);
    g_signal_connect(G_OBJECT(propdialog), "key-press-event",
		     G_CALLBACK(key_press_for_cancelling_dialog), NULL);
    ui_data->label_widget = lookup_widget(propdialog, "folderName");

    gtk_widget_show_all(propdialog);

    return propdialog;
}

void newdialog_return_to_normal()
{
    g_assert(app_data != NULL);
    g_assert(app_data->app_ui_data != NULL);
    
    gtk_window_set_modal(GTK_WINDOW(app_data->app_ui_data->add_feed_dialog), TRUE);
    gtk_widget_set_sensitive(app_data->app_ui_data->newfeedbtn, TRUE);
    gtk_widget_set_sensitive(app_data->app_ui_data->changefolderbtn, TRUE);
    gtk_widget_set_sensitive(app_data->app_ui_data->browsebtn, TRUE);
    gtk_widget_set_sensitive(app_data->app_ui_data->ui_data->sourceEntry, TRUE);
//     gtk_widget_set_sensitive(app_data->app_ui_data->cancelbtn, TRUE);
}

static gboolean idle_refresh(gpointer data)
{
    gtk_widget_queue_draw(app_data->app_ui_data->html_scrollpane);
    gtk_widget_queue_draw(app_data->app_ui_data->rss_news);

    return FALSE;
}

void newdialog_destroy()
{
    g_assert(app_data != NULL);
    g_assert(app_data->app_ui_data != NULL);  
    if (app_data->app_ui_data->fsdialog != NULL)
    {
          
      gtk_dialog_response(app_data->app_ui_data->fsdialog,GTK_RESPONSE_CANCEL);
          
    }  

    gtk_widget_destroy(GTK_WIDGET(app_data->app_ui_data->add_feed_dialog));
	    
    app_data->app_ui_data->dialog = NULL;
    app_data->app_ui_data->add_feed_dialog = NULL;
	 
    g_free(app_data->app_ui_data->ui_data);
	    
    app_data->app_ui_data->ui_data = NULL;

    g_idle_add(idle_refresh, NULL);
}

void enable_sensitive_newfeedbtn(GtkWidget * widget, gpointer user_data)
{
    struct fp_prop_ui_data *data = (struct fp_prop_ui_data *)user_data;
    
    ULOG_INFO("Enable sensitive newfeedbtn");

    gtk_widget_set_sensitive(data->newfeedbtn, 
        ui_feed_newdialog_can_close(data, FALSE, NULL));
}
