/**
 * @file rule.c feed/vfolder rule handling
 *
 * Copyright (C) 2003, 2004 Lars Lindner <lars.lindner@gmx.net>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

/* For strcasestr */
#define _GNU_SOURCE

#include <osso-log.h>

#include <string.h>
#include "support.h"
#include "feed.h"
#include "item.h"
#include "rule.h"

/* rule function interface, each function requires a item
   structure which it matches somehow against its values and
   returns TRUE if the rule was fulfilled and FALSE if not.*/
typedef gboolean(*ruleCheckFuncPtr) (rulePtr rp, itemPtr ip);

/* the rule function list is used to create popup menues in ui_vfolder.c */
struct ruleInfo *ruleFunctions = NULL;
gint nrOfRuleFunctions = 0;

/************************************************************************/
/*                        PRIVATE FUNCTIONS                             */
/************************************************************************/

static gchar *wildcard_search(const gchar * haystack, const gchar * needle)
{
    gboolean leftWildcard = FALSE, rightWildcard = FALSE;

    gchar *n = g_strdup(needle);
    gchar *s = n;
    if (s[0] == '*') {
	leftWildcard = TRUE;
	s++;
    }

    int l = strlen(s);
    if (s[l - 1] == '*') {
	s[l - 1] = 0;
	rightWildcard = TRUE;
	l--;
    }

    if (leftWildcard && rightWildcard) {
	//we can cut some corners if there are leading AND trailing wildcards
	gchar *m = strcasestr(haystack, s);
	g_free(n);
	return m;
    }

    gchar *p = (gchar *) haystack;
    gchar *q;
    gboolean ok;
    gchar c;

    while (TRUE) {
	//iterate through all occurrences of the search string...
	q = strcasestr(p, s);
	if (q == NULL) {
	    break;
	}

	ok = TRUE;

	//validate word boundaries
	if ((!leftWildcard) && (q > haystack))	//if we are at the beginning of haystack, this is not considered.
	{
	    c = *(q - 1);

#define DELIM_IF if (!(c == ' ' || c == '\t' || c == '.' || c == ',' || c == ':' || c == ';' || c == '?' || \
                       c == '"' || c == '(' || c == ')' || c == '\'' || c == '\r' || c == '\n' || c == '\0'))

	    DELIM_IF {
		ok = FALSE;
	    }
	}
	if (!rightWildcard)	//end of haystack is denoted by a zero so no overflow condition needs to be checked here.
	{
	    c = *(q + l);

	    DELIM_IF {
		ok = FALSE;
	    }
	}

	if (ok) {
	    g_free(n);
	    return q;
	}
	//advance past the current occurrence
	p = q + l;
    }

    g_free(n);
    return NULL;
}

/* -------------------------------------------------------------------- */
/* rule checking implementations					*/
/* -------------------------------------------------------------------- */

static gboolean rule_exact_title_match(rulePtr rp, itemPtr ip)
{

    if (NULL != item_get_title(ip)) {
	if (NULL != wildcard_search(item_get_title(ip), rp->value))
	    return TRUE;
    }

    return FALSE;
}

static gboolean rule_exact_description_match(rulePtr rp, itemPtr ip)
{

    if (NULL != item_get_description(ip)) {
	if (NULL != wildcard_search(item_get_description(ip), rp->value))
	    return TRUE;
    }

    return FALSE;
}

static gboolean rule_exact_match(rulePtr rp, itemPtr ip)
{

    if (rule_exact_title_match(rp, ip))
	return TRUE;

    if (rule_exact_description_match(rp, ip))
	return TRUE;

    return FALSE;
}

static gboolean rule_is_unread(rulePtr rp, itemPtr ip)
{

    return !(item_get_read_status(ip));
}

static gboolean rule_is_flagged(rulePtr rp, itemPtr ip)
{

    return item_get_mark(ip);
}

/* rule initialization */

static void rule_add(ruleCheckFuncPtr func, gchar * ruleId, gchar * title,
		     gchar * positive, gchar * negative,
		     gboolean needsParameter)
{

    ruleFunctions =
	(ruleInfoPtr) g_realloc(ruleFunctions,
				sizeof(struct ruleInfo) *
				(nrOfRuleFunctions + 1));
    if (NULL == ruleFunctions)
	ULOG_ERR("could not allocate memory!");
    ruleFunctions[nrOfRuleFunctions].ruleFunc = func;
    ruleFunctions[nrOfRuleFunctions].ruleId = ruleId;
    ruleFunctions[nrOfRuleFunctions].title = title;
    ruleFunctions[nrOfRuleFunctions].positive = positive;
    ruleFunctions[nrOfRuleFunctions].negative = negative;
    ruleFunctions[nrOfRuleFunctions].needsParameter = needsParameter;
    nrOfRuleFunctions++;
}

/************************************************************************/
/*                        PUBLIC FUNCTIONS                              */
/************************************************************************/

rulePtr rule_new(feedPtr fp, const gchar * ruleId, const gchar * value,
		 gboolean additive)
{
    ruleInfoPtr ri = NULL;
    rulePtr rp = NULL;
    int i = 0;
    ULOG_DEBUG("\n\n\nrule_new: ruleId = %s, value = %s, fp name = %s",ruleId, value, fp->title);
    for (i = 0, ri = ruleFunctions; i < nrOfRuleFunctions; i++, ri++) {
	if (0 == strcmp(ri->ruleId, ruleId)) {
	    rp = (rulePtr) g_new0(struct rule, 1);
	    rp->fp = fp;
	    rp->value = g_strdup(value);
	    rp->ruleInfo = ri;
	    rp->additive = additive;
	    return rp;
	}
    }
    return NULL;
}

gboolean rule_check_item(rulePtr rp, itemPtr ip)
{
    g_assert(ip != NULL);

    return (*((ruleCheckFuncPtr) rp->ruleInfo->ruleFunc)) (rp, ip);
}

void rule_free(rulePtr rp)
{
    g_assert(rp != NULL);

    g_free(rp->value);
    g_free(rp);
}

/* Not localized, these should not appear in display */
void rule_init(void)
{

    rule_add(rule_exact_match, "exact", "Item", "does contain",
	     "does not contain", TRUE);
    rule_add(rule_exact_title_match, "exact_title", "Item title",
	     "does match", "does not match", TRUE);
    rule_add(rule_exact_description_match, "exact_desc", "Item body",
	     "does match", "does not match", TRUE);
    rule_add(rule_is_unread, "unread", "Read status", "is unread",
	     "is read", FALSE);
    rule_add(rule_is_flagged, "flagged", "Flag status", "is flagged",
	     "is unflagged", FALSE);
}
