/**
 * @file opml.c generic OPML 1.0 support
 * 
 * Copyright (C) 2003, 2004 Lars Lindner <lars.lindner@gmx.net>
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Library General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 * 
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Library General Public License for more details.
 * 
 * You should have received a copy of the GNU Library General Public License
 * along with this library; see the file COPYING.LIB.  If not, write to
 * the Free Software Foundation, Inc., 59 Temple Place - Suite 330,
 * Boston, MA 02111-1307, USA.
 */

#include <string.h>

#include <libxml/xmlmemory.h>
#include <libxml/parser.h>
#include <osso-log.h>
#include <hildon-widgets/hildon-note.h>
#include "support.h"
#include "conf.h"
#include "common.h"
#include "feed.h"
#include "item.h"
#include "opml.h"
#include "callbacks.h"
#include "htmlview.h"
#include "ui_feed_directory.h"
#include "ui_mainwindow.h"
#include "appdata.h"

extern AppData *app_data;

GSList *feed_list; /* List of feeds in the current feed directory */

/* you can find the OPML specification at Userland:

   http://www.opml.org/
   
 */

/* ---------------------------------------------------------------------------- */
/* OPML parsing and HTML output	 						*/
/* ---------------------------------------------------------------------------- */

/************************************************************************/
/*                        PRIVATE FUNCTIONS                             */
/************************************************************************/

/** Displays confirmation dialog for adding the list of
  * feeds to the feed tree */
/* unused atm
static void display_confirmation_dialog(gint feeds)
{
    HildonNote *note = NULL;
    gint resp = CONFRESP_ERROR;
    gchar *str = NULL, *str2 = NULL;
    folderPtr selected_parent = NULL;
    folderPtr parent = NULL;
    int i = 0;
    
    g_assert(app_data != NULL);
    g_assert(app_data->app_ui_data != NULL);
    
    str = g_strdup_printf("%s", _("rss_nc_add_all_feeds"));
    str2 = g_strdup_printf(str, feeds);
    
    note = HILDON_NOTE
	(hildon_note_new_confirmation_add_buttons
	 (NULL,
	  str2,
	  _("rss_bd_ok"),
	  CONFRESP_OK, _("rss_bd_cancel"), CONFRESP_CANCEL, NULL, NULL));
    
    if (note == NULL) {
	ULOG_CRIT("Unable to create HildonNote");
	g_free(str);
	g_free(str2);
	str = NULL;
	str2 = NULL;
	return;
    }

    resp = gtk_dialog_run(GTK_DIALOG(note));
    gtk_widget_destroy(GTK_WIDGET(note));
    g_free(str);
    g_free(str2);
    str = NULL;
    str2 = NULL;
            
    if(resp == CONFRESP_OK)
    {
        selected_parent = app_data->app_ui_data->ui_data->parent_folder;
	     
        parent = (selected_parent != NULL)
	         ? selected_parent : ui_feedlist_get_root_folder();
    
        ULOG_DEBUG("OK");
	
	for (i = 0; i < g_slist_length(feed_list); i++) 
	{
	    gchar *url = NULL;

	    url = (gchar *) g_slist_nth_data(feed_list, i);

	    if (url != NULL)
	    {
	        ui_feedlist_new_subscription_for_fd(url, NULL,
					            FEED_REQ_RESET_TITLE |
					            FEED_REQ_RESET_UPDATE_INT |
					            FEED_REQ_AUTO_DISCOVER, parent, TRUE, FALSE);
            }						   		    
	}
	ui_feedlist_update();
	app_data->app_ui_data->feed_directory_cancelled = FALSE;
    }
    else
        app_data->app_ui_data->feed_directory_cancelled = TRUE;
}*/

/** Frees allocated memory
  */
static void feed_list_free()
{
    int i = 0;

    if (feed_list != NULL) {
	for (i = 0; i < g_slist_length(feed_list); i++) {
	    gchar *url = NULL;

	    url = (gchar *) g_slist_nth_data(feed_list, i);

            if (url != NULL)
	        g_free(url);
	}
	g_slist_free(feed_list);
    }
}

/** Process outline tags of type rss and collapse the hieararchical structure
  * into one level
  *
  * @param cur the current node
  */
static void process_outline_tags(xmlNodePtr cur)
{
    gchar *url = NULL;
    xmlNodePtr child = NULL;
    static int depth = 0;

    /* Do not iterate the opml tree infinitely */
    if (depth >= MAX_DEPTH)
	return;

    depth++;
    
    g_assert(cur != NULL);
    
    /* process all <outline> tags */
    child = cur->xmlChildrenNode;
    while (child != NULL) {
	/* If the outline is of type rss, save the necessary data */
	if (!xmlStrcmp(child->name, BAD_CAST "outline") &&
	    ui_feed_directory_outline_type_is_rss(child)) {

	    url = utf8_fix(xmlGetProp(child, BAD_CAST "xmlUrl"));

	    if (url != NULL)
	        feed_list = g_slist_append(feed_list, url);
	}
	/* Otherwise check if the outline has children */
	else if (!xmlStrcmp(child->name, BAD_CAST "outline") &&
		 child->xmlChildrenNode != NULL) {
	    
	    process_outline_tags(child);
	}
	child = child->next;
    }
    depth--;
}

/** Parses the feed directory contents provided in OPML format
  *
  * @param doc the document to parse
  * @param cur the current node
  */
static void opml_parse(feedPtr fp, xmlDocPtr doc, xmlNodePtr cur)
{
    int error = 0;

    g_assert(doc != NULL);
    g_assert(cur != NULL);

    feed_list = NULL;
    
    feed_set_feed_directory(fp, TRUE);
    
    ULOG_DEBUG("Parsing opml document");
        
    if (xmlStrcmp(cur->name, BAD_CAST "opml") &&
	xmlStrcmp(cur->name, BAD_CAST "oml") &&
	xmlStrcmp(cur->name, BAD_CAST "outlineDocument")) {
	ULOG_DEBUG
	    ("OPML Could not find OPML header. Unable to parse feed directory.");
	xmlFreeDoc(doc);
	error = 1;
    }

    cur = cur->xmlChildrenNode;

    while (cur && xmlIsBlankNode(cur)) {
	cur = cur->next;
    }

    while (cur != NULL) {
	if (!xmlStrcmp(cur->name, BAD_CAST "body"))
	    process_outline_tags(cur);
	cur = cur->next;
    }
    
//    display_confirmation_dialog(g_slist_length(feed_list));
    app_data->app_ui_data->feed_directory_count = g_slist_length(feed_list);
    
    ULOG_DEBUG("Found %d feeds", g_slist_length(feed_list));
    feed_list_free();
}

static gboolean opml_format_check(xmlDocPtr doc, xmlNodePtr cur)
{
    g_assert(cur != NULL);
    
    if (!xmlStrcmp(cur->name, BAD_CAST "opml") ||
	!xmlStrcmp(cur->name, BAD_CAST "oml") ||
	!xmlStrcmp(cur->name, BAD_CAST "outlineDocument")) {

	return TRUE;
    }
    return FALSE;
}

/* ---------------------------------------------------------------------------- */
/* initialization								*/
/* ---------------------------------------------------------------------------- */

/************************************************************************/
/*                        PUBLIC FUNCTIONS                              */
/************************************************************************/

feedHandlerPtr opml_init_feed_handler(void)
{
    feedHandlerPtr fhp = NULL;

    fhp = g_new0(struct feedHandler, 1);

    /* prepare feed handler structure */
    fhp->typeStr = "opml";
    fhp->icon = ICON_RSS_NEWS_FEED;
    fhp->directory = FALSE;
    fhp->feedParser = opml_parse;
    fhp->checkFormat = opml_format_check;
    fhp->merge = FALSE;

    return fhp;
}
