/**
 * @file main.c Liferea main program
 *
 * Copyright (C) 2003 Lars Lindner <lars.lindner@gmx.net>
 * 
 * Some code like the command line handling was inspired by 
 *
 * Pan - A Newsreader for Gtk+
 * Copyright (C) 2002  Charles Kerr <charles@rebelbase.com>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

#include "main.h"

#include <settings.h>
#include <dbus/dbus-glib.h>

AppData *app_data;
extern GtkWidget *newsbox;
extern time_t rss_updated;
extern gboolean quit_reset_state;

GThread *mainThread = NULL;
gboolean lifereaStarted = FALSE;

/************************************************************************/
/*                        PRIVATE FUNCTIONS                             */
/************************************************************************/

/** Shows the command line parameters supported by the program
  */
static void show_help(void)
{
    GString *str = g_string_new(NULL);

    g_string_append_c(str, '\n');
    g_string_append_printf(str, "Liferea %s\n\n", VERSION);
    g_string_append_printf(str, "%s\n",
			   "  --version        Prints Liferea's version number");
    g_string_append_printf(str, "%s\n",
			   "  --help           Prints this message and exits");
    g_string_append_printf(str, "%s\n",
			   "  --iconify        Starts the program iconified");
    g_string_append_c(str, '\n');
    g_string_append_printf(str, "%s\n",
			   "  --debug-cache    Print debugging messages for the cache handling");
    g_string_append_printf(str, "%s\n",
			   "  --debug-conf     Print debugging messages of the configuration handling");
    g_string_append_printf(str, "%s\n",
			   "  --debug-update   Print debugging messages of the feed update processing");
    g_string_append_printf(str, "%s\n",
			   "  --debug-parsing  Print debugging messages of all parsing functions");
    g_string_append_printf(str, "%s\n",
			   "  --debug-gui      Print debugging messages of all GUI functions");
    g_string_append_printf(str, "%s\n",
			   "  --debug-trace    Print debugging messages when entering/leaving functions");
    g_string_append_printf(str, "%s\n",
			   "  --debug-all      Print debugging messages of all types");
    g_string_append_printf(str, "%s\n",
			   "  --debug-verbose  Print verbose debugging messages");

    g_string_append_c(str, '\n');
    ULOG_INFO("%s", str->str);
    g_string_free(str, TRUE);
}

static void initialize_app_ui_data()
{
    app_data->app_ui_data->start_on_background = FALSE;
    app_data->app_ui_data->find_string = "";
    app_data->app_ui_data->rss_search_found = 0;
    app_data->app_ui_data->automatic_update_source = 0;
    app_data->app_ui_data->download_dequeue_source = 0;
    app_data->app_ui_data->iap_action = OSSO_IAP_NO_ACTION;
    app_data->app_ui_data->errors_in_refreshing = FALSE;
    app_data->app_ui_data->add_feed_dialog = NULL;
    app_data->app_ui_data->ui_data = NULL;
    app_data->app_ui_data->source = NULL;
    app_data->app_ui_data->filter = NULL;
    app_data->app_ui_data->draggednode = NULL;
    app_data->app_ui_data->feed_directory_cancelled = FALSE;
    app_data->app_ui_data->network_iap = NULL;
    app_data->app_ui_data->application_topped = TRUE;
    app_data->app_ui_data->refresh_dialog = NULL;
    app_data->app_ui_data->dialog_closed = FALSE;
    app_data->gconf_client = NULL;
}

/** Back to sigterm approach. Its not a good idea to do a state restore
  * every time the application is topped so a full state save is not done
  * on untopping. That would require clearing the item list. A part
  * of the state is saved on untopping the program and the rest is done here */
static void sigterm_cb(int signum)
{
    (void) signum;

    g_assert(app_data != NULL);

    save_state(app_data);

    on_quit(FALSE);
}

void quit_cleanup()
{
    ULOG_DEBUG("quit cleanup");
    
	download_done();
    
    gtkhtml_deinit();

    if(app_data && app_data->app_ui_data) {
        g_object_unref(app_data->app_ui_data->button_image_search);
        g_object_unref(app_data->app_ui_data->button_image_stop);
        g_object_unref(app_data->app_ui_data->search_field);
        g_object_unref(app_data->app_ui_data->button_image_refresh);
        g_object_unref(app_data->app_ui_data->progress_bar);

        if (app_data->app_ui_data->actions) {
                osso_uri_free_actions (app_data->app_ui_data->actions);
                app_data->app_ui_data->actions = NULL;
        }
    }
    
    g_assert(app_data != NULL);
    
    //free the image cache
    if(app_data->app_ui_data != NULL)
        g_object_unref(G_OBJECT(app_data->app_ui_data->img_cache));
    
    if(quit_reset_state)
        reset_state_file(app_data->osso);
    
#ifdef USE_SM
    /* unplug */
    session_end();
#endif

    deinit_osso(app_data);
    
    if (app_data)
    {
        ULOG_DEBUG(">Freeing app_data");
        
        if(app_data->app_ui_data)
        {
            ULOG_DEBUG(">Freeing app_ui_data");
            g_free(app_data->app_ui_data);
            app_data->app_ui_data = NULL;
        }

        g_free(app_data);
        app_data = NULL;
    }

    ULOG_DEBUG("quit cleanup ended");
}

/************************************************************************/
/*                        PUBLIC FUNCTIONS                              */
/************************************************************************/

extern gboolean just_empty_items;

int main(int argc, char *argv[])
{
    gulong debug_flags = 0;
    gboolean startIconified = FALSE;
    gboolean headline_clicked = FALSE;
    const char *arg = NULL;
    gchar *filename = NULL;
    GTimer *tm = g_timer_new();
    gint i = 0;
    
    /* Allocate application data */
    app_data = g_new0(AppData, 1);
    app_data->app_ui_data = g_new0(AppUIData, 1);

    g_assert(app_data != NULL);
    g_assert(app_data->app_ui_data != NULL);
    
    initialize_app_ui_data();
    
    ULOG_DEBUG("Application starting");
    ULOG_DEBUG("Time0: %f", g_timer_elapsed(tm, NULL));

    signal( SIGTERM, sigterm_cb );

    /* Use this safety patch from bug 25738 */
    if (!g_thread_supported()) {
      g_thread_init(NULL);
    }
    dbus_g_thread_init();
    gdk_threads_init();
    gdk_threads_enter();

    init_osso(app_data);
    
    ULOG_DEBUG("Time (after init_osso): %f", g_timer_elapsed(tm, NULL));
    filename =
	g_strdup_printf("%s" G_DIR_SEPARATOR_S ".osso_rss_feed_reader"
			G_DIR_SEPARATOR_S "background", g_get_home_dir());
    if (g_file_test(filename, G_FILE_TEST_EXISTS)) {
	startIconified = TRUE;
	app_data->app_ui_data->start_on_background = TRUE;
	g_unlink(filename);
    }
    g_free(filename);

    filename =
	g_strdup_printf("%s" G_DIR_SEPARATOR_S ".osso_rss_feed_reader"
			G_DIR_SEPARATOR_S "headline_clicked", g_get_home_dir());
    if (g_file_test(filename, G_FILE_TEST_EXISTS)) {
	headline_clicked = TRUE;
	g_unlink(filename);
    }
    g_free(filename);
    
#ifdef USE_SM
    gchar *opt_session_arg = NULL, *opt_config_dir_arg = NULL;
#endif

#ifdef ENABLE_NLS
    setlocale(LC_ALL, "");
    bindtextdomain(GETTEXT_PACKAGE, PACKAGE_LOCALE_DIR);
    bind_textdomain_codeset(GETTEXT_PACKAGE, "UTF-8");
    textdomain(GETTEXT_PACKAGE);
#endif
    gnome_vfs_init();
    ULOG_DEBUG("Time (after i10n): %f", g_timer_elapsed(tm, NULL));
    /* parse arguments  */
    debug_flags = 0;
/*	debug_flags = DEBUG_TRACE|DEBUG_CACHE|DEBUG_CONF|DEBUG_UPDATE|DEBUG_PARSING|DEBUG_GUI; */
    for (i = 1; i < argc; ++i) {
	arg = argv[i];

	if (!strcmp(arg, "--debug-cache"))
	    debug_flags |= DEBUG_CACHE;
	else if (!strcmp(arg, "--debug-conf"))
	    debug_flags |= DEBUG_CONF;
	else if (!strcmp(arg, "--debug-update"))
	    debug_flags |= DEBUG_UPDATE;
	else if (!strcmp(arg, "--debug-parsing"))
	    debug_flags |= DEBUG_PARSING;
	else if (!strcmp(arg, "--debug-gui"))
	    debug_flags |= DEBUG_GUI;
	else if (!strcmp(arg, "--debug-trace"))
	    debug_flags |= DEBUG_TRACE;
	else if (!strcmp(arg, "--debug-all"))
	    debug_flags |=
		DEBUG_TRACE | DEBUG_CACHE | DEBUG_CONF | DEBUG_UPDATE |
		DEBUG_PARSING | DEBUG_GUI;
	else if (!strcmp(arg, "--debug-verbose"))
	    debug_flags |= DEBUG_VERBOSE;
	else if (!strcmp(arg, "--version") || !strcmp(arg, "-v")) {
	    ULOG_INFO("liferea %s\n", VERSION);
	    return 0;
	} else if (!strcmp(arg, "--help") || !strcmp(arg, "-h")) {
	    show_help();
	    return 0;
	} else if (!strcmp(arg, "--empty"))
	    just_empty_items = TRUE;
	else if (!strcmp(arg, "--iconify")) {
	    startIconified = TRUE;
        app_data->app_ui_data->start_on_background = TRUE;
#ifdef USE_SM
	} else if (!strcmp(arg, "--session")) {
	    i++;
	    if (i < argc) {
		opt_session_arg = g_strdup(argv[i]);
	    } else
		ULOG_DEBUG
		    ("The --session argument must be given a parameter.\n");
#endif
	} else {
	    ULOG_DEBUG("Liferea encountered an unknown argument: %s\n",
		       arg);
	}
    }
    set_debug_level(debug_flags);

    ULOG_DEBUG("Time (after set_debug_level): %f", g_timer_elapsed(tm, NULL));
    gtk_set_locale();

    
    gtk_init(&argc, &argv);

    ULOG_DEBUG("Time (after gtk_init): %f", g_timer_elapsed(tm, NULL));

    mainThread = g_thread_self();	/* we need to now this when locking in ui_queue.c */

    /* order is important! */
    conf_init();		/* initialize gconf */
    ULOG_DEBUG("Time (after conf_init): %f", g_timer_elapsed(tm, NULL));
#ifdef USE_SM
    session_init(BIN_DIR G_DIR_SEPARATOR_S "liferea", opt_session_arg);
    session_set_cmd(NULL, startIconified);
#endif
/* We don't use plugin architecture because of performance concerns */

    download_init();		/* Initialize the download subsystem */
    ULOG_DEBUG("Time (after download init): %f", g_timer_elapsed(tm, NULL));

    gtkhtml_init();
    ULOG_DEBUG("Time (after gtkhtml_init): %f", g_timer_elapsed(tm, NULL));

    metadata_init();
    ULOG_DEBUG("Time (after metadata_init): %f", g_timer_elapsed(tm, NULL));

    feed_init();		/* register feed types */
    ULOG_DEBUG("Time (after feed_init): %f", g_timer_elapsed(tm, NULL));

//    general_load();		/* retrieve stored general data */
    ULOG_DEBUG("Time (after general_load): %f", g_timer_elapsed(tm, NULL));

    /* Create application UI */
    ui_create_main_window(app_data);
    ULOG_DEBUG("Time (after ui_create_main_window): %f", g_timer_elapsed(tm, NULL));

    ui_mainwindow_load_search_entries();
    ULOG_DEBUG("Time (after ui_mainwindow_load_search_entries): %f", g_timer_elapsed(tm, NULL));
            
    g_assert(app_data != NULL);
    g_assert(app_data->app_ui_data != NULL);
    
    ui_init(startIconified, app_data->app_ui_data);	/* setup mainwindow and initialize gconf configured GUI behaviour */
    ULOG_DEBUG("Time (after ui_init): %f", g_timer_elapsed(tm, NULL));
    ULOG_DEBUG("tvh main.c: ui_init done ");
    /* tvh: commented this out. This registers 'windows-1252 encoding handler
    This hangs with feeds http://gravyboy.com/fanboyalmanac/FanboyAlmanacFeed.xml
     */
    //register_encoding_handlers();
    ULOG_DEBUG("tvh main.c: encode handler registered ");

    g_assert(app_data != NULL);
    ULOG_DEBUG("tvh main.c: app is not NULL ");

    if(!startIconified)
    {
        ui_feedlist_delete_read_items(NULL);    
        /* Refresh the applet since the feed contents might have been changed */	
        rss_dbus_refresh_completed();
        restore_state(app_data);
        app_data->app_ui_data->auto_refresh_is_ongoing=FALSE;
    }

    ui_folder_handle_empty_root();
    ULOG_DEBUG("Time (after ui_folder_handle_empty_root): %f", g_timer_elapsed(tm, NULL));
    ULOG_DEBUG("tvh main.c: ui_folder_handle empty root done ");

    setup_alarm_event(FALSE);
    
    
    lifereaStarted = TRUE;
    ULOG_DEBUG("\n\n\n****\nGtk_main now \n****\n\n");
    gtk_main();
    ULOG_DEBUG("\n\n\n****\nGtk_main exited \n****\n\n");
    gdk_threads_leave();
	
    quit_cleanup();

    ULOG_DEBUG("Clean exit");
    
    return 0;
}

/* Function to keep the "new" items, and delete the old items*/
void filter_on_exit(feedPtr fp)
{
	ui_feed_set_new_items_hidden(fp);
	ui_feed_delete_old_items(fp);
	ui_feed_remove_hidden_attribute(fp);
	feed_save(fp);
}

void save_application_data()
{
    conf_feedlist_save();	/* should save feedlist and folder states */
	
  
	/*Delete the old items, keep the news, //only, if the app was not started by the alarmd*/
  //if (!app_data->app_ui_data->start_on_background)
	ui_feedlist_do_for_all(NULL,
			   ACTION_FILTER_FEED | ACTION_FILTER_DIRECTORY,
			   filter_on_exit);

    /* should save all feeds still in memory */
    ui_feedlist_do_for_all(NULL,
			   ACTION_FILTER_FEED | ACTION_FILTER_DIRECTORY,
			   feed_unload);

    /* save itemlist properties */
    setBooleanConfValue(LAST_ITEMLIST_MODE, !itemlist_mode);

    /* save pane proportions */
    ULOG_DEBUG("\n\n\n\n!!!!!!!!!!!!!!!!Saving pane position 1");
    if (GTK_IS_WIDGET(newsbox)) {
            gint x = gtk_paned_get_position(GTK_PANED(GTK_PANED(newsbox)));
            ULOG_DEBUG("Pane position: %d", x);
            setNumericConfValue(LAST_VPANE_POS, x);
    }

}
