/**
 * @file htmlview.c common interface for browser module implementations
 * and module loading functions
 *
 * Copyright (C) 2003, 2004 Lars Lindner <lars.lindner@gmx.net>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

#ifdef HAVE_CONFIG_H
#  include <config.h>
#endif

#include <sys/stat.h>
#include <string.h>
#include <glib.h>
#include <gmodule.h>

#include <settings.h>

#include "common.h"
#include "conf.h"
#include "callbacks.h"
#include "ui_queue.h"
#include "support.h"
#include "htmlview.h"
#include "debug.h"


#include <ossoemailinterface.h>
#include <osso-log.h>

extern AppData *app_data;

/* function types for the imported symbols */
typedef htmlviewPluginInfo *(*infoFunction) ();
htmlviewPluginInfo *htmlviewInfo = NULL;

//GSList *availableBrowserModules = NULL;

static GModule *handle;

extern GtkWidget *mainwindow;

extern char *proxyname;
extern char *proxyusername;
extern char *proxypassword;
extern int proxyport;


//extern gchar * rendered_content;
//extern GtkWidget * gtkhtml_htmlview;
/* -------------------------------------------------------------------- */
/* module loading and initialisation					*/
/* -------------------------------------------------------------------- */

/************************************************************************/
/*                        PRIVATE FUNCTIONS                             */
/************************************************************************/

/** Method which tries to load the functions listed in the array
  * symbols from the specified module name libname.  If testmode
  * is true no error messages are issued. The function returns
  * TRUE on success. 
  *
  * @param libname the module
  * @param testmode used to control debugging output
  * @return TRUE on success
  */
static inline gboolean ui_htmlview_load_symbols(gchar * libname,
					 gboolean testmode)
{
    testmode = FALSE;
    infoFunction ptr = NULL;
    gchar *filename = NULL;

    filename =
	g_strdup_printf("%s%s%s", PACKAGE_LIB_DIR, G_DIR_SEPARATOR_S,
			libname);
    ULOG_INFO("loading HTML widget module (%s)", filename);

    if ((handle = g_module_open(filename, 0)) == NULL) {
	if (!testmode)
	    g_warning
		("Failed to open HTML widget module (%s) specified in configuration!\n%s\n",
		 filename, g_module_error());
	else
	    debug2(DEBUG_GUI,
		   "Failed to open HTML widget module (%s) specified in configuration!\n%s\n",
		   filename, g_module_error());
	g_free(filename);
	return FALSE;
    }
    g_free(filename);

    if (g_module_symbol(handle, "htmlview_plugin_getinfo", (void *) &ptr)) {
	htmlviewInfo = (*ptr) ();
	if (htmlviewInfo->api_version != HTMLVIEW_API_VERSION) {
	    if (!testmode)
		g_warning("Htmlview API mismatch!");
	    else
		debug0(DEBUG_GUI, "Htmlview API mismatch!");
	    g_module_close(handle);
	    return FALSE;
	}
    } else {
	if (!testmode)
	    g_warning("Detected module is not a valid htmlview module!");
	else
	    debug0(DEBUG_GUI,
		   "Detected module is not a valid htmlview module!");
	g_module_close(handle);
	return FALSE;
    }

    return TRUE;
}

/************************************************************************/
/*                        PUBLIC FUNCTIONS                              */
/************************************************************************/

/* function to load the module specified by module */
/* seems not to be used
void ui_htmlview_init(void)
{
    return;
    gboolean success = FALSE;
    gint filenamelen = 0;
    gchar *filename = NULL;
    struct browserModule *info = NULL;
    GSList *tmp = NULL;
    GError *error = NULL;
    GDir *dir = NULL;
*/
    /* Check to see if gmodule is supported */
/*
    if (!g_module_supported())
	ULOG_ERR("Cannot load HTML widget module (%s)!", g_module_error());
*/
    /* now we determine a list of all available modules
       to present in the preferences dialog and to load
       one just in case there was no configured module
       or it did not load when trying... */
/*
    ULOG_INFO("Available browser modules (%s):", PACKAGE_LIB_DIR);
    dir = g_dir_open(PACKAGE_LIB_DIR, 0, &error);
    if (!error) {
*/    
	/* maybe no good solution, library name syntax: 
	   liblihtml<one letter code>.<library extension> */
/*
	filenamelen = 11 + strlen(G_MODULE_SUFFIX);
	filename = (gchar *) g_dir_read_name(dir);
	while (NULL != filename) {
	    if ((filenamelen == strlen(filename))
		&& (0 == strncmp("liblihtml", filename, 9))) {
		ULOG_INFO("trying %s ... ", filename);
*/
		/* now lets filter the files with correct library suffix */
/*
		if (0 ==
		    strncmp(G_MODULE_SUFFIX, filename + 11,
			    strlen(G_MODULE_SUFFIX))) {
		    ULOG_INFO("trying %s, filename suffix correct.",
			    filename);
*/			    
		    /* if we find one, try to load all symbols and if successful
		       add it to the available module list */
/*
		    if (TRUE == ui_htmlview_load_symbols(filename, TRUE)) {
			ULOG_INFO("trying %s, symbols loaded ok.",
				filename);
			info = g_new0(struct browserModule, 1);
			info->libname = g_strdup(filename);
			info->description = g_strdup(htmlviewInfo->name);
			availableBrowserModules =
			    g_slist_append(availableBrowserModules,
					   (gpointer) info);
			ULOG_INFO("-> %s (%s)", info->description,
				info->libname);
			g_module_close(handle);
		    } else {
			ULOG_INFO
			    ("did not try %s, symbols loading failed.",
			     filename);
		    }

		} else {
		    ULOG_INFO("did not try %s, filename suffix wrong.",
			    filename);
		}
	    } else {
		ULOG_INFO("did not try %s, filename did not match.",
			filename);
	    }
	    filename = (gchar *) g_dir_read_name(dir);
	}
	g_dir_close(dir);
    } else {
	g_warning("g_dir_open(%s) failed. Reason: %s\n", PACKAGE_LIB_DIR,
		  error->message);
	g_error_free(error);
	error = NULL;
    }
*/
    /* load configured module, we get a empty string if nothing is configured */
/*
    filename = getStringConfValue(BROWSER_MODULE);
    if (0 != strlen(filename)) {
	ULOG_INFO("Loading configured browser module (%s)!", filename);
	success = ui_htmlview_load_symbols(filename, FALSE);
    } else {
	ULOG_INFO("No browser module configured!");
    }
    g_free(filename);
    if (!success) {
*/    
	/* try to load one of the available modules */
/*
	tmp = availableBrowserModules;
	while (NULL != tmp) {
	    info = (struct browserModule *) tmp->data;
	    ULOG_INFO("trying to load browser module %s (%s)",
		    info->description, info->libname);
	    if (TRUE ==
		(success = ui_htmlview_load_symbols(info->libname, FALSE)))
		break;
	    tmp = g_slist_next(tmp);
	}
    }

    if (success) {
	htmlviewInfo->init();
	ui_htmlview_set_proxy(proxyname, proxyport, proxyusername,
			      proxypassword);
    } else
	ULOG_ERR
	    ("Sorry, I was not able to load any installed browser modules!"
	      "Try the --debug-all option to get debug information!");
}
*/
/* unused
void ui_htmlview_deinit()
{
    (htmlviewInfo->deinit) ();
}
*/
/* -------------------------------------------------------------------- */
/* browser module interface functions					*/
/* -------------------------------------------------------------------- */

GtkWidget *ui_htmlview_new()
{
    GtkWidget *htmlview = NULL;
    if (NULL != htmlviewInfo) {
    	htmlview = htmlviewInfo->create();
    } else {
       ULOG_DEBUG("tvh: htmlview.c: into gtkhtml now......");
	   htmlview = gtkhtml_new();
    }

    //ui_htmlview_clear(htmlview); tvh: IMPORTANT: todo 

    return htmlview;
}

void ui_htmlview_write(GtkWidget * htmlview, gchar * string,
		             gchar * base, nodePtr node)
{
    //g_assert(string != NULL);
    //g_assert(htmlview != NULL);
    ULOG_DEBUG("ui_htmlview_write: 1");
    /*
    if (base == NULL)
    	base = g_strdup("file:///");
    */
    if (string != NULL)
    {
        ULOG_DEBUG("ui_htmlview_write: 1.5");
       if (!g_utf8_validate((const gchar *) string, -1, NULL)) {

	      /* Its really a bug if we get invalid encoded UTF-8 here!!! */
	      g_warning("Invalid encoded UTF8 buffer passed to HTML widget!");

           gchar *buffer = g_strdup(string);
    
    
	      /* to prevent crashes inside the browser */
	      buffer = utf8_fix(buffer);
	      if (NULL != htmlviewInfo) 
          {  
	          (htmlviewInfo->write) (htmlview, buffer, base, node);
    	      g_free(buffer);

                return;
          }
	      else 
            {
	          gtkhtml_write_html(htmlview, buffer, base, node);
    	      g_free(buffer);

            return;
            }
       }
    }
    if (NULL != htmlviewInfo) {
        ULOG_DEBUG("ui_htmlview_write: 2");
	   (htmlviewInfo->write) (htmlview, string, base, node);
    } else {
            ULOG_DEBUG("ui_htmlview_write: 3");
	   gtkhtml_write_html(htmlview, string, base, node);
    }
}

GtkWidget *ui_htmlview_get_buffer()
{
	ULOG_DEBUG("Entering ui_htmlview_get_buffer\n");
    if (NULL != htmlviewInfo) {
		ULOG_DEBUG("With function\n");
	return (htmlviewInfo->get_buffer) ();
    } else {
	//return textview_get_buffer();
		ULOG_DEBUG("with text\n");
    //return rendered_content;
    return NULL;
    }
	ULOG_DEBUG("Leaving ui_htmlview_get_buffer\n");
}

void ui_htmlview_clear(GtkWidget * htmlview)
{
    gchar *buffer = strdup("");

    ui_htmlview_write(htmlview, buffer, NULL, NULL);
    g_free(buffer);
}

gboolean ui_htmlview_scroll()
{
    gboolean result = FALSE;

    if (NULL != htmlviewInfo) {
	result =
	    htmlviewInfo->
	    scrollPagedown(ui_mainwindow_get_active_htmlview());
    } else {
	result =
	    gtkhtml_scroll_pagedown(ui_mainwindow_get_active_htmlview
					());
    }

    return result;
}

void ui_htmlview_set_proxy(gchar * hostname, int port, gchar * username,
			   gchar * password)
{
    if (htmlviewInfo != NULL && htmlviewInfo->setProxy != NULL)
	htmlviewInfo->setProxy(hostname, port, username, password);
}

int send_text_via_email(const gchar *post)
{
    g_assert(NULL != post);
    g_assert(app_data != NULL);
    
    ULOG_DEBUG("Calling osso_email_url_email()");
    osso_return_t retcode =
	osso_email_url_email(app_data->osso,
			     post);

    /* CHECK: There is no error feedback defined in UI spec. */
    switch (retcode) {
    case OSSO_OK:
	ULOG_DEBUG("osso_email_url_email() returned OK!\n");
	break;
    case OSSO_ERROR:
	ULOG_DEBUG("osso_email_url_email() returned ERROR!\n");
	break;
    case OSSO_INVALID:
	ULOG_DEBUG("osso_email_url_email() returned INVALID!\n");
	break;
    case OSSO_RPC_ERROR:
	ULOG_DEBUG("osso_email_url_email() returned OSSO_RPC_ERROR!\n");
	break;
    default:
	ULOG_DEBUG("osso_email_url_email() returned %d!\n", retcode);
	break;
    }

    return 0;
}

int send_item(itemPtr item, gboolean bluetooth)
{
    gchar *post = NULL;
    gchar *esc = NULL;
    gchar *desc = NULL;
    
    g_assert(NULL != item);

    g_assert(app_data != NULL);
    g_assert(app_data != NULL);
    
    if (item->source)
        esc = g_markup_escape_text(item->source,-1);
    else
        esc = g_markup_escape_text(item->title,-1);
    
    desc = unhtmlize(g_strdup(item->description) );
    
    post = g_strconcat(esc,"\n\n", desc, NULL);

    post = g_strstrip(post);
    
    if (bluetooth)
        bt_send(item->title, post, strlen(post));
    else
        send_text_via_email(post);
    
    g_free(post);
    g_free(esc);
    g_free(desc);
    
    return 0;
}

/* -------------------------------------------------------------------- */
/* htmlview callbacks 							*/
/* -------------------------------------------------------------------- */

void ui_htmlview_render_item(itemPtr ip)
{
    if (NULL != htmlviewInfo) {
	(htmlviewInfo->render_item) (ip);
    } else {
        ULOG_DEBUG("ui_htmlview_render_item: appending another ip to search view ");
        gtkhtml_searchview_append_item(ip);
    }
}
