/**
 * @file feed.h common feed handling
 * 
 * Copyright (C) 2003 Lars Lindner <lars.lindner@gmx.net>
 * Copyright (C) 2004 Nathan J. Conrad <t98502@users.sourceforge.net>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version. 
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

#ifndef _FEED_H
#define _FEED_H

#include <glib.h>
#include "common.h"
#include "item.h"
#include "folder.h"

#define DEFAULT_MAX_ITEMS 100

/* ------------------------------------------------------------ */
/* feed list view entry types (FS_TYPE) 			*/
/* ------------------------------------------------------------ */

enum node_types {
    FST_INVALID = 0,			/**< invalid type */
    FST_FOLDER = 1,			/**< the folder type */

    FST_VFOLDER = 9,			/**< special type for VFolders */
    FST_FEED = 10,			/**< Any type of feed */
};

/** Flags used in the request structure */
enum feed_request_flags {
    FEED_REQ_SHOW_PROPDIALOG = 1,
    /**< Feed's title should be reset to default upon update */
    FEED_REQ_RESET_TITLE = 2,
    /**< Feed's update interval should be reset to default upon update */		
    FEED_REQ_RESET_UPDATE_INT = 4,
    /**< Feed auto-discovery attempts should be made */
    FEED_REQ_AUTO_DISCOVER = 8,		

    /**< set to signal that this is an important user triggered request */
    FEED_REQ_PRIORITY_HIGH = 16,	
    /**< set to make the favicon be updated after the feed is downloaded */
    FEED_REQ_DOWNLOAD_FAVICON = 32,	
    /**< set to make an auth request dialog to be created after 401 errors */
    FEED_REQ_AUTH_DIALOG = 64,	      
};

struct feedhandler;
struct request;
/* ------------------------------------------------------------ */
/* Feed structure                                               */
/* ------------------------------------------------------------ */

enum cache_limit {
    /* Values > 0 are used to specify certain limits */
    CACHE_DISABLE = 0,
    CACHE_DEFAULT = -1,
    CACHE_UNLIMITED = -2,
};

/** common structure to access feed info structures */

typedef struct feed {
    gint type;					/**< feed type (first position is important!!!) */
    gpointer *ui_data;				/**< per-feed UI data (second position is important!!!) */

    struct feedHandler *fhp;			/**< Feed handler saved by the ->typeStr attribute. */

    gchar *id;					/**< unique feed identifier string */
    gint unreadCount;				/**< number of unread items */
    gint markCount;				/**< number of marked items */
    gint newCount;				/**< number of new items */
    gint defaultInterval;			/**< update interval as specified by the feed */
    gint loaded;				/**< counter which is non-zero if items are to be kept in memory */
    gboolean needsCacheSave;			/**< flag set when the feed's cache needs to be resaved */
    gchar *parseErrors;				/**< textual/HTML description of parsing errors */
    gchar *errorDescription;			/**< textual/HTML description of download/parsing errors */

    gpointer icon;				/**< pointer to pixmap, if there is a favicon */

    time_t addedTime;				/**< Feeds adding date */
    time_t time;				/**< Feeds modified date */
    GHashTable *tmpdata;			/**< tmp data hash used during stateful parsing */

    gboolean new_subscription;                  /**< This is set when a new feed is added from the Add Feed -dialog
                                                     and an internet connect has been established */
    gboolean feed_directory;                    /**< This is set if the feed is a feed directory */
    /* feed properties needed to be saved */
    gboolean available;				/**< flag to signalize loading errors */
    gboolean discontinued;			/**< flag to avoid updating after HTTP 410 */

    gchar *title;				/**< feed/channel title */
    gchar *htmlUrl;				/**< URL of HTML version of the feed */
    gchar *imageUrl;				/**< URL of the optional feed image */
    gchar *description;				/**< HTML string describing the feed */
    gchar *source;				/**< feed source */
    gchar *filtercmd;				/**< feed filter command */
    gint updateInterval;			/**< user defined update interval in minutes */
    GSList *metadata;				/**< metadata of this feed */

    gulong lastItemNr;				/**< internal counter used to uniqely assign item id's. */
    GSList *items;				/**< list of pointers to the item structures of this channel */

    GSList *rules;				/**< list of rules if this is a vfolder */
    time_t newestPost;                          /**< The newest post in the feed */
    time_t feedRead;                            /**< The time the feed has been read */
    
    /* feed properties used for updating */
    gchar *lastModified;		/**< Last modified string as sent by the server */
    gchar *etag;			/**< E-Tag sent by the server */
    GTimeVal lastPoll;				/**< time at which the feed was last updated */
    GTimeVal lastFaviconPoll;			/**< time at which the feed was last updated */
    struct request *request;			/**< update request structure used when downloading xml content */
    GSList *otherRequests;			/**< list of other update request structures used for 
                                                     downloading anything (favicon, blogChannel stuff, ...) */
    gint cacheLimit;				/**< Amount of cache to save: See the cache_limit enum */
    gboolean noIncremental;			/**< Do merging for this feed but drop old items */
    gint sortColumn;				/**< Sorting column. Set to either IS_TITLE, or IS_TIME */
    gboolean sortReversed;			/**< Sort in the reverse order? */
    gboolean nonremovable;		/**< Is the removing of the feed disabled. */
    gboolean err;               /**< Set to TRUE on parse errors, FALSE otherwise */
    
    gboolean downloadImages;    /**< if true the images are downloaded on the feed */
} *feedPtr;

/* ------------------------------------------------------------ */
/* feed handler interface					*/
/* ------------------------------------------------------------ */

/** a function which parses the feed data given with the feed ptr fp */
typedef void (*feedParserFunc) (feedPtr fp, xmlDocPtr doc, xmlNodePtr cur);
typedef gboolean(*checkFormatFunc) (xmlDocPtr doc, xmlNodePtr cur);	 /**< Returns true if correct format */

typedef struct feedHandler {
    const gchar *typeStr;		/**< string representation of the feed type */
    int icon;				/**< Icon number used for available feeds/directories */
    gboolean directory;			/**< Determines if a feed should be autoupdated and updated 
                                             when "update all" is selected */
    feedParserFunc feedParser;		/**< feed type parse function */
    checkFormatFunc checkFormat;	/**< Parser for the feed type*/
    gboolean merge;			/**< flag if feed type supports merging */

} *feedHandlerPtr;

/* ------------------------------------------------------------ */
/* feed creation/modification interface				*/
/* ------------------------------------------------------------ */

/** 
 * Initializes feed parsing handlers. Should be called 
 * only once on program startup.
 */
void feed_init(void);

/**
 * Create a new feed structure.
 * @returns the new, empty feed
 */
feedPtr feed_new(void);

/**
 * General feed source parsing function. Parses the passed feed source
 * and tries to determine the source type. If the type is HTML and 
 * autodiscover is TRUE the function tries to find a feed, tries to
 * download it and parse the feed's source instead of the passed source.
 *
 * @param fp		the feed structure to be filled
 * @param data		the feed source
 * @param dataLength the length of the 'data' string
 * @param autodiscover	TRUE if auto discovery should be possible
 */
feedHandlerPtr feed_parse(feedPtr fp, gchar * data, size_t dataLength,
			  gboolean autodiscover);

/**
 * Loads a feed from a cache file.
 *
 * @param type the type of feed being loaded. This effects the
 * extension of the cache file.
 *
 * @param id the name of the cache file used. Some types of feed have
 * an extension, such as ocs, that is appended to the id, to generate
 * the cache filename.
 *
 * @returns FALSE if the feed file could not be opened and TRUE if it
 * was opened or was already loaded.
 */
gboolean feed_load(feedPtr fp);

/** Only some feed informations are kept in memory to lower memory
  * usage. This method unloads everything besides necessary infos.
  *
  * @param fp a pointer to a feed structure
  */
gboolean feed_unload(feedPtr fp);

void feed_merge(feedPtr old_fp, feedPtr new_fp);
void feed_remove(feedPtr fp);
void feed_schedule_update(feedPtr fp, gint flags);
gboolean feed_save(feedPtr fp);

/**
 * Can be used to add a single item to a feed. But it's better to
 * use feed_add_items() to keep the item order of parsed feeds.
 * Should be used for vfolders only.
 */
void feed_add_item(feedPtr fp, itemPtr ip);

/**
 * To be used by parser implementation to merge a new orderd list of
 * items to a feed. Ensures properly ordered joint item list. The
 * passed GList is free'd afterwards!
 */
void feed_add_items(feedPtr fp, GList * items);

void feed_remove_item(feedPtr fp, itemPtr ip);

/** 
 * To lookup an item given by it's unique item nr 
 */
itemPtr feed_lookup_item_by_nr(feedPtr fp, gulong nr);

/** 
 * To lookup an item given by it's unique item nr 
 */
itemPtr feed_lookup_item_by_nr_extended(feedPtr fp, gulong nr, gchar *orig_fp_id);

/** 
 * To lookup an item given by it's unique item id 
 * TODO: This might not be needed
 */
//itemPtr feed_lookup_item_by_id(feedPtr fp, gchar * id, gchar * orig_fp_id);

/**
 * Mark a feeds update requests as canceled.
 */
void feed_free_requests(nodePtr ptr);

void feed_free(feedPtr fp);

/**
 * This is a timeout callback to check for feed update results.
 * If there is a result pending its data is parsed and merged
 * against the feed it belongs to.
 */
void feed_process_update_result(struct request *request);

/**
 * Stores general data about the feed cache for applet.
 */
void general_save();

/**
 * Retrieves general data about the feed cache stored for applet.
 */
void general_load();

/* ------------------------------------------------------------ */
/* feed property get/set 					*/
/* ------------------------------------------------------------ */

gpointer feed_get_favicon(feedPtr fp);

/**
 * Sets the type of feed
 * @param fp feed to modify
 * @param type type to set
 */
void feed_set_type(feedPtr fp, int type);
gint feed_get_type(feedPtr fp);

/**
 * Lookup a feed type string from the feed type number
 */
feedHandlerPtr feed_type_str_to_fhp(const gchar * str);
const gchar *feed_type_fhp_to_str(feedHandlerPtr fhp);

void feed_increase_unread_counter(feedPtr fp);
void feed_decrease_unread_counter(feedPtr fp);
gint feed_get_unread_counter(feedPtr fp);

void feed_increase_new_counter(feedPtr fp);
void feed_decrease_new_counter(feedPtr fp);
gint feed_get_new_counter(feedPtr fp);

void feed_increase_mark_counter(feedPtr fp);
void feed_decrease_mark_counter(feedPtr fp);
gint feed_get_mark_counter(feedPtr fp);

gint feed_get_default_update_interval(feedPtr fp);
void feed_set_default_update_interval(feedPtr fp, gint interval);

gint feed_get_update_interval(feedPtr fp);
void feed_set_update_interval(feedPtr fp, gint interval);

void feed_reset_update_counter(feedPtr fp);

gboolean feed_get_available(feedPtr fp);
void feed_set_available(feedPtr fp, gboolean available);

gboolean feed_get_download_images(feedPtr fp);
void feed_set_download_images(feedPtr fp, gboolean download);

gboolean feed_get_discontinued(feedPtr fp);
void feed_set_discontinued(feedPtr fp, gboolean discontinued);

gboolean feed_get_nonremovable(feedPtr fp);
void feed_set_nonremovable(feedPtr fp, gboolean nonremovable);

/**
 * Returns a HTML string describing the last retrieval error 
 * of this feed. Should only be called when getFeedAvailable
 * returns FALSE.
 *
 * @param fp		feed
 * @return HTML error description
 */
gchar *feed_get_error_description(feedPtr fp);

/** Get feed id
  *
  * @param fp the feed to get the id for
  */
const gchar *feed_get_id(feedPtr fp);

/** Set a new id for a feed
  *
  * @param fp the feed pointer for which to set the id
  * @param id the new id
  */
void feed_set_id(feedPtr fp, const gchar * id);

/** Get the modified time for a feed
  *
  * @param fp the feed to get the modified time for
  * @return the modified time
  */
const time_t feed_get_time(feedPtr fp);

/** Set the modified time for a feed
  *
  * @param fp the feed to set the time for
  * @param time the new time
  */
void feed_set_time(feedPtr fp, time_t time);

/** Get the title of a feed
  *
  * @param fp the feed to get the title for
  * @return the title
  */
const gchar *feed_get_title(feedPtr fp);

/** Set a new title for a feed
  *
  * @param fp the feed to set the title for
  * @param title the new title
  */
void feed_set_title(feedPtr fp, const gchar * title);

/** Get the description of a feed
  *
  * @param the feed to get the description for
  */
const gchar *feed_get_description(feedPtr fp);

/** Set the description for a feed
  *
  * @param fp the feed to set the description for
  * @param description the new description
  */
void feed_set_description(feedPtr fp, const gchar * description);

/** Get the source of a feed
  *
  * @param fp the feed to get the source for
  * @return the source of the feed
  */
const gchar *feed_get_source(feedPtr fp);

/** Set a source for a feed
  *
  * @param fp the feed to set the source for
  * @param source the new source
  */
void feed_set_source(feedPtr fp, const gchar * source);

/** Get the filter command for a feed
  *
  * @param fp the feed to get the filter command for
  * @return the filter command
  */
const gchar *feed_get_filter(feedPtr fp);

/** Set a filter command for a feed
  *
  * @param fp the feed to set the filter command for
  * @param filter the new filter command
  */
void feed_set_filter(feedPtr fp, const gchar * filter);

/** Get the url of the html version of the feed
  *
  * @param fp the feed to get the url for
  * @return the url of the html version of the feed
  */
const gchar *feed_get_html_url(feedPtr fp);

/** Sets the url of the html version of the feed
  *
  * @param fp the feed to set the url for
  * @param url the new url
  */
void feed_set_html_url(feedPtr fp, const gchar * url);

/** Get the url of the feed image
  *
  * @param fp the feed to get the url for
  * @return the url of the feed image
  */
const gchar *feed_get_image_url(feedPtr fp);

/** Set the feed image url
  *
  * @param fp the feed to set the url for
  * @param url the new url
  */
void feed_set_image_url(feedPtr fp, const gchar * url);

/** Get the added time of a feed
  *
  * @param fp the feed to get the added time for
  * @return the added time
  */
const time_t feed_get_added(feedPtr fp);

/** Sets the added time for a feed
  *
  * @param fp the feed to set the added time for
  * @param added the new added time
  */
void feed_set_added(feedPtr fp, const time_t added);

/** Gets the last modified string for a feed
  *
  * @param fp the feed to get the last modified string for
  * @return the last modified string
  */
const gchar *feed_get_lastmodified(feedPtr fp);

/** Sets the last modified string for a feed
  *
  * @param fp the feed to set the last modified string for
  * @param lastmodified the new last modified string
  */
void feed_set_lastmodified(feedPtr fp, const gchar * lastmodified);

/** Gets the etag for a feed
  *
  * @param fp the feed to get the etag for
  * @return the etag of the feed
  */
const gchar *feed_get_etag(feedPtr fp);

/** Sets the etag for a feed
  *
  * @param fp the feed to set the etag for
  * @param etag the new etag
  */
void feed_set_etag(feedPtr fp, const gchar * etag);

/** Get the feed handler for a feed
  *
  * @param fp the feed to get the feed handler for
  * @return the feed handler
  */
const feedHandlerPtr feed_get_fhp(feedPtr fp);

/** Gets the item list for a feed
  *
  * @param fp the feed to get the item list for
  * @return the item list for the feed
  */
GSList *feed_get_item_list(feedPtr fp);

/** Set the sort colum for a feed
  *
  * @param fp the feed to set the sort column for
  * @param sortColumn the new sort column for the feed, either IS_TITLE or IS_TIME
  * @param reversed set to TRUE if sorting should be done in reverse order
  */
void feed_set_sort_column(feedPtr fp, gint sortColumn, gboolean reversed);

/** Method to free all items structures of a feed, does not mean
  * that it removes items from cache! This method is used 
  * for feed unloading.
  *
  * @param fp the feed to clear the item list for
  */
void feed_clear_item_list(feedPtr fp);

/** Method to permanently remove all items from the given feed
  *
  * @param fp the feed to remove the items from
  */
void feed_remove_items(feedPtr fp);

/** Method to permanently remove unmarked items from feed
  *
  * @param fp the feed to remove the unmarked items from
  * @param save_unread if unread posts should be saved
  */
void feed_remove_unmarked(feedPtr fp, gboolean save_unread);

/** Marks all the items of a feed read
  *
  * @param fp the feed from which to mark the items
  */
void feed_mark_all_items_read(feedPtr fp);

/** Returns a HTML rendering of a feed. The returned string must be
  * freed
  *
  *  @param fp the feed to get the HTML rendering for
  */
gchar *feed_render(feedPtr fp);

/** Gets the newest post of the feed
  *
  * @param fp the feed to get the newest post for
  */
time_t feed_get_newest_post(feedPtr fp);

/** Sets the newest post of the feed
  *
  * @param fp the feed to set the newest post for
  */
void feed_set_newest_post(feedPtr fp, time_t newestPost);

/** Adds items to a feed and deletes unmarked posts
  *
  * @param fp the feed to add items to
  * @param items the list of items to add
  */
void feed_add_items_delete_old(feedPtr fp, GList *items);

/** Adds an item to a feed if it is newer than any items before, which
  * means that no feed gets downloaded twice
  *
  * @param fp the feed to add an item to
  * @param new_ip the new item to add
  * @return TRUE if an item has been added
  */
gboolean feed_add_item_ignore_old(feedPtr fp, itemPtr new_ip);

/** Sets the time when the feed has been read
  *
  * @param fp the feed to set the time for
  * @param time the time to set for the feed being read or -1 for the current time
  */
void feed_set_feed_read(feedPtr fp, time_t time);

/** Sets parse errors
  *
  * @param fp the feed to set parse errors for
  * @param parse_errors TRUE if there were parse errors, FALSE otherwise
  */
void feed_set_parse_errors(feedPtr fp, gboolean parse_errors);

/** Gets the time when the feed has been read
  *
  * @param fp the feed to get the time for
  */
time_t feed_get_feed_read(feedPtr fp);

/** Writes a string to a file
  *
  * @param cache the file to write
  * @param string the string to write
  */
gboolean cache_write_str(FILE * cache, const gchar * string);

/** Reads a string from a file
  *
  * @param cache the file to read from
  */
gchar *cache_read_str(FILE * cache);

/** Writes an integer to a file
  *
  * @param cache the file to write
  * @param data the integer to write
  */
gboolean cache_write_int(FILE * cache, int data);

/** Reads an integer from a file
  *
  * @param cache the file to read from
  */
int cache_read_int(FILE * cache);

/** This is set when a new subscription is made and an internet connection has been
  * established
  *
  * @param fp the feed to set the new subscription for
  * @param new_subscription the value to set
  */
void feed_set_new_subscription(feedPtr fp, gboolean new_subscription);

/** Gets the value of the new_subscription field
  *
  * @param fp the feed to get the new subscription field for
  */
gboolean feed_get_new_subscription(feedPtr fp);

/** This is set if the feed is a feed directory
  *
  * @param fp the feed to set as a feed directory
  * @param feed_directory the value to set
  */
void feed_set_feed_directory(feedPtr fp, gboolean feed_directory);

/** Gets the value of the feed_directory field
  *
  * @param fp the feed to get the value for
  */
gboolean feed_get_feed_directory(feedPtr fp);

/** Gets the parse errors field
  *
  * @param fp the feed to get the parse errors for
  * @return TRUE if there were parse errors, FALSE otherwise
  */
gboolean feed_get_parse_errors(feedPtr fp);

void feed_download_images(feedPtr fp);

extern gboolean conf_keep_feeds_in_memory;

#endif
