/**
 * @file feed.c common feed handling
 * 
 * Copyright (C) 2003, 2004 Lars Lindner <lars.lindner@gmx.net>
 * Copyright (C) 2004 Nathan J. Conrad <t98502@users.sourceforge.net>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version. 
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

#include <glib.h>
#include <libxml/xmlmemory.h>
#include <libxml/parser.h>
#include <string.h>
#include <time.h>
#include <unistd.h>        /* For unlink() */
#include <stdlib.h>
#include <hildon-widgets/gtk-infoprint.h>
#include <errno.h>

#include <settings.h>

#include "conf.h"
#include "common.h"
#include "osso-log.h"

#include "support.h"
#include "html.h"
#include "cdf_channel.h"
#include "rss_channel.h"
#include "atom10.h" 
#include "pie_feed.h"
#include "ocs_dir.h"
#include "opml.h"
#include "vfolder.h"
#include "feed.h"
#include "folder.h"
#include "favicon.h"
#include "callbacks.h"
#include "update.h"
#include "debug.h"
#include "metadata.h"
#include "interface.h"

//#include "enclosure.h"

#include "ui_feed.h"        /* FIXME: Remove ui_* include from core code */
#include "ui_feedlist.h"
#include "htmlview.h"
#include "ui_feed_directory.h"
#include "ui_queue.h"

#define READSTR str = cache_read_str(cache);
#define FREESTR if (str) {g_free(str); str=NULL;}
#define RETURN_IF_ERROR {fclose(cache); g_free(filename); g_free(filename_temp); unlink(filename_temp); return FALSE;}

extern long rss_unread;
extern time_t rss_updated;
extern AppData *app_data;
extern feedPtr searchFeed;
extern gboolean just_empty_items;

/* auto detection lookup table */
static GSList *feedhandlers;

struct feed_type {
    gint id_num;
    gchar *id_str;
};

gboolean conf_keep_feeds_in_memory = FALSE;
const glong C_MAGICHEAD = 0xCA204220;
const glong C_VERSION = 2;

/************************************************************************/
/*                        PUBLIC FUNCTIONS                              */
/************************************************************************/

/* initializing function, only called upon startup */
void feed_init(void)
{

    feedhandlers = g_slist_append(feedhandlers, ocs_init_feed_handler());    /* Must come before RSS/RDF */
    feedhandlers = g_slist_append(feedhandlers, rss_init_feed_handler());
    feedhandlers = g_slist_append(feedhandlers, cdf_init_feed_handler());
    /*tvh: import atom handler from liferea 1.0.4 */
    feedhandlers = g_slist_append(feedhandlers, atom10_init_feed_handler());  /* Must be before pie */
    feedhandlers = g_slist_append(feedhandlers, pie_init_feed_handler());
    feedhandlers = g_slist_append(feedhandlers, opml_init_feed_handler());
    feedhandlers =
    g_slist_append(feedhandlers, vfolder_init_feed_handler());
}

/* function to create a new feed structure */
feedPtr feed_new(void)
{
    feedPtr fp;

    fp = g_new0(struct feed, 1);

    /* we don't allocate a request structure this is done
       during cache loading or first update! */

    fp->updateInterval = -1;
    fp->defaultInterval = -1;
/*    Currently done where needed. */
    fp->addedTime = time(NULL); 
//    get_corrected_time(&(fp->addedTime));
    fp->type = FST_FEED;
    fp->cacheLimit = CACHE_DEFAULT;
    fp->newestPost = 0;
    fp->feedRead = 0;
    fp->new_subscription = FALSE;
    fp->feed_directory = FALSE;
    fp->tmpdata =
    g_hash_table_new_full(g_str_hash, g_str_equal, NULL, g_free);
    fp->downloadImages = TRUE;

    return fp;
}

/* ------------------------------------------------------------ */
/* feed type registration                    */
/* ------------------------------------------------------------ */

const gchar *feed_type_fhp_to_str(feedHandlerPtr fhp)
{
    if (fhp == NULL)
    return NULL;
    return fhp->typeStr;
}

feedHandlerPtr feed_type_str_to_fhp(const gchar * str)
{
    GSList *iter = NULL;
    feedHandlerPtr fhp = NULL;

    if (str == NULL)
    return NULL;

    for (iter = feedhandlers; iter != NULL; iter = iter->next) {
    fhp = (feedHandlerPtr) iter->data;
    if (!strcmp(str, fhp->typeStr))
        return fhp;
    }

    return NULL;
}

/**
 * General feed source parsing function. Parses the passed feed source
 * and tries to determine the source type. If the type is HTML and 
 * autodiscover is TRUE the function tries to find a feed, tries to
 * download it and parse the feed's source instead of the passed source.
 *
 * @param fp        the feed structure to be filled
 * @param data        the feed source
 * @param dataLength the length of the 'data' string
 * @param autodiscover    TRUE if auto discovery should be possible
 */
feedHandlerPtr feed_parse(feedPtr fp, gchar * data, size_t dataLength,
              gboolean autodiscover)
{
    gchar *source = NULL;
    xmlDocPtr doc = NULL;
    xmlNodePtr cur = NULL;
    GSList *handlerIter = NULL;
    gboolean handled = FALSE;
    feedHandlerPtr handler = NULL;

    debug_enter("feed_parse");
//    ULOG_DEBUG("String before: %s\n", data);
#ifdef feed_eliminate_end_sign_used
    data=g_strdup(feed_eliminate_end_sign(g_strdup(data)));
#else
    /*
     * feed_eliminate_end_sign_used was modified with "return" as first stmt,
     * turning it into dead code and creating memory leak.
     * I enclosed that function (in common.c) 
     * in #ifdef feed_eliminate_end_sign_used
     * (NB which is not defined now!)
     *
     * calling it as above:g_strdup(feed_eliminate_end_sign(g_strdup(data)));
     * would result in effective call g_strdup(g_strdup(data));
     * causing leaked result of first strdup.
     * (that comment added by Olev Kartau while looking for memory leaks 2007-04-20)
     */
    data=g_strdup(data);
#endif /* feed_eliminate_end_sign_used */
//    ULOG_DEBUG("String: %s\n", data);
    
    if(fp != NULL)
        g_free(fp->parseErrors);
    
    /* try to parse buffer with XML and to create a DOM tree */
    do {
    if (NULL ==
        (doc = parseBuffer(data, dataLength, &(fp->parseErrors)))) {
        gchar *msg =
        g_strdup_printf
        ("<p>XML error occurred. Unable to load feed. \"%s\"</p>",
         fp->source);
        addToHTMLBuffer(&(fp->parseErrors), msg);
        g_free(msg);
        break;
    }
    if (NULL == (cur = xmlDocGetRootElement(doc))) {
        addToHTMLBuffer(&(fp->parseErrors), "<p>Empty document</p>");
        break;
    }
    while (cur && xmlIsBlankNode(cur)) {
        cur = cur->next;
    }
    if (NULL == cur->name) {
        addToHTMLBuffer(&(fp->parseErrors), "<p>Invalid XML</p>");
        break;
    }

    /* determine the syndication format */
    handlerIter = feedhandlers;
    while (handlerIter != NULL) {
        handler = (feedHandlerPtr) (handlerIter->data);
        if (handler != NULL && handler->checkFormat != NULL
        && (*(handler->checkFormat)) (doc, cur)) {

        /* free old temp. parsing data, don't free right after parsing because
           it can be used until the last feed request is finished, move me 
           to the place where the last request in list otherRequests is 
           finished :-) */
        g_hash_table_destroy(fp->tmpdata);
        fp->tmpdata =
            g_hash_table_new_full(g_str_hash, g_str_equal, NULL,
                      g_free);

        /* we always drop old metadata */
        metadata_list_free(fp->metadata);
        fp->metadata = NULL;

        if (FALSE == handler->merge)
            feed_clear_item_list(fp);    /* for directories */

//        if ((nodePtr) displayed_node == (nodePtr) fp) {
//            ui_itemlist_clear();
//        }
            /*feed_remove_unmarked(fp);*/
//            ui_itemlist_load((nodePtr) fp);
//        } /*else
//            feed_remove_unmarked(fp);*/

       //2006May31: If it's opml, launch the feed directory dialog,
       //rather than a confirmation dialog: "Add all xyz feeds?"
       // No need to call the opml_parse func as this
       // will be done when creating the feed directory dialog anyway
       if (strcmp ((handler->typeStr), "opml") == 0)
       {
           newdialog_destroy();
           folderPtr folder = ui_feedlist_get_target_folder();

           ULOG_DEBUG("feed_parse: creating the feed dir dialog now");
           ui_feed_directory_from_add_feed_dialog( doc, cur, folder, (gchar *)feed_get_source(fp) );

           //free stuff and just return
           //g_free(source);
           if (doc != NULL)
               xmlFreeDoc(doc);
                   g_free(data);

           return handler;
        }

        (*(handler->feedParser)) (fp, doc, cur);    /* parse it */
        handled = TRUE;
        break;
        }
        handlerIter = handlerIter->next;
    }
    } while (0);

    if (handled)
        ULOG_DEBUG("handled %d: fp->title: %s",handled,fp->title);
    
    if (!handled) {
    handler = NULL;

//    ULOG_DEBUG("testing if we have a html page");
    
    /* test if we have a HTML page */
    if (autodiscover && ((NULL != strstr(data, "<html>"))
                 || (NULL != strstr(data, "<HTML>"))
                 || (NULL != strstr(data, "<html "))
                 || (NULL != strstr(data, "<HTML "))
        )) {
        /* if yes we should scan for links */
        debug1(DEBUG_UPDATE,
           "HTML detected, starting feed auto discovery (%s)",
           feed_get_source(fp));
        if (NULL !=
        (source =
         html_auto_discover_feed(data, feed_get_source(fp)))) {
        /* now download the first feed link found */
        struct request *request = download_request_new();
        debug1(DEBUG_UPDATE, "feed link found: %s", source);
        request->source = g_strdup(source);
        download_process(request);
        if (NULL != request->data) {
            debug0(DEBUG_UPDATE, "feed link download successful!");
            feed_set_source(fp, source);
            handler =
            feed_parse(fp, request->data, request->size,
                   FALSE);
        } else {
            /* if the download fails we do nothing except
               unsetting the handler so the original source
               will get a "unsupported type" error */
            handler = NULL;
            debug0(DEBUG_UPDATE, "feed link download failed!");
        }
        g_free(source);
        download_request_free(request);
        } else {
        debug0(DEBUG_UPDATE, "no feed link found!");
        }
    } else {
      handler = NULL;
      feed_set_available(fp, FALSE);
	if (!app_data->app_ui_data->start_on_background)
            gtk_infoprint(GTK_WINDOW(app_data->app_ui_data->app), _("rss_ni_errors_while_parsing_feed"));           
        addToHTMLBuffer(&(fp->parseErrors),
                "<p>Unable to specify the feed type. Check feed type and ensure it is <a href="
                "\"http://feedvalidator.org\">valid</a> and in a <a href=\"http://liferea."
                "sourceforge.net/index.php#supported_formats\">supported format</a>.</p>");
    }
    } else {
    debug1(DEBUG_UPDATE, "discovered feed format: %s",
           feed_type_fhp_to_str(handler));
    }

//    ULOG_DEBUG("Freeing document");
    if (doc != NULL)
    xmlFreeDoc(doc);
    
    ULOG_DEBUG("Exiting parser");
    debug_exit("feed_parse");
        g_free(data);

    return handler;
}

gchar *cache_read_str(FILE * cache)
{
    int len = 0;
    fread(&len, sizeof(len), 1, cache);    /* read length */

    if (feof(cache)) {
    return NULL;
    }

    if (len == 0) {
    return NULL;
    }

    gchar *string = g_malloc(len + 1);
    fread(string, len, 1, cache);
    string[len] = 0;
    return string;
}

int cache_read_int(FILE * cache)
{
    int data = 0;
    fread(&data, sizeof(data), 1, cache);
    return data;
}

long cache_read_long(FILE * cache)
{
    long data = 0;
    fread(&data, sizeof(data), 1, cache);
    return data;
}

gboolean cache_read_boolean(FILE * cache)
{
    char c = 0;
    fread(&c, sizeof(c), 1, cache);

    if (c == 0)
    return FALSE;

    return TRUE;
}

itemPtr cache_read_feed_item(FILE * cache, gint ver, gint size)
{
    gchar *str = NULL;
    gint count_enclosure=0, length=0;
    gint pos=0;
    struct enclosure_attribute *attrib=NULL;
    itemPtr ip = item_new();
    ip->newStatus = FALSE;

    if (ver > 0)
        pos = cache_read_long(cache);

    READSTR item_set_title(ip, str);    /* can be null */
    FREESTR if (feof(cache)) {
        g_free(ip->title);
        g_free(ip);
        return NULL;
    }

    READSTR item_set_description(ip, str);    /* can be null */
    FREESTR READSTR item_set_source(ip, str);    /* can be null */
    FREESTR READSTR item_set_real_source_title(ip, str);    /* can be null */
    FREESTR READSTR item_set_real_source_url(ip, str);    /* can be null */
    FREESTR READSTR item_set_id(ip, str);    /* can be null */
    FREESTR ip->nr = cache_read_long(cache);

    gboolean b;

    b = cache_read_boolean(cache);
    item_set_read_status(ip, b);

    b = cache_read_boolean(cache);
    /* we don't call item_set_mark here because it would
     * update the UI 
     item_set_mark(ip,b);*/
    ip->marked = b;
    //ip->delete_count=cache_read_int(cache);
    item_set_time(ip, cache_read_long(cache));

    if (ver >= C_VERSION) {
        /* Load the enclosure datas too, if it has some*/
        count_enclosure=cache_read_int(cache);

        while (count_enclosure)
        {
            attrib=g_new0(struct enclosure_attribute,1);
            READSTR enclosure_set_url(attrib,str);
            FREESTR READSTR 
            if (str)
            {
                enclosure_set_type(attrib,str);
                FREESTR
            }
            length=cache_read_int(cache);
            enclosure_set_length(attrib,length);
            ip->metadata=metadata_list_append_enclosure_attr(ip->metadata,"enclosure",attrib);
                length=0;
            count_enclosure--;
        }
    }
    if (ver && pos < size)
        fseek(cache, pos, SEEK_SET);

    return ip;
}


gboolean cache_write_str(FILE * cache, const gchar * string)
{
    if (string == NULL) {
    int i = 0;
    if (fwrite(&i, sizeof(i), 1, cache)!=1)      /* write just a zero */
    {
      return TRUE;
    }
    else
    {
      return FALSE;
    }
    }

    int len = strlen(string);
    if (fwrite(&len, sizeof(len), 1, cache)!=1)    /* write the size of string */
    {
      return TRUE;
    }
    if (fwrite(string, len, 1, cache)!=1)    /* trailing zero is not written */
    {
      return TRUE;
    }
    return FALSE;
}

gboolean cache_write_int(FILE * cache, int data)
{
    if (fwrite(&data, sizeof(data), 1, cache)!=1)
    {
      return TRUE;
    }
  return FALSE;
}

gboolean cache_write_long(FILE * cache, long data)
{
    if (fwrite(&data, sizeof(data), 1, cache)!=1)
    {
      return TRUE;
    }
    return FALSE;
}

gboolean cache_write_boolean(FILE * cache, gboolean data)
{
    char c = (data ? 1 : 0);

    if (fwrite(&c, sizeof(c), 1, cache)!=1)
    {
      return TRUE;
    }
    return FALSE;
}

gboolean cache_write_feed_item(FILE * cache, itemPtr ip)
{
    GSList *enclosures=NULL;
    gint pos, tmp;
    pos = ftell(cache);
    if (cache_write_long(cache, 0))
      return TRUE;
    if (cache_write_str(cache, item_get_title(ip)))    /* can be null */
      return TRUE;
    if (cache_write_str(cache, item_get_description(ip)))    /* can be null */
      return TRUE;
    if (cache_write_str(cache, item_get_source(ip)))    /* can be null */
      return TRUE;
    if (cache_write_str(cache, item_get_real_source_title(ip)))    /* can be null */
      return TRUE;
    if (cache_write_str(cache, item_get_real_source_url(ip)))    /* can be null */
      return TRUE;
    if (cache_write_str(cache, item_get_id(ip)))    /* can be null */
      return TRUE;
    if (cache_write_long(cache, ip->nr))
      return TRUE;
    if (cache_write_boolean(cache, item_get_read_status(ip)))
      return TRUE;
    if (cache_write_boolean(cache, item_get_mark(ip)))
      return TRUE;
    //cache_write_int(cache,item_get_delete_count(ip));
    if (cache_write_long(cache, item_get_time(ip)))
      return TRUE;

    /* Save the enclosure datas too*/
    enclosures=metadata_list_get(ip->metadata,"enclosure");
    /*Write out how long is the enclosure list*/
    if (enclosures)
    {
        //enclosures=enclosures->data;
        if (cache_write_int(cache,g_slist_length(enclosures)))
          return TRUE;
    }
    else
    {
        if (cache_write_int(cache, 0))
          return TRUE;
    }
    /* Write the enclosures by one by one */
    while(enclosures)
    {
        struct enclosure_attribute *current_enclosure=(struct enclosure_attribute *)enclosures->data;
        /*Write the URL*/
        if (cache_write_str(cache,enclosure_get_url(current_enclosure)))
          return TRUE;

        /*Write the type*/
        if (cache_write_str(cache,enclosure_get_type(current_enclosure)))
          return TRUE;

        /*Write the length*/
        if (cache_write_int(cache,enclosure_get_length(current_enclosure)))
          return TRUE;

        enclosures=g_slist_next(enclosures);
    }
    tmp = ftell(cache);
    fseek(cache, pos, SEEK_SET);
    if (cache_write_long(cache, tmp))
      return TRUE;
    fseek(cache, tmp, SEEK_SET);
    return FALSE;
}

void general_save()
{
/*    gchar *filename = NULL;

//    ULOG_DEBUG("general_save");
    
    filename =
    g_strdup_printf("%s" G_DIR_SEPARATOR_S "general",
            common_get_cache_path());
    FILE *cache = fopen(filename, "wb");
    g_free(filename);
    
    if(cache == NULL) {
        perror("general_save failed");
	return;
    }
    
    cache_write_long(cache, rss_unread);
    cache_write_long(cache, rss_updated);

    fflush(cache);
    fclose(cache);*/
}

void general_load()
{
/*    gchar *filename = NULL;

//    ULOG_DEBUG("general_load");
    
    filename =
    g_strdup_printf("%s" G_DIR_SEPARATOR_S "general",
            common_get_cache_path());
    FILE *cache = fopen(filename, "rb");
    g_free(filename);

    if (NULL == cache) {   
    rss_unread = 0;
    rss_updated = 0;
    } else {
    rss_unread = cache_read_long(cache);
    rss_updated = cache_read_long(cache);

    fclose(cache);
    }*/
}

/*
 * Feeds caches are marked to be saved at a few different places:
 * (1) Inside whe feed_set_* functions where an item is marked or made read or unread
 * (2) Inside of feed_process_result
 * (3) The callback where items are removed from the itemlist
 */
gboolean feed_save(feedPtr fp)
{
    GSList *itemlist = NULL;
    gchar *filename = NULL;
    gchar *filename_temp = NULL;
  
    itemPtr ip = NULL;
    gint saveCount = 0;
    gint saveMaxCount = 0;
    gint pos, tmp;

    debug_enter("feed_save");

    if (fp->needsCacheSave == FALSE) {
    return;
    }

    debug1(DEBUG_CACHE, "saving feed: %s", fp->title);
    g_assert(0 != fp->loaded);

    saveMaxCount = fp->cacheLimit;
    if (saveMaxCount == CACHE_DEFAULT)
    saveMaxCount = DEFAULT_MAX_ITEMS;

    filename =
    common_create_cache_filename("cache" G_DIR_SEPARATOR_S "feeds",
                     fp->id, NULL);
    filename_temp=g_strdup_printf("%s~",filename);
    FILE *cache = fopen(filename_temp, "wb");
    if (!cache) {
        perror("no space on device");
        return FALSE;
    }
//    g_assert(cache != NULL);

    if (cache_write_long(cache, C_MAGICHEAD))
      RETURN_IF_ERROR
      
    if (cache_write_long(cache, C_VERSION))
      RETURN_IF_ERROR
    pos = ftell(cache);
    if (cache_write_long(cache, 0))
      RETURN_IF_ERROR
    if (cache_write_str(cache, feed_get_title(fp)))
      RETURN_IF_ERROR
    if (cache_write_str(cache, feed_get_source(fp)))
      RETURN_IF_ERROR
    if (cache_write_str(cache, fp->description))    //can be null
      RETURN_IF_ERROR
    if (cache_write_str(cache, feed_get_image_url(fp)))    //can be null
      RETURN_IF_ERROR
    //cache_write_int(cache, fp->defaultInterval); now common for all feeds...
    if (cache_write_boolean(cache, feed_get_available(fp)))
      RETURN_IF_ERROR
    if (just_empty_items) {
        if (cache_write_boolean(cache, FALSE))
          RETURN_IF_ERROR
        if (cache_write_long(cache, feed_get_added(fp)))
          RETURN_IF_ERROR
        if (cache_write_str(cache, 0))
          RETURN_IF_ERROR
        if (cache_write_long(cache, 0))
          RETURN_IF_ERROR
        if (cache_write_long(cache, 0))
          RETURN_IF_ERROR
    } else {
        if (cache_write_boolean(cache, feed_get_discontinued(fp)))
          RETURN_IF_ERROR
        if (cache_write_long(cache, feed_get_added(fp)))
          RETURN_IF_ERROR
        if (cache_write_str(cache, feed_get_lastmodified(fp)))    //can be null
          RETURN_IF_ERROR
        if (cache_write_long(cache, feed_get_newest_post(fp)))
          RETURN_IF_ERROR
        if (cache_write_long(cache, feed_get_feed_read(fp)))
          RETURN_IF_ERROR
    }

    tmp = ftell(cache);
    fseek(cache, pos, SEEK_SET);
    if (cache_write_long(cache, tmp))
      RETURN_IF_ERROR
    fseek(cache, tmp, SEEK_SET);
    //cache_write_boolean(cache, feed_get_download_images(fp));
    //FIXME: metadata should maybe also be saved..
    //metadata_add_xml_nodes(fp->metadata, feedNode);

    if (!just_empty_items) {
        itemlist = feed_get_item_list(fp);
        for (itemlist = feed_get_item_list(fp); itemlist != NULL;
                itemlist = g_slist_next(itemlist)) {
            ip = itemlist->data;
            g_assert(NULL != ip);

            if (saveMaxCount != CACHE_UNLIMITED &&
                saveCount >= saveMaxCount &&
                (fp->fhp == NULL || fp->fhp->directory == FALSE) &&
                !item_get_mark(ip)) {
                continue;
            }

            if (cache_write_feed_item(cache, ip))    //item_save(ip, feedNode)
              RETURN_IF_ERROR
            saveCount++;
        }
    }

    fclose(cache);
    if (rename(filename_temp,filename)<0)
    {
      g_free(filename);
      g_free(filename_temp);
      return FALSE;
    }
    fp->needsCacheSave = FALSE;
    g_free(filename);
    g_free(filename_temp);
    debug_exit("feed_save");
    
    return TRUE;
}

/* Function which is called to load a feed's into memory. This function
   might be called multiple times even if the feed was already loaded.
   Each time the method is called a reference counter is incremented. */
gboolean feed_load(feedPtr fp)
{
    GList *items = NULL;
    gchar *filename = NULL;
    gint pos=0, tmp, size;
    gint ver = 0;

    g_assert(NULL != fp);
    g_assert(NULL != fp->id);
    
    if (0 != (fp->loaded)++) {
        return TRUE;
    }

    if (FST_VFOLDER == feed_get_type(fp)) {
        debug0(DEBUG_CACHE, "it's a vfolder, nothing to do...");
        fp->loaded++;
        return TRUE;
    }

    filename =
        common_create_cache_filename("cache" G_DIR_SEPARATOR_S "feeds",
                     fp->id, NULL);
    debug1(DEBUG_CACHE, "loading cache file \"%s\"", filename);
    ULOG_DEBUG ("\n--------------feed_load: loading cache file \"%s\"", filename);

    fp->unreadCount = 0;
    fp->markCount = 0;
    fp->newCount = 0;
    metadata_list_free(fp->metadata);
    fp->metadata = NULL;

    gchar *str;
    FILE *cache = fopen(filename, "rb");
    if (cache == NULL) {
        g_free(filename);
        debug0(DEBUG_CACHE, "A cache file could not be loaded\n");
        return FALSE;        //TRUE?
    }

    fseek(cache, 0, SEEK_END);
    size = ftell(cache);
    fseek(cache, 0, SEEK_SET);

    tmp = cache_read_long(cache);
    if (tmp == C_MAGICHEAD) {
      ver = cache_read_long(cache);
      pos = cache_read_long(cache);
    } else {
      fseek(cache, 0, SEEK_SET);
    }
    READSTR
/* CHECK: Feed title can be set manually in OPML, so we should override that.
 * However, can this cause problems somewhere? Alternate solution is to
 * save feed cache after title has been changed. */
    if (NULL == feed_get_title(fp)) {
        ULOG_WARN("Feed %s title was somehow NULL, setting %s", filename,
              str);
        feed_set_title(fp, str);
    }
    FREESTR READSTR feed_set_source(fp, str);
    FREESTR READSTR g_free(fp->description);
    fp->description = str;
    /* no need to free this as it's not duplicated
     * - is there a reason we don't use feed_set_description? */

    READSTR feed_set_image_url(fp, str);
    FREESTR feed_set_available(fp, cache_read_boolean(cache));
    feed_set_discontinued(fp, cache_read_boolean(cache));

    feed_set_added(fp, cache_read_long(cache));

    READSTR feed_set_lastmodified(fp, str);
    FREESTR
    feed_set_newest_post(fp, cache_read_long(cache));
    feed_set_feed_read(fp, cache_read_long(cache));

    if (ver && pos < size)
      fseek(cache, pos, SEEK_SET);

    while (!feof(cache)) {
        itemPtr ip = cache_read_feed_item(cache, ver, size);
        if (ip != NULL) {
            items = g_list_append(items, ip);
        }
    }

    feed_add_items(fp, items);
    favicon_load(fp);

    fclose(cache);
    g_free(filename);

    debug_exit("feed_load");
    return TRUE;
}

/* Only some feed informations are kept in memory to lower memory
   usage. This method unloads everything besides necessary infos. 
   
   If the feed parameter is NULL the function is called for all feeds.
   
   Each time this function is called the reference counter of all
   feeds is decremented and if it zero the unnecessary feed infos are 
   free'd */
gboolean feed_unload(feedPtr fp)
{
    gint unreadCount = 0;

    g_assert(NULL != fp);
    g_assert(0 <= fp->loaded);    /* could indicate bad loaded reference counting */
    if (0 != fp->loaded) {
    if (feed_save(fp) == FALSE)        /* save feed before unloading */
      return FALSE;
    
//    if (!getBooleanConfValue(KEEP_FEEDS_IN_MEMORY)) {
    if (conf_keep_feeds_in_memory) {
        debug_enter("feed_unload");
        if (1 == fp->loaded) {
        if (FST_FEED == feed_get_type(fp)) {
//            ULOG_DEBUG("feed_unload (%s)",
//               feed_get_source(fp));

            /* free items */
            unreadCount = fp->unreadCount;
            feed_clear_item_list(fp);
            fp->unreadCount = unreadCount;
        } else {
            debug1(DEBUG_CACHE, "not unloading vfolder (%s)",
               feed_get_title(fp));
        }
        } else {
        debug2(DEBUG_CACHE,
               "not unloading (%s) because it's used (%d references)...",
               feed_get_source(fp), fp->loaded);
        }
        fp->loaded--;
        debug_exit("feed_unload");
    }
    }
}

/**
 * method to be called to schedule a feed to be updated
 */
void feed_schedule_update(feedPtr fp, gint flags)
{
    const gchar *source = NULL;
    struct request *request = NULL;
    g_assert(NULL != fp);

    ULOG_DEBUG("Scheduling %s to be updated", feed_get_title(fp));
    
/*    if (fp->request != NULL) {
    return;
    }*/

    if (feed_get_discontinued(fp)) {
    ULOG_DEBUG("The feed was discontinued. Liferea won't update it anymore!");
    return;
    }
    
    if (NULL == (source = feed_get_source(fp))) {
    g_warning
        ("Feed source is NULL! This should never happen - cannot update!");
    return;
    }
    /* This causes the news reader to crash! */
    //We don't need to call this here. We should update the time only when the download was finished
    //feed_reset_update_counter(fp);
    request = download_request_new();
    //fp->request = request;
    request->callback = ui_feed_process_update_result;

    request->user_data = fp;
    request->feed = fp;
    /* prepare request url (strdup because it might be
       changed on permanent HTTP redirection in netio.c) */
    request->source = g_strdup(source);
    if (feed_get_lastmodified(fp) != NULL)
    request->lastmodified = g_strdup(feed_get_lastmodified(fp));
    if (feed_get_etag(fp) != NULL)
    request->etag = g_strdup(feed_get_etag(fp));
    request->flags = flags;
    request->priority = (flags & FEED_REQ_PRIORITY_HIGH) ? 1 : 0;
    if (feed_get_filter(fp) != NULL)
    request->filtercmd = g_strdup(feed_get_filter(fp));

    download_queue(request);

}

/**
 * To be used by parser implementation to merge a new orderd list of
 * items to a feed. Ensures properly ordered joint item list. The
 * passed GList is free'd afterwards!
 */
void feed_add_items(feedPtr fp, GList * items)
{
    GList *iter = NULL;
    
    g_assert(NULL != fp);

    if (NULL == items) {    /* Nothing to do - hopefully not an error */
    /*ULOG_WARN("Feed %s attempt to add NULL list.", fp->title);*/
    return;
    }

    /* The parser implementations read the items from the
       feed from top to bottom. Adding them directly would
       mean to reverse there order. */
    iter = g_list_last(items);
    while (iter != NULL) {
    feed_add_item(fp, ((itemPtr) iter->data));
    iter = g_list_previous(iter);
    }
    
    g_list_free(items);
}

void feed_add_items_delete_old(feedPtr fp, GList *items)
{
    GList *iter = NULL;
    GSList *gs_iter = NULL;
    time_t newestPost = 0;
    itemPtr ip = NULL;
    gboolean item_added = FALSE;
    
    g_assert(NULL != fp);
    
    if (NULL == items) {    /* Nothing to do - hopefully not an error */
    /*ULOG_WARN("Feed %s attempt to add NULL list.", fp->title);*/
    return;
    }

    ULOG_DEBUG("Adding new feed items ...");
    /* The parser implementations read the items from the
       feed from top to bottom. Adding them directly would
       mean to reverse there order. */
    iter = g_list_last(items);
    while (iter != NULL) {
    if(feed_add_item_ignore_old(fp, ((itemPtr) iter->data)) &&
       !item_added)
        item_added = TRUE;
    iter = g_list_previous(iter);
    }
    
    //We don't do this... this could be from older version?
    //The automatic deletion function is changed.
    /*ULOG_DEBUG("Removing unmarked ...");
    if(item_added)
        feed_remove_unmarked(fp, TRUE);*/
        
    ULOG_DEBUG("Calculating newest post ...");
    gs_iter = fp->items;
    
    ip = NULL;    
    while (gs_iter != NULL) {
        ip = gs_iter->data;
    
    if(ip->time > newestPost)
        newestPost = ip->time;
    gs_iter = g_slist_next(gs_iter);
    }
    if(newestPost > feed_get_newest_post(fp))
        feed_set_newest_post(fp, newestPost);    
    
    ULOG_DEBUG("Freeing items ...");
    g_list_free(items);
    ULOG_DEBUG("Done ...");
}
/**
 * Can be used to add a single item to a feed. But it's better to
 * use feed_add_items() to keep the item order of parsed feeds.
 * Should be used for vfolders only.
 *
 * Adds an item to the feed. Realizes the item merging logic based on
 * the item id's. 
 *
 * Note: Does not do merging for vfolders. Would be hazardous!
 * Note: This version adds vfolder results automatically to display.
 */
void feed_add_item(feedPtr fp, itemPtr new_ip)
{
    GSList *old_items = NULL;
    itemPtr old_ip = NULL;
    gboolean found = FALSE, equal = FALSE;

    g_assert(NULL != fp);
    g_assert(NULL != new_ip);
    g_assert((0 != fp->loaded) || (FST_VFOLDER == feed_get_type(fp)));

    if (FST_VFOLDER != feed_get_type(fp)) {
        /* determine if we should add it... */
       debug1(DEBUG_VERBOSE, "check new item for merging: \"%s\"",
           item_get_title(new_ip));

       /* compare to every existing item in this feed */
        found = FALSE;
        old_items = feed_get_item_list(fp);
        while (NULL != old_items) {
            old_ip = old_items->data;

            /* try to compare the two items */

            /* trivial case: one item has item the other doesn't -> they can't be equal */
            if (((item_get_id(old_ip) == NULL)
                 && (item_get_id(new_ip) != NULL))
                || ((item_get_id(old_ip) != NULL)
                && (item_get_id(new_ip) == NULL))) {
                /* cannot be equal (different ids) so compare to 
               next old item */
                ULOG_DEBUG("Comparing %s vs %s, id mismatch",
                   item_get_title(old_ip), item_get_title(new_ip));
                old_items = g_slist_next(old_items);
            continue;
            }

            /* just for the case there are no ids: compare titles and HTML descriptions */
            equal = TRUE;

            if (((item_get_title(old_ip) != NULL)
                 && (item_get_title(new_ip) != NULL))
                && (0 !=
                    strcmp(item_get_title(old_ip),
                       item_get_title(new_ip))))
                equal = FALSE;

            if (((item_get_description(old_ip) != NULL)
                 && (item_get_description(new_ip) != NULL))
                    && (0 !=
                        strcmp(item_get_description(old_ip),
                       item_get_description(new_ip))))
                equal = FALSE;

            /* best case: they both have ids (position important: id check is useless 
               without knowing if the items are different!) */
            if (NULL != item_get_id(old_ip)) {
                if (0 == strcmp(item_get_id(old_ip), item_get_id(new_ip))) {
                    found = TRUE;
                    break;
                }
            }

            if (equal) {
                found = TRUE;
                break;
            }

            old_items = g_slist_next(old_items);
        }

        if (!found) {
            if (0 == new_ip->nr) {
                new_ip->nr = ++(fp->lastItemNr);
                fp->needsCacheSave = TRUE;
            }

            /* ensure that the feed last item nr is at maximum */
            if (new_ip->nr > fp->lastItemNr)
            fp->lastItemNr = new_ip->nr;

            /* ensure that the item nr's are unique */
            if (NULL != feed_lookup_item_by_nr(fp, new_ip->nr)) {
                g_warning
                    ("The item number to be added is not unique! Feed (%s) Item (%s) (%ld)\n",
                     fp->id, new_ip->title, new_ip->nr);
                new_ip->nr = ++(fp->lastItemNr);
                fp->needsCacheSave = TRUE;
            }

            if (FALSE == item_get_read_status(new_ip))
                feed_increase_unread_counter(fp);
            if (TRUE == item_get_mark(new_ip))
                feed_increase_mark_counter(fp);
            if (TRUE == new_ip->newStatus) {
                fp->newCount++;
            }
            fp->items = g_slist_prepend(fp->items, (gpointer) new_ip);
            new_ip->fp = fp;

            /* check if the item matches any vfolder rules */
            vfolder_check_item(new_ip);

            debug0(DEBUG_VERBOSE, "-> item added to feed itemlist");

        } else {
            /* if the item was found but has other contents -> update contents */
            if (!equal) {
            /* no item_set_new_status() - we don't treat changed items as new items! */
                item_set_title(old_ip, item_get_title(new_ip));
                item_set_description(old_ip, item_get_description(new_ip));
                item_set_time(old_ip, item_get_time(new_ip));
                item_set_unread(old_ip);    /* FIXME: needed? */
                metadata_list_free(old_ip->metadata);
                old_ip->metadata = new_ip->metadata;
                vfolder_update_item(old_ip);
                debug0(DEBUG_VERBOSE,
                   "-> item already existing and was updated");
            } else {
                debug0(DEBUG_VERBOSE, "-> item already exists");
            }
        } //while end
    } else {
        /* vfolder/search - just add the item */
        if (FALSE == item_get_read_status(new_ip))
            feed_increase_unread_counter(fp);
        fp->items = g_slist_append(fp->items, (gpointer) new_ip);
        new_ip->fp = fp;
        /* item number should already be set by item_copy() */

        //ui_itemlist_search_append(new_ip);
    }
}

gboolean feed_add_item_ignore_old(feedPtr fp, itemPtr new_ip)
{
    GSList *old_items = NULL;
    itemPtr old_ip = NULL;
    gboolean found = FALSE, equal = FALSE;

    g_assert(NULL != fp);
    g_assert(NULL != new_ip);
    g_assert((0 != fp->loaded) || (FST_VFOLDER == feed_get_type(fp)));

    if (FST_VFOLDER != feed_get_type(fp)) {
    /* determine if we should add it... */
    debug1(DEBUG_VERBOSE, "check new item for merging: \"%s\"",
           item_get_title(new_ip));

    /* compare to every existing item in this feed */
    found = FALSE;
    old_items = feed_get_item_list(fp);
    while (NULL != old_items) {
        old_ip = old_items->data;

        /* try to compare the two items */

        /* trivial case: one item has item the other doesn't -> they can't be equal */
        if (((item_get_id(old_ip) == NULL)
         && (item_get_id(new_ip) != NULL))
        || ((item_get_id(old_ip) != NULL)
            && (item_get_id(new_ip) == NULL))) {
        /* cannot be equal (different ids) so compare to 
           next old item */
        ULOG_DEBUG("feed_add_item_ignore_old Comparing %s vs %s, id mismatch",
               item_get_title(old_ip), item_get_title(new_ip));
        old_items = g_slist_next(old_items);
        continue;
        }

        /* just for the case there are no ids: compare titles and HTML descriptions */
        equal = TRUE;

        if (((item_get_title(old_ip) != NULL)
         && (item_get_title(new_ip) != NULL))
        && (0 !=
            strcmp(item_get_title(old_ip),
               item_get_title(new_ip))))
        equal = FALSE;

        if (((item_get_description(old_ip) != NULL)
         && (item_get_description(new_ip) != NULL))
        && (0 !=
            strcmp(item_get_description(old_ip),
               item_get_description(new_ip))))
        equal = FALSE;

        /* best case: they both have ids (position important: id check is useless 
           without knowing if the items are different!) */
        if (NULL != item_get_id(old_ip)) {
        if (0 == strcmp(item_get_id(old_ip), item_get_id(new_ip))) {
            found = TRUE;
            break;
        }
        }

        if (equal) {
        found = TRUE;
        break;
        }

        old_items = g_slist_next(old_items);
    }

    if (!found) {
        if(item_get_time(new_ip) <= feed_get_newest_post(fp)) {
            item_free_ignore_metadata(new_ip);
             return FALSE;
        }
        if (0 == new_ip->nr) {
        new_ip->nr = ++(fp->lastItemNr);
        fp->needsCacheSave = TRUE;
        }

        /* ensure that the feed last item nr is at maximum */
        if (new_ip->nr > fp->lastItemNr)
        fp->lastItemNr = new_ip->nr;

        /* ensure that the item nr's are unique */
        if (NULL != feed_lookup_item_by_nr(fp, new_ip->nr)) {
        g_warning
            ("The item number to be added is not unique! Feed (%s) Item (%s) (%ld)\n",
             fp->id, new_ip->title, new_ip->nr);
        new_ip->nr = ++(fp->lastItemNr);
        fp->needsCacheSave = TRUE;
        }

        if (FALSE == item_get_read_status(new_ip))
        feed_increase_unread_counter(fp);
        if (TRUE == item_get_mark(new_ip))
        feed_increase_mark_counter(fp);
        if (TRUE == new_ip->newStatus) {
        fp->newCount++;
        }
        fp->items = g_slist_prepend(fp->items, (gpointer) new_ip);
        new_ip->fp = fp;

        /* check if the item matches any vfolder rules */
        vfolder_check_item(new_ip);

        debug0(DEBUG_VERBOSE, "-> item added to feed itemlist");

    } else {
        /* if the item was found but has other contents -> update contents */
        if (!equal) {
        /* no item_set_new_status() - we don't treat changed items as new items! */
        item_set_title(old_ip, item_get_title(new_ip));
        item_set_description(old_ip, item_get_description(new_ip));
        item_set_time(old_ip, item_get_time(new_ip));
        item_set_unread(old_ip);    /* FIXME: needed? */
        metadata_list_free(old_ip->metadata);
        old_ip->metadata = new_ip->metadata;
        vfolder_update_item(old_ip);
        debug0(DEBUG_VERBOSE,
               "-> item already existing and was updated");
        } else {
        debug0(DEBUG_VERBOSE, "-> item already exists");
        }
        item_free_ignore_metadata(new_ip);
    }
    } else {
    /* vfolder/search - just add the item */
    if (FALSE == item_get_read_status(new_ip))
        feed_increase_unread_counter(fp);
    fp->items = g_slist_append(fp->items, (gpointer) new_ip);
    new_ip->fp = fp;
    /* item number should already be set by item_copy() */

    ui_itemlist_search_append(new_ip);
    
    found = TRUE;
    }
    return !found;
}

void feed_set_feed_read(feedPtr fp, time_t time)
{
    GTimeVal current_time;
    
    g_assert(fp != NULL);
    
    if(time == -1)
    {
        g_get_current_time(&current_time);

        fp->feedRead = current_time.tv_sec;
    }
    else
        fp->feedRead = time;
    fp->needsCacheSave = TRUE;
}

time_t feed_get_feed_read(feedPtr fp)
{
    g_assert(fp != NULL);
    
    return fp->feedRead;
}
void feed_remove_item(feedPtr fp, itemPtr ip)
{

    fp->items = g_slist_remove(fp->items, ip);
    vfolder_remove_item(ip);    /* remove item copies */
    remove_links_from_cache(ip);
    item_free(ip);        /* remove the item */
    fp->needsCacheSave = TRUE;
}

itemPtr feed_lookup_item_by_nr(feedPtr fp, gulong nr)
{
    GSList *items = NULL;
    itemPtr ip = NULL;

    g_assert((0 != fp->loaded) || (FST_VFOLDER == feed_get_type(fp)));
    items = fp->items;
    while (NULL != items) {
    ip = (itemPtr) (items->data);
    if (ip->nr == nr)
        return ip;
    items = g_slist_next(items);
    }

    return NULL;
}
/* Looks for ip that has fp as the parent fp
    But also its original fp matches orig_fp id (== sourceFeed. Look at item_copy
    We need the 3rd param for virtual feed search
    One virtual feed can have 2 items that have the same id due to
    one feed can be added twice. Or even when the items from 
    different feeds can accidentally have the same id

    TODO: NOW CHANGED BY itemPtr feed_lookup_item_by_nr_extended()
*/

//itemPtr feed_lookup_item_by_id(feedPtr fp, gchar *id, gchar * orig_fp_id)
itemPtr feed_lookup_item_by_nr_extended(feedPtr fp, gulong nr, gchar *orig_fp_id)
{
    GSList *items = NULL;
    itemPtr ip = NULL;
    
    if (FST_VFOLDER == feed_get_type(fp) )
        ULOG_DEBUG("feed_lookup_item_by_id: Looking in a vfolder ");

    if ( (0 == fp->loaded) && (FST_VFOLDER != feed_get_type(fp)) )
        //feed_load(fp);
        ULOG_DEBUG("feed_lookup_item_by_id: feed not loaded and feed not vfolder. \n\
                    something is wrong. Should we have loaded the feed first? ");
                       
    items = fp->items;
    while (NULL != items) {
        ip = (itemPtr) (items->data);
        //ULOG_DEBUG("ip->id = %s, id = %s", ip->id, id);
        //ULOG_DEBUG("ip->id = %d, id = %d", ip->nr, nr);
        //just to make sure id is not NULL
        /*
        if (NULL == ip->id)
        {
            if ( ip->source == NULL)
                ULOG_DEBUG("feed_lookup_item_by_id: This feed is fcuked, both \
                            id and source are NULL");
            else
                item_set_id(ip, item_get_source(ip));   
        }    
        */
        /* if we're searching for item from a virtual feed
        Then there's a risk of there are 2 items found which 
        are from 2 different feeds but exactly the same feed.
        (like a feed is added twice 
         */
        if (FST_VFOLDER == feed_get_type(fp))
        {
            //if ( ( strcmp(ip->sourceFeed->id, orig_fp_id)==0) && (strcmp(ip->id, id) == 0) )
            if ( ( strcmp(ip->sourceFeed->id, orig_fp_id)==0) && (ip->nr == nr) )

                return ip;
        }
        /* normal feed */
        else 
        {
            //if (strcmp(ip->id, id) == 0)
            if (ip->nr == nr)
                return ip;
        }
        items = g_slist_next(items);
    }

    return NULL;
}

/* ---------------------------------------------------------------------------- */
/* feed attributes encapsulation                        */
/* ---------------------------------------------------------------------------- */

void feed_set_id(feedPtr fp, const gchar * id)
{
    g_assert(fp != NULL);
    if (NULL != fp->id )
        g_free(fp->id);
    /*TODO: an appropriate remove_white_spaces is preferrable
    this is just a hack to some feeds that have '\n' at the beginning
    of the guid or their souce. Weird though
    */
    //g_strstrip(id);
        fp->id = g_strdup(id);
}
const gchar *feed_get_id(feedPtr fp)
{
    g_assert(fp != NULL);
    return fp->id;
}

void feed_set_type(feedPtr fp, gint type)
{
    g_assert(fp != NULL);
    fp->type = type;
    conf_feedlist_schedule_save();
}

gint feed_get_type(feedPtr fp)
{
    g_assert(fp != NULL);
    return fp->type;
}

gpointer feed_get_favicon(feedPtr fp)
{
    g_assert(fp != NULL);
    return fp->icon;
}

void feed_increase_unread_counter(feedPtr fp)
{
    g_assert(fp != NULL);
    fp->unreadCount++;
}

void feed_decrease_unread_counter(feedPtr fp)
{
    g_assert(fp != NULL);
    fp->unreadCount--;
}

gint feed_get_unread_counter(feedPtr fp)
{
    g_assert(fp != NULL);
    return fp->unreadCount;
}

void feed_increase_new_counter(feedPtr fp)
{
    g_assert(fp != NULL);
    fp->newCount++;
}

void feed_decrease_new_counter(feedPtr fp)
{
    g_assert(fp != NULL);
    fp->newCount--;
}

gint feed_get_new_counter(feedPtr fp)
{
    g_assert(fp != NULL);
    return fp->newCount;
}

void feed_increase_mark_counter(feedPtr fp)
{
    g_assert(fp != NULL);
    fp->markCount++;
}

void feed_decrease_mark_counter(feedPtr fp)
{
    g_assert(fp != NULL);
    fp->markCount--;
}

gint feed_get_mark_counter(feedPtr fp)
{
    g_assert(fp != NULL);
    return fp->markCount;
}

gint feed_get_default_update_interval(feedPtr fp)
{
    g_assert(fp != NULL);
    return fp->defaultInterval;
}

void feed_set_default_update_interval(feedPtr fp, gint interval)
{
    g_assert(fp != NULL);
    fp->defaultInterval = interval;
}

gint feed_get_update_interval(feedPtr fp)
{
    g_assert(fp != NULL);
    return fp->updateInterval;
}

void feed_set_update_interval(feedPtr fp, gint interval)
{
    g_assert(fp != NULL);
    fp->updateInterval = interval;

    conf_feedlist_schedule_save();
}

const feedHandlerPtr feed_get_fhp(feedPtr fp)
{
    g_assert(fp != NULL);
    return fp->fhp;
}

void feed_reset_update_counter(feedPtr fp)
{
    g_assert(fp != NULL);
    g_get_current_time(&fp->lastPoll);
    //get_corrected_time(&(fp->lastPoll));
    /*fp->lastPoll.tv_sec = get_current_time();*/
    conf_feedlist_schedule_save();
    debug2(DEBUG_CONF, "Reseting last poll counter for %s to %ld.\n",
       fp->title, fp->lastPoll.tv_sec);
}

void feed_set_available(feedPtr fp, gboolean available)
{
    g_assert(fp != NULL);
    fp->available = available;
}

gboolean feed_get_available(feedPtr fp)
{
    g_assert(fp != NULL);
    return fp->available;
}

void feed_set_download_images(feedPtr fp, gboolean download)
{
    g_assert(fp != NULL);
    fp->downloadImages = download;
}

gboolean feed_get_download_images(feedPtr fp)
{
    g_assert(fp != NULL);
    return fp->downloadImages;
}

void feed_set_discontinued(feedPtr fp, gboolean discontinued)
{
    g_assert(fp != NULL);
    fp->discontinued = discontinued;
}

gboolean feed_get_discontinued(feedPtr fp)
{
    g_assert(fp != NULL);
    return fp->discontinued;
}

void feed_set_nonremovable(feedPtr fp, gboolean nonremovable)
{
    g_assert(fp != NULL);
    fp->nonremovable = nonremovable;
}

gboolean feed_get_nonremovable(feedPtr fp)
{
    g_assert(fp != NULL);
    return fp->nonremovable;
}

/* Returns a HTML string describing the last retrieval error 
   of this feed. Should only be called when feed_get_available
   returns FALSE. Caller must free returned string! */
gchar *feed_get_error_description(feedPtr fp)
{
    gchar *tmp1 = NULL;

    if (feed_get_discontinued(fp)) {
    addToHTMLBufferFast(&tmp1, UPDATE_ERROR_START);
    addToHTMLBufferFast(&tmp1, HTTP410_ERROR_TEXT);
    addToHTMLBufferFast(&tmp1, UPDATE_ERROR_END);
    }
    addToHTMLBuffer(&tmp1, fp->errorDescription);
    return tmp1;
}

const time_t feed_get_time(feedPtr fp)
{
    return (fp != NULL ? fp->time : 0);
}
void feed_set_time(feedPtr fp, const time_t t)
{
    g_assert(fp != NULL);
    fp->time = t;
}

const gchar *feed_get_title(feedPtr fp)
{
    g_assert(fp != NULL);

    if (NULL != fp->title)
    return fp->title;
    else if (NULL != fp->source)
    return fp->source;
    else
    return NULL;
}

void feed_set_title(feedPtr fp, const gchar * title)
{
    g_assert(fp != NULL);
    g_free(fp->title);
    if (title != NULL)
    fp->title = g_strstrip(g_strdup(title));
    else
    fp->title = NULL;
    conf_feedlist_schedule_save();
}


const gchar *feed_get_description(feedPtr fp)
{
    g_assert(fp != NULL);
    return fp->description;
}
void feed_set_description(feedPtr fp, const gchar * description)
{
    g_assert(fp != NULL);
    g_free(fp->description);
    if (description != NULL)
    {
    fp->description = g_strdup(description);
    }
    else
    fp->description = NULL;
}

const gchar *feed_get_source(feedPtr fp)
{
    g_assert(fp != NULL);
    return fp->source;
}
const gchar *feed_get_filter(feedPtr fp)
{
    g_assert(fp != NULL);
    return fp->filtercmd;
}

void feed_set_source(feedPtr fp, const gchar * source)
{
    g_assert(fp != NULL);
    g_free(fp->source);

    fp->source = g_strdup(source);
    conf_feedlist_schedule_save();
}

void feed_set_filter(feedPtr fp, const gchar * filter)
{
    g_assert(fp != NULL);
    g_free(fp->filtercmd);

    fp->filtercmd = g_strdup(filter);
    conf_feedlist_schedule_save();
}

const gchar *feed_get_html_url(feedPtr fp)
{
    g_assert(fp != NULL);
    return fp->htmlUrl;
};
void feed_set_html_url(feedPtr fp, const gchar * htmlUrl)
{
    g_assert(fp != NULL);

    g_free(fp->htmlUrl);
    if (htmlUrl != NULL)
    fp->htmlUrl = g_strdup(htmlUrl);
    else
    fp->htmlUrl = NULL;
}

const time_t feed_get_added(feedPtr fp)
{
    return (fp != NULL ? fp->addedTime : 0);
}
void feed_set_added(feedPtr fp, const time_t t)
{
    g_assert(fp != NULL);
    if (t == 0)
        fp->addedTime = time(NULL); 
        //get_corrected_time(&fp->addedTime);
    else
        fp->addedTime = t;
}

const gchar *feed_get_lastmodified(feedPtr fp)
{
    g_assert(fp != NULL);
    return fp->lastModified;
};
void feed_set_lastmodified(feedPtr fp, const gchar * lastmodified)
{
    g_assert(fp != NULL);

    g_free(fp->lastModified);
    if (lastmodified != NULL)
    fp->lastModified = g_strdup(lastmodified);
    else
    fp->lastModified = NULL;
}

const gchar *feed_get_etag(feedPtr fp)
{
    g_assert(fp != NULL);
    return fp->etag;
};
void feed_set_etag(feedPtr fp, const gchar * etag)
{
    g_assert(fp != NULL);

    g_free(fp->etag);
    if (etag != NULL)
    fp->etag = g_strdup(etag);
    else
    fp->etag = NULL;
}

const gchar *feed_get_image_url(feedPtr fp)
{
    g_assert(fp != NULL);
    return fp->imageUrl;
};
void feed_set_image_url(feedPtr fp, const gchar * imageUrl)
{
    g_assert(fp != NULL);

    g_free(fp->imageUrl);
    if (imageUrl != NULL)
    fp->imageUrl = g_strdup(imageUrl);
    else
    fp->imageUrl = NULL;
}

/* returns feed's list of items, if necessary loads the feed from cache */
GSList *feed_get_item_list(feedPtr fp)
{
    g_assert(fp != NULL);
    g_assert((0 != fp->loaded) || (FST_VFOLDER == fp->type));
    return fp->items;
}

void feed_set_sort_column(feedPtr fp, gint sortColumn, gboolean reversed)
{
    g_assert(fp != NULL);
    fp->sortColumn = sortColumn;
    fp->sortReversed = reversed;
    conf_feedlist_schedule_save();
}

/**
 * Method to free all items structures of a feed, does not mean
 * that it removes items from cache! This method is used 
 * for feed unloading.
 */
void feed_clear_item_list(feedPtr fp)
{
    GSList *item = NULL;

    g_assert(fp != NULL);
    item = fp->items;

    while (NULL != item) {
    item_free(item->data);
    item = g_slist_next(item);
    }
    g_slist_free(fp->items);
    fp->items = NULL;
    /* explicitly not forcing feed saving to allow feed unloading */
}

/** 
 * Method to permanently removing all items from the given feed
 */
void feed_remove_items(feedPtr fp)
{
    GSList *item = NULL;

    g_assert(fp != NULL);
    //for proper removing of vfolder's items, feed has to be loaded 
    feed_load(fp);
    item = fp->items;

    while (NULL != item) {
    vfolder_remove_item(item->data);    /* remove item copies */
    remove_links_from_cache(item->data);
    item_free(item->data);    /* remove the item */
    item = g_slist_next(item);
    }
    g_slist_free(fp->items);
    fp->items = NULL;
    fp->needsCacheSave = TRUE;    /* force feed saving to make it permanent */
    feed_unload(fp);
}

/** 
 * Method to permanently remove unmarked items from feed
 */
void feed_remove_unmarked(feedPtr fp, gboolean save_unread)
{
    GSList *item = NULL;
    GSList *newlist = NULL;
//    gboolean load_displayed_feed = FALSE;
    
    g_assert(fp != NULL);
    
    item = fp->items;
    
/*    if((nodePtr)fp == displayed_node)
    {
//        load_displayed_feed = TRUE;
        load_displayed_feed = FALSE;
        ui_itemlist_clear();
    }*/

    while (NULL != item) {
    if (item_get_mark((itemPtr) item->data)) {
        newlist = g_slist_append(newlist, item->data);
    } else if (save_unread && !item_get_read_status((itemPtr) item->data)) {
        newlist = g_slist_append(newlist, item->data);
    } else {
        vfolder_remove_item(item->data);    /* remove item copies */
        remove_links_from_cache(item->data);
        item_free(item->data);    /* remove the item */
    }
    item = g_slist_next(item);
    }
    g_slist_free(fp->items);
    fp->items = newlist;
    fp->needsCacheSave = TRUE;    /* force feed saving to make it permanent */
    
/*    if(load_displayed_feed)
        ui_feedlist_load_selected((nodePtr)fp);*/
}

void feed_mark_all_items_read(feedPtr fp)
{
    GSList *item = NULL;

    feed_load(fp);
    item = fp->items;
    while (NULL != item) {
    item_set_read((itemPtr) item->data);
    item = g_slist_next(item);
    }
    feed_unload(fp);
}

gchar *feed_render(feedPtr fp)
{
    struct displayset displayset;
    gchar *buffer = NULL;
    gchar *tmp = NULL, *tmp2 = NULL;
    gboolean migration = FALSE;

    g_assert(0 != fp);
    g_assert(0 != fp->loaded);
    displayset.headtable = NULL;
    displayset.head = NULL;
    displayset.body = g_strdup(feed_get_description(fp));
    displayset.foot = NULL;
    displayset.foottable = NULL;

    /* I hope this is unique enough... */
    if ((NULL != displayset.body) && (NULL != strstr(displayset.body, "class=\"itemhead\"")))
    migration = TRUE;

    if (FALSE == migration) {
    metadata_list_render(fp->metadata, &displayset);

    /* Error description */
    if (NULL != (tmp = feed_get_error_description(fp))) {
        addToHTMLBufferFast(&buffer, tmp);
        g_free(tmp);
    }

    /* Head table */
    addToHTMLBufferFast(&buffer, HEAD_START);
    /*  -- Feed line */
    if (feed_get_html_url(fp) != NULL)
        tmp = g_strdup_printf("<a href=\"%s\">%s</a>",
                  feed_get_html_url(fp),
                  feed_get_title(fp));
    else
        tmp = g_strdup(feed_get_title(fp));

    tmp2 = g_strdup_printf(HEAD_LINE, tmp);
    g_free(tmp);
    addToHTMLBufferFast(&buffer, tmp2);
    g_free(tmp2);

    /*  -- Source line */
    if ((NULL != feed_get_source(fp))
        && (FST_VFOLDER != feed_get_type(fp))) {
        tmp =
        g_strdup_printf("<a href=\"%s\">%s</a>",
                feed_get_source(fp), feed_get_source(fp));

        tmp2 = g_strdup_printf(HEAD_LINE, tmp);
        g_free(tmp);
        addToHTMLBufferFast(&buffer, tmp2);
        g_free(tmp2);
    }

    addToHTMLBufferFast(&buffer, displayset.headtable);
    g_free(displayset.headtable);
    addToHTMLBufferFast(&buffer, HEAD_END);

    /* Head */
    if (displayset.head != NULL) {
        addToHTMLBufferFast(&buffer, displayset.head);
        g_free(displayset.head);
    }

    /* feed/channel image */
    if (NULL != feed_get_image_url(fp)) {
        addToHTMLBufferFast(&buffer, "<img class=\"feed\" src=\"");
        addToHTMLBufferFast(&buffer, feed_get_image_url(fp));
        addToHTMLBufferFast(&buffer, "\"><br>");
    }
    }

    /* Body */
    if (displayset.body != NULL) {
    addToHTMLBufferFast(&buffer, displayset.body);
    g_free(displayset.body);
    }

    if (FALSE == migration) {
    /* Foot */
    if (displayset.foot != NULL) {
        addToHTMLBufferFast(&buffer, displayset.foot);
        g_free(displayset.foot);
    }

    if (displayset.foottable != NULL) {
        addToHTMLBufferFast(&buffer, FEED_FOOT_TABLE_START);
        addToHTMLBufferFast(&buffer, displayset.foottable);
        addToHTMLBufferFast(&buffer, FEED_FOOT_TABLE_START);
        g_free(displayset.foottable);
    }
    }
    return buffer;
}

time_t feed_get_newest_post(feedPtr fp)
{
    g_assert(fp != NULL);
    
    return fp->newestPost;
}    

void feed_set_newest_post(feedPtr fp, time_t newestPost)
{
    g_assert(fp != NULL);
    
    fp->newestPost = newestPost;
}

void feed_free_requests(nodePtr ptr)
{
    GSList *iter;
    feedPtr fp = (feedPtr) ptr;
    g_assert(NULL != ptr);

    /* Don't free active feed requests here, because they might still
       be processed in the update queues! Abandoned requests are
       free'd in feed_process. They must be freed in the main thread
       for locking reasons. */
    if (fp->request != NULL)
    fp->request->callback = NULL;

    /* same goes for other requests */
    iter = fp->otherRequests;
    while (NULL != iter) {
    ((struct request *) iter->data)->callback = NULL;
    iter = g_slist_next(iter);
    }
    g_slist_free(fp->otherRequests);
}

/* method to totally erase a feed, remove it from the config, etc.... */
void feed_free(feedPtr fp)
{
    gchar *filename = NULL;

    /* This is not strictly necessary. It just speeds deletion of an entire itemlist. */
    if (displayed_node == (nodePtr) fp)
    {
        ULOG_DEBUG("feed: feed_free");
        ui_mainwindow_show_statistics();
    }
    if (FST_VFOLDER == fp->type) {
    vfolder_free(fp);    /* some special preparations for vfolders */
    } else {
    g_assert(FST_FEED == fp->type);
    }

    /* free items */
    feed_remove_items(fp);

    if (fp->id && fp->id[0] != '\0')
    filename =
        common_create_cache_filename("cache" G_DIR_SEPARATOR_S "feeds",
                     fp->id, NULL);

    /* FIXME: Move this to a better place. The cache file does not
       need to always be deleted, for example when freeing a
       feedstruct used for updating. */
    if (filename && 0 != unlink(filename))
    /* Oh well.... Can't do anything about it. 99% of the time,
       this is spam anyway. */ ;
    g_free(filename);

    feed_free_requests((nodePtr) fp);

    if (fp->icon != NULL)
    g_object_unref(fp->icon);

    if (fp->id) {
    favicon_remove(fp);
    conf_feedlist_schedule_save();
    g_free(fp->id);
    }

    g_free(fp->title);
    g_free(fp->description);
    g_free(fp->errorDescription);
    g_free(fp->source);
    g_free(fp->filtercmd);
    g_free(fp->htmlUrl);
    g_free(fp->imageUrl);
    g_free(fp->parseErrors);
    g_free(fp->lastModified);
    g_free(fp->etag);
    metadata_list_free(fp->metadata);
    g_hash_table_destroy(fp->tmpdata);
    g_free(fp);
}

void feed_set_new_subscription(feedPtr fp, gboolean new_subscription)
{
    g_assert(fp != NULL);
    
    fp->new_subscription = new_subscription;
}

gboolean feed_get_new_subscription(feedPtr fp)
{
    g_assert(fp != NULL);

    return fp->new_subscription;
}

void feed_set_feed_directory(feedPtr fp, gboolean feed_directory)
{
    g_assert(fp != NULL);
    
    fp->feed_directory = feed_directory;
}

gboolean feed_get_feed_directory(feedPtr fp)
{
    g_assert(fp != NULL);

    return fp->feed_directory;
}

void feed_set_parse_errors(feedPtr fp, gboolean parse_errors)
{
    g_assert(fp != NULL);

    fp->err = parse_errors;
}

gboolean feed_get_parse_errors(feedPtr fp)
{
    g_assert(fp != NULL);

    return fp->err;
}

void
download_image_link(gchar *link, gpointer user_data)
{
//    ULOG_DEBUG("Adding image link to download: <%s>", link);
    download_queue_image(link, NULL, FALSE);
}

void feed_download_images(feedPtr fp)
{
    g_assert(fp != NULL);
    
    GSList *iter = NULL;

    feed_load(fp);
    iter = fp->items;
    while (iter) {
        if (!item_get_hidden((itemPtr) iter->data))
            find_image_links((gchar *)item_get_description((itemPtr) iter->data), 
                download_image_link, NULL);
        iter = g_slist_next(iter);
    }
    feed_unload(fp);
}
