/**
    @file configuration.c

    Application configuration access through Settings component.

    Copyright (c) 2004-2006 Nokia Corporation.
	
    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA
    02111-1307 USA
*/


#include <string.h>
#include <glib.h>

#include "appdata.h"
#include "configuration.h"
#include "constant.h"
#include "settings.h"
#include "utility.h"
#include "pdfviewer.h"
#include "ui/callbacks.h"
#include "debug.h"

/**
	Load application configuration from GConf and command line options

	@param argc number of arguments passed from main
	@param *argv[] value structure for arguments
	@param app_data Application data
	@returns gboolean TRUE if successful; FALSE otherwise
*/
gboolean get_configuration(gint argc, gchar * argv[], AppData * app_data)
{
    GSList        *list        = NULL;
    gchar         *str_recent_items;
    gchar        **files       = NULL;
    RecentItem    *recent_item = NULL;
    int            n;
    gboolean       show_images = TRUE;
    gboolean       first_run = TRUE;
    GnomeVFSResult vfs_result; 

    TDB("Get conf\n");
    /* default images on */
    first_run = settings_get_bool(GCONF_KEY_FIRST_RUN);
    if (!first_run)
    {
        settings_set_bool(GCONF_KEY_IMAGES,TRUE);
        settings_set_bool(GCONF_KEY_FIRST_RUN,TRUE);
    }
    
    /* get show images option */
    show_images = settings_get_bool(GCONF_KEY_IMAGES);

    /* there was no user conf saved in gconf (yet) */
    /* having this check always makes it true, commented out for bug 21710 */
/*
    if (show_images == NULL) {
      show_images=TRUE;
    }
*/  
/*
    if ( !show_images ) {
      show_images = TRUE;
    }
*/
  
    /* setting show images flag */ 
/*
    if (show_images !=  
	PDF_FLAGS_IS_SET(app_data->app_ui_data->flags, PDF_FLAGS_SHOW_IMAGES) )
      PDF_FLAGS_TOGGLE(app_data->app_ui_data->flags, PDF_FLAGS_SHOW_IMAGES);
*/
    if (show_images == TRUE)
    {
	  PDF_FLAGS_SET(app_data->app_ui_data->flags, PDF_FLAGS_SHOW_IMAGES);
    }
    else
    {
	  PDF_FLAGS_UNSET(app_data->app_ui_data->flags, PDF_FLAGS_SHOW_IMAGES);
    }
			    
    str_recent_items = settings_get_string(GCONF_KEY_RECENT_ITEMS);
    TDB("get_conf: %s\n", str_recent_items);
    if (!str_recent_items) {
	/* store empty string value to create new gconf key */
	settings_set_string(GCONF_KEY_RECENT_ITEMS, "");
	return TRUE;
    }

    /* separate recent items and store them into a list */
    files = g_strsplit(str_recent_items, RECENT_ITEM_SEPARATOR, 0);
    for (n = 0; files[n] != NULL; n++) {
	/* ignore trailing separator */
        if (strlen(files[n]) > 0) {
          gboolean obex = strncmp(files[n], "obex:", 5) == 0;
	  /* filter out not supported formats */
	  if(obex || file_is_supported(files[n])) {
	    recent_item = g_new0(RecentItem, 1);
	    recent_item->uri = g_strdup(files[n]);

            TDB("%s, %d\n", files[n], obex);
            if (obex) 
                vfs_result = GNOME_VFS_OK;
            else
                vfs_result = gnome_vfs_monitor_add(&(recent_item->monitor_handle),
					       files[n],
					       GNOME_VFS_MONITOR_FILE,
					       on_file_changed,
					       app_data);
	    
	    if (vfs_result == GNOME_VFS_OK)
	      list = g_slist_append(list, recent_item);
	    else
	      OSSO_LOG_CRIT("Couldn't add monitor for file"
			    "not added to recent list!");

	  }
	}
    }
    g_strfreev(files);
    g_free(str_recent_items);

    /* store recent items list to appdata */
    app_data->recent_list = list;

    return TRUE;
}


/**
	Save application configuration to GConf
	
	@param app_data Application data
	@returns gboolean TRUE if successful; FALSE otherwise
*/
gboolean save_configuration(AppData * app_data)
{
    GSList *iter = NULL;
    gchar *str_recent_items, *tmp;

    g_return_val_if_fail(app_data, FALSE);

    if (app_data->recent_list) {
	/* combine recent list items into a single string */
	str_recent_items = g_strdup("");
	for (iter = app_data->recent_list; iter != NULL; iter = iter->next) {
	    tmp = str_recent_items;
	    str_recent_items =
	      g_strconcat(str_recent_items, 
			  (gchar *) ((RecentItem*)iter->data)->uri,
			  NULL);
	    g_free(tmp);
	    tmp = str_recent_items;
	    str_recent_items =
		g_strconcat(str_recent_items, RECENT_ITEM_SEPARATOR, NULL);
	    g_free(tmp);
	}

	/* store string into gconf */
	settings_set_string(GCONF_KEY_RECENT_ITEMS, str_recent_items);
	g_free(str_recent_items);
    }

    /* store boolean value into gconf */
    settings_set_bool(GCONF_KEY_IMAGES,
		      (gboolean)( PDF_FLAGS_IS_SET( app_data->app_ui_data->flags,
					PDF_FLAGS_SHOW_IMAGES ) ) ) ;

    return TRUE;
}


/* EOF */
