/**
  @file bt.c

  @author Johan Hedberg <johan.hedberg@nokia.com>

  Copyright (C) 2004 Nokia. All rights reserved.

  This program is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation; either version 2 of the License, or
  (at your option) any later version.

  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.
    
  You should have received a copy of the GNU General Public License
  along with this program; if not, write to the Free Software
  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

*/

#include <stdlib.h>
#include <stdio.h>
#include <stdint.h>
#include <unistd.h>
#include <errno.h>
#include <sys/poll.h>
#include <sys/types.h>
#include <sys/ioctl.h>
#include <sys/socket.h>
#include <netinet/in.h>
#include <glib.h>

#include <bluetooth/bluetooth.h>
#include <bluetooth/hci.h>
#include <bluetooth/hci_lib.h>
#include <bluetooth/rfcomm.h>
#include <bluetooth/sdp.h>
#include <bluetooth/sdp_lib.h>

#include "bt.h"

static const unsigned char
nokia_ftp_uuid[16] = { 0x00, 0x00, 0x50, 0x05, 0x00, 0x00, 0x10, 0x00,
                       0x80, 0x00, 0x00, 0x02, 0xee, 0x00, 0x00, 0x01 };

static BtDev *bt_dev_new(const char *bda, uint16_t clock, uint8_t cod[3], const char *name)
{
    BtDev *dev;

    dev = g_new0(BtDev, 1);
    if (bda)
        dev->bda  = g_strdup(bda);
    if (name)
        dev->name = g_strdup(name);
    dev->clock = clock;
    memcpy(dev->cod, cod, 3);

    return dev;
}

static void bt_dev_free(BtDev *dev)
{
    g_free(dev->bda);
    g_free(dev->name);
    g_free(dev);
}


static int make_inquiry_socket(void)
{
    struct hci_filter flt;
    struct sockaddr_hci addr;
    int sk;

    sk = socket(AF_BLUETOOTH, SOCK_RAW, BTPROTO_HCI);
    if (sk < 0) {
        printf("Can't create HCI socket: %s\n", g_strerror(errno));
        return -1;
    }

    hci_filter_clear(&flt);
    hci_filter_set_ptype(HCI_EVENT_PKT, &flt);

    hci_filter_set_event(EVT_INQUIRY_RESULT, &flt);
    hci_filter_set_event(EVT_INQUIRY_RESULT_WITH_RSSI, &flt);
    hci_filter_set_event(EVT_INQUIRY_COMPLETE, &flt);
    hci_filter_set_event(EVT_CMD_STATUS, &flt);
    hci_filter_set_event(EVT_REMOTE_NAME_REQ_COMPLETE, &flt);

    if (setsockopt(sk, SOL_HCI, HCI_FILTER, &flt, sizeof(flt)) < 0) {
        printf("Can't set HCI filter: %s\n", g_strerror(errno));
        return -1;
    }


    /* Bind socket to the HCI device */
    addr.hci_family = AF_BLUETOOTH;
    addr.hci_dev = 0;
    if (bind(sk, (struct sockaddr *)&addr, sizeof(addr)) < 0) {
        printf("Can't attach to device hci0: %s\n", g_strerror(errno));
        close(sk);
        return -1;
    }

    return sk;
}

static int send_inquiry_command(int sock)
{
    inquiry_cp inq;

    inq.num_rsp = 100;
    inq.length  = 8;

    /* General/Unlimited Inquiry Access Code (GIAC) */
    inq.lap[0] = 0x33;
    inq.lap[1] = 0x8b;
    inq.lap[2] = 0x9e;

    return hci_send_cmd(sock, OGF_LINK_CTL, OCF_INQUIRY,
                        INQUIRY_CP_SIZE , &inq);
}

static gint bt_dev_cmp_bda(BtDev *a, const char *bda)
{
    return strcmp(a->bda, bda);
}

static gint bt_dev_cmp_name(BtDev *a, const char *bda)
{
    if (a->name)
        return strcmp(a->name, bda);
    return 1;
}

static GSList *inquiry_result_with_rssi(char *data, GSList *l)
{
    int i;
    uint8_t num_resp = data[0];
    char   *ptr      = &(data[1]);

    for (i = 0; i < (int)num_resp; i++) {
        BtDev *dev;
        char bda[18];

        inquiry_info_with_rssi *info = (inquiry_info_with_rssi *)ptr;
        ptr += INQUIRY_INFO_WITH_RSSI_SIZE;

        ba2str(&info->bdaddr, bda);

        /* Ignore duplicates */
        if (g_slist_find_custom(l, bda, (GCompareFunc)bt_dev_cmp_bda))
            continue;
       
        dev = bt_dev_new(bda, info->clock_offset, info->dev_class, NULL);
        l = g_slist_append(l, dev);
    }

    return l;
}

static GSList *inquiry_result(char *data, GSList *l)
{
    int i;
    uint8_t num_resp = data[0];
    char   *ptr      = &(data[1]);

    for (i = 0; i < (int)num_resp; i++) {
        BtDev *dev;
        char bda[18];

        inquiry_info *info = (inquiry_info *)ptr;
        ptr += INQUIRY_INFO_SIZE;

        ba2str(&info->bdaddr, bda);

        /* Ignore duplicates */
        if (g_slist_find_custom(l, bda, (GCompareFunc)bt_dev_cmp_bda))
            continue;
       
        dev = bt_dev_new(bda, info->clock_offset, info->dev_class, NULL);
        l = g_slist_append(l, dev);
    }

    return l;
}


static gboolean process_data(char *data, int data_len, GSList **l)
{
    hci_event_hdr *hdr;
    char *body;

    if (data[0] != HCI_EVENT_PKT)
        return TRUE;

    body = &data[HCI_EVENT_HDR_SIZE + 1];
    hdr = (hci_event_hdr *) &data[1];

    switch (hdr->evt) {
        case EVT_INQUIRY_RESULT:
            *l = inquiry_result(body, *l);
            break;
        case EVT_INQUIRY_RESULT_WITH_RSSI:
            *l = inquiry_result_with_rssi(body, *l);
            break;
        case EVT_INQUIRY_COMPLETE:
            return FALSE;
        case EVT_CMD_STATUS:
            {
                evt_cmd_status *ev = (evt_cmd_status *)body;
                if (ev->status != 0x00)
                    return FALSE;
            }
            break;
        default:
            break;
    }

    return TRUE;
}

static GSList *get_inquiry_results(int sk)
{
    GSList *l;
    char *data;

    l          = NULL;
    data       = g_malloc(HCI_MAX_EVENT_SIZE);

    while (TRUE) {
        int ret;
        struct pollfd pfd;

        pfd.fd      = sk;
        pfd.events  = POLLIN;

        ret = poll(&pfd, 1, 10000);
        if (ret < 0) {
            printf("poll: %s\n", g_strerror(errno));
            break;
        }
        else if (ret == 0) {
            printf("Timeout while waiting for data on HCI socket\n");
            break;
        }
        else if (pfd.revents & POLLIN) {
            int data_len;

            data_len = read(sk, data, HCI_MAX_EVENT_SIZE);
            if (data_len < 0) {
                printf("Unable to read from HCI socket: %s",
                        g_strerror(errno));
                break;
            }

            if (!process_data(data, data_len, &l))
                break;
        }
        else {
            printf("Error or hangup on HCI socket\n");
            break;
        }
    }

    g_free(data);
    return l;
}

static int bt_sdp_rec_get_ch(sdp_record_t *rec)
{
    sdp_list_t *protos = NULL;
    int ch = -1;

    if (sdp_get_access_protos(rec, &protos) == 0) {
        ch = sdp_get_proto_port(protos, RFCOMM_UUID);
        sdp_list_foreach(protos, (sdp_list_func_t)sdp_list_free, NULL);
        sdp_list_free(protos, NULL);
    }

    return ch;
}

static int bt_sdp_session_get_ch(sdp_session_t *sess, uuid_t *uuid)
{
    int ch = -1;
    uint32_t range = 0x0000ffff;
    sdp_list_t *attrid, *search, *seq, *next;

    attrid = sdp_list_append(0, &range);
    search = sdp_list_append(0, uuid);

    if (sdp_service_search_attr_req(sess, search, SDP_ATTR_REQ_RANGE, attrid, &seq)) {
        fprintf(stderr, "Service Discovery failed: %s\n", g_strerror(errno));
        return -1;
    }
    sdp_list_free(attrid, 0);
    sdp_list_free(search, 0);

    for (; ch < 1 && seq; seq = next) {
        sdp_record_t *rec = (sdp_record_t *) seq->data;

        ch = bt_sdp_rec_get_ch(rec);

        next = seq->next;
        free(seq);
        sdp_record_free(rec);
    }

    return ch;
}

static int bt_sdp_get_channel(bdaddr_t *ba, gboolean ftp)
{
    uuid_t uuid;
    int ch;
    sdp_session_t *sess;

    sess = sdp_connect(BDADDR_ANY, ba, SDP_RETRY_IF_BUSY);
    if (sess == NULL) {
        fprintf(stderr, "SDP connect failed\n");
        return -1;
    }

    if (ftp) {
        sdp_uuid128_create(&uuid, nokia_ftp_uuid);
        ch = bt_sdp_session_get_ch(sess, &uuid);
        if (ch < 0) {
            sdp_uuid16_create(&uuid, OBEX_FILETRANS_SVCLASS_ID);
            ch = bt_sdp_session_get_ch(sess, &uuid);
        }
    }
    else {
        sdp_uuid16_create(&uuid, OBEX_OBJPUSH_SVCLASS_ID);
        ch = bt_sdp_session_get_ch(sess, &uuid);
    }

    sdp_close(sess);

    return ch;
}

static int bt_rfcomm_connect(bdaddr_t *ba, uint8_t channel)
{
    struct sockaddr_rc laddr, raddr;
    int sk;

    memset(&laddr, 0, sizeof(laddr));
    memset(&raddr, 0, sizeof(raddr));

    laddr.rc_family = AF_BLUETOOTH;
    bacpy(&laddr.rc_bdaddr, BDADDR_ANY);
    laddr.rc_channel = 0;

    raddr.rc_family = AF_BLUETOOTH;
    bacpy(&raddr.rc_bdaddr, ba);
    raddr.rc_channel = channel;

    sk = socket(AF_BLUETOOTH, SOCK_STREAM, BTPROTO_RFCOMM);
    if (sk < 0) {       
        fprintf(stderr, "Can't create RFCOMM socket: %s\n", g_strerror(errno));
        return -1;
    }

    if (bind(sk, (struct sockaddr *)&laddr, sizeof(laddr)) < 0) {
        fprintf(stderr, "Can't bind RFCOMM socket: %s\n", g_strerror(errno));
        close(sk);
        return -1;
    }

    if (connect(sk, (struct sockaddr *)&raddr, sizeof(raddr)) < 0) {
        fprintf(stderr, "connect(): %s\n", g_strerror(errno));
        close(sk);
        return -1;
    }

    return sk;
}

int bt_connect(const char *bda,
               int channel,
               gboolean ftp,
               gboolean auth,
               gboolean encrypt,
               const char *role)
{
    int chan, sk;
    bdaddr_t ba;

    g_assert(bda != NULL);

    if (str2ba(bda, &ba) < 0) {
        fprintf(stderr, "Invalid BDA: %s\n", bda);
        return -1;
    }

    if (channel < 1 || channel > 255) {
        chan = bt_sdp_get_channel(&ba, ftp);
        if (chan < 1) {
            fprintf(stderr, "Unable to get RFCOMM channel from SDP server\n");
            return -1;
        }
    }
    else
        chan = channel;

    printf("Connecting to channel %d on %s\n", chan, bda);

    sk = bt_rfcomm_connect(&ba, (uint8_t)chan);
    if (sk < 0) {
        fprintf(stderr, "Could not connect to RFCOMM channel %d on %s\n",
                chan, bda);
        return -1;
    }

    return sk;
}

char *bt_get_remote_name(const char *bda)
{
    int sk;
    char name[248];
    bdaddr_t ba;

    if (str2ba(bda, &ba) < 0) {
        fprintf(stderr, "Invalid BDA: %s\n", bda);
        return NULL;
    }

    sk = hci_open_dev(hci_get_route(&ba));
    if (sk < 0) {
        fprintf(stderr, "hci_open_dev failed: %s", g_strerror(errno));
        return NULL;
    }

    if (hci_read_remote_name(sk, &ba, sizeof(name), name, 5000) < 0) {
        fprintf(stderr, "hci_read_remote_name failed: %s\n",
                g_strerror(errno));
        close(sk);
        return NULL;
    }

    close(sk);

    return g_strdup(name);
}

static void resolve_name(BtDev *dev, gpointer data)
{
    char name[248];
    bdaddr_t ba;
    int sk = GPOINTER_TO_INT(data);

    if (dev->name)
        return;

    str2ba(dev->bda, &ba);

    if (hci_read_remote_name(sk, &ba, sizeof(name), name, 5000) < 0)
        return;

    dev->name = g_strdup(name);
}

void bt_scan_free(GSList *l)
{
    g_slist_foreach(l, (GFunc)bt_dev_free, NULL);
    g_slist_free(l);
}

GSList *bt_scan(void)
{
    GSList *l;
    int sk;

    sk = make_inquiry_socket();
    if (sk < 0)
        return NULL;

    if (send_inquiry_command(sk) < 0) {
        printf("Could not send inquiry command\n");
        return NULL;
    }

    printf("Searching...\n"); fflush(stdout);

    l = get_inquiry_results(sk);

    if (l) {
        printf("%d devices found. Resolving names...", g_slist_length(l));
        fflush(stdout);
        g_slist_foreach(l, (GFunc)resolve_name, GINT_TO_POINTER(sk));
        printf("done.\n");
    }
    else
        printf("No devices were found.\n");
    

    close(sk);

    return l;
    
}

BtDev *bt_scan_find_name(GSList *l, const char *name)
{
    GSList *match;
    match = g_slist_find_custom(l, name, (GCompareFunc)bt_dev_cmp_name);
    if (match)
        return match->data;
    return NULL;
}
