/**
    @file callbacks.c

    Provides callbacks for the user interface.

    Copyright (c) 2004, 2005 Nokia Corporation.
	
    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA
    02111-1307 USA
*/

/* Function prototypes */
#include <gdk/gdkkeysyms.h>
#include <glib/gstdio.h>
#include <hildon-widgets/gtk-infoprint.h>
#include <libgnomevfs/gnome-vfs.h>
#include <osso-helplib.h>
#include "callbacks.h"
#include "state_save.h"

#define PAUSE_WITH_HW_KEY "/apps/osso/games_startup/pause_with_hw_key"
#define ENABLE_KEY_TIMEOUT 1000


static gboolean enable_fullscreen(gpointer data)
{
  AppData *appdata=(AppData*)data;
  appdata->app_ui_data->timeouts=0;
  appdata->app_ui_data->got_pause=FALSE;
  return FALSE;
  
}

void undimm_everything(AppData *ad)
{
  gtk_widget_set_sensitive(hildon_appview_get_menu(ad->app_ui_data->main_view),TRUE);
  if (GTK_WIDGET_VISIBLE(ad->app_ui_data->play_button))
		{
			gtk_widget_set_sensitive(ad->app_ui_data->play_button,TRUE);
		} 
		if (GTK_WIDGET_VISIBLE(ad->app_ui_data->continue_button))
		{
			gtk_widget_set_sensitive(ad->app_ui_data->continue_button,TRUE);
		}
    if (get_game_state()==gs_paused)
    {
		  if (GTK_WIDGET_VISIBLE(ad->app_ui_data->restart_button))
		  {
			  gtk_widget_set_sensitive(ad->app_ui_data->restart_button,TRUE);
		  }
    }
		if (ad->app_ui_data->plugin->info->plugin_cb)
		{
			ad->app_ui_data->plugin->info->plugin_cb(NULL,(gpointer)MA_GAME_PLAYING);
		}
  return;
}

void dimm_everything(AppData *ad)
{
  gtk_widget_set_sensitive(hildon_appview_get_menu(ad->app_ui_data->main_view),FALSE);
  if (GTK_WIDGET_VISIBLE(ad->app_ui_data->play_button))
  {
	  gtk_widget_set_sensitive(ad->app_ui_data->play_button,FALSE);
  } 
  if (GTK_WIDGET_VISIBLE(ad->app_ui_data->continue_button))
  {
	  gtk_widget_set_sensitive(ad->app_ui_data->continue_button,FALSE);
  } 
  if (GTK_WIDGET_VISIBLE(ad->app_ui_data->restart_button))
  {
	  gtk_widget_set_sensitive(ad->app_ui_data->restart_button,FALSE);
  }
  if (ad->app_ui_data->plugin->info->plugin_cb)
  {
    ad->app_ui_data->plugin->info->plugin_cb(NULL,(gpointer)MA_GAME_PLAYING_START);
  }
  return;
}

void        save_nomem_response      (GtkDialog *dialog,
                                            gint       arg1,
                                            gpointer   user_data)
{
  AppUIData *app_ui_data=(AppUIData*)user_data;
  gtk_widget_destroy(app_ui_data->nomem_dialog);
  app_ui_data->nomem_dialog=NULL;
}


gboolean dialog_key_press(GtkWidget *widget, GdkEventKey *event, gpointer data)
{
  (void)data;
  
  if (event->state & (GDK_CONTROL_MASK |
                      GDK_SHIFT_MASK |
                      GDK_MOD1_MASK |
                      GDK_MOD3_MASK |
                      GDK_MOD4_MASK |
                      GDK_MOD5_MASK)) {
    return FALSE;
  }

  switch (event->keyval)
  {
    case GDK_Escape:
        gtk_dialog_response(GTK_DIALOG(widget), GAME_RESPONSE_CANCEL);
      return TRUE;
      break;
    default:
      break;
  }
  
  return FALSE;
}


gboolean save_nomem_key_press(GtkWidget *widget, GdkEventKey *event, gpointer data)
{
  AppUIData *app_ui_data=(AppUIData*)data;
  
  if (event->state & (GDK_CONTROL_MASK |
                      GDK_SHIFT_MASK |
                      GDK_MOD1_MASK |
                      GDK_MOD3_MASK |
                      GDK_MOD4_MASK |
                      GDK_MOD5_MASK)) {
    return FALSE;
  }

  switch (event->keyval)
  {
    case GDK_Escape:
      
        gtk_widget_destroy(GTK_DIALOG(widget));
        app_ui_data->nomem_dialog=NULL;
      return TRUE;
      break;
    default:
      break;
  }
  
  return FALSE;
}


/* The following function is from INDT's games-startup code. The copyrights 
belongs to them. The INDT's games-startup was released under LGPL license. 
This function is modified under LGPL license*/

/* Callback function when a game returns */
void 
startup_ui_top_cb (AppUIData *app, 
                   gpointer    data)
{
	osso_log (LOG_DEBUG, "%s\n", __FUNCTION__);
  AppUIData *ui = (AppUIData*) (data);
  gtk_window_present (GTK_WINDOW (ui->app));
}

/* The following function is from INDT's games-startup code. The copyrights 
belongs to them. The INDT's games-startup was released under LGPL license. 
This function is modified under LGPL license*/

/* RPC event handler for indt games */
gint startup_app_rpc_event_cb (const gchar *interface, 
	  	          const gchar *method,
                          GArray      *args, 
                          gpointer     data, 
                          osso_rpc_t  *retval)
{
  AppUIData *app;
  GameState oldstate;
	osso_rpc_t val;
  osso_log (LOG_DEBUG, "%s\n", __FUNCTION__);
  g_return_val_if_fail (data != NULL, OSSO_ERROR);
  g_return_val_if_fail (method != NULL, OSSO_ERROR);
  g_return_val_if_fail (retval != NULL, OSSO_ERROR);
  app = (AppUIData *) (data);
 

  retval->type = DBUS_TYPE_BOOLEAN;
  retval->value.b = TRUE;
  
	oldstate=get_game_state();
  /* Handle pause method */
  if (g_ascii_strcasecmp (method, GAME_PAUSE_METHOD) == 0) {
    //newstate = GAME_PAUSED;
    int with_hw_key=settings_get_int(PAUSE_WITH_HW_KEY);
		osso_log (LOG_DEBUG, "%s: Got pause message\n", __FUNCTION__);
    app->got_pause=TRUE;
		if ((with_hw_key==286)||(with_hw_key==GDK_F5))
		{
			gtk_window_iconify(GTK_WINDOW(app->app));
			settings_set_int(PAUSE_WITH_HW_KEY,0);
			osso_log (LOG_DEBUG, "%s: ..with home key\n", __FUNCTION__);
		}
    else
    {
      gtk_window_present(GTK_WINDOW(app->app));
    }
	  set_gamestate(gs_paused);
    if (GTK_WIDGET_VISIBLE(app->play_button))
		{
			gtk_widget_grab_focus(app->play_button);
		} 
		if (GTK_WIDGET_VISIBLE(app->continue_button))
		{
      gtk_widget_grab_focus(app->continue_button);
    }
  }
  else if (g_ascii_strcasecmp (method, GAME_CLOSE_METHOD) == 0 ||
           g_ascii_strcasecmp (method, GAME_CRASHED_METHOD) == 0 || 
            g_ascii_strcasecmp (method, GAME_END_METHOD) == 0) {
    //newstate = GAME_CLOSED;
    osso_log (LOG_DEBUG, "%s: Got end message\n", __FUNCTION__);
    int with_hw_key=settings_get_int(PAUSE_WITH_HW_KEY);
    if ((with_hw_key==286)||(with_hw_key==GDK_F5))
		{
			gtk_window_iconify(GTK_WINDOW(app->app));
			settings_set_int(PAUSE_WITH_HW_KEY,0);
			osso_log (LOG_DEBUG, "%s: ..with home key\n", __FUNCTION__);
		}
			   set_gamestate(gs_closed);
    if (app->plugin->info->plugin_cb)
    {
            app->plugin->info->plugin_cb(NULL,(gpointer)MA_GAME_ENDED);
    }
  }
  else /* Handle home method */
  if (g_ascii_strcasecmp (method, GAME_HOME_METHOD) == 0) {
	  gtk_window_iconify(GTK_WINDOW(app->app));
    //newstate = GAME_PAUSED;
	  set_gamestate(gs_paused);
    if (GTK_WIDGET_VISIBLE(app->play_button))
		{
			gtk_widget_grab_focus(app->play_button);
		} 
		if (GTK_WIDGET_VISIBLE(app->continue_button))
		{
      gtk_widget_grab_focus(app->continue_button);
    }
  } else /* Handle Load method */
	if (g_ascii_strcasecmp (method, GAME_MIME_OPEN) == 0)
	{
		val	= g_array_index(args, osso_rpc_t, (guint)0);
		osso_log (LOG_DEBUG, "Loading game\n");
		if ((val.type == DBUS_TYPE_STRING)
        && (val.value.s != NULL)) {
      game_mime_open((gchar *)val.value.s);
              
      /* success*/
      retval->type=DBUS_TYPE_BOOLEAN;
      retval->value.b=TRUE;
      return OSSO_OK;
    } else {
      retval->type=DBUS_TYPE_BOOLEAN;
      retval->value.b=FALSE;
    }
	}
	else {
    retval->value.b = FALSE;
    return OSSO_ERROR;
  }

  /*oldstate = app->state;
  app->state = newstate;*/
  if (app->state_change_cb)
  {
    app->state_change_cb (app, oldstate, get_game_state(), app->state_change_cb_data);
  }
  return OSSO_OK;
}

/* The following function is from INDT's games-startup code. The copyrights 
belongs to them. The INDT's games-startup was released under LGPL license. 
This function is modified under LGPL license*/

/* top event cb for indt games */
void 
startup_app_top_event_cb (const gchar *arguments, 
                    gpointer     data)
{
  AppUIData *app;
  osso_log (LOG_DEBUG, "%s\n", __FUNCTION__);

 
  g_return_if_fail (data != NULL);

  app = (AppUIData*) (data);

  if (app->top_cb)
    app->top_cb (app, app->top_cb_data);
}

/* The following function is from INDT's games-startup code. The copyrights 
belongs to them. The INDT's games-startup was released under LGPL license. 
This function is modified under LGPL license*/

/* The callback function to close the application */
void 
startup_ui_game_quit_cb (GtkWidget *widget,
                         gpointer   data)
{
  //AppUIData *ui = data;
	AppData *appdata = get_app_data();
	AppUIData *ui= appdata->app_ui_data;
  //guint state;
	GameState state;
  if (ui->timeouts)
  {
    g_source_remove(ui->timeouts);
  }
  osso_log(LOG_DEBUG,"%s",__FUNCTION__);
  if (ui->async_command_sending==TRUE)
  {
    osso_log(LOG_DEBUG,"Sending is in progress");
    
	  set_game_state_async(GAME_CLOSE);
  }

  if (ui->plugin && ui->plugin->info->write_config)
    ui->plugin->info->write_config ();

  if (ui->plugin && ui->plugin->send_game_close){  
    //state = get_game_state ();
    //if (state == GAME_RUNNING || state == GAME_PAUSED)
	  if (get_game_state() == gs_running || get_game_state() == gs_paused)
      //startup_app_send_game_msg (ui->app, GAME_CLOSE);
    {
      osso_log(LOG_DEBUG,"Send game close needed");
      set_game_state_async(GAME_CLOSE);
    }
  }
  if (ui->plugin && ui->plugin->info->unload)
    ui->plugin->info->unload ();
  /*gtk_widget_hide (GTK_WIDGET (ui->app));
  gtk_main_iteration ();
  gtk_main_quit (); */
}

/* The following function is from INDT's games-startup code. The copyrights 
belongs to them. The INDT's games-startup was released under LGPL license. 
This function is modified under LGPL license*/

/* The callback function to restart the game. It is connected to the reset button */
void 
startup_ui_game_restart_cb (GtkWidget *widget, 
                            gpointer   data)
{
  //gboolean answer = FALSE;
	AppData *appdata = get_app_data();
	AppUIData *ui= appdata->app_ui_data;
  if (ui->low_memory)
  {
    show_infoprint(__("memr_ib_operation_disabled"));
    return;
  }
  //AppUIData *ui = data;
  //if (paused){
	if (get_game_state()==gs_paused){
    gchar *restart_needed=NULL;
    restart_needed=startup_config_read_entry(ui->indt_startup_config,"ShowRestartDialog",NULL);
    if (restart_needed)
    {
      if ( g_ascii_strncasecmp(restart_needed,"FALSE",6)==0)
      {
        if (ui->plugin && ui->plugin->info->write_config)
        ui->plugin->info->write_config ();
      
        hildon_app_set_title (ui->app, ui->game_title);
        if (ui->plugin->info->plugin_cb)
	      {
		      ui->plugin->info->plugin_cb(widget,(gpointer)MA_GAME_PLAYING_START);
	      }
        //startup_app_send_game_msg (ui->app, GAME_RESTART);
	      set_game_state_async(GAME_RESTART);
        return;
      }
    }
    if(ui_show_restart_game_dialog(ui)==GAME_RESPONSE_OK){
      if (ui->plugin && ui->plugin->info->write_config)
        ui->plugin->info->write_config ();
      
      hildon_app_set_title (ui->app, ui->game_title);
      if (ui->plugin->info->plugin_cb)
	    {
		    ui->plugin->info->plugin_cb(widget,(gpointer)MA_GAME_PLAYING_START);
	    }
      //startup_app_send_game_msg (ui->app, GAME_RESTART);
	  set_game_state_async(GAME_RESTART);
    }
  }
}

/* The following function is from INDT's games-startup code. The copyrights 
belongs to them. The INDT's games-startup was released under LGPL license. 
This function is modified under LGPL license*/

/* The callback function to start the INDT game. It is connected to the play button */
void 
startup_ui_game_play_cb (GtkWidget *widget, 
                         gpointer   data)
{
	//osso_rpc_t ret;
  //AppUIData *ui =  (AppUIData *)data;
	AppData *appdata = get_app_data();
	AppUIData *ui= appdata->app_ui_data;
  if (ui->low_memory)
  {
    show_infoprint(__("memr_ib_operation_disabled"));
    return;
  }
	if (get_game_state() == gs_paused){
    if (ui->plugin->info->plugin_cb)
		{
			ui->plugin->info->plugin_cb(widget,(gpointer)MA_GAME_RESET_OLD_SETTINGS);
		}
  }
	if (ui->plugin->info->plugin_cb)
	{
		ui->plugin->info->plugin_cb(widget,(gpointer)MA_GAME_PLAYING_START);
	}
  if (ui->plugin && ui->plugin->info->write_config)
    ui->plugin->info->write_config ();
    //guint state = startup_app_get_game_state (ui->app);
  //if (state == GAME_PAUSED || paused){
  if (get_game_state() == gs_paused){
    //paused = FALSE;
    //startup_app_send_game_msg (ui->app, GAME_CONTINUE);
	  set_game_state_async(GAME_CONTINUE);
  }
  else {
    //startup_app_send_game_msg (ui->app, GAME_RUN);
	  set_game_state_async(GAME_RUN);
  }
  set_gamestate(gs_running);
}

void set_save_menu_state(gboolean state)
{
  osso_log (LOG_DEBUG, "%s\n", __FUNCTION__);
  AppData *appdata = get_app_data();
  AppUIData *ui= appdata->app_ui_data;
  if (state==TRUE)
	{/*Check, if the file is read-only, or not*/
		gchar *file = settings_get_string(SETTINGS_CHESS_SAVE_FILE);
    osso_log (LOG_DEBUG, "%s  44\n", __FUNCTION__);
		osso_log (LOG_DEBUG, "Fname: %s\n", file);
		GnomeVFSFileInfo *file_info = gnome_vfs_file_info_new();
    GnomeVFSFileInfo *dir_info = gnome_vfs_file_info_new();
		GnomeVFSResult vfs_res = file ? gnome_vfs_get_file_info(file,
      file_info,
      GNOME_VFS_FILE_INFO_FOLLOW_LINKS) : GNOME_VFS_ERROR_GENERIC;
    gchar *directory_uri=NULL,*last_slash=NULL;
    if (last_slash=g_strrstr(file,"/"))
    {
      directory_uri=g_strndup(file,last_slash-file);
    }
    GnomeVFSResult vfs_res_dir = directory_uri ? gnome_vfs_get_file_info(directory_uri,
      dir_info,
      GNOME_VFS_FILE_INFO_FOLLOW_LINKS) : GNOME_VFS_ERROR_GENERIC;
    gboolean dir_ok=TRUE;
    if (directory_uri&&(g_strncasecmp(directory_uri,"obex",4)!=0))
    {
      dir_ok=(dir_info->type == GNOME_VFS_FILE_TYPE_DIRECTORY) && (dir_info->permissions&GNOME_VFS_PERM_USER_WRITE);
    }
    else
    {
      dir_ok=TRUE;
    }
		if ((strlen(file)<=1)||((vfs_res == GNOME_VFS_OK) &&
      (file_info->type == GNOME_VFS_FILE_TYPE_REGULAR) && (file_info->permissions&GNOME_VFS_PERM_USER_WRITE)&&
    dir_ok))
		{
			gtk_widget_set_sensitive(ui->game_save_menu, state);
			}
			else
			{
  			gtk_widget_set_sensitive(ui->game_save_menu, FALSE);
				g_signal_connect(G_OBJECT(ui->game_save_menu),
		 "insensitive_press",
		 G_CALLBACK(on_insensitive_menu_press),
		 dgettext("hildon-common-strings","ckdg_ib_game_is_read_only"));

			}
			if (file)
			{
				g_free(file);
			}
      if (directory_uri)
      {
        g_free(directory_uri);
      }
      gnome_vfs_file_info_unref(dir_info);
      gnome_vfs_file_info_unref(file_info);
	}
	else
	{
		gtk_widget_set_sensitive(ui->game_save_menu, state);
	}
	
}

/* The following function is from INDT's games-startup code. The copyrights 
belongs to them. The INDT's games-startup was released under LGPL license. 
This function is modified under LGPL license*/

void 
startup_ui_state_change_cb (AppUIData *app, 
                            guint       oldstate,
                            guint       newstate,
                            gpointer    data)
{
  //AppUIData *ui = (AppUIData *) (data);
	osso_log (LOG_DEBUG, "%s\n", __FUNCTION__);
	AppData *appdata = get_app_data();
	AppUIData *ui= appdata->app_ui_data;
	g_assert(ui);
	//app changed to ui!!
  switch (newstate) {
    case gs_unloaded:
      /*gtk_label_set_text (GTK_LABEL (ui->controls_state_label),
        _("game_ia_startup_not_started"));

      gtk_button_set_label (GTK_BUTTON (ui->play_button), 
        _("game_bv_startup_play"));

      gtk_widget_set_sensitive (ui->restart_button, FALSE);

      gtk_widget_set_sensitive(ui->game_restart_menu, FALSE);
      gtk_widget_set_sensitive (ui->restart_button, FALSE);
      if (ui->game_save_menu && ui->game_saveas_menu){
        gtk_widget_set_sensitive(ui->game_save_menu, FALSE);
        gtk_widget_set_sensitive(ui->game_saveas_menu, FALSE);
      }*/
      game_state_changed(gs_unloaded);
      break;

    case gs_paused:
      ui->timeouts=g_timeout_add(ENABLE_KEY_TIMEOUT,enable_fullscreen,appdata);
      /*gtk_label_set_text (GTK_LABEL (ui->controls_state_label),
        _("game_ia_startup_paused"));

      gtk_button_set_label (GTK_BUTTON (ui->play_button), 
        _("game_bv_startup_continue"));
        
      gtk_widget_set_sensitive (ui->restart_button, TRUE);
			gtk_widget_set_sensitive (ui->play_button, TRUE);*/
    game_state_changed(gs_paused);
    if (ui->plugin)
      if (ui->plugin->info)
        if (ui->plugin->info->plugin_cb)
    ui->plugin->info->plugin_cb(NULL,(gpointer)MA_GAME_PLAYING);

      gtk_widget_set_sensitive(ui->game_restart_menu, TRUE);
      if (ui->game_save_menu && ui->game_saveas_menu){
		set_save_menu_state(TRUE);
        gtk_widget_set_sensitive(ui->game_saveas_menu, TRUE);
      }
      if (ui->plugin && ui->plugin->info->update_menu)
        ui->plugin->info->update_menu ();
      //paused = TRUE;
      break;
      
    case gs_closed:
      /*gtk_label_set_text (GTK_LABEL (ui->controls_state_label),
        _("game_ia_startup_not_started"));

      gtk_button_set_label (GTK_BUTTON (ui->play_button), 
        _("game_bv_startup_play"));

      gtk_widget_set_sensitive (ui->restart_button, FALSE);

      gtk_widget_set_sensitive(ui->game_restart_menu, FALSE);

      if (ui->game_save_menu && ui->game_saveas_menu){
        gtk_widget_set_sensitive(ui->game_save_menu, FALSE);
        gtk_widget_set_sensitive(ui->game_saveas_menu, FALSE);
      }
       */
    game_state_changed(gs_closed);
      if (ui->plugin && ui->plugin->info->update_menu)
        ui->plugin->info->update_menu ();
      break;

    default:
      break;
  }
  set_gamestate(newstate);
}

/* The following function is from INDT's games-startup code. The copyrights 
belongs to them. The INDT's games-startup was released under LGPL license. 
This function is modified under LGPL license*/

/* callback for the menu */
void 
startup_ui_menu_action_cb (gpointer   data, 
                           guint      action, 
                           GtkWidget *menu_item)
{
  //AppUIData *ui = (AppUIData*) (data);
	AppData *appdata = get_app_data();
	AppUIData *ui= appdata->app_ui_data;
  
  switch (action) {
    case MA_GAME_PLAY:
      startup_ui_game_play_cb (menu_item, ui);
      break;
      
    case MA_GAME_RESTART:
      startup_ui_game_restart_cb (menu_item, ui);
      break;
      
    case MA_GAME_CLOSE:
      //startup_ui_game_quit_cb (menu_item, ui);
			game_quit_cb(menu_item,ui);
      break;
  }
}

/**	
  Private variable for AppUIData, initialized using cb_init()
  This is used by the insensitive menu callback(s) 
*/
static AppUIData *_app_ui= NULL;

/**
 Helper function to get an bool entry, returning defined value, if not found.
 @param key GConf key to be get.
 @param fall Fallback to this, if not found, or invalid.
 @return Value got from GConf or the value specified in fall.
*/
static gboolean settings_get_bool_fallback(gchar *key, gboolean fall);

static gboolean settings_get_bool_fallback(gchar *key, gboolean fall)
{
  GConfValue *gc_val = settings_get(key);
  
  if (gc_val)
  {
    if (gc_val->type == GCONF_VALUE_BOOL) {
      g_free(gc_val);
      return settings_get_bool(key);
    }
    g_free(gc_val);
  }
  return fall;
}

/**
 Helper function to get an int entry, returning defined value, if not found.
 @param key GConf key to be get.
 @param fall Fallback to this, if not found, or invalid.
 @return Value got from GConf or the value specified in fall.
*/
static gint settings_get_int_fallback(gchar *key, gint fall);

static gint settings_get_int_fallback(gchar *key, gint fall)
{
  GConfValue *gc_val = settings_get(key);
  if (gc_val)
  {
    if (gc_val->type == GCONF_VALUE_INT) {
      g_free(gc_val);
      return settings_get_int(key);
    }
    g_free(gc_val);
  }
  return fall;
}

/*
  I have written three different callbacks for menus mostly for the
  sake of clarity. Of course it's possible to handle all menu events
  in one big function. --JK
*/

void menu_action( gpointer cb_data, guint cb_action, GtkWidget *menu_item )
{
  AppUIData *app_ui_data;

  app_ui_data = (AppUIData *) cb_data;

  switch( cb_action ) {

  case MA_GAME_PLAY:
    if (get_game_state() == gs_paused)
      game_continue_cb(menu_item,(gpointer)app_ui_data);
    else
      game_play_cb(menu_item,(gpointer)app_ui_data);
    break;
  case MA_GAME_RESTART:
    game_restart_cb(menu_item,(gpointer)app_ui_data);
    break;
  case MA_GAME_RESET:
    game_reset_cb(menu_item,(gpointer)app_ui_data);
    break;
  case MA_GAME_OPEN:
    game_load_cb(menu_item,(gpointer)app_ui_data);
    break;
  case MA_GAME_SAVE:
    game_save_cb(menu_item,(gpointer)app_ui_data);
    break;
  case MA_GAME_SAVE_AS:
    game_save_as_cb(menu_item,(gpointer)app_ui_data);
    break;
  case MA_GAME_HELP:
    game_help_cb(menu_item,(gpointer)app_ui_data);
    break;
  case MA_GAME_RECENT_1:
    game_load_recent_cb(menu_item,1,(gpointer)app_ui_data);
    break;
  case MA_GAME_RECENT_2:
    game_load_recent_cb(menu_item,2,(gpointer)app_ui_data);
    break;
  case MA_GAME_RECENT_3:
    game_load_recent_cb(menu_item,3,(gpointer)app_ui_data);
    break;
  case MA_GAME_RECENT_4:
    game_load_recent_cb(menu_item,4,(gpointer)app_ui_data);
    break;
  case MA_GAME_RECENT_5:
    game_load_recent_cb(menu_item,5,(gpointer)app_ui_data);
    break;
  case MA_GAME_RECENT_6:
    game_load_recent_cb(menu_item,6,(gpointer)app_ui_data);
    break;
  case MA_GAME_HIGH_SCORES:
    mahjongg_scores_cb(menu_item, (gpointer)app_ui_data);
    break;
  case MA_GAME_CLOSE:
    game_clean_up_and_quit(menu_item, (gpointer)app_ui_data);
    break;

  }
  return;
}

/* Radiobutton menuitem callback */
void menu_radio( gpointer cb_data, guint cb_action, GtkWidget *menu_item )
{
  AppUIData *app_ui_data = (AppUIData *) cb_data;

  switch( cb_action ) {
  case MR_GAME_HUMAN_WHITE:
    chess_human_menu_cb(menu_item, (gpointer)app_ui_data,
      chess_human_white);
    break;
  case MR_GAME_HUMAN_BLACK:
    chess_human_menu_cb(menu_item, (gpointer)app_ui_data,
      chess_human_black);
    break;
  case MR_GAME_OPPONENT_1:
    chess_opponent_menu_cb(menu_item, (gpointer)app_ui_data,
      chess_opponent_1);
    break;
  case MR_GAME_OPPONENT_2:
    chess_opponent_menu_cb(menu_item, (gpointer)app_ui_data,
      chess_opponent_2);
    break;
  case MR_GAME_OPPONENT_3:
    chess_opponent_menu_cb(menu_item, (gpointer)app_ui_data,
      chess_opponent_3);
    break;
  case MR_GAME_OPPONENT_4:
    chess_opponent_menu_cb(menu_item, (gpointer)app_ui_data,
      chess_opponent_4);
    break;
  case MR_GAME_OPPONENT_HUMAN:
    chess_opponent_menu_cb(menu_item, (gpointer)app_ui_data,
      chess_opponent_human);
    break;
  case MR_GAME_BOARD_1:
    mahjong_board_menu_cb(menu_item, (gpointer)app_ui_data,
      mahjong_board_1);
    break;
  case MR_GAME_BOARD_2:
    mahjong_board_menu_cb(menu_item, (gpointer)app_ui_data,
      mahjong_board_2);
    break;
  case MR_GAME_BOARD_3:
    mahjong_board_menu_cb(menu_item, (gpointer)app_ui_data,
      mahjong_board_3);
    break;
  case MR_GAME_BOARD_4:
    mahjong_board_menu_cb(menu_item, (gpointer)app_ui_data,
      mahjong_board_4);
    break;
  case MR_GAME_BOARD_5:
    mahjong_board_menu_cb(menu_item, (gpointer)app_ui_data,
      mahjong_board_5);
    break;
  case MR_GAME_BOARD_6:
    mahjong_board_menu_cb(menu_item, (gpointer)app_ui_data,
      mahjong_board_6);
    break;
  case MR_GAME_BOARD_7:
    mahjong_board_menu_cb(menu_item, (gpointer)app_ui_data,
      mahjong_board_7);
    break;
  case MR_GAME_BOARD_8:
    mahjong_board_menu_cb(menu_item, (gpointer)app_ui_data,
      mahjong_board_8);
    break;

  case MR_GAME_LMARBLES_EASY:
    lmarbles_difficulty_menu_cb(menu_item, (gpointer)app_ui_data,
      lmarbles_difficulty_1);
    break;
  case MR_GAME_LMARBLES_NORMAL:
    lmarbles_difficulty_menu_cb(menu_item, (gpointer)app_ui_data,
      lmarbles_difficulty_2);
    break;
  case MR_GAME_LMARBLES_HARD:
    lmarbles_difficulty_menu_cb(menu_item, (gpointer)app_ui_data,
      lmarbles_difficulty_3);
    break;
  case MR_GAME_LMARBLES_BRAINSTORM:
    lmarbles_difficulty_menu_cb(menu_item, (gpointer)app_ui_data,
      lmarbles_difficulty_4);
    break;
  }
  return;
}

/* Toggle menuitem callback */
void menu_toggle( gpointer cb_data, guint cb_action, GtkWidget *menu_item )
{
  AppUIData *app_ui_data;

  app_ui_data = (AppUIData *) cb_data;

  switch( cb_action ) {
  case MT_GAME_SOUND:
    game_menu_sound_cb(menu_item,(gpointer)app_ui_data);
    break;
  case MT_GAME_LEGAL_MOVES:
    game_menu_moves_cb(menu_item,(gpointer)app_ui_data);
    break;
  }
  return;
}

GtkTreeModel *create_highscore_list(struct HighScore *scores)
{
  int i=0;
  GtkTreeStore *tree_store;
  GtkTreeIter iter;

  if (scores==NULL) return NULL;

  /* New tree store with HIGHSCORE_COLS columns, specifying three
     first of them to be string */
  tree_store = gtk_tree_store_new(
    HIGHSCORE_COLS,
    G_TYPE_STRING,
    G_TYPE_STRING,
    G_TYPE_STRING
    );
  
  /* Building contents for the tree store */
  for (i=0; i<HIGHSCORE_SIZE; i++) {
    if (scores[i].level!=NULL /*&& 
        scores[i].time!=NULL && 
        scores[i].date!=NULL*/) {
      /* New row */
      gtk_tree_store_append( tree_store,
        &iter, NULL);
      /* Setting string values for the specified columns */
      gtk_tree_store_set( tree_store, &iter,
        HIGHSCORE_LEVEL_COLUMN, scores[i].level,
        HIGHSCORE_TIME_COLUMN, scores[i].time,
        HIGHSCORE_DATE_COLUMN, scores[i].date,
        -1);
    }
  }

  return GTK_TREE_MODEL(tree_store);
}

void chess_human_cb(GtkWidget *widget, gpointer data)
{
  gint active = gtk_combo_box_get_active(
      GTK_COMBO_BOX( widget ));
  gchar *path = NULL;

  /* Getting path of menu item */
  switch (active) {
    default:
    case chess_human_invalid:
    case chess_human_white:
      path = "/gamemechessmainmenusettings"
             "/gamemechessmenusettingshuman"
             "/gamemechessmenusettingshumanwhite";
    break;

    case chess_human_black:
      path = "/gamemechessmainmenusettings"
             "/gamemechessmenusettingshuman"
             "/gamemechessmenusettingshumanblack";
    break;
  }

  /* Setting selected path item to be selected*/
  if (ui_menu_item_get_active((AppUIData*)data, path)==FALSE) {
    ((AppUIData*)data)->skip_box_update = TRUE;
    ui_menu_item_set_active((AppUIData*)data,
      path,
      TRUE);
  }  
}

void chess_human_menu_cb(GtkWidget *widget, gpointer data, guint active)
{
  if (!((AppUIData*)data)->skip_box_update == TRUE) {

    /* Change human choice */
    gtk_combo_box_set_active(
      GTK_COMBO_BOX( ((AppUIData*)data)->chess_human_choice ),
      active);
  }

  ((AppUIData*)data)->skip_box_update = FALSE;
}

void chess_opponent_cb(GtkWidget *widget, gpointer data)
{
  gint active = gtk_combo_box_get_active(
      GTK_COMBO_BOX( widget ));
  gchar *path = NULL;

  /* Get path to correct opponent selection */
  switch (active) {
    default:
    case chess_opponent_1:
      path = "/gamemechessmainmenusettings"
             "/gamemechessmenusettingsopponent"
             "/gamemechessmenusettingsopponent1";
    break;

    case chess_opponent_2:
      path = "/gamemechessmainmenusettings"
             "/gamemechessmenusettingsopponent"
             "/gamemechessmenusettingsopponent2";
    break;

    case chess_opponent_3:
      path = "/gamemechessmainmenusettings"
             "/gamemechessmenusettingsopponent"
             "/gamemechessmenusettingsopponent3";
    break;

    case chess_opponent_4:
      path = "/gamemechessmainmenusettings"
             "/gamemechessmenusettingsopponent"
             "/gamemechessmenusettingsopponent4";
    break;

    case chess_opponent_human:
      path = "/gamemechessmainmenusettings"
             "/gamemechessmenusettingsopponent"
             "/gamemechessmenusettingsopponenthuman";
    break;
  }

  /* Set corresponding menu item active */
  if (ui_menu_item_get_active((AppUIData*)data, path)==FALSE) {
    ((AppUIData*)data)->skip_box_update = TRUE;
    ui_menu_item_set_active((AppUIData*)data,
      path,
      TRUE);
  }  
}

void chess_opponent_menu_cb(GtkWidget *widget, gpointer data, guint active)
{
  if (!((AppUIData*)data)->skip_box_update == TRUE)

  gtk_combo_box_set_active(
    GTK_COMBO_BOX( ((AppUIData*)data)->chess_opponent_choice ),
    active
    );

  ((AppUIData*)data)->skip_box_update = FALSE;
}

void mahjong_board_cb(GtkWidget *widget, gpointer data)
{
  gint active = gtk_combo_box_get_active(
      GTK_COMBO_BOX( widget ));
  gchar *path = NULL;

  /* Getting path of menu item */
  switch (active) {
    default:
    case mahjong_board_invalid:
    case mahjong_board_1:
      path = "/gamememahjongmainmenuboard"
             "/gamememahjongmenuboard1";
    break;

    case mahjong_board_2:
      path = "/gamememahjongmainmenuboard"
             "/gamememahjongmenuboard2";
    break;

    case mahjong_board_3:
      path = "/gamememahjongmainmenuboard"
             "/gamememahjongmenuboard3";
    break;

    case mahjong_board_4:
      path = "/gamememahjongmainmenuboard"
             "/gamememahjongmenuboard4";
    break;

    case mahjong_board_5:
      path = "/gamememahjongmainmenuboard"
             "/gamememahjongmenuboard5";
    break;

    case mahjong_board_6:
      path = "/gamememahjongmainmenuboard"
             "/gamememahjongmenuboard6";
    break;

    case mahjong_board_7:
      path = "/gamememahjongmainmenuboard"
             "/gamememahjongmenuboard7";
    break;

    case mahjong_board_8:
      path = "/gamememahjongmainmenuboard"
             "/gamememahjongmenuboard8";
    break;
  }

  /* Setting selected path item to be selected*/
  if (ui_menu_item_get_active((AppUIData*)data, path)==FALSE) {
    ((AppUIData*)data)->skip_box_update = TRUE;
    ui_menu_item_set_active((AppUIData*)data,
      path,
      TRUE);
  }  
}

void mahjong_board_menu_cb(GtkWidget *widget, gpointer data, guint active)
{
  if (!((AppUIData*)data)->skip_box_update == TRUE)

  gtk_combo_box_set_active(
    GTK_COMBO_BOX( ((AppUIData*)data)->mahjong_board_choice ),
    active
    );

  ((AppUIData*)data)->skip_box_update = FALSE;
}

void lmarbles_difficulty_cb(GtkWidget *widget, gpointer data)
{
  gint active = gtk_combo_box_get_active(
      GTK_COMBO_BOX( widget ));
  gchar *path = NULL;

  /* Getting path of menu item */
  switch (active) {
    default:
    case lmarbles_difficulty_invalid:
    case lmarbles_difficulty_1:
      path = "/gamemelmarblesmainmenusettings"
             "/gamemelmarblesmenusettingseasy";
    break;

    case lmarbles_difficulty_2:
      path = "/gamemelmarblesmainmenusettings"
             "/gamemelmarblesmenusettingsnormal";
    break;

    case lmarbles_difficulty_3:
      path = "/gamemelmarblesmainmenusettings"
             "/gamemelmarblesmenusettingshard";
    break;

    case lmarbles_difficulty_4:
      path = "/gamemelmarblesmainmenusettings"
             "/gamemelmarblesmenusettingsbrainstorm";
    break;

  }

  /* Setting selected path item to be selected*/
  if (ui_menu_item_get_active((AppUIData*)data, path)==FALSE) {
    ((AppUIData*)data)->skip_box_update = TRUE;
    ui_menu_item_set_active((AppUIData*)data,
      path,
      TRUE);
  }
}

void lmarbles_difficulty_menu_cb(GtkWidget *widget, gpointer data,
  guint active)
{
  if (!((AppUIData*)data)->skip_box_update == TRUE)

  gtk_combo_box_set_active(
    GTK_COMBO_BOX( ((AppUIData*)data)->lmarbles_difficulty_choice ),
    active
    );

  ((AppUIData*)data)->skip_box_update = FALSE;
}

void menu_set_save_dimmed(AppUIData *app_ui_data, gboolean dimmed) {
  if (app_ui_data->plugin&&app_ui_data->plugin->menu_open_save) {
    gtk_widget_set_sensitive(app_ui_data->game_save_menu, dimmed);
    gtk_widget_set_sensitive(app_ui_data->game_saveas_menu, dimmed);
  }
  
#if 0
  gchar *path = "";

  /* Setting save options (un)dimmed in menu */
  path = "/gamemechessmainmenugame/gamemechessmenugamesave";
  ui_menu_item_set_sensitive(app_ui_data, path, dimmed);

  path = "/gamemechessmainmenugame/gamemechessmenugamesaveas";
  ui_menu_item_set_sensitive(app_ui_data, path, dimmed);  
#endif
}

void game_sound_cb(GtkWidget *widget, gpointer data)
{
  gchar *path = "";

  /* Get path to current game sound setting in menu */
  switch (get_game_type()) {
    default:
    case gt_chess:
    path = "/gamemechessmainmenusettings"
      "/gamemechessmenusettingssound";
    break;

    case gt_mahjong:
    path = "/gamememahjongmainmenusettings"
      "/gamememahjongmenusettingssound";
    break;

    case gt_lmarbles:
    path = "/gamemelmarblesmainmenusettings"
      "/gamemelmarblesmenusettingssound";
    break;
  }

  /* Set menu item activity */
  ui_menu_item_set_active((AppUIData*)data,
    path,
    gtk_toggle_button_get_active(
      GTK_TOGGLE_BUTTON( widget ))
    );

  set_settings((AppUIData*)data);
}

void game_moves_cb(GtkWidget *widget, gpointer data)
{
  /* Set menu item activity */
  ui_menu_item_set_active((AppUIData*)data,
    "/gamemechessmainmenusettings/gamemechessmenusettingslegalmoves",
    gtk_toggle_button_get_active(
      GTK_TOGGLE_BUTTON( widget ))
    );

  set_settings((AppUIData*)data);
}

void game_menu_sound_cb(GtkWidget *widget, gpointer data)
{
  GtkWidget *sound_check = NULL;

  /* Select sound setting widget by game type */
  switch (get_game_type()) {
    default:
    case gt_chess:
    sound_check = ((AppUIData*)data)->chess_sound_check;
    break;

    case gt_mahjong:
    sound_check = ((AppUIData*)data)->mahjong_sound_check;
    break;

    case gt_lmarbles:
    sound_check = ((AppUIData*)data)->lmarbles_sound_check;
    break;
  }

  /* Set activity by menu item */
  gtk_toggle_button_set_active(
    GTK_TOGGLE_BUTTON( sound_check ),
    gtk_check_menu_item_get_active(
      GTK_CHECK_MENU_ITEM( widget ) )
    );
}

void game_menu_moves_cb(GtkWidget *widget, gpointer data)
{
  /* Set activity by menu item */
  gtk_toggle_button_set_active(
    GTK_TOGGLE_BUTTON( ((AppUIData *)data)->chess_moves_check ),
    gtk_check_menu_item_get_active(
      GTK_CHECK_MENU_ITEM( widget ) )
    );
}

gboolean game_quit_cb(GtkWidget *widget, gpointer data)
{
  AppData *appdata=get_app_data();
  AppUIData *ui=appdata->app_ui_data;
  
  if (ui->plugin)
  {
	  startup_ui_game_quit_cb(NULL,NULL);
	  //return;
  }
  else
  {
	  set_settings( get_app_data()->app_ui_data );
    game_close();
  }
  gtk_widget_hide( GTK_WIDGET(get_app_data()->app_ui_data->app) );
  gtk_main_iteration();
//  clear_state((gpointer)get_app_data());
  //game_close();
  gtk_main_quit();
  return TRUE;
}

void game_untop_cb(GtkWidget *widget, gpointer data)
{
  AppData *ad = get_app_data();
  data=ad;
  save_state((AppData *)data);
  hildon_app_set_killable(((AppData *)data)->app_ui_data->app, TRUE);
  
  /* Remove menu when app lose focus */
  gtk_menu_popdown(((AppData *)data)->app_ui_data->main_menu_bar);
  if (ad->app_ui_data->plugin->info->plugin_cb)
	{
    ad->app_ui_data->plugin->info->plugin_cb(NULL,(gpointer)STARTUP_UNTOP);
  }
}

void game_top_cb(GtkWidget *widget, gpointer data)
{
	osso_log (LOG_DEBUG, "%s\n", __FUNCTION__);
  hildon_app_set_killable(((AppData *)data)->app_ui_data->app, FALSE);
}

void get_settings(AppUIData *app_ui_data)
{
  gboolean white = TRUE;
  gboolean human = FALSE;
  gboolean sound = FALSE;
  gboolean moves = FALSE;
  gint player = -1;
  gint difficulty = -1;

  switch (get_game_type()) {
    /* Chess settings */
    case gt_chess:

    /* Get settings */
    white = settings_get_bool_fallback(SETTINGS_CHESS_PLAYER_COLOR, TRUE);
    sound = settings_get_bool_fallback(SETTINGS_CHESS_ENABLE_SOUND, TRUE);

    difficulty = settings_get_int_fallback(SETTINGS_CHESS_DIFFICULTY, 4);

    human = settings_get_bool(SETTINGS_CHESS_OPPONENT_TYPE);
    moves = settings_get_bool(SETTINGS_CHESS_SHOW_LEGAL_MOVES);

    /* Sound */
    gtk_toggle_button_set_active(
      GTK_TOGGLE_BUTTON( app_ui_data->chess_sound_check), sound );

    /* Legal moves */
    gtk_toggle_button_set_active(
      GTK_TOGGLE_BUTTON( app_ui_data->chess_moves_check), moves );

    /* Human color */
    if (white == TRUE) {
      player = chess_human_white;
    } else {
      player = chess_human_black;
    }
    gtk_combo_box_set_active(
      GTK_COMBO_BOX( app_ui_data->chess_human_choice ), player );

    /* Opponent difficulty and type */
    gtk_combo_box_set_active(
      GTK_COMBO_BOX( app_ui_data->chess_opponent_choice ), difficulty );
    break;

    /* Mahjong settings */
    case gt_mahjong:

    /* Get settings */
    sound = settings_get_bool_fallback(SETTINGS_MAHJONG_ENABLE_SOUND, TRUE);
    difficulty = settings_get_int(SETTINGS_MAHJONG_BOARD_LAYOUT);

    /* Sound */
    gtk_toggle_button_set_active(
      GTK_TOGGLE_BUTTON( app_ui_data->mahjong_sound_check), sound );

    /* Board type */
    gtk_combo_box_set_active(
      GTK_COMBO_BOX( app_ui_data->mahjong_board_choice ), difficulty );
    break;

    /* Lmarbles settings */
    case gt_lmarbles:
    sound = settings_get_bool_fallback(SETTINGS_LMARBLES_ENABLE_SOUND, TRUE);
    difficulty = settings_get_int_fallback(SETTINGS_LMARBLES_DIFFICULTY, 1);

    gtk_toggle_button_set_active(
        GTK_TOGGLE_BUTTON( app_ui_data->lmarbles_sound_check ), sound);
    gtk_combo_box_set_active(
        GTK_COMBO_BOX( app_ui_data->lmarbles_difficulty_choice ), difficulty);
    break;

    default:
    case gt_none:
    break;
  }

}

/* Read settings and store them to GConf */
void set_settings(AppUIData *app_ui_data)
{
  gboolean white = TRUE;
  gboolean human = FALSE;
  gboolean sound = TRUE;
  gboolean legal_moves = FALSE;
  gint player = -1;
  gint difficulty = -1;

  g_assert(app_ui_data);
	
	osso_log(LOG_DEBUG,"%s\n",__FUNCTION__);

  switch (get_game_type()) {
    /* Chess settings */
    case gt_chess:

    /* Sound */
    sound = gtk_toggle_button_get_active(
      GTK_TOGGLE_BUTTON( app_ui_data->chess_sound_check) );

    /* Human color */
    player = gtk_combo_box_get_active(
      GTK_COMBO_BOX( app_ui_data->chess_human_choice ) );
    if (player == chess_human_white) white = TRUE;
    else white = FALSE;

    /* Opponent difficulty and type */
    difficulty = gtk_combo_box_get_active(
      GTK_COMBO_BOX( app_ui_data->chess_opponent_choice ) );

    if (difficulty == chess_opponent_human) human = TRUE;
    else human = FALSE;

    /* Show legal moves */
    legal_moves = gtk_toggle_button_get_active(
      GTK_TOGGLE_BUTTON( app_ui_data->chess_moves_check) );

    /* Set settings */
    settings_set_bool(SETTINGS_CHESS_PLAYER_COLOR,
      white);
    settings_set_bool(SETTINGS_CHESS_ENABLE_SOUND,
      sound);
    settings_set_bool(SETTINGS_CHESS_SHOW_LEGAL_MOVES,
      legal_moves);
    settings_set_int(SETTINGS_CHESS_DIFFICULTY,
      difficulty);
    settings_set_bool(SETTINGS_CHESS_OPPONENT_TYPE,
      human);
    break;

    /* Mahjong settings */
    case gt_mahjong:

    /* Sound */
    sound = gtk_toggle_button_get_active(
      GTK_TOGGLE_BUTTON( app_ui_data->mahjong_sound_check) );

    /* Board type */
    difficulty = gtk_combo_box_get_active(
      GTK_COMBO_BOX( app_ui_data->mahjong_board_choice ) );

    settings_set_bool(SETTINGS_MAHJONG_ENABLE_SOUND,
      sound);
    settings_set_int(SETTINGS_MAHJONG_BOARD_LAYOUT,
      difficulty);
    break;

    /* TODO No support for Lmarbles games yet */
    default:
    /* Sound */
    sound = gtk_toggle_button_get_active(
      GTK_TOGGLE_BUTTON( app_ui_data->lmarbles_sound_check) );

    /* Board type */
    difficulty = gtk_combo_box_get_active(
      GTK_COMBO_BOX( app_ui_data->lmarbles_difficulty_choice ) );

    settings_set_bool(SETTINGS_LMARBLES_ENABLE_SOUND,
      sound);
    settings_set_int(SETTINGS_LMARBLES_DIFFICULTY,
      difficulty);
    break;
  }
}

void game_play_cb(GtkWidget *widget, gpointer data)
{
  AppData *app=get_app_data();
  if (app->app_ui_data->low_memory)
  {
    show_infoprint(__("memr_ib_operation_disabled"));
    return;
  }
  GameState gs = get_game_state();
	
	if (get_game_type()==gt_indt)
	{
		startup_ui_game_play_cb(widget,data);
		return;
	}
	
  
  if ((gs == gs_unloaded) || (gs == gs_closed)) { 
    set_settings((AppUIData*)data);
    game_play();
    ui_update_status((AppUIData*)data);
  }
}

void game_continue_cb(GtkWidget *widget, gpointer data)
{
  AppData *app=get_app_data();
  GameState gs = get_game_state();
  osso_log(LOG_DEBUG,"%s\n",__FUNCTION__);
  if (app->app_ui_data->low_memory)
  {
    show_infoprint(__("memr_ib_operation_disabled"));
    return;
  }
	
  startup_ui_game_play_cb(NULL,NULL);
  /*if (gs==gs_paused) {
  if (app->app_ui_data->plugin->info->plugin_cb)
	{
		app->app_ui_data->plugin->info->plugin_cb(widget,(gpointer)MA_GAME_PLAYING_START);
	}
    set_settings((AppUIData*)data);
    game_continue();
    ui_update_status((AppUIData*)data);
  }*/
}

void game_restart_cb(GtkWidget *widget, gpointer data)
{
  guint answer = 0;
  GameType gt = get_game_type();
  GameState gs = get_game_state();
  AppData *app=get_app_data();
  if (app->app_ui_data->low_memory)
  {
    show_infoprint(__("memr_ib_operation_disabled"));
    return;
  }

  if (gt == gt_chess || gt == gt_mahjong) {
    if (gs == gs_paused) {
      answer = ui_show_restart_game_dialog((AppUIData *)data);
    }
  } else if (gt == gt_lmarbles) {
    answer = GAME_RESPONSE_OK;
  } else if (gt==gt_indt)
  {
	  startup_ui_game_restart_cb(widget,data);
	  return;
  }

  if (answer == GAME_RESPONSE_OK) {
    hildon_appview_set_title(((AppUIData*)data)->main_view, "");
    set_settings((AppUIData*)data);
    game_restart();
    ui_update_status((AppUIData*)data);
  }  
}

void game_reset_cb(GtkWidget *widget, gpointer data)
{
  gchar *path = NULL;

  path = "/gamemelmarblesmainmenusettings"
         "/gamemelmarblesmenusettingsreset"
         "/gamemelmarblesmenusettingsresetprogress";
  
  if (ui_menu_item_get_active((AppUIData*)data, path) == FALSE)
    return;
  
  if (ui_show_reset_game_progress_dialog((AppUIData *)data) == GAME_RESPONSE_OK)
  {
    gchar *profile = g_build_filename(g_get_home_dir(),
        ".lmarbles_profile", NULL);
    g_unlink(profile);
    g_free(profile);
    gchar *state = g_build_filename(g_get_home_dir(),
        ".lmarbles_state", NULL);
    g_unlink(state);
    g_free(state);
  } else {
    /* Unset menu item activity */
    ui_menu_item_set_active((AppUIData*)data, path, FALSE);

    set_settings((AppUIData*)data);  
  }
}

void game_load_recent_cb(GtkWidget *widget, int num, gpointer data)
{
  AppUIData *app_ui_data = data;
    
  gchar *result = g_slist_nth_data(app_ui_data->chess_recent_items, num - 1);

  if (!result)
    return;
  if (!settings_set_string(SETTINGS_CHESS_SAVE_FILE, result))
    return;

  /* Check file existing */
  GnomeVFSURI *src_uri;
  gchar *src_text;
  gchar *uri_scheme;
  gboolean file_not_found = FALSE;

  src_text = g_strdup(result);
  uri_scheme = gnome_vfs_get_uri_scheme(src_text);
  if (!uri_scheme)
    src_text = gnome_vfs_get_uri_from_local_path(src_text);
  else
    g_free(uri_scheme);

  src_uri = gnome_vfs_uri_new(src_text);
  if (!gnome_vfs_uri_exists(src_uri)) {
    file_not_found = TRUE;
    ui_show_load_err(GAME_LOAD_FILE_NOT_FOUND, app_ui_data, result);
  }
  g_free(src_text);
  gnome_vfs_uri_unref(src_uri);
  if (file_not_found)
    return;
  /* End check file existing */  
  
  add_recent_item(app_ui_data, result);
  ui_update_recent_items(app_ui_data);

  set_settings(app_ui_data);

  game_load();
  
  ui_update_status((AppUIData*)data);
}

void game_load_cb(GtkWidget *widget, gpointer data)
{
  if (ui_show_file_chooser(TRUE,(AppUIData*)data) == TRUE) {
    set_settings((AppUIData*)data);
    ui_update_recent_items((AppUIData *)data);
    game_load();
  }
  ui_update_status((AppUIData*)data);
}

void game_save_cb(GtkWidget *widget, gpointer data)
{
  gchar *file = settings_get_string(SETTINGS_CHESS_SAVE_FILE);
  GnomeVFSFileInfo *file_info = gnome_vfs_file_info_new();
  GnomeVFSResult vfs_res = file ? gnome_vfs_get_file_info(file,
      file_info,
      GNOME_VFS_FILE_INFO_FOLLOW_LINKS) : GNOME_VFS_ERROR_GENERIC;

  set_settings((AppUIData *)data);
  if (vfs_res == GNOME_VFS_OK &&
      file_info->type == GNOME_VFS_FILE_TYPE_REGULAR) {
    is_new_file = FALSE;
    game_save();
  }
  else if (ui_show_file_chooser(FALSE,(AppUIData*)data) == TRUE) {
    is_new_file = TRUE;
    game_save();
    ui_update_recent_items((AppUIData *)data);
  }

  gnome_vfs_file_info_unref(file_info);

  ui_update_status((AppUIData*)data);
  
  if (file != NULL) 
    g_free(file);
}

void game_help_cb(GtkWidget *widget, gpointer data)
{
  const gchar *help_path;
  switch (get_game_type())
  {
    case gt_lmarbles:
      help_path = OSSO_MARBLES_HELP_PATH;
      break;
    case gt_mahjong:
      help_path = OSSO_MAHJONG_HELP_PATH;
      break;
    case gt_chess:
    default:
      help_path = OSSO_CHESS_HELP_PATH;
      break;
  }
  ossohelp_show(get_app_data()->app_osso_data->osso,
      help_path,
      0);
}

void game_save_as_cb(GtkWidget *widget, gpointer data)
{
  gchar *old_filename = settings_get_string(SETTINGS_CHESS_SAVE_FILE);
  
  if (ui_show_file_chooser(FALSE,(AppUIData*)data) == TRUE) {
    gchar *new_filename = settings_get_string(SETTINGS_CHESS_SAVE_FILE);
    if (g_strcasecmp(old_filename, new_filename) != 0) {
      is_new_file = TRUE;
    } else {
      is_new_file = FALSE;
    }
    if (new_filename) {
      g_free(new_filename);
    }
    
    game_save();
    ui_update_recent_items((AppUIData *)data);
  }
  ui_update_status((AppUIData*)data);
  
  if (old_filename) {
    g_free(old_filename);
  }
}

void chess_change_cb(GtkWidget *widget, gpointer data)
{
  set_game_type(GAME_CHESS);

  ui_update_status(data);
}

void mahjong_change_cb(GtkWidget *widget, gpointer data)
{
  set_game_type(GAME_MAHJONG);

  ui_update_status(data);
}

void lmarbles_change_cb(GtkWidget *widget, gpointer data)
{
  set_game_type(GAME_LMARBLES);

  ui_update_status(data);
}

/* Callback for Mahjongg high scores */
void mahjongg_scores_cb(GtkWidget* w, gpointer data)
{
  struct HighScore scores[HIGHSCORE_SIZE];
  GameState gs = get_game_state();
  
  if (gs != gs_running) {
    highscore_load(scores);

    ((AppUIData*)data)->mahjong_tree_model = (GtkWidget*)create_highscore_list(scores);
    ui_show_highscore((AppUIData*)data);
    
    highscore_unload(scores);
  }
}

/* Set restart option state in menu */
void menu_set_restart_dimmed( AppUIData *app_ui_data, gboolean mode )
{
    gtk_widget_set_sensitive(app_ui_data->game_restart_menu, mode);
#if 0
  switch (get_game_type()) {
    default:
    case gt_chess:
    ui_menu_item_set_sensitive(app_ui_data,
    "/gamemechessmainmenugame/gamemechessmenugamerestart",
    mode);
    break;

    case gt_mahjong:
    ui_menu_item_set_sensitive(app_ui_data,
    "/gamememahjongmainmenugame/gamememahjongmenugamerestart",
    mode);
    break;

    case gt_lmarbles:
    ui_menu_item_set_sensitive(app_ui_data,
    "/gamemelmarblesmainmenugame/gamemelmarblesmenugamerestart",
    mode);
    break;
  }
#endif
}


gboolean key_press(GtkWidget *widget, GdkEventKey *event, gpointer data)
{
  AppUIData *app_ui_data = ((AppData *)data)->app_ui_data;
  GameState gs;
  if (event->state & (GDK_CONTROL_MASK |
                      GDK_SHIFT_MASK |
                      GDK_MOD1_MASK |
                      GDK_MOD3_MASK |
                      GDK_MOD4_MASK |
                      GDK_MOD5_MASK)) {
    return FALSE;
  }

  switch (event->keyval)
  {
    //case GDK_Return:
    case GDK_F6:
          gs = get_game_state();
    if (!app_ui_data->got_pause)
    {
#ifndef VERSION_3
      if (gtk_widget_is_focus(app_ui_data->play_button)||gtk_widget_is_focus(app_ui_data->continue_button))
      {
#endif
      if ((gs == gs_unloaded) || (gs == gs_closed) || (gs == gs_paused)) {
#ifndef VERSION_3
        if (ui_show_start_game_dialog(app_ui_data) == GAME_RESPONSE_OK) {
#endif
          //menu_action(app_ui_data, MA_GAME_PLAY, NULL);
          startup_ui_game_play_cb(NULL,NULL);
#ifndef VERSION_3
        }

      }
      return TRUE;
      }
      else
      {
        return FALSE;
      }
#else
    }
      return TRUE;
#endif
  }
      break;  
    case GDK_Escape:
      if (app_ui_data->chess_save_done == GAME_SAVE_SAVING) {
        saving_dialog_cb(GTK_DIALOG(app_ui_data->chess_saving_dialog), 
                         GTK_RESPONSE_CANCEL, data);
      }
      return FALSE;
      break;
    case GDK_F5:
      if (app_ui_data->plugin->info->plugin_cb)
		{
			app_ui_data->plugin->info->plugin_cb(NULL,(gpointer)MA_GAME_HOME_PRESSED);
		}
      return FALSE;
      break;
    
  }
  return FALSE;
}

/* Initializes data for callbacks module */
void cb_init(AppUIData * app_ui)
{
  /* store app ui data pointer */
  _app_ui = app_ui;
}

void show_infoprint(gchar *msg)
{
  AppData *ad = get_app_data();
  if (ad!=NULL) {
    gtk_infoprint( GTK_WINDOW(ad->app_ui_data->app), msg);
  }
}

/* Handles insensitive menu press events, display infoprint*/
void on_insensitive_menu_press(GtkWidget * widget, gchar * message)
{
  gtk_infoprint( GTK_WINDOW (_app_ui->app), message);
}

/* Created */
gboolean on_delete_event( GtkWidget * widget, gpointer data)
{
  g_assert(data);

  game_clean_up_and_quit(widget, data);

  return TRUE;    /* we've handled it (FALSE would _not_ work!) */
}

void game_clean_up_and_quit( GtkWidget *widget, gpointer data)
{
  /*
  GameState gs = get_game_state();
  AppData *ad = get_app_data();

  if (get_game_type()==gt_chess &&
      !ad->app_ui_data->chess_saved &&
      (gs==gs_running || gs==gs_paused)) {
    gint answer = ui_show_save_before_closing_dialog( (AppUIData*)data );

    if (answer == GAME_RESPONSE_OK) game_save();
    else if ((answer == GAME_RESPONSE_NO ) || (answer == GAME_RESPONSE_CANCEL)) return;
  }
  */
  
  GnomeVFSResult gnomevfs_res;
  if (gnomevfs_handle != NULL) {
    gnomevfs_res = gnome_vfs_monitor_cancel(gnomevfs_handle);
  }

  ui_clean_up((AppUIData *) data);
  game_quit_cb(widget, data);
}

/*
Notes: When renaming a file, GNOME_VFS_MONITOR_EVENT_DELETED is emitted for the old 
name and GNOME_VFS_MONITOR_EVENT_CREATED is emitted for the new name. Must have a 
monitor added for the location of the new file in order to receive that 
GNOME_VFS_MONITOR_EVENT_CREATED notification.

Only process CHESS_FILE_GLOB file extension
*/
void chess_saved_file_monitor( GnomeVFSMonitorHandle *handle,
			const gchar *monitor_uri,
			const gchar *info_uri,
			GnomeVFSMonitorEventType event_type,
			gpointer user_data)
{
  AppUIData *app_ui_data = (AppUIData *)user_data;

  gchar *file_ext, *chess_file_glob;
  gchar *deleted_file_name;
  
  static gboolean rename_current_file = FALSE;

  if (handle != gnomevfs_handle)
    return;
    
  /* Get current event file extension */
  file_ext = g_strrstr(info_uri, ".");
  if (!file_ext) {
    return;
  }

  /* Get chess saved file extension */
  chess_file_glob = g_strrstr(CHESS_FILE_GLOB, ".");
  if (!chess_file_glob) {
    return;
  }
  
  if (g_strcasecmp(file_ext, chess_file_glob) != 0) {
    return;
  }
  
  switch (event_type) {
      case GNOME_VFS_MONITOR_EVENT_DELETED:
        deleted_file_name = settings_get_string(SETTINGS_CHESS_SAVE_FILE);
        if (g_strcasecmp(deleted_file_name, info_uri) == 0) {// current open file was renamed
          rename_current_file = TRUE;
        } else {
          rename_current_file = FALSE;
        }
        g_free(deleted_file_name);

        ui_update_recent_items(app_ui_data);  
        break;
      case GNOME_VFS_MONITOR_EVENT_CREATED:
        if (rename_current_file == TRUE) {
          settings_set_string(SETTINGS_CHESS_SAVE_FILE, info_uri);
          add_recent_item(app_ui_data, info_uri);
          ui_update_recent_items(app_ui_data);
          
          rename_current_file = FALSE;
        }
        break;
      case GNOME_VFS_MONITOR_EVENT_CHANGED:
      case GNOME_VFS_MONITOR_EVENT_STARTEXECUTING:
      case GNOME_VFS_MONITOR_EVENT_STOPEXECUTING:
      case GNOME_VFS_MONITOR_EVENT_METADATA_CHANGED:
      	break;
  }
    
  return;
}

void saving_dialog_cb(GtkDialog *dialog, gint button, gpointer user_data)
{
  AppUIData *app_ui_data = ((AppData*)user_data)->app_ui_data;
  
  if (button == GTK_RESPONSE_CANCEL) {
    if (app_ui_data->chess_save_done == GAME_SAVE_SAVING) {
        ui_cancel_saving();
    }
  }
}

void volume_unmounted_cb(GnomeVFSVolumeMonitor *vfsvolumemonitor,
  GnomeVFSVolume *volume, gpointer user_data)
{
  AppUIData *app_ui_data = (AppUIData *)user_data;
  gchar *dst_text = NULL;
  gchar *volume_text = NULL;
  GnomeVFSURI *dst_uri;
  gchar *uri_scheme;

  g_return_if_fail(app_ui_data != NULL);

  dst_text = settings_get_string(SETTINGS_CHESS_SAVE_FILE);
  uri_scheme = gnome_vfs_get_uri_scheme(dst_text);
  if (!uri_scheme)
    dst_text = gnome_vfs_get_uri_from_local_path(dst_text);
  else
    g_free(uri_scheme);
  dst_uri = gnome_vfs_uri_new(dst_text);    
  g_free(dst_text);
  
  dst_text = gnome_vfs_uri_to_string(dst_uri, GNOME_VFS_URI_HIDE_NONE);

  if (dst_text != NULL) {
    /* the mount point of the changed volume */
    volume_text = gnome_vfs_volume_get_activation_uri(volume);
  }
  
  /* check if unmounting this volume has anything to do with us */
  if(volume_text && dst_text && (g_str_has_prefix(dst_text, volume_text))) {
    app_ui_data->is_mmc = (app_ui_data->mmc_uri 
      && g_str_has_prefix(volume_text, app_ui_data->mmc_uri));
  }
  
  if (dst_text) g_free (dst_text);
  if (volume_text) g_free(volume_text);
}
