/*   ScwEntry
 *   Copyright 2005  Kalle Vahlman
 *
 *   This library is free software; you can redistribute it and/or
 *   modify it under the terms of the GNU Lesser General Public
 *   License as published by the Free Software Foundation; either
 *   version 2.1 of the License, or (at your option) any later version.
 *
 *   This library is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 *   Lesser General Public License for more details.
 *
 *   You should have received a copy of the GNU Lesser General Public
 *   License along with this library; if not, write to the Free Software
 *   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307, USA
 *
 */

#include <time.h>

#include <gdk/gdkkeysyms.h>
#include <gtk/gtk.h>

#include "scwentry.h"

#define DEFAULT_HISTORY_SIZE	20
#define MAX_HISTORY_SIZE	200


enum
{
  PROP_0,
  PROP_HISTORY_SIZE
};

enum
{
  FAKE_SIGNAL,
  LAST_SIGNAL
};

/*
static guint scw_entry_signals[LAST_SIGNAL] = { 0 };
*/

typedef struct _ScwEntryPrivate ScwEntryPrivate;

struct _ScwEntryPrivate
{
  gint history_size;
  
  GList *history;
  GList *current;
};

#define SCW_ENTRY_GET_PRIVATE(obj) (G_TYPE_INSTANCE_GET_PRIVATE ((obj), SCW_TYPE_ENTRY, ScwEntryPrivate))

static void scw_entry_class_init  (ScwEntryClass    *klass);
static void scw_entry_init        (ScwEntry         *entry);
static void scw_entry_finalize    (GObject          *object);

void
scw_entry_set_property (GObject      *object,
                              guint         prop_id,
                              const GValue *value,
                              GParamSpec   *pspec);
void
scw_entry_get_property (GObject      *object,
                              guint         prop_id,
                              GValue *value,
                              GParamSpec   *pspec);

void scw_entry_activate    (ScwEntry  *entry,
                            gpointer  user_data);
gboolean    scw_entry_key_pressed (GtkWidget *widget,
                                   GdkEventKey *event,
                                   gpointer user_data);
/*** Callbacks */

/*** Utility */

static GtkEntryClass *parent_class = NULL;


GType
scw_entry_get_type (void)
{
  static GType entry_type = 0;

  if (!entry_type)
    {
      static const GTypeInfo scwentry_info =
      {
	sizeof (ScwEntryClass),
	NULL,		/* base_init */
	NULL,		/* base_finalize */
	(GClassInitFunc) scw_entry_class_init,
	NULL,		/* class_finalize */
	NULL,		/* class_data */
	sizeof (ScwEntry),
	0,		/* n_preallocs */
	(GInstanceInitFunc) scw_entry_init,
	NULL,		/* value_table */
      };

      entry_type = g_type_register_static (GTK_TYPE_ENTRY, "ScwEntry", 
					 &scwentry_info, 0);
    }

  return entry_type;
}

static void
scw_entry_class_init (ScwEntryClass *scw_entry_class)
{
  GObjectClass *gobject_class = G_OBJECT_CLASS (scw_entry_class);

  parent_class = g_type_class_peek_parent (scw_entry_class);

  gobject_class->finalize = scw_entry_finalize;
  gobject_class->set_property = scw_entry_set_property;
  gobject_class->get_property = scw_entry_get_property;

  /* Properties */

  g_object_class_install_property (gobject_class,
    PROP_HISTORY_SIZE,
    g_param_spec_int ("history-size",
                      "History Size",
                      "Number of history items for this entry",
                      0,
                      MAX_HISTORY_SIZE,
                      DEFAULT_HISTORY_SIZE,
                      G_PARAM_READWRITE));

  g_type_class_add_private (gobject_class, sizeof (ScwEntryPrivate));
}

static void
scw_entry_init (ScwEntry *entry)
{
  ScwEntryPrivate *priv;
  
  priv = SCW_ENTRY_GET_PRIVATE (entry);

  priv->history_size = DEFAULT_HISTORY_SIZE;
  
  priv->history = NULL;
  priv->current = NULL;

  g_signal_connect (G_OBJECT (entry), "key-press-event",
                    G_CALLBACK (scw_entry_key_pressed), NULL);  
  g_signal_connect (G_OBJECT (entry), "activate",
                    G_CALLBACK (scw_entry_activate), NULL);  

}

static void
scw_entry_finalize (GObject *object)
{
  ScwEntry *entry = SCW_ENTRY (object);
  ScwEntryPrivate *priv = SCW_ENTRY_GET_PRIVATE (entry);

  /* free all history items */
  g_list_free(priv->history);

  priv->history = NULL;
  priv->current = NULL;
}

void
scw_entry_set_property (GObject      *object,
                              guint         prop_id,
                              const GValue *value,
                              GParamSpec   *pspec)
{
  ScwEntry *entry = SCW_ENTRY (object);
/*
  ScwEntryPrivate *priv = SCW_ENTRY_GET_PRIVATE (entry);
*/  
  switch (prop_id)
  {
    case PROP_HISTORY_SIZE:
      scw_entry_set_history_size (entry, g_value_get_int (value));
      break;
    default:
      G_OBJECT_WARN_INVALID_PROPERTY_ID (object, prop_id, pspec);
      break;
  }    
  
}

void
scw_entry_get_property (GObject      *object,
                              guint         prop_id,
                              GValue *value,
                              GParamSpec   *pspec)
{
  ScwEntry *entry = SCW_ENTRY (object);
  ScwEntryPrivate *priv = SCW_ENTRY_GET_PRIVATE (entry);
  
  switch (prop_id)
  {
    case PROP_HISTORY_SIZE:
      g_value_set_int (value, priv->history_size);
      break;
    default:
      G_OBJECT_WARN_INVALID_PROPERTY_ID (object, prop_id, pspec);
      break;
  }    
  
}

/**
 * scw_entry_set_history_size:
 * @entry:  The #ScwEntry.
 * @size :  Desired maximum size for the history.
 *
 * Sets the maximum history size for the ScwEntry. The list will be truncated
 * if longer than specified length.
 *
 */
void
scw_entry_set_history_size (ScwEntry	*entry,
                            gint      size)
{
  ScwEntryPrivate *priv;
  
  g_return_if_fail (SCW_IS_ENTRY(entry));

  priv = SCW_ENTRY_GET_PRIVATE (entry);

  size = CLAMP (size, 0, MAX_HISTORY_SIZE);

  while (g_list_length(priv->history) > size)
    {
      priv->history = g_list_delete_link(priv->history,
                                         g_list_last(priv->history));
    }

  priv->history_size = size;
}

/**
 * scw_entry_get_history_size:
 * @entry:     The #ScwEntry.
 *
 * Gets the size of the history list for the entry.
 *
 * Returns: The size of the history list.
 *
 */
gint
scw_entry_get_history_size (ScwEntry *entry)
{
  ScwEntryPrivate *priv;
  
  g_return_val_if_fail (SCW_IS_ENTRY (entry), 0);

  priv = SCW_ENTRY_GET_PRIVATE (entry);

  return priv->history_size;
}

/**
 * scw_entry_get_history:
 * @entry:     The #ScwEntry.
 *
 * Gets the actual history list for the entry.
 * NOTE: The list should not be modified in any way!
 *
 * Returns: The history list for the widget.
 *
 */
GList *
scw_entry_get_history (ScwEntry *entry)
{
  ScwEntryPrivate *priv;
  
  g_return_val_if_fail (SCW_IS_ENTRY (entry), NULL);

  priv = SCW_ENTRY_GET_PRIVATE (entry);
  
  return priv->history;
}

/**
 * scw_entry_history_get_current:
 * @entry:     The #ScwEntry.
 *
 * Gets the current item in the history list for the entry.
 * NOTE: The list should not be modified in any way!
 *
 * Returns: The current item in the history list for the widget.
 *
 */
GList *
scw_entry_history_get_current (ScwEntry *entry)
{
  ScwEntryPrivate *priv;
  
  g_return_val_if_fail (SCW_IS_ENTRY (entry), NULL);

  priv = SCW_ENTRY_GET_PRIVATE (entry);
  
  return priv->current;
}

/**
 * scw_entry_history_append:
 * @entry:  The #ScwEntry.
 * @item :  Item to append to the history.
 *
 * Appends the specified item to the history list. Identical succesive items
 * will be "collapsed" to one item.
 *
 */
void
scw_entry_history_append (ScwEntry *entry,
                          const gchar    *item)
{
  ScwEntryPrivate *priv;
  
  g_return_if_fail (SCW_IS_ENTRY (entry));
  g_return_if_fail (item != NULL);

  priv = SCW_ENTRY_GET_PRIVATE (entry);

  priv->current = NULL;

  if (priv->history != NULL
      && g_utf8_collate (priv->history->data, item) == 0)
    {
      return;
    }
  
  priv->history = g_list_prepend (priv->history, g_strdup (item));
  
  while (g_list_length (priv->history) > priv->history_size)
    {
      priv->history = g_list_delete_link (priv->history,
                                          g_list_last (priv->history));
    }
}

/**
 * scw_entry_history_step_back:
 * @entry:  The #ScwEntry.
 *
 * Steps one item back in the history and replaces current text in the entry
 * with the text from the item in history. If at the end of history, does
 * nothing.
 *
 */
void
scw_entry_history_step_back (ScwEntry *entry)
{
  GList *prev;
  ScwEntryPrivate *priv;
  
  g_return_if_fail (SCW_IS_ENTRY (entry));

  priv = SCW_ENTRY_GET_PRIVATE (entry);

  if (priv->history == NULL)
    {
      return;
    }

  /* If there's no current, start with the first one */
  if (priv->current == NULL)
    {
      prev = priv->history;
    }
  else
    {
      prev = g_list_next (priv->current);
    }
  
  if (prev != NULL)
    {
      if (prev->data != NULL)
        {
          gtk_entry_set_text (GTK_ENTRY (entry), prev->data);
        }
      priv->current = prev;
    }
  else
    {
      /* FIXME: Visible notification */
    }
}


/**
 * scw_entry_history_step_forward:
 * @entry:  The #ScwEntry.
 *
 * Steps one item forward in the history and replaces current text in the entry
 * with the text from the item in history. If no history item is active, appends
 * the current text to the history
 *
 */
void
scw_entry_history_step_forward (ScwEntry *entry)
{
  GList *prev;
  ScwEntryPrivate *priv;
  
  g_return_if_fail (SCW_IS_ENTRY (entry));

  priv = SCW_ENTRY_GET_PRIVATE (entry);

  if (priv->history == NULL)
    {
      return;
    }

  /* If there's no current, do nothing
   * We don't support wrapping the history around
   */
  if (priv->current == NULL)
    {
      return;
    }

  prev = g_list_previous (priv->current);
  
  if (prev != NULL)
    {
      if (prev->data != NULL)
        {
          gtk_entry_set_text (GTK_ENTRY (entry), prev->data);
        }
      priv->current = prev;
    }
  /* As a special case, if there was no previous item, reset
   * both the current item and the entry
   */
  else
    {
      gtk_entry_set_text (GTK_ENTRY (entry), "");
      priv->current = NULL;
    }
}

/**
 * scw_entry_history_peek_backwards:
 * @entry:     The #ScwEntry.
 *
 * Peeks what is the previous item in the history list.
 * The returned string should be freed after use.
 *
 * Returns: The previous history item, or NULL if not available.
 *
 */
const gchar *
scw_entry_history_peek_backwards (ScwEntry *entry)
{
  GList *next;
  ScwEntryPrivate *priv;
  
  g_return_val_if_fail (SCW_IS_ENTRY (entry), NULL);

  priv = SCW_ENTRY_GET_PRIVATE (entry);

  if (priv->history == NULL)
    {
      return NULL;
    }

  /* If there's no current, return the first item */
  if (priv->current == NULL)
    {
      return priv->history->data;
    }

  next = g_list_next (priv->current);
  
  if (next != NULL)
    {
      if (next->data != NULL)
        {
          /* Note that this is not a copy */
          return next->data;
        }
    }

  return NULL;  
  
}

/**
 * scw_entry_history_peek_forward:
 * @entry:     The #ScwEntry.
 *
 * Peeks what is the next item in the history list.
 * The returned string should be freed after use.
 *
 * Returns: The next history item, or NULL if not available.
 *
 */
const gchar *
scw_entry_history_peek_forward (ScwEntry *entry)
{
  GList *prev;
  ScwEntryPrivate *priv;
  
  g_return_val_if_fail (SCW_IS_ENTRY (entry), NULL);

  priv = SCW_ENTRY_GET_PRIVATE (entry);

  if (priv->history == NULL)
    {
      return NULL;
    }

  /* If there's no current, return NULL
   * We don't support wrapping the history around
   */
  if (priv->current == NULL)
    {
      return NULL;
    }

  prev = g_list_previous (priv->current);
  
  if (prev != NULL)
    {
      if (prev->data != NULL)
        {
          /* Note that this is not a copy */
          return prev->data;
        }
    }

  return NULL;  
}

gboolean    scw_entry_key_pressed (GtkWidget *widget,
                                   GdkEventKey *event,
                                   gpointer user_data)
{
  ScwEntryPrivate *priv;
  
  g_return_val_if_fail (SCW_IS_ENTRY (widget), FALSE);

  priv = SCW_ENTRY_GET_PRIVATE (widget);

  if (priv->history == NULL)
    {
      return FALSE;
    }

  switch (event->keyval)
    {
      case GDK_Up:
        scw_entry_history_step_back (SCW_ENTRY (widget));
        break;
        
      case GDK_Down:
        scw_entry_history_step_forward (SCW_ENTRY (widget));
        break;
        
      default:
        return FALSE;
    }

#if 0
  g_print ("scw_entry_key_press(): ");
  
  for (i = priv->history; i; i = i->next)
    {
      if (i == priv->current)
        {
          g_print ("_%s_ -> ", (gchar*)i->data);
        }
      else
        {
          g_print ("%s -> ", (gchar*)i->data);
        }
    }
  
  g_print ("\n");
#endif

  return TRUE;
}


void scw_entry_activate    (ScwEntry  *entry,
                            gpointer  user_data)
{
  ScwEntryPrivate *priv;
  
  g_return_if_fail (SCW_IS_ENTRY (entry));

  priv = SCW_ENTRY_GET_PRIVATE (entry);
    
  if (g_utf8_strlen (gtk_entry_get_text (GTK_ENTRY (entry)), -1) > 0)
    {
      scw_entry_history_append (entry, gtk_entry_get_text (GTK_ENTRY (entry)));
    }
}

