/*
GPS daemon manager API. The API is used by those applications that
wish to use services provided by gps daemon i.e., they wish to receive
GPS data from the daemon. See README file for more details.

Copyright (C) 2006 Nokia Corporation. All rights reserved.

Author: Jukka Rissanen <jukka.rissanen@nokia.com>

Redistribution and use in source and binary forms, with or without modification, are permitted provided that the following conditions are met:

Redistributions of source code must retain the above copyright notice, this list of conditions and the following disclaimer.
Redistributions in binary form must reproduce the above copyright notice, this list of conditions and the following disclaimer in the documentation and/or other materials provided with the distribution.
The name of the author may not be used to endorse or promote products derived from this software without specific prior written permission.

THIS SOFTWARE IS PROVIDED BY THE AUTHOR ``AS IS'' AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.

*/

/* $Id:$ */

#include <stdio.h>
#include <errno.h>
#include <sys/stat.h>
#include <unistd.h>
#include <fcntl.h>
#include <string.h>
#include <sys/types.h>
#include <signal.h>
#include <sys/stat.h>
#include <sys/wait.h>
#include <sys/time.h>
#include <time.h>
#include <sys/select.h>

#include "gpsmgr.h"

#define DEBUG

/* max. number of applications using this API */
static int max_applications = GPSMGR_MAX_APPLICATIONS;


static int unlock_file(gpsmgr_t *ctx, int do_unlock_slot);
static int read_pid(char *file, pid_t *pid);
static int gpsmgr_gpsd_started(int *fd_param);

/* ----------------------------------------------------------------------- */
static int debug_mode = 0;

#ifdef DEBUG
#define PDEBUG(fmt...) do {						\
    if (debug_mode) {							\
      struct timeval tv;						\
      gettimeofday(&tv, 0);						\
      printf("DEBUG[%d]:%ld.%ld:%s:%s():%d: ",				\
	     getpid(),							\
	     tv.tv_sec, tv.tv_usec,					\
	     __FILE__, __FUNCTION__, __LINE__);				\
      printf(fmt);							\
      fflush(stdout);							\
    }									\
  }while(0);
#else
#define PDEBUG(fmt...)
#endif

/* ----------------------------------------------------------------------- */
int gpsmgr_set_debug_mode(int mode)
{
#ifdef DEBUG
  int old_mode = debug_mode;
  debug_mode = mode;
  return old_mode;
#else
  return -1;
#endif
}


/* ----------------------------------------------------------------------- */
static int lock(int fd, int cmd, int type, off_t offset, int whence,
		off_t len)
{
  struct flock fl = {0};

  fl.l_type = type;
  fl.l_start = offset;
  fl.l_whence = whence;
  fl.l_len = len;

  return (fcntl(fd, cmd, &fl));
}

/* ----------------------------------------------------------------------- */
static int lock_master(int fd)
{
  int st = lock(fd, F_SETLK, F_WRLCK, 0, SEEK_SET, 1);
  PDEBUG("master lock %s (fd=%d)\n", st<0 ? "failed" : "succeed", fd);
  return st;
}

static int unlock_master(int fd)
{
  int st = lock(fd, F_SETLK, F_UNLCK, 0, SEEK_SET, 1);
  PDEBUG("master unlock %s (fd=%d)\n", st<0 ? "failed" : "succeed", fd);
  return st;
}

/* ----------------------------------------------------------------------- */
static int lock_check(int fd, int slot)
{
  int ret;
  struct flock fl;

  memset(&fl, 0, sizeof(fl));

  fl.l_type = F_WRLCK;
  fl.l_start = slot;
  fl.l_whence = SEEK_SET;
  fl.l_len = 1;
  
  ret = fcntl(fd, F_GETLK, &fl);

  if (fl.l_type == F_UNLCK) {
    ret = 0;  /* is not locked */
  } else {
    if (fl.l_pid != 0)
      ret = 1; /* is locked */
    else
      ret = 0; /* pid 0 cannot lock anything */
  }

#if 0
  printf("1) slot %d%slocked (pid = %d)\n", slot, ret==0 ? " not ": " ", fl.l_pid);
#endif

  return ret;
}

/* ----------------------------------------------------------------------- */
static int lock_slot(int fd, int slot)
{
  return lock(fd, F_SETLK, F_WRLCK, slot, SEEK_SET, 1);
}

static int unlock_slot(int fd, int slot)
{
  return lock(fd, F_SETLK, F_UNLCK, slot, SEEK_SET, 1);
}


/* ----------------------------------------------------------------------- */
/* Lock file for locking slot access */
static int lock2(int *fd)
{
  int fd2, ret;
  char *file = GPSMGR_LOCK2;

  fd2 = open(file, O_RDWR | O_CREAT, S_IRWXU | S_IRWXG | S_IRWXO);
  if (fd2 < 0) {
    PDEBUG("Cannot open %s [%s/%d]\n", file, strerror(errno), errno);
    return -1;
  }

  /* When the lock is got, then we know the slot access can be done */
  PDEBUG("Waiting for lock...\n");
  ret = lock(fd2, F_SETLKW, F_WRLCK, 0, SEEK_SET, 0);
  if (fd)
    *fd = fd2;
  return ret;
}


static int unlock2(int fd)
{
  int ret;

  PDEBUG("Releasing the lock...\n");
  ret = lock(fd, F_SETLKW, F_UNLCK, 0, SEEK_SET, 0);
  close(fd);
  return ret;
}


/* ----------------------------------------------------------------------- */
/* Get an advisory lock.
 * Returns:
 *   <0 Lock file cannot be opened or lock cannot be get, see errno
 *    0 Lock could not be acquired
 *    1 Lock acquired
 */
static int lock_file_slot(gpsmgr_t *ctx)
{
  int ret, i, st, slot = 0;
  int fd = ctx->lock_file_desc;

#if 0
  int fd2 = -1;
#endif

  if (ctx->locking_slot) {
    /* We already have a locking slot */
    return 1;
  }

#if 0
  PDEBUG("Locking slot access\n");
  ret = lock2(&fd2);
  if (ret<0) {
    PDEBUG("Cannot lock slot access [%s/%d]\n", strerror(errno), errno);
    return -1;
  }
#endif

  /* Ok, find available slot in the file. Note that we do not use slot 0
   * because that is used as a master lock.
   */
  for (i=1; i<max_applications; i++) {

    errno = 0;

    ret = lock_slot(fd, i);
    PDEBUG("Try to lock slot %d, ret=%d [%s/%d]\n", i, ret, strerror(errno), errno);
    if (ret<0) {
      if ((errno != EACCES) && (errno != EAGAIN)) {
	PDEBUG("Cannot lock slot %d in %s [%s/%d]\n", i, ctx->file, strerror(errno), errno);
      }
      continue; /* get next slot */
    }

    if (!ret) {
      /* Ok, we got the lock for slot */
      PDEBUG("Slot %d locked in %s (fd=%d)\n", i, ctx->file, fd);
      slot = i;
      break;
    }
  }

  if (slot) {
    ctx->locking_slot = slot;
    st = 1;
  } else {
    /* slot could not be found */
    ctx->locking_slot = 0;
    st = 0;
  }

#if 0
  ret = unlock2(fd2);
  PDEBUG("Unlocking slot access\n");
  if (ret<0) {
    PDEBUG("Cannot unlock slot access [%s/%d]\n", strerror(errno), errno);
    return -1;
  }
#endif

  return st;
}


/* ----------------------------------------------------------------------- */
/* Release lock.
 * Returns:
 *   <0 Lock file cannot be opened or lock cannot be get, see errno
 *    0 Lock released
 */
static int unlock_file_slot(gpsmgr_t *ctx, int all)
{
  int ret, st = 1, id;
  int fd = ctx->lock_file_desc;
  int i = max_applications + 1;

#if 0
  int fd2 = -1;
#endif

  if (!ctx->locking_slot) {
    /* we didn't had locking slot for us */
    errno = EINVAL;
    PDEBUG("All slots locked already in fd %d [%s/%d]\n", fd, strerror(errno),errno);
    return -1;
  }

#if 0
  PDEBUG("Locking slot access\n");
  ret = lock2(&fd2);
  if (ret<0) {
    PDEBUG("Cannot lock slot access [%s/%d]\n", strerror(errno), errno);
    return -1;
  }
#endif

  id  = ctx->locking_slot;

  do {
    PDEBUG("Unlocking slot %d\n", id);
    ret = unlock_slot(fd, id);
    i--;
    id  = i;
  } while (all && i>0);


  PDEBUG("Slot %d in file fd %d released\n", ctx->locking_slot, fd);
  PDEBUG("ret = %d, errno=%d, %s\n", ret, errno, strerror(errno));

  ctx->locking_slot = 0;

  if (ret<0) {
    st = -1;
  }

#if 0
  ret = unlock2(fd2);
  PDEBUG("Unlocking slot access\n");
  if (ret<0) {
    PDEBUG("Cannot unlock slot access [%s/%d]\n", strerror(errno), errno);
    return -1;
  }
#endif

  return st;
}



/* ----------------------------------------------------------------------- */
/* Check number of locks (==number of applications needing gps data)
 * Returns:
 */
static int check_locks(int fd, int locking_slot, int *num_of_locks, int peek)
{
  int ret, i, free_slots = 0;

  for (i=1; i<max_applications; i++) {

    if (locking_slot == i) /* do not try to check ourselves */
      continue;

    if (peek) {
      /* just get an estimate of number of users */
      ret = lock_check(fd, i);
    } else {
      ret = lock_slot(fd, i);
    }
    if (!ret) {
      /* Ok, we got the lock for slot */
      free_slots++;
    }
  }

#if 0
  PDEBUG("Free slots = %d, max slots = %d\n", free_slots, max_applications);
#endif

  if ((free_slots+1+1) == max_applications) {
    /* We are the only one, now we have also all slots locked */
    if (num_of_locks)
      *num_of_locks = 1;

#if 0
    PDEBUG("Num of locks = %d\n", 1);
#endif

    return 1;
  }

  if (!peek) {
    /* Free the slots that we locked because now we know that there
     * is an application still running.
     */
    for (i=1; i<max_applications; i++) {

      if (locking_slot == i) /* do not try to check ourselves */
	continue;

      ret = lock(fd, F_SETLK, F_UNLCK, i, SEEK_SET, 1);
    }
  }

  if (num_of_locks)
    *num_of_locks = max_applications - 1 - free_slots; /* this process is also counted */

  return 0;
}


/* ----------------------------------------------------------------------- */
/* Get an advisory lock.
 * Returns:
 *   <0 Lock file cannot be opened or lock cannot be get, see errno
 *    0 Lock could not be acquired
 *    1 Lock acquired
 */
static int lock_file(gpsmgr_t *ctx, int do_lock_slot)
{
  int ret;

#ifdef DEBUG
  pid_t pid = ctx->mgr_pid;
#endif

  ret = lock_master(ctx->lock_file_desc);
  PDEBUG("Global slot locked, ret = %d, pid=%d\n", ret, pid);
  if (ret<0) {

    if ((errno == EACCES) || (errno = EAGAIN)) {
      PDEBUG("File %s already locked\n", ctx->file);
      ret = 0;
      goto OUT;
    }

    PDEBUG("File %s could not be locked [%s/%d]\n", ctx->file, strerror(errno), errno);
    return -1;
  }

  ret = 1;

  PDEBUG("File %s locked (fd=%d)\n", ctx->file, ctx->lock_file_desc);

 OUT:
  if (do_lock_slot) {
    /* increment the reference count i.e., mark us as a gpsd user */
    PDEBUG("Trying to get slot for us\n");
    ret = lock_file_slot(ctx);
    PDEBUG("Locking slot = %d\n", ctx->locking_slot);
    if (ret == 0) {
      PDEBUG("All slots occupied, cannot continue.\n");
      unlock_file(ctx, 0);
      return -1;
    } else if (ret < 0) {
      PDEBUG("Error while getting the slot lock [%s/%d]\n", strerror(errno), errno);
      unlock_file(ctx, 0);
      return -1;
    }
  }

  return ret;
}


/* ----------------------------------------------------------------------- */
/* Release lock.
 * Returns:
 *   <0 Lock file cannot be opened or lock cannot be get, see errno
 *    0 Lock released
 */
static int unlock_file(gpsmgr_t *ctx, int do_unlock_slot)
{
  int ret;

  if (do_unlock_slot) {
    unlock_file_slot(ctx, 0);
  }

  ret = unlock_master(ctx->lock_file_desc);
  if (ret<0) {
    return -1;
  }

  PDEBUG("File unlocked (fd=%d)\n", ctx->lock_file_desc);

  return 0;
}


/* ----------------------------------------------------------------------- */
/* Check master file locking.
 * Returns:
 *   <0 Lock file cannot be opened or lock cannot be get, see errno
 *    0 Lock available (gpsd no running)
 *    1 Lock not available (gpsd running)
 *    2 gpsd was not running but the lock is now acquired
 */
static int check_gpsd_lock(int lock_or_not)
{
  int ret, fd;
  char *file = GPSMGR_GPSD_LOCK;

  fd = open(file, O_RDWR | O_CREAT, S_IRWXU | S_IRWXG | S_IRWXO);
  if (fd < 0) {
    PDEBUG("Cannot open %s [%s/%d]\n", file, strerror(errno), errno);
    return -1;
  }

  ret = lock(fd, F_SETLK, F_WRLCK, 0, SEEK_SET, 0);
  if (ret == 0) {

    /* Ok, we got the lock, just start gpsd */
    if (lock_or_not == GPSMGR_MODE_JUST_CHECK) {
      lock(fd, F_SETLK, F_UNLCK, 0, SEEK_SET, 0);
      ret = 0;
    } else {
      ret = 2; /* locked by us but perhaps not yet running */
    }
  } else {

    if ((errno==EACCES) || (errno==EAGAIN)) {
      /* already locked i.e., process is running */
      ret = 1;
    }
  }

  close(fd);
  return ret;
}



/* ----------------------------------------------------------------------- */
static int exec_bg(char *prog, char *args[])
{
  pid_t pid;
  int ret;

  if ((pid=fork())<0) {
    return -1;
  } else if (pid!=0) {
    return pid;  /* parent returns */
  }

  /* Child continues */
  setsid();
  chdir("/");
  umask(0);

#if 1
  if (debug_mode) {
#define MAX_TMP_BUF 1024
    char buf[MAX_TMP_BUF+1];
    int idx = 0, loc = 0;
    while (args[idx]) {
      loc += snprintf(buf+loc, MAX_TMP_BUF-loc, "%s ", args[idx]);
      idx++;
    }
    if (idx>0) {
      PDEBUG("exec: params=%d, args=%s\n", idx, buf);
    }
    PDEBUG("Exec %s %s\n", prog, buf);
  }
#endif

  /* We create lock here so that gpsd source need not to be touched */
  ret = gpsmgr_gpsd_started(NULL);
  if (ret != 0) {
    PDEBUG("GPS daemon already running.\n");
    exit(-1);
  }

  if (execvp(prog, args)<0) {
    PDEBUG("Cannot exec %s [%s/%d]\n", prog, strerror(errno), errno);
    return -1;
  }

  return 0;  /* never returns */
}


/* ----------------------------------------------------------------------- */
int gpsmgr_start(char *path, char **gps_devices, char *ctrl_sock,
		 int debug_level, short port, gpsmgr_t *ctx)
{
  int ret, fd, fd2, i=0;
  struct stat buf;
  char debug_level_str[32];
  char port_str[6];

  /* The gps device addresses are put to the end of the arg list, there is
   * space for max. 8 devices, if this is a problem, then increase the array
   * or make it dynamic (its your call). The arg list must be null terminated.
   */
  char *args[]={path,                   /* 0   */
		"-n",                   /* 1   */
		"-N",                   /* 2   */
		"-F", ctrl_sock,        /* 3-4 */
		"-S", port_str,         /* 5-6 */
		"-D", debug_level_str,  /* 7-8 */
		0,0,0,0,0,0,0,0,        /* 9-16 (8 slots for rfcomm devices) */
		0};
  int pos = 9; /* from where to move */
  int end = sizeof(args) / sizeof(char *) - 1; /* null at the end! */
  int pos2 = pos;  /* where to put arguments (next index to debug_level_str) */

  if (!ctx) {
    errno = EINVAL;
    return -1;
  }

  /* gpsd requires that either device or control socket is passed as a
     parameter */
  if (!gps_devices && !ctrl_sock) {
    errno = EINVAL;
    return -1;
  }

  ret = lock2(&fd2);
  if (ret<0) {
    PDEBUG("Cannot lock slot access [%s/%d]\n", strerror(errno), errno);
    return -1;
  }

  ctx->mgr_pid = getpid();
  ctx->file = GPSMGR_LOCK;

  if (debug_level) {
    snprintf(debug_level_str, 32, "%d", debug_level);
    gpsmgr_set_debug_mode(debug_level);
  } else {
    memmove(&args[pos-2], &args[pos], (end-pos)*sizeof(char *));
    pos2 -= 2;
    end -= 2;
  }
  pos -= 2;

  if (port) {
    snprintf(port_str, 6, "%u", port);
  } else {
    PDEBUG("port missing, moving args[%d] -> %d (%d bytes)\n", pos, pos-2, (end-pos)*sizeof(char *));
    memmove(&args[pos-2], &args[pos], (end-pos)*sizeof(char *));
    pos2 -= 2;
    end -= 2;
  }
  pos -= 2;

  if (ctrl_sock) {
    args[4] = ctrl_sock;
  } else {
    PDEBUG("ctrl socket missing, moving args[%d] -> %d (%d bytes)\n", pos, pos-2, (end-pos)*sizeof(char *));
    memmove(&args[pos-2], &args[pos], (end-pos)*sizeof(char *));
    pos2 -= 2;
    end -= 2;
  }
  pos -= 2;

  if (gps_devices) {
    while (gps_devices[i] && (i<(end-pos2))) {
      PDEBUG("devices: i=%d, end=%d, pos2=%d, gps_devices[%d]=%s\n", i, end, pos2, i, gps_devices[i]);
      args[pos2++] = gps_devices[i++];
    }
  }

  fd = open(ctx->file, O_RDWR | O_CREAT, S_IRWXU | S_IRWXG | S_IRWXO);
  if (fd<0) {
    PDEBUG("Cannot open %s [%s/%d]\n", ctx->file, strerror(errno), errno);
    return -1;
  }

  /* Fill the file if it is empty (just in case) */
  ret = fstat(fd, &buf);
  if (!ret) {
    if (S_ISREG(buf.st_mode) && (buf.st_size < max_applications)) {
      char *tmp = (char *)calloc(max_applications, 1);
      PDEBUG("Create %s as %d bytes long\n", ctx->file, max_applications);
      if (tmp) {
	write(fd, tmp, max_applications);
	free(tmp);
      }
    }
  }

  lseek(fd, 0, SEEK_SET);  /* start from the beginning */

  ctx->lock_file_desc = fd;

  ret = lock_file(ctx, 1);
  if (ret == 0) {
    PDEBUG("GPS daemon started already, we have slot %d\n", ctx->locking_slot);
    return 0; /* everything ok, someone is already started gpsd because the file
	       * is locked
	       */
  }

  if (ret < 0) {
    PDEBUG("Cannot continue [%s/%d]\n", strerror(errno), errno);
    unlock2(fd2);
    return -1;
  }

  ret = unlock2(fd2);
  if (ret<0) {
    PDEBUG("Cannot unlock slot access [%s/%d]\n", strerror(errno), errno);
  }

  /* Ok, we are the only application running that wishes to start gpsd */
  if (!ctx->already_locked_by_us) {
    ret = check_gpsd_lock(GPSMGR_MODE_LOCK_IF_POSSIBLE);
    if (ret == 1) {
      /* Already running */
      return 0;
    } else if (ret < 0) {
      /* Error */
      return -1;
    }
  } else {
    PDEBUG("Locked by us but gpsd not yet started.\n");
  }

  /* Start the program */
  ret = exec_bg(path, args);
  if (ret<0) {
    unlock_file(ctx, 1);
    return -1;
  }

  ctx->pid = ret;

  return 0;
}

/* ----------------------------------------------------------------------- */
int gpsmgr_stop(gpsmgr_t *ctx)
{
  int ret, fd2, st = 0;

  if (!ctx) {
    errno = EINVAL;
    return -1;
  }

  ret = lock2(&fd2);
  if (ret<0) {
    PDEBUG("Cannot lock slot access [%s/%d]\n", strerror(errno), errno);
    return -1;
  }

  PDEBUG("Checking locks to %s, fd=%d\n", ctx->file, ctx->lock_file_desc);

  /* Get the master lock */
  ret = lock_file(ctx, 0);
  if (ret == 0) {
    /* already locked, just return */
    st = 1;
    goto OUT;
  }

  if (ret < 0) {
    /* error */
    st = -1;
    goto OUT;
  }

  /* We have the lock */

  ret = check_gpsd_lock(GPSMGR_MODE_LOCK_IF_POSSIBLE);
  if (ret == 0) {
    /* Not running */
    PDEBUG("gpsd is not running\n");

  } else if (ret < 0) {
    /* Error */
    PDEBUG("Cannot continue [%s/%d]\n", strerror(errno), errno);
    st = -1;

  } else {

    /* Ok, we have a lock, try to check how meny users there are */
    ret = check_locks(ctx->lock_file_desc, ctx->locking_slot, NULL, 0);
    if (ret == 1) {
      /* we were the only one, stop gpsd now */
      if (ctx->pid>0) {
	/* making sure that we only kill one process */
	int killed = 0;

      KILL:
	PDEBUG("No users for gpsd, killing it (pid=%d)\n", ctx->pid);
	kill(ctx->pid, SIGTERM);

	/* wait child in order to avoid zombie, try not to block */
	{
	  int maxcount = 10*3;  /* three seconds timeout */
	  int pid_st;
	  struct timeval tv;

	  while (maxcount > 0) {
	    /* try not to leave zombies */
	    pid_st = waitpid(ctx->pid, NULL, WNOHANG);
	    if (pid_st>0) {
	      PDEBUG("gpsd killed (pid=%d)\n", pid_st);
	      killed = 1;
	      break; /* waited ok */
	    }
	    maxcount--;

	    PDEBUG("gpsd waited, %d counts left\n", maxcount);

	    /* sleep 100ms */
	    tv.tv_sec = 0;
	    tv.tv_usec = 100*1000;

	    select(ctx->lock_file_desc, NULL, NULL, NULL, &tv);
	  }
	}

	/* If gpsd is still running, just kill it using force */
	if (!killed && !kill(ctx->pid, 0)) {
	   PDEBUG("gpsd (pid=%d) refuses to die, using the force\n", ctx->pid);
	   kill(ctx->pid, SIGSEGV);
	}

	st = 0;
      } else {
	/* pid was not set, read it from file */
	read_pid(GPSMGR_GPSD_LOCK, &ctx->pid);
	if (ctx->pid)
	  goto KILL;

	PDEBUG("Pid is %d, will not kill it!\n", ctx->pid);
	st = -1;
      }
    } else
      st = 2;

  }

  unlock_file(ctx, 1);
  close(ctx->lock_file_desc);
  ctx->already_locked_by_us = 0;

 OUT:
  ret = unlock2(fd2);
  if (ret<0) {
    PDEBUG("Cannot unlock slot access [%s/%d]\n", strerror(errno), errno);
  }

  return st;
}


/* ----------------------------------------------------------------------- */
int gpsmgr_is_gpsd_running(gpsmgr_t *ctx, int *num_of_clients, int mode)
{
  int ret, st = 0;

  ret = check_gpsd_lock(mode);
  if (ret <= 0) {
    st = 0;  /* not running or error */
  } else if (ret == 2) {
    st = 2;
    if (ctx)
      ctx->already_locked_by_us = 1;
  } else {
    /* File already locked, gpsd is running, this is ok */
    st = 1;
    if (ctx)
      ctx->already_locked_by_us = 0;
  }

  if (ctx && (mode == GPSMGR_MODE_JUST_CHECK) && num_of_clients)
    check_locks(ctx->lock_file_desc, ctx->locking_slot, num_of_clients, 1);

  return st;
}


/* ----------------------------------------------------------------------- */
static int read_pid(char *file, pid_t *pid)
{
#define PID_LEN 10
  char pid_buf[PID_LEN+1];
  int fd, st, ret, p;

  fd = open(file, O_RDONLY, S_IRWXU | S_IRWXG | S_IRWXO);
  if (fd<0) {
    PDEBUG("Cannot open %s [%s/%d]\n", file, strerror(errno), errno);
    return -1;
  }

  ret = read(fd, pid_buf, PID_LEN);
  if (ret) {
    p = atoi(pid_buf);
    if (!p) {
      PDEBUG("Pid %s is not a valid number (file %s)\n", pid_buf, file);
      st = -1;
      goto OUT;
    }
    if (pid)
      *pid = p;

    PDEBUG("File %s, pid %d\n", file, p);

  } else {

    PDEBUG("Pid file %s is empty\n", file);
    st = -1;
    goto OUT;

  }

  st = 0;

 OUT:
  close(fd);
  return st;
}


/* ----------------------------------------------------------------------- */
static int write_pid(int fd, pid_t pid)
{
#define PID_LEN 10
  char pid_buf[PID_LEN+1];
  int pid_len;
  int st;

  if (ftruncate(fd, 0) < 0) {
    PDEBUG("File (fd=%d) truncate failed [%s/%d]\n", fd, strerror(errno), errno);
    st = -1;
    goto OUT;
  }

  snprintf(pid_buf, PID_LEN, "%ld\n", (long)pid);
  pid_len = strlen(pid_buf);
  if (write(fd, pid_buf, pid_len) != pid_len) {
    PDEBUG("File (fd=%d) pid %d write failed [%s/%d]\n", fd, pid, strerror(errno), errno);
    st = -1;
    goto OUT;
  }

  st = 0;

 OUT:
  return st;
}

/* ----------------------------------------------------------------------- */
static int gpsmgr_gpsd_started(int *fd_param)
{
  int fd = -1, ret;
  pid_t pid = getpid();
  char *file = GPSMGR_GPSD_LOCK;

  fd = open(file, O_RDWR | O_CREAT, S_IRWXU | S_IRWXG | S_IRWXO);
  if (fd<0) {
    PDEBUG("Cannot open %s [%s/%d]\n", file, strerror(errno), errno);
    return -1;
  }

  ret = lock(fd, F_SETLKW, F_WRLCK, 0, SEEK_SET, 0); /* we must wait for the lock */
  if (ret<0) {

    if ((errno == EACCES) || (errno = EAGAIN)) {
      PDEBUG("File %s already locked\n", file);
      ret = 0;
      goto OUT;
    }

    PDEBUG("File %s could not be locked [%s/%d]\n", file, strerror(errno), errno);
    return -1;
  } else {
    write_pid(fd, pid);
    ret = 0;
  }
  /* Note that this lock file is not closed (because we would loose the lock
   * then. The lock is released when the gpsd exists.
   */

 OUT:
  if (fd_param)
    *fd_param = fd;

  return ret;
}


/* ======================================================================= */
/* Finally module test the API */
#ifdef TEST_GPSMGR

/* 
   ${CC:=gcc} -g -DDEBUG -DTEST_GPSMGR gpsmgr.c
*/

#include <assert.h>

int term_flag = 0;

void terminate(int sig)
{
  PDEBUG("signal %d received\n", sig);
  term_flag = 1;
}


int check_locks_held(void)
{
  pid_t pid;
  struct flock fl;
  char *file;
  int fd;
  int count = 0, i, ret;

  /* Check the held locks */
  pid = getpid();
  file = GPSMGR_LOCK;

  fd = open(file, O_RDWR | O_CREAT, S_IRWXU | S_IRWXG | S_IRWXO);
  if (fd<0) {
    PDEBUG("Cannot open %s [%s/%d]\n", file, strerror(errno), errno);
    return -1;
  }

  lseek(fd, max_applications, SEEK_END); /* extend the length of the file just in case */

  for (i=1; i<max_applications; i++) {
    ret = lock(fd, F_SETLK, F_WRLCK, i, SEEK_SET, 1);
    if (ret<0) {

      if ((errno == EACCES) || (errno = EAGAIN)) {
	PDEBUG("File %s slot %d already locked\n", file, i);
	count++;
      }
    }
  }

  close(fd);

  return count;
}


#define MAX_CLIENTS 10

#define MAX_DIR_LEN 128
static char cwd[MAX_DIR_LEN+1] = {0};
static int cwd_len = 0;

int main(int argc, char **argv)
{
  char *path = argv[0];
  char *dev = "/dev/rfcomm0";
  char *ctrl_sock = "/tmp/control";
  gpsmgr_t ctx = {0};
  pid_t pid = 0;
  int ret, i;
  int len, count = 0, locks, num_clients = MAX_CLIENTS;
  char *devices[]={ "barfoo", "foobar", "rfcomm0", 0 };

  gpsmgr_set_debug_mode(1);

  if (!cwd[0]) {
    char *p = getenv("WORKING_DIR");
    if (!p) {
      getcwd(cwd, MAX_DIR_LEN);
    } else
      strncpy(cwd, p, MAX_DIR_LEN);

    cwd_len = strlen(cwd);
    cwd[cwd_len] = '/';
  }

#if 1
  PDEBUG("--------------\n");
  PDEBUG("argc=%d\n", argc);
  for (i=0; i<argc; i++) {
    PDEBUG("argv[%d]=%s\n", i, argv[i]);
  }
  PDEBUG("--------------\n");
#endif

  if (argc==1 || argc==2) {

    /* simulate application that wishes to start gpsd */
    /* we start several applications here and each app tries to
     * call gpsmgr_start() (only one of them should be able to
     * start gpsd)
     */
    /* This branch is checked first (called from MODULE_TEST) */

    /* Start the program i.e., myself */

    pid_t *clients;

    if (argc==2)
      num_clients = atoi(argv[1]);

    clients = calloc(num_clients, sizeof(pid_t));

    PDEBUG("clients to start = %d\n", num_clients);

    for (i=0; i<num_clients; i++) {
      char id[32+1];
      char *args[]={ path, "-id", id, "foobar", 0};  /* 4 or more parameters are needed here */
      snprintf(id, 32, "%d", i);

      clients[i] = exec_bg(path, args);
      if (!clients[i]) {
	PDEBUG("ERROR client starts\n");
	ret = -1;
	goto QUIT;
      } else {
	PDEBUG("started application %d\n", clients[i]);
	count++;
	sleep(1);
      }
    }

    PDEBUG("let the children (%d) live and then kill them, hah haa...\n", count);
    sleep(2);

    for (i=0; i<num_clients; i++) {
      if (clients[i]>0) {
	PDEBUG("killing %d\n", clients[i]);
	kill(clients[i], SIGTERM);
	sleep(1);
      }
    }

    /* wait children, otherwise the executable file could be removed by MODULE_TEST script */
    i = 0;
    while(1) {
      pid = wait(&ret);
      if (pid<0) {
	if (errno == ECHILD)
	  break;
	perror("wait");
	break;
      } else {
	PDEBUG("%d stopped, %d left\n", pid, num_clients-i-1);
      }
      if (i>=num_clients) {
	PDEBUG("All clients waited.\n");
	break;
      }
      i++;
    }

    PDEBUG("done\n");

    free(clients);

    locks = check_locks_held();
    if (locks) {
      PDEBUG("Killed %d clients but still %d locks held\n", count, locks);
      ret = -1;
      goto OUT;
    }

    ret = 0;

  } else if (argc==4) {

    /* This branch is second (called from first branch) */

    /* this simulates the application that uses gpsmgr API
     * i.e., this program will start gpsd
     */

    pid_t p = getpid();
    char *prog;
    int num_of_clients = 0, cnt = 0;

    prog = argv[0];

#if 0
    for (i=0; i<argc; i++) {
      PDEBUG("%d: [%d]=%s\n", p, i, argv[i]);
    }
#endif

    /* wait signal that parent sends to us */
    signal(SIGTERM, terminate);

    PDEBUG("Application program %s started.\n", prog);

    ret = gpsmgr_start(prog, devices, NULL, 0, 0, &ctx);

#if 0
    if (ctx.pid == 0) {
      /* This is parent so we just wait the children now */
      PDEBUG("waiting...\n");
      wait(&ret);
      goto OUT;
    }
#endif

    if (ret != 0) {
      PDEBUG("App: start ret = %d\n", ret);
      perror("start");
      goto OUT;
    } else {
      PDEBUG("App: started gpsd %d\n", ctx.pid);
    }

    while(!term_flag) {

#if 1
      /* At least one instance must be running */
      if (!gpsmgr_is_gpsd_running(&ctx, &num_of_clients, GPSMGR_MODE_JUST_CHECK)) {
	PDEBUG("App: ERROR, gpsd is not running, clients = %d\n", num_of_clients);
	ret = -1;
	goto OUT;
      }
      if (!(cnt % 120)) {
	PDEBUG("App: Num of clients running = %d\n", num_of_clients);
      }
      cnt++;
#endif

      usleep(50);
    }
    PDEBUG("App: stop manager\n");

    ret = gpsmgr_stop(&ctx);
    if (ret != 0) {
      PDEBUG("App: stop ret = %d\n", ret);
      perror("stop");
    } else {
      PDEBUG("App: stopped %d\n", ctx.pid);
    }

    PDEBUG("Application program %s stopped.\n", argv[0]);

  } else {

    int fd = -1;
    int cnt = 0;

    /* This branch is third (called from second branch) */

    /* recursive call, now we simulate gpsd */
#if 0
    ret = gpsmgr_gpsd_started(&fd);
    if (ret != 0) {
      PDEBUG("GPS daemon already running.\n");
      printf("ERROR\n");
      ret = -1;
      goto QUIT;
    }
#endif

    PDEBUG("gpsd started, pid=%d\n", getpid());

    /* wait signal that parent sends to us */
    signal(SIGTERM, terminate);

    while(!term_flag) {
      if (!(cnt % 10)) {
	PDEBUG("gpsd running\n");
      }

      //usleep(5000);
      sleep(1);
      cnt++;
    }
    PDEBUG("stop gpsd\n");

  QUIT:
    PDEBUG("gpsd program stopped\n");
    close(fd);
    exit(0);
    
  }


 OUT:
  PDEBUG("Program stopped.\n");
  return ret;
}

#endif

