/*
 * test2.c - Farsight tests
 *
 * Farsight Voice+Video library test suite
 *  Copyright 2005,2006 Collabora Ltd.
 *  Copyright 2005,2006 Nokia Corporation
 *   @author: Rob Taylor <rob.taylor@collabora.co.uk>.
 *   @author: Philippe Khalaf <philippe.khalaf@collabora.co.uk>.
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

/* this test is to be run on a 770 */

#include <glib.h>
#include <gst/gst.h>
#include <farsight/farsight-session.h>
#include <farsight/farsight-stream.h>
#include <farsight/farsight-transport.h>

GMainLoop *mainloop = NULL;

static void
stream_error (FarsightStream *stream,
       FarsightStreamError error,
       const gchar *debug)
{
    g_print ("%s: stream error: stream=%p error=%s\n", __FUNCTION__, stream, debug);
}

static void
session_error (FarsightSession *stream,
       FarsightSessionError error,
       const gchar *debug)
{
    g_print ("%s: session error: session=%p error=%s\n", __FUNCTION__, stream, debug);
}


static void
new_active_candidate_pair (FarsightStream *stream, gchar* native_candidate, gchar *remote_candidate)
{
    g_print ("%s: new-native-candidate-pair: stream=%p\n", __FUNCTION__, stream);
}

static void
codec_changed (FarsightStream *stream, gint codec_id)
{
    g_print ("%s: codec-changed: codec_id=%d, stream=%p\n", __FUNCTION__, codec_id, stream);
}

static void
native_candidates_prepared (FarsightStream *stream)
{
    const GList *transport_candidates, *lp;
    FarsightTransportInfo *info;

    g_print ("%s: preparation-complete: stream=%p\n", __FUNCTION__, stream);

    transport_candidates = farsight_stream_get_native_candidate_list (stream); 
    for (lp = transport_candidates; lp; lp = g_list_next (lp)) 
    {
      info = (FarsightTransportInfo*)lp->data;
      g_message ("Local transport candidate: %s %d %s %s %s:%d, pref %f", 
          info->candidate_id, info->component, (info->proto == FARSIGHT_NETWORK_PROTOCOL_TCP)?"TCP":"UDP",
          info->proto_subtype, info->ip, info->port, (double) info->preference);
    }
    //g_main_loop_quit(mainloop);
}

static void 
state_changed (FarsightStream *stream, 
               FarsightStreamState state,
               FarsightStreamDirection dir)
{
    switch (state) {
      case FARSIGHT_STREAM_STATE_CONNECTING:
        g_message ("%s: %p connecting\n", __FUNCTION__, stream);
        break;
      case FARSIGHT_STREAM_STATE_CONNECTED:
        g_message ("%s: %p connected\n", __FUNCTION__, stream);
        break;
      case FARSIGHT_STREAM_STATE_DISCONNECTED:
            g_message ("%s: %p disconnected\n", __FUNCTION__, stream);
            break;
    }
}

FarsightSession *setup_rtp_session();
FarsightStream *setup_rtp_stream(FarsightSession *session);

 gboolean
create_recv_pipeline (gpointer user_data)
{
  FarsightStream *stream = user_data;

  g_message ("Preloading coeed 18 return %d",
      farsight_stream_preload_receive_pipeline (stream, 18));


  return FALSE;
}



int main(int argc, char **argv)
{
    g_type_init();
    gst_init (&argc, &argv);

    if (argc != 3)
    {
        g_print("usage : test remoteip remoteport\n");
        return -1;
    }

    mainloop = g_main_loop_new (NULL, FALSE);

    FarsightSession *session;
    session = setup_rtp_session();

    FarsightStream *stream1;

    GList *candidate_glist = NULL;
    FarsightTransportInfo *trans = NULL;

    stream1 = setup_rtp_stream(session);

        /* let's create our candidate from ip:port given on command line */
    trans = g_new0 (FarsightTransportInfo,1);
    trans->candidate_id = g_strdup_printf ("L1");
    trans->component = 1;
    trans->ip = g_strdup (argv[1]);
    trans->port = atoi(argv[2]);
    trans->proto = FARSIGHT_NETWORK_PROTOCOL_UDP;
    trans->proto_subtype = "RTP";
    trans->proto_profile = "AVP";
    trans->preference = 1.0;
    trans->type = FARSIGHT_CANDIDATE_TYPE_LOCAL;

    candidate_glist = g_list_append(candidate_glist, trans);

    farsight_stream_set_remote_candidate_list(stream1, candidate_glist);


    /* this assumes both hosts have the same codecs */
    farsight_stream_set_remote_codecs(stream1,
        farsight_stream_get_local_codecs(stream1));

    g_free(trans);
    g_list_free(candidate_glist);

    farsight_stream_set_active_candidate_pair(stream1, "L1", "L1");

    g_timeout_add (2000, create_recv_pipeline, stream1);


    farsight_stream_start(stream1);

    g_print ("STARTING\n");

    g_main_loop_run(mainloop); 
    return 0;
}

FarsightSession *setup_rtp_session()
{
    FarsightSession *session;

    session = farsight_session_factory_make ("rtp");

    if (!session) {
      g_error("RTP plugin not found");
      exit(1);
    }
    g_print ("protocol details:\n name: %s\n description: %s\n author: %s\n",
             farsight_plugin_get_name (session->plugin),
             farsight_plugin_get_description (session->plugin),
             farsight_plugin_get_author (session->plugin));
    g_signal_connect (G_OBJECT (session), "error", 
                      G_CALLBACK (session_error), NULL);

 
    return session;
}

FarsightStream *setup_rtp_stream(FarsightSession *session)
{
    FarsightStream *stream;
    const GList *possible_codecs, *lp;
    FarsightCodec *codec;

    stream = farsight_session_create_stream (session,
            FARSIGHT_MEDIA_TYPE_AUDIO, FARSIGHT_STREAM_DIRECTION_BOTH);
    g_object_set (G_OBJECT (stream), "transmitter", "rawudp", NULL);
    g_signal_connect (G_OBJECT (stream), "error", 
                      G_CALLBACK (stream_error), NULL);
    g_signal_connect (G_OBJECT (stream), "new-active-candidate-pair", 
                      G_CALLBACK (new_active_candidate_pair), NULL);
    g_signal_connect (G_OBJECT (stream), "codec-changed", 
                      G_CALLBACK (codec_changed), NULL);
    g_signal_connect (G_OBJECT (stream), "native-candidates-prepared", 
                      G_CALLBACK (native_candidates_prepared), NULL);
    g_signal_connect (G_OBJECT (stream), "state-changed", 
                      G_CALLBACK (state_changed), NULL);

    possible_codecs = farsight_stream_get_local_codecs (stream);

    for (lp = possible_codecs; lp; lp = g_list_next (lp)) 
    {
      codec = (FarsightCodec*) lp->data;
      g_message ("codec: %d: %s/%d found", codec->id, codec->encoding_name, 
                                       codec->clock_rate);
    }
    farsight_stream_prepare_transports(stream);

    return stream;
}
