/******************************************************************
*
*	CyberNet for C
*
*	Copyright (C) Satoshi Konno 2005
*
*       Copyright (C) 2006 Nokia Corporation. All rights reserved.
*
*       This is licensed under BSD-style license with patent exclusion,
*       see file COPYING.
*
*	File: cinterface_function.cpp
*
*	Revision:
*
*	02/09/05
*		- first revision
*
******************************************************************/

#include <cybergarage/net/cinterface.h>
#include <cybergarage/net/csocket.h>
#include <cybergarage/util/clog.h>
#include <cybergarage/upnp/ssdp/cssdp.h>

#ifdef HAVE_CONFIG_H
#  include "config.h"
#endif

#include <stdio.h>

#if (defined(WIN32) || defined(__CYGWIN__)) && !defined(ITRON)
	#include <Iptypes.h>
	#include <Iphlpapi.h>
#elif defined(BTRON) || (defined(TENGINE) && !defined(CG_TENGINE_NET_KASAGO))
	#include <net/sock_com.h>
	#include <btron/bsocket.h>
#elif defined(ITRON)
	#include <kernel.h>
#elif defined(TENGINE) && defined(CG_TENGINE_NET_KASAGO)
	#include <tk/tkernel.h>
	#include <btron/kasago.h>
	#include <sys/svc/ifkasago.h>
#else
	#if defined(HAVE_IFADDRS_H)
		#include <ifaddrs.h>
	#else
		#include <sys/ioctl.h>
	#endif
	#include <netdb.h>
	#include <net/if.h>
	#include <sys/socket.h>
	#include <netinet/in.h>
        #include <arpa/inet.h>
#endif

#if defined(TENGINE) && defined(CG_TENGINE_NET_KASAGO)
extern ttUserInterface kaInterfaceHandle;
#endif

#if defined(ITRON)
char InterfaceAddress[CG_NET_IPV6_ADDRSTRING_MAXSIZE];
BOOL IsInterfaceAddressInitialized = FALSE;
#endif

/****************************************
* cg_net_gethostinterfaces (WIN32)
****************************************/

#if (defined(WIN32) || defined(__CYGWIN__)) && !defined(ITRON)

int cg_net_gethostinterfaces(CgNetworkInterfaceList *netIfList)
{
	cg_log_debug_l4("Entering...\n");

#if !defined(CG_USE_WIN32_GETHOSTADDRESSES)
	CgNetworkInterface *netIf;
	SOCKET sd;
	int nNumInterfaces;
	INTERFACE_INFO InterfaceList[20];
	unsigned long nBytesReturned;
	struct sockaddr_in *pAddress;
	char *host;
	u_long nFlags;
	int i;

	cg_socket_startup();
	cg_net_interfacelist_clear(netIfList);

	sd = WSASocket(AF_INET, SOCK_DGRAM, 0, 0, 0, 0);
	if (sd == SOCKET_ERROR)
		return 0;

	if (WSAIoctl(sd, SIO_GET_INTERFACE_LIST, 0, 0, &InterfaceList, sizeof(InterfaceList), &nBytesReturned, 0, 0) == SOCKET_ERROR)
		return 0;

	nNumInterfaces = nBytesReturned / sizeof(INTERFACE_INFO);
    for (i = 0; i < nNumInterfaces; ++i) {
		pAddress = (struct sockaddr_in *) & (InterfaceList[i].iiAddress);
		host = inet_ntoa(pAddress->sin_addr);
		nFlags = InterfaceList[i].iiFlags;
		//if (CyberNet::HostInterface::USE_ONLY_IPV4_ADDR == false) {
			if (nFlags & IFF_LOOPBACK)
				continue;
		//}
		if (!(nFlags & IFF_UP))
			continue;
		//if (IsUseAddress(host) == false)
		//	continue;
		netIf = cg_net_interface_new();
		cg_net_interface_setaddress(netIf, host);
		cg_net_interfacelist_add(netIfList, netIf);
	}

#else
	IP_ADAPTER_ADDRESSES *pAdapterAddresses, *ai;
	DWORD ifFlags;
	ULONG outBufLen;
	IP_ADAPTER_UNICAST_ADDRESS *uai;
	SOCKET_ADDRESS sockaddr;
	SOCKADDR *saddr;
	INT saddrlen;
	char addr[NI_MAXHOST];
	char port[NI_MAXSERV];
	int namInfoRet;
	int ifIdx;
	CgNetworkInterface *netIf;

	cg_socket_startup();
	cg_net_interfacelist_clear(netIfList);

	outBufLen = 0;
	ifFlags = 
		GAA_FLAG_SKIP_ANYCAST | 
		GAA_FLAG_SKIP_FRIENDLY_NAME | 
		GAA_FLAG_SKIP_MULTICAST | 
		GAA_FLAG_SKIP_DNS_SERVER;

	GetAdaptersAddresses(AF_UNSPEC, ifFlags, NULL, NULL, &outBufLen);
	pAdapterAddresses = (IP_ADAPTER_ADDRESSES *) LocalAlloc(LMEM_ZEROINIT, outBufLen);
	GetAdaptersAddresses(AF_UNSPEC, ifFlags, NULL, pAdapterAddresses, &outBufLen);
	ai = pAdapterAddresses;
	while (ai != NULL) {
		if (ai->OperStatus != IfOperStatusUp) {
			ai = ai->Next;
			continue;
		}
		if (ai->IfType == IF_TYPE_SOFTWARE_LOOPBACK) {
			ai = ai->Next;
			continue;
		}
		if (ai->IfType == IF_TYPE_TUNNEL) {
			ai = ai->Next;
			continue;
		}
		uai = ai->FirstUnicastAddress;
		while (uai != NULL) {
			sockaddr = uai->Address;
			saddr = sockaddr.lpSockaddr;
			saddrlen = sockaddr.iSockaddrLength;
			namInfoRet = getnameinfo(saddr, saddrlen, addr, sizeof(addr), port, sizeof(port), NI_NUMERICHOST | NI_NUMERICSERV);
			if (namInfoRet == 0) {
				//if (IsUseAddress(addr) == true) {
					ifIdx = 0;
					if (cg_net_isipv6address(addr) == TRUE)
						ifIdx = cg_net_getipv6scopeid(addr);
					netIf = cg_net_interface_new();
					cg_net_interface_setaddress(netIf, addr);
					cg_net_interfacelist_add(netIfList, netIf);
				//}
			}
			else {
				int err = WSAGetLastError();
			}
			uai = uai->Next;
		}
		ai = ai->Next;
	}
	LocalFree(pAdapterAddresses);

#endif

	return cg_net_interfacelist_size(netIfList);

	cg_log_debug_l4("Leaving...\n");
}

#else

/****************************************
* cg_net_gethostinterfaces (UNIX)
****************************************/

#if defined(HAVE_IFADDRS_H)

int cg_net_gethostinterfaces(CgNetworkInterfaceList *netIfList)
{
	cg_log_debug_l4("Entering...\n");

	CgNetworkInterface *netIf;
	struct ifaddrs *ifaddr;
	char addr[NI_MAXHOST+1];
	char *ifname;
	
	cg_net_interfacelist_clear(netIfList);
	
	if (getifaddrs(&ifaddr) != 0)
	{
		cg_log_debug("No addresses for interfaces!\n");
		return 0;
	}
	
	struct ifaddrs *i;
	for (i = ifaddr; i != NULL; i = i->ifa_next) {
		if (!(i->ifa_flags & IFF_UP))
			continue;
		if (i->ifa_flags & IFF_LOOPBACK)
			continue;
		if (getnameinfo(i->ifa_addr, sizeof(struct sockaddr), addr, NI_MAXHOST, NULL, 0, NI_NUMERICHOST) == 0) {
			ifname = i->ifa_name;
			cg_log_debug("Interface name: %s, address: %s\n",  ifname, addr);
			netIf = cg_net_interface_new();
			cg_net_interface_setname(netIf, ifname);
			cg_net_interface_setaddress(netIf, addr);
			cg_net_interfacelist_add(netIfList, netIf);
		}
	}
	freeifaddrs(ifaddr);
	return cg_net_interfacelist_size(netIfList);

	cg_log_debug_l4("Leaving...\n");
}

#elif !defined(BTRON) && !defined(ITRON) && !defined(TENGINE)

#error This implementation lacks the interface id and cannot be currently used.

static const char *PATH_PROC_NET_DEV = "/proc/net/dev";

int cg_net_gethostinterfaces(CgNetworkInterfaceList *netIfList)
{
	cg_log_debug_l4("Entering...\n");

	CgNetworkInterface *netIf;
	FILE *fd;
	int s;
	char buffer[256+1];
	char ifaddr[20+1];
	char *ifname;
	char *sep;
	
	cg_net_interfacelist_clear(netIfList);
	
	s = socket(AF_INET, SOCK_DGRAM, 0);
	if (s < 0)
		return 0;
	fd = fopen(PATH_PROC_NET_DEV, "r");
	fgets(buffer, sizeof(buffer)-1, fd);
	fgets(buffer, sizeof(buffer)-1, fd);
	while (!feof(fd)) {
		ifname = buffer;
		sep;
		if (fgets(buffer, sizeof(buffer)-1, fd) == NULL)
			break;
		sep = strrchr(buffer, ':');
		if (sep)
			*sep = 0;
		while (*ifname == ' ')
			ifname++;
		struct ifreq req;
		strcpy(req.ifr_name, ifname);
		if (ioctl(s, SIOCGIFFLAGS, &req) < 0)
			continue;
		if (!(req.ifr_flags & IFF_UP))
			continue;
		if (req.ifr_flags & IFF_LOOPBACK)
			continue;
		if (ioctl(s, SIOCGIFADDR, &req) < 0)
			continue;
		strncpy(ifaddr, inet_ntoa(((struct sockaddr_in*)&req.ifr_addr)->sin_addr), sizeof(struct ifaddr)-1);
		netIf = cg_net_interface_new();
		cg_net_interface_setname(netIf, ifname);
		cg_net_interface_setaddress(netIf, ifaddr);
		cg_net_interfacelist_add(netIfList, netIf);
		cg_log_debug("Interface name: %s, address: %s\n", ifname, ifaddr);
		//cout << ifname << ", " << ifaddr << endl;
	}
	fclose(fd);
	close(s);
	return cg_net_interfacelist_size(netIfList);

	cg_log_debug_l4("Leaving...\n");
}

#endif

char *cg_net_selectaddr(struct sockaddr *remoteaddr)
{
	struct ifaddrs *ifaddrs, *ifaddr;
	uint32_t laddr, lmask, raddr;
	char *address_candidate = NULL, *auto_ip_address_candidate = NULL;

	raddr = ntohl(((struct sockaddr_in *)remoteaddr)->sin_addr.s_addr);

	if ( 0 != getifaddrs(&ifaddrs) )
	{
		return NULL;
	}

	for ( ifaddr = ifaddrs; NULL != ifaddr; ifaddr = ifaddr->ifa_next )
	{
		if (!(ifaddr->ifa_flags & IFF_UP))
			continue;
		if (ifaddr->ifa_flags & IFF_LOOPBACK)
			continue;
		if (ifaddr->ifa_flags & IFF_POINTOPOINT) 
			continue;

		laddr = ntohl(((struct sockaddr_in *)ifaddr->ifa_addr)->sin_addr.s_addr);
		if ( NULL != (struct sockaddr_in *)ifaddr->ifa_netmask )
			lmask = ntohl(((struct sockaddr_in *)ifaddr->ifa_netmask)->sin_addr.s_addr);
		else {
			cg_log_debug_s("No netmask for address %u!\n", laddr);
			continue;
		}

		/* Checking if we have an exact subnet match */
		if ( ( laddr & lmask ) == ( raddr & lmask ) ) {
			if ( NULL != address_candidate ) free(address_candidate);
			address_candidate = cg_strdup(
					inet_ntoa((struct in_addr)((struct sockaddr_in *)ifaddr->ifa_addr)->sin_addr));
			cg_log_debug_s("Address match! Selecting local address (%u)\n", laddr);
			break;
		}

		/* Checking if we have and auto ip address */
		if ( ( laddr & lmask ) == CG_AUTO_IP_NET ) {
			cg_log_debug_s("Found auto ip address. Selecting it for second address candidate (%u)\n", laddr);
			if ( NULL != auto_ip_address_candidate ) free(auto_ip_address_candidate);
			auto_ip_address_candidate = cg_strdup(
					inet_ntoa((struct in_addr)((struct sockaddr_in *)ifaddr->ifa_addr)->sin_addr));
		}
		/* Good. We have others than auto ips present. */
		else {
			cg_log_debug_s("Didn't have an exact subnet match, but non auto ip address anyway... (%u)\n", laddr);
			if ( NULL != address_candidate ) free(address_candidate);
			address_candidate = cg_strdup(
					inet_ntoa((struct in_addr)((struct sockaddr_in *)ifaddr->ifa_addr)->sin_addr));
		}
	}

	freeifaddrs(ifaddrs);

	if ( NULL != address_candidate )
		return address_candidate;

	if ( NULL != auto_ip_address_candidate )
		return auto_ip_address_candidate;

	/* Starting to feel desperate and returning local address.*/

	return cg_strdup("127.0.0.1");
}

#endif

/****************************************
* cg_net_gethostinterfaces (BTRON)
****************************************/

#if defined(BTRON) || (defined(TENGINE) && !defined(CG_TENGINE_NET_KASAGO))

int cg_net_gethostinterfaces(CgNetworkInterfaceList *netIfList)
{
	cg_log_debug_l4("Entering...\n");

	CgNetworkInterface *netIf;
	struct hostent hostEnt;
	B buf[HBUFLEN];
	ERR err;
	char *ifname = CG_NET_DEFAULT_IFNAME;
	char ifaddr[32];

	cg_net_interfacelist_clear(netIfList);
	
	err = so_gethostbyname("localhost", &hostEnt, buf);
	if (err != 0)
		return 0;
	
	inet_ntop(hostEnt.h_addrtype, hostEnt.h_addr, ifaddr, sizeof(ifname));
	
	netIf = cg_net_interface_new();
	cg_net_interface_setname(netIf, ifname);
	cg_net_interface_setaddress(netIf, ifaddr);
	cg_net_interfacelist_add(netIfList, netIf);
	
	return cg_net_interfacelist_size(netIfList);

	cg_log_debug_l4("Leaving...\n");
}

#endif

/****************************************
* cg_net_gethostinterfaces (TENGINE-KASAGO)
****************************************/

#if defined(TENGINE) && defined(CG_TENGINE_NET_KASAGO)

int cg_net_gethostinterfaces(CgNetworkInterfaceList *netIfList)
{
	cg_log_debug_l4("Entering...\n");

	CgNetworkInterface *netIf;
    struct in_addr inAddr;
    char ipaddr[CG_NET_IPV6_ADDRSTRING_MAXSIZE];
	int kaRet;
	
	cg_socket_startup();

    inAddr.s_addr = 0;
    kaRet = ka_tfGetIpAddress(kaInterfaceHandle, &(inAddr.s_addr), 0);
    if(kaRet != 0)
		return 0;
		
    ka_tfInetToAscii((unsigned long)inAddr.s_addr, ipaddr);
	
	netIf = cg_net_interface_new();
	cg_net_interface_setname(netIf, CG_NET_DEFAULT_IFNAME);
	cg_net_interface_setaddress(netIf, ipaddr);
	cg_net_interfacelist_add(netIfList, netIf);
	
	return cg_net_interfacelist_size(netIfList);

	cg_log_debug_l4("Leaving...\n");
}

#endif

/****************************************
* cg_net_gethostinterfaces (TENGINE-KASAGO)
****************************************/

#if defined(TENGINE) && defined(CG_TENGINE_NET_KASAGO)

int cg_net_gethostinterfaces(CgNetworkInterfaceList *netIfList)
{
	cg_log_debug_l4("Entering...\n");

	CgNetworkInterface *netIf;
    struct in_addr inAddr;
    char ipaddr[CG_NET_IPV6_ADDRSTRING_MAXSIZE];
	int kaRet;
	
	cg_socket_startup();

    inAddr.s_addr = 0;
    kaRet = ka_tfGetIpAddress(kaInterfaceHandle, &(inAddr.s_addr), 0);
    if(kaRet != 0)
		return 0;
		
    ka_tfInetToAscii((unsigned long)inAddr.s_addr, ipaddr);
	
	netIf = cg_net_interface_new();
	cg_net_interface_setname(netIf, CG_NET_DEFAULT_IFNAME);
	cg_net_interface_setaddress(netIf, ipaddr);
	cg_net_interfacelist_add(netIfList, netIf);
	
	return cg_net_interfacelist_size(netIfList);

	cg_log_debug_l4("Leaving...\n");
}

#endif

/****************************************
* cg_net_gethostinterfaces (ITRON)
****************************************/

#if defined(ITRON)

void cg_net_setinterface(char *ifaddr)
{
	cg_log_debug_l4("Entering...\n");

	cg_strcpy(InterfaceAddress, ifaddr);
	IsInterfaceAddressInitialized = TRUE;

	cg_log_debug_l4("Leaving...\n");
}

int cg_net_gethostinterfaces(CgNetworkInterfaceList *netIfList)
{
	cg_log_debug_l4("Entering...\n");

	CgNetworkInterface *netIf;

	if (IsInterfaceAddressInitialized == FALSE)
			return 0;

	netIf = cg_net_interface_new();
	cg_net_interface_setname(netIf, "");
	cg_net_interface_setaddress(netIf, InterfaceAddress);
	cg_net_interfacelist_add(netIfList, netIf);
	
	return cg_net_interfacelist_size(netIfList);

	cg_log_debug_l4("Leaving...\n");
}

#endif
