/**
 * Clinkc-av control point test tool
 *
 * Content Directory Service tests
 *
 * Copyright (c) 2006 Nokia Corporation. All rights reserved.
 *
 * This is licensed under BSD-style license with patent exclusion,
 * see file COPYING.
 */

#include <cybergarage/contentdirectory/ccontentdirectory.h>
#include <cybergarage/contentdirectory/ccdscontrolpoint.h>
#include <cybergarage/contentdirectory/ccdsactionarguments.h>
#include <cybergarage/contentdirectory/cdidllite.h>
#include <cybergarage/avcontrol/cavcontrolpoint.h>

#include <math.h>

#include "cptest.h"
#include "cdstest.h"

AVCPSubTestCase cds_tc[NUM_CDS_TEST_CASES] = 
{ 
	{TEST_CASE_CDS_GET_SEARCH_CAPS, CG_UPNP_STATUS_ACTION_FAILED, cp_test_cds_get_search_caps, NULL},
	{TEST_CASE_CDS_GET_SORT_CAPS, CG_UPNP_STATUS_ACTION_FAILED, cp_test_cds_get_sort_caps, NULL},
	{TEST_CASE_CDS_GET_SYSTEM_UPDATE_ID, CG_UPNP_STATUS_ACTION_FAILED, cp_test_cds_get_system_update_id, NULL},
	{TEST_CASE_CDS_BROWSE_DIRECT_CHILDREN, CG_UPNP_STATUS_ACTION_FAILED, cp_test_cds_browse_direct_children, NULL},
	{TEST_CASE_CDS_SEARCH, CG_UPNP_STATUS_ACTION_FAILED, cp_test_cds_search, NULL}
};

int cp_test_cds_print_statistics()
{
	int i = 0;
	int passed = 0;
	
	printf("\n************************ Content Directory test results ************************\n");

	/* Print results */
	for (i = 0; i < NUM_CDS_TEST_CASES; i++)
	{
		if (cg_upnp_av_control_iserrorcodesuccessful(cds_tc[i].result))
		{
			printf("%s: PASS\n", cds_tc[i].name);
			passed++;
		}
		else
		{
			printf("%s: FAIL (%d)\n",
				cds_tc[i].name, cds_tc[i].result);
		}
	}
	
	printf("\nContent Directory pass rate: %d of %d (%.0f%%)\n",
		passed, NUM_CDS_TEST_CASES, 
		ceil(((float) passed / (float) NUM_CDS_TEST_CASES) * (float) 100));
	
	return passed;
}

int cp_test_cds(CgUpnpControlPoint* cp)
{
	int i = 0;
	char* udn = NULL;
	BOOL retval = TRUE;
	SUB_TEST_CASE_FUNCTION func = NULL;

	printf("\nBEGIN: cp_test_cds\n");

	if (cp == NULL)
	{
		printf("No control point\n");
		printf("END: cp_test_cds\n");

		return FALSE;
	}

	/* Get a device with CDS capabilities */
	udn = cp_test_get_device_udn_with_service(cp, CG_UPNP_CDS_SERVICE_TYPE);
	if (udn == NULL)
	{
		printf("END: cp_test_cds_browse\n");
		return FALSE;
	}

	/* Run thru test cases */
	for (i = 0; i < NUM_CDS_TEST_CASES; i++)
	{
		func = cds_tc[i].func;
		cds_tc[i].result = func(cp, udn);

		/* One failed sub test case is enough to fail the CDS test */
		if (!cg_upnp_av_control_iserrorcodesuccessful(cds_tc[i].result))
		{
			retval = FALSE;
		}
	}
	
	/* Free extra data */
	for (i = 0; i < NUM_CDS_TEST_CASES; i++)
	{
		if (cds_tc[i].result_data != NULL)
		{
			free(cds_tc[i].result_data);
			cds_tc[i].result_data = NULL;
		}
	}
	
	free(udn);
	
	printf("END: cp_test_cds\n");
	
	return retval;
}

int cp_test_cds_get_search_caps(CgUpnpControlPoint* cp, char* udn)
{
	int retval = 0;
	char* errorString = NULL;
	int index = -1;	
	char* searchCaps = NULL;
	
	printf("BEGIN: cp_test_cds_get_search_caps\n");
	
	/* Invoke a GetSearchCaps action */
	cg_upnp_controlpoint_lock(cp);
	
	retval = cg_upnp_av_cds_control_getsearchcaps(
			cg_upnp_controlpoint_getdevicebyudn(cp, udn),
			&errorString,
			&searchCaps);
	
	cg_upnp_controlpoint_unlock(cp);

	if (cg_upnp_av_control_iserrorcodesuccessful(retval))
	{
		/* Save the result text to test results */
		index = cp_test_find_test_case((AVCPTestCase*) cds_tc, 
				NUM_CDS_TEST_CASES,
				TEST_CASE_CDS_GET_SEARCH_CAPS);

		cds_tc[index].result_data = (void*) searchCaps;
		
		printf("Search Capabilities: %s\n", searchCaps);
	}
	else
	{
		printf("Unable to invoke action: %d (%s)\n",
			retval, errorString);
		free(errorString);
	}
	
	printf("END: cp_test_cds_get_search_caps\n");
	
	return retval;
}

int cp_test_cds_get_sort_caps(CgUpnpControlPoint* cp, char* udn)
{
	int retval = 0;
	char* errorString = NULL;
	int index = -1;	
	char* sortCaps = NULL;
	
	printf("BEGIN: cp_test_cds_get_sort_caps\n");
	
	/* Invoke a GetSearchCaps action */
	cg_upnp_controlpoint_lock(cp);
	
	retval = cg_upnp_av_cds_control_getsortcaps(
			cg_upnp_controlpoint_getdevicebyudn(cp, udn),
			&errorString,
			&sortCaps);
	
	cg_upnp_controlpoint_unlock(cp);

	if (cg_upnp_av_control_iserrorcodesuccessful(retval))
	{
		/* Save the result text to test results */
		index = cp_test_find_test_case((AVCPTestCase*) cds_tc, 
				NUM_CDS_TEST_CASES,
				TEST_CASE_CDS_GET_SORT_CAPS);
		
		cds_tc[index].result_data = (void*) sortCaps;
		
		printf("Sort Capabilities: %s\n", sortCaps);
	}
	else
	{
		printf("Unable to invoke action: %d (%s)\n",
			retval, errorString);
		free(errorString);
	}
	
	printf("END: cp_test_cds_get_sort_caps\n");
	
	return retval;
}

int cp_test_cds_get_system_update_id(CgUpnpControlPoint* cp, char* udn)
{
	int retval = 0;
	char* errorString = NULL;
	int index = -1;	
	char* id = NULL;
	
	printf("BEGIN: cp_test_cds_get_system_update_id\n");
	
	/* Invoke a GetSearchCaps action */
	cg_upnp_controlpoint_lock(cp);
	
	retval = cg_upnp_av_cds_control_getsystemupdateid(
			cg_upnp_controlpoint_getdevicebyudn(cp, udn),
			&errorString,
			&id);
	
	cg_upnp_controlpoint_unlock(cp);

	if (cg_upnp_av_control_iserrorcodesuccessful(retval))
	{
		/* Save the result text to test results */
		index = cp_test_find_test_case((AVCPTestCase*) cds_tc, 
				NUM_CDS_TEST_CASES,
				TEST_CASE_CDS_GET_SYSTEM_UPDATE_ID);
		
		cds_tc[index].result_data = (void*) id;

		printf("System Update ID: %s\n", id);
	}
	else
	{
		printf("Unable to invoke action: %d (%s)\n",
			retval, errorString);
		free(errorString);
	}
	
	printf("END: cp_test_cds_get_system_update_id\n");
	
	return retval;
}

int cp_test_cds_browse_direct_children(CgUpnpControlPoint* cp, char* udn)
{
	int action_result = 0;
	char* errorString = NULL;
	BOOL retval = FALSE;
	int index = -1;
	
	char* objectID = CG_UPNP_AV_CDS_BROWSE_OBJECTID_ROOT;
	char* browseFlag = CG_UPNP_AV_CDS_BROWSE_FLAG_DIRECTCHILDREN;
	char* filter = CG_UPNP_AV_CDS_BROWSE_FILTER_ALL;
	char* startingIndex = CG_UPNP_AV_CDS_BROWSE_STARTINGINDEX_FIRST;
	char* requestedCount = CG_UPNP_AV_CDS_BROWSE_REQUESTEDCOUNT_ALL;
	char* sortCriteria = CG_UPNP_AV_CDS_BROWSE_SORTCRITERIA_NONE;
	char* result = NULL;
	char* numberReturned = NULL;
	char* totalMatches = NULL;
	char* updateID = NULL;
	
	CgString* treeDump = cg_string_new();
	CgXmlNodeList* nodelist = NULL;
	
	printf("BEGIN: cp_test_cds_browse_direct_children\n");
	
	/* Create a new nodelist for the DIDL result */
	nodelist = cg_xml_nodelist_new();

	/* Invoke a browse action */
	cg_upnp_controlpoint_lock(cp);
	
	action_result = cg_upnp_av_cds_control_browse(
			cg_upnp_controlpoint_getdevicebyudn(cp, udn),
			&errorString,
			&objectID,
			&browseFlag, 
			&filter, 
			&startingIndex,
			&requestedCount, 
			&sortCriteria, 
			&result,
			&numberReturned,
			&totalMatches,
			&updateID);
			
	cg_upnp_controlpoint_unlock(cp);

	if (cg_upnp_av_control_iserrorcodesuccessful(action_result))
	{
		/* Create an XML nodelist */
		retval = cg_upnp_av_cds_create_cg_xml(result, nodelist);
		
		/* Parse the XML nodelist to the tree model */
		if (retval == TRUE)
		{
			/* Save the result text to test results */
			index = cp_test_find_test_case((AVCPTestCase*) cds_tc, 
					NUM_CDS_TEST_CASES,
					TEST_CASE_CDS_BROWSE_DIRECT_CHILDREN);
			
			cds_tc[index].result_data = (void*) result;
			
			/* Check the validity of the DIDL-Lite document and get the
			 * child items that actually contain the interesting information */
			if (!cg_upnp_av_cds_didllite_getchildren(nodelist, FALSE))
			{
				printf("Browse result XML isn't valid DIDL-Lite\n");
				action_result = -1;
			}
			else
			{
				printf("XML Tree: \n%s\n",
					cg_xml_node_tostring(
						cg_xml_nodelist_gets(nodelist), 
						TRUE,
						treeDump));
				
				cg_string_delete(treeDump);
			}
		}
		else
		{
			printf("Browse result didn't produce valid XML\n");
			action_result = -1;
		}
	}
	else
	{
		printf("Unable to invoke action: %d (%s)\n",
			retval, errorString);
		free(errorString);
	}
	
	/* Get rid of the XML nodelist */
	cg_xml_nodelist_delete(nodelist);

	free(numberReturned);
	free(totalMatches);
	free(updateID);
	
	printf("END: cp_test_cds_browse_direct_children\n");
	
	return action_result;
}

int cp_test_cds_search(CgUpnpControlPoint* cp, char* udn)
{
	int action_result = 0;
	char* errorString = NULL;
	BOOL retval = FALSE;
	int index = -1;
	
	char* containerID = CG_UPNP_AV_CDS_BROWSE_OBJECTID_ROOT;
	char* searchCriteria = "*";
	char* filter = CG_UPNP_AV_CDS_BROWSE_FILTER_ALL;
	char* startingIndex = CG_UPNP_AV_CDS_BROWSE_STARTINGINDEX_FIRST;
	char* requestedCount = CG_UPNP_AV_CDS_BROWSE_REQUESTEDCOUNT_ALL;
	char* sortCriteria = CG_UPNP_AV_CDS_BROWSE_SORTCRITERIA_NONE;
	char* result = NULL;
	char* numberReturned = NULL;
	char* totalMatches = NULL;
	char* updateID = NULL;
	
	CgString* treeDump = cg_string_new();
	CgXmlNodeList* nodelist = NULL;
	
	printf("BEGIN: cp_test_cds_search\n");
	
	/* Create a new nodelist for the DIDL result */
	nodelist = cg_xml_nodelist_new();

	/* Invoke a browse action */
	cg_upnp_controlpoint_lock(cp);
	
	action_result = cg_upnp_av_cds_control_browse(
			cg_upnp_controlpoint_getdevicebyudn(cp, udn),
			&errorString,
			&containerID,
			&searchCriteria, 
			&filter, 
			&startingIndex,
			&requestedCount, 
			&sortCriteria, 
			&result,
			&numberReturned,
			&totalMatches,
			&updateID);
			
	cg_upnp_controlpoint_unlock(cp);

	if (cg_upnp_av_control_iserrorcodesuccessful(action_result))
	{
		/* Create an XML nodelist */
		retval = cg_upnp_av_cds_create_cg_xml(result, nodelist);
		
		/* Parse the XML nodelist to the tree model */
		if (retval == TRUE)
		{
			/* Save the result text to test results */
			index = cp_test_find_test_case((AVCPTestCase*) cds_tc, 
					NUM_CDS_TEST_CASES,
					TEST_CASE_CDS_SEARCH);
			
			cds_tc[index].result_data = (void*) result;
			
			/* Check the validity of the DIDL-Lite document and get the
			 * child items that actually contain the interesting information */
			if (!cg_upnp_av_cds_didllite_getchildren(nodelist, FALSE))
			{
				printf("Search result XML isn't valid DIDL-Lite\n");
				action_result = -1;
			}
			else
			{
				printf("XML Tree: \n%s\n",
					cg_xml_node_tostring(
						cg_xml_nodelist_gets(nodelist), 
						TRUE,
						treeDump));
				
				cg_string_delete(treeDump);
			}
		}
		else
		{
			printf("Search result didn't produce valid XML\n");
			action_result = -1;
		}
	}
	else
	{
		printf("Unable to invoke action: %d (%s)\n",
			retval, errorString);
		free(errorString);
	}
	
	/* Get rid of the XML nodelist */
	cg_xml_nodelist_delete(nodelist);

	free(numberReturned);
	free(totalMatches);
	free(updateID);
	
	printf("END: cp_test_cds_search\n");
	
	return action_result;
}

