/**
 * Clinkc-av control point test tool
 *
 * AV Transport Service tests
 *
 * Copyright (c) 2006 Nokia Corporation. All rights reserved.
 *
 * This is licensed under BSD-style license with patent exclusion,
 * see file COPYING.
 */

#include <cybergarage/avtransport/cavtransport.h>
#include <cybergarage/avtransport/cavtcontrolpoint.h>
#include <cybergarage/avtransport/cavtactionarguments.h>
#include <cybergarage/avcontrol/cavcontrolpoint.h>

#include <cybergarage/contentdirectory/ccdscontrolpoint.h>
#include <cybergarage/contentdirectory/ccontentdirectory.h>
#include <cybergarage/contentdirectory/cdidllite.h>

#include <math.h>

#include "cptest.h"
#include "avttest.h"
#include "cdstest.h"

AVCPSubTestCase avt_tc[NUM_AVT_TEST_CASES] = 
{
	{TEST_CASE_AVT_SET_AVTRANSPORT_URI, CG_UPNP_STATUS_ACTION_FAILED, cp_test_avt_set_avtransport_uri, NULL},
	{TEST_CASE_AVT_GET_DEVICE_CAPABILITIES, CG_UPNP_STATUS_ACTION_FAILED, cp_test_avt_get_device_capabilities, NULL},
	{TEST_CASE_AVT_GET_TRANSPORT_SETTINGS, CG_UPNP_STATUS_ACTION_FAILED, cp_test_avt_get_transport_settings, NULL},
	{TEST_CASE_AVT_STOP, CG_UPNP_STATUS_ACTION_FAILED, cp_test_avt_stop, NULL},
	{TEST_CASE_AVT_PLAY, CG_UPNP_STATUS_ACTION_FAILED, cp_test_avt_play, NULL},
	{TEST_CASE_AVT_SEEK, CG_UPNP_STATUS_ACTION_FAILED, cp_test_avt_seek, NULL},
	{TEST_CASE_AVT_GET_MEDIA_INFO, CG_UPNP_STATUS_ACTION_FAILED, cp_test_avt_get_media_info, NULL},	
	{TEST_CASE_AVT_GET_TRANSPORT_INFO, CG_UPNP_STATUS_ACTION_FAILED, cp_test_avt_get_transport_info, NULL},
	{TEST_CASE_AVT_GET_POSITION_INFO, CG_UPNP_STATUS_ACTION_FAILED, cp_test_avt_get_position_info, NULL},
	{TEST_CASE_AVT_NEXT, CG_UPNP_STATUS_ACTION_FAILED, cp_test_avt_next, NULL},
	{TEST_CASE_AVT_PREVIOUS, CG_UPNP_STATUS_ACTION_FAILED, cp_test_avt_previous, NULL}
};

extern AVCPSubTestCase cds_tc[];

int cp_test_avt_print_statistics()
{
	int i = 0;
	int passed = 0;
	
	printf("\n************************** AV Transport test results ***************************\n");

	/* Print results */
	for (i = 0; i < NUM_AVT_TEST_CASES; i++)
	{
		if (cg_upnp_av_control_iserrorcodesuccessful(avt_tc[i].result))
		{
			printf("%s: PASS\n", avt_tc[i].name);
			passed++;
		}
		else
		{
			printf("%s: FAIL (%d)\n",
				avt_tc[i].name, avt_tc[i].result);
		}
	}
	
	printf("\nAV Transport pass rate: %d of %d (%.0f%%)\n",
		passed, NUM_AVT_TEST_CASES, 
		ceil(((float) passed / (float) NUM_AVT_TEST_CASES) * (float) 100));
	
	return passed;
}

int cp_test_avt(CgUpnpControlPoint* cp)
{
	int i = 0;
	BOOL retval = TRUE;
	char* udn = NULL;
	SUB_TEST_CASE_FUNCTION func = NULL;
	
	printf("\nBEGIN: cp_test_avt\n");

	/* Get a device with AVT capabilities */
	udn = cp_test_get_device_udn_with_service(cp, CG_UPNP_AVT_SERVICE_TYPE);
	if (udn == NULL)
	{
		printf("END: cp_test_avt\n");
		return FALSE;
	}
	
	/* Run thru test cases */
	for (i = 0; i < NUM_AVT_TEST_CASES; i++)
	{
		func = avt_tc[i].func;
		avt_tc[i].result = func(cp, udn);

		/* One failed sub test case is enough to fail the test */
		if (!cg_upnp_av_control_iserrorcodesuccessful(avt_tc[i].result))
		{
			retval = FALSE;
		}
	}
	
	/* Free extra data */
	for (i = 0; i < NUM_AVT_TEST_CASES; i++)
	{
		if (avt_tc[i].result_data != NULL)
		{
			free(avt_tc[i].result_data);
			avt_tc[i].result_data = NULL;
		}
	}
	
	free(udn);
	
	printf("END: cp_test_avt\n");
	
	return retval;
}

int cp_test_avt_set_avtransport_uri(CgUpnpControlPoint* cp, char* udn)
{
	int retval = 0;
	int index = 0;
	char* errorString = NULL;
	char* instanceID = "0";
	char* uri = NULL;
	char* uriMetaData = NULL;
	
	printf("BEGIN: cp_test_avt_set_avtransport_uri\n");

	/* Get the previous result from CDS browse test */
	index = cp_test_find_test_case((AVCPTestCase*) cds_tc, 
					NUM_CDS_TEST_CASES,
					TEST_CASE_CDS_BROWSE_DIRECT_CHILDREN);

	if (cds_tc[index].result_data == NULL)
	{
		printf("No successful result from a previous browse action.\n");
		printf("Setting empty uri.\n");
	}
	else
	{
		cp_test_avt_get_res_from_didllite(cds_tc[index].result_data, 
						  &uri);
		
		printf("Using URI: %s\n", uri);
	}
	
	cg_upnp_controlpoint_lock(cp);
	
	retval = cg_upnp_av_avt_control_setavtransporturi(
		cg_upnp_controlpoint_getdevicebyudn(cp, udn),
		&errorString,
		&instanceID,
		&uri,
		&uriMetaData);

	cg_upnp_controlpoint_unlock(cp);
	
	if (cg_upnp_av_control_iserrorcodesuccessful(retval) == FALSE)
	{
		printf("Unable to invoke action: %d (%s)\n", 
			retval, errorString);
		free(errorString);
	}

	printf("END: cp_test_avt_set_avtransport_uri\n");
	
	return retval;
}

BOOL cp_test_avt_get_res_from_didllite(char* doc, char** uri)
{
	int retval = 0;
	CgXmlNodeList* nodelist = cg_xml_nodelist_new();
	CgXmlNodeList* children = NULL;
	CgXmlNode* node = NULL;
	CgXmlNode* subnode = NULL;
	
	/* Create the XML nodelist */
	if (cg_upnp_av_cds_create_cg_xml(doc, nodelist) == FALSE)
	{
		printf("Browse result is not valid XML\n");
		cg_xml_nodelist_delete(nodelist);
		return FALSE;
	}

	/* Get the child nodes */
	children = cg_upnp_av_cds_didllite_getchildren(nodelist, TRUE);
	if (children == NULL)
	{
		printf("Browse result isn't valid DIDL-Lite\n");
		cg_xml_nodelist_delete(nodelist);
		return FALSE;
	}
	else
	{
		/* The outer loop iterates <item> and <container> nodes */
		for (node = cg_upnp_av_cds_didllite_nodelist_gets(children);
		     node != NULL;
		     node = cg_upnp_av_cds_didllite_node_next(node))
		{
			/* The inner loop iterates <res> etc. */
			for (subnode = cg_upnp_av_cds_didllite_nodelist_gets(cg_upnp_av_cds_didllite_node_getchildnodelist(node));
			     subnode != NULL;
			     subnode = cg_upnp_av_cds_didllite_node_next(subnode))
			{
				if (cg_upnp_av_cds_didllite_node_isname(subnode, DIDL_LITE_RES))
				{
					*uri = cg_strdup(cg_upnp_av_cds_didllite_node_getvalue(subnode));
					break;
				}
			}
			
			if (*uri)
			{
				/* Stop iteration if uri was found */
				break;
			}
		}
	}
	
	if (*uri)
	{
		retval = TRUE;
	}
	else
	{
		retval = FALSE;
	}
	
	cg_xml_nodelist_delete(nodelist);
	
	return retval;
}

int cp_test_avt_get_media_info(CgUpnpControlPoint* cp, char* udn)
{
	int retval = 0;
	char* errorString = NULL;
	char* instanceID = "0";
	char* nrTracks = NULL;
	char* mediaDuration = NULL;
	char* currentUri = NULL;
	char* currentUriMetaData = NULL;
	char* nextUri = NULL;
	char* nextUriMetaData = NULL;
	char* playMedium = NULL;
	char* recordMedium = NULL;
	char* writeStatus = NULL;
	
	printf("BEGIN: cp_test_avt_get_media_info\n");
	
	cg_upnp_controlpoint_lock(cp);
	
	retval = cg_upnp_av_avt_control_getmediainfo(
			cg_upnp_controlpoint_getdevicebyudn(cp, udn),
			&errorString,
			&instanceID,
			&nrTracks,
			&mediaDuration,
			&currentUri,
			&currentUriMetaData,
			&nextUri,
			&nextUriMetaData,
			&playMedium,
			&recordMedium,
			&writeStatus);

	cg_upnp_controlpoint_unlock(cp);
	
	if (cg_upnp_av_control_iserrorcodesuccessful(retval) == FALSE)
	{
		printf("Unable to invoke action: %d (%s)\n", 
			retval, errorString);
		free(errorString);
	}
	else
	{
		printf("Number of Tracks: %s\n", nrTracks);
		printf("Media Duration: %s\n", mediaDuration);
		printf("Current URI: %s\n", currentUri);
		printf("Current URI Metadata: [%s]\n", currentUriMetaData);
		printf("Next URI: %s\n", nextUri);
		printf("Next URI Metadata: [%s]\n", nextUriMetaData);
		printf("Play Medium: %s\n", playMedium);
		printf("Record Medium: %s\n", recordMedium);
		printf("Write Status: %s\n", writeStatus);
	}
	
	free(nrTracks);
	free(mediaDuration);
	free(currentUri);
	free(currentUriMetaData);
	free(nextUri);
	free(nextUriMetaData);
	free(playMedium);
	free(recordMedium);
	free(writeStatus);
	
	printf("END: cp_test_avt_get_media_info\n");
	
	return retval;
}

int cp_test_avt_get_transport_info(CgUpnpControlPoint* cp, char* udn)
{
	int retval = 0;
	char* errorString = NULL;

	char* instanceID = "0";
	char* currentTransportState = NULL;
	char* currentTransportStatus = NULL;
	char* currentSpeed = NULL;

	printf("BEGIN: cp_test_avt_get_transport_info\n");
	
	cg_upnp_controlpoint_lock(cp);
	
	retval = cg_upnp_av_avt_control_gettransportinfo(
			cg_upnp_controlpoint_getdevicebyudn(cp, udn),
			&errorString,
			&instanceID,
			&currentTransportState,
			&currentTransportStatus,
			&currentSpeed);

	cg_upnp_controlpoint_unlock(cp);
	
	if (cg_upnp_av_control_iserrorcodesuccessful(retval) == FALSE)
	{
		printf("Unable to invoke action: %d (%s)\n", 
			retval, errorString);
		free(errorString);
	}
	else
	{
		printf("Current Transport State: %s\n", currentTransportState);
		printf("Current Transport Status: %s\n", currentTransportStatus);
		printf("Current Speed: %s\n", currentSpeed);
	}
	
	free(currentTransportState);
	free(currentTransportStatus);
	free(currentSpeed);

	printf("END: cp_test_avt_get_transport_info\n");
	
	return retval;
}

int cp_test_avt_get_position_info(CgUpnpControlPoint* cp, char* udn)
{
	int retval = 0;
	char* errorString = NULL;
	char* instanceID = "0";
	char* track = NULL;
	char* trackDuration = NULL;
	char* trackMetaData = NULL;
	char* trackURI = NULL;
	char* relTime = NULL;
	char* absTime = NULL;
	char* relCount = NULL;
	char* absCount = NULL;

	printf("BEGIN: cp_test_avt_get_position_info\n");

	cg_upnp_controlpoint_lock(cp);
	
	retval = cg_upnp_av_avt_control_getpositioninfo(
			cg_upnp_controlpoint_getdevicebyudn(cp, udn),
			&errorString,
			&instanceID,
			&track,
			&trackDuration,
			&trackMetaData,
			&trackURI,
			&relTime,
			&absTime,
			&relCount,
			&absCount);

	cg_upnp_controlpoint_unlock(cp);
	
	if (cg_upnp_av_control_iserrorcodesuccessful(retval) == FALSE)
	{
		printf("Unable to invoke action: %d (%s)\n",
			retval, errorString);
		free(errorString);
	}
	else
	{
		printf("Track: %s\n", track);
		printf("Track Duration: %s\n", trackDuration);
		printf("Track Metadata: %s\n", trackMetaData);
		printf("Track URI: %s\n", trackURI);
		printf("Relative Time: %s\n", relTime);
		printf("Absolute Time: %s\n", absTime);
		printf("Relative Count: %s\n", relCount);
		printf("Absolute Count: %s\n", absCount);
	}

	free(track);
	free(trackDuration);
	free(trackMetaData);
	free(trackURI);
	free(relTime);
	free(absTime);
	free(relCount);
	free(absCount);
	
	printf("END: cp_test_avt_get_position_info\n");
	
	return retval;
}

int cp_test_avt_get_device_capabilities(CgUpnpControlPoint* cp, char* udn)
{
	int retval = 0;
	char* errorString = NULL;
	char* instanceID = "0";
	char* playMedia = NULL;
	char* recMedia = NULL;
	char* recQualityModes = NULL;
	
	printf("BEGIN: cp_test_avt_get_device_capabilities\n");

	cg_upnp_controlpoint_lock(cp);
	
	retval = cg_upnp_av_avt_control_getdevicecapabilities(
			cg_upnp_controlpoint_getdevicebyudn(cp, udn),
			&errorString,
			&instanceID,
			&playMedia,
			&recMedia,
			&recQualityModes);
	
	cg_upnp_controlpoint_unlock(cp);
	
	if (cg_upnp_av_control_iserrorcodesuccessful(retval) == FALSE)
	{
		printf("Unable to invoke action: %d (%s)\n",
			retval, errorString);
		free(errorString);
	}
	else
	{
		printf("Play Media: %s\n", playMedia);
		printf("Record Media: %s\n", recMedia);
		printf("Record Quality Modes: %s\n", recQualityModes);
	}

	free(playMedia);
	free(recMedia);
	free(recQualityModes);
	
	printf("END: cp_test_avt_get_device_capabilities\n");
	
	return retval;
}

int cp_test_avt_get_transport_settings(CgUpnpControlPoint* cp, char* udn)
{
	int retval = 0;
	char* errorString = NULL;
	char* instanceID = "0";
	char* playMode = NULL;
	char* recQualityMode = NULL;
	
	printf("BEGIN: cp_test_avt_get_transport_settings\n");

	cg_upnp_controlpoint_lock(cp);
	
	cg_upnp_av_avt_control_gettransportsettings(
		cg_upnp_controlpoint_getdevicebyudn(cp, udn),
		&errorString,
		&instanceID,
		&playMode,
		&recQualityMode);
	
	cg_upnp_controlpoint_unlock(cp);

	if (cg_upnp_av_control_iserrorcodesuccessful(retval) == FALSE)
	{
		printf("Unable to invoke action: %d (%s)\n",
			retval, errorString);
		free(errorString);
	}
	else
	{
		printf("Play Mode: %s\n", playMode);
		printf("Record Quality Mode: %s\n", recQualityMode);
	}
	
	free(playMode);
	free(recQualityMode);
	
	printf("END: cp_test_avt_get_transport_settings\n");

	return retval;
}

int cp_test_avt_stop(CgUpnpControlPoint* cp, char* udn)
{
	int retval = 0;
	char* errorString = "0";
	char* instanceID = NULL;

	printf("BEGIN: cp_test_avt_stop\n");

	cg_upnp_controlpoint_lock(cp);
	
	/* Invoke stop action */
	retval = cg_upnp_av_avt_control_stop(
		cg_upnp_controlpoint_getdevicebyudn(cp, udn),
		&errorString, 
		&instanceID);

	cg_upnp_controlpoint_unlock(cp);

	if (cg_upnp_av_control_iserrorcodesuccessful(retval) == FALSE)
	{
		printf("Unable to invoke action: %d (%s)\n", retval, errorString);
		free(errorString);
	}
	
	printf("END: cp_test_avt_stop\n");
	
	return retval;
}

int cp_test_avt_play(CgUpnpControlPoint* cp, char* udn)
{
	int retval = 0;
	char* errorString = NULL;
	char* instanceID = "0";
	char* speed = "1";

	printf("BEGIN: cp_test_avt_play\n");
	
	cg_upnp_controlpoint_lock(cp);
	
	/* Invoke play action */
	retval = cg_upnp_av_avt_control_play(
		cg_upnp_controlpoint_getdevicebyudn(cp, udn),
		&errorString,
		&instanceID,
		&speed);

	cg_upnp_controlpoint_unlock(cp);

	if (cg_upnp_av_control_iserrorcodesuccessful(retval) == FALSE)
	{
		printf("Unable to invoke action: %d (%s)\n",
			retval, errorString);
		free(errorString);
	}

	printf("END: cp_test_avt_play\n");
	
	return retval;
}

int cp_test_avt_seek(CgUpnpControlPoint* cp, char* udn)
{
	int retval = 0;
	
	printf("BEGIN: cp_test_avt_seek\n");
	printf("END: cp_test_avt_seek\n");
	
	return retval;
}

int cp_test_avt_next(CgUpnpControlPoint* cp, char* udn)
{
	int retval = 0;
	char* errorString = NULL;
	char* instanceID = "0";

	printf("BEGIN: cp_test_avt_next\n");
	
	cg_upnp_controlpoint_lock(cp);
	
	/* Invoke next action */
	retval = cg_upnp_av_avt_control_next(
		cg_upnp_controlpoint_getdevicebyudn(cp, udn), 
		&errorString,
		&instanceID);

	cg_upnp_controlpoint_unlock(cp);

	if (cg_upnp_av_control_iserrorcodesuccessful(retval) == FALSE)
	{
		printf("Unable to invoke action: %d (%s)\n",
			retval, errorString);
		free(errorString);
	}

	printf("END: cp_test_avt_next\n");
	
	return retval;
}

int cp_test_avt_previous(CgUpnpControlPoint* cp, char* udn)
{
	int retval = 0;
	char* errorString = NULL;
	char* instanceID = "0";

	printf("BEGIN: cp_test_avt_previous\n");
	
	cg_upnp_controlpoint_lock(cp);
	
	/* Invoke previous action */
	retval = cg_upnp_av_avt_control_previous(
		cg_upnp_controlpoint_getdevicebyudn(cp, udn), 
		&errorString, 
		&instanceID);

	cg_upnp_controlpoint_unlock(cp);

	if (cg_upnp_av_control_iserrorcodesuccessful(retval) == FALSE)
	{
		printf("Unable to invoke action: %d (%s)\n",
			retval, errorString);
		free(errorString);
	}

	printf("END: cp_test_avt_previous\n");
	
	return retval;
}
