/*
 * UPnP Browser for Maemo
 *
 * interface.c
 *
 * Copyright 2005 Nokia Corporation. All rights reserved.
 *
 * This is licensed under BSD-style license with patent exclusion,
 * see file COPYING.
 */

#ifdef MAEMO
#include <hildon-widgets/hildon-app.h>
#include <hildon-widgets/hildon-appview.h>
#include <hildon-widgets/hildon-defines.h>
#include <hildon-widgets/gtk-infoprint.h>
#endif

#include <gtk/gtk.h>

#include "interface.h"
#include "upnp.h"
#include "settings.h"
#include "preset_dialog.h"
#include "browser.h"
#include "now_playing.h"
#include "media_control.h"
#include "helper.h"
#include "playlist.h"
#include "playlist_dialog.h"

/**
 * Global struct that holds the application widgets
 */
UPnPBrowserWidgets* widgets;

int create_widgets()
{	
	/* Tooltips */
	widgets->tips = gtk_tooltips_new();
	
	/* Toolbar/Browsers Main VBOX */
	widgets->vpaned = gtk_vpaned_new();
	gtk_widget_set_name(widgets->vpaned, "vpaned");
	gtk_widget_show(widgets->vpaned);
	gtk_container_add(GTK_CONTAINER(widgets->appview), widgets->vpaned);
	
	/* Browser hpaned */
	widgets->hpaned = gtk_hpaned_new();
	gtk_widget_set_name(widgets->hpaned, "hpaned");
	gtk_widget_show(widgets->hpaned);

	/* Toolbar VBOX */
	widgets->toolbar_vbox = gtk_vbox_new(FALSE, 0);
	gtk_widget_set_name(widgets->toolbar_vbox, "toolbar_vbox");
	gtk_widget_show(widgets->toolbar_vbox);

	/* Add browser and now playing toolbar hpaned to vpaned */
	gtk_container_add(GTK_CONTAINER(widgets->vpaned), widgets->hpaned);
	gtk_container_add(GTK_CONTAINER(widgets->vpaned),
			  widgets->toolbar_vbox);
#ifdef MAEMO
	gtk_paned_set_position(GTK_PANED(widgets->hpaned), 350);
	gtk_paned_set_position(GTK_PANED(widgets->vpaned), 300);
#else
	gtk_paned_set_position(GTK_PANED(widgets->hpaned), 500);
	gtk_paned_set_position(GTK_PANED(widgets->vpaned), 450);
#endif
	
#ifdef MAEMO
	/* Set mute */
	widgets->volume = 50;
	widgets->mute = FALSE;
#endif

	/* Create the device & folder views & models */
	create_folder_model_view();
	create_device_view();

	/* Add the scrolled window to the main hpaned */
	gtk_container_add(GTK_CONTAINER(widgets->hpaned),
			  widgets->device_scrolled_window);
	
	/* Add the scrolled window to the main hpaned */
	gtk_container_add(GTK_CONTAINER(widgets->hpaned),
			  widgets->folder_scrolled_window);
	
	g_signal_connect(G_OBJECT(gtk_tree_view_get_selection
                         (GTK_TREE_VIEW(widgets->device_view))), 
			 "changed",
			 G_CALLBACK(device_view_row_selected), 
			 widgets);

	g_signal_connect_after(G_OBJECT(widgets->folder_view),
			 "row-expanded",
			 G_CALLBACK(folder_view_row_expanded),
			 widgets);
	
	create_now_playing_bar();
	
	create_menu();

	create_toolbar();
	
	return 0;
}

/**
 * Create application menu
 */
int create_menu()
{
	GtkWidget* menu_playback = NULL;
	GtkWidget* item_playback = NULL;
	GtkWidget* item_play = NULL;
	GtkWidget* item_pause = NULL;
	GtkWidget* item_stop = NULL;

	GtkWidget* item_download = NULL;
	GtkWidget* item_update = NULL;
	GtkWidget* item_settings = NULL;
	GtkWidget* item_close = NULL;

#ifdef MAEMO
	widgets->menu = GTK_WIDGET(
		hildon_appview_get_menu(HILDON_APPVIEW(widgets->appview)));
#else
	widgets->menu = gtk_menu_new();
#endif
	
	/* Create the main menu items */
	item_download = gtk_menu_item_new_with_label("Download...");

	/* Create a "Playback" sub menu */
	menu_playback = gtk_menu_new();
	item_playback = gtk_menu_item_new_with_label ("Playback");

	item_play = gtk_menu_item_new_with_label ("Play");
	gtk_menu_append(menu_playback, item_play);

	item_pause = gtk_menu_item_new_with_label ("Pause");
	gtk_menu_append(menu_playback, item_pause);

	item_stop = gtk_menu_item_new_with_label ("Stop");
	gtk_menu_append(menu_playback, item_stop);

	/* Update */
	item_update = gtk_menu_item_new_with_label ("Update");

	/* Settings */
	item_settings = gtk_menu_item_new_with_label("Settings...");
	
	/* Close */
	item_close = gtk_menu_item_new_with_label("Close");

	/* These are deprecated functions */
	gtk_menu_append(widgets->menu, item_download);
	gtk_menu_append(widgets->menu, item_playback);
	gtk_menu_append(widgets->menu, item_update);
	gtk_menu_append(widgets->menu, gtk_separator_menu_item_new());
	gtk_menu_append(widgets->menu, item_settings);
	gtk_menu_append(widgets->menu, item_close);
	
	/* Add playback submenu to the "Playback" item */
	gtk_menu_item_set_submenu(GTK_MENU_ITEM(item_playback), menu_playback);

	/* Download signal */
	g_signal_connect(G_OBJECT(item_download), "activate",
			 GTK_SIGNAL_FUNC(download_clicked), NULL);

	/* Play signal */
	g_signal_connect(G_OBJECT(item_play), "activate",
			 GTK_SIGNAL_FUNC(play_clicked), NULL);

	/* Pause signal */
	g_signal_connect(G_OBJECT(item_pause), "activate",
			 GTK_SIGNAL_FUNC(pause_clicked), NULL);

	/* Stop signal */
	g_signal_connect(G_OBJECT(item_stop), "activate",
			 GTK_SIGNAL_FUNC(stop_clicked), NULL);

	/* Update servers signal */
	g_signal_connect(G_OBJECT(item_update), "activate",
			 GTK_SIGNAL_FUNC(update_clicked), NULL);

	/* Settings signal */
	g_signal_connect(G_OBJECT(item_settings), "activate",
			 GTK_SIGNAL_FUNC(settings_clicked), NULL);

	/* Close signal */
	g_signal_connect(G_OBJECT(item_close), "activate",
			 GTK_SIGNAL_FUNC(close_clicked), NULL);

	/* Make all menu widgets visible */
	gtk_widget_show_all(GTK_WIDGET(widgets->menu));
    
	return 0;
}

/**
 * Create the device tree model & view
 */
int create_toolbar()
{
	GtkToolItem *item;
#ifdef MAEMO
	GtkWidget *image;
#endif
	
	/* If toolbar hasn't been created, create a standard GTK toolbar */
	if (widgets->toolbar == NULL)
	{
		widgets->toolbar = gtk_toolbar_new();
		gtk_widget_set_size_request(widgets->toolbar, 0, 0);
		gtk_container_add(GTK_CONTAINER(widgets->toolbar_vbox),
				widgets->toolbar);
	}

	/* Renderer preset menu button */
#ifdef MAEMO
	image = gtk_image_new_from_icon_name("qgn_widg_wizard",
					     HILDON_ICON_SIZE_SMALL);
	item = gtk_tool_button_new(image, NULL);
#else
	item = gtk_tool_button_new_from_stock(GTK_STOCK_INDEX);
	gtk_tool_button_set_label(GTK_TOOL_BUTTON(item), "Presets");
#endif
	gtk_tool_item_set_expand(item, FALSE);
	gtk_tool_item_set_tooltip(item, widgets->tips, "Presets", "Presets");
	gtk_toolbar_insert(GTK_TOOLBAR(widgets->toolbar), item, -1);
	g_signal_connect(G_OBJECT(item), "clicked",
			 G_CALLBACK(renderer_presets_clicked), NULL);
	widgets->presets_button = item;
	gtk_widget_set_sensitive(GTK_WIDGET(item), FALSE);
	
	/* Separator */
	item = gtk_separator_tool_item_new();
	gtk_separator_tool_item_set_draw(GTK_SEPARATOR_TOOL_ITEM(item), 
					 TRUE);
	gtk_toolbar_insert(GTK_TOOLBAR(widgets->toolbar), item, -1);

	/* Volume Decrease */
#ifdef MAEMO
	image = gtk_image_new_from_icon_name("qgn_indi_gene_minus",
					     HILDON_ICON_SIZE_SMALL);
	item = gtk_tool_button_new(image, "Decrease Volume");
#else
	item = gtk_tool_button_new_from_stock(GTK_STOCK_REMOVE);
	gtk_tool_button_set_label(GTK_TOOL_BUTTON(item), "Softer");
#endif
	gtk_tool_item_set_tooltip(item, widgets->tips, "Decrease Volume",
				  "Decrease Volume" );
	gtk_toolbar_insert(GTK_TOOLBAR(widgets->toolbar), item, -1);
	g_signal_connect(G_OBJECT(item), "clicked",
			 G_CALLBACK(volume_decrease_clicked), NULL);
	widgets->volume_dec_button = item;

	/* Volume Icon / Mute */
#ifdef MAEMO
	image = gtk_image_new_from_icon_name("qgn_indi_gene_volume",
					     HILDON_ICON_SIZE_26);
	item = gtk_tool_button_new(image, NULL);
#else
	item = gtk_tool_button_new_from_stock(GTK_STOCK_CLOSE);
	gtk_tool_button_set_label(GTK_TOOL_BUTTON(item), "Mute");
#endif
	gtk_tool_item_set_tooltip(item, widgets->tips, "Mute", "Mute");
	gtk_toolbar_insert(GTK_TOOLBAR(widgets->toolbar), item, -1);
	g_signal_connect(G_OBJECT(item), "clicked",
			 G_CALLBACK(mute_clicked), NULL);
	widgets->mute_button = item;
	
	/* Volume Increase */
#ifdef MAEMO
	image = gtk_image_new_from_icon_name("qgn_indi_gene_plus",
					     HILDON_ICON_SIZE_SMALL);
	item = gtk_tool_button_new(image, NULL);
#else
	item = gtk_tool_button_new_from_stock(GTK_STOCK_ADD);
	gtk_tool_button_set_label(GTK_TOOL_BUTTON(item), "Louder");
#endif
	gtk_tool_item_set_tooltip(item, widgets->tips, "Increase Volume",
				  "Increase Volume");
	gtk_toolbar_insert(GTK_TOOLBAR(widgets->toolbar), item, -1);
	g_signal_connect(G_OBJECT(item), "clicked",
			 G_CALLBACK(volume_increase_clicked), NULL);
	widgets->volume_inc_button = item;
	
	/* Separator */
	item = gtk_separator_tool_item_new();
	gtk_separator_tool_item_set_draw(GTK_SEPARATOR_TOOL_ITEM(item), 
					 TRUE);
	gtk_toolbar_insert(GTK_TOOLBAR(widgets->toolbar), item, -1);

	/* Previous */
#ifdef MAEMO
	image = gtk_image_new_from_icon_name("qgn_widg_mplayer_rew",
					     HILDON_ICON_SIZE_26);
	item = gtk_tool_button_new(image, NULL);
#else
	item = gtk_tool_button_new_from_stock(GTK_STOCK_GO_BACK);
#endif
	gtk_tool_item_set_tooltip(item, widgets->tips, "Previous", "Previous");
	gtk_toolbar_insert(GTK_TOOLBAR(widgets->toolbar), item, -1);
	g_signal_connect(G_OBJECT(item), "clicked",
			 G_CALLBACK(prev_clicked), NULL);
	
	/* Play */
#ifdef MAEMO
	image = gtk_image_new_from_icon_name("qgn_widg_mplayer_play",
					     HILDON_ICON_SIZE_26);
	item = gtk_tool_button_new(image, NULL);
#else
	item = gtk_tool_button_new_from_stock(GTK_STOCK_JUMP_TO);
	gtk_tool_button_set_label(GTK_TOOL_BUTTON(item), "Play");
#endif
	gtk_tool_item_set_tooltip(item, widgets->tips, "Play", "Play");
	gtk_toolbar_insert(GTK_TOOLBAR(widgets->toolbar), item, -1);
	g_signal_connect(G_OBJECT(item), "clicked",
			 G_CALLBACK(play_clicked), NULL);
	
	/* Pause */
#ifdef MAEMO
	image = gtk_image_new_from_icon_name("qgn_widg_mplayer_pause",
					     HILDON_ICON_SIZE_26);
	item = gtk_tool_button_new(image, NULL);
#else
	item = gtk_tool_button_new_from_stock(GTK_STOCK_GO_DOWN);
	gtk_tool_button_set_label(GTK_TOOL_BUTTON(item), "Pause");
#endif
	gtk_tool_item_set_tooltip(item, widgets->tips, "Pause", "Pause");
	gtk_toolbar_insert(GTK_TOOLBAR(widgets->toolbar), item, -1);
	g_signal_connect(G_OBJECT(item), "clicked",
			 G_CALLBACK(pause_clicked), NULL);
	
	/* Stop */
#ifdef MAEMO
	image = gtk_image_new_from_icon_name("qgn_widg_mplayer_stop",
					     HILDON_ICON_SIZE_26);
	item = gtk_tool_button_new(image, NULL);
#else
	item = gtk_tool_button_new_from_stock(GTK_STOCK_STOP);
#endif
	gtk_tool_item_set_tooltip(item, widgets->tips, "Stop", "Stop");
	gtk_toolbar_insert(GTK_TOOLBAR(widgets->toolbar), item, -1);
	g_signal_connect(G_OBJECT(item), "clicked",
			 G_CALLBACK(stop_clicked), NULL);
	
	/* Next */
#ifdef MAEMO
	image = gtk_image_new_from_icon_name("qgn_widg_mplayer_fwd",
					     HILDON_ICON_SIZE_26);
	item = gtk_tool_button_new(image, NULL);
#else
	item = gtk_tool_button_new_from_stock(GTK_STOCK_GO_FORWARD);
#endif
	gtk_tool_item_set_tooltip(item, widgets->tips, "Next", "Next");
	gtk_toolbar_insert(GTK_TOOLBAR(widgets->toolbar), item, -1);
	g_signal_connect(G_OBJECT(item), "clicked",
			 G_CALLBACK(next_clicked), NULL);

	/* Separator */
	item = gtk_separator_tool_item_new();
	gtk_separator_tool_item_set_draw(GTK_SEPARATOR_TOOL_ITEM(item), TRUE);
	gtk_toolbar_insert(GTK_TOOLBAR(widgets->toolbar), item, -1);

	/* Seek bar */
	widgets->seek_bar = gtk_hscale_new_with_range(0, 100, 1);
	gtk_scale_set_draw_value(GTK_SCALE(widgets->seek_bar), FALSE);
 
	/* Seek tool item */
	item = gtk_tool_item_new();
	GTK_WIDGET_UNSET_FLAGS(item, GTK_CAN_FOCUS);
	gtk_tool_item_set_expand(item, TRUE);
	gtk_container_add(GTK_CONTAINER(item), widgets->seek_bar);
	gtk_tool_item_set_tooltip(item, widgets->tips, "Seek", "Seek");
	gtk_widget_set_sensitive(widgets->seek_bar, FALSE);

	gtk_toolbar_insert(GTK_TOOLBAR(widgets->toolbar), item, -1);

	/* Slider moved signal */
	widgets->seek_bar_signal = 
		g_signal_connect(G_OBJECT(GTK_RANGE(widgets->seek_bar)), 
				 "value-changed",
				 GTK_SIGNAL_FUNC(seek_bar_move), NULL);

#if 0
	/* Playlist button */
#ifdef MAEMO
	image = gtk_image_new_from_icon_name("qgn_list_gene_playlist",
					     HILDON_ICON_SIZE_26);
	item = gtk_tool_button_new(image, NULL);
#else
	item = gtk_tool_button_new_from_stock(GTK_STOCK_INDEX);
	gtk_tool_button_set_label(GTK_TOOL_BUTTON(item), "Playlist");
#endif
	gtk_tool_item_set_tooltip(item, widgets->tips, "Playlist", "Playlist");
	gtk_toolbar_insert(GTK_TOOLBAR(widgets->toolbar), item, -1);
	g_signal_connect(G_OBJECT(item), "clicked",
			 G_CALLBACK(playlist_clicked), NULL);

	/* Add to playlist button */
#ifdef MAEMO
	image = gtk_image_new_from_icon_name("qgn_widg_mplayer_add",
					     HILDON_ICON_SIZE_26);
	item = gtk_tool_button_new(image, NULL);
#else
	item = gtk_tool_button_new_from_stock(GTK_STOCK_INDEX);
	gtk_tool_button_set_label(GTK_TOOL_BUTTON(item), "Add");
#endif
	gtk_tool_item_set_tooltip(item, widgets->tips, "Add", "Add");
	gtk_toolbar_insert(GTK_TOOLBAR(widgets->toolbar), item, -1);
	g_signal_connect(G_OBJECT(item), "clicked",
			 G_CALLBACK(playlist_add_clicked), NULL);

	/* Add dir to playlist button */
#ifdef MAEMO
	image = gtk_image_new_from_icon_name("qgn_list_btno_misc",
					     HILDON_ICON_SIZE_26);
	item = gtk_tool_button_new(image, NULL);
#else
	item = gtk_tool_button_new_from_stock(GTK_STOCK_INDEX);
	gtk_tool_button_set_label(GTK_TOOL_BUTTON(item), "Add all");
#endif
	gtk_tool_item_set_tooltip(item, widgets->tips, "Add all", "Add all");
	gtk_toolbar_insert(GTK_TOOLBAR(widgets->toolbar), item, -1);
	g_signal_connect(G_OBJECT(item), "clicked",
			 G_CALLBACK(playlist_add_all_clicked), NULL);
#endif

#ifndef MAEMO
	/* Separator */
	item = gtk_separator_tool_item_new();
	gtk_separator_tool_item_set_draw(GTK_SEPARATOR_TOOL_ITEM(item), 
					 TRUE);
	gtk_toolbar_insert(GTK_TOOLBAR(widgets->toolbar), item, -1);

	/* Update Media Servers */
	item = gtk_tool_button_new_from_stock(GTK_STOCK_REFRESH);
	gtk_tool_button_set_label(GTK_TOOL_BUTTON(item), "Update");

	gtk_tool_item_set_tooltip(item, widgets->tips, "Update", "Update");
	gtk_toolbar_insert(GTK_TOOLBAR(widgets->toolbar), item, -1);
	g_signal_connect(G_OBJECT(item), "clicked",
			 G_CALLBACK(update_clicked), NULL);

	/* Download */
	item = gtk_tool_button_new_from_stock(GTK_STOCK_SAVE);
	
	gtk_tool_item_set_tooltip(item, widgets->tips, "Download file",
				  "Download file");
	gtk_toolbar_insert(GTK_TOOLBAR(widgets->toolbar), item, -1);
	g_signal_connect(G_OBJECT(item), "clicked",
			 G_CALLBACK(download_clicked), NULL);

	/* Settings */
	item = gtk_tool_button_new_from_stock(GTK_STOCK_PREFERENCES);
	
	gtk_tool_item_set_tooltip(item, widgets->tips, "Preferences",
				  "Preferences");
	gtk_toolbar_insert(GTK_TOOLBAR(widgets->toolbar), item, -1);
	g_signal_connect(G_OBJECT(item), "clicked",
			 G_CALLBACK(settings_clicked), NULL);

#endif
	return 0;
}

/*****************************************************************************
 * Callbacks
 *****************************************************************************/

void close_clicked()
{
	gtk_main_quit();
}

void settings_clicked()
{
	settings_edit_dialog(widgets->settings);
}

void download_clicked()
{
	save_data_to_file();
}

void update_clicked()
{
	cg_upnp_controlpoint_search(controlPoint,
				    CG_UPNP_DEVICE_UPNP_ROOTDEVICE);
}

void renderer_presets_clicked()
{
	preset_dialog_run();
}

void volume_decrease_clicked()
{
	decrease_volume();
}

void mute_clicked()
{
	mute_media();
}

void volume_increase_clicked()
{
	increase_volume();
}

void prev_clicked()
{
	previous_media();
}

void play_clicked()
{
	play_media();
}

void pause_clicked()
{
	pause_media();
}

void stop_clicked()
{
	stop_media();
}

void next_clicked()
{
	next_media();
}

void playlist_clicked()
{
	playlist_dialog_run();
}

void playlist_add_clicked()
{
}

void playlist_add_all_clicked()
{
}

void seek_bar_move(GtkRange* range, gpointer user_data)
{
	g_return_if_fail(range != NULL);

	g_signal_handler_block(range, widgets->seek_bar_signal);

	seek_media((long) gtk_range_get_value(range));

	g_signal_handler_unblock(range, widgets->seek_bar_signal);
}
